/* refbase.hh - base class for using Glib::RefPtr
 * Copyright 2003 Bas Wijnen <wijnen@debian.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef SHEVEK_REFBASE_HH
#define SHEVEK_REFBASE_HH

#include <sigc++/sigc++.h>
#include <glibmm/refptr.h>

namespace shevek
{
  /// Base class for classes which want reference counting through Glib::RefPtr.
  class refbase
  {
  public:
    /// Identical to GLib::RefPtr <>::cast_dynamic, but nicer to type.
    template <typename _T> Glib::RefPtr <_T> cast_dynamic ()
    { return Glib::RefPtr <_T>::cast_dynamic (refptr_this <refbase> ()); }
  protected:
    /// Constructor, increments reference count.
    refbase ();
    /// Destructor, decrements reference count and destroys the object if it reaches 0.
    virtual ~refbase ();
    /// Get a RefPtr to this, protected because only members should need it.
    /** This function allows member functions, which have a pointer to the
     *  object, but not a Glib::RefPtr, to pass a RefPtr to others.
     */
    template <typename T> Glib::RefPtr <T> refptr_this ();
  private:
    // Not copyable
    refbase (refbase const &that);
    refbase &operator= (refbase const &that);
    // Reference functions, needed for Glib::RefPtr.
    virtual void reference ();
    virtual void unreference ();
    // Allow Glib::RefPtr to call the above.
    template <typename T> friend class Glib::RefPtr;
    // Reference count.
    unsigned m_refcount;
  };

  template <typename T>
  Glib::RefPtr <T> refbase::refptr_this ()
  {
    // This is a bit of a hack, but I don't see a way to solve it otherwise.
    // The idea is that members of refbase-derived classes need a refptr to
    // their this pointer, to give away.
    // The only usable constructor of RefPtr for that purpose is the one which
    // is meant for create ().  However, that does not increment the reference
    // counter.  So I do that by hand.
    reference ();
    return Glib::RefPtr <T> (dynamic_cast <T *> (this) );
  }
}

#endif
