/*
 * Copyright (c) 2003-2010 Alexandre Ratchov <alex@caoua.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 	- Redistributions of source code must retain the above
 * 	  copyright notice, this list of conditions and the
 * 	  following disclaimer.
 *
 * 	- Redistributions in binary form must reproduce the above
 * 	  copyright notice, this list of conditions and the
 * 	  following disclaimer in the documentation and/or other
 * 	  materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * implements misc built-in functions available through the
 * interpreter
 *
 * each function is described in the manual.html file
 */

#include "dbg.h"
#include "defs.h"
#include "node.h"
#include "exec.h"
#include "data.h"
#include "cons.h"

#include "textio.h"
#include "lex.h"
#include "parse.h"

#include "mux.h"
#include "mididev.h"

#include "track.h"
#include "song.h"
#include "user.h"
#include "builtin.h"
#include "smf.h"
#include "saveload.h"

struct song *usong;
unsigned user_flag_batch = 0;
unsigned user_flag_verb = 0;

/* -------------------------------------------------- some tools --- */

/*
 * execute the script in the given file inside the 'exec' environment.
 * the script has acces to all global variables, but not to the local
 * variables of the calling proc. Thus it can be safely used from a
 * procedure
 */
unsigned
exec_runfile(struct exec *exec, char *filename)
{
	struct parse *parse;
	struct name **locals;
	struct node *root;
	struct data *data;
	unsigned res;

	res = 0;
	root = NULL;
	data = NULL;
	parse = parse_new(filename);
	if (!parse) {
		return 0;
	}
	locals = exec->locals;
	exec->locals = &exec->globals;
	if (parse_prog(parse, &root)) {
		res = node_exec(root, exec, &data);
	}
	exec->locals = locals;
	parse_delete(parse);
	node_delete(root);
	return res;
}

/*
 * find the pointer to the songtrk contained in 'var' ('var' must be a
 * reference)
 */
unsigned
exec_lookuptrack(struct exec *o, char *var, struct songtrk **res)
{
	char *name;
	struct songtrk *t;
	if (!exec_lookupname(o, var, &name)) {
		return 0;
	}
	t = song_trklookup(usong, name);
	if (t == NULL) {
		cons_errs(name, "no such track");
		return 0;
	}
	*res = t;
	return 1;
}

/*
 * find the (dev, ch) couple for the channel referenced by 'var'
 * 'var' can be
 *	- a reference to a songchan
 *	- a list of two integers (like '{dev chan}')
 */
unsigned
exec_lookupchan_getnum(struct exec *o, char *var,
    unsigned *dev, unsigned *ch, int input) {
	struct var *arg;

	arg = exec_varlookup(o, var);
	if (!arg) {
		dbg_puts("exec_lookupchan_getnum: no such var\n");
		dbg_panic();
	}
	if (!data_list2chan(arg->data, dev, ch, input)) {
		return 0;
	}
	return 1;
}

/*
 * find the pointer to an existing songchan that is referenced by
 * 'var'.  ('var' must be a reference)
 */
unsigned
exec_lookupchan_getref(struct exec *o, char *var, 
    struct songchan **res, int input)
{
	struct var *arg;
	struct songchan *i;

	arg = exec_varlookup(o, var);
	if (!arg) {
		dbg_puts("exec_lookupchan: no such var\n");
		dbg_panic();
	}
	if (arg->data->type == DATA_REF) {
		i = song_chanlookup(usong, arg->data->val.ref, input);
	} else {
		cons_err("bad channel name");
		return 0;
	}
	if (i == NULL) {
		cons_errs(arg->data->val.ref, "no such chan");
		return 0;
	}
	*res = i;
	return 1;
}

/*
 * find the pointer to the songfilt contained in 'var' ('var' must be
 * a reference)
 */
unsigned
exec_lookupfilt(struct exec *o, char *var, struct songfilt **res)
{
	char *name;
	struct songfilt *f;
	if (!exec_lookupname(o, var, &name)) {
		return 0;
	}
	f = song_filtlookup(usong, name);
	if (f == NULL) {
		cons_errs(name, "no such filt");
		return 0;
	}
	*res = f;
	return 1;
}

/*
 * find the pointer to the songsx contained in 'var' ('var' must be a
 * reference)
 */
unsigned
exec_lookupsx(struct exec *o, char *var, struct songsx **res)
{
	char *name;
	struct songsx *t;
	if (!exec_lookupname(o, var, &name)) {
		return 0;
	}
	t = song_sxlookup(usong, name);
	if (t == NULL) {
		cons_errs(name, "no such sysex");
		return 0;
	}
	*res = t;
	return 1;
}

/*
 * fill the event with the one referenced by 'var'
 * 'var' can be:
 * 	- { noff chan xxx yyy }
 * 	- { non chan xxx yyy }
 * 	- { ctl chan xxx yyy }
 * 	- { kat chan xxx yyy }
 * 	- { cat chan xxx }
 * 	- { pc chan xxx }
 * 	- { bend chan xxx }
 *	- { xctl chan xxx yyy }
 *	- { xpc chan xxx yyy }
 *	- { rpn chan xxx yyy }
 *	- { nrpn chan xxx yyy }
 * where 'chan' is in the same as in lookupchan_getnum
 * and 'xxx' and 'yyy' are integers
 */
unsigned
exec_lookupev(struct exec *o, char *name, struct ev *ev, int input)
{
	struct var *arg;
	struct data *d;
	unsigned dev, ch, max, num;

	arg = exec_varlookup(o, name);
	if (!arg) {
		dbg_puts("exec_lookupev: no such var\n");
		dbg_panic();
	}
	d = arg->data;

	if (d->type != DATA_LIST) {
		cons_err("event spec must be a list");
		return 0;
	}
	d = d->val.list;
	if (!d || d->type != DATA_REF ||
	    !ev_str2cmd(ev, d->val.ref) ||
	    !EV_ISVOICE(ev)) {
		cons_err("bad status in event spec");
		return 0;
	}
	d = d->next;
	if (!d) {
		cons_err("no channel in event spec");
		return 0;
	}
	if (!data_list2chan(d, &dev, &ch, input)) {
		return 0;
	}
	ev->dev = dev;
	ev->ch = ch;
	d = d->next;
	if (ev->cmd == EV_XCTL || ev->cmd == EV_CTL) {
		if (!d || !data_getctl(d, &num)) {
			return 0;
		}
		ev->ctl_num = num;
	} else {
		if (ev->cmd == EV_BEND || ev->cmd == EV_NRPN || ev->cmd == EV_RPN) {
			max = EV_MAXFINE;
		} else {
			max = EV_MAXCOARSE;
		}
		if (!d || d->type != DATA_LONG || d->val.num < 0 || d->val.num > max) {
			cons_err("bad byte0 in event spec");
			return 0;
		}
		ev->v0 = d->val.num;
	}
	d = d->next;
	if (ev->cmd == EV_PC || ev->cmd == EV_CAT || ev->cmd == EV_BEND) {
		if (d) {
			cons_err("extra data in event spec");
			return 0;
		}
	} else {
		if (ev->cmd == EV_XPC || ev->cmd == EV_XCTL ||
		    ev->cmd == EV_NRPN || ev->cmd == EV_RPN) {
			max = EV_MAXFINE;
		} else {
			max = EV_MAXCOARSE;
		}
		if (!d || d->type != DATA_LONG ||
		    d->val.num < 0 || d->val.num > max) {
			cons_err("bad byte1 in event spec");
			return 0;
		}
		ev->v1 = d->val.num;
	}

	/*
	 * convert all CTL -> XCTL and PC -> XPC
	 */
	if (ev->cmd == EV_CTL) {
		ev->cmd = EV_XCTL;
		ev->ctl_val <<= 7;
	}
	if (ev->cmd == EV_PC) {
		ev->cmd = EV_XPC;
		ev->pc_bank = EV_UNDEF;
	}
	return 1;
}

/*
 * fill the evspec with the one referenced by var.
 * var is of this form
 * 	- { [type [chanrange [xxxrange [yyyrange]]]] }
 * (brackets mean argument is optionnal)
 */
unsigned
exec_lookupevspec(struct exec *o, char *name, struct evspec *e, int input)
{
	struct var *arg;
	struct data *d;
	struct songchan *i;
	unsigned lo, hi, min, max;

	arg = exec_varlookup(o, name);
	if (!arg) {
		dbg_puts("exec_lookupev: no such var\n");
		dbg_panic();
	}
	d = arg->data;
	if (d->type != DATA_LIST) {
		cons_err("list expected in event range spec");
		return 0;
	}
	evspec_reset(e);

	/*
	 * find the event type
	 */
	d = d->val.list;
	if (!d) {
		goto done;
	}
	if (d->type != DATA_REF ||
	    !evspec_str2cmd(e, d->val.ref)) {
		cons_err("bad status in event spec");
		return 0;
	}
	e->v0_min = evinfo[e->cmd].v0_min;
	e->v0_max = evinfo[e->cmd].v0_max;
	e->v1_min = evinfo[e->cmd].v1_min;
	e->v1_max = evinfo[e->cmd].v1_max;

	/*
	 * find the {device channel} pair
	 */
	d = d->next;
	if (!d) {
		goto done;
	}
	if ((evinfo[e->cmd].flags & (EV_HAS_DEV | EV_HAS_CH)) == 0) {
		goto toomany;
	}
	if (d->type == DATA_REF) {
		i = song_chanlookup(usong, d->val.ref, input);
		if (i == NULL) {
			cons_err("no such chan name");
			return 0;
		}
		e->dev_min = e->dev_max = i->dev;
		e->ch_min = e->ch_max = i->ch;
	} else if (d->type == DATA_LIST) {
		if (!d->val.list) {		/* empty list = any chan/dev */
			/* nothing */
		} else if (d->val.list &&
		    d->val.list->next &&
		    !d->val.list->next->next) {
			if (!data_list2range(d->val.list, 0, EV_MAXDEV, &lo, &hi)) {
				return 0;
			}
			e->dev_min = lo;
			e->dev_max = hi;
			if (!data_list2range(d->val.list->next, 0, EV_MAXCH, &lo, &hi)) {
				return 0;
			}
			e->ch_min = lo;
			e->ch_max = hi;
		} else {
			cons_err("bad channel range spec");
			return 0;
		}
	} else if (d->type == DATA_LONG) {
		if (d->val.num < 0 || d->val.num > EV_MAXDEV) {
			cons_err("bad device number");
			return 0;
		}
		e->dev_min = e->dev_max = d->val.num;
		e->ch_min = 0;
		e->ch_max = EV_MAXCH;
	} else { 
		cons_err("list or ref expected as channel range spec");
		return 0;
	}

	/*
	 * find the first parameter range
	 */
	d = d->next;
	if (!d) {
		goto done;
	}
	if (evinfo[e->cmd].nranges < 1) {
		goto toomany;
	}
	if ((e->cmd == EVSPEC_CTL || e->cmd == EV_XCTL) &&
	    d->type == DATA_REF) {
		if (!data_getctl(d, &hi)) {
			return 0;
		}
		e->v0_min = e->v0_max = hi;
	} else {
		if (!data_list2range(d, e->v0_min, e->v0_max, &min, &max)) {
			return 0;
		}
		e->v0_min = min;
		e->v0_max = max;
	}

	/*
	 * find the second parameter range; we desallow CTL/XCTL values
	 * in order not to interefere with states
	 */
	d = d->next;
	if (!d) {
		goto done;
	}
	if (evinfo[e->cmd].nranges < 2) {
		goto toomany;
	}
	if (!data_list2range(d, e->v1_min, e->v1_max, &min, &max)) {
		return 0;
	}
	e->v1_min = min;
	e->v1_max = max;
	d = d->next;
	if (d) {
		goto toomany;
	}
 done:
	/*
	 * convert PC->XPC and CTL->XCTL
	 */
	if (e->cmd == EVSPEC_PC) {
		e->cmd = EVSPEC_XPC;
		e->v1_min = evinfo[e->cmd].v1_min;
		e->v1_max = evinfo[e->cmd].v1_max;
	} else if (e->cmd == EVSPEC_CTL) {
		e->cmd = EVSPEC_XCTL;
		e->v1_min =  (e->v1_min << 7) & 0x3fff;
		e->v1_max = ((e->v1_max << 7) & 0x3fff) | 0x7f;
	}

#if 0
	dbg_puts("lookupevspec: ");
	evspec_dbg(e);
	dbg_puts("\n");
#endif

	return 1;
 toomany:
	cons_err("too many ranges/values in event spec");
	return 0;
}

/*
 * find the (hi lo) couple for the 14bit controller
 * number:
 *	- an integer for 7bit controller
 *	- a list of two integers (like '{hi lo}')
 */
unsigned
exec_lookupctl(struct exec *o, char *var, unsigned *num)
{
	struct var *arg;

	arg = exec_varlookup(o, var);
	if (!arg) {
		dbg_puts("exec_lookupctl: no such var\n");
		dbg_panic();
	}
	return data_getctl(arg->data, num);
}

/*
 * print a struct data to the console
 */
void
data_print(struct data *d)
{
	struct data *i;

	switch(d->type) {
	case DATA_NIL:
		textout_putstr(tout, "nil");
		break;
	case DATA_LONG:
		if (d->val.num < 0) {
			textout_putstr(tout, "-");
			textout_putlong(tout, -d->val.num);
		} else {
			textout_putlong(tout, d->val.num);
		}
		break;
	case DATA_STRING:
		textout_putstr(tout, "\"");
		textout_putstr(tout, d->val.str);
		textout_putstr(tout, "\"");
		break;
	case DATA_REF:
		textout_putstr(tout, d->val.ref);
		break;
	case DATA_LIST:
		textout_putstr(tout, "{");
		for (i = d->val.list; i != NULL; i = i->next) {
			data_print(i);
			if (i->next) {
				textout_putstr(tout, " ");
			}
		}
		textout_putstr(tout, "}");
		break;
	case DATA_RANGE:
		textout_putlong(tout, d->val.range.min);
		textout_putstr(tout, ":");
		textout_putlong(tout, d->val.range.max);
		break;
	default:
		dbg_puts("data_print: unknown type\n");
		break;
	}
}

/*
 * convert 2 integer lists to channels
 */
unsigned
data_num2chan(struct data *o, unsigned *res_dev, unsigned *res_ch)
{
	long dev, ch;

	if (o == NULL ||
	    o->next == NULL ||
	    o->next->next != NULL ||
	    o->type != DATA_LONG ||
	    o->next->type != DATA_LONG) {
		cons_err("bad {dev midichan} in spec");
		return 0;
	}
	dev = o->val.num;
	ch = o->next->val.num;
	if (ch < 0 || ch > EV_MAXCH ||
	    dev < 0 || dev > EV_MAXDEV) {
		cons_err("bad dev/midichan ranges");
		return 0;
	}
	*res_dev = dev;
	*res_ch = ch;
	return 1;
}

/*
 * lookup the value of the given controller
 */
unsigned
exec_lookupval(struct exec *o, char *n, unsigned isfine, unsigned *r)
{
	struct var *arg;
	unsigned max;

	arg = exec_varlookup(o, n);
	if (!arg) {
		dbg_puts("exec_lookupval: no such var\n");
		dbg_panic();
	}
	if (arg->data->type == DATA_NIL) {
		*r = EV_UNDEF;
		return 1;
	} else if (arg->data->type == DATA_LONG) {
		   max = isfine ? EV_MAXFINE : EV_MAXCOARSE;
		   if (arg->data->val.num < 0 || arg->data->val.num > max) {
			   cons_err("controller value out of range");
			   return 0;
		   }
		   *r = arg->data->val.num;
		   return 1;
	} else {
		cons_err("bad type of controller value");
		return 0;
	}
}

/*
 * convert lists to channels, 'data' can be
 * 	- a reference to an existing songchan
 *	- a pair of integers '{ dev midichan }'
 */
unsigned
data_list2chan(struct data *o, unsigned *res_dev, unsigned *res_ch, int input)
{
	struct songchan *i;

	if (o->type == DATA_LIST) {
		return data_num2chan(o->val.list, res_dev, res_ch);
	} else if (o->type == DATA_REF) {
		i = song_chanlookup(usong, o->val.ref, input);
		if (i == NULL) {
			cons_errs(o->val.ref, "no such chan name");
			return 0;
		}
		*res_dev = i->dev;
		*res_ch = i->ch;
		return 1;
	} else {
		cons_err("bad channel specification");
		return 0;
	}
}

/*
 * convert a struct data to a pair of integers
 * data can be:
 * 	- a liste of 2 integers
 *	- a single integer (then min = max)
 */
unsigned
data_list2range(struct data *d, unsigned min, unsigned max,
    unsigned *lo, unsigned *hi) {
    	if (d->type == DATA_LONG) {
		*lo = *hi = d->val.num;
	} else if (d->type == DATA_LIST) {
		d = d->val.list;
		if (!d) {
			*lo = min;
			*hi = max;
			return 1;
		}
		if (!d->next || d->next->next ||
		    d->type != DATA_LONG || d->next->type != DATA_LONG) {
			cons_err("exactly 0 or 2 numbers expected in range spec");
			return 0;
		}
		*lo = d->val.num;
		*hi = d->next->val.num;
	} else if (d->type == DATA_RANGE) {
		*lo = d->val.range.min;
		*hi = d->val.range.max;
	} else {
		cons_err("range or number expected in range spec");
		return 0;
	}
	if (*lo < min || *lo > max || *hi < min || *hi > max || *lo > *hi) {
		cons_err("range values out of bounds");
		return 0;
	}
	return 1;
}

/*
 * convert a list to bitmap of continuous controllers
 */
unsigned
data_list2ctlset(struct data *d, unsigned *res)
{
	unsigned ctlset;

	ctlset = 0;
	while (d) {
		if (d->type != DATA_LONG) {
			cons_err("not-a-number in controller set");
			return 0;
		}
		if (d->val.num < 0 || d->val.num >= 32) {
			cons_err("controller number out of range 0..31");
			return 0;
		}
		if (evctl_isreserved(d->val.num)) {
			cons_erru(d->val.num, "controller number reserved");
			return 0;
		}
		ctlset |= (1 << d->val.num);
		d = d->next;
	}
	*res = ctlset;
	return 1;
}


/*
 * check if the pattern in data (list of integers)
 * match the beggining of the given sysex
 */
unsigned
data_matchsysex(struct data *d, struct sysex *sx, unsigned *res)
{
	unsigned i;
	struct chunk *ck;

	i = 0;
	ck = sx->first;
	while (d) {
		if (d->type != DATA_LONG) {
			cons_err("not-a-number in sysex pattern");
			return 0;
		}
		for (;;) {
			if (!ck) {
				*res = 0;
				return 1;
			}
			if (i < ck->used) {
				break;
			}
			ck = ck->next;
			i = 0;
		}
		if (d->val.num != ck->data[i++]) {
			*res = 0;
			return 1;
		}
		d = d->next;
	}
	*res = 1;
	return 1;
}

/*
 * convert a 'struct data' to a controller number
 */
unsigned
data_getctl(struct data *d, unsigned *num)
{
    	if (d->type == DATA_LONG) {
		if (d->val.num < 0 || d->val.num > EV_MAXCOARSE) {
			cons_err("7bit ctl number out of bounds");
			return 0;
		}
		if (evctl_isreserved(d->val.num)) {
			cons_err("controller is reserved for bank, rpn, nrpn");
			return 0;
		}
		*num = d->val.num;
	} else if (d->type == DATA_REF) {
		if (!evctl_lookup(d->val.ref, num)) {
			cons_errs(d->val.ref, "no such controller");
			return 0;
		}
	} else {
		cons_err("number or identifier expected in ctl spec");
		return 0;
	}
	return 1;
}

/* ---------------------------------------- interpreter functions --- */


unsigned
user_func_info(struct exec *o, struct data **r)
{
	exec_dumpprocs(o);
	exec_dumpvars(o);
	return 1;
}


unsigned
user_func_shut(struct exec *o, struct data **r)
{
	unsigned i;
	struct ev ev;
	struct mididev *dev;

	/*
	 * XXX: should raise mode to SONG_IDLE and
	 * use mixout
	 */
	if (!song_try_mode(usong, 0)) {
		return 0;
	}
	mux_open();
	for (dev = mididev_list; dev != NULL; dev = dev->next) {
		for (i = 0; i < EV_MAXCH; i++) {
			ev.cmd = EV_XCTL;
			ev.dev = dev->unit;
			ev.ch = i;
			ev.ctl_num = 121;
			ev.ctl_val = 0;
			mux_putev(&ev);
			ev.cmd = EV_XCTL;
			ev.dev = dev->unit;
			ev.ch = i;
			ev.ctl_num = 123;
			ev.ctl_val = 0;
			mux_putev(&ev);
			ev.cmd = EV_BEND;
			ev.dev = dev->unit;
			ev.ch = i;
			ev.bend_val = EV_BEND_DEFAULT;
			mux_putev(&ev);
		}
	}
	mux_close();
	return 1;
}

unsigned
user_func_proclist(struct exec *o, struct data **r)
{
	struct proc *i;
	struct data *d, *n;

	d = data_newlist(NULL);
	PROC_FOREACH(i, o->procs) {
		if (i->code->vmt == &node_vmt_slist) {
			n = data_newref(i->name.str);
			data_listadd(d, n);
		}
	}
	*r = d;
	return 1;
}

unsigned
user_func_builtinlist(struct exec *o, struct data **r)
{
	struct proc *i;
	struct data *d, *n;

	d = data_newlist(NULL);
	PROC_FOREACH(i, o->procs) {
		if (i->code->vmt == &node_vmt_builtin) {
			n = data_newref(i->name.str);
			data_listadd(d, n);
		}
	}
	*r = d;
	return 1;
}

unsigned
user_mainloop(void)
{
	struct parse *parse;
	struct exec *exec;
	struct node *root;
	struct data *data;
	unsigned result, exitcode;

	/*
	 * create the project (ie the song) and
	 * the execution environment of the interpreter
	 */
	usong = song_new();
	exec = exec_new();

	/*
	 * register built-in functions
	 */
	exec_newbuiltin(exec, "print", blt_print,
			name_newarg("value", NULL));
	exec_newbuiltin(exec, "err", blt_err,
			name_newarg("message", NULL));
	exec_newbuiltin(exec, "h", blt_h,
			name_newarg("function", NULL));
	exec_newbuiltin(exec, "exec", blt_exec,
			name_newarg("filename", NULL));
	exec_newbuiltin(exec, "debug", blt_debug,
			name_newarg("flag",
			name_newarg("value", NULL)));
	exec_newbuiltin(exec, "panic", blt_panic, NULL);
	exec_newbuiltin(exec, "info", user_func_info, NULL);

	exec_newbuiltin(exec, "getunit", blt_getunit, NULL);
	exec_newbuiltin(exec, "setunit", blt_setunit,
			name_newarg("tics_per_unit", NULL));
	exec_newbuiltin(exec, "getfac", blt_getfac, NULL);
	exec_newbuiltin(exec, "fac", blt_fac,
			name_newarg("tempo_factor", NULL));
	exec_newbuiltin(exec, "getpos", blt_getpos, NULL);
	exec_newbuiltin(exec, "g", blt_goto,
			name_newarg("measure", NULL));
	exec_newbuiltin(exec, "getlen", blt_getlen, NULL);
	exec_newbuiltin(exec, "sel", blt_sel,
			name_newarg("length", NULL));
	exec_newbuiltin(exec, "getq", blt_getq, NULL);
	exec_newbuiltin(exec, "setq", blt_setq,
			name_newarg("step", NULL));
	exec_newbuiltin(exec, "ev", blt_ev,
			name_newarg("evspec", NULL));
	exec_newbuiltin(exec, "gett", blt_gett, NULL);
	exec_newbuiltin(exec, "ct", blt_ct,
			name_newarg("trackname", NULL));
	exec_newbuiltin(exec, "getf", blt_getf, NULL);
	exec_newbuiltin(exec, "cf", blt_cf,
			name_newarg("filtname", NULL));
	exec_newbuiltin(exec, "getx", blt_getx, NULL);
	exec_newbuiltin(exec, "cx", blt_cx,
			name_newarg("sysexname", NULL));
	exec_newbuiltin(exec, "geti", blt_geti, NULL);
	exec_newbuiltin(exec, "ci", blt_ci,
			name_newarg("channame", NULL));
	exec_newbuiltin(exec, "geto", blt_geto, NULL);
	exec_newbuiltin(exec, "co", blt_co,
			name_newarg("channame", NULL));
	exec_newbuiltin(exec, "mute", blt_mute,
			name_newarg("trackname", NULL));
	exec_newbuiltin(exec, "unmute", blt_unmute,
			name_newarg("trackname", NULL));
	exec_newbuiltin(exec, "getmute", blt_getmute,
			name_newarg("trackname", NULL));
	exec_newbuiltin(exec, "ls", blt_ls, NULL);
	exec_newbuiltin(exec, "save", blt_save,
			name_newarg("filename", NULL));
	exec_newbuiltin(exec, "load", blt_load,
			name_newarg("filename", NULL));
	exec_newbuiltin(exec, "reset", blt_reset, NULL);
	exec_newbuiltin(exec, "export", blt_export,
			name_newarg("filename", NULL));
	exec_newbuiltin(exec, "import", blt_import,
			name_newarg("filename", NULL));
	exec_newbuiltin(exec, "i", blt_idle, NULL);
	exec_newbuiltin(exec, "p", blt_play, NULL);
	exec_newbuiltin(exec, "r", blt_rec, NULL);
	exec_newbuiltin(exec, "s", blt_stop, NULL);
	exec_newbuiltin(exec, "t", blt_tempo,
			name_newarg("beats_per_minute", NULL));
	exec_newbuiltin(exec, "mins", blt_mins,
			name_newarg("amount",
			name_newarg("sig", NULL)));
	exec_newbuiltin(exec, "mcut", blt_mcut, NULL);
	exec_newbuiltin(exec, "mdup", blt_mdup,
			name_newarg("where", NULL));
	exec_newbuiltin(exec, "minfo", blt_minfo, NULL);
	exec_newbuiltin(exec, "mtempo", blt_mtempo, NULL);
	exec_newbuiltin(exec, "msig", blt_msig, NULL);
	exec_newbuiltin(exec, "mend", blt_mend, NULL);
	exec_newbuiltin(exec, "ctlconf", blt_ctlconf,
			name_newarg("name",
			name_newarg("ctl",
			name_newarg("defval", NULL))));
	exec_newbuiltin(exec, "ctlconfx", blt_ctlconfx,
			name_newarg("name",
			name_newarg("ctl",
			name_newarg("defval", NULL))));
	exec_newbuiltin(exec, "ctlunconf", blt_ctlunconf,
			name_newarg("name", NULL));
	exec_newbuiltin(exec, "ctlinfo", blt_ctlinfo, NULL);
	exec_newbuiltin(exec, "m", blt_metro,
			name_newarg("onoff", NULL));
	exec_newbuiltin(exec, "metrocf", blt_metrocf,
			name_newarg("eventhi",
			name_newarg("eventlo", NULL)));

	exec_newbuiltin(exec, "tlist", blt_tlist, NULL);
	exec_newbuiltin(exec, "tnew", blt_tnew,
			name_newarg("trackname", NULL));
	exec_newbuiltin(exec, "tdel", blt_tdel, NULL);
	exec_newbuiltin(exec, "tren", blt_tren,
			name_newarg("newname", NULL));
	exec_newbuiltin(exec, "texists", blt_texists,
			name_newarg("trackname", NULL));
	exec_newbuiltin(exec, "taddev", blt_taddev,
			name_newarg("measure",
			name_newarg("beat",
			name_newarg("tic",
			name_newarg("event", NULL)))));
	exec_newbuiltin(exec, "tsetf", blt_tsetf,
			name_newarg("filtname", NULL));
	exec_newbuiltin(exec, "tgetf", blt_tgetf, NULL);
	exec_newbuiltin(exec, "tcheck", blt_tcheck, NULL);
	exec_newbuiltin(exec, "tcut", blt_tcut, NULL);
	exec_newbuiltin(exec, "tclr", blt_tclr, NULL);
	exec_newbuiltin(exec, "tpaste", blt_tpaste, NULL);
	exec_newbuiltin(exec, "tcopy", blt_tcopy, NULL);
	exec_newbuiltin(exec, "tins", blt_tins,
			name_newarg("amount", NULL));
	exec_newbuiltin(exec, "tmerge", blt_tmerge,
			name_newarg("source", NULL));
	exec_newbuiltin(exec, "tquant", blt_tquant,
			name_newarg("rate", NULL));
	exec_newbuiltin(exec, "ttransp", blt_ttransp,
			name_newarg("halftones", NULL));
	exec_newbuiltin(exec, "tevmap", blt_tevmap,
			name_newarg("from",
			name_newarg("to", NULL)));
	exec_newbuiltin(exec, "tclist", blt_tclist, NULL);
	exec_newbuiltin(exec, "tinfo", blt_tinfo, NULL);

	exec_newbuiltin(exec, "ilist", blt_ilist, NULL);
	exec_newbuiltin(exec, "iexists", blt_iexists,
			name_newarg("channame", NULL));
	exec_newbuiltin(exec, "iset", blt_iset,
			name_newarg("channum", NULL));
	exec_newbuiltin(exec, "inew", blt_inew,
			name_newarg("channame",
			name_newarg("channum", NULL)));
	exec_newbuiltin(exec, "idel", blt_idel, NULL);
	exec_newbuiltin(exec, "iren", blt_iren,
			name_newarg("newname", NULL));
	exec_newbuiltin(exec, "iinfo", blt_iinfo, NULL);
	exec_newbuiltin(exec, "igetc", blt_igetc, NULL);
	exec_newbuiltin(exec, "igetd", blt_igetd, NULL);
	exec_newbuiltin(exec, "iaddev", blt_iaddev,
			name_newarg("event", NULL));
	exec_newbuiltin(exec, "irmev", blt_irmev,
			name_newarg("evspec", NULL));

	exec_newbuiltin(exec, "olist", blt_olist, NULL);
	exec_newbuiltin(exec, "oexists", blt_oexists,
			name_newarg("channame", NULL));
	exec_newbuiltin(exec, "oset", blt_oset,
			name_newarg("channum", NULL));
	exec_newbuiltin(exec, "onew", blt_onew,
			name_newarg("channame",
			name_newarg("channum", NULL)));
	exec_newbuiltin(exec, "odel", blt_odel, NULL);
	exec_newbuiltin(exec, "oren", blt_oren,
			name_newarg("newname", NULL));
	exec_newbuiltin(exec, "oinfo", blt_oinfo, NULL);
	exec_newbuiltin(exec, "ogetc", blt_ogetc, NULL);
	exec_newbuiltin(exec, "ogetd", blt_ogetd, NULL);
	exec_newbuiltin(exec, "oaddev", blt_oaddev,
			name_newarg("event", NULL));
	exec_newbuiltin(exec, "ormev", blt_ormev,
			name_newarg("evspec", NULL));

	exec_newbuiltin(exec, "flist", blt_flist, NULL);
	exec_newbuiltin(exec, "fexists", blt_fexists,
			name_newarg("filtname", NULL));
	exec_newbuiltin(exec, "fnew", blt_fnew,
			name_newarg("filtname", NULL));
	exec_newbuiltin(exec, "fdel", blt_fdel, NULL);
	exec_newbuiltin(exec, "fren", blt_fren,
			name_newarg("newname", NULL));
	exec_newbuiltin(exec, "finfo", blt_finfo, NULL);
	exec_newbuiltin(exec, "freset", blt_freset, NULL);
	exec_newbuiltin(exec, "fmap", blt_fmap,
			name_newarg("from",
			name_newarg("to", NULL)));
	exec_newbuiltin(exec, "funmap", blt_funmap,
			name_newarg("from",
			name_newarg("to", NULL)));
	exec_newbuiltin(exec, "ftransp", blt_ftransp,
			name_newarg("evspec",
			name_newarg("plus", NULL)));
	exec_newbuiltin(exec, "fvcurve", blt_fvcurve,
			name_newarg("evspec",
			name_newarg("weight", NULL)));
	exec_newbuiltin(exec, "fchgin", blt_fchgin,
			name_newarg("from",
			name_newarg("to", NULL)));
	exec_newbuiltin(exec, "fchgout", blt_fchgout,
			name_newarg("from",
			name_newarg("to", NULL)));
	exec_newbuiltin(exec, "fswapin", blt_fswapin,
			name_newarg("from",
			name_newarg("to", NULL)));
	exec_newbuiltin(exec, "fswapout", blt_fswapout,
			name_newarg("from",
			name_newarg("to", NULL)));

	exec_newbuiltin(exec, "xlist", blt_xlist, NULL);
	exec_newbuiltin(exec, "xexists", blt_xexists,
			name_newarg("sysexname", NULL));
	exec_newbuiltin(exec, "xnew", blt_xnew,
			name_newarg("sysexname", NULL));
	exec_newbuiltin(exec, "xdel", blt_xdel, NULL);
	exec_newbuiltin(exec, "xren", blt_xren,
			name_newarg("newname", NULL));
	exec_newbuiltin(exec, "xinfo", blt_xinfo, NULL);
	exec_newbuiltin(exec, "xrm", blt_xrm,
			name_newarg("data", NULL));
	exec_newbuiltin(exec, "xsetd", blt_xsetd,
			name_newarg("devnum",
			name_newarg("data", NULL)));
	exec_newbuiltin(exec, "xadd", blt_xadd,
			name_newarg("devnum",
			name_newarg("data", NULL)));

	exec_newbuiltin(exec, "shut", user_func_shut, NULL);
	exec_newbuiltin(exec, "proclist", user_func_proclist, NULL);
	exec_newbuiltin(exec, "builtinlist", user_func_builtinlist, NULL);

	exec_newbuiltin(exec, "dnew", blt_dnew,
			name_newarg("devnum",
			name_newarg("path",
			name_newarg("mode", NULL))));
	exec_newbuiltin(exec, "ddel", blt_ddel,
			name_newarg("devnum", NULL));
	exec_newbuiltin(exec, "dmtcrx", blt_dmtcrx,
			name_newarg("devnum", NULL));
	exec_newbuiltin(exec, "dmmctx", blt_dmmctx,
			name_newarg("devlist", NULL));
	exec_newbuiltin(exec, "dclktx", blt_dclktx,
			name_newarg("devlist", NULL));
	exec_newbuiltin(exec, "dclkrx", blt_dclkrx,
			name_newarg("devnum", NULL));
	exec_newbuiltin(exec, "dclkrate", blt_dclkrate,
			name_newarg("devnum",
			name_newarg("tics_per_unit", NULL)));
	exec_newbuiltin(exec, "dinfo", blt_dinfo,
			name_newarg("devnum", NULL));
	exec_newbuiltin(exec, "dixctl", blt_dixctl,
			name_newarg("devnum",
			name_newarg("ctlset", NULL)));
	exec_newbuiltin(exec, "doxctl", blt_doxctl,
			name_newarg("devnum",
			name_newarg("ctlset", NULL)));

	/*
	 * run the user startup script: $HOME/.midishrc or /etc/midishrc
	 */
	if (!user_flag_batch) {
		exec_runrcfile(exec);
		if (mididev_list == NULL)
			cons_err("Warning, no MIDI devices configured.");
	}

	/*
	 * create the parser and start parsing standard input
	 */
	parse = parse_new(NULL);
	if (parse == NULL) {
		return 0;
	}

	cons_putpos(usong->curpos, 0, 0);

	root = NULL;
	data = NULL;
	for (;;) {
		/*
		 * print mem_alloc() and mem_free() stats, useful to
		 * track memory leaks
		 */
		mem_stats();

		/*
		 * parse a block
		 */
		if (!parse_getsym(parse)) {
			goto err;
		}
		if (parse->lex.id == TOK_EOF) {
			/* end-of-file (user quit) */
			exitcode = 1;
			break;
		}
		parse_ungetsym(parse);
		if (!parse_line(parse, &root)) {
			node_delete(root);
			root = NULL;
			goto err;
		}

		/*
		 * at this stage no parse error, execute the tree
		 */
		result = node_exec(root, exec, &data);
		node_delete(root);
		root = NULL;
		if (result == RESULT_OK) {
			continue;
		}
		if (result == RESULT_EXIT) {
			exitcode = 1;	/* 1 means success */
			break;
		}

	err:
		/*
		 * in batch mode stop on the first error
		 */
		if (user_flag_batch) {
			exitcode = 0;	/* 0 means failure */
			break;
		}
	}

	parse_delete(parse);
	exec_delete(exec);
	song_delete(usong);
	usong = NULL;
	mididev_listdone();
	return exitcode;
}
