import unittest

from miro import autodiscover
from miro import feed
from miro import opml

from miro.fileobject import FilenameType

from miro.test.framework import MiroTestCase

URL_1 = "http://www.domain-1.com/videos/rss.xml"
URL_2  = "http://www.domain-2.com/videos/rss.xml"
URL_3 = "http://www.domain-1.com/videos/atom.xml"
URL_4 = "http://www.domain-2.com/videos/atom.xml"

# -----------------------------------------------------------------------------

def _get_subs(entry):
    """Removes most of the boilerplate for getting subscriptions given
    some specific entry or entries in the <body> ... </body> section.
    """
    teststring = u"""\
<?xml version="1.0" encoding="utf-8" ?>
<!-- OPML generated by Miro v3.1-git on Thu Jul 22 15:23:10 2010 -->
<opml version="2.0"
 xmlns:miro="http://getmiro.com/opml/subscriptions">
<head>
  <title>miro_subscriptions.opml</title>
</head>
<body>
  %s
</body>
</opml>""" % entry

    return autodiscover.parse_content(teststring)

class TestImporter(unittest.TestCase):
    def test_simple_flat(self):
        teststring = u"""\
<?xml version="1.0" encoding="utf-8"?>
<opml version="2.0"
 xmlns:miro="http://getmiro.com/opml/subscriptions">
  <head>
    <title>Sample OPML Flat Subscription List</title>
  </head>
  <body>
    <outline text="Sample Dummy RSS Feed 1" type="rss" xmlUrl="%s" />
    <outline text="Sample Dummy Atom Feed 1" type="rss" xmlUrl="%s" />
    <outline text="Sample Dummy RSS Feed 2" type="rss" xmlUrl="%s" />
    <outline text="Sample Dummy Atom Feed 2" type="rss" xmlUrl="%s" />
  </body>
</opml>
""" % (URL_1, URL_3, URL_2, URL_4)

        subscriptions = autodiscover.parse_content(teststring)
        self.assertEquals(len(subscriptions), 4)
        for feed_ in subscriptions:
            self.assertEquals(feed_['type'], 'feed')
        self.assertEquals(subscriptions[0]['url'], URL_1)
        self.assertEquals(subscriptions[1]['url'], URL_3)
        self.assertEquals(subscriptions[2]['url'], URL_2)
        self.assertEquals(subscriptions[3]['url'], URL_4)

    def test_simple_nested(self):
        teststring = u"""\
<?xml version="1.0" encoding="utf-8"?>
<opml version="2.0"
 xmlns:miro="http://getmiro.com/opml/subscriptions">
  <head>
    <title>Sample OPML Flat Subscription List</title>
  </head>
  <body>
    <outline text="folder 1">
      <outline text="Sample Dummy RSS Feed 1" type="rss" xmlUrl="%s" />
      <outline text="folder1-1">
        <outline text="Sample Dummy Atom Feed 1" type="rss" xmlUrl="%s" />
      </outline>
    </outline>
    <outline text="folder 2">
      <outline text="folder2-1">
        <outline text="Sample Dummy RSS Feed 2" type="rss" xmlUrl="%s" />
      </outline>
      <outline text="Sample Dummy Atom Feed 2" type="rss" xmlUrl="%s" />
    </outline>
  </body>
</opml>
""" % (URL_1, URL_3, URL_2, URL_4)

        subscriptions = autodiscover.parse_content(teststring)
        self.assertEquals(len(subscriptions), 4)
        for feed_ in subscriptions:
            self.assertEquals(feed_['type'], 'feed')

        self.assertEquals(subscriptions[0]['url'], URL_1)
        self.assertEquals(subscriptions[1]['url'], URL_3)
        self.assertEquals(subscriptions[2]['url'], URL_2)
        self.assertEquals(subscriptions[3]['url'], URL_4)

    def test_basic_feed(self):
        subs = _get_subs("""\
  <outline type="rss"
    text="GeekBeat.TV (Large MP4)"
    xmlUrl="http://revision3.com/geekbeattv/feed/MP4-Large" />
""")
        feed_ = subs[0]

        self.assertEquals(feed_['type'], 'feed')
        self.assertEquals(
            feed_['url'], u"http://revision3.com/geekbeattv/feed/MP4-Large")
        self.assertEquals(
            feed_['title'], u"GeekBeat.TV (Large MP4)")

    def test_basic_feed_2(self):
        subs = _get_subs("""\
  <outline type="rss"
    text="The Economist: All video"
    xmlUrl="http://feeds2.feedburner.com/feedroom/xDDO"
    miro:autoDownload="off"/>
""")
        feed_ = subs[0]

        self.assertEquals(feed_['type'], 'feed')
        self.assertEquals(
            feed_['url'], u"http://feeds2.feedburner.com/feedroom/xDDO")
        self.assertEquals(
            feed_['title'], u"The Economist: All video")
        self.assertEquals(
            feed_["auto_download_mode"], u"off")

    def test_url(self):
        # if there's no url, there shouldn't be a subscription
        subs = _get_subs("""\
  <outline type="rss" />
""")
        self.assert_(len(subs) == 0)

        # if there's a url, we pick it up
        subs = _get_subs("""\
  <outline type="rss"
    xmlUrl="http://example.com/feed/"/>
""")
        self.assertEquals(subs[0]["url"], u"http://example.com/feed/")

    def test_title(self):
        # if there's no title, then the sub won't have a title key
        subs = _get_subs("""\
  <outline type="rss"
    xmlUrl="http://feeds2.feedburner.com/feedroom/xDDO"
    miro:autoDownload="off"/>  
""")
        self.assert_("title" not in subs[0])

        # if there's no title, then the sub won't have a title key
        subs = _get_subs("""\
  <outline type="rss"
    xmlUrl="http://feeds2.feedburner.com/feedroom/xDDO"
    text=""
    miro:autoDownload="off"/>  
""")
        self.assert_("title" not in subs[0])

        # if title == url, then the sub won't have a title key
        subs = _get_subs("""\
  <outline type="rss"
    xmlUrl="http://feeds2.feedburner.com/feedroom/xDDO"
    text="http://feeds2.feedburner.com/feedroom/xDDO"
    miro:autoDownload="off"/>  
""")
        self.assert_("title" not in subs[0])

    def test_auto_download(self):
        # if there's no autodownload, then there's no key
        subs = _get_subs("""\
  <outline type="rss"
    xmlUrl="http://feeds2.feedburner.com/feedroom/xDDO"/>
""")
        self.assert_("auto_download_mode" not in subs[0])

        # autodownload can be autoDownload
        subs = _get_subs("""\
  <outline type="rss"
    xmlUrl="http://feeds2.feedburner.com/feedroom/xDDO"
    autoDownload="all"/>
""")
        self.assertEquals(subs[0]["auto_download_mode"], "all")

        # autodownload can be miro:autoDownload
        subs = _get_subs("""\
  <outline type="rss"
    xmlUrl="http://feeds2.feedburner.com/feedroom/xDDO"
    miro:autoDownload="all"/>
""")
        self.assertEquals(subs[0]["auto_download_mode"], "all")

        # must be either "all", "off", or "new"
        subs = _get_subs("""\
  <outline type="rss"
    xmlUrl="http://feeds2.feedburner.com/feedroom/xDDO"
    miro:autoDownload="foo"/>
""")
        self.assert_("auto_download_mode" not in subs[0])

    def test_expiry_time(self):
        # if there's no expiryTime, then there's no key
        subs = _get_subs("""\
  <outline type="rss"
    xmlUrl="http://feeds2.feedburner.com/feedroom/xDDO"/>
""")
        self.assert_("expiry_time" not in subs[0])

        # expiry_time can be expiryTime
        subs = _get_subs("""\
  <outline type="rss"
    xmlUrl="http://feeds2.feedburner.com/feedroom/xDDO"
    expiryTime="240"/>
""")
        self.assertEquals(subs[0]["expiry_time"], 240)

        # expiry_time can be miro:expiryTime
        subs = _get_subs("""\
  <outline type="rss"
    xmlUrl="http://feeds2.feedburner.com/feedroom/xDDO"
    miro:expiryTime="240"/>
""")
        self.assertEquals(subs[0]["expiry_time"], 240)

        # expiry_time can be int or "never" or "system"
        subs = _get_subs("""\
  <outline type="rss"
    xmlUrl="http://feeds2.feedburner.com/feedroom/xDDO"
    miro:expiryTime="never"/>
""")
        self.assertEquals(subs[0]["expiry_time"], "never")

        subs = _get_subs("""\
  <outline type="rss"
    xmlUrl="http://feeds2.feedburner.com/feedroom/xDDO"
    miro:expiryTime="system"/>
""")
        self.assertEquals(subs[0]["expiry_time"], "system")

        subs = _get_subs("""\
  <outline type="rss"
    xmlUrl="http://feeds2.feedburner.com/feedroom/xDDO"
    miro:expiryTime="foo"/>
""")
        self.assert_("expiry_time" not in subs[0])

    # FIXME - test folders
    # FIXME - test sites

class TestExporter(MiroTestCase):
    def _get_export(self, feeds, sites):
        return opml.Exporter().export_content(
            FilenameType("/tmp/foo.opml"), feeds, sites)

    def test_simple(self):
        feed1 = feed.Feed(u"http://example.com/feed/")
        feed1.finish_generate_feed(
            feed.RSSFeedImpl(u"http://example.com/feed/",
                             feed1,
                             u"Foo feed"))
            
        data = self._get_export([feed1], [])

        subs = autodiscover.parse_content(data)
        self.assertEquals(subs[0]["url"], u"http://example.com/feed/")
        self.assertEquals(subs[0]["title"], u"Foo feed")
        self.assertEquals(subs[0]["type"], "feed")

    # FIXME - add more export tests
    # FIXME - test folders
    # FIXME - test sites
