/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Oracle and Java are registered trademarks of Oracle and/or its affiliates.
 * Other names may be trademarks of their respective owners.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 * 
 * Contributor(s):
 * 
 * Portions Copyrighted 2008 Sun Microsystems, Inc.
 */
package org.netbeans.modules.cnd.remote.server;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.logging.Level;
import java.util.prefs.Preferences;
import javax.swing.SwingUtilities;
import javax.swing.event.ChangeListener;
import org.netbeans.api.project.Project;
import org.netbeans.modules.cnd.api.remote.ServerList;
import org.netbeans.modules.nativeexecution.api.ExecutionEnvironmentFactory;
import org.netbeans.modules.cnd.api.remote.ServerRecord;
import org.netbeans.modules.cnd.utils.CndPathUtilitities;
import org.netbeans.modules.cnd.remote.support.RemoteCommandSupport;
import org.netbeans.modules.cnd.remote.support.RemoteProjectSupport;
import org.netbeans.modules.cnd.remote.support.RemoteUtil;
import org.netbeans.modules.cnd.spi.remote.RemoteSyncFactory;
import org.netbeans.modules.cnd.spi.remote.ServerListImplementation;
import org.netbeans.modules.cnd.utils.CndUtils;
import org.netbeans.modules.nativeexecution.api.ExecutionEnvironment;
import org.netbeans.modules.nativeexecution.api.util.ConnectionListener;
import org.netbeans.modules.nativeexecution.api.util.ConnectionManager;
import org.netbeans.modules.nativeexecution.api.util.PasswordManager;
import org.openide.util.ChangeSupport;
import org.openide.util.Lookup;
import org.openide.util.NbPreferences;
import org.openide.util.WeakListeners;

/**
 * The cnd.remote implementation of ServerList.
 * 
 * @author gordonp
 */
@org.openide.util.lookup.ServiceProvider(service = ServerListImplementation.class)
public class RemoteServerList implements ServerListImplementation, ConnectionListener {

    private static final String CND_REMOTE = "cnd.remote"; // NOI18N
    private static final String REMOTE_SERVERS = CND_REMOTE + ".servers"; // NOI18N
    private static final String DEFAULT_INDEX = CND_REMOTE + ".default"; // NOI18N
    private int defaultIndex;
    private final PropertyChangeSupport pcs;
    private final ChangeSupport cs;
    private final ArrayList<RemoteServerRecord> unlisted;
    private final ArrayList<RemoteServerRecord> items = new ArrayList<RemoteServerRecord>();

    public RemoteServerList() {
        defaultIndex = getPreferences().getInt(DEFAULT_INDEX, 0);
        pcs = new PropertyChangeSupport(this);
        cs = new ChangeSupport(this);
        unlisted = new ArrayList<RemoteServerRecord>();

        // Creates the "localhost" record and any remote records cached in remote.preferences

        // "localhost" record
        RemoteServerRecord localRecord = new RemoteServerRecord(ExecutionEnvironmentFactory.getLocal(), null, RemoteSyncFactory.getDefault(), false);
        localRecord.setState(RemoteServerRecord.State.ONLINE);
        items.add(localRecord);

        // now remote records cached in remote.preferences
        String slist = getPreferences().get(REMOTE_SERVERS, null);
        if (slist != null) {
            List<RemoteServerRecord> toAdd = RemoteServerRecord.fromString(slist);
            for (RemoteServerRecord record : toAdd) {
                record.setState(RemoteServerRecord.State.OFFLINE);
                items.add(record);
            }
        }
        defaultIndex = Math.min(defaultIndex, items.size() - 1);
        refresh();
        ConnectionManager.getInstance().addConnectionListener(WeakListeners.create(ConnectionListener.class, this, null));
    }

    @Override
    public void connected(ExecutionEnvironment env) {
        Collection<RemoteServerRecord> recordsToNotify = new ArrayList<RemoteServerRecord>();
        synchronized (this) {
            for (RemoteServerRecord rec : items) {
                if (rec.getExecutionEnvironment().equals(env)) {
                    recordsToNotify.add(rec);
                }
            }
        }
        // previously, it was done by RemoteFileSupport, but it is moved to dlight.remote
        if (recordsToNotify.isEmpty()) {
            // inlined RemoteServerListUI.revalidate
            ServerRecord record = get(env);
            if (record.isDeleted()) {
                addServer(record.getExecutionEnvironment(), record.getDisplayName(), record.getSyncFactory(), false, true);
            } else if (!record.isOnline()) {
                record.validate(true);
            }            
        } else {
            for (RemoteServerRecord rec : recordsToNotify) {
                rec.checkHostInfo();
            }
        }
    }

    @Override
    public void disconnected(ExecutionEnvironment env) {
    }

    /**
     * Get a ServerRecord pertaining to env. If needed, create the record.
     * 
     * @param env specifies the host
     * @return A RemoteServerRecord for env
     */
    @Override
    public synchronized ServerRecord get(ExecutionEnvironment env) {
        return get(env, true);
    }

    public synchronized RemoteServerRecord get(ExecutionEnvironment env, boolean create) {

        // Search the active server list
        for (RemoteServerRecord record : items) {
            if (env.equals(record.getExecutionEnvironment())) {
                return record;
            }
        }

        // Search the unlisted servers list. These are records created by Tools->Options
        // which haven't been added yet (and won't until/unless OK is pressed in T->O).
        for (RemoteServerRecord record : unlisted) {
            if (env.equals(record.getExecutionEnvironment())) {
                return record;
            }
        }

        if (create) {
            // Create a new unlisted record and return it
            RemoteServerRecord record = new RemoteServerRecord(env, null, RemoteSyncFactory.getDefault(), false);
            unlisted.add(record);
            return record;
        } else {
            return null;
        }
    }


    @org.netbeans.api.annotations.common.SuppressWarnings("UG") // since get(ExecutionEnvironment) is synchronized
    @Override
    public ServerRecord get(Project project) {
        ExecutionEnvironment execEnv = RemoteProjectSupport.getExecutionEnvironment(project);
        if( execEnv != null) {
            return get(execEnv);
        }
        return null;
    }

    @Override
    public synchronized ServerRecord getDefaultRecord() {
        return items.get(defaultIndex);
    }

    private synchronized void setDefaultIndexImpl(int defaultIndex) {
        int oldValue = this.defaultIndex;
        this.defaultIndex = defaultIndex;
        getPreferences().putInt(DEFAULT_INDEX, defaultIndex);
        firePropertyChange(ServerList.PROP_DEFAULT_RECORD, oldValue, defaultIndex);
    }

    @Override
    public synchronized void setDefaultRecord(ServerRecord record) {
        assert record != null;
        for (int i = 0; i < items.size(); i++) {
            if (items.get(i).equals(record)) {
                setDefaultIndexImpl(i);
                return;
            }
        }
        CndUtils.assertTrue(false, "Can not set nonexistent record as default");
    }

    @Override
    public synchronized List<ExecutionEnvironment> getEnvironments() {
        List<ExecutionEnvironment> result = new ArrayList<ExecutionEnvironment>(items.size());
        for (RemoteServerRecord item : items) {
            result.add(item.getExecutionEnvironment());
        }
        return result;
    }

    @Override
    public synchronized ServerRecord addServer(final ExecutionEnvironment execEnv, String displayName,
            RemoteSyncFactory syncFactory, boolean asDefault, boolean connect) {

        RemoteServerRecord record = null;
        if (syncFactory == null) {
            syncFactory = RemoteSyncFactory.getDefault();
        }

        // First off, check if we already have this record
        for (RemoteServerRecord r : items) {
            if (r.getExecutionEnvironment().equals(execEnv)) {
                if (asDefault) {
                    defaultIndex = items.indexOf(r);
                    getPreferences().putInt(DEFAULT_INDEX, defaultIndex);
                }
                return r;
            }
        }

        // Now see if its unlisted (created in Tools->Options but cancelled with no OK)
        for (RemoteServerRecord r : unlisted) {
            if (r.getExecutionEnvironment().equals(execEnv)) {
                record = r;
                break;
            }
        }

        if (record == null) {
            record = new RemoteServerRecord(execEnv, displayName, syncFactory, connect);
        } else {
            record.setDeleted(false);
            record.setDisplayName(displayName);
            record.setSyncFactory(syncFactory);
            unlisted.remove(record);
        }
        ArrayList<RemoteServerRecord> oldItems = new ArrayList<RemoteServerRecord>(items);
        items.add(record);
        Collections.sort(items, RECORDS_COMPARATOR);
        if (asDefault) {
            defaultIndex = items.indexOf(record);
        }
        refresh();
        storePreferences();
        getPreferences().putInt(DEFAULT_INDEX, defaultIndex);
        firePropertyChange(ServerList.PROP_RECORD_LIST, oldItems, new ArrayList<RemoteServerRecord>(items));
        return record;
    }

    public static RemoteServerList getInstance() {
        RemoteServerList instance = null;
        for (ServerListImplementation inst : Lookup.getDefault().lookupAll(ServerListImplementation.class)) {
            if (inst instanceof RemoteServerList) {
                instance = (RemoteServerList) inst;
                break;
            }
        }
        return instance;
    }

    public static void storePreferences() {
        RemoteServerList instance = getInstance();
        if (instance == null)  {
            RemoteUtil.LOGGER.warning("Can not find RemoteServerList instance");
            return;
        }
        List<RemoteServerRecord> records = new ArrayList<RemoteServerRecord>();
        synchronized (instance) {
            for (RemoteServerRecord record : instance.items) {
                if (record.isRemote()) {
                    records.add(record);
                }
            }
        }
        getPreferences().put(REMOTE_SERVERS, RemoteServerRecord.toString(records));
    }

    @Override
    public synchronized void set(List<ServerRecord> records, ServerRecord defaultRecord) {
        ArrayList<RemoteServerRecord> oldItems = new ArrayList<RemoteServerRecord>(items);
        RemoteUtil.LOGGER.log(Level.FINEST, "ServerList: set {0}", records);
        Collection<ExecutionEnvironment> removed = clear();
        List<ExecutionEnvironment> allEnv = new ArrayList<ExecutionEnvironment>();
        for (ServerRecord rec : records) {
            addServer(rec.getExecutionEnvironment(), rec.getDisplayName(), rec.getSyncFactory(), false, false);
            removed.remove(rec.getExecutionEnvironment());
            allEnv.add(rec.getExecutionEnvironment());
        }
        setDefaultRecord(defaultRecord);
        PasswordManager.getInstance().setServerList(allEnv);
        firePropertyChange(ServerList.PROP_RECORD_LIST, oldItems, new ArrayList<RemoteServerRecord>(items));
    }

    @Override
    public void save() {
        unlisted.clear();
    }

    private synchronized Collection<ExecutionEnvironment> clear() {
        Collection<ExecutionEnvironment> removed = new ArrayList<ExecutionEnvironment>();
        for (RemoteServerRecord record : items) {
            record.setDeleted(true);
            removed.add(record.getExecutionEnvironment());
        }
        getPreferences().remove(REMOTE_SERVERS);
        unlisted.addAll(items);
        items.clear();
        return removed;
    }

    private void refresh() {
        cs.fireChange();
    }

    public synchronized RemoteServerRecord getLocalhostRecord() {
        return items.get(0);
    }

    //TODO: why this is here?
    //TODO: deprecate and remove
    @Override
    public boolean isValidExecutable(ExecutionEnvironment env, String path) {
        if (path == null || path.length() == 0) {
            return false;
        }
        if (SwingUtilities.isEventDispatchThread()) {
            RemoteUtil.LOGGER.warning("RemoteServerList.isValidExecutable from EDT"); // NOI18N
        }
        int exit_status = RemoteCommandSupport.run(env, "test", "-x", path); // NOI18N
        if (exit_status != 0 && !CndPathUtilitities.isPathAbsolute(path)) {
            // Validate 'path' against user's PATH.
            exit_status = RemoteCommandSupport.run(env, "test", "-x", "`which " + path + "`"); // NOI18N
        }
        return exit_status == 0;
    }

    @Override
    public synchronized Collection<? extends ServerRecord> getRecords() {
        return new ArrayList<RemoteServerRecord>(items);
    }

    // TODO: Are these still needed?
    public void addChangeListener(ChangeListener listener) {
        cs.addChangeListener(listener);
    }

    public void removeChangeListener(ChangeListener listener) {
        cs.removeChangeListener(listener);
    }

    @Override
    public void addPropertyChangeListener(PropertyChangeListener listener) {
        pcs.addPropertyChangeListener(listener);
    }

    @Override
    public void removePropertyChangeListener(PropertyChangeListener listener) {
        pcs.removePropertyChangeListener(listener);
    }

    private void firePropertyChange(String property, Object oldValue, Object newValue) {
        pcs.firePropertyChange(property, oldValue, newValue);
    }

    private static Preferences getPreferences() {
        return NbPreferences.forModule(RemoteServerList.class);
    }

    private static final Comparator<RemoteServerRecord> RECORDS_COMPARATOR = new Comparator<RemoteServerRecord> () {
        @Override
        public int compare(RemoteServerRecord o1, RemoteServerRecord o2) {
            if (o1 == o2) {
                return 0;
            }

            // make localhosts first in the list
            boolean o1local = o1.getExecutionEnvironment().isLocal();
            boolean o2local = o2.getExecutionEnvironment().isLocal();
            if (o1local != o2local) {
                if (o1local) {
                    return -1;
                } else if (o2local) {
                    return 1;
                }
            }

            // others sort in alphabetical order
            return o1.getServerName().compareTo(o2.getServerName());
        }
    };

    @Override
    public ServerRecord createServerRecord(ExecutionEnvironment env, String displayName, RemoteSyncFactory syncFactory) {
        return new RemoteServerRecord(env, displayName, syncFactory, false);
    }
}
