/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Oracle and Java are registered trademarks of Oracle and/or its affiliates.
 * Other names may be trademarks of their respective owners.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.debugger.jpda;

import com.sun.jdi.AbsentInformationException;
import com.sun.jdi.Bootstrap;
import com.sun.jdi.ClassType;
import com.sun.jdi.IncompatibleThreadStateException;
import com.sun.jdi.IntegerValue;
import com.sun.jdi.InterfaceType;
import com.sun.jdi.InternalException;
import com.sun.jdi.InvalidStackFrameException;
import com.sun.jdi.LocalVariable;
import com.sun.jdi.Method;
import com.sun.jdi.ObjectCollectedException;
import com.sun.jdi.ObjectReference;
import com.sun.jdi.ReferenceType;
import com.sun.jdi.StackFrame;
import com.sun.jdi.StringReference;
import com.sun.jdi.ThreadGroupReference;
import com.sun.jdi.ThreadReference;
import com.sun.jdi.TypeComponent;
import com.sun.jdi.Value;
import com.sun.jdi.VirtualMachine;

import java.awt.EventQueue;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyVetoException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.security.auth.Refreshable;

import org.netbeans.api.debugger.DebuggerEngine;
import org.netbeans.api.debugger.DebuggerInfo;
import org.netbeans.api.debugger.DebuggerManager;
import org.netbeans.api.debugger.LazyActionsManagerListener;
import org.netbeans.api.debugger.Properties;

import org.netbeans.api.debugger.jpda.DeadlockDetector;
import org.netbeans.api.debugger.jpda.InvalidExpressionException;
import org.netbeans.api.debugger.jpda.JPDAClassType;
import org.netbeans.api.debugger.jpda.JPDAThreadGroup;
import org.netbeans.modules.debugger.jpda.actions.CompoundSmartSteppingListener;
import org.netbeans.modules.debugger.jpda.jdi.ClassNotPreparedExceptionWrapper;
import org.netbeans.modules.debugger.jpda.jdi.InternalExceptionWrapper;
import org.netbeans.modules.debugger.jpda.jdi.ObjectCollectedExceptionWrapper;
import org.netbeans.modules.debugger.jpda.jdi.VMDisconnectedExceptionWrapper;
import org.netbeans.modules.debugger.jpda.models.ObjectTranslation;
import org.netbeans.modules.debugger.jpda.util.JPDAUtils;
import org.netbeans.spi.debugger.ContextProvider;
import org.netbeans.api.debugger.Session;
import org.netbeans.api.debugger.jpda.AbstractDICookie;
import org.netbeans.api.debugger.jpda.AttachingDICookie;
import org.netbeans.api.debugger.jpda.CallStackFrame;
import org.netbeans.api.debugger.jpda.DebuggerStartException;
import org.netbeans.api.debugger.jpda.JPDABreakpoint;
import org.netbeans.api.debugger.jpda.event.JPDABreakpointEvent;
import org.netbeans.api.debugger.jpda.JPDADebugger;
import org.netbeans.api.debugger.jpda.JPDAThread;
import org.netbeans.api.debugger.jpda.SmartSteppingFilter;
import org.netbeans.api.debugger.jpda.Variable;
import org.netbeans.api.debugger.jpda.JPDAStep;
import org.netbeans.api.debugger.jpda.ListeningDICookie;

import org.netbeans.api.debugger.jpda.ObjectVariable;
import org.netbeans.modules.debugger.jpda.breakpoints.BreakpointsEngineListener;
import org.netbeans.modules.debugger.jpda.expr.EvaluatorExpression;
import org.netbeans.modules.debugger.jpda.models.JPDAThreadImpl;
import org.netbeans.modules.debugger.jpda.models.LocalsTreeModel;
import org.netbeans.modules.debugger.jpda.models.CallStackFrameImpl;
import org.netbeans.modules.debugger.jpda.models.JPDAClassTypeImpl;
import org.netbeans.modules.debugger.jpda.models.ThreadsCache;
import org.netbeans.modules.debugger.jpda.util.Operator;
import org.netbeans.modules.debugger.jpda.expr.JDIVariable;
import org.netbeans.modules.debugger.jpda.jdi.ClassTypeWrapper;
import org.netbeans.modules.debugger.jpda.jdi.IllegalThreadStateExceptionWrapper;
import org.netbeans.modules.debugger.jpda.jdi.IntegerValueWrapper;
import org.netbeans.modules.debugger.jpda.jdi.InvalidStackFrameExceptionWrapper;
import org.netbeans.modules.debugger.jpda.jdi.MirrorWrapper;
import org.netbeans.modules.debugger.jpda.jdi.StackFrameWrapper;
import org.netbeans.modules.debugger.jpda.jdi.ThreadReferenceWrapper;
import org.netbeans.modules.debugger.jpda.jdi.ValueWrapper;
import org.netbeans.modules.debugger.jpda.jdi.VirtualMachineWrapper;
import org.netbeans.spi.debugger.DebuggerEngineProvider;
import org.netbeans.spi.debugger.DelegatingSessionProvider;

import org.netbeans.spi.debugger.jpda.Evaluator;
import org.netbeans.spi.viewmodel.TreeModel;
import org.openide.ErrorManager;
import org.openide.util.Exceptions;
import org.openide.util.Lookup;
import org.openide.util.NbBundle;
import org.openide.util.RequestProcessor;
import org.openide.util.WeakSet;
import org.openide.util.lookup.Lookups;

/**
* Representation of a debugging session.
*
* @author   Jan Jancura
*/
@JPDADebugger.Registration(path="netbeans-JPDASession")
public class JPDADebuggerImpl extends JPDADebugger {

    private static final Logger logger = Logger.getLogger("org.netbeans.modules.debugger.jpda");

    private static final boolean SINGLE_THREAD_STEPPING = !Boolean.getBoolean("netbeans.debugger.multiThreadStepping");


    // variables ...............................................................

    //private DebuggerEngine              debuggerEngine;
    private VirtualMachine              virtualMachine = null;
    private final Object                virtualMachineLock = new Object();
    private Throwable                   throwable;
    private int                         state = 0;
    private final Object                stateLock = new Object();
    private Operator                    operator;
    private PropertyChangeSupport       pcs;
    public  PropertyChangeSupport       varChangeSupport = new PropertyChangeSupport(this);
    private JPDAThreadImpl              currentThread;
    private CallStackFrame              currentCallStackFrame;
    private final Object                currentThreadAndFrameLock = new Object();
    private int                         suspend = (SINGLE_THREAD_STEPPING) ? SUSPEND_EVENT_THREAD : SUSPEND_ALL;
    public final ReentrantReadWriteLock accessLock = new DebuggerReentrantReadWriteLock(true);
    private final Object                LOCK2 = new Object ();
    private boolean                     starting;
    private AbstractDICookie            attachingCookie;
    private JavaEngineProvider          javaEngineProvider;
    private Set<String>                 languages;
    private String                      lastStratumn;
    private ContextProvider             lookupProvider;
    private ObjectTranslation           threadsTranslation;
    private ObjectTranslation           localsTranslation;
    private ExpressionPool              expressionPool;
    private ThreadsCache                threadsCache;
    private DeadlockDetector            deadlockDetector;
    private ThreadsCollectorImpl        threadsCollector;
    private final Object                threadsCollectorLock = new Object();
    private final Map<Long, String>     markedObjects = new LinkedHashMap<Long, String>();
    private final Map<String, ObjectVariable> markedObjectLabels = new LinkedHashMap<String, ObjectVariable>();
    private final Map<ClassType, List>  allInterfacesMap = new WeakHashMap<ClassType, List>();

    private StackFrame      altCSF = null;  //PATCH 48174

    private boolean                     doContinue = true; // Whether resume() will actually resume
    private Boolean                     singleThreadStepResumeDecision = null;
    private Boolean                     stepInterruptByBptResumeDecision = null;

    // init ....................................................................

    public JPDADebuggerImpl (ContextProvider lookupProvider) {
        this.lookupProvider = lookupProvider;
        
        Properties p = Properties.getDefault().getProperties("debugger.options.JPDA");
        int stepResume = p.getInt("StepResume", (SINGLE_THREAD_STEPPING) ? 1 : 0);
        suspend = (stepResume == 1) ? SUSPEND_EVENT_THREAD : SUSPEND_ALL;

        pcs = new PropertyChangeSupport (this);
        List l = lookupProvider.lookup (null, DebuggerEngineProvider.class);
        int i, k = l.size ();
        for (i = 0; i < k; i++)
            if (l.get (i) instanceof JavaEngineProvider)
                javaEngineProvider = (JavaEngineProvider) l.get (i);
        if (javaEngineProvider == null)
            throw new IllegalArgumentException
                ("JavaEngineProvider have to be used to start JPDADebugger!");
        languages = new HashSet<String>();
        languages.add ("Java");
        threadsTranslation = ObjectTranslation.createThreadTranslation(this);
        localsTranslation = ObjectTranslation.createLocalsTranslation(this);
        this.expressionPool = new ExpressionPool();
    }


    // JPDADebugger methods ....................................................

    /**
     * Returns current state of JPDA debugger.
     *
     * @return current state of JPDA debugger
     * @see #STATE_STARTING
     * @see #STATE_RUNNING
     * @see #STATE_STOPPED
     * @see #STATE_DISCONNECTED
     */
    @Override
    public int getState () {
        synchronized (stateLock) {
            return state;
        }
    }

    /**
     * Gets value of suspend property.
     *
     * @return value of suspend property
     */
    @Override
    public int getSuspend () {
        synchronized (stateLock) {
            return suspend;
        }
    }

    /**
     * Sets value of suspend property.
     *
     * @param s a new value of suspend property
     */
    @Override
    public void setSuspend (int s) {
        int old;
        synchronized (stateLock) {
            if (s == suspend) return;
            old = suspend;
            suspend = s;
        }
        firePropertyChange (PROP_SUSPEND, Integer.valueOf(old), Integer.valueOf(s));
    }

    /**
     * Returns current thread or null.
     *
     * @return current thread or null
     */
    @Override
    public JPDAThread getCurrentThread () {
        return currentThread;
    }

    /**
     * Returns current stack frame or null.
     *
     * @return current stack frame or null
     */
    @Override
    public CallStackFrame getCurrentCallStackFrame () {
        synchronized (currentThreadAndFrameLock) {
            return currentCallStackFrame;
        }
    }

    /**
     * Evaluates given expression in the current context.
     *
     * @param expression a expression to be evaluated
     *
     * @return current value of given expression
     */
    @Override
    public Variable evaluate (String expression) throws InvalidExpressionException {
        return evaluate(expression, null, null);
    }

    /**
     * Evaluates given expression in the context of the variable.
     *
     * @param expression a expression to be evaluated
     *
     * @return current value of given expression
     */
    public Variable evaluate (String expression, ObjectVariable var)
    throws InvalidExpressionException {
        return evaluate(expression, null, var);
    }

    /**
     * Evaluates given expression in the current context.
     *
     * @param expression a expression to be evaluated
     *
     * @return current value of given expression
     */
    public Variable evaluate (String expression, CallStackFrame csf)
    throws InvalidExpressionException {
        return evaluate(expression, csf, null);
    }

    /**
     * Evaluates given expression in the current context.
     *
     * @param expression a expression to be evaluated
     *
     * @return current value of given expression
     */
    public Variable evaluate (String expression, CallStackFrame csf, ObjectVariable var)
    throws InvalidExpressionException {
        return evaluateGeneric(new EvaluatorExpression(expression), csf, var);
    }

    /**
     * Waits till the Virtual Machine is started and returns
     * {@link DebuggerStartException} if any.
     *
     * @throws DebuggerStartException if some problems occurs during debugger
     *         start
     *
     * @see AbstractDICookie#getVirtualMachine()
     */
    @Override
    public void waitRunning () throws DebuggerStartException {
        waitRunning(0);
    }

    /**
     * Waits till the Virtual Machine is started.
     *
     * @return <code>true</code> when debugger started or finished in the mean time,
     *         <code>false</code> when the debugger is still starting
     * @throws DebuggerStartException if some problems occurs during debugger
     *         start
     *
     * @see AbstractDICookie#getVirtualMachine()
     */
    private boolean waitRunning(long timeout) throws DebuggerStartException {
        synchronized (LOCK2) {
            int state = getState();
            if (state == STATE_DISCONNECTED) {
                if (throwable != null)
                    throw new DebuggerStartException (throwable);
                else
                    return true;
            }
            if (!starting && state != STATE_STARTING || throwable != null) {
                return true; // We're already running
            }
            try {
                logger.fine("JPDADebuggerImpl.waitRunning(): starting = "+starting+", state = "+state+", exception = "+throwable);
                LOCK2.wait (timeout);
                if ((starting || state == STATE_STARTING) && throwable == null) {
                    return false; // We're still not running and the time expired
                }
            } catch (InterruptedException e) {
                 throw new DebuggerStartException (e);
            }

            if (throwable != null)
                throw new DebuggerStartException (throwable);
            else
                return true;
        }
    }

    /**
     * Returns <code>true</code> if this debugger supports Pop action.
     *
     * @return <code>true</code> if this debugger supports Pop action
     */
    @Override
    public boolean canPopFrames () {
        VirtualMachine vm = getVirtualMachine ();
        if (vm == null) return false;
        return VirtualMachineWrapper.canPopFrames0(vm);
    }

    /**
     * Returns <code>true</code> if this debugger supports fix & continue
     * (HotSwap).
     *
     * @return <code>true</code> if this debugger supports fix & continue
     */
    @Override
    public boolean canFixClasses () {
        VirtualMachine vm = getVirtualMachine ();
        if (vm == null) return false;
        return VirtualMachineWrapper.canRedefineClasses0(vm);
    }

    /**
     * Implements fix & continue (HotSwap). Map should contain class names
     * as a keys, and byte[] arrays as a values.
     *
     * @param classes a map from class names to be fixed to byte[]
     */
    @Override
    public void fixClasses (Map<String, byte[]> classes) {
        PropertyChangeEvent evt = null;
        accessLock.writeLock().lock();
        try {
            // 1) redefine classes
            Map<ReferenceType, byte[]> map = new HashMap<ReferenceType, byte[]>();
            Iterator<Map.Entry<String, byte[]>> e = classes.entrySet ().iterator ();
            VirtualMachine vm = getVirtualMachine();
            if (vm == null) {
                return ; // The session has finished
            }
            try {
                while (e.hasNext ()) {
                    Map.Entry<String, byte[]> classEntry = e.next();
                    String className = classEntry.getKey();
                    byte[] bytes = classEntry.getValue();
                    List<ReferenceType> classRefs = VirtualMachineWrapper.classesByName (vm, className);
                    int j, jj = classRefs.size ();
                    for (j = 0; j < jj; j++)
                        map.put (
                            classRefs.get (j),
                            bytes
                        );
                }
                VirtualMachineWrapper.redefineClasses (vm, map);
            } catch (InternalExceptionWrapper ex) {
            } catch (VMDisconnectedExceptionWrapper ex) {
            }

            // update breakpoints
            fixBreakpoints();

            firePropertyChange(JPDADebugger.PROP_CLASSES_FIXED, null, null);

            // 2) pop obsoleted frames
            JPDAThread t = getCurrentThread ();
            if (t != null && t.isSuspended()) {
                try {
                    if (t.getStackDepth () < 2) return;
                    CallStackFrame frame;
                    try {
                        CallStackFrame[] frames = t.getCallStack(0, 1);
                        if (frames.length == 0) return ;
                        frame = frames[0]; // Retrieve the new, possibly obsoleted frame and check it.
                    } catch (AbsentInformationException ex) {
                        return;
                    }

                    //PATCH #52209
                    if (frame.isObsolete () && ((CallStackFrameImpl) frame).canPop()) {
                        frame.popFrame ();
                        setState (STATE_RUNNING);
                        evt = updateCurrentCallStackFrameNoFire (t);
                        setState (STATE_STOPPED);
                    }
                } catch (InvalidStackFrameException ex) {
                }
            }

        } finally {
            accessLock.writeLock().unlock();
        }
        if (evt != null) {
            firePropertyChange(evt);
        }
    }

    public void fixBreakpoints() {
        // Just reset the time stamp so that new line numbers are taken.
        EditorContextBridge.getContext().disposeTimeStamp(this);
        EditorContextBridge.getContext().createTimeStamp(this);
        Session s = getSession();
        DebuggerEngine de = s.getEngineForLanguage ("Java");
        if (de == null) {
            de = s.getCurrentEngine();
        }
        if (de != null) {
            BreakpointsEngineListener bel = null;
            List lazyListeners = de.lookup(null, LazyActionsManagerListener.class);
            for (int li = 0; li < lazyListeners.size(); li++) {
                Object service = lazyListeners.get(li);
                if (service instanceof BreakpointsEngineListener) {
                    bel = (BreakpointsEngineListener) service;
                    break;
                }
            }
            if (bel != null) bel.fixBreakpointImpls ();
        }
    }

    public Session getSession() {
        return lookupProvider.lookupFirst(null, Session.class);
    }

    public RequestProcessor getRequestProcessor() {
        return javaEngineProvider.getRequestProcessor();
    }

    private Boolean canBeModified;
    private final Object canBeModifiedLock = new Object();

    @Override
    public boolean canBeModified() {
        VirtualMachine vm = getVirtualMachine ();
        if (vm == null) return false;
        synchronized (canBeModifiedLock) {
            if (canBeModified == null) {
                canBeModified = vm.canBeModified();
            }
            return canBeModified.booleanValue();
        }
    }

    private SmartSteppingFilter smartSteppingFilter;

    /**
     * Returns instance of SmartSteppingFilter.
     *
     * @return instance of SmartSteppingFilter
     */
    @Override
    public SmartSteppingFilter getSmartSteppingFilter () {
        if (smartSteppingFilter == null) {
            smartSteppingFilter = lookupProvider.lookupFirst(null, SmartSteppingFilter.class);
        }
        return smartSteppingFilter;
    }

    CompoundSmartSteppingListener compoundSmartSteppingListener;

    private CompoundSmartSteppingListener getCompoundSmartSteppingListener () {
        if (compoundSmartSteppingListener == null)
            compoundSmartSteppingListener = lookupProvider.lookupFirst(null, CompoundSmartSteppingListener.class);
        return compoundSmartSteppingListener;
    }

    /**
     * Test whether we should stop here according to the smart-stepping rules.
     */
    boolean stopHere(JPDAThread t) {
        return getCompoundSmartSteppingListener ().stopHere
                     (lookupProvider, t, getSmartSteppingFilter());
    }

    /**
     * Helper method that fires JPDABreakpointEvent on JPDABreakpoints.
     *
     * @param breakpoint a breakpoint to be changed
     * @param event a event to be fired
     */
    @Override
    public void fireBreakpointEvent (
        JPDABreakpoint breakpoint,
        JPDABreakpointEvent event
    ) {
        super.fireBreakpointEvent (breakpoint, event);
    }

    /**
    * Adds property change listener.
    *
    * @param l new listener.
    */
    @Override
    public void addPropertyChangeListener (PropertyChangeListener l) {
        pcs.addPropertyChangeListener (l);
    }

    /**
    * Removes property change listener.
    *
    * @param l removed listener.
    */
    @Override
    public void removePropertyChangeListener (PropertyChangeListener l) {
        pcs.removePropertyChangeListener (l);
    }

    /**
    * Adds property change listener.
    *
    * @param l new listener.
    */
    @Override
    public void addPropertyChangeListener (String propertyName, PropertyChangeListener l) {
        pcs.addPropertyChangeListener (propertyName, l);
    }

    /**
    * Removes property change listener.
    *
    * @param l removed listener.
    */
    @Override
    public void removePropertyChangeListener (String propertyName, PropertyChangeListener l) {
        pcs.removePropertyChangeListener (propertyName, l);
    }


    // internal interface ......................................................

    public void popFrames (ThreadReference thread, StackFrame frame) {
        PropertyChangeEvent evt = null;
        accessLock.readLock().lock();
        try {
            JPDAThreadImpl threadImpl = getThread(thread);
            setState (STATE_RUNNING);
            try {
                threadImpl.popFrames(frame);
                evt = updateCurrentCallStackFrameNoFire(threadImpl);
            } catch (IncompatibleThreadStateException ex) {
                ErrorManager.getDefault().notify(ex);
            } finally {
                setState (STATE_STOPPED);
            }
        } finally {
            accessLock.readLock().unlock();
        }
        if (evt != null) {
            firePropertyChange(evt);
        }
    }

    public void setException (Throwable t) {
        if (logger.isLoggable(Level.FINE)) {
            logger.fine("JPDADebuggerImpl.setException("+t+")");
        }
        synchronized (LOCK2) {
            throwable = t;
            starting = false;
            LOCK2.notifyAll ();
        }
    }

    public void setCurrentThread (JPDAThread thread) {
        Object oldT;
        PropertyChangeEvent event;
        CallStackFrame topFrame = getTopFrame(thread);
        synchronized (currentThreadAndFrameLock) {
            oldT = currentThread;
            currentThread = (JPDAThreadImpl) thread;
            event = updateCurrentCallStackFrameNoFire(topFrame);
        }
        if (thread != oldT) {
            firePropertyChange (PROP_CURRENT_THREAD, oldT, thread);
        }
        if (event != null) {
            firePropertyChange(event);
        }
        setState(thread.isSuspended() ? STATE_STOPPED : STATE_RUNNING);
    }

    /**
     * Set the current thread and call stack, but do not fire changes.
     * @return The PropertyChangeEvent associated with this change, it can have
     *         attached other PropertyChangeEvents as a propagation ID.
     */
    private PropertyChangeEvent setCurrentThreadNoFire(JPDAThread thread) {
        Object oldT;
        PropertyChangeEvent evt = null;
        PropertyChangeEvent evt2;
        CallStackFrame topFrame = getTopFrame(thread);
        synchronized (currentThreadAndFrameLock) {
            oldT = currentThread;
            currentThread = (JPDAThreadImpl) thread;
            evt2 = updateCurrentCallStackFrameNoFire(topFrame);
        }
        if (thread != oldT) {
            evt = new PropertyChangeEvent(this, PROP_CURRENT_THREAD, oldT, thread);
        }
        if (evt == null) evt = evt2;
        else if (evt2 != null) evt.setPropagationId(evt2);
        return evt;
    }

    public void setCurrentCallStackFrame (CallStackFrame callStackFrame) {
        CallStackFrame old = setCurrentCallStackFrameNoFire(callStackFrame);
        if (old == callStackFrame) return ;
        firePropertyChange (
            PROP_CURRENT_CALL_STACK_FRAME,
            old,
            callStackFrame
        );
    }

    private CallStackFrame setCurrentCallStackFrameNoFire (CallStackFrame callStackFrame) {
        CallStackFrame old;
        synchronized (currentThreadAndFrameLock) {
            if (callStackFrame == currentCallStackFrame) return callStackFrame;
            old = currentCallStackFrame;
            currentCallStackFrame = callStackFrame;
        }
        return old;
    }

    /**
     * Used by AbstractVariable and watches.
     */
    // * Might be changed to return a variable with disabled collection. When not used any more,
    // * it's collection must be enabled again.
    public Value evaluateIn (String expression) throws InvalidExpressionException {
        return evaluateIn(new EvaluatorExpression(expression), null);
    }

    /**
     * Used by BreakpointImpl.
     */
    // * Might be changed to return a variable with disabled collection. When not used any more,
    // * it's collection must be enabled again.
    public Value evaluateIn (EvaluatorExpression expression, CallStackFrame csf) throws InvalidExpressionException {
        return evaluateIn(expression, csf, null);
    }

    // * Might be changed to return a variable with disabled collection. When not used any more,
    // * it's collection must be enabled again.
    public Value evaluateIn (EvaluatorExpression expression, CallStackFrame csf, ObjectVariable var) throws InvalidExpressionException {
        Variable variable = evaluateGeneric(expression, csf, var);
        if (variable instanceof JDIVariable) {
            return ((JDIVariable) variable).getJDIValue();
        } else {
            return null;
        }
    }

    //PATCH 48174
    public void setAltCSF(StackFrame sf) {
        altCSF = sf;
    }

    public StackFrame getAltCSF() {
        return altCSF;
    }

    /**
     * Used by WatchesModel & BreakpointImpl.
     */
    // * Might be changed to return a variable with disabled collection. When not used any more,
    // * it's collection must be enabled again.
    public Value evaluateIn (EvaluatorExpression expression) throws InvalidExpressionException {
        return evaluateIn(expression, null, null);
    }


    InvalidExpressionException methodCallsUnsupportedExc;

    /**
     * Evaluates given expression in the current context.
     *
     * @param expression a expression to be evaluated
     *
     * @return current value of given expression
     */
    private Variable evaluateGeneric(EvaluatorExpression expression, CallStackFrame c, ObjectVariable var)
            throws InvalidExpressionException {

        Session s = getSession();
        Evaluator e = s.lookupFirst(s.getCurrentLanguage(), Evaluator.class);
        logger.fine("Have evaluator "+e+" for language "+s.getCurrentLanguage());   // NOI18N

        if (e == null) {
            e = new JavaEvaluator(s);
        }

        CallStackFrameImpl csf;
        if (c instanceof CallStackFrameImpl) {
            csf = (CallStackFrameImpl) c;
        } else {
            csf = (CallStackFrameImpl) getCurrentCallStackFrame ();
        }
        if (csf == null) {
            StackFrame sf = getAltCSF();
            if (sf != null) {
                try {
                    ThreadReference t = StackFrameWrapper.thread(sf);
                    JPDAThread tr = getThread(t);
                    CallStackFrame[] stackFrames = tr.getCallStack(0, 1);
                    if (stackFrames.length > 0) {
                        c = stackFrames[0];
                        csf = (CallStackFrameImpl) c;
                    }
                } catch (InternalExceptionWrapper ex) {
                } catch (InvalidStackFrameExceptionWrapper ex) {
                } catch (VMDisconnectedExceptionWrapper ex) {
                } catch (AbsentInformationException aiex) {
                }
            }
        }
        if (csf == null) {
            throw new InvalidExpressionException
                (NbBundle.getMessage(JPDADebuggerImpl.class, "MSG_NoCurrentContextStackFrame"));
        }

        JPDAThread frameThread = csf.getThread();
        JPDAThreadImpl frameThreadImpl = (JPDAThreadImpl) frameThread;
        //Object pendingAction = frameThreadImpl.getPendingAction();
        //if (pendingAction != null) { For the case that evaluation should be blocked by pending action
        //    //return frameThreadImpl.getPendingVariable(pendingActions);
        //    throw new InvalidExpressionException(frameThreadImpl.getPendingString(pendingAction));
        //}
        Lock lock = frameThreadImpl.accessLock.writeLock();
        lock.lock();
        Variable vr;
        try {
            if (!frameThread.isSuspended() && !((JPDAThreadImpl) frameThread).isSuspendedNoFire()) {
                // Thread not suspended => Can not start evaluation
                throw new InvalidExpressionException
                    (NbBundle.getMessage(JPDADebuggerImpl.class, "MSG_NoCurrentContextStackFrame"));
            }
            ObjectReference v = null;
            if (var instanceof JDIVariable) {
                v = (ObjectReference) ((JDIVariable) var).getJDIValue();
            }
            Evaluator.Result result;
            final JPDAThreadImpl[] resumedThread = new JPDAThreadImpl[] { null };
            try {
                StackFrame sf = csf.getStackFrame();
                int stackDepth = csf.getFrameDepth();
                final ThreadReference tr = StackFrameWrapper.thread(sf);
                Runnable methodToBeInvokedNotifier = new Runnable() {
                        @Override
                        public void run() {
                            if (resumedThread[0] == null) {
                                JPDAThreadImpl theResumedThread = getThread(tr);
                                try {
                                    theResumedThread.notifyMethodInvoking();
                                } catch (PropertyVetoException pvex) {
                                    throw new RuntimeException(
                                        new InvalidExpressionException (pvex.getMessage()));
                                } catch (ThreadDeath td) {
                                    throw td;
                                } catch (Throwable t) {
                                    Exceptions.printStackTrace(t);
                                }
                                resumedThread[0] = theResumedThread;
                            }
                        }
                    };
                Lookup context;
                if (var != null) {
                    if (v != null) {
                        context = Lookups.fixed(csf, var, sf, stackDepth, v, methodToBeInvokedNotifier);
                    } else {
                        context = Lookups.fixed(csf, var, sf, stackDepth, methodToBeInvokedNotifier);
                    }
                } else {
                    context = Lookups.fixed(csf, sf, stackDepth, methodToBeInvokedNotifier);
                }
                result = expression.evaluate(e, new Evaluator.Context(context));
            } catch (InternalExceptionWrapper ex) {
                return null;
            } catch (InvalidStackFrameExceptionWrapper ex) {
                throw new InvalidExpressionException
                    (NbBundle.getMessage(JPDADebuggerImpl.class, "MSG_NoCurrentContextStackFrame"));
            } catch (VMDisconnectedExceptionWrapper ex) {
                // Causes kill action when something is being evaluated.
                return null;
            } catch (InternalException ex) {
                InvalidExpressionException isex = new InvalidExpressionException(ex.getLocalizedMessage());
                isex.initCause(ex);
                Exceptions.attachMessage(isex, "Expression = '"+expression+"'");
                throw isex;
            } catch (RuntimeException rex) {
                Throwable cause = rex.getCause();
                if (cause instanceof InvalidExpressionException) {
                    Exceptions.attachMessage(cause, "Expression = '"+expression+"'");
                    throw (InvalidExpressionException) cause;
                } else {
                    throw rex;
                }
            } finally {
                if (resumedThread[0] != null) {
                    resumedThread[0].notifyMethodInvokeDone();
                }
            }
            vr = result.getVariable();
            if (vr == null) {
                vr = getLocalsTreeModel ().getVariable (result.getValue());
            }
        } finally {
            lock.unlock();
        }
        return vr;
    }

    /**
     * Used by AbstractVariable.
     */
    public Value invokeMethod (
        ObjectReference reference,
        Method method,
        Value[] arguments
    ) throws InvalidExpressionException {
        return invokeMethod(null, reference, method, arguments, 0);
    }

    public Value invokeMethod (
        ObjectReference reference,
        Method method,
        Value[] arguments,
        int maxLength
    ) throws InvalidExpressionException {
        return invokeMethod(null, reference, method, arguments, maxLength);
    }

    /**
     * Used by AbstractVariable.
     */
    public Value invokeMethod (
        JPDAThreadImpl thread,
        ObjectReference reference,
        Method method,
        Value[] arguments
    ) throws InvalidExpressionException {
        return invokeMethod(thread, reference, method, arguments, 0);
    }

    private Value invokeMethod (
        JPDAThreadImpl thread,
        ObjectReference reference,
        Method method,
        Value[] arguments,
        int maxLength
    ) throws InvalidExpressionException {
        synchronized (currentThreadAndFrameLock) {
            if (thread == null && currentThread == null)
                throw new InvalidExpressionException
                        (NbBundle.getMessage(JPDADebuggerImpl.class, "MSG_NoCurrentContext"));
            if (thread == null) {
                thread = currentThread;
            }
        }
        thread.accessLock.writeLock().lock();
        try {
            if (methodCallsUnsupportedExc != null) {
                throw methodCallsUnsupportedExc;
            }
            boolean threadSuspended = false;
            JPDAThread frameThread = null;
            CallStackFrameImpl csf = null;
            try {
                // Remember the current stack frame, it might be necessary to re-set.
                csf = (CallStackFrameImpl) getCurrentCallStackFrame ();
                if (csf != null) {
                    try {
                        frameThread = csf.getThread();
                    } catch (InvalidStackFrameException isfex) {}
                }
                ThreadReference tr = thread.getThreadReference();
                try {
                    thread.notifyMethodInvoking();
                    threadSuspended = true;
                } catch (PropertyVetoException pvex) {
                    throw new InvalidExpressionException (pvex.getMessage());
                } catch (RuntimeException rex) {
                    // Give up
                    thread.notifyMethodInvokeDone();
                    throw rex;
                } catch (ThreadDeath td) {
                    throw td;
                } catch (Error e) {
                    // Give up
                    thread.notifyMethodInvokeDone();
                    throw e;
                }
                try {
                    Value v = org.netbeans.modules.debugger.jpda.expr.TreeEvaluator.
                        invokeVirtual (
                            reference,
                            method,
                            tr,
                            Arrays.asList (arguments),
                            this
                        );
                    if (maxLength > 0 && maxLength < Integer.MAX_VALUE && (v instanceof StringReference)) {
                        v = cutLength((StringReference) v, maxLength, tr);
                    }
                    return v;
                } catch (InternalException e) {
                    InvalidExpressionException ieex = new InvalidExpressionException (e.getLocalizedMessage());
                    ieex.initCause(e);
                    throw ieex;
                }
            } catch (InvalidExpressionException ieex) {
                if (ieex.getTargetException() instanceof UnsupportedOperationException) {
                    methodCallsUnsupportedExc = ieex;
                }
                throw ieex;
            } finally {
                if (threadSuspended) {
                    thread.notifyMethodInvokeDone();
                }
                if (frameThread != null) {
                    try {
                        csf.getThread();
                    } catch (InvalidStackFrameException isfex) {
                        // The current frame is invalidated, set the new current...
                        int depth = csf.getFrameDepth();
                        try {
                            CallStackFrame csf2 = frameThread.getCallStack(depth, depth + 1)[0];
                            setCurrentCallStackFrameNoFire(csf2);
                        } catch (AbsentInformationException aiex) {
                            setCurrentCallStackFrame(null);
                        }
                    }
                }
            }
        } finally {
            thread.accessLock.writeLock().unlock();
        }
    }

    private Value cutLength(StringReference sr, int maxLength, ThreadReference tr) throws InvalidExpressionException {
        try {
            Method stringLengthMethod;
                stringLengthMethod = ClassTypeWrapper.concreteMethodByName(
                        (ClassType) ValueWrapper.type (sr), "length", "()I"); // NOI18N
            List<Value> emptyArgs = Collections.emptyList();
            IntegerValue lengthValue = (IntegerValue) org.netbeans.modules.debugger.jpda.expr.TreeEvaluator.
                invokeVirtual (
                    sr,
                    stringLengthMethod,
                    tr,
                    emptyArgs,
                    this
                );
                if (IntegerValueWrapper.value(lengthValue) > maxLength) {
                    Method subStringMethod = ClassTypeWrapper.concreteMethodByName(
                            (ClassType) ValueWrapper.type (sr), "substring", "(II)Ljava/lang/String;");  // NOI18N
                    if (subStringMethod != null) {
                        sr = (StringReference) org.netbeans.modules.debugger.jpda.expr.TreeEvaluator.
                            invokeVirtual (
                                sr,
                                subStringMethod,
                                tr,
                                Arrays.asList(new Value [] { VirtualMachineWrapper.mirrorOf(MirrorWrapper.virtualMachine(sr), 0),
                                               VirtualMachineWrapper.mirrorOf(MirrorWrapper.virtualMachine(sr), maxLength) }),
                                this
                            );
                    }
                }
        } catch (InternalExceptionWrapper ex) {
        } catch (ObjectCollectedExceptionWrapper ex) {
        } catch (VMDisconnectedExceptionWrapper ex) {
        } catch (ClassNotPreparedExceptionWrapper ex) {
        }
        return sr;
    }

    public static String getGenericSignature (TypeComponent component) {
        if (tcGenericSignatureMethod == null) return null;
        try {
            return (String) tcGenericSignatureMethod.invoke
                (component, new Object[0]);
        } catch (IllegalAccessException e) {
            Exceptions.printStackTrace(e);
            return null;    // should not happen
        } catch (InvocationTargetException e) {
            Exceptions.printStackTrace(e);
            return null;    // should not happen
        }
    }

    public static String getGenericSignature (LocalVariable component) {
        if (lvGenericSignatureMethod == null) return null;
        try {
            return (String) lvGenericSignatureMethod.invoke(component, new Object[0]);
        } catch (IllegalAccessException e) {
            Exceptions.printStackTrace(e);
            return null;    // should not happen
        } catch (InvocationTargetException e) {
            Exceptions.printStackTrace(e);
            return null;    // should not happen
        }
    }

    public VirtualMachine getVirtualMachine () {
        synchronized (virtualMachineLock) {
            return virtualMachine;
        }
    }

    public Operator getOperator () {
        synchronized (virtualMachineLock) {
            return operator;
        }
    }

    public void setStarting () {
        setState (STATE_STARTING);
    }

    public synchronized void setAttaching(AbstractDICookie cookie) {
        this.attachingCookie = cookie;
    }

    public void setRunning (VirtualMachine vm, Operator o) {
        if (logger.isLoggable(Level.FINE)) {
            logger.fine("Start - JPDADebuggerImpl.setRunning ()");
            JPDAUtils.printFeatures (logger, vm);
        }
        synchronized (LOCK2) {
            starting = true;
        }
        synchronized (virtualMachineLock) {
            virtualMachine = vm;
            operator = o;
        }
        synchronized (canBeModifiedLock) {
            canBeModified = null; // Reset the can be modified flag
        }

        initGenericsSupport ();
        EditorContextBridge.getContext().createTimeStamp(this);


//        Iterator i = getVirtualMachine ().allThreads ().iterator ();
//        while (i.hasNext ()) {
//            ThreadReference tr = (ThreadReference) i.next ();
//            if (tr.isSuspended ()) {
//                if (startVerbose)
//                    System.out.println("\nS JPDADebuggerImpl.setRunning () - " +
//                        "thread supended"
//                    );
//                setState (STATE_RUNNING);
//                synchronized (LOCK) {
//                    virtualMachine.resume ();
//                }
//                if (startVerbose)
//                    System.out.println("\nS JPDADebuggerImpl.setRunning () - " +
//                        "thread supended - VM resumed - end"
//                    );
//                synchronized (LOCK2) {
//                    LOCK2.notify ();
//                }
//                return;
//            }
//        }

        synchronized (threadsCollectorLock) {
            if (threadsCache != null) {
                threadsCache.setVirtualMachine(vm);
            }
        }

        setState (STATE_RUNNING);
        synchronized (virtualMachineLock) {
            vm = virtualMachine; // re-take the VM, it can be nulled by finish()
        }
        if (vm != null) {
            notifyToBeResumedAll();
            accessLock.writeLock().lock();
            try {
                VirtualMachineWrapper.resume(vm);
            } catch (VMDisconnectedExceptionWrapper e) {
            } catch (InternalExceptionWrapper e) {
            } finally {
                accessLock.writeLock().unlock();
            }
        }

        logger.fine("   JPDADebuggerImpl.setRunning () finished, VM resumed.");
        synchronized (LOCK2) {
            starting = false;
            LOCK2.notifyAll ();
        }
    }

    /**
    * Performs stop action.
    */
    public void setStoppedState (ThreadReference thread, boolean stoppedAll) {
        PropertyChangeEvent evt;
        accessLock.readLock().lock();
        try {
            // this method can be called in stopped state to switch
            // the current thread only
            JPDAThread c = getCurrentThread();
            JPDAThread t = getThread (thread);
            if (!stoppedAll && c != null && c != t && c.isSuspended()) {
                // We already have a suspended current thread, do not switch in that case.
                return ;
            }
            checkJSR45Languages (t);
            evt = setCurrentThreadNoFire(t);
            PropertyChangeEvent evt2 = setStateNoFire(STATE_STOPPED);

            if (evt == null) evt = evt2;
            else if (evt2 != null) {
                PropertyChangeEvent evt3 = evt;
                while(evt3.getPropagationId() != null) evt3 = (PropertyChangeEvent) evt3.getPropagationId();
                evt3.setPropagationId(evt2);
            }
        } finally {
            accessLock.readLock().unlock();
        }
        if (evt != null) {
            do {
                firePropertyChange(evt);
                evt = (PropertyChangeEvent) evt.getPropagationId();
            } while (evt != null);
        }
    }

    /**
     * Can be called if the current thread is resumed after stop.
     */
    public void setRunningState() {
        setState(STATE_RUNNING);
    }

    /**
    * Performs stop action and disable a next call to resume()
    */
    public void setStoppedStateNoContinue (ThreadReference thread) {
        PropertyChangeEvent evt;
        accessLock.readLock().lock();
        try {
            // this method can be called in stopped state to switch
            // the current thread only
            evt = setStateNoFire(STATE_RUNNING);
            JPDAThread t = getThread (thread);
            checkJSR45Languages (t);
            PropertyChangeEvent evt2 = setCurrentThreadNoFire(t);

            if (evt == null) evt = evt2;
            else if (evt2 != null) evt.setPropagationId(evt2);

            evt2 = setStateNoFire(STATE_STOPPED);

            if (evt == null) evt = evt2;
            else if (evt2 != null) {
                PropertyChangeEvent evt3 = evt;
                while(evt3.getPropagationId() != null) evt3 = (PropertyChangeEvent) evt3.getPropagationId();
                evt3.setPropagationId(evt2);
            }

            doContinue = false;
        } finally {
            accessLock.readLock().unlock();
        }
        if (evt != null) {
            do {
                firePropertyChange(evt);
                evt = (PropertyChangeEvent) evt.getPropagationId();
            } while (evt != null);
        }
    }


    private boolean finishing;

    /**
     * Used by KillActionProvider.
     */
    public void finish () {
        try {
            synchronized (this) {
                if (finishing) {
                    // Can easily be called twice - from the operator termination
                    return ;
                }
                finishing = true;
            }
            logger.fine("StartActionProvider.finish ()");
            if (getState () == STATE_DISCONNECTED) return;
            AbstractDICookie di = lookupProvider.lookupFirst(null, AbstractDICookie.class);
            Operator o = getOperator();
            if (o != null) o.stop();
            stopAttachListening();
            try {
                boolean startedUp;
                do {
                    startedUp = waitRunning(500); // First wait till the debugger comes up
                    if (!startedUp) {
                        stopAttachListening();
                        continue;
                    }
                } while (false);
            } catch (DebuggerStartException dsex) {
                // We do not want to start it anyway when we're finishing - do not bother
            }
            VirtualMachine vm;
            synchronized (virtualMachineLock) {
                vm = virtualMachine;
                virtualMachine = null;
            }
            setState (STATE_DISCONNECTED);
            if (jsr45EngineProviders != null) {
                for (Iterator<JSR45DebuggerEngineProvider> i = jsr45EngineProviders.iterator(); i.hasNext();) {
                    JSR45DebuggerEngineProvider provider = i.next();
                    DebuggerEngine.Destructor d = provider.getDesctuctor();
                    if (d != null) {
                        d.killEngine();
                    }
                }
                jsr45EngineProviders = null;
            }
            DebuggerEngine.Destructor d = javaEngineProvider.getDestructor();
            if (d != null) {
                d.killEngine ();
            }
            if (vm != null) {
                try {
                    if (di instanceof AttachingDICookie) {
                        JPDAThreadImpl t;
                        synchronized (currentThreadAndFrameLock) {
                            t = currentThread;
                        }
                        if (t != null && t.isMethodInvoking()) {
                            try {
                                t.waitUntilMethodInvokeDone(5000); // Wait 5 seconds at most
                            } catch (InterruptedException ex) {}
                        }
                        logger.fine(" StartActionProvider.finish() VM dispose");
                        VirtualMachineWrapper.dispose (vm);
                    } else {
                        logger.fine(" StartActionProvider.finish() VM exit");
                        VirtualMachineWrapper.exit (vm, 0);
                    }
                } catch (InternalExceptionWrapper e) {
                    logger.fine(" StartActionProvider.finish() VM exception " + e);
                } catch (VMDisconnectedExceptionWrapper e) {
                    logger.fine(" StartActionProvider.finish() VM exception " + e);
                    // debugee VM is already disconnected (it finished normally)
                }
            }
            logger.fine (" StartActionProvider.finish() end.");

            //Notify LOCK2 so that no one is waiting forever
            synchronized (LOCK2) {
                starting = false;
                LOCK2.notifyAll ();
            }
            EditorContextBridge.getContext().disposeTimeStamp(this);
        } finally {
            finishing = false; // for safety reasons
        }
    }

    private synchronized void stopAttachListening() {
        if (attachingCookie != null) {
            if (attachingCookie instanceof ListeningDICookie) {
                ListeningDICookie listeningCookie = (ListeningDICookie) attachingCookie;
                try {
                    listeningCookie.getListeningConnector().stopListening(listeningCookie.getArgs());
                } catch (java.io.IOException ioex) {
                } catch (com.sun.jdi.connect.IllegalConnectorArgumentsException icaex) {
                } catch (IllegalArgumentException iaex) {
                }
            }
        }
    }

    /**
     * Suspends the target virtual machine (if any).
     * Used by PauseActionProvider.
     *
     * @see  com.sun.jdi.ThreadReference#suspend
     */
    public void suspend () {
        VirtualMachine vm;
        synchronized (virtualMachineLock) {
            vm = virtualMachine;
        }
        accessLock.writeLock().lock();
        try {
            if (vm != null) {
                logger.fine("VM suspend");
                try {
                    VirtualMachineWrapper.suspend (vm);
                    // Check the suspended count
                    List<ThreadReference> threads = VirtualMachineWrapper.allThreads(vm);
                    for (ThreadReference t : threads) {
                        try {
                            while (ThreadReferenceWrapper.suspendCount(t) > 1) {
                                ThreadReferenceWrapper.resume(t);
                            }
                        } catch (IllegalThreadStateExceptionWrapper e) {
                        } catch (ObjectCollectedExceptionWrapper e) {
                        } catch (InternalExceptionWrapper e) {
                        }
                    }
                } catch (VMDisconnectedExceptionWrapper e) {
                    return ;
                } catch (InternalExceptionWrapper e) {
                    return ;
                }
            }
            setState (STATE_STOPPED);
        } finally {
            accessLock.writeLock().unlock();
        }
        notifySuspendAll(true, true);
    }

    public List<PropertyChangeEvent> notifySuspendAll(boolean doFire, boolean explicitelyPaused) {
        return notifySuspendAll(doFire, explicitelyPaused, null);
    }

    public List<PropertyChangeEvent> notifySuspendAll(boolean doFire, boolean explicitelyPaused,
                                                      Set<ThreadReference> ignoredThreads) {
        Collection threads = threadsTranslation.getTranslated();
        List<PropertyChangeEvent> events = new ArrayList<PropertyChangeEvent>(threads.size());
        for (Iterator it = threads.iterator(); it.hasNext(); ) {
            Object threadOrGroup = it.next();
            if (threadOrGroup instanceof JPDAThreadImpl) {
                int status = ((JPDAThreadImpl) threadOrGroup).getState();
                boolean invalid = (status == JPDAThread.STATE_NOT_STARTED ||
                                   status == JPDAThread.STATE_UNKNOWN ||
                                   status == JPDAThread.STATE_ZOMBIE);
                if (!invalid && (ignoredThreads == null || !ignoredThreads.contains(((JPDAThreadImpl) threadOrGroup).getThreadReference()))) {
                    try {
                        PropertyChangeEvent event = ((JPDAThreadImpl) threadOrGroup).notifySuspended(doFire, explicitelyPaused);
                        if (event != null) {
                            events.add(event);
                        }
                    } catch (ObjectCollectedException ocex) {
                        invalid = true;
                    }
                } else if (status == JPDAThread.STATE_UNKNOWN || status == JPDAThread.STATE_ZOMBIE) {
                    threadsTranslation.remove(((JPDAThreadImpl) threadOrGroup).getThreadReference());
                }
            }
        }
        return events;
    }

    public void notifySuspendAllNoFire() {
        Collection threads = threadsTranslation.getTranslated();
        for (Iterator it = threads.iterator(); it.hasNext(); ) {
            Object threadOrGroup = it.next();
            if (threadOrGroup instanceof JPDAThreadImpl) {
                int status = ((JPDAThreadImpl) threadOrGroup).getState();
                boolean invalid = (status == JPDAThread.STATE_NOT_STARTED ||
                                   status == JPDAThread.STATE_UNKNOWN ||
                                   status == JPDAThread.STATE_ZOMBIE);
                if (!invalid) {
                    try {
                        ((JPDAThreadImpl) threadOrGroup).notifySuspendedNoFire();
                    } catch (ObjectCollectedException ocex) {
                        invalid = true;
                    }
                } else if (status == JPDAThread.STATE_UNKNOWN || status == JPDAThread.STATE_ZOMBIE) {
                    threadsTranslation.remove(((JPDAThreadImpl) threadOrGroup).getThreadReference());
                }
            }
        }
    }

    /**
     * Used by ContinueActionProvider & StepActionProvider.
     */
    public void resume () {
        accessLock.readLock().lock();
        try {
            if (!doContinue) {
                doContinue = true;
                // Continue the next time and do nothing now.
                return ;
            }
        } finally {
            accessLock.readLock().unlock();
        }
        PropertyChangeEvent stateChangeEvent;
        //notifyToBeResumedAll();
        VirtualMachine vm;
        synchronized (virtualMachineLock) {
            vm = virtualMachine;
        }
        if (vm != null) {
            logger.fine("VM resume");
            List<JPDAThread> allThreads = getAllThreads();
            accessLock.writeLock().lock();
            logger.finer("Debugger WRITE lock taken.");
            stateChangeEvent = setStateNoFire(STATE_RUNNING);
            // We must resume only threads which are regularly suspended.
            // Otherwise we may unexpectedly resume threads which just hit an event!
            
            // However, this can not be done right without an atomic resume of a set of threads.
            // Since this functionality is not available in the backend, we will
            // call VirtualMachine.resume() if all threads are suspended
            // (so that the model of of suspend all/resume all works correctly)
            // and call ThreadReference.resume() on individual suspended threads
            // (so that suspend of event thread together with continue works correctly).

            // Other cases (some threads suspended and some running with events suspending all threads)
            // will NOT WORK CORRECTLY. Because while resuming threads one at a time,
            // if the first one hits an event which suspends all, other resumes will resume the
            // suspended threads.
            // But this looks like a reasonable trade-off considering the available functionality.

            List<JPDAThreadImpl> threadsToResume = new ArrayList<JPDAThreadImpl>();
            for (JPDAThread t : allThreads) {
                if (t.isSuspended()) {
                    threadsToResume.add((JPDAThreadImpl) t);
                }
            }
            try {
                for (int i = 0; i < threadsToResume.size(); i++) {
                    JPDAThreadImpl t = threadsToResume.get(i);
                    boolean can = t.cleanBeforeResume();
                    if (!can) {
                        threadsToResume.remove(i);
                        i--;
                    }
                }
                if (allThreads.size() == threadsToResume.size()) {
                    // Resuming all
                    VirtualMachineWrapper.resume(vm);
                    for (JPDAThreadImpl t : threadsToResume) {
                        t.setAsResumed();
                    }
                    logger.finer("All threads resumed.");
                } else {
                    for (JPDAThreadImpl t : threadsToResume) {
                        t.resumeAfterClean();
                        t.setAsResumed();
                    }
                }
            } catch (VMDisconnectedExceptionWrapper e) {
            } catch (InternalExceptionWrapper e) {
            } finally {
                accessLock.writeLock().unlock();
                logger.finer("Debugger WRITE lock released.");
                if (stateChangeEvent != null) {
                    firePropertyChange(stateChangeEvent);
                }
                for (JPDAThreadImpl t : threadsToResume) {
                    try {
                        t.fireAfterResume();
                    } catch (ThreadDeath td) {
                        throw td;
                    } catch (Throwable th) {
                        Exceptions.printStackTrace(th);
                    }
                }
            }
        }
    }

    /** DO NOT CALL FROM ANYWHERE BUT JPDAThreadImpl.resume(). */
    public boolean currentThreadToBeResumed() {
        accessLock.readLock().lock();
        try {
            if (!doContinue) {
                doContinue = true;
                // Continue the next time and do nothing now.
                return false;
            }
        } finally {
            accessLock.readLock().unlock();
        }
        setState (STATE_RUNNING);
        return true;
    }

    public void resumeCurrentThread() {
        accessLock.readLock().lock();
        try {
            if (!doContinue) {
                doContinue = true;
                // Continue the next time and do nothing now.
                return ;
            }
        } finally {
            accessLock.readLock().unlock();
        }
        setState (STATE_RUNNING);
        currentThread.resume();
    }

    public void notifyToBeResumedAll() {
        Collection threads = threadsTranslation.getTranslated();
        for (Iterator it = threads.iterator(); it.hasNext(); ) {
            Object threadOrGroup = it.next();
            if (threadOrGroup instanceof JPDAThreadImpl) {
                int status = ((JPDAThreadImpl) threadOrGroup).getState();
                boolean invalid = (status == JPDAThread.STATE_NOT_STARTED ||
                                   status == JPDAThread.STATE_UNKNOWN ||
                                   status == JPDAThread.STATE_ZOMBIE);
                if (!invalid) {
                    ((JPDAThreadImpl) threadOrGroup).notifyToBeResumed();
                } else if (status == JPDAThread.STATE_UNKNOWN || status == JPDAThread.STATE_ZOMBIE) {
                    threadsTranslation.remove(((JPDAThreadImpl) threadOrGroup).getThreadReference());
                }
            }
        }
    }

    public void notifyToBeResumedAllNoFire() {
        Collection threads = threadsTranslation.getTranslated();
        for (Iterator it = threads.iterator(); it.hasNext(); ) {
            Object threadOrGroup = it.next();
            if (threadOrGroup instanceof JPDAThreadImpl) {
                int status = ((JPDAThreadImpl) threadOrGroup).getState();
                boolean invalid = (status == JPDAThread.STATE_NOT_STARTED ||
                                   status == JPDAThread.STATE_UNKNOWN ||
                                   status == JPDAThread.STATE_ZOMBIE);
                if (!invalid) {
                    ((JPDAThreadImpl) threadOrGroup).notifyToBeResumedNoFire();
                } else if (status == JPDAThread.STATE_UNKNOWN || status == JPDAThread.STATE_ZOMBIE) {
                    threadsTranslation.remove(((JPDAThreadImpl) threadOrGroup).getThreadReference());
                }
            }
        }
    }

    private Set<JPDAThreadGroup> interestedThreadGroups = new WeakSet<JPDAThreadGroup>();

    public void interestedInThreadGroup(JPDAThreadGroup tg) {
        interestedThreadGroups.add(tg);
    }

    public boolean isInterestedInThreadGroups() {
        return !interestedThreadGroups.isEmpty();
    }

    public ThreadsCache getThreadsCache() {
        synchronized (threadsCollectorLock) {
            if (threadsCache == null) {
                threadsCache = new ThreadsCache(this);
                threadsCache.addPropertyChangeListener(new PropertyChangeListener() {
                    //  Re-fire the changes
                    @Override
                    public void propertyChange(PropertyChangeEvent evt) {
                        String propertyName = evt.getPropertyName();
                        if (ThreadsCache.PROP_THREAD_STARTED.equals(propertyName)) {
                            firePropertyChange(PROP_THREAD_STARTED, null, getThread((ThreadReference) evt.getNewValue()));
                        }
                        if (ThreadsCache.PROP_THREAD_DIED.equals(propertyName)) {
                            firePropertyChange(PROP_THREAD_DIED, getThread((ThreadReference) evt.getOldValue()), null);
                        }
                        if (ThreadsCache.PROP_GROUP_ADDED.equals(propertyName)) {
                            firePropertyChange(PROP_THREAD_GROUP_ADDED, null, getThreadGroup((ThreadGroupReference) evt.getNewValue()));
                        }
                    }
                });
            }
            return threadsCache;
        }
    }

    List<JPDAThread> getAllThreads() {
        ThreadsCache tc = getThreadsCache();
        if (tc == null) {
            return Collections.emptyList();
        }
        List<ThreadReference> threadList = tc.getAllThreads();
        int n = threadList.size();
        List<JPDAThread> threads = new ArrayList<JPDAThread>(n);
        for (int i = 0; i < n; i++) {
            threads.add(getThread(threadList.get(i)));
        }
        return Collections.unmodifiableList(threads);
    }

    public JPDAThreadGroup[] getTopLevelThreadGroups() {
        ThreadsCache tc = getThreadsCache();
        if (tc == null) {
            return new JPDAThreadGroup[0];
        }
        List<ThreadGroupReference> groupList = tc.getTopLevelThreadGroups();
        JPDAThreadGroup[] groups = new JPDAThreadGroup[groupList.size()];
        for (int i = 0; i < groups.length; i++) {
            groups[i] = getThreadGroup((ThreadGroupReference) groupList.get(i));
        }
        return groups;
    }

    public JPDAThreadImpl getThread (ThreadReference tr) {
        return (JPDAThreadImpl) threadsTranslation.translate (tr);
    }

    public JPDAThreadImpl getExistingThread (ThreadReference tr) {
        return (JPDAThreadImpl) threadsTranslation.translateExisting(tr);
    }

    public JPDAThreadGroup getThreadGroup (ThreadGroupReference tgr) {
        return (JPDAThreadGroup) threadsTranslation.translate (tgr);
    }

    public Variable getLocalVariable(LocalVariable lv, Value v) {
        return (Variable) localsTranslation.translate(lv, v);
    }

    public JPDAClassType getClassType(ReferenceType cr) {
        return (JPDAClassType) localsTranslation.translate (cr);
    }

    public Variable getVariable (Value value) {
        return getLocalsTreeModel ().getVariable (value);
    }

    public void markObject(ObjectVariable var, String label) {
        synchronized (markedObjects) {
            long uid = var.getUniqueID();
            String oldLabel = markedObjects.remove(uid);
            if (oldLabel != null) {
                markedObjectLabels.remove(oldLabel);
            }
            if (label != null) {
                markedObjects.put(uid, label);
                ObjectVariable markedVar = markedObjectLabels.get(label);
                if (markedVar != null) {
                    markedObjects.remove(markedVar.getUniqueID());
                }
                markedObjectLabels.put(label, var);
            }
        }
    }

    /*public Map<ObjectVariable, String> getMarkedObjects() {
        Map<ObjectVariable, String> map;
        synchronized (markedObjects) {
            map = new LinkedHashMap<ObjectVariable, String>(markedObjects);
        }
        return map;
    }*/

    public String getLabel(ObjectVariable var) {
        if (var instanceof Refreshable) {
            if (!((Refreshable) var).isCurrent()) {
                return null;
            }
        }
        synchronized (markedObjects) {
            return markedObjects.get(var.getUniqueID());
        }
    }

    public ObjectVariable getLabeledVariable(String label) {
        synchronized (markedObjects) {
            return markedObjectLabels.get(label);
        }
    }

    public Map<String, ObjectVariable> getAllLabels() {
        synchronized (markedObjects) {
            return new HashMap<String, ObjectVariable>(markedObjectLabels);
        }
    }

    public ExpressionPool getExpressionPool() {
        return expressionPool;
    }

    synchronized void setSingleThreadStepResumeDecision(Boolean decision) {
        singleThreadStepResumeDecision = decision;
    }

    synchronized Boolean getSingleThreadStepResumeDecision() {
        return singleThreadStepResumeDecision;
    }

    public synchronized void setStepInterruptByBptResumeDecision(Boolean decision) {
        stepInterruptByBptResumeDecision = decision;
    }

    public synchronized Boolean getStepInterruptByBptResumeDecision() {
        return stepInterruptByBptResumeDecision;
    }


    // private helper methods ..................................................

    private static final java.util.regex.Pattern jvmVersionPattern =
            java.util.regex.Pattern.compile ("(\\d+)\\.(\\d+)\\.(\\d+)(_\\d+)?(-\\w+)?");
    private static java.lang.reflect.Method  tcGenericSignatureMethod;
    private static java.lang.reflect.Method  lvGenericSignatureMethod;


    private void initGenericsSupport () {
        tcGenericSignatureMethod = null;
        if (Bootstrap.virtualMachineManager ().minorInterfaceVersion () >= 5) {
            VirtualMachine vm;
            synchronized (virtualMachineLock) {
                vm = virtualMachine;
            }
            if (vm == null) return ;
            try {
                java.util.regex.Matcher m = jvmVersionPattern.matcher(VirtualMachineWrapper.version(vm));
                if (m.matches ()) {
                    int minor = Integer.parseInt (m.group (2));
                    if (minor >= 5) {
                        try {
                            tcGenericSignatureMethod = TypeComponent.class.
                                getMethod ("genericSignature", new Class [0]);
                            lvGenericSignatureMethod = LocalVariable.class.
                                getMethod ("genericSignature", new Class [0]);
                        } catch (NoSuchMethodException e) {
                            // the method is not available, ignore generics
                        }
                    }
                }
            } catch (InternalExceptionWrapper e) {
            } catch (VMDisconnectedExceptionWrapper e) {
            }
        }
    }

    private PropertyChangeEvent setStateNoFire (int state) {
        int o;
        synchronized (stateLock) {
            if (state == this.state) return null;
            o = this.state;
            this.state = state;
        }
        //PENDING HACK see issue 46287
        System.setProperty(
            "org.openide.awt.SwingBrowserImpl.do-not-block-awt",
            String.valueOf (state != STATE_DISCONNECTED)
        );
        return new PropertyChangeEvent(this, PROP_STATE, new Integer (o), new Integer (state));
    }

    private void setState (int state) {
        if (state == STATE_RUNNING) {
            synchronized (currentThreadAndFrameLock) {
                currentCallStackFrame = null;
            }
        }
        PropertyChangeEvent evt = setStateNoFire(state);
        if (evt != null) {
            firePropertyChange(evt);
        }
    }

    /**
     * Fires property change.
     */
    private void firePropertyChange (String name, Object o, Object n) {
        pcs.firePropertyChange (name, o, n);
        //System.err.println("ALL Change listeners count = "+pcs.getPropertyChangeListeners().length);
    }

    /**
     * Fires property change.
     */
    private void firePropertyChange (PropertyChangeEvent evt) {
        pcs.firePropertyChange (evt);
        //System.err.println("ALL Change listeners count = "+pcs.getPropertyChangeListeners().length);
    }

    private SourcePath engineContext;
    public synchronized SourcePath getEngineContext () {
        if (engineContext == null)
            engineContext = lookupProvider.lookupFirst(null, SourcePath.class);
        return engineContext;
    }

    private LocalsTreeModel localsTreeModel;
    private LocalsTreeModel getLocalsTreeModel () {
        if (localsTreeModel == null)
            localsTreeModel = (LocalsTreeModel) lookupProvider.
                lookupFirst ("LocalsView", TreeModel.class);
        return localsTreeModel;
    }

    private ThreadReference getEvaluationThread () {
        synchronized (currentThreadAndFrameLock) {
            if (currentThread != null) return currentThread.getThreadReference ();
        }
        VirtualMachine vm;
        synchronized (virtualMachineLock) {
            vm = virtualMachine;
        }
        if (vm == null) return null;
        List l;
        try {
            l = VirtualMachineWrapper.allThreads(vm);
        } catch (InternalExceptionWrapper ex) {
            return null;
        } catch (VMDisconnectedExceptionWrapper ex) {
            return null;
        }
        if (l.size () < 1) return null;
        int i, k = l.size ();
        ThreadReference thread = null;
        for (i = 0; i < k; i++) {
            ThreadReference t = (ThreadReference) l.get (i);
            try {
                if (ThreadReferenceWrapper.isSuspended (t)) {
                    thread = t;
                    if (ThreadReferenceWrapper.name (t).equals ("Finalizer"))
                        return t;
                }
            } catch (InternalExceptionWrapper ex) {
            } catch (IllegalThreadStateExceptionWrapper ex) {
            } catch (ObjectCollectedExceptionWrapper ex) {
            } catch (VMDisconnectedExceptionWrapper ex) {
            }
        }
        return thread;
    }

    private PropertyChangeEvent updateCurrentCallStackFrameNoFire (JPDAThread thread) {
        CallStackFrame oldSF;
        CallStackFrame newSF;
        if ((thread == null) || (thread.getStackDepth () < 1)) {
            newSF = null;
        } else {
            try {
                CallStackFrame[] csfs = thread.getCallStack(0, 1);
                if (csfs.length > 0) {
                    newSF = csfs[0];
                } else {
                    newSF = null;
                }
            } catch (AbsentInformationException e) {
                newSF = null;
            }
        }
        oldSF = setCurrentCallStackFrameNoFire(newSF);
        if (oldSF == newSF) return null;
        else return new PropertyChangeEvent(this, PROP_CURRENT_CALL_STACK_FRAME,
                                            oldSF, newSF);
    }

    private CallStackFrame getTopFrame(JPDAThread thread) {
        CallStackFrame callStackFrame;
        if ((thread == null) || (thread.getStackDepth () < 1)) {
            callStackFrame = null;
        } else {
            try {
                CallStackFrame[] csfs = thread.getCallStack(0, 1);
                if (csfs.length > 0) {
                    callStackFrame = csfs[0];
                } else {
                    callStackFrame = null;
                }
            } catch (AbsentInformationException e) {
                callStackFrame = null;
            }
        }
        return callStackFrame;
    }

    /**
     * @param thread The thread to take the top frame from
     * @return A PropertyChangeEvent or <code>null</code>.
     */
    private PropertyChangeEvent updateCurrentCallStackFrameNoFire(CallStackFrame callStackFrame) {
        CallStackFrame old;
        old = setCurrentCallStackFrameNoFire(callStackFrame);
        if (old == callStackFrame) return null;
        else return new PropertyChangeEvent(this, PROP_CURRENT_CALL_STACK_FRAME,
                                            old, callStackFrame);
    }

    private void checkJSR45Languages (JPDAThread t) {
        if (t.getStackDepth () > 0)
            try {
                CallStackFrame[] frames = t.getCallStack (0, 1);
                if (frames.length < 1) return ; // Internal error or disconnected
                CallStackFrame f = frames [0];
                List<String> l = f.getAvailableStrata ();
                String stratum = f.getDefaultStratum ();
                //String sourceDebugExtension;
                    //sourceDebugExtension = (String) f.getClass().getMethod("getSourceDebugExtension").invoke(f);
                if (l.size() == 1 && "Java".equals(l.get(0))) {     // NOI18N
                    // Hack for non-Java languages that do not define stratum:
                    String sourceName = f.getSourceName(null);
                    int ext = sourceName.lastIndexOf('.');
                    if (ext > 0) {
                        String extension = sourceName.substring(++ext);
                        extension = extension.toUpperCase();
                        if (!"JAVA".equals(extension)) {    // NOI18N
                            l = Collections.singletonList(extension);
                            stratum = extension;
                        }
                    }
                }
                int i, k = l.size ();
                for (i = 0; i < k; i++) {
                    if (!languages.contains (l.get (i))) {
                        String language = l.get (i);
                        DebuggerManager.getDebuggerManager ().startDebugging (
                            createJSR45DI (language)
                        );
                        languages.add (language);
                    }
                } // for
                if ( (stratum != null) &&
                     (!stratum.equals (lastStratumn))
                )
                    javaEngineProvider.getSession ().setCurrentLanguage (stratum);
                lastStratumn = stratum;
            } catch (AbsentInformationException e) {
            }
    }

    private Set<JSR45DebuggerEngineProvider> jsr45EngineProviders;

    private DebuggerInfo createJSR45DI (final String language) {
        if (jsr45EngineProviders == null) {
            jsr45EngineProviders = new HashSet<JSR45DebuggerEngineProvider>(1);
        }
        JSR45DebuggerEngineProvider provider = new JSR45DebuggerEngineProvider(language);
        jsr45EngineProviders.add(provider);
        return DebuggerInfo.create (
            "netbeans-jpda-JSR45DICookie-" + language,
            new Object[] {
                new DelegatingSessionProvider () {
                    @Override
                    public Session getSession (
                        DebuggerInfo debuggerInfo
                    ) {
                        return javaEngineProvider.getSession ();
                    }
                },
                provider
            }
        );
    }

    @Override
    public JPDAStep createJPDAStep(int size, int depth) {
        Session session = lookupProvider.lookupFirst(null, Session.class);
        return new JPDAStepImpl(this, session, size, depth);
    }

    /*public synchronized Heap getHeap() {
        if (virtualMachine != null && canGetInstanceInfo(virtualMachine)) {
            return new HeapImpl(virtualMachine);
        } else {
            return null;
        }
    }*/

    @Override
    public List<JPDAClassType> getAllClasses() {
        //assert !java.awt.EventQueue.isDispatchThread() : "All classes retrieving in AWT Event Queue!";
        List<ReferenceType> classes;
        synchronized (virtualMachineLock) {
            if (virtualMachine == null) {
                classes = Collections.emptyList();
            } else {
                classes = VirtualMachineWrapper.allClasses0(virtualMachine);
            }
        }
        return new ClassTypeList(this, classes);
    }

    @Override
    public List<JPDAClassType> getClassesByName(String name) {
        List<ReferenceType> classes;
        synchronized (virtualMachineLock) {
            if (virtualMachine == null) {
                classes = Collections.emptyList();
            } else {
                classes = VirtualMachineWrapper.classesByName0(virtualMachine, name);
            }
        }
        return new ClassTypeList(this, classes);
    }

    @Override
    public long[] getInstanceCounts(List<JPDAClassType> classTypes) throws UnsupportedOperationException {
            //assert !java.awt.EventQueue.isDispatchThread() : "Instance counts retrieving in AWT Event Queue!";
            VirtualMachine vm;
            synchronized (virtualMachineLock) {
                vm = virtualMachine;
            }
            if (vm == null) {
                return new long[classTypes.size()];
            }
            List<ReferenceType> types;
            if (classTypes instanceof ClassTypeList) {
                ClassTypeList cl = (ClassTypeList) classTypes;
                types = cl.getTypes();
            } else {
                types = new ArrayList<ReferenceType>(classTypes.size());
                for (JPDAClassType clazz : classTypes) {
                    types.add(((JPDAClassTypeImpl) clazz).getType());
                }
            }
            try {
                return VirtualMachineWrapper.instanceCounts(vm, types);
            } catch (InternalExceptionWrapper e) {
                return new long[classTypes.size()];
            } catch (VMDisconnectedExceptionWrapper e) {
                return new long[classTypes.size()];
            }
    }

    @Override
    public boolean canGetInstanceInfo() {
        synchronized (virtualMachineLock) {
            return virtualMachine != null && virtualMachine.canGetInstanceInfo();
        }
    }

    @Override
    public ThreadsCollectorImpl getThreadsCollector() {
        synchronized (threadsCollectorLock) {
            if (threadsCollector == null) {
                threadsCollector = new ThreadsCollectorImpl(this);
            }
            return threadsCollector;
        }
    }

    DeadlockDetector getDeadlockDetector() {
        synchronized (threadsCollectorLock) {
            if (deadlockDetector == null) {
                deadlockDetector = new DeadlockDetectorImpl(this);
            }
            return deadlockDetector;
        }
    }

    public List<JPDAClassType> getAllInterfaces(ClassType ct) {
        try {
            List allInterfaces;
            boolean toCompute = false;
            synchronized (allInterfacesMap) {
                allInterfaces = allInterfacesMap.get(ct);
                if (allInterfaces == null) {
                    allInterfaces = new ArrayList();
                    allInterfaces.add("computing");             // NOI18N
                    allInterfacesMap.put(ct, allInterfaces);
                    toCompute = true;
                }
            }
            if (toCompute) {
                List<InterfaceType> interfaces = null;
                try {
                    //assert !javax.swing.SwingUtilities.isEventDispatchThread();
                    interfaces = ClassTypeWrapper.allInterfaces0(ct);
                } finally {
                    if (interfaces == null) {
                        synchronized (allInterfacesMap) {
                            allInterfacesMap.remove(ct);
                        }
                    }
                    synchronized (allInterfaces) {
                        allInterfaces.clear();
                        if (interfaces != null) {
                            for (InterfaceType it : interfaces) {
                                allInterfaces.add(new JPDAClassTypeImpl(this, it));
                            }
                        }
                        allInterfaces.notifyAll();
                    }
                }
            } else {
                synchronized (allInterfaces) {
                    if (allInterfaces.contains("computing")) {  // NOI18N
                        try {
                            allInterfaces.wait();
                        } catch (InterruptedException ex) {
                            return null;
                        }
                    }
                }
            }
            return Collections.unmodifiableList(allInterfaces);
        } catch (ClassNotPreparedExceptionWrapper cnpex) {
            return null;
        }
    }

    public boolean hasAllInterfaces(ClassType ct) {
        List allInterfaces;
        synchronized (allInterfacesMap) {
            allInterfaces = allInterfacesMap.get(ct);
        }
        if (allInterfaces == null) {
            return false;
        } else {
            synchronized (allInterfaces) {
                if (allInterfaces.contains("computing")) {      // NOI18N
                    return false;
                }
            }
        }
        return true;
    }


    private static class DebuggerReentrantReadWriteLock extends ReentrantReadWriteLock {

        private ReadLock readerLock;
        private WriteLock writerLock;

        public DebuggerReentrantReadWriteLock(boolean fair) {
            super(fair);
            readerLock = new DebuggerReadLock(this);
            writerLock = new DebuggerWriteLock(this);
        }

        @Override
        public ReadLock readLock() {
            return readerLock;
        }

        @Override
        public WriteLock writeLock() {
            return writerLock;
        }

        private static class DebuggerReadLock extends ReentrantReadWriteLock.ReadLock {

            protected DebuggerReadLock(ReentrantReadWriteLock lock) {
                super(lock);
            }

            @Override
            public void lock() {
                assert !EventQueue.isDispatchThread() : "Debugger lock taken in AWT Event Queue!";
                super.lock();
            }

        }

        private static class DebuggerWriteLock extends ReentrantReadWriteLock.WriteLock {

            protected DebuggerWriteLock(ReentrantReadWriteLock lock) {
                super(lock);
            }

            @Override
            public void lock() {
                assert !EventQueue.isDispatchThread() : "Debugger lock taken in AWT Event Queue!";
                super.lock();
            }

        }

    }

}
