/* spawn_network.c
  
   The spawn_network and related commands.

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris

   This is part of Network Expect (nexp)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"
#include "nexp_speakers.h"
#include "util-tcl.h"

#define SNAPLEN 65535

static int spawn_id;

/*
 * Linked list of all active listeners. The -i parameter to the
 * expect_network command specifies one of these listeners.
 */
static struct nexp_listener *listeners;

/*
 * Returns a pointer to the listener structure that corresponds to the
 * listener ID passed as a parameter.
 *
 * Returns NULL if no listener with the specified ID exists.
 */
struct nexp_listener *
lookup_listener(const char *listener_id)
{
    struct nexp_listener *l;

    for (l = listeners; l; l = l->next)
	if (!strcmp(l->name, listener_id) )
	    break;

    return l;
}

/*
 * Returns the file descriptor associated with a listener. Having a function
 * to do this simplifies the code because, due to the use of a union inside
 * the nexp_listener structure, there is always the need to look at the
 * listener type to be able to obtain the correct file descriptor.
 */
int
listener_fd(struct nexp_listener *l)
{
    if (!l)
	return -1;

    switch (l->type) {
    case LISTENER_LIVE:
	return l->_live.fd;
	break;
    case LISTENER_SAVEFILE:
	return l->_pcap.fd;
	break;
    case LISTENER_STREAM:
    case LISTENER_DGRAM:
	return l->_socket.fd;
	break;
    default:
	return -1;
    }
}

static void
close_listener(struct nexp_listener *l)
{
    struct nexp_listener *previous_l;

    switch (l->type) {
    case LISTENER_SAVEFILE:
	pcap_close(l->_pcap.pd);
	free(l->_pcap.cfilter);
	free(l->_pcap.rfilter);
	free(l->_pcap.delays);
	free( (void *) l->_pcap.fname);
	free(l->_pcap.packet_list);
	dfilter_free(l->_pcap.rfcode);
	break;
    case LISTENER_LIVE:
	pcap_close(l->_live.pd);
	free(l->_live.cfilter);
	free(l->_live.rfilter);
	free( (void *) l->_live.ifname);
	dfilter_free(l->_live.rfcode);
	break;
    case LISTENER_STREAM:
    case LISTENER_DGRAM:
	close(l->_socket.fd);
	free(l->_socket.dsthost);
	free(l->_socket.dstport);
	break;
    }

    /* Remove listener from linked list of listeners. */
    if (listeners == l)
	/* Element to remove is at the beginning of the list. */
	listeners = l->next;
    else {
	/* Find linked list item previous to one we want to remove. */
	for (previous_l = listeners;
	     previous_l->next != l;
	     previous_l = previous_l->next);

	previous_l->next = l->next;
    }

    free(l);
}

/*
 * Close all active listeners.
 */
static void
close_listeners(void)
{
    while (listeners) {
#ifdef DEBUG
	printf("Closing listener %s\n", listeners->name);
#endif
	close_listener(listeners);
    }
}

static void
listeners_info(void)
{
    struct nexp_listener *l;
    int nlisteners, i;
    static const char *listener_types_names[] = {
	[LISTENER_LIVE] = "live capture",
	[LISTENER_SAVEFILE] = "PCAP file",
	[LISTENER_STREAM] = "TCP socket",
	[LISTENER_DGRAM] = "UDP socket"
    };
    struct pcap_stat ps;

    printf("Listeners:\n");

    for (l = listeners, nlisteners = 0; l; l = l->next, nlisteners++);

    printf("  Number of listeners: %d\n", nlisteners);

    for (l = listeners, i = 0; l; l = l->next, i++) {
	printf("  Listener #%d:\n", i);
	printf("    Name: %s\n", l->name);
	printf("    Type: %s\n", listener_types_names[l->type]);
	switch (l->type) {
	case LISTENER_LIVE:
	    printf("    Capture (pcap) filter: %s\n",
		   l->_pcap.cfilter && *l->_pcap.cfilter ? l->_live.cfilter
							 : "none");
	    printf("    Display (libwireshark) filter: %s\n",
		   l->_pcap.rfilter && *l->_pcap.rfilter ? l->_live.rfilter
							 : "none");
	    printf("    Packet capture descriptor at: %p\n", l->_live.pd);
	    printf("    File descriptor: %d\n", l->_live.fd);
	    printf("    Reading packets from: interface %s\n",
		   l->_live.ifname);

	    if (pcap_stats(l->_live.pd, &ps) == 0) {
		printf("    Number of packets received: %u\n", ps.ps_recv);
		printf("    Number of packets dropped: %u\n", ps.ps_drop);
	    }
	    break;
	case LISTENER_SAVEFILE:
	    printf("    Capture (pcap) filter: %s\n",
		   l->_pcap.cfilter && *l->_pcap.cfilter ? l->_live.cfilter
							 : "none");
	    printf("    Display (libwireshark) filter: %s\n",
		   l->_pcap.rfilter && *l->_pcap.rfilter ? l->_live.rfilter
							 : "none");
	    printf("    Packet capture descriptor at: %p\n", l->_pcap.pd);
	    printf("    File descriptor: %d\n", l->_pcap.fd);
	    printf("    Reading packets from: PCAP savefile %s\n",
		   l->_pcap.fname);
	    if (!l->_pcap.fullspeed) {
		/*
		 * We have only pre-processed the savefile if we're not
		 * reading at full speed.
		 */
		printf("    Number of packets in savefile: %d\n",
		       l->_pcap.npackets);
		printf("    Packets read so far: %d\n", l->_pcap.curr_pkt);
	    }
	    break;
	case LISTENER_STREAM:
	case LISTENER_DGRAM:
	    printf("    File descriptor: %d\n", l->_socket.fd);
	    printf("    Host: %s\n", l->_socket.dsthost);
	    printf("    Port: %s\n", l->_socket.dstport);
	    break;
	}
    }
}

static void
listeners_and_speakers_info(void)
{
    listeners_info();
    speakers_info();
}

static int
socket_listener_info(struct Tcl_Interp *interp, const char *listener)
{
    struct nexp_listener *l;
    struct sockaddr_in sin_remote, sin_local;
    socklen_t sin_len;
    Tcl_Obj *list_obj, *obj;
    struct addr local_addr, remote_addr;
    char *ip;

    l = lookup_listener(listener);
    if (!l) {
	nexp_error(interp, "\"%s\" is not a listener", listener);
	return -1;
    }

    if (l->type != LISTENER_STREAM && l->type != LISTENER_DGRAM) {
	nexp_error(interp, "\"%s\" is not a TCP or UDP listener", listener);
	return -1;
    }

    sin_len = sizeof sin_local;

    if (getsockname(l->_socket.fd, (struct sockaddr *) &sin_local, &sin_len)
	== -1) {
	nexp_error(interp, "getsockname(): %s", strerror(errno) );
	return -1;
    }

    sin_len = sizeof sin_remote;

    if (getpeername(l->_socket.fd, (struct sockaddr *) &sin_remote, &sin_len)
	== -1) {
	nexp_error(interp, "getpeername(): %s", strerror(errno) );
	return -1;
    }

    if (addr_ston( (struct sockaddr *) &sin_local, &local_addr) == -1) {
	nexp_error(interp, "addr_ston(): %s", strerror(errno) );
	return -1;
    }

    if (addr_ston( (struct sockaddr *) &sin_remote, &remote_addr) == -1) {
	nexp_error(interp, "addr_ston(): %s", strerror(errno) );
	return -1;
    }

    list_obj = Tcl_NewListObj(0, NULL);

    ip = ipaddr2str(local_addr.addr_ip);
    obj = Tcl_NewStringObj(ip, strlen(ip) );
    Tcl_ListObjAppendElement(interp, list_obj, obj);

    obj = Tcl_NewIntObj(ntohs(sin_local.sin_port) );
    Tcl_ListObjAppendElement(interp, list_obj, obj);

    ip = ipaddr2str(remote_addr.addr_ip);
    obj = Tcl_NewStringObj(ip, strlen(ip) );
    Tcl_ListObjAppendElement(interp, list_obj, obj);

    obj = Tcl_NewIntObj(ntohs(sin_remote.sin_port) );
    Tcl_ListObjAppendElement(interp, list_obj, obj);

    Tcl_SetObjResult(interp, list_obj);

    return 0;
}

/*
 * Pre-process a PCAP savefile by calculating the time deltas between
 * packets and the number of packets (for information purposes only)
 * in the savefile.
 *
 * Note: no error checking for return values from pcap_compile() and
 * pcap_setfilter() because this function gets called after we've used
 * them before.
 */
static int
calc_delays(const char *savefile, char *filter, struct nexp_listener *l)
{
    pcap_t *pd;
    char errbuf[PCAP_ERRBUF_SIZE];
    struct pcap_pkthdr h;
    struct bpf_program fcode;
    int curr_pkt;
    struct timeval previous, delta;

    if (vflag)
	printf("Pre-processing savefile %s\n", savefile);
 
    pd = pcap_open_offline(savefile, errbuf);
    pcap_compile(pd, &fcode, filter, 1, 0);
    pcap_setfilter(pd, &fcode);
    pcap_freecode(&fcode);

    /*
     * Read first packet to obtain the initial time.
     */
    if (!pcap_next(pd, &h) ) {
	warn("PCAP savefile %s is empty!", savefile);
	pcap_close(pd);
	return -1;
    }

    l->_pcap.delays = xrealloc(l->_pcap.delays, 1*sizeof(struct timeval) );
    l->_pcap.delays[0].tv_sec = l->_pcap.delays[0].tv_usec = 0;

    for (previous = h.ts, curr_pkt = 1;
	 pcap_next(pd, &h);
	 previous = h.ts, curr_pkt++) {
	l->_pcap.delays = xrealloc(l->_pcap.delays,
				   (curr_pkt + 1)*sizeof(struct timeval) );

	timersub(&h.ts, &previous, &delta);

	l->_pcap.delays[curr_pkt] = delta;
    }

    /*
     * Add a fake delta of 0 seconds at the end of the list so after reading
     * the last packet of the savefile we can proceed to perform one last read
     * which will give us EOF. If we don't this we'll read random junk which
     * may lead us into an infinite loop, or at least a loop lasting several
     * years, depending on the random junk we read.
     */
    l->_pcap.delays = xrealloc(l->_pcap.delays,
			       (curr_pkt + 1)*sizeof(struct timeval) );
    delta.tv_sec = delta.tv_usec = 0;
    l->_pcap.delays[curr_pkt] = delta;

    l->_pcap.npackets = curr_pkt;

    if (vflag)
	printf("Pre-processed %d packets from savefile %s\n",
	       l->_pcap.npackets, savefile);

    pcap_close(pd);

    return 0;
}

/*
 * Prepares a packet capture file descriptor for reading. If the input flag
 * "buffered" is false, it prepares the packet capture file descriptor for
 * reading in real-time, i.e. no buffering.
 *
 * Returns -1 on failure or a file descriptor that can be passed to select().
 */
static int
prepare_pd(Tcl_Interp *interp, pcap_t *pd, int buffered)
{
    int fd;
    char errbuf[PCAP_ERRBUF_SIZE];

    if ( (fd = pcap_get_selectable_fd(pd) ) == -1) {
	nexp_error(interp, "pcap_get_selectable_fd() returned error");
	return -1;
    }

    if (buffered)
	/*
	 * Nothing else to do here since we want a capture file
	 * descriptor that uses buffering.
	 */
	return fd;

    /*
     * This really isn't needed for Linux but it doesn't hurt. It is
     * definitely needed for platforms where select() times out when there
     * is more than one packet ready to be read for a particular file
     * descriptor.  See comments in expect_network.c regarding this.
     */
    if (pcap_setnonblock(pd, 1, errbuf) == -1) {
	nexp_error(interp, "Can't set non-blocking mode: %s",
		   pcap_geterr(pd) );
	return -1;
    }

#if (defined(BSD) || defined(__OpenBSD__) ) && defined(BIOCIMMEDIATE)
    /*
     * For some of the BSDs this is very important since otherwise
     * we won't be able to read data as soon as it arrives. Instead,
     * the OS will buffer it and hand it out once the buffer is full.
     */
    if (ioctl(fd, BIOCIMMEDIATE, (int []) {1}) < 0)
	warn("ioctl() with BIOCIMMEDIATE returned an error. errno = %d",
	     errno);
#elif __sun__
    /*
     * Under Solaris, select() keeps waiting until the next packet,
     * because it is buffered, so we have to set timeout and
     * chunk size to zero
     */
    {
	struct timeval time_zero = {0, 0};

	if (ioctl(fd, SBIOCSCHUNK, (int []) {0}) < 0)
	    warn("ioctl() with SBIOCSCHUNK returned an error (%d): %s",
		 errno, strerror(errno) );

	if (ioctl(fd, SBIOCSTIME, &time_zero) < 0)
	    warn("ioctl() with SBIOCSTIME returned an error (%d): %s",
		 errno, strerror(errno) );
    }
#endif

    return fd;
}

/*
 * Sets the PCAP filter for a packet capture file descriptor.
 *
 * Returns 0 on success and -1 on failure.
 */
static int
set_pcap_filter(Tcl_Interp *interp, pcap_t *pd, char *filter)
{
    struct bpf_program fcode;

    if (pcap_compile(pd, &fcode, filter, 1, 0) < 0) {
	nexp_error(interp, "Can't compile filter: %s", pcap_geterr(pd) );
	return -1;
    }

    if (pcap_setfilter(pd, &fcode) < 0) {
	pcap_freecode(&fcode);
	nexp_error(interp, "Can't set filter: %s", pcap_geterr(pd) );
	return -1;
    }

    /*
     * pcap(3) says that we can release the memory used by the BPF
     * program after it has been made the filter program via a call to
     * pcap_setfilter().
     */
    pcap_freecode(&fcode);

    return 0;
}

static int
process_packet_list(struct nexp_listener *l, const char *packet_list)
{
    char *p, *s, *t;
    int *list = NULL;
    int npackets = 0;
    int a, b, n;
    int i, j;

    p = xstrdup(packet_list);

    for (s = strtok(p, ","); s; s = strtok(NULL, ",") ) {
	t = strchr(s, '-');
	if (t) {
	    /* It's a packet range */
	    *t++ = '\0';
	    a = strtoul(s, NULL, 0);
	    b = strtoul(t, NULL, 0);
	    if (a > b) {
		/* Invalid range */
		free(p);
		free(list);
		return -1;
	    }

	    n = b - a + 1;

	    list = xrealloc(list, sizeof(int)*(npackets + n) );
	    for (j = a, i = 0; i < n; i++, j++)
		list[npackets + i] = j - 1;

	    npackets += n;
	} else {
	    /* Single packet */
	    list = xrealloc(list, sizeof(int)*(npackets + 1) );
	    list[npackets++] = strtoul(s, NULL, 0) - 1;
	}
    }

    qsort(list, npackets, sizeof(int), compare_integers);

    l->_pcap.packet_list = list;
    l->_pcap.npacketsl = npackets;

    free(p);

    return 0;
}

static
void usage(void)
{
    fprintf(stderr, "\
usage: spawn_network [-r <PCAP file>] [-w <PCAP file>] [-hex] [-stdout]\n\
         [-info] [-4tuple] [-ip] [-ip6] [-i <interface>] [-p] [-o <interface>]\n\
         [-tcp] [-udp] [-fullspeed|-normalspeed] [-buffered] \n\
         [<PCAP filter> | host port]");
}

/********************************************************************
 *                          spawn_network                           *
 ********************************************************************/

static int
NExp_SpawnNetworkCmd(ClientData clientData _U_, Tcl_Interp *interp, int argc,
		     const char **argv)
{
    int i, fd;
    const char *interfacein = NULL;
    const char *pcapfilein = NULL;
    int promisc = 1;
    int fullspeed = 0;
    pcap_t *pd;
    char errbuf[PCAP_ERRBUF_SIZE];
    char *cmdbuf;
    struct nexp_listener *l, *new_listener;
    struct nexp_speaker speaker_parms, *new_speaker;
    enum listener_types listener_type = LISTENER_NONE;
    int sockfd = -1; /* Initialize to shut compiler warning up */
    struct addr a;
    struct sockaddr_in sin;
    const char *packet_list = NULL;
    int buffered = 0;
    const gchar *rfilter = NULL;

    memset(&speaker_parms, 0, sizeof speaker_parms);
    speaker_parms.type = SPEAKER_NONE;

    for (i = 1; i < argc && argv[i][0] == '-'; i++) {
	if (!strcmp(argv[i], "-info") ) {
	    listeners_and_speakers_info();
	    return TCL_OK;
	} else if (!strcmp(argv[i], "-4tuple") ) {
	    if (++i >= argc) {
		nexp_error(interp, "%s requires an argument.", argv[i - 1]);
		return TCL_ERROR;
	    }

	    return socket_listener_info(interp, argv[i]) == -1 ? TCL_ERROR
							       : TCL_OK;
	} else if (!strcmp(argv[i], "-i") ) {
	    if (++i >= argc) {
		nexp_error(interp, "%s requires an argument.", argv[i - 1]);
		return TCL_ERROR;
	    }

	    listener_type = LISTENER_LIVE;
	    interfacein = argv[i];
	} else if (!strcmp(argv[i], "-o") ) {
	    if (++i >= argc) {
		nexp_error(interp, "%s requires an argument.", argv[i - 1]);
		return TCL_ERROR;
	    }

	    speaker_parms._iface.ifname = argv[i];
	    speaker_parms.type = SPEAKER_ETHER;
	} else if (!strcmp(argv[i], "-readpcap") || !strcmp(argv[i], "-r") ) {
	    if (++i >= argc) {
		nexp_error(interp, "%s requires an argument.", argv[i - 1]);
		return TCL_ERROR;
	    }

	    listener_type = LISTENER_SAVEFILE;
	    pcapfilein = argv[i];
	} else if (!strcmp(argv[i], "-promiscuous")
		   || !strcmp(argv[i], "-p") ) {
	    promisc = 0;
	} else if (!strcmp(argv[i], "-writepcap") || !strcmp(argv[i], "-w") ) {
	    if (++i >= argc) {
		nexp_error(interp, "%s requires an argument.", argv[i - 1]);
		return TCL_ERROR;
	    }

	    speaker_parms.type = SPEAKER_PCAP;
	    speaker_parms._pcap.fname = argv[i];
	} else if (!strcmp(argv[i], "-ip") || !strcmp(argv[i], "-4") ) {
	    speaker_parms.type = SPEAKER_IPV4;
#ifdef __linux__
	} else if (!strcmp(argv[i], "-ip6") || !strcmp(argv[i], "-6") ) {
	    speaker_parms.type = SPEAKER_IPV6;
#endif
	} else if (!strcmp(argv[i], "-hex") ) {
	    speaker_parms.type = SPEAKER_HEX;
	} else if (!strcmp(argv[i], "-stdout") ) {
	    speaker_parms.type = SPEAKER_STDOUT;
	} else if (!strcmp(argv[i], "-fullspeed") || !strcmp(argv[i], "-f") ) {
	    fullspeed = 1;
	} else if (!strcmp(argv[i], "-normalspeed") ) {
	    fullspeed = 0;
	} else if (!strcmp(argv[i], "-tcp") ) {
	    listener_type = LISTENER_STREAM;
	    speaker_parms.type = SPEAKER_STREAM;
	} else if (!strcmp(argv[i], "-udp") ) {
	    listener_type = LISTENER_DGRAM;
	    speaker_parms.type = SPEAKER_DGRAM;
	} else if (!strcmp(argv[i], "-packets") ) {
	    if (++i >= argc) {
		nexp_error(interp, "%s requires an argument.", argv[i - 1]);
		return TCL_ERROR;
	    }

	    packet_list = argv[i];
	} else if (!strcmp(argv[i], "-buffered") ) {
	    buffered = 1;
	} else if (!strcmp(argv[i], "-R") ) {
	    if (++i >= argc) {
		nexp_error(interp, "%s requires an argument (display filter.)",
			   argv[i - 1]);
		return TCL_ERROR;
	    }

	    rfilter = argv[i];
	} else {
	    nexp_error(interp, "Unknown argument %s.", argv[i]);
	    return TCL_ERROR;
	}
    }

    /*
     * Sanity check: make sure that the user didn't specify a TCP/UDP
     * listener/speaker and then specified a different type of
     * listener/speaker.
     */
    if (   (listener_type == LISTENER_STREAM
	    && speaker_parms.type != SPEAKER_STREAM)
	|| (listener_type == LISTENER_DGRAM
	    && speaker_parms.type != SPEAKER_DGRAM) ) {
	nexp_error(interp,
		   "Can't have TCP/UDP listener and non-TCP/UDP speaker");
	return TCL_ERROR;
    }

    /*
     * Make sure the user is not doing a no-operation.
     */
    if (listener_type == LISTENER_NONE && speaker_parms.type == SPEAKER_NONE) {
	usage();
	return TCL_ERROR;
    }

    if (listener_type == LISTENER_NONE)
	goto nolistener;

    /*********************************************************************
     *                          Listener creation                        *
     *********************************************************************/

    /* Create new listener and initialize its fields. */
    new_listener = xmalloc(sizeof(struct nexp_listener) );
    memset(new_listener, 0, sizeof(struct nexp_listener) );
    new_listener->type = listener_type;
    snprintf(new_listener->name, sizeof(new_listener->name), "nexp%d",
	     spawn_id++);

    switch (listener_type) {
    case LISTENER_LIVE:
	*errbuf = '\0';
	/*
	 * Can't use 0 for the read timeout because this would cause
	 * problems, i.e. select() never returning, on platforms that
	 * do support the read timeout. Setting the timeout to the lowest
	 * possible value (1 msec).
	 */
	pd = pcap_open_live(interfacein, SNAPLEN, promisc, 1, errbuf);
	if (!pd) {
	    nexp_error(interp, "pcap_open_live(): %s", errbuf);
	    free(new_listener);
	    return TCL_ERROR;
	} else if (*errbuf)
	    nexpDiagLog("Warning: %s", errbuf);

	fd = prepare_pd(interp, pd, buffered);
	if (fd == -1) {
	    pcap_close(pd);
	    free(new_listener);
	    return TCL_ERROR;
	}

	/*
	 * Deal with the (optional) PCAP filter.
	 */
	cmdbuf = copy_argv( (char **) &argv[i]);
	if (!cmdbuf)
	    cmdbuf = xstrdup("");

	if (set_pcap_filter(interp, pd, cmdbuf) == -1) {
	    pcap_close(pd);
	    free(cmdbuf);
	    free(new_listener);
	    return TCL_ERROR;
	}

	/*
	 * Deal with the (optional) Wireshark display filter.
	 */
	if (!rfilter)
	    rfilter = "";

	if (!dfilter_compile(rfilter, &new_listener->_live.rfcode) ) {
	    nexp_error(interp, "%s", dfilter_error_msg);
	    pcap_close(pd);
	    free(cmdbuf);
	    free(new_listener);
	    return TCL_ERROR;
	}

	new_listener->_live.pd = pd;
	new_listener->_live.fd = fd;
	new_listener->_live.datalink_type = pcap_datalink(pd);
	new_listener->_live.cfilter = xstrdup(cmdbuf);
	new_listener->_live.rfilter = xstrdup(rfilter);
	new_listener->_live.ifname = xstrdup(interfacein);

	free(cmdbuf);
	break;
    case LISTENER_SAVEFILE:
	if (packet_list)
	    /*
	     * User specified a specific list of packets to use. Let's
	     * process this list to convert into our internal representation.
	     */
	    if (process_packet_list(new_listener, packet_list) == -1) {
		nexp_error(interp, "Couldn't process packet list");
		free(new_listener);
		return TCL_ERROR;
	    }

	*errbuf = '\0';
	pd = pcap_open_offline(pcapfilein, errbuf);
	if (!pd) {
	    nexp_error(interp, "pcap_open_offline(): %s", errbuf);
	    free(new_listener);
	    return TCL_ERROR;
	} else if (*errbuf)
	    nexpDiagLog("Warning: %s", errbuf);

	fd = prepare_pd(interp, pd, buffered);
	if (fd == -1) {
	    pcap_close(pd);
	    free(new_listener);
	    return TCL_ERROR;
	}

	/*
	 * Deal with the (optional) PCAP filter.
	 */
	cmdbuf = copy_argv( (char **) &argv[i]);
	if (!cmdbuf)
	    cmdbuf = xstrdup("");

	if (set_pcap_filter(interp, pd, cmdbuf) == -1) {
	    pcap_close(pd);
	    free(cmdbuf);
	    free(new_listener);
	    return TCL_ERROR;
	}

	/*
	 * Deal with the (optional) Wireshark display filter.
	 */
	if (!rfilter)
	    rfilter = "";

	if (!dfilter_compile(rfilter, &new_listener->_pcap.rfcode) ) {
	    nexp_error(interp, "%s", dfilter_error_msg);
	    pcap_close(pd);
	    free(cmdbuf);
	    free(new_listener);
	    return TCL_ERROR;
	}

	new_listener->_pcap.pd = pd;
	new_listener->_pcap.fd = fd;
	new_listener->_pcap.datalink_type = pcap_datalink(pd);
	new_listener->_pcap.cfilter = xstrdup(cmdbuf);
	new_listener->_pcap.rfilter = xstrdup(rfilter);
	new_listener->_pcap.fname = xstrdup(pcapfilein);
	new_listener->_pcap.fullspeed = fullspeed;

	if (!fullspeed) {
	    /*
	     * Can't see any other way; we need to have the time deltas
	     * between packets in a savefile if we are to read from the
	     * savefile at the same speed at which packets were captured.
	     * Pre-processing is the easiest way out...
	     */
	    if (calc_delays(pcapfilein, cmdbuf, new_listener) == -1) {
		nexp_error(interp, "Couldn't pre-process PCAP savefile.");
		pcap_close(pd);
		free(new_listener);
		return TCL_ERROR;
	    }
	}

	free(cmdbuf);
	break;
    case LISTENER_STREAM:
	if (argc - i != 2) {
	    /*
	     * We're missing destination host and destination port.
	     */
	    usage();
	    free(new_listener);
	    return TCL_ERROR;
	}

        if (addr_pton(argv[i], &a) == -1) {
	    nexp_error(interp, "can't resolve '%s'", argv[i]);
	    free(new_listener);
	    return TCL_ERROR;
        }

	if (addr_ntos(&a, (struct sockaddr *) &sin) == -1) {
	    nexp_error(interp, "addr_ntos(): %s", strerror(errno) );
	    free(new_listener);
	    return TCL_ERROR;
	}

	sin.sin_port = htons(atoi(argv[i + 1]) );

	new_listener->_socket.fd = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
	if (new_listener->_socket.fd == -1) {
	    nexp_error(interp, "Can't create TCP socket");
	    free(new_listener);
	    return TCL_ERROR;
	}
	new_listener->_socket.dsthost = xstrdup(argv[i]);
	new_listener->_socket.dstport = xstrdup(argv[i + 1]);

	sockfd = new_listener->_socket.fd;

	if (connect(sockfd, (struct sockaddr *) &sin, sizeof sin) == -1) {
	    nexp_error(interp, "connect(): %s", strerror(errno) );
	    free(new_listener);
	    return TCL_ERROR;
	}

	/*
	 * Save socket parameters for speaker creation phase (below.)
	 */
	speaker_parms._socket.fd = sockfd;
	speaker_parms._socket.dsthost = new_listener->_socket.dsthost;
	speaker_parms._socket.dstport = new_listener->_socket.dstport;

	break;
    case LISTENER_DGRAM:
	if (argc - i != 2) {
	    /*
	     * We're missing destination host and destination port.
	     */
	    usage();
	    free(new_listener);
	    return TCL_ERROR;
	}

        if (addr_pton(argv[i], &a) == -1) {
	    nexp_error(interp, "can't resolve '%s'", argv[i]);
	    free(new_listener);
	    return TCL_ERROR;
        }

	if (addr_ntos(&a, (struct sockaddr *) &sin) == -1) {
	    nexp_error(interp, "addr_ntos(): %s", strerror(errno) );
	    free(new_listener);
	    return TCL_ERROR;
	}

	sin.sin_port = htons(atoi(argv[i + 1]) );

	new_listener->_socket.fd = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
	if (new_listener->_socket.fd == -1) {
	    nexp_error(interp, "Can't create TCP socket");
	    free(new_listener);
	    return TCL_ERROR;
	}
	new_listener->_socket.dsthost = xstrdup(argv[i]);
	new_listener->_socket.dstport = xstrdup(argv[i + 1]);

	sockfd = new_listener->_socket.fd;

	if (connect(sockfd, (struct sockaddr *) &sin, sizeof sin) == -1) {
	    nexp_error(interp, "connect(): %s", strerror(errno) );
	    free(new_listener);
	    return TCL_ERROR;
	}

	/*
	 * Save socket parameters for speaker creation phase (below.)
	 */
	speaker_parms._socket.fd = sockfd;
	speaker_parms._socket.dsthost = new_listener->_socket.dsthost;
	speaker_parms._socket.dstport = new_listener->_socket.dstport;

	break;
    }

    /* Add new listener to linked list of listeners. */
    if (!listeners)
	listeners = new_listener;
    else {
	for (l = listeners; l->next; l = l->next);

	l->next = new_listener;
    }

    /*
     * Tell user of new spawn id.
     */
    Tcl_SetVar(interp, LISTENER_SPAWN_ID_VARNAME, new_listener->name, 0);

    /*********************************************************************
     *                           Speaker creation                        *
     *********************************************************************/

    /*
     * We jump here after processing of command-line arguments if none of the
     * listener-creation options is specified. In this case the user probably
     * wants to just create a speaker.
     */
nolistener:

    if (speaker_parms.type == SPEAKER_NONE)
	goto nospeaker;

    new_speaker = nexp_newspeaker(&speaker_parms, errbuf);
    if (!new_speaker) {
	nexp_error(interp, "%s", errbuf);
	return TCL_ERROR;
    }

    /*
     * Tell user of new spawn id.
     */
    Tcl_SetVar(interp, SPEAKER_SPAWN_ID_VARNAME, new_speaker->name, 0);

nospeaker:

    return TCL_OK;
}

/********************************************************************
 *                          close_network                           *
 ********************************************************************/

static int
NExp_CloseNetworkCmd(ClientData clientData _U_, Tcl_Interp *interp, int argc,
		     const char **argv)
{
    struct nexp_listener *l;
    struct nexp_speaker *s;

    if (argc != 2) {
	nexp_error(interp, "Usage: %s <listener/speaker ID>.", argv[0]);
	return TCL_ERROR;
    }

    l = lookup_listener(argv[1]);
    if (l)
	close_listener(l);

    s = lookup_speaker(argv[1]);
    if (s)
	close_speaker(s);

    if (!l && !s) {
	nexp_error(interp, "No listener or speaker named \"%s\"", argv[1]);
	return TCL_ERROR;
    } else
	return TCL_OK;
}

static struct nexp_cmd_data cmd_data[]  = {
    {"spawn_network", NULL, NExp_SpawnNetworkCmd, 0, 0},
    {"close_network", NULL, NExp_CloseNetworkCmd, 0, 0},

    {NULL, NULL, NULL, 0, 0}
};

void
nexp_init_spawn_cmds(Tcl_Interp *interp)
{
    if (atexit(close_listeners) != 0) {
	fprintf(stderr, "cannot set exit function close_listeners()\n");
	exit(EXIT_FAILURE);
    }

    if (atexit(close_speakers) != 0) {
	fprintf(stderr, "cannot set exit function close_speakers()\n");
	exit(EXIT_FAILURE);
    }

    create_default_speakers();

    /*
     * Set the "speaker_id" Tcl variable so the send_network command can be
     * used right away, without the need to use spawn_network to create a
     * speaker. Note that we default to the IPv4 (kernel-routed) speaker if an
     * IPv4 speaker was created or to the hex speaker if not.
     */
    if (!lookup_speaker("ip") )
	Tcl_SetVar(interp, SPEAKER_SPAWN_ID_VARNAME,
		   DEFAULT_HEX_SPEAKER_NAME, 0);
    else
	Tcl_SetVar(interp, SPEAKER_SPAWN_ID_VARNAME,
		   DEFAULT_IPV4_SPEAKER_NAME, 0);

    nexp_create_commands(interp, cmd_data);
}
