from ply import lex, yacc
import logging, os
import ldap.schema

log = logging.getLogger(__name__)

class ParseError(Exception):
    pass

def error(t, msg):
    t.lexer.error_count += 1
    t.lexer.log.error("%s:%d: %s"%(t.lexer.path, t.lexer.lineno, msg))

tokens = (
	'ATTRIBUTETYPE',
	'OBJECTCLASS',
	'LPAREN', 'RPAREN',
	'COMMENT',
	'MISC',
)

t_LPAREN = r'\('
t_RPAREN = r'\)'

def t_COMMENT(t):
    r'\#.*'
    pass

def t_ATTRIBUTETYPE(t):
    r'\battributetype\b'
    return t

def t_OBJECTCLASS(t):
    r'\bobjectclass\b'
    return t

def t_space(t):
    r'[ \t\n]+'
    t.lexer.lineno += t.value.count('\n')

def t_MISC(t):
    r"([^()' \t\n]+|'[^']*')+"
    t.lexer.lineno += t.value.count('\n')
    return t

def t_error(t):
    error(t, "Illegal character '%s'"%t.value[0])
    t.lexer.skip(1)


def p_schema0(p):
    'schema :'
    p[0] = {'objectClasses': [], 'attributeTypes': []}

def p_schema1(p):
    'schema : schema schema_element'
    st, se = p[2]
    if se:
	p[1][st].append(se)
    p[0] = p[1]

def p_attributetype(p):
    'schema_element : ATTRIBUTETYPE delimited'
    try:
	p[0] = 'attributeTypes', p[2]
    except AssertionError:
	error(p, "Invalid attribute specification.")

def p_objectclass(p):
    'schema_element : OBJECTCLASS delimited'
    try:
	p[0] = 'objectClasses', p[2]
    except AssertionError:
	error(p, "Invalid attribute specification.")

def p_delimited(p):
    'delimited : LPAREN misc RPAREN'
    p[0] = '(' + p[2] + ')'

def p_misc0(p):
    'misc :'
    p[0] = ''

def p_misc1(p):
    '''misc : misc MISC
	    | misc delimited'''
    p[0] = p[1] + ' ' + p[2]

def p_error(p):
    if p is None:
	return
    p.lexer.error_count += 1
    error(p, 'Syntax error before "%s"'% p.value)


def parse(path, log = log, builddir = '.'):
    lexer = lex.lex()
    lexer.path = path
    lexer.log = log
    lexer.error_count = 0
    parser = yacc.yacc(outputdir = builddir,
		       tabmodule = 'ldap_schema_parser',
		       debugfile = 'ldap_schema_parser.out')
    fh = open(path)
    text = fh.read()
    fh.close()
    r = parser.parse(text, lexer = lexer)
    if lexer.error_count:
	raise ParseError('Failed to parse schema file.')
    return ldap.schema.subentry.SubSchema(r)

# logging.basicConfig(loglevel = logging.DEBUG)
# print parse('/etc/ldap/schema/GLUE20.schema')
