#include "jdxblock.h"
#include <tjutils/tjlist_code.h>

#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif

void set_c_locale() {
  Log<JcampDx> odinlog("JcampDxBlock","set_c_locale");

#ifdef HAVE_LOCALE_H
 // set locale to default before reading/writing blocks
 // to get decimal point on all platforms
  const char* retval=setlocale(LC_NUMERIC, "C");
  ODINLOG(odinlog,normalDebug) << "LC_NUMERIC=" << retval << STD_endl;
#endif
}

JcampDxBlock::JcampDxBlock(const STD_string& title,compatMode mode) : garbage(0), embed(true) {
  Log<JcampDx> odinlog(title.c_str(),"JcampDxBlock(title)");
  set_label(title);
  JcampDxBlock::set_compatmode(mode); // Do NOT call virtual function in constructor!
}

JcampDxBlock::JcampDxBlock(const JcampDxBlock& block) : garbage(0) {
  JcampDxBlock::operator = (block);
}

JcampDxBlock::~JcampDxBlock() {
  Log<JcampDx> odinlog(this,"~JcampDxBlock");
  if(garbage) {
    ODINLOG(odinlog,normalDebug) << "Removing " << JDXList::size() << " parameters from block" << STD_endl;
    JDXList::clear(); // remove before deleting
    ODINLOG(odinlog,normalDebug) << "Deleting " << garbage->size() << " parameters in garbage" << STD_endl;
    for(STD_list<JcampDxClass*>::iterator it=garbage->begin(); it!=garbage->end(); ++it) {
      ODINLOG(odinlog,normalDebug) << "Deleting >" << (*it)->get_label() << "<" << STD_endl;
      delete (*it);
    }
    delete garbage;
  }
}


JcampDxBlock& JcampDxBlock::operator = (const JcampDxBlock& block) {
  Log<JcampDx> odinlog(this,"JcampDxBlock::operator = ");
  JcampDxClass::operator = (block);
//  JDXList::operator = (block); // Unneccessary since the copied list would be cleared in the next line
  JDXList::clear(); // start with empty block so that members are copied by value and not by pointer (as done in List)
  embed=block.embed;
  return *this;
}


JDXList::constiter JcampDxBlock::ldr_exists(const STD_string& label) const {
  Log<JcampDx> odinlog(this,"ldr_exists");
  for(constiter it=get_const_begin(); it!=get_const_end(); ++it) {

    ODINLOG(odinlog,normalDebug) << "comparing LDR  >" << label << "< and >" << (*it)->get_label() << "<" << STD_endl;

    if( (*it)->get_label()==label ) {
      ODINLOG(odinlog,normalDebug) <<  "LDR found" << STD_endl;
      return it;
    }
  }
  ODINLOG(odinlog,normalDebug) << "LDR not found" << STD_endl;
  return get_const_end();
}


STD_string JcampDxBlock::extract_parlabel(const STD_string& parstring) {
  STD_string parlabel(extract(parstring,"##","="));
  if (parlabel[0]=='$') {
    parlabel+="=";
    parlabel=extract(parlabel,"$","=");
  }
  if(parlabel=="TITLE") {
    parlabel=extract(parstring,"##TITLE=","\n");
  }
  return parlabel;
}


int JcampDxBlock::parse_ldr_list(STD_string& parstring) {
  Log<JcampDx> odinlog(this,"parse_ldr_list");

  parstring+="##"; // set mark at end so rmblock(parstring,"##","##",... always works
  ODINLOG(odinlog,normalDebug) << "parstring=>" << parstring << "<" << STD_endl;

  // get 1st parameter before starting while loop
  STD_string parlabel(extract_parlabel(parstring));

  constiter ldr_it=get_const_begin();
  int n_parsed=0;
  while(parlabel!="") {
    if( (ldr_it=ldr_exists(parlabel) )!=get_const_end() ) {
      if(!(*ldr_it)->parse(parstring)) {
        ODINLOG(odinlog,normalDebug) << "parsing of >" << parlabel << "< failed" << STD_endl;
        return -1;
      } else {
        ODINLOG(odinlog,normalDebug) << "parsing of >" << parlabel << "< successful" << STD_endl;
        n_parsed++;
      }
    } else {
      parstring=rmblock(parstring,"##","##",true,false,false,false);
    }

    // get next parameter
    parlabel=extract_parlabel(parstring);
  }
  return n_parsed;
}



bool JcampDxBlock::parse(STD_string& parstring) {
  Log<JcampDx> odinlog(this,"parse");
  if(parseblock(parstring)<0) return false;
  ODINLOG(odinlog,normalDebug) << "parstring(pre)=>" << parstring << "<" << STD_endl;
  // eat up my part from parstring
  parstring+="##END="; // make sure we have an end tag which might got lost
  STD_string blockbody(extract(parstring,"##TITLE=","##END=",true));
  parstring=replaceStr(parstring,"##TITLE="+blockbody+"##END=","");
  ODINLOG(odinlog,normalDebug) << "parstring(post)=>" << parstring << "<" << STD_endl;

  return true;
}


STD_string JcampDxBlock::print_header() const {
  STD_string result="##TITLE="+get_label()+"\n";
  result+="##JCAMPDX=4.24\n";
  result+="##DATATYPE=Parameter Values\n";
  return result;
}

STD_string JcampDxBlock::print_tail() const {
  return "##END=\n";
}


STD_string JcampDxBlock::print() const {
  Log<JcampDx> odinlog(this,"print");

  STD_string retstring(print_header());
  ODINLOG(odinlog,normalDebug) << "printing " << numof_pars() << " parameters" << STD_endl;
  for(constiter it=get_const_begin(); it!=get_const_end(); ++it) {
    retstring+=(*it)->print();
  }
  retstring+=print_tail();
  return retstring;
}


STD_ostream& JcampDxBlock::print2stream(STD_ostream& os) const {
  os << print_header();
  for(constiter it=get_const_begin(); it!=get_const_end(); ++it) {
    if((*it)->get_filemode()!=exclude) {
      os << (*it)->get_jdx_prefix();
      (*it)->print2stream(os);
      os << (*it)->get_jdx_postfix();
    }
  }
  return os << print_tail();
}


STD_string JcampDxBlock::printval(const STD_string& parameterName, bool append_unit) const {
  Log<JcampDx> odinlog(this,"printval");
  STD_string retstring;
  constiter it=ldr_exists(parameterName);
  if(it!=get_const_end()) {
    retstring=(*it)->printvalstring();
    if(append_unit) retstring+=(*it)->get_unit();
  }
  return retstring;
}


bool JcampDxBlock::parseval(const STD_string& parameterName, const STD_string& value) {
  Log<JcampDx> odinlog(this,"parseval");
  bool result=false;
  constiter it=ldr_exists(parameterName);
  if(it!=get_const_end()) {
    STD_string valdummy=value;
    result=(*it)->parsevalstring(valdummy);
  }
  return result;
}


STD_string JcampDxBlock::get_parx_code(parxCodeType type, const ParxEquiv& equiv) const {
#ifdef PARAVISION_PLUGIN
  STD_string result=JcampDxClass::get_parx_code(type,equiv);

  if(type==parx_def) {
    for(constiter it=get_const_begin(); it!=get_const_end(); ++it) {
      if((*it)->isUserDefParameter()) result+=(*it)->get_parx_code(parx_def,(*it)->get_parx_equiv());
    }
  }

  if(type==parx_parclass_def) {
    result="parclass {\n";
    for(constiter it=get_const_begin(); it!=get_const_end(); ++it) {
      if((*it)->isUserDefParameter()) result+=STD_string((*it)->get_label())+";\n";
    }
    result+="}  "+STD_string(get_label())+";\n";
  }

  if(type==parx_parclass_init) {
    STD_string visabilityString;
    result="PARX_hide_class (NOT_HIDDEN, \""+STD_string(get_label())+"\");\n";
    for(constiter it=get_const_begin(); it!=get_const_end(); ++it) {
      if((*it)->isUserDefParameter()) {
        if( (*it)->get_parmode() == hidden) visabilityString="HIDDEN";
        else visabilityString="NOT_HIDDEN";
        result+="PARX_hide_pars ("+visabilityString+", \""+(*it)->get_label()+"\");\n";
        result+="PARX_hide_pars (HIDE_IN_FILE, \""+STD_string((*it)->get_label())+"\");\n";
        if( (*it)->get_parmode() == noedit) {
          result+="PARX_noedit (TRUE, \""+STD_string((*it)->get_label())+"\");\n";
        }
      }
    }
  }

  if(type==parx_parclass_passval) {
    for(constiter it=get_const_begin(); it!=get_const_end(); ++it) {
      if((*it)->isUserDefParameter() && (*it)->get_parx_equiv().name!="" ) {
        result+=(*it)->get_parx_code(parx_passval_head,(*it)->get_parx_equiv());
      }
    }
//    result+="PARX_read_file(\""+STD_string(get_label())+"\",\""+equiv.name+"\");\n";
    result+="ParxRelsReadClass(\""+STD_string(get_label())+"\","+equiv.name+");\n";
    for(constiter it=get_const_begin(); it!=get_const_end(); ++it) {
      if((*it)->isUserDefParameter() && (*it)->get_parx_equiv().name!="" ) {
        result+=(*it)->get_parx_code(parx_passval,(*it)->get_parx_equiv());
      }
    }
  }

  return result;
#else
  return "";
#endif
}


int JcampDxBlock::load(const STD_string &filename) {
  Log<JcampDx> odinlog(this,"load");
  set_c_locale(); // Make sure we have right locale
  STD_string blockstr;
  int retval=::load(blockstr,filename);
  ODINLOG(odinlog,normalDebug) <<  "retval(" << filename << ")=" << retval << STD_endl;
  if(retval) return -1;
  return parseblock(dos2unix(blockstr));
}


int JcampDxBlock::parseblock(const STD_string& source) {
  Log<JcampDx> odinlog(this,"parseblock");
  STD_string blocklabel;
  int result=-1;
  STD_string parlabel=extract(source,"##","=");
  if(parlabel=="TITLE") {
    STD_string source_without_comments=rmblock(source, "\n$$", "\n", true, false); // Remove single-line comments
    source_without_comments=rmblock(source_without_comments, "$$", "\n", true, false); // Remove remaining end-line comments
    blocklabel=extract(source_without_comments,"##TITLE=","\n");
    set_label(blocklabel);
    STD_string blockbody(extract(source_without_comments,"##TITLE=","##END=",true)); // Extract body of parameter block, regarding nested structure (blocks within blocks)
//    blockbody=replaceStr(blockbody,blocklabel+"\n",""); // This seems to be unecessary
    result=parse_ldr_list(blockbody);
  }
  return result;
}


int JcampDxBlock::write(const STD_string &filename) const {
  Log<JcampDx> odinlog(this,"write");

  set_c_locale(); // Make sure we have right locale

  STD_ofstream outfile(filename.c_str());
  JcampDxBlock::print2stream(outfile);
  outfile.close();

  return 0;
}


unsigned int JcampDxBlock::numof_pars() const {
  Log<JcampDx> odinlog(this,"numof_pars");
  unsigned int n=0;
  ODINLOG(odinlog,normalDebug) << "size()=" << size() << STD_endl;
  for(constiter it=get_const_begin(); it!=get_const_end(); ++it) {
    ODINLOG(odinlog,normalDebug) << "trying to count parameter ..." << STD_endl;
    if((*it)->isUserDefParameter()) n++;
    ODINLOG(odinlog,normalDebug) << "counted parameter >" << (*it)->get_label() << "<" << STD_endl;
  }
  return n;
}



JcampDxBlock& JcampDxBlock::merge(JcampDxBlock& block, bool onlyUserPars) {
  Log<JcampDx> odinlog(this,"merge");
  for(iter it=block.get_begin(); it!=block.get_end(); ++it) {
    if( onlyUserPars ) {
      if( (*it)->isUserDefParameter() ) append(**it);
    } else append(**it);
  }
  return *this;
}


JcampDxBlock& JcampDxBlock::unmerge(JcampDxBlock& block) {
  Log<JcampDx> odinlog(this,"unmerge");
  for(iter it=block.get_begin(); it!=block.get_end(); ++it) {
    remove(**it);
  }
  return *this;
}


JcampDxClass* JcampDxBlock::get_parameter(const STD_string& ldrlabel) {
  Log<JcampDx> odinlog(this,"get_parameter");
  for(iter it=get_begin(); it!=get_end(); ++it) {
    if( (*it)->get_label()==ldrlabel ) {
      return (*it);
    }
  }
  return 0;
}


bool JcampDxBlock::parameter_exists(const STD_string& ldrlabel) const {
  Log<JcampDx> odinlog(this,"parameter_exists");
    bool result=false;
    if(ldr_exists(ldrlabel)!=get_const_end()) result=true;
    return result;
}


JcampDxBlock& JcampDxBlock::set_prefix(const STD_string& prefix) {
  Log<JcampDx> odinlog(this,"set_prefix");
  if( STD_string(get_label()).find(prefix) == STD_string::npos ) set_label(prefix+"_"+get_label());

  for(iter it=get_begin(); it!=get_end(); ++it) {
    if((*it)->isUserDefParameter()) {
      if( STD_string((*it)->get_label()).find(prefix) != 0 )
              (*it)->set_label(prefix+"_"+(*it)->get_label());
    }
  }
  return *this;
}


JcampDxClass& JcampDxBlock::set_compatmode(compatMode compat_mode) {
  JcampDxClass::set_compatmode(compat_mode);
  for(iter it=get_begin(); it!=get_end(); ++it) (*it)->set_compatmode(compat_mode);
  return *this;
}


JcampDxClass& JcampDxBlock::set_parmode(parameterMode parameter_mode) {
  JcampDxClass::set_parmode(parameter_mode);
  for(iter it=get_begin(); it!=get_end(); ++it) (*it)->set_parmode(parameter_mode);
  return *this;
}


JcampDxClass& JcampDxBlock::set_filemode(fileMode file_mode) {
  JcampDxClass::set_filemode(file_mode);
  for(iter it=get_begin(); it!=get_end(); ++it) (*it)->set_filemode(file_mode);
  return *this;
}


JcampDxClass& JcampDxBlock::operator [] (unsigned int i) {
  Log<JcampDx> odinlog(this,"operator []");
  unsigned int n=numof_pars();
  if(i>=n) return *this;
  unsigned int j=0;
  for(iter it=get_begin(); it!=get_end(); ++it) {
    if((*it)->isUserDefParameter()) {
      if(j==i) return *(*it);
      j++;
    }
  }
  return *this;
}


const JcampDxClass& JcampDxBlock::operator [] (unsigned int i) const {
  Log<JcampDx> odinlog(this,"operator [] const");
  unsigned int n=numof_pars();
  if(i>=n) return *this;
  unsigned int j=0;
  for(constiter it=get_const_begin(); it!=get_const_end(); ++it) {
    if((*it)->isUserDefParameter()) {
      if(j==i) return *(*it);
      j++;
    }
  }
  return *this;
}


JcampDxClass& JcampDxBlock::get_parameter_by_id(int id) {
  for(iter it=get_begin(); it!=get_end(); ++it) {
    if((*it)->get_parameter_id()==id) return *(*it);
  }
  return *this;
}

JcampDxBlock& JcampDxBlock::append_copy(const JcampDxClass& src) {
  if(!garbage) garbage=new STD_list<JcampDxClass*>;
  JcampDxClass* ldrcopy=src.create_copy();
  garbage->push_back(ldrcopy);
  append(*ldrcopy);
  return *this;
}

JcampDxBlock& JcampDxBlock::create_copy(const JcampDxBlock& src) {
  JcampDxBlock::operator = (src);
  if(!garbage) garbage=new STD_list<JcampDxClass*>;
  for(constiter ldr_it=src.get_const_begin(); ldr_it!=src.get_const_end(); ++ldr_it) {
    if((*ldr_it)->isUserDefParameter())
      append_copy(**ldr_it);
  }
  return *this;
}


JcampDxBlock& JcampDxBlock::copy_ldr_vals(const JcampDxBlock& src) {
  Log<JcampDx> odinlog(this,"copy_ldr_vals");

  for(constiter ldr_it=src.get_const_begin(); ldr_it!=src.get_const_end(); ++ldr_it) {
    ODINLOG(odinlog,normalDebug) << "copying " << (*ldr_it)->get_label() << STD_endl;
    constiter it;
    if( (it=ldr_exists((*ldr_it)->get_label()))!=get_const_end() ) {
      STD_string valstring((*ldr_it)->printvalstring());
      ODINLOG(odinlog,normalDebug) << "valstring=" << valstring << STD_endl;
      (*it)->parsevalstring(valstring);
    }
  }
  return *this;
}



bool JcampDxBlock::compare(const JcampDxBlock& rhs, const STD_list<STD_string>* exclude, double accuracy) const {
  Log<JcampDx> odinlog(this,"compare");

  unsigned int mynpars=numof_pars();
  unsigned int rhnpars=rhs.numof_pars();
  ODINLOG(odinlog,normalDebug) << "mynpars/rhnpars=" << mynpars << "/" << rhnpars << STD_endl;
  if(mynpars!=rhnpars) return mynpars<rhnpars;

  constiter my_it;

  // Instances required for STL replacement
  STD_list<STD_string>::const_iterator exclude_begin,exclude_end;
  if(exclude) {
    exclude_begin=exclude->begin();
    exclude_end=exclude->end();
  }

  for(constiter rh_it=rhs.get_const_begin(); rh_it!=rhs.get_const_end(); ++rh_it) {
    STD_string parlabel((*rh_it)->get_label());
    my_it=ldr_exists(parlabel);

    if( exclude && (STD_find(exclude_begin, exclude_end, parlabel) != exclude_end ) ) {
      ODINLOG(odinlog,normalDebug) << "Excluding " << parlabel << " from comparison" << STD_endl;
    } else {

      if(my_it==get_const_end()) return true; // 'this' does not include all pars of 'rhs' -> it is 'smaller'

      // check for type
      STD_string mytype=(*my_it)->get_typeInfo();
      STD_string rhtype=(*rh_it)->get_typeInfo();
      ODINLOG(odinlog,normalDebug) << "mytype/rhtype(" << parlabel << ")=" << mytype << "/" << rhtype << STD_endl;
      if( mytype!=rhtype ) return mytype<rhtype;


      // Cast to double/float if possible
      double myfloat=0.0;
      double rhfloat=0.0;
      float* dumfloat=0;
      double* dumdouble=0;
      bool my_is_float=false;
      bool rh_is_float=false;

      if( (dumfloat=(*my_it)->cast(dumfloat)) )   {myfloat=(*dumfloat);  my_is_float=true;}
      if( (dumdouble=(*my_it)->cast(dumdouble)) ) {myfloat=(*dumdouble); my_is_float=true;}
      if( (dumfloat=(*rh_it)->cast(dumfloat)) )   {rhfloat=(*dumfloat);  rh_is_float=true;}
      if( (dumdouble=(*rh_it)->cast(dumdouble)) ) {rhfloat=(*dumdouble); rh_is_float=true;}

      ODINLOG(odinlog,normalDebug) << "my_is_float/rh_is_float(" << parlabel << ")=" << my_is_float << "/" << rh_is_float << STD_endl;

      if(my_is_float && rh_is_float) {
        ODINLOG(odinlog,normalDebug) << "myfloat/rhfloat(" << parlabel << ")=" << myfloat << "/" << rhfloat << STD_endl;

        if(fabs(myfloat-rhfloat)>accuracy) { // Otherwise, values are considered equal
          ODINLOG(odinlog,normalDebug) << "Unequal myfloat/rhfloat(" << parlabel << ")=" << myfloat << "/" << rhfloat << STD_endl;
          return myfloat<rhfloat;
        }
      } else {

        // compare value by string
        STD_string myval=(*my_it)->printvalstring();
        STD_string rhval=(*rh_it)->printvalstring();
        ODINLOG(odinlog,normalDebug) << "myval/rhval(" << parlabel << ")=" << myval << "/" << rhval << STD_endl;
        if( myval!=rhval) {
          ODINLOG(odinlog,normalDebug) << "Unequal myval/rhval(" << parlabel << ")=" << myval << "/" << rhval << STD_endl;
          return myval<rhval;
        }
      }
    }
  }

  return false; // all equal
}


JcampDxBlock& JcampDxBlock::append_member(JcampDxClass& ldr,const STD_string ldrlabel) {
  Log<JcampDx> odinlog(this,"append_member");
  if(ldrlabel!=STD_string("")) ldr.set_label(ldrlabel);
  append(ldr);
  return *this;
}



#ifndef NO_CMDLINE
JcampDxBlock& JcampDxBlock::parse_cmdline_options(int argc, char *argv[], bool modify) {
  char optval[ODIN_MAXCHAR];
  for(constiter ldr_it=get_const_begin(); ldr_it!=get_const_end(); ++ldr_it) {
    STD_string opt((*ldr_it)->get_cmdline_option());
    if(opt!="") {
      STD_string optstr("-"+opt);
      bool* booldummy=(*ldr_it)->cast(booldummy=NULL);
      if(booldummy) {
        if(isCommandlineOption(argc,argv,optstr.c_str())) (*booldummy)=true;
      } else {
        if(getCommandlineOption(argc,argv,optstr.c_str(),optval,ODIN_MAXCHAR,modify)) (*ldr_it)->parsevalstring(optval);
      }
    }
  }
  return *this;
}

STD_map<STD_string,STD_string> JcampDxBlock::get_cmdline_options() const {
  STD_map<STD_string,STD_string> result;
  for(constiter ldr_it=get_const_begin(); ldr_it!=get_const_end(); ++ldr_it) {
    STD_string opt((*ldr_it)->get_cmdline_option());
    if(opt!="") {
      STD_string descr=(*ldr_it)->get_description();

      STD_string unit=(*ldr_it)->get_unit();
      if(unit!="") descr+=" ["+unit+"]";

      // Exclude default value for Boolean types
      STD_string defval;
      bool* booldummy=(*ldr_it)->cast(booldummy=0);
      if(!booldummy) defval=(*ldr_it)->printvalstring();

      svector alt=(*ldr_it)->get_alternatives();

      if(defval!="" || alt.size()) {
        descr+=" (";
        if(alt.size()) descr+="options="+tokenstring(alt,0)+", ";
        if(defval!="") descr+="default="+defval+unit;
        descr+=")";
      }

      result[opt]=descr;
    }
  }
  return result;
}


STD_string JcampDxBlock::get_cmdline_usage(const STD_string& lineprefix) const {
  STD_string result;
  STD_map<STD_string,STD_string> optmap=get_cmdline_options();
  for(STD_map<STD_string,STD_string>::const_iterator it=optmap.begin(); it!=optmap.end(); ++it) {
    result += lineprefix+"-"+it->first+": "+it->second+"\n";
  }
  return result;
}
#endif


void JcampDxBlock::init_static() {
  Log<JcampDx> odinlog("JcampDxBlock","init_static");
  set_c_locale();
}


void JcampDxBlock::destroy_static() {
  Log<JcampDx> odinlog("JcampDxBlock","destroy_static");

}



EMPTY_TEMPL_LIST bool StaticHandler<JcampDxBlock>::staticdone=false;


// Template instantiations
template class ListItem<JcampDxClass>;
template class List<JcampDxClass,JcampDxClass*,JcampDxClass&>;
