/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "VirtualMachine_p.h"

// JIT
#include <llvm/ExecutionEngine/ExecutionEngine.h>
#include <llvm/Module.h>
#include <llvm/DerivedTypes.h>
#include <llvm/Target/TargetOptions.h>
#include <llvm/LLVMContext.h>
#include <llvm/Support/CommandLine.h>
#include <llvm/Support/Threading.h>
#include <llvm/Support/ErrorHandling.h>
#include <llvm/Support/TargetSelect.h>

#include "Function_p.h"
#include "Value.h"

#include "Debug.h"

#include "Macros_p.h"

using namespace GTLCore;

struct VirtualMachine::Private {
  llvm::ExecutionEngine* executionEngine;
  static VirtualMachine* virtualMachine;
  static int optimLevel;
};

VirtualMachine* VirtualMachine::Private::virtualMachine = 0;
int VirtualMachine::Private::optimLevel = 2;

void gtl_llvm_error_handler(void *user_data, const std::string& reason)
{
  GTL_ABORT(reason);
}


VirtualMachine::VirtualMachine() : d(new Private)
{
  d->executionEngine = 0;
  
#ifdef LLVM_27
  llvm::llvm_install_error_handler(&gtl_llvm_error_handler);
#else
  llvm::install_fatal_error_handler(&gtl_llvm_error_handler);
#endif
  
  llvm::llvm_start_multithreaded();

  std::string errorMessage;
  llvm::LLVMContext& dummyContext = llvm::getGlobalContext(); // TODO someday (llvm 2.7 ?) use own context
  llvm::InitializeNativeTarget();
  
  llvm::CodeGenOpt::Level OLvl = llvm::CodeGenOpt::Default;
  switch (Private::optimLevel) {
    case 0: OLvl = llvm::CodeGenOpt::None; break;
    case 3: OLvl = llvm::CodeGenOpt::Aggressive; break;
    case 1:
    case 2:
    default:
      OLvl = llvm::CodeGenOpt::Default; break;
  }
  
  llvm::EngineBuilder builder( new llvm::Module("dummy", dummyContext));
#ifdef OPENGTL_32_BITS
  std::list<std::string> attrs;
  attrs.push_back("-mmx"); // http://llvm.org/bugs/show_bug.cgi?id=3287
  builder.setMAttrs(attrs);
#endif
  builder.setOptLevel(OLvl);
  builder.setEngineKind(llvm::EngineKind::JIT);
  builder.setErrorStr(&errorMessage);
  d->executionEngine = builder.create();
  GTL_DEBUG("Error while creating execution engine : " << errorMessage);
  GTL_ASSERT(d->executionEngine);
}

VirtualMachine::~VirtualMachine()
{
  llvm::llvm_stop_multithreaded();
#ifndef LLVM_27_OR_28
  delete d->executionEngine;
#endif
  delete d;
}

STATIC_DELETER(DeleteVirtualMachine)
{
  delete VirtualMachine::Private::virtualMachine;
}

void VirtualMachine::registerModule( llvm::Module* mp)
{
  GTL_ASSERT(d->executionEngine);
  d->executionEngine->addModule(mp);
}

void VirtualMachine::unregisterModule( llvm::Module* module)
{
  if(d->executionEngine)
  {
    d->executionEngine->removeModule(module);
    for( llvm::Module::FunctionListType::iterator it = module->getFunctionList().begin();
         it != module->getFunctionList().end(); ++it)
    {
//      d->executionEngine->freeMachineCodeForFunction(it);
      d->executionEngine->updateGlobalMapping( it, 0);
    }
    for( llvm::Module::GlobalListType::iterator it = module->getGlobalList().begin();
         it != module->getGlobalList().end(); ++it)
    {
      d->executionEngine->updateGlobalMapping( it, 0);
    }
  }
}

void *VirtualMachine::getPointerToFunction(llvm::Function *F)
{
  return d->executionEngine->getPointerToFunction(F);
}

void *VirtualMachine::getPointerToFunction(Function *F)
{
  return getPointerToFunction( F->d->data->function( ) );
}

void* VirtualMachine::getGlobalVariablePointer( const llvm::GlobalVariable* _pointer)
{
    return d->executionEngine->getPointerToGlobal( _pointer );
}

#if 0
void VirtualMachine::setGlobalVariable( llvm::GlobalVariable *GV, const GTLCore::Value& value)
{
  void* ptr = d->executionEngine->getPointerToGlobal( GV );
  if( GV->getType()->getElementType() == llvm::Type::Int32Ty )
  {
    *(int*)ptr = value.asInt32();
  } else if( GV->getType()->getElementType() == llvm::Type::Int1Ty )
  {
    *(bool*)ptr = value.asBoolean();
  } else if( GV->getType()->getElementType() == llvm::Type::FloatTy )
  {
    GTL_DEBUG("Set " << value.asFloat() << " on ptr " << ptr << " from value = " << *(float*)ptr);
    *(float*)ptr = value.asFloat();
    GTL_DEBUG( *(float*)ptr );
  }
}

GTLCore::Value VirtualMachine::getGlobalVariable( llvm::GlobalVariable *GV)
{
  void* ptr = d->executionEngine->getPointerToGlobal( GV );
  if( GV->getType()->getElementType() == llvm::Type::Int32Ty )
  {
    return GTLCore::Value( *(int*)ptr);
  }
  if( GV->getType()->getElementType() == llvm::Type::Int1Ty )
  {
    return GTLCore::Value( *(bool*)ptr);
  }
  if( GV->getType()->getElementType() == llvm::Type::FloatTy )
  {
    return GTLCore::Value( *(float*)ptr);
  }
  return GTLCore::Value();
}
#endif

VirtualMachine* VirtualMachine::instance()
{
  if( not Private::virtualMachine)
  {
    Private::virtualMachine = new VirtualMachine;
  }
  return Private::virtualMachine;
}

void VirtualMachine::setOptimizationLevel(int v)
{
  GTL_ASSERT(Private::virtualMachine == 0);
  Private::optimLevel = v;
}

llvm::ExecutionEngine* VirtualMachine::executionEngine()
{
  return d->executionEngine;
}
