//                                               -*- C++ -*-
/**
 *  @file  PiecewiseLinearEvaluationImplementation.cxx
 *  @brief The evaluation part of a linear piecewise scalar function
 *
 *  Copyright (C) 2005-2013 EDF-EADS-Phimeca
 *
 *  This library is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 *  @author dutka
 *  @date   2008-05-21 11:21:38 +0200 (Wed, 21 May 2008)
 */
#include <algorithm>
#include "PiecewiseLinearEvaluationImplementation.hxx"
#include "OSS.hxx"
#include "PersistentObjectFactory.hxx"
#include "Description.hxx"
#include "Exception.hxx"

BEGIN_NAMESPACE_OPENTURNS

CLASSNAMEINIT(PiecewiseLinearEvaluationImplementation);

static Factory<PiecewiseLinearEvaluationImplementation> RegisteredFactory("PiecewiseLinearEvaluationImplementation");


/* Default constructor */
PiecewiseLinearEvaluationImplementation::PiecewiseLinearEvaluationImplementation()
  : NumericalMathEvaluationImplementation()
  , locations_(0)
  , values_(0, 0)
{
  // Nothing to do
}


/* Parameters constructor */
PiecewiseLinearEvaluationImplementation::PiecewiseLinearEvaluationImplementation(const NumericalPoint & locations,
    const NumericalPoint & values)
  : NumericalMathEvaluationImplementation()
  , locations_(0)
  , values_(0, 0)
{
  // Convert the values into a sample
  const UnsignedLong size(values.getSize());
  NumericalSample sampleValues(size, 1);
  for (UnsignedLong i = 0; i < size; ++i) sampleValues[i][0] = values[i];
  // Check the input
  setLocationsAndValues(locations, sampleValues);
}

/* Parameters constructor */
PiecewiseLinearEvaluationImplementation::PiecewiseLinearEvaluationImplementation(const NumericalPoint & locations,
    const NumericalSample & values)
  : NumericalMathEvaluationImplementation()
  , locations_(0)
  , values_(0, values.getDimension())
{
  setLocationsAndValues(locations, values);
}


/* Virtual constructor */
PiecewiseLinearEvaluationImplementation * PiecewiseLinearEvaluationImplementation::clone() const
{
  return new PiecewiseLinearEvaluationImplementation(*this);
}


/* String converter */
String PiecewiseLinearEvaluationImplementation::__repr__() const
{
  return OSS() << "class=" << GetClassName()
         << " locations=" << locations_
         << " values=" << values_;
}

String PiecewiseLinearEvaluationImplementation::__str__(const String & offset) const
{
  return OSS() << offset << __repr__();
}


/* Evaluation operator */
NumericalPoint PiecewiseLinearEvaluationImplementation::operator () (const NumericalPoint & inP) const
{
  if (inP.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: expected an input point of dimension 1, got dimension=" << inP.getDimension();
  const NumericalScalar x(inP[0]);
  UnsignedLong iLeft(0);
  if (x <= locations_[iLeft]) return values_[iLeft];
  UnsignedLong iRight(locations_.getSize() - 1);
  if (x >= locations_[iRight]) return values_[iRight];
  // Find the segment containing x by bisection
  while (iRight - iLeft > 1)
  {
    const UnsignedLong iMiddle((iRight + iLeft) / 2);
    if (x < locations_[iMiddle]) iRight = iMiddle;
    else iLeft = iMiddle;
  }
  const NumericalScalar xLeft(locations_[iLeft]);
  const NumericalScalar xRight(locations_[iRight]);
  const NumericalScalar dx(xLeft - xRight);
  const NumericalPoint vLeft(values_[iLeft]);
  const NumericalPoint vRight(values_[iRight]);
  const UnsignedLong dimension(getOutputDimension());
  NumericalPoint value(dimension);
  const NumericalScalar alpha((x - xRight) / dx);
  const NumericalScalar beta((xLeft - x) / dx);
  for (UnsignedLong i = 0; i < dimension; ++i) value[i] = alpha * vLeft[i] + beta * vRight[i];
  return value;
}

/* Locations accessor */
NumericalPoint PiecewiseLinearEvaluationImplementation::getLocations() const
{
  return locations_;
}

void PiecewiseLinearEvaluationImplementation::setLocations(const NumericalPoint & locations)
{
  if (locations.getDimension() != values_.getDimension()) throw InvalidArgumentException(HERE) << "Error: the number of locations=" << locations.getDimension() << " must match the number of previously set values=" << values_.getDimension();
  locations_ = locations;
  std::stable_sort(locations_.begin(), locations_.end());
}

/* Values accessor */
NumericalSample PiecewiseLinearEvaluationImplementation::getValues() const
{
  return values_;
}

void PiecewiseLinearEvaluationImplementation::setValues(const NumericalPoint & values)
{
  const UnsignedLong size(values.getSize());
  if (size != locations_.getSize()) throw InvalidArgumentException(HERE) << "Error: the number of values=" << size << " must match the number of previously set locations=" << locations_.getSize();
  NumericalSample sampleValues(size, 1);
  for (UnsignedLong i = 0; i < size; ++i) sampleValues[i][0] = values[i];
  values_ = sampleValues;
}

void PiecewiseLinearEvaluationImplementation::setValues(const NumericalSample & values)
{
  if (values.getSize() != locations_.getSize()) throw InvalidArgumentException(HERE) << "Error: the number of values=" << values.getSize() << " must match the number of previously set locations=" << locations_.getSize();
  values_ = values;
}

void PiecewiseLinearEvaluationImplementation::setLocationsAndValues(const NumericalPoint & locations,
    const NumericalSample & values)
{
  const UnsignedLong size(locations.getSize());
  if (size != values.getSize()) throw InvalidArgumentException(HERE) << "Error: the number of values=" << values.getSize() << " must match the number of locations=" << size;
  // Sort the data in increasing order according to the locations
  const UnsignedLong dimension(values.getDimension());
  NumericalSample data(size, 1 + dimension);
  for (UnsignedLong i = 0; i < size; ++i)
  {
    data[i][0] = locations[i];
    for (UnsignedLong j = 0; j < dimension; ++j)
      data[i][j + 1] = values[i][j];
  }
  data = data.sortAccordingToAComponent(0);
  locations_ = NumericalPoint(size);
  values_ = NumericalSample(size, dimension);
  for (UnsignedLong i = 0; i < size; ++i)
  {
    locations_[i] = data[i][0];
    for (UnsignedLong j = 0; j < dimension; ++j)
      values_[i][j] = data[i][j + 1];
  }
}

/* Input dimension accessor */
UnsignedLong PiecewiseLinearEvaluationImplementation::getInputDimension() const
{
  return 1;
}

/* Output dimension accessor */
UnsignedLong PiecewiseLinearEvaluationImplementation::getOutputDimension() const
{
  return values_.getDimension();
}


/* Method save() stores the object through the StorageManager */
void PiecewiseLinearEvaluationImplementation::save(Advocate & adv) const
{
  NumericalMathEvaluationImplementation::save(adv);
  adv.saveAttribute( "locations_", locations_ );
  adv.saveAttribute( "values_", values_ );
}


/* Method load() reloads the object from the StorageManager */
void PiecewiseLinearEvaluationImplementation::load(Advocate & adv)
{
  NumericalMathEvaluationImplementation::load(adv);
  adv.loadAttribute( "locations_", locations_ );
  adv.loadAttribute( "values_", values_ );
}


END_NAMESPACE_OPENTURNS
