///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/animation/controller/StandardControllers.h>
#include <core/gui/properties/FloatControllerUI.h>
#include <core/gui/properties/BooleanPropertyUI.h>
#include <core/undo/UndoManager.h>

#include "PositionDataChannel.h"
#include "AtomTypeDataChannel.h"
#include <atomviz/atoms/AtomsObject.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(PositionDataChannel, DataChannel)
DEFINE_FLAGS_REFERENCE_FIELD(PositionDataChannel, FloatController, "GlobalAtomRadiusScale", PROPERTY_FIELD_ALWAYS_DEEP_COPY, _globalAtomRadiusScale)
DEFINE_PROPERTY_FIELD(PositionDataChannel, "HighQualityRenderingInViewports", _useHighQualityRenderingInViewports)
DEFINE_PROPERTY_FIELD(PositionDataChannel, "FlatAtomRendering", _flatAtomRendering)
SET_PROPERTY_FIELD_LABEL(PositionDataChannel, _globalAtomRadiusScale, "Atom radius")
SET_PROPERTY_FIELD_LABEL(PositionDataChannel, _useHighQualityRenderingInViewports, "High-quality rendering in viewports")
SET_PROPERTY_FIELD_LABEL(PositionDataChannel, _flatAtomRendering, "Flat atoms")

/******************************************************************************
* Serialization constructor.
******************************************************************************/
PositionDataChannel::PositionDataChannel(bool isLoading) : DataChannel(isLoading),
	renderBufferValidity(TimeNever), _useHighQualityRenderingInViewports(false),
	_flatAtomRendering(false)
{
	INIT_PROPERTY_FIELD(PositionDataChannel, _globalAtomRadiusScale);
	INIT_PROPERTY_FIELD(PositionDataChannel, _useHighQualityRenderingInViewports);
	INIT_PROPERTY_FIELD(PositionDataChannel, _flatAtomRendering);
	if(!isLoading) {
		// Set default global atom scaling.
		setGlobalAtomRadiusScaleController(CONTROLLER_MANAGER.createDefaultController<FloatController>());
		setGlobalAtomRadiusScale(1.0f);
	}
}


/******************************************************************************
* Special constructor to create a standard data channel.
******************************************************************************/
PositionDataChannel::PositionDataChannel(DataChannelIdentifier which) : DataChannel(which),
	renderBufferValidity(TimeNever), _useHighQualityRenderingInViewports(false),
	_flatAtomRendering(false)
{
	INIT_PROPERTY_FIELD(PositionDataChannel, _globalAtomRadiusScale);
	INIT_PROPERTY_FIELD(PositionDataChannel, _useHighQualityRenderingInViewports);
	INIT_PROPERTY_FIELD(PositionDataChannel, _flatAtomRendering);
	// Set default global atom scaling.
	setGlobalAtomRadiusScaleController(CONTROLLER_MANAGER.createDefaultController<FloatController>());
	setGlobalAtomRadiusScale(1.0f);
}

/******************************************************************************
* Render the atoms into the viewport.
******************************************************************************/
void PositionDataChannel::render(TimeTicks time, Viewport* vp, AtomsObject* atoms, ObjectNode* contextNode)
{
	try {
		// Update the rendering buffer for atoms if it is not valid.
		if(_useHighQualityRenderingInViewports)
			renderBuffer.prepare(vp, _flatAtomRendering, AtomsRenderer::DEFAULT_HQ_RENDERING_METHOD);
		else
			renderBuffer.prepare(vp, _flatAtomRendering);
		if(!renderBufferValidity.contains(time) || !renderBuffer.isFilled()) {
			renderBufferValidity.setInfinite();
			if(!fillRenderBuffer(time, atoms, renderBuffer, renderBufferValidity)) return;

			// Rendering buffer should be valid now.
			OVITO_ASSERT(renderBufferValidity.contains(time));
		}
		renderBuffer.render(vp);
	}
	catch(const Exception& ex) {
		ex.logError();
	}
}

/******************************************************************************
* Renders the channel's contents in high-quality mode to an offscreen buffer.
******************************************************************************/
void PositionDataChannel::renderHQ(TimeTicks time, AtomsObject* atoms, const CameraViewDescription& view, ObjectNode* contextNode, int imageWidth, int imageHeight, Window3D* glcontext)
{
	AtomsRenderer arb;
	arb.prepare(glcontext, _flatAtomRendering, AtomsRenderer::DEFAULT_HQ_RENDERING_METHOD);
	TimeInterval validity;
	if(!fillRenderBuffer(time, atoms, arb, validity))
		return;

	arb.renderOffscreen(view.isPerspective, view.projectionMatrix, QSize(imageWidth, imageHeight));
}

/******************************************************************************
* Lets the channel clear all its internal caches.
******************************************************************************/
void PositionDataChannel::clearCaches()
{
	DataChannel::clearCaches();
	renderBufferValidity.setEmpty();
}

/******************************************************************************
* Creates a copy of this object.
******************************************************************************/
RefTarget::SmartPtr PositionDataChannel::clone(bool deepCopy, CloneHelper& cloneHelper)
{
	// Let the base class create an instance of this class.
	PositionDataChannel::SmartPtr clone = static_object_cast<PositionDataChannel>(DataChannel::clone(deepCopy, cloneHelper));

	// Clear render cache of original data channel, as it is probably no longer needed.
	clearCaches();
	renderBuffer.reset();

	return clone;
}

/******************************************************************************
* Fills the render buffer with atom data.
******************************************************************************/
bool PositionDataChannel::fillRenderBuffer(TimeTicks time, AtomsObject* atoms, AtomsRenderer& renderBuffer, TimeInterval& renderBufferValidity)
{
	PositionDataChannel* posChannel = this;
	DataChannel* colorChannel = atoms->getStandardDataChannel(DataChannel::ColorChannel);
	AtomTypeDataChannel* typeChannel = static_object_cast<AtomTypeDataChannel>(atoms->getStandardDataChannel(DataChannel::AtomTypeChannel));
	DataChannel* radiusChannel = atoms->getStandardDataChannel(DataChannel::RadiusChannel);
	DataChannel* selectionChannel = atoms->getStandardDataChannel(DataChannel::SelectionChannel);

	// Setup the source for the atom positions.
	if(!posChannel->size()) {
		renderBuffer.beginAtoms(0);
		renderBuffer.endAtoms();
		return false;
	}

	FloatType globalRadiusScaling = 1;
	if(globalAtomRadiusScaleController()) {
		globalAtomRadiusScaleController()->getValue(time, globalRadiusScaling, renderBufferValidity);
		OVITO_ASSERT(globalRadiusScaling >= 0);
	}

	AffineTransformation simCell = atoms->simulationCell()->cellMatrix();

	renderBuffer.beginAtoms(atoms->atomsCount());

	const Point3* posIter = posChannel->constDataPoint3();
	const int* typeIter = NULL;
	if(typeChannel && typeChannel->isVisible())
		typeIter = typeChannel->constDataInt();

	// Setup the source for the atoms radii. This can be either the radius data channel that
	// specifies the radius for each atom or the radius parameter of the atom's type.
	FloatType radius = globalRadiusScaling;
	const FloatType* radiusIter = NULL;
	QVector<FloatType> typeRadii;
	if(radiusChannel && radiusChannel->isVisible()) {
		radiusIter = radiusChannel->constDataFloat();
	}
	else if(typeIter) {
		// Get the atom type radii.
		typeRadii.insert(0, typeChannel->atomTypes().size(), 1);
		for(int i=0; i<typeChannel->atomTypes().size(); i++) {
			AtomType* atype = typeChannel->atomTypes()[i];
			if(atype && atype->radiusController()) {
				atype->radiusController()->getValue(time, typeRadii[i], renderBufferValidity);
				typeRadii[i] *= globalRadiusScaling;
			}
		}
	}

	// Setup the source for colors. This can be either the color channel that
	// specifies an individual color for each atom or the colors are derived from the atom type.
	// In this case the color assigned to the atom type is used. Finally the selection
	// channel can be visible. In this case the color of all selected a	toms is overridden with the selection color.
	const Vector3* colorIter = NULL;
	const int* selectionIter = NULL;

	Color selectionColor = Color(1,0,0);
	QVector<Color> typeColors;
	if(colorChannel && colorChannel->isVisible()) {
		colorIter = colorChannel->constDataVector3();
	}
	else if(typeIter) {
		// Get the atom type colors.
		typeColors.insert(0, typeChannel->atomTypes().size(), Color(1,1,1));
		for(int i=0; i<typeChannel->atomTypes().size(); i++) {
			AtomType* atype = typeChannel->atomTypes()[i];
			if(atype && atype->colorController())
				atype->colorController()->getValue(time, typeColors[i], renderBufferValidity);
		}
	}
	if(selectionChannel && selectionChannel->isVisible()) {
		selectionIter = selectionChannel->constDataInt();
	}

	int nAtoms = atoms->atomsCount();

	while(nAtoms--) {
		Color color;
		if(colorIter) color = *colorIter++;
		else if(!typeColors.empty()) color = typeColors[(*typeIter) % typeColors.size()];
		else color = Color(1,1,1);
		if(selectionIter && *selectionIter++) color = selectionColor;
		if(radiusIter) radius = *radiusIter++;
		else if(!typeRadii.empty()) radius = typeRadii[(*typeIter) % typeRadii.size()];
		renderBuffer.specifyAtom(*posIter,
			(GLubyte)(min(color.r,(FloatType)1)*255.0), (GLubyte)(min(color.g,(FloatType)1)*255.0), (GLubyte)(min(color.b,(FloatType)1)*255.0), radius);
		++posIter;
		if(typeIter) ++typeIter;
	}

	renderBuffer.endAtoms();

	return true;
}

/******************************************************************************
* Returns the bounding box of the channel's geometry when rendered in the viewports.
******************************************************************************/
Box3 PositionDataChannel::boundingBox(TimeTicks time, AtomsObject* atoms, ObjectNode* contextNode, TimeInterval& validityInterval)
{
	Box3 bb;
	if(size() == 0) return bb;

	// Compute bounding box of all the atoms.
	bb.addPoints(constDataPoint3(), size());

	// Add maximum atoms radius
	FloatType globalRadiusScaling = 1;
	if(globalAtomRadiusScaleController()) {
		globalAtomRadiusScaleController()->getValue(time, globalRadiusScaling, validityInterval);
		OVITO_ASSERT(globalRadiusScaling >= 0);
	}
	FloatType maximumRadius = 0;
	const FloatType* radiusIter = NULL;
	DataChannel* radiusChannel = atoms->getStandardDataChannel(DataChannel::RadiusChannel);
	AtomTypeDataChannel* typeChannel = static_object_cast<AtomTypeDataChannel>(atoms->getStandardDataChannel(DataChannel::AtomTypeChannel));
	if(radiusChannel && radiusChannel->isVisible() && radiusChannel->size() != 0) {
		maximumRadius = *max_element(radiusChannel->constDataFloat(), radiusChannel->constDataFloat() + radiusChannel->size()) * globalRadiusScaling;
	}
	else if(typeChannel) {
		for(int i=0; i<typeChannel->atomTypes().size(); i++) {
			AtomType* atype = typeChannel->atomTypes()[i];
			if(atype && atype->radiusController()) {
				FloatType typeRadius;
				atype->radiusController()->getValue(time, typeRadius, validityInterval);
				maximumRadius = max(maximumRadius, typeRadius * globalRadiusScaling);
			}
		}
	}
	else maximumRadius = globalRadiusScaling;
	return bb.padBox(maximumRadius);
}

IMPLEMENT_PLUGIN_CLASS(PositionDataChannelEditor, PropertiesEditor)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void PositionDataChannelEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create a rollout.
	QWidget* rollout = createRollout(tr("Position Data Channel"), rolloutParams, "atomviz.data_channels.position.parameters");

    // Create the rollout contents.
	QGridLayout* layout = new QGridLayout(rollout);
	layout->setContentsMargins(4,4,4,4);
#ifndef Q_WS_MAC
	layout->setSpacing(0);
#endif
	layout->setColumnStretch(1, 1);

	BooleanPropertyUI* showAtomsUI = new BooleanPropertyUI(this, "isVisible", tr("Show atoms"));
	layout->addWidget(showAtomsUI->checkBox(), 0, 0, 1, 3);

	BooleanPropertyUI* highQualityDisplayUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(PositionDataChannel, _useHighQualityRenderingInViewports));
	layout->addWidget(highQualityDisplayUI->checkBox(), 1, 0, 1, 3);

	BooleanPropertyUI* flatAtomDisplayUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(PositionDataChannel, _flatAtomRendering));
	layout->addWidget(flatAtomDisplayUI->checkBox(), 2, 0, 1, 3);

	// Radius scaling parameter.
	FloatControllerUI* globalRadiusScaleUI = new FloatControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(PositionDataChannel, _globalAtomRadiusScale));
	layout->addWidget(globalRadiusScaleUI->label(), 3, 0);
	layout->addLayout(globalRadiusScaleUI->createFieldLayout(), 3, 1);
	globalRadiusScaleUI->setMinValue(0);
}


};	// End of namespace AtomViz
