#!/bin/bash -e
# Copyright (C) 2022 Simo Sorce <simo@redhat.com>
# SPDX-License-Identifier: Apache-2.0

: "${TESTBLDDIR=.}"

title()
{
    case "$1" in
    "SECTION")
        shift 1
        echo "########################################"
        echo "## $*"
        echo ""
        ;;
    "ENDSECTION")
        echo ""
        echo "                                      ##"
        echo "########################################"
        echo ""
        ;;
    "PARA")
        shift 1
        echo ""
        echo "## $*"
        if [ -f "${PPDBGFILE}" ]; then
            echo "[TEST]: $*" >> "${PPDBGFILE}"
        fi
        ;;
    "LINE")
        shift 1
        echo "$*"
        ;;
    *)
        echo "$*"
        ;;
    esac
}

cleanup_server()
{
    echo "killing $1 server"
    kill -9 -- "$2"
}

helper_emit=1

ossl()
{
    helper_output=""
    echo "# r $1" >> "${TMPPDIR}/gdb-commands.txt"
    echo "$CHECKER openssl $1"
    # shellcheck disable=SC2086  # this is intentionally split by words
    __out=$(eval $CHECKER openssl $1)
    __res=$?
    if [ "${2:-0}" -eq "$helper_emit" ]; then
        # shellcheck disable=SC2034  # used externally by caller
        helper_output="$__out"
    else
        echo "$__out"
    fi
    return $__res
}

gen_unsetvars() {
    grep "^export" "${TMPPDIR}/testvars" \
    | sed -e 's/export/unset/' -e 's/=.*$//' \
    >> "${TMPPDIR}/unsetvars"
}

kill_children() {
    # make sure it is killed before we continue
    jobs -p | xargs -r kill -9 || true
}

# macOS uses BSD sed, which expects the argument after -i (with a space after
# it!) to be the backup suffix, while GNU sed expects a potential backup suffix
# directly after -i and interprets -i <expression> as in-place editing with no
# backup.
#
# Use "${sed_inplace[@]}" to make that work transparently by setting it to the
# arguments required to achieve in-place editing without backups depending on
# the version of sed.
if sed --version 2>/dev/null | grep -q 'GNU sed'; then
	export sed_inplace=("-i")
else
	export sed_inplace=("-i" "")
fi

# Generate self-signed certificate using OpenSSL
# This is useful for the RSA-PSS certificates that can not be generated by
# certtool
selfsign_cert() {
    KEY="$1"
    export CERT="$2"
    shift 2
    export AARGS="$*"

    TMPCA=${TMPPDIR}/tmpca
    TMP_OPENSSL_CONF=${OPENSSL_CONF}
    sed -e "s|^dir .*|dir = ${TMPCA}|" \
        "${OPENSSL_CONF}" > "${OPENSSL_CONF}.tmpca"
    export OPENSSL_CONF=${OPENSSL_CONF}.tmpca

    mkdir -p "${TMPCA}/newcerts" "${TMPCA}/private"
    if [ ! -e "${TMPCA}/serial" ]; then
        echo "01" > "${TMPCA}/serial"
    fi
    touch "${TMPCA}/index.txt"

    title PARA "Generating a new selfsigned certificate for ${KEY}"
    ossl 'req -batch -noenc -x509 -new -key ${KEY} ${AARGS} -out ${CERT}'
    echo "$helper_output"
    OPENSSL_CONF=${TMP_OPENSSL_CONF}
}
