/*
 *  PlanFacile (Easy plan, in french) is a small tool to help people to
 *  write a document on a particular subject.
 *  Copyright (C) 2005  Julien BRUGUIER
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "localisationfichier.h"

static Resultat localisationfichier_copieinterne(LocalisationFichier *original , LocalisationFichier **copie)
{
	/* Ralise une copie de la liste des positions.
	 * Renvoie RESULTAT_ERREUR_MEMOIRE si un allocation choue.
	 */
	DEBUTZONESECURISEE
	if(original==NULL)
		return RESULTAT_OK;
	SECURISE(localisationfichier_copieinterne(original->inclusion,copie));
	SECURISE(localisationfichier_ajout(copie,original->nom,original->nomabsolu,original->descripteur,original->position));
	FINZONESECURISEE
	return RESULTAT_OK;
}

static Resultat localisationfichier_copieinterne_commande(LocalisationFichier *original , LocalisationFichier **copie)
{
	/* Ralise une copie de la liste des positions.
	 * Renvoie RESULTAT_ERREUR_MEMOIRE si un allocation choue.
	 */
	DEBUTZONESECURISEE
	if(original==NULL)
		return RESULTAT_OK;
	SECURISE(localisationfichier_copieinterne(original->inclusion,copie));
	SECURISE(localisationfichier_ajout(copie,original->nom,NULL,NULL,original->position));
	FINZONESECURISEE
	return RESULTAT_OK;
}

Resultat localisationfichier_initialisation(LocalisationFichier **localisationfichier)
{
	/* Cre une localisation de fichier vide.
	 */
	*localisationfichier=NULL;
	return RESULTAT_OK;
}

Resultat localisationfichier_ajout(LocalisationFichier **localisationfichier , NomFichier nom , NomFichier nomabsolu , DescripteurFichier descripteur , PositionFichier position)
{
	/* Ajoute une localisation de fichier.
	 * Utilis lors de l'inclusion d'un fichier.
	 * Renvoie RESULTAT_ERREUR_MEMOIRE si l'allocation choue.
	 */
	LocalisationFichier *fichierinclus;
	if((fichierinclus=(LocalisationFichier*)(malloc(sizeof(LocalisationFichier))))==NULL)
		return RESULTAT_ERREUR_MEMOIRE;
	fichierinclus->descripteur=descripteur;
	fichierinclus->position=position;
	if((fichierinclus->nom=(NomFichier)(strdup((char*)(nom))))==NULL)
	{
		free(fichierinclus);
		return RESULTAT_ERREUR_MEMOIRE;
	}
	if(nomabsolu==NULL)
		fichierinclus->nomabsolu=NULL;
	else
		if((fichierinclus->nomabsolu=(NomFichier)(strdup((char*)(nomabsolu))))==NULL)
		{
			free(fichierinclus->nom);
			free(fichierinclus);
			return RESULTAT_ERREUR_MEMOIRE;
		}
	fichierinclus->inclusion=*localisationfichier;
	*localisationfichier=fichierinclus;
	return RESULTAT_OK;
}

Resultat localisationfichier_modification(LocalisationFichier *localisationfichier , PositionFichier position)
{
	/* Modifie la position dans le fichier courant.
	 */
	if(localisationfichier==NULL)
		return RESULTAT_ERREUR;
	localisationfichier->position=position;
	return RESULTAT_OK;
}

Resultat localisationfichier_retrait(LocalisationFichier **localisationfichier)
{
	/* Retire une localisation de fichier.
	 * Ne referme pas le fichier dsign par le descripteur.
	 * Utilis lors d'un retour de fichier inclus.
	 */
	LocalisationFichier *fichierinclus;
	if((*localisationfichier)==NULL)
		return RESULTAT_ERREUR;
	fichierinclus=*localisationfichier;
	*localisationfichier=fichierinclus->inclusion;
	free(fichierinclus->nom);
	if(fichierinclus->nomabsolu!=NULL)
		free(fichierinclus->nomabsolu);
	free(fichierinclus);
	return RESULTAT_OK;
}

Resultat localisationfichier_lecture_nom(LocalisationFichier *localisationfichier , NomFichier *nom)
{
	/* Lit le nom du fichier.
	 * Renvoie RESULTAT_ERREUR si localisationfichier est NULL.
	 */
	if(localisationfichier==NULL)
		return RESULTAT_ERREUR;
	*nom=localisationfichier->nom;
	return RESULTAT_OK;
}

Resultat localisationfichier_lecture_nomabsolu(LocalisationFichier *localisationfichier , NomFichier *nomabsolu)
{
	/* Lit le nom absolu du fichier.
	 * Renvoie RESULTAT_ERREUR si localisationfichier est NULL.
	 */
	if(localisationfichier==NULL)
		return RESULTAT_ERREUR;
	*nomabsolu=localisationfichier->nomabsolu;
	return RESULTAT_OK;
}

Resultat localisationfichier_lecture_descripteur(LocalisationFichier *localisationfichier , DescripteurFichier *descripteur)
{
	/* Lit le descripteur du dernier fichier inclus.
	 * Renvoie RESULTAT_ERREUR si localisationfichier est NULL.
	 */
	if(localisationfichier==NULL)
		return RESULTAT_ERREUR;
	*descripteur=localisationfichier->descripteur;
	return RESULTAT_OK;
}
Resultat localisationfichier_lecture_position(LocalisationFichier *localisationfichier , PositionFichier *position)
{
	/* Lit la position dans le dernier fichier inclus.
	 * Renvoie RESULTAT_ERREUR si localisationfichier est NULL.
	 */
	if(localisationfichier==NULL)
		return RESULTAT_ERREUR;
	*position=localisationfichier->position;
	return RESULTAT_OK;
}

Resultat localisationfichier_lecture_inclusion(LocalisationFichier *localisationfichier , LocalisationFichier **inclusion)
{
	/* Renvoie la position du fichier parent dans l'ordre d'inclusion.
	 * Renvoie RESULTAT_ERREUR si localisationfichier est NULL.
	 */
	if(localisationfichier==NULL)
		return RESULTAT_ERREUR;
	*inclusion=localisationfichier->inclusion;
	return RESULTAT_OK;
}

Resultat localisationfichier_vide(LocalisationFichier *localisationfichier , Booleen *vide)
{
	/* Indique si la localisationfichier est vide ou non.
	 */
	if(localisationfichier==NULL)
		*vide=VRAI;
	else
		*vide=FAUX;
	return RESULTAT_OK;
}

Resultat localisationfichier_test_fichier(LocalisationFichier *localisationfichier , NomFichier nomabsolu , Booleen *present)
{
	/* Indique si un nom de fichier est prsent dans la localisation
	 * de fichier, afin de savoir si on ne ralise pas une inclusion
	 * rcursive. Le rsultat est renvoy dans le boolen pass en
	 * paramtre.
	 */
	DEBUTZONESECURISEE
	if(localisationfichier==NULL)
	{
		*present=FAUX;
		return RESULTAT_OK;
	}
	SECURISE(localisationfichier_test_fichier(localisationfichier->inclusion,nomabsolu,present));
	if((*present)==VRAI)
		return RESULTAT_OK;
	if((nomabsolu==NULL)||(localisationfichier->nomabsolu==NULL))
		return RESULTAT_OK;
	if((strcmp((char*)(nomabsolu),(char*)(localisationfichier->nomabsolu)))==0)
		*present=VRAI;
	FINZONESECURISEE
	return RESULTAT_OK;
}

Resultat localisationfichier_copie(LocalisationFichier *localisationfichier , LocalisationFichier **copie)
{
	/* Effectue une copie d'une localisation de fichier.
	 * Renvoie RESULTAT_ERREUR_MEMOIRE si une allocation choue.
	 */
	DEBUTZONESECURISEE
	SECURISE(localisationfichier_destruction(copie));
	SECURISE(localisationfichier_initialisation(copie));
	SECURISE(localisationfichier_copieinterne(localisationfichier,copie));
	FINZONESECURISEE
	return RESULTAT_OK;
}

Resultat localisationfichier_copie_commande(LocalisationFichier *localisationfichier , LocalisationFichier **copie)
{
	/* Effectue une copie d'une localisation de fichier,
	 * en ne recopiant ni le descripteur de fichier, ni le nom
	 * absolu ventuel.
	 * Renvoie RESULTAT_ERREUR_MEMOIRE si une allocation choue.
	 */
	DEBUTZONESECURISEE
	SECURISE(localisationfichier_destruction(copie));
	SECURISE(localisationfichier_initialisation(copie));
	SECURISE(localisationfichier_copieinterne_commande(localisationfichier,copie));
	FINZONESECURISEE
	return RESULTAT_OK;
}

Resultat localisationfichier_destruction(LocalisationFichier **localisationfichier)
{
	/* Dtruit une localisation de fichier.
	 */
	DEBUTZONESECURISEE
	if((*localisationfichier)==NULL)
		return RESULTAT_OK;
	SECURISE(localisationfichier_destruction(&((*localisationfichier)->inclusion)));
	free((*localisationfichier)->nom);
	if((*localisationfichier)->nomabsolu!=NULL)
		free((*localisationfichier)->nomabsolu);
	free(*localisationfichier);
	*localisationfichier=NULL;
	FINZONESECURISEE
	return RESULTAT_OK;
}

