"""Python library to connect deCONZ and Home Assistant to work together."""

import enum
from typing import NotRequired, TypedDict

from . import LightBase


class TypedCoverState(TypedDict):
    """Cover state type definition."""

    bri: int
    lift: NotRequired[int]
    open: NotRequired[bool]
    sat: NotRequired[int]
    tilt: NotRequired[int]


class TypedCover(TypedDict):
    """Cover type definition."""

    state: TypedCoverState


class CoverAction(enum.StrEnum):
    """Possible cover actions."""

    CLOSE = enum.auto()
    OPEN = enum.auto()
    STOP = enum.auto()


class Cover(LightBase):
    """Cover and Damper class."""

    raw: TypedCover

    @property
    def is_open(self) -> bool:
        """Is cover open."""
        if "open" not in self.raw["state"]:  # Legacy support
            return self.state is False
        return self.raw["state"]["open"]

    @property
    def lift(self) -> int:
        """Amount of closed position.

        Supported values:
          0-100 - 0 is open / 100 is closed
        """
        if "lift" not in self.raw["state"]:  # Legacy support
            return int(self.raw["state"]["bri"] / 2.54)
        return self.raw["state"]["lift"]

    @property
    def tilt(self) -> int | None:
        """Amount of tilt.

        Supported values:
          0-100 - 0 is open / 100 is closed
        """
        if "tilt" in self.raw["state"]:
            if 0 <= (tilt := self.raw["state"]["tilt"]) <= 100:
                return tilt
        elif "sat" in self.raw["state"]:  # Legacy support
            return int(self.raw["state"]["sat"] / 2.54)
        return None

    @property
    def supports_tilt(self) -> bool:
        """Supports tilt."""
        return "tilt" in self.raw["state"]
