\name{quadscheme.logi}
\alias{quadscheme.logi}
\title{Generate a Logistic Regression Quadrature Scheme from a Point Pattern}
\description{
  Generates a logistic regression quadrature scheme (an object of class
  \code{"logiquad"} inheriting from \code{"quad"})
  from point patterns of data and dummy points.
}
\usage{
 quadscheme.logi(data, dummy, dummytype = "stratrand",
                 nd = NULL, mark.repeat = FALSE, \dots)
}
\arguments{
  \item{data}{
    The observed data point pattern.
    An object of class \code{"ppp"}
    or in a format recognised by \code{\link{as.ppp}()}
  }
  \item{dummy}{
    The pattern of dummy points for the quadrature. 
    An object of class \code{"ppp"}
    or in a format recognised by \code{\link{as.ppp}()}.
    If missing a sensible default is generated.
  }
  \item{dummytype}{
    The name of the type of dummy points to use when \code{"dummy"}
    is missing. Currently available
    options are: \code{"stratrand"} (default), \code{"binomial"},
    \code{"poisson"}, \code{"grid"} and \code{"transgrid"}.
  }
  \item{nd}{
    Integer, or integer vector of length 2 controlling the intensity of
    dummy points when \code{"dummy"} is missing.
  }
  \item{mark.repeat}{
    Repeating the dummy points for each level of a marked data pattern
    when \code{"dummy"} is missing. (See details.)
  }
  \item{\dots}{
    Ignored.
  }
} 
\value{
  An object of class \code{"logiquad"} inheriting from \code{"quad"}
  describing the quadrature scheme 
  (data points, dummy points, and quadrature weights)
  suitable as the argument \code{Q} of the function \code{\link{ppm}()} for
  fitting a point process model.

  The quadrature scheme can be inspected using the
  \code{print} and \code{plot} methods for objects
  of class \code{"quad"}. 
}
\details{
  This is the primary method for producing a quadrature schemes
  for use by \code{\link{ppm}} when the logistic regression
  approximation (Baddeley et al. 2013) to the pseudolikelihood of the
  model is applied (i.e. when \code{method="logi"} in \code{\link{ppm}}).
  
  The function \code{\link{ppm}} fits a point process model to an
  observed point pattern. When used with the option \code{method="logi"}
  it requires a quadrature scheme consisting of 
  the original data point pattern and an additional pattern of dummy points.
  Such quadrature schemes are represented by objects of class
  \code{"logiquad"}.

  Quadrature schemes are created by the function
  \code{quadscheme.logi}.
  The arguments \code{data} and \code{dummy} specify the data and dummy
  points, respectively. There is a sensible default for the dummy 
  points.
  Alternatively the dummy points
  may be specified arbitrarily and given in any format recognised by
  \code{\link{as.ppp}}.
 
  The quadrature region is the region over which we are
  integrating, and approximating integrals by finite sums.
  If \code{dummy} is a point pattern object (class \code{"ppp"})
  then the quadrature region is taken to be \code{dummy$window}.
  If \code{dummy} is just a list of \eqn{x, y} coordinates
  then the quadrature region defaults to the observation window
  of the data pattern, \code{data$window}.

  If \code{dummy} is missing, then a pattern of dummy points will be
  generated, taking account of the optional arguments \code{dummytype},
  \code{nd}, and \code{mark.repeat}.

  The currently accepted values for \code{dummytype} are:
  \itemize{
  \item \code{"grid"} where the frame of the window
  is divided into a \code{nd * nd} or \code{nd[1] * nd[2]} regular grid
  of tiles and the centers constitutes the dummy points.
  \item \code{"transgrid"} where a regular grid as above is translated
  by a random vector.
  \item \code{"stratrand"} where each point of a regular grid as above
  is randomly translated within its tile.
  \item \code{"binomial"} where \code{nd * nd} or \code{nd[1] * nd[2]}
  points are generated uniformly in the frame of the
  window.
  \code{"poisson"} where a homogeneous Poisson point process with
  intensity \code{nd * nd} or \code{nd[1] * nd[2]} is
  generated within the frame of observation window.
}
Then if the window is not rectangular, any dummy points
lying outside it are deleted. 

If \code{data} is a multitype point pattern the dummy points should also
be marked (with the same levels of the marks as \code{data}). If
\code{dummy} is missing and the dummy pattern is generated by
\code{quadscheme.logi} the default behaviour is to attach a uniformly
distributed mark (from the levels of the marks) to each dummy
point. Alternatively, if \code{mark.repeat=TRUE} each dummy point is
repeated as many times as there are levels of the marks with a distinct
mark value attached to it.

Finally, each point (data and dummy) is assigned the weight 1. The
weights are never used and only appear to be compatible with the class
\code{"quad"} from which the \code{"logiquad"} object inherits.
}
\references{
  Baddeley, A., Coeurjolly, J.-F., Rubak, E. and Waagepetersen, R. (2013)
  \emph{A logistic regression estimating function
    for spatial Gibbs point processes.}
  Research Report, Centre for Stochastic Geometry and Bioimaging,
  Denmark. \url{www.csgb.dk}
}
\seealso{
  \code{\link{ppm}},
  \code{\link{as.ppp}}
}
\examples{
  data(simdat)

  Q <- quadscheme.logi(simdat)
  
}
\author{Adrian Baddeley
  \email{Adrian.Baddeley@uwa.edu.au}
  \url{http://www.maths.uwa.edu.au/~adrian/},
  Rolf Turner \email{r.turner@auckland.ac.nz}
  and Ege Rubak
  \email{rubak@math.aau.dk}.
}
\keyword{spatial}
\keyword{datagen}
