# ifndef _SKIT_CGS_H
# define _SKIT_CGS_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
/*Class:cgs
NAME: cgs - cgs-conjugate gradient method 
SYNOPSIS:
    
    "int cgs" ("const Matrix &A, Vector &x, const Vector &b",
    |   const Preconditioner &M, int &max_iter, Real &tol);

EXAMPLE:

  The simplest call to 'cgs' has the folling form:

  |    int status = cgs(a, x, b, EYE, 100, 1e-7);

DESCRIPTION:       

 'cgs' solves the unsymmetric linear system Ax = b 
 using the conjugate gradient squared method.

  The return value indicates convergence within max_iter (input)
  iterations (0), or no convergence within max_iter iterations (1).
  Upon successful return, output arguments have the following values:

    "x"  approximate solution to Ax = b

    "max_iter" the number of iterations performed before the tolerance was reached

    "tol" the residual after the final iteration

SEE ALSO:

  "cg", "bicg", "bichstab", class "csr", class "vec"

NOTE: 

  'cgs' is an iterative template routine.

  'cgs' follows the algorithm described on p. 26 in

	Templates for the Solution of Linear Systems: Building Blocks for Iterative Methods, 
	2nd Edition, 
        R. Barrett, M. Berry, T. F. Chan, J. Demmel, J. Donato, J. Dongarra, V. Eijkhout,
	R. Pozo, C. Romine, H. Van der Vorst,
        SIAM, 1994, "ftp.netlib.org/templates/templates.ps".

  The present implementation is inspired from "IML++ 1.2" iterative method library,
  "http://math.nist.gov/iml++".

AUTHOR: 

    Pierre Saramito
    | Pierre.Saramito@imag.fr
    LMC-IMAG, 38041 Grenoble cedex 9, France

DATE: 
    
    12 march 1997

METHODS: @cgs
End:
*/
// ========== [ method implementation ] ====================================

namespace rheolef { 

//<cgs:
template < class Matrix, class Vector, class Preconditioner, class Real >
int 
cgs(const Matrix &A, Vector &x, const Vector &b,
    const Preconditioner &M, int &max_iter, Real &tol)
{
  Real resid;
  Real rho_1, rho_2=0, alpha, beta;
  Vector p, phat, q, qhat, vhat, u, uhat;

  Real normb = norm(b);
  Vector r = b - A*x;
  Vector rtilde = r;

  if (normb == Real(0))
    normb = 1;
  
  if ((resid = norm(r) / normb) <= tol) {
    tol = resid;
    max_iter = 0;
    return 0;
  }

  for (int i = 1; i <= max_iter; i++) {
    rho_1 = dot(rtilde, r);
    if (rho_1 == Real(0)) {
      tol = norm(r) / normb;
      return 2;
    }
    if (i == 1) {
      u = r;
      p = u;
    } else {
      beta = rho_1 / rho_2;
      u = r + beta * q;
      p = u + beta * (q + beta * p);
    }
    phat = M.solve(p);
    vhat = A*phat;
    alpha = rho_1 / dot(rtilde, vhat);
    q = u - alpha * vhat;
    uhat = M.solve(u + q);
    x += alpha * uhat;
    qhat = A * uhat;
    r -= alpha * qhat;
    rho_2 = rho_1;
    if ((resid = norm(r) / normb) < tol) {
      tol = resid;
      max_iter = i;
      return 0;
    }
  }
  
  tol = resid;
  return 1;
}
//>cgs:
}// namespace rheolef
#endif // _SKIT_CGS_H
