#ifndef _RHEOLEF_COMPILER_H
#define _RHEOLEF_COMPILER_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
//
// compiler-dependent part for rheolef implementation
//
// author: Pierre.Saramito@imag.fr
//
// date: 27 january 2000
//

#include "rheolef/config.h"   /* as generated by configure */

// ----------------------------------------------------------------------------
// i/o
// ----------------------------------------------------------------------------
#include <ios>
#include <istream>
#include <ostream>
#include <iostream>
#include <fstream>
#include <iomanip>

// ----------------------------------------------------------------------------
// macro utilities:
// ----------------------------------------------------------------------------
//	fatal_macro
//	error_macro
//	trace_macro
//	check_macro
//	assert_macro
//
# define fatal_macro(message) \
	{ std::cerr << "fatal(" << __FILE__ << "," << __LINE__ << "): " << message << std::endl; exit(1); }
# define error_macro(message) \
	fatal_macro(message)
# define warning_macro(message) \
	{ std::cerr << "warning(" << __FILE__ << "," << __LINE__ << "): " << message << std::endl; }
#  define check_macro(ok_condition, message) \
        { if (!(ok_condition)) fatal_macro(message); }

// ----------------------------------------------------------------------------
// debug
// ----------------------------------------------------------------------------
# ifdef _RHEOLEF_PARANO
#  ifdef  NDEBUG
#    undef  NDEBUG
#  endif
#  define trace_macro(message) \
        { std::cerr << "trace(" << __FILE__ << "," << __LINE__ << "): " << message << std::endl; }
#  define assert_macro(ok_condition, message) \
	check_macro(ok_condition,message)
# else // not _RHEOLEF_PARANO
#  ifndef NDEBUG
#    define NDEBUG
#    define BOOST_UBLAS_NDEBUG // disable all bounds, structure and similar checks of uBLAS
#  endif
#  define trace_macro(message)
#  define assert_macro(ok_condition, message)
# endif // _RHEOLEF_PARANO

// ----------------------------------------------------------------------------
// some std libs
// ----------------------------------------------------------------------------
#include <memory>
#include <iterator>
#include <cstdlib>
#include <cstdio>
#include <string>
#include <utility> // std::pair<A,B>
#include <vector>
#include <list>
#include <set>
#include <map>

// ----------------------------------------------------------------------------
// C++ malloc debug
// ----------------------------------------------------------------------------
# ifdef _RHEOLEF_HAVE_DMALLOC

// macros wrapper for new and delette
// macros wrapper for new and delette
// when debuging with malloc_dbg

#define DMALLOC_DISABLE /* do not re-define free() and such as macros: clash with boost that calls std::free()  */
#include <dmalloc.h> // link also with -ldmalloc
#include "rheolef/dmalloc_return.h"  // from dmalloc lib

// for eigen: dmallloc is not alligned to 16 bits
#ifdef   EIGEN_MALLOC_ALREADY_ALIGNED
#undef   EIGEN_MALLOC_ALREADY_ALIGNED
#endif //EIGEN_MALLOC_ALREADY_ALIGNED
#define  EIGEN_MALLOC_ALREADY_ALIGNED 0

// defined in "util/dmallocxx/dmallocc.cc"
// put it in global namespace:
#define USE_NEW_INLINE
#ifdef  USE_NEW_INLINE
inline
void *operator new(size_t size, const char *const file, int line)
{
  if (size == 0) return (void*)0;
  return dmalloc_malloc(file, line, size, DMALLOC_FUNC_NEW,
			0 /* no alignment */, 0 /* no xalloc messages */);
}
inline
void *operator new[](size_t size,  const char *const file, int line)
{
  if (size == 0) return (void*)0;
  return dmalloc_malloc(file, line, size, DMALLOC_FUNC_NEW_ARRAY,
			0 /* no alignment */, 0 /* no xalloc messages */);
}
inline
void
operator delete(void *pnt)
{
  char	*file;
  GET_RET_ADDR(file);
  dmalloc_free(file, 0, pnt, DMALLOC_FUNC_DELETE);
}
inline
void
operator delete[](void *pnt)
{
  char	*file;
  GET_RET_ADDR(file);
  dmalloc_free(file, 0, pnt, DMALLOC_FUNC_DELETE_ARRAY);
}
#else // USE_NEW_INLINE
void *operator new(size_t, const char *const, int);
void *operator new[](size_t,  const char *const, int);
void  operator delete(void *);
void  operator delete[](void *);
#endif // USE_NEW_INLINE
# define 	new_macro(obj)            new (__FILE__ , __LINE__) obj
# define 	new_tab_macro(typ, n)     (new (__FILE__ , __LINE__) typ [(n)])
# define        delete_macro(ptr)         { if (ptr) delete (ptr); }
# define        delete_tab_macro(ptr)     { if (ptr) delete [] (ptr); }

# else // ! _RHEOLEF_HAVE_DMALLOC : standard c++ new, delete

# define 	new_macro(obj)            (new obj)
# define 	new_tab_macro(typ, n)     (new typ [(n)])
# define        delete_macro(ptr)         { if (ptr) delete (ptr); }
# define        delete_tab_macro(ptr)     { if (ptr) delete [] (ptr); }

# endif // _RHEOLEF_HAVE_DMALLOC

// ----------------------------------------------------------------------------
// floats
// ----------------------------------------------------------------------------
#include "rheolef/Float.h"

#endif // _RHEOLEF_COMPILER_H
