/*  RipOff - Plugin based CD Ripper
 *  Copyright (C) 2006 Bobby Ryan Newberry
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <FLAC/export.h>

#if !defined(FLAC_API_VERSION_CURRENT) || FLAC_API_VERSION_CURRENT <= 7
#include <FLAC/seekable_stream_encoder.h>
#else
#include <FLAC/stream_encoder.h>
#endif

#include <FLAC/metadata.h>
#include "lib/RipOffPluginRawInterface.h"
#define UI_NAME "FLAC Encoder Plugin v1.1"
#define ENCODER "ENCODER=RipOff "VERSION
#define ARTIST "ARTIST="
#define GENRE "GENRE="
#define ALBUM "ALBUM="
#define TITLE "TITLE="
#define DATE "DATE="
#define TRACKNUMBER "TRACKNUMBER="
#define DESCRIPTION "DESCRIPTION=Ripped by "NAME" "WEBSITE


struct FlacPlugin_
{
	gchar *name;
	gchar *label;
        #if !defined(FLAC_API_VERSION_CURRENT) || FLAC_API_VERSION_CURRENT <= 7
	FLAC__SeekableStreamEncoder *encoder;
        #else
        FLAC__StreamEncoder *encoder;
        #endif
	FLAC__StreamMetadata *metadata[1];
	guint8 compression_level;
};
typedef struct FlacPlugin_ * FlacPlugin;

#if !defined(FLAC_API_VERSION_CURRENT) || FLAC_API_VERSION_CURRENT <= 7
FLAC__StreamEncoderWriteStatus write_callback(	const FLAC__SeekableStreamEncoder *encoder, 
						const FLAC__byte buffer[], 
						unsigned bytes, 
						unsigned samples, 
						unsigned current_frame, 
						void *client_data);

FLAC__SeekableStreamEncoderSeekStatus seek_callback(	const FLAC__SeekableStreamEncoder *encoder, 
							FLAC__uint64 absolute_byte_offset, 
							void *client_data);

FLAC__SeekableStreamEncoderTellStatus tell_callback(	const FLAC__SeekableStreamEncoder *encoder, 
							FLAC__uint64 *absolute_byte_offset, 
							void *client_data);
#else
FLAC__StreamEncoderWriteStatus write_callback(	const FLAC__StreamEncoder *encoder, 
						const FLAC__byte buffer[], 
						unsigned bytes, 
						unsigned samples, 
						unsigned current_frame, 
						void *client_data);

FLAC__StreamEncoderSeekStatus seek_callback(	const FLAC__StreamEncoder *encoder, 
							FLAC__uint64 absolute_byte_offset, 
							void *client_data);

FLAC__StreamEncoderTellStatus tell_callback(	const FLAC__StreamEncoder *encoder, 
							FLAC__uint64 *absolute_byte_offset, 
							void *client_data);
#endif

void value_changed(GtkAdjustment *adjustment, gpointer data);
void setup_metadata(FlacPlugin plugin, RipOffTrack track);
void setup_compression_level(FlacPlugin plugin);

RipOffPluginRaw ripoff_plugin_raw_new(xmlDocPtr ptr)
{
	FlacPlugin raw = g_new(struct FlacPlugin_, 1);
	ripoff_config_file_set_config_file_pointer(ptr);
	raw->name = UI_NAME;
	raw->label = "ripoff_default_flac_plugin";
	raw->compression_level = ripoff_config_file_retrieve_int(raw->label, "compression_level", 4.0);
	
	/* this function should take care of the stored preferences loading */
	return  (RipOffPluginRaw) raw;
}

gchar *ripoff_plugin_raw_get_name(RipOffPluginRaw raw)
{
	return raw->name;
}

gchar *ripoff_plugin_raw_get_label(RipOffPluginRaw raw)
{
	return raw->label;
}

const gchar *ripoff_plugin_raw_get_extension(RipOffPluginRaw raw)
{
	return "flac";
}

gboolean ripoff_plugin_raw_has_prefs(RipOffPluginRaw raw)
{
	return TRUE;
}

gboolean ripoff_plugin_raw_has_about(RipOffPluginRaw raw)
{
	return TRUE;
}

GtkWidget *ripoff_plugin_raw_prefs(RipOffPluginRaw raw/*, GtkWidget *pref_window*/)
{
	GtkWidget *close_button;
	GtkWidget *window;
	GtkObject *compression_level_adjustment;
	GtkWidget *compression_level_hscale;
	GtkWidget *label;
	GtkWidget *table;
	GtkTooltips *compression_level_tips;
	FlacPlugin plugin = (FlacPlugin) raw;
	/*struct helper_struct *helper;
	helper = g_new(struct helper_struct, 1);*/

	table = gtk_table_new(2, 2, FALSE);

	window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_widget_set_size_request (GTK_WIDGET (window), 400, 140);
	gtk_window_set_title (GTK_WINDOW (window), UI_NAME" Preferences");

	label = gtk_label_new("FLAC Compression Level");
	/* Creates quality setting radio button and scale bar*/
	compression_level_tips = gtk_tooltips_new();



	/* create compression level rating adjustments and scales */
	compression_level_adjustment = gtk_adjustment_new(	plugin->compression_level,
								0.0,
								8.0,
								4.0,
								1.0,
								0.0); 

	compression_level_hscale = gtk_hscale_new(GTK_ADJUSTMENT(compression_level_adjustment));
	gtk_scale_set_digits(GTK_SCALE(compression_level_hscale), 0);
	gtk_tooltips_set_tip(compression_level_tips, compression_level_hscale, "A higher compression level allows your outputted FLAC files to be smaller at the cost of a longer encoding session.", NULL);

	g_signal_connect(	compression_level_adjustment,
				"value-changed",
				G_CALLBACK (value_changed),
				&(plugin->compression_level));


	/* close window button setup */
	close_button = gtk_button_new_from_stock(GTK_STOCK_CLOSE);


	g_signal_connect_swapped(	G_OBJECT (close_button), 
					"clicked",
                              		G_CALLBACK (gtk_widget_destroy),
                              		G_OBJECT (window));

	gtk_table_attach (	GTK_TABLE(table),
				label, 0, 1, 0, 1,
				GTK_FILL | GTK_EXPAND,
				GTK_FILL | GTK_EXPAND,
				5, 5);

	gtk_table_attach (	GTK_TABLE(table),
				compression_level_hscale, 1, 2, 0, 1,
				GTK_FILL | GTK_EXPAND,
				GTK_FILL | GTK_EXPAND,
				5, 5);

	gtk_table_attach (	GTK_TABLE(table),
				close_button, 1, 2, 1, 2,
				GTK_SHRINK,
				GTK_SHRINK,
				0, 5);


	gtk_container_add(GTK_CONTAINER(window), table);

	return window;
}

GtkWidget *ripoff_plugin_raw_about(RipOffPluginRaw raw)
{
	return about_window_draw(	        "FLAC Encoder", 
						UI_NAME, 
						"FLAC Encoding Plugin for RipOff",
						"Copyright © "YEAR" Bobby Ryan Newberry",
						WEBSITE);
}

gboolean ripoff_plugin_raw_perform_setup(	RipOffPluginRaw raw,
						long total_bytes_to_encode,
						FILE *output_descriptor,
						RipOffTrack track)
{	
	FlacPlugin plugin = (FlacPlugin) raw;

        #if !defined(FLAC_API_VERSION_CURRENT) || FLAC_API_VERSION_CURRENT <= 7
	plugin->encoder = FLAC__seekable_stream_encoder_new();
	setup_metadata(plugin, track);
	setup_compression_level(plugin);

	FLAC__seekable_stream_encoder_set_write_callback(plugin->encoder, write_callback);
	FLAC__seekable_stream_encoder_set_seek_callback(plugin->encoder, seek_callback);
	FLAC__seekable_stream_encoder_set_tell_callback(plugin->encoder, tell_callback);
	FLAC__seekable_stream_encoder_set_client_data(plugin->encoder, output_descriptor);
	FLAC__seekable_stream_encoder_set_total_samples_estimate(plugin->encoder, total_bytes_to_encode / 4);
	FLAC__seekable_stream_encoder_init(plugin->encoder);
        #else
	plugin->encoder = FLAC__stream_encoder_new();
	setup_metadata(plugin, track);
	setup_compression_level(plugin);
	FLAC__stream_encoder_set_total_samples_estimate(plugin->encoder, total_bytes_to_encode / 4);

        FLAC__stream_encoder_init_stream(       plugin->encoder,
                                                write_callback,
                                                seek_callback,
                                                tell_callback,
                                                NULL,
                                                output_descriptor);
        #endif

	return TRUE;
}

gboolean ripoff_plugin_raw_encode_buffer(	RipOffPluginRaw raw,
						long total_bytes_to_encode,
						int16_t *audio_data,
						FILE *output_descriptor,
						RipOffTrack track)
{
	FlacPlugin plugin = (FlacPlugin) raw;
	int j;
	int32_t conversion_buffer[CDIO_CD_FRAMESIZE_RAW /2];

	/* we are doing this because FLAC understands int32_t while
	 * libparanoia rips to int16_t thus requiring some conversion
	 */
	for(j = 0; j < CDIO_CD_FRAMESIZE_RAW / 2; ++j)
	{
		conversion_buffer[j] = audio_data[j];
		/*((char *)&(audio_data[j]))[0] = ((char *)&(audio_data[j+1]))[1];
		((char *)&(audio_data[j]))[1] = ((char *)&(audio_data[j+1]))[0];*/
	}

	if(plugin->encoder == NULL)
		g_print("NULL!\n");

        #if !defined(FLAC_API_VERSION_CURRENT) || FLAC_API_VERSION_CURRENT <= 7
	FLAC__seekable_stream_encoder_process_interleaved(	plugin->encoder, 
								conversion_buffer, 
								CDIO_CD_FRAMESIZE_RAW / 4);
        #else
	FLAC__stream_encoder_process_interleaved(	plugin->encoder, 
								conversion_buffer, 
								CDIO_CD_FRAMESIZE_RAW / 4);
        #endif
	return TRUE;
}

gboolean ripoff_plugin_raw_perform_cleanup(	RipOffPluginRaw raw,
						long total_bytes_to_encode,
						FILE *output_descriptor,
						RipOffTrack track)
{
	FlacPlugin plugin = (FlacPlugin) raw;
	
        #if !defined(FLAC_API_VERSION_CURRENT) || FLAC_API_VERSION_CURRENT <= 7
	FLAC__seekable_stream_encoder_finish(plugin->encoder);
	FLAC__seekable_stream_encoder_delete(plugin->encoder);
        #else
	FLAC__stream_encoder_finish(plugin->encoder);
	FLAC__stream_encoder_delete(plugin->encoder);
        #endif
        FLAC__metadata_object_delete(plugin->metadata[0]);

	return TRUE;
}

void ripoff_plugin_raw_close(RipOffPluginRaw raw)
{
	FlacPlugin plugin = (FlacPlugin) raw;
	ripoff_config_file_save_int(plugin->label, "compression_level", plugin->compression_level);

}

#if !defined(FLAC_API_VERSION_CURRENT) || FLAC_API_VERSION_CURRENT <= 7
FLAC__StreamEncoderWriteStatus write_callback(	const FLAC__SeekableStreamEncoder *encoder, 
						const FLAC__byte buffer[], 
						unsigned bytes, 
						unsigned samples, 
						unsigned current_frame, 
						void *client_data)
#else
FLAC__StreamEncoderWriteStatus write_callback(	const FLAC__StreamEncoder *encoder, 
						const FLAC__byte buffer[], 
						unsigned bytes, 
						unsigned samples, 
						unsigned current_frame, 
						void *client_data)
#endif
{
	FILE *output_descriptor = client_data;

	fwrite(buffer, bytes, 1, output_descriptor);
	return FLAC__STREAM_ENCODER_WRITE_STATUS_OK;
}

#if !defined(FLAC_API_VERSION_CURRENT) || FLAC_API_VERSION_CURRENT <= 7
FLAC__SeekableStreamEncoderSeekStatus seek_callback(	const FLAC__SeekableStreamEncoder *encoder, 
							FLAC__uint64 absolute_byte_offset, 
							void *client_data)
#else
FLAC__StreamEncoderSeekStatus seek_callback(	        const FLAC__StreamEncoder *encoder, 
							FLAC__uint64 absolute_byte_offset, 
							void *client_data)
#endif
{
	FILE *output_descriptor = client_data;

	fseek(output_descriptor, absolute_byte_offset, SEEK_SET);
	return TRUE;
}

#if !defined(FLAC_API_VERSION_CURRENT) || FLAC_API_VERSION_CURRENT <= 7
FLAC__SeekableStreamEncoderTellStatus tell_callback(	const FLAC__SeekableStreamEncoder *encoder, 
							FLAC__uint64 *absolute_byte_offset, 
							void *client_data)
#else
FLAC__StreamEncoderTellStatus tell_callback(	        const FLAC__StreamEncoder *encoder, 
							FLAC__uint64 *absolute_byte_offset, 
							void *client_data)
#endif
{
	FILE *output_descriptor = client_data;
	*absolute_byte_offset = ftell(output_descriptor);

        #if !defined(FLAC_API_VERSION_CURRENT) || FLAC_API_VERSION_CURRENT <= 7
	return FLAC__SEEKABLE_STREAM_ENCODER_TELL_STATUS_OK;
        #else
        return FLAC__STREAM_ENCODER_TELL_STATUS_OK;
        #endif
}

void value_changed(GtkAdjustment *adjustment, gpointer data)
{
	*((guint8 *)data) = gtk_adjustment_get_value(adjustment);
}

void setup_metadata(FlacPlugin plugin, RipOffTrack track)
{
	gchar buffer[255];
	plugin->metadata[0] = FLAC__metadata_object_new(FLAC__METADATA_TYPE_VORBIS_COMMENT);
	FLAC__metadata_object_vorbiscomment_resize_comments(plugin->metadata[0], 8);
	FLAC__StreamMetadata_VorbisComment_Entry entry;

	/* metadata entry for the artist */
	strncpy(buffer, ARTIST, 255);
	strncat(buffer, ripoff_track_get_artist(track), 255);
	entry.entry = buffer;
	entry.length = strlen(buffer);

	FLAC__metadata_object_vorbiscomment_set_comment(plugin->metadata[0], 0, entry, TRUE);

	/* metadata entry for the genre */
	strncpy(buffer, GENRE, 255);
	strncat(buffer, ripoff_track_get_genre(track), 255);
	entry.entry = buffer;
	entry.length = strlen(buffer);

	FLAC__metadata_object_vorbiscomment_set_comment(plugin->metadata[0], 1, entry, TRUE);

	/* metadata entry for the album */
	strncpy(buffer, ALBUM, 255);
	strncat(buffer, ripoff_track_get_album_title(track), 255);
	entry.entry = buffer;
	entry.length = strlen(buffer);

	FLAC__metadata_object_vorbiscomment_set_comment(plugin->metadata[0], 2, entry, TRUE);

	/* metadata entry for the track title */
	strncpy(buffer, TITLE, 255);
	strncat(buffer, ripoff_track_get_track_title(track), 255);
	entry.entry = buffer;
	entry.length = strlen(buffer);

	FLAC__metadata_object_vorbiscomment_set_comment(plugin->metadata[0], 3, entry, TRUE);

	/* metadata entry for the year */
	strncpy(buffer, DATE, 255);
	strncat(buffer, ripoff_track_get_year(track), 255);
	entry.entry = buffer;
	entry.length = strlen(buffer);

	FLAC__metadata_object_vorbiscomment_set_comment(plugin->metadata[0], 4, entry, TRUE);

	/* metadata entry for the track number */
	strncpy(buffer, TRACKNUMBER, 255);
	strncat(buffer, ripoff_track_get_track_num_string(track, TRUE), 255);
	entry.entry = buffer;
	entry.length = strlen(buffer);

	FLAC__metadata_object_vorbiscomment_set_comment(plugin->metadata[0], 5, entry, TRUE);

	/* metadata entry for the comment */
	strncpy(buffer, DESCRIPTION, 255);
	entry.entry = buffer;
	entry.length = strlen(buffer);

	FLAC__metadata_object_vorbiscomment_set_comment(plugin->metadata[0], 6, entry, TRUE);

	/* metadata entry for the encoder */
	strncpy(buffer, ENCODER, 255);
	entry.entry = buffer;
	entry.length = strlen(buffer);

	FLAC__metadata_object_vorbiscomment_set_comment(plugin->metadata[0], 7, entry, TRUE);

        #if !defined(FLAC_API_VERSION_CURRENT) || FLAC_API_VERSION_CURRENT <= 7
	FLAC__seekable_stream_encoder_set_metadata(plugin->encoder, plugin->metadata, 1);
        #else
        FLAC__stream_encoder_set_metadata(plugin->encoder, plugin->metadata, 1);
        #endif
}

void setup_compression_level(FlacPlugin plugin)
{
        #if !defined(FLAC_API_VERSION_CURRENT) || FLAC_API_VERSION_CURRENT <= 7
	switch(plugin->compression_level)
	{
		case 0:	FLAC__seekable_stream_encoder_set_max_lpc_order(plugin->encoder, 0);
			FLAC__seekable_stream_encoder_set_blocksize(plugin->encoder, 1152);
			FLAC__seekable_stream_encoder_set_do_mid_side_stereo(plugin->encoder, FALSE);
			FLAC__seekable_stream_encoder_set_loose_mid_side_stereo(plugin->encoder, FALSE);
			FLAC__seekable_stream_encoder_set_min_residual_partition_order(plugin->encoder, 2);
			FLAC__seekable_stream_encoder_set_max_residual_partition_order(plugin->encoder, 2);
			FLAC__seekable_stream_encoder_set_do_exhaustive_model_search(plugin->encoder, FALSE);
			break;

		case 1: FLAC__seekable_stream_encoder_set_max_lpc_order(plugin->encoder, 0);
			FLAC__seekable_stream_encoder_set_blocksize(plugin->encoder, 1152);
			FLAC__seekable_stream_encoder_set_do_mid_side_stereo(plugin->encoder, TRUE);
			FLAC__seekable_stream_encoder_set_loose_mid_side_stereo(plugin->encoder, TRUE);
			FLAC__seekable_stream_encoder_set_min_residual_partition_order(plugin->encoder, 2);
			FLAC__seekable_stream_encoder_set_max_residual_partition_order(plugin->encoder, 2);
			FLAC__seekable_stream_encoder_set_do_exhaustive_model_search(plugin->encoder, FALSE);
			break;
		
		case 2: FLAC__seekable_stream_encoder_set_max_lpc_order(plugin->encoder, 0);
			FLAC__seekable_stream_encoder_set_blocksize(plugin->encoder, 1152);
			FLAC__seekable_stream_encoder_set_do_mid_side_stereo(plugin->encoder, TRUE);
			FLAC__seekable_stream_encoder_set_loose_mid_side_stereo(plugin->encoder, FALSE);
			FLAC__seekable_stream_encoder_set_min_residual_partition_order(plugin->encoder, 3);
			FLAC__seekable_stream_encoder_set_do_exhaustive_model_search(plugin->encoder, FALSE);
			break;
			


		case 3: FLAC__seekable_stream_encoder_set_max_lpc_order(plugin->encoder, 6);
			FLAC__seekable_stream_encoder_set_blocksize(plugin->encoder, 4608);
			FLAC__seekable_stream_encoder_set_do_mid_side_stereo(plugin->encoder, TRUE);
			FLAC__seekable_stream_encoder_set_loose_mid_side_stereo(plugin->encoder, FALSE);
			FLAC__seekable_stream_encoder_set_min_residual_partition_order(plugin->encoder, 3);
			FLAC__seekable_stream_encoder_set_max_residual_partition_order(plugin->encoder, 3);
			FLAC__seekable_stream_encoder_set_do_exhaustive_model_search(plugin->encoder, FALSE);
			break;

		case 4: FLAC__seekable_stream_encoder_set_max_lpc_order(plugin->encoder, 8);
			FLAC__seekable_stream_encoder_set_blocksize(plugin->encoder, 4608);
			FLAC__seekable_stream_encoder_set_do_mid_side_stereo(plugin->encoder, TRUE);
			FLAC__seekable_stream_encoder_set_loose_mid_side_stereo(plugin->encoder, TRUE);
			FLAC__seekable_stream_encoder_set_min_residual_partition_order(plugin->encoder, 3);
			FLAC__seekable_stream_encoder_set_max_residual_partition_order(plugin->encoder, 3);
			FLAC__seekable_stream_encoder_set_do_exhaustive_model_search(plugin->encoder, FALSE);
			break;



		case 5: FLAC__seekable_stream_encoder_set_max_lpc_order(plugin->encoder, 8);
			FLAC__seekable_stream_encoder_set_blocksize(plugin->encoder, 4608);
			FLAC__seekable_stream_encoder_set_do_mid_side_stereo(plugin->encoder, TRUE);
			FLAC__seekable_stream_encoder_set_loose_mid_side_stereo(plugin->encoder, FALSE);
			FLAC__seekable_stream_encoder_set_min_residual_partition_order(plugin->encoder, 3);
			FLAC__seekable_stream_encoder_set_max_residual_partition_order(plugin->encoder, 3);
			FLAC__seekable_stream_encoder_set_do_exhaustive_model_search(plugin->encoder, FALSE);
			break;

		case 6: FLAC__seekable_stream_encoder_set_max_lpc_order(plugin->encoder, 8);
			FLAC__seekable_stream_encoder_set_blocksize(plugin->encoder, 4608);
			FLAC__seekable_stream_encoder_set_do_mid_side_stereo(plugin->encoder, TRUE);
			FLAC__seekable_stream_encoder_set_loose_mid_side_stereo(plugin->encoder, FALSE);
			FLAC__seekable_stream_encoder_set_min_residual_partition_order(plugin->encoder, 4);
			FLAC__seekable_stream_encoder_set_do_exhaustive_model_search(plugin->encoder, FALSE);
			break;

		case 7: FLAC__seekable_stream_encoder_set_max_lpc_order(plugin->encoder, 8);
			FLAC__seekable_stream_encoder_set_blocksize(plugin->encoder, 4608);
			FLAC__seekable_stream_encoder_set_do_mid_side_stereo(plugin->encoder, TRUE);
			FLAC__seekable_stream_encoder_set_loose_mid_side_stereo(plugin->encoder, FALSE);
			FLAC__seekable_stream_encoder_set_min_residual_partition_order(plugin->encoder, 6);
			FLAC__seekable_stream_encoder_set_do_exhaustive_model_search(plugin->encoder, TRUE);
			break;

		case 8: FLAC__seekable_stream_encoder_set_max_lpc_order(plugin->encoder, 12);
			FLAC__seekable_stream_encoder_set_blocksize(plugin->encoder, 4608);
			FLAC__seekable_stream_encoder_set_do_mid_side_stereo(plugin->encoder, TRUE);
			FLAC__seekable_stream_encoder_set_loose_mid_side_stereo(plugin->encoder, FALSE);
			FLAC__seekable_stream_encoder_set_min_residual_partition_order(plugin->encoder, 6);
			FLAC__seekable_stream_encoder_set_do_exhaustive_model_search(plugin->encoder, TRUE);
			break;
	}
        #else
	switch(plugin->compression_level)
	{
		case 0:	FLAC__stream_encoder_set_max_lpc_order(plugin->encoder, 0);
			FLAC__stream_encoder_set_blocksize(plugin->encoder, 1152);
			FLAC___stream_encoder_set_do_mid_side_stereo(plugin->encoder, FALSE);
			FLAC__stream_encoder_set_loose_mid_side_stereo(plugin->encoder, FALSE);
			FLAC__stream_encoder_set_min_residual_partition_order(plugin->encoder, 2);
			FLAC__stream_encoder_set_max_residual_partition_order(plugin->encoder, 2);
			FLAC__stream_encoder_set_do_exhaustive_model_search(plugin->encoder, FALSE);
			break;

		case 1: FLAC__stream_encoder_set_max_lpc_order(plugin->encoder, 0);
			FLAC__stream_encoder_set_blocksize(plugin->encoder, 1152);
			FLAC__stream_encoder_set_do_mid_side_stereo(plugin->encoder, TRUE);
			FLAC__stream_encoder_set_loose_mid_side_stereo(plugin->encoder, TRUE);
			FLAC__stream_encoder_set_min_residual_partition_order(plugin->encoder, 2);
			FLAC__stream_encoder_set_max_residual_partition_order(plugin->encoder, 2);
			FLAC__stream_encoder_set_do_exhaustive_model_search(plugin->encoder, FALSE);
			break;
		
		case 2: FLAC__stream_encoder_set_max_lpc_order(plugin->encoder, 0);
			FLAC__stream_encoder_set_blocksize(plugin->encoder, 1152);
			FLAC__stream_encoder_set_do_mid_side_stereo(plugin->encoder, TRUE);
			FLAC__stream_encoder_set_loose_mid_side_stereo(plugin->encoder, FALSE);
			FLAC__stream_encoder_set_min_residual_partition_order(plugin->encoder, 3);
			FLAC__stream_encoder_set_do_exhaustive_model_search(plugin->encoder, FALSE);
			break;
			


		case 3: FLAC__stream_encoder_set_max_lpc_order(plugin->encoder, 6);
			FLAC__stream_encoder_set_blocksize(plugin->encoder, 4608);
			FLAC__stream_encoder_set_do_mid_side_stereo(plugin->encoder, TRUE);
			FLAC__stream_encoder_set_loose_mid_side_stereo(plugin->encoder, FALSE);
			FLAC__stream_encoder_set_min_residual_partition_order(plugin->encoder, 3);
			FLAC__stream_encoder_set_max_residual_partition_order(plugin->encoder, 3);
			FLAC__stream_encoder_set_do_exhaustive_model_search(plugin->encoder, FALSE);
			break;

		case 4: FLAC__stream_encoder_set_max_lpc_order(plugin->encoder, 8);
			FLAC__stream_encoder_set_blocksize(plugin->encoder, 4608);
			FLAC__stream_encoder_set_do_mid_side_stereo(plugin->encoder, TRUE);
			FLAC__stream_encoder_set_loose_mid_side_stereo(plugin->encoder, TRUE);
			FLAC__stream_encoder_set_min_residual_partition_order(plugin->encoder, 3);
			FLAC__stream_encoder_set_max_residual_partition_order(plugin->encoder, 3);
			FLAC__stream_encoder_set_do_exhaustive_model_search(plugin->encoder, FALSE);
			break;



		case 5: FLAC__stream_encoder_set_max_lpc_order(plugin->encoder, 8);
			FLAC__stream_encoder_set_blocksize(plugin->encoder, 4608);
			FLAC__stream_encoder_set_do_mid_side_stereo(plugin->encoder, TRUE);
			FLAC__stream_encoder_set_loose_mid_side_stereo(plugin->encoder, FALSE);
			FLAC__stream_encoder_set_min_residual_partition_order(plugin->encoder, 3);
			FLAC__stream_encoder_set_max_residual_partition_order(plugin->encoder, 3);
			FLAC__stream_encoder_set_do_exhaustive_model_search(plugin->encoder, FALSE);
			break;

		case 6: FLAC__stream_encoder_set_max_lpc_order(plugin->encoder, 8);
			FLAC__stream_encoder_set_blocksize(plugin->encoder, 4608);
			FLAC__stream_encoder_set_do_mid_side_stereo(plugin->encoder, TRUE);
			FLAC__stream_encoder_set_loose_mid_side_stereo(plugin->encoder, FALSE);
			FLAC__stream_encoder_set_min_residual_partition_order(plugin->encoder, 4);
			FLAC__stream_encoder_set_do_exhaustive_model_search(plugin->encoder, FALSE);
			break;

		case 7: FLAC__stream_encoder_set_max_lpc_order(plugin->encoder, 8);
			FLAC__stream_encoder_set_blocksize(plugin->encoder, 4608);
			FLAC__stream_encoder_set_do_mid_side_stereo(plugin->encoder, TRUE);
			FLAC__stream_encoder_set_loose_mid_side_stereo(plugin->encoder, FALSE);
			FLAC__stream_encoder_set_min_residual_partition_order(plugin->encoder, 6);
			FLAC__stream_encoder_set_do_exhaustive_model_search(plugin->encoder, TRUE);
			break;

		case 8: FLAC__stream_encoder_set_max_lpc_order(plugin->encoder, 12);
			FLAC__stream_encoder_set_blocksize(plugin->encoder, 4608);
			FLAC__stream_encoder_set_do_mid_side_stereo(plugin->encoder, TRUE);
			FLAC__stream_encoder_set_loose_mid_side_stereo(plugin->encoder, FALSE);
			FLAC__stream_encoder_set_min_residual_partition_order(plugin->encoder, 6);
			FLAC__stream_encoder_set_do_exhaustive_model_search(plugin->encoder, TRUE);
			break;
	}
        #endif

}

