% -----------------------------------------------------------------------------
%  (C) Altran Praxis Limited
% -----------------------------------------------------------------------------
% 
%  The SPARK toolset is free software; you can redistribute it and/or modify it
%  under terms of the GNU General Public License as published by the Free
%  Software Foundation; either version 3, or (at your option) any later
%  version. The SPARK toolset is distributed in the hope that it will be
%  useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
%  Public License for more details. You should have received a copy of the GNU
%  General Public License distributed with the SPARK toolset; see file
%  COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
%  the license.
% 
% =============================================================================

%###############################################################################
% PURPOSE
%-------------------------------------------------------------------------------
% Predicate to support saving, merging and viewing of usage data.
%###############################################################################

%###############################################################################
% MODULE
%###############################################################################

%###############################################################################
% DEPENDENCIES
%###############################################################################

:- use_module(gauge_simplifier,
              [view/1]).

:- use_module(data__switches,
              [get_switch_usage/1]).

:- use_module(library(file_systems)).

%###############################################################################
% TYPES
%###############################################################################

%###############################################################################
% DATA
%###############################################################################

%
% Predicate used to save all the usage data across across multiple
% executions of the Simplifier.
%

:- dynamic usage_data/3.

%
% Values for selection and resolution used in predicate profile_data.
%

selection([calls, choice_points, instructions]).
resolution([predicate, clause]).

%###############################################################################
% PREDICATES
%###############################################################################

%###############################################################################
% save_usage_data
%-------------------------------------------------------------------------------
% Save usage data. If usage data already exists then load in the data, merge
% and then save it.
%###############################################################################

save_usage_data :-
    get_switch_usage(provided_usage_file(Filename)),
    save_usage_data_x(Filename),
    !.

% File exists.
save_usage_data_x(Filename):-
    file_exists(Filename),
    !,
    load_files(Filename),
    process_all_usage_data,
    save_predicates([usage_data/3], Filename).

% File does not exist.
save_usage_data_x(Filename):-
    !,
    process_all_usage_data,
    save_predicates([usage_data/3], Filename).

%###############################################################################
% process_all_usage_data
%-------------------------------------------------------------------------------
% Collect new usage data and merge it with saved existing usage data if
% they exist.
%###############################################################################

process_all_usage_data :-
    selection(Sel_List),
    resolution(Res_List),
    process_usage_data_sel_list_res_list(Sel_List, Res_List).

process_usage_data_sel_list_res_list(Sel_List, Res_List):-
    member(Selection, Sel_List),
    member(Resolution, Res_List),
    process_usage_data_sel_res(Selection, Resolution),
    fail.

process_usage_data_sel_list_res_list(_Sel_List, _Res_List):-
    !.

process_usage_data_sel_res(Selection, Resolution):-
    usage_data(Selection, Resolution, ExistingData),
    !,
    profile_data([_], Selection, Resolution, Data),
    retract(usage_data(Selection, Resolution, ExistingData)),
    merge_pair_lists(Data, ExistingData, NewData),
    assert(usage_data(Selection, Resolution, NewData)).

process_usage_data_sel_res(Selection, Resolution):-
    \+ usage_data(Selection, Resolution, _),
    !,
    profile_data([_], Selection, Resolution, Data),
    assert(usage_data(Selection, Resolution, Data)).

%###############################################################################
% merge_pair_lists(+List1, +List2, -ListMerged)
%-------------------------------------------------------------------------------
% Merge two lists containing usage data together. Note that keysort
% does not remove duplicates so no data is lost.
%###############################################################################

merge_pair_lists(List1, List2, MergedList):-
    append(List1, List2, TmpList),
    keysort(TmpList, SortedList),
    process_duplicate_keys(SortedList, MergedList),
    !.

%###############################################################################
% process_duplicate_keys(+SortedList, -MergedSortedList[]).
%-------------------------------------------------------------------------------
% Merge two lists containing usage data. Note that keysort (see above)
% does not remove duplicates so no data is lost.
%###############################################################################

process_duplicate_keys([], []) :- !.

process_duplicate_keys([H], [H]) :- !.

% If the two elements at the head of the list have the same key then merge them.

process_duplicate_keys([H1-V1,H1-V2|T], [H1-V_sum|L]) :-
    V_sum is V1 + V2,
    process_duplicate_keys(T, L),
    !.

% Case when two head elements in the list have different keys.

process_duplicate_keys([H1-V1,H2-V2|T], [H1-V1|L]) :-
    process_duplicate_keys([H2-V2|T], L),
    !.

%###############################################################################
% view_usage.
%
% Predicate for viewing collected usage data through predicates provided
% by the modified gauge_simplifier.
%###############################################################################

view_usage:-
    get_switch_usage(provided_usage_file(Filename)),
    file_exists(Filename),
    load_files(Filename),
    view([_]).

%###############################################################################
% END-OF-FILE

