/* ====================================================================
 * Copyright (c) 2007-2008  Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "LayoutSettingsHandler.h"
#include "../config/ConfigData.h"

// qt
#include <QtCore/QRegExp>
#include <Qt3Support/Q3Header>


int LayoutSettingsHandler::getValue( ConfigData* cfg, const QString& key, int def )
{
  ConfigValue value = cfg->getValue( sc::String(key.toUtf8()) );
  
  if( value.isNull() )
  {
    return def;
  }
  
  return value.getNumericValue();
}

void LayoutSettingsHandler::setValue( ConfigData* cfg, const QString& key, int value )
{
  ConfigValue val( sc::String(key.toUtf8()), (long)value);
  cfg->setValue( val.getKey(), val );
}

QSize LayoutSettingsHandler::getSize( ConfigData* cfg, const QString& key, const QSize& def )
{
  ConfigValue sizeValue = cfg->getValue( sc::String(key.toUtf8()) );
  
  if( sizeValue.isNull() )
  {
    return def;
  }

  QString sizeVal = (const char*)sizeValue.getStringValue();
  QRegExp sizeExp("(\\d+),(\\d+)");
  sizeExp.exactMatch(sizeVal);

  return QSize( sizeExp.cap(1).toInt(), sizeExp.cap(2).toInt() );
}

void LayoutSettingsHandler::setSize( ConfigData* cfg, const QString& key, const QSize& size )
{
  QString sizeVal = QString("%1,%2").arg(size.width()).arg(size.height());
  
  ConfigValue value( sc::String(key.toUtf8()), sc::String(sizeVal.toUtf8()) );
  cfg->setValue( value.getKey(), value );
}

QRect LayoutSettingsHandler::getRect( ConfigData* cfg, const QString& key, const QRect& def )
{
  ConfigValue rectValue = cfg->getValue( sc::String(key.toUtf8()) );
  
  if( rectValue.isNull() )
  {
    return def;
  }
  
  QString rectVal = (const char*)rectValue.getStringValue();
  QRegExp rectExp("(\\d+),(\\d+),(\\d+),(\\d+)");
  rectExp.exactMatch(rectVal);

  QRect rect;
  rect.setLeft( rectExp.cap(1).toInt() );
  rect.setTop( rectExp.cap(2).toInt() );
  rect.setRight( rectExp.cap(3).toInt() );
  rect.setBottom( rectExp.cap(4).toInt() );
  return rect;
}

void LayoutSettingsHandler::setRect( ConfigData* cfg, const QString& key, const QRect& rect )
{
  QString rectVal  = QString("%1,%2,%3,%4")
    .arg(rect.left()).arg(rect.top()).arg(rect.right()).arg(rect.bottom());
  
  ConfigValue value( sc::String(key.toUtf8()), sc::String(rectVal.toUtf8()) );
  cfg->setValue( value.getKey(), value );
}

QByteArray LayoutSettingsHandler::getByteArray( ConfigData* cfg, const QString& key, const QByteArray& def )
{
  ConfigValue bytearrayValue = cfg->getValue( sc::String(key.toUtf8()) );
  
  if( bytearrayValue.isNull() )
  {
    return def;
  }

  return QByteArray::fromHex((const char*)bytearrayValue.getStringValue());
}

void LayoutSettingsHandler::setByteArray( ConfigData* cfg, const QString& key, const QByteArray& bytearray )
{
  ConfigValue value( sc::String(key.toUtf8()), sc::String(bytearray.toHex()) );
  cfg->setValue( value.getKey(), value );
}

void LayoutSettingsHandler::getHeaderColumns( ConfigData* cfg, const QString& key, Q3Header* header )
{
  ConfigValue value = cfg->getValue( sc::String(key.toUtf8()) );

  if( value.isNull() )
  {
    return;
  }
  
  QString qval = (const char*)value.getStringValue();

  int cnt = 0;
  while(true)
  {    
    QString section = qval.section( ',', cnt );

    if( section.isEmpty() )
      break;

    QRegExp exp("(\\d+):(\\d+)");
    exp.exactMatch(section);
    
    header->moveSection( exp.cap(1).toInt(), cnt );
    header->resizeSection( exp.cap(1).toInt(), exp.cap(2).toInt() );
    cnt++;
  }
}

void LayoutSettingsHandler::setHeaderColumns( ConfigData* cfg, const QString& key, const Q3Header* header )
{
  QString sections;
  
  for( int i = 0; i < header->count(); i++ )
  {
    QString v;
    int s = header->mapToSection(i);
    v.setNum(s);
    sections += v;
    sections += ":";    
    v.setNum( header->sectionSize(s) );
    sections += v;
    sections += ",";    
  }
  // strip trailing ","
  sections.truncate( sections.length()-1 );

  ConfigValue value( sc::String(key.toUtf8()), sc::String(sections.toUtf8()) );
  cfg->setValue( value.getKey(), value );
}
