// Copyright (C) 2009 Martin Sandve Alnes
//
// This file is part of SyFi.
//
// SyFi is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// SyFi is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with SyFi. If not, see <http://www.gnu.org/licenses/>.
//
// First added:  2009-01-01
// Last changed: 2009-04-01
//
// This demo program solves the hyperelasticity
// equations in 3D with a Fung type model.

#include <ctime>
#define TMIN 3.0
#define MMIN 1000

clock_t __tic_time;

void tic()
{
  __tic_time = clock();
}

double toc()
{
  clock_t __toc_time = clock();
  double elapsed_time = ((double) (__toc_time - __tic_time)) / CLOCKS_PER_SEC;
  return elapsed_time;
}


#include <iostream>
#include <vector>
#include <dolfin.h>
#include "generated_code/HyperElasticityFung.h"

using dolfin::message;
using dolfin::Function;
using dolfin::MeshFunction;
using dolfin::FunctionSpace;
using dolfin::SubDomain;
using dolfin::uint;
using dolfin::UnitCube;
using dolfin::DirichletBC;
using dolfin::BoundaryCondition;
using dolfin::VariationalProblem;
using dolfin::Vector;
using dolfin::Matrix;
using dolfin::File;

using namespace HyperElasticityFung;

using std::cout;
using std::endl;

const double eps = DOLFIN_EPS;

class Value: public Function
{
public:
  Value(const FunctionSpace & V, double value):
    Function(V), value(value)
  {}

  void eval(double* values, const double* x) const
  {
    values[0] = value;
  }

  double value;
};

class FiberField: public Function
{
public:
  FiberField(const FunctionSpace & V):
    Function(V)
  {}

  void eval(double* values, const double* x) const
  {
    const double theta = 0.0;
//    const double ct = cos(theta);
//    const double st = sin(theta);
    const double ct = 1.0;  
    const double st = 0.0; 


    // Fiber direction
    values[0] = ct;
    values[1] = st;
    values[2] = 0.0;

    // Sheet direction
    values[3] = -st;
    values[4] = ct;
    values[5] = 0.0;

    // Sheet normal direction
    values[6] = 0.0;
    values[7] = 0.0;
    values[8] = 1.0;
  }
};

const double a = 0.01;

class BodyForce: public Function
{
public:
  BodyForce(const FunctionSpace & V):
    Function(V)
  {}

  void eval(double* values, const double* x) const
  {
    values[0] = 0; 
    values[1] = 0; 
    values[2] = 0; 
  }
};

class BoundaryPressure: public Function
{
public:
  BoundaryPressure(const FunctionSpace & V):
    Function(V)
  {}

  void eval(double* values, const double* x) const
  {
    // TODO: Set some nonzero pressure
    values[0] = 0.0;
  }
};

class Displacement: public Function
{
public:
  Displacement(const FunctionSpace & V):
    Function(V)
  {}

  void eval(double* values, const double* x) const
  {
    const double dx = x[0] - 0.5;
    const double dy = x[1] - 0.5;
    const double dz = x[2] - 0.5;

    // Working:
//  values[0] =  0.05*dx;
//  values[1] = -0.05*dy;
//  values[2] =  0.0;

    // Not working:
//  values[0] =  0.1*dx;
//  values[1] = -0.1*dy;
//  values[2] =  0.0;

    // Not working:
//  values[0] =  sin(x[1]);
//  values[1] =  sin(x[0]);
//  values[2] =  0.0;

    // Not working:
//  values[0] = 0.1 * sin(x[1]);
//  values[1] = 0.1 * sin(x[0]);
//  values[2] = 0.1 * 0.0;

    // Working:
//  values[0] = 0.01 * sin(x[1]);
//  values[1] = 0.01 * sin(x[0]);
//  values[2] = 0.01 * 0.0;


    // Not working:
  values[0] = 0.0001*x[0];
  values[1] = 0.0; 
  values[2] = 0.0;


//    values[0] = a*( sin(x[1])-x[1]);
//    values[1] = -a*( x[2]-sin(x[2]));
//    values[2] = -( x[0]-sin(x[0]))*a;
  }
};

class PressureBoundary: public SubDomain
{
  bool inside(const double* x, bool on_boundary) const
  {
    //return (x[0] < eps) || (x[0] > 1.0-eps) || (x[1] < eps) || (x[1] > 1.0-eps) || (x[2] < eps) || (x[2] > 1.0-eps);
    return false;
  }
};

class DirichletBoundary: public SubDomain
{
  bool inside(const double* x, bool on_boundary) const
  {
    return x[0] < DOLFIN_EPS || x[0] > 1.0-DOLFIN_EPS;
//    return (x[0] < eps) || (x[0] > 1.0-eps) || (x[1] < eps) || (x[1] > 1.0-eps) || (x[2] < eps) || (x[2] > 1.0-eps);
  }
};

int main(int argc, char**argv)
{
    // Geometry
    message("Mesh");
    unsigned n = 6;
    UnitCube mesh(n, n, n);
    mesh.disp();

    // Boundaries
    message("Boundaries");
    unsigned pressure_boundary_marker  = 0;
    unsigned dirichlet_boundary_marker = 1;

    MeshFunction<unsigned> boundaries(mesh, mesh.geometry().dim()-1);

    PressureBoundary pressure_boundary;
    pressure_boundary.mark(boundaries, pressure_boundary_marker);
    
    DirichletBoundary dirichlet_boundary;
    dirichlet_boundary.mark(boundaries, dirichlet_boundary_marker);

    // Function spaces
    message("Function spaces");
    Form_a_F::TestSpace V(mesh);
    CoefficientSpace_sigma_0 P(mesh);
    CoefficientSpace_K C(mesh);
    CoefficientSpace_A Q(mesh);
    
    // Coefficient functions
    message("Functions");
    
    // Displacement at current and two previous timesteps
    Function u(V);
    Function up(V);
    Function upp(V);
    u.vector().zero();
    up.vector().zero();
    upp.vector().zero();

    // Time parameters
    Value dt(C, 1.0);
    
    // Fiber field
    FiberField A(Q);
    
    // External forces
    BodyForce g(V);
    BoundaryPressure sigma_0(P);
    
    // Material parameters
    Value rho(C, 1.0);
    Value K(C, 0.876);      // kPa // TODO: Get the right units
    Value C_compr(C, 10.0); // kPa
    Value bff(C, 18.48);
    Value bfx(C, 3.58);
    Value bxx(C, 2.8);
    
    // Analytical manufactured solution
    Displacement usol(V);

    // Attach coefficients to forms
    CoefficientSet coeffs;
    coeffs.u       = u;
    coeffs.up      = up;
    coeffs.upp     = upp;
    coeffs.dt      = dt;
    coeffs.A       = A;
    coeffs.g       = g;
    coeffs.sigma_0 = sigma_0;
    coeffs.rho     = rho;
    coeffs.K       = K;
    coeffs.C_compr = C_compr;
    coeffs.bff     = bff;
    coeffs.bfx     = bfx;
    coeffs.bxx     = bxx;
    
    // Forms
    message("Forms");
    Form_a_f f(coeffs);
    Form_a_F L(V, coeffs);
    Form_a_J a(V, V, coeffs);

    // Setup boundary conditions
    message("Boundary conditions");

    // Dirichlet boundary conditions
    DirichletBC bc(V, usol, boundaries, dirichlet_boundary_marker);

    std::vector<BoundaryCondition*> bcs;
    bcs.push_back(&bc);

    // Create variational problem (a, L, bcs, cell_domains, exterior_facet_domains, interior_facet_domains, nonlinear)
    message("Variational problem");
    VariationalProblem problem(a, L, bcs, 0, &boundaries, 0, true);
    
    // Compute and show linear system
    if(true)
    {
        message("================================================================================");
        message("Printing linear system");

        Matrix J;
        tic();
        problem.J(J, u.vector());
        cout << "Time to assemble J = " << toc() << endl;
        //J.disp();

        Vector F;
        tic();
        problem.F(F, u.vector());
        cout << "Time to assemble F = " << toc() << endl;
        //F.disp();
        cout << "Size of system = " << F.size() << endl;
        message("================================================================================");
    }
    return 0;

    // Solve!
    message("Solving");
    problem.set("linear solver", "direct");
    problem.solve(u);

    // Postprocessing
    message("Postprocessing");
    
    // Interpolate usol
    Function usol_h(V);
    usol.interpolate(usol_h.vector(), V);

    // Compute functional value of u and exact solution
    f.u = u;
    double f_value = assemble(f);
    f.u = usol_h;
    double f_value2 = assemble(f);

    // Print limits of u
    cout << endl;
    cout << "==========================" << endl;
    cout << "Norm of u_h = " << u.vector().norm() << endl;
    cout << "Min  of u_h = " << u.vector().min() << endl;
    cout << "Max  of u_h = " << u.vector().max() << endl;
    cout << endl;
    cout << "Norm of u_e = " << usol_h.vector().norm() << endl;
    cout << "Min  of u_e = " << usol_h.vector().min() << endl;
    cout << "Max  of u_e = " << usol_h.vector().max() << endl;
    cout << endl;
    cout << "f(u) = " << f_value << endl;
    cout << "f(u_exact) = " << f_value2 << endl;
    cout << "(u_exact is not correct)" << endl;
    cout << "==========================" << endl;

    // Compute F at u for verification
    Vector F;
    problem.F(F, u.vector());
    cout << endl;
    cout << "==========================" << endl;
    cout << "Norm of F = " << F.norm() << endl;
    cout << "Min  of F = " << F.min() << endl;
    cout << "Max  of F = " << F.max() << endl;
    cout << "==========================" << endl;
    
    // Compute e_h = usol_h - u
    Function e_h(V);
    e_h.vector() = u.vector();
    e_h.vector() -= usol_h.vector();
    cout << endl;
    cout << "==========================" << endl;
    cout << "Norm of e_h = " << e_h.vector().norm() << endl;
    cout << "Min  of e_h = " << e_h.vector().min() << endl;
    cout << "Max  of e_h = " << e_h.vector().max() << endl;
    cout << "==========================" << endl;
    
    // Interpolate fiber field
    Function Ah(Q);
    A.interpolate(Ah.vector(), Q);

    //plot(w);
    
    // Write functions to file
    message("Writing to file");
    
    File ufile("u.pvd");
    ufile << u;
    
    File usolfile("usol.pvd");
    usolfile << usol_h;
    
    File efile("e.pvd");
    efile << e_h;

    File Afile("A.pvd");
    Afile << Ah;

    message("Done!");
}

