/*****************************************************************************
    TRAVIS - Trajectory Analyzer and Visualizer
    http://www.travis-analyzer.de/

    Copyright (c) 2009-2017 Martin Brehm
                  2012-2017 Martin Thomas

    This file written by Martin Brehm.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/


// This must always be the first include directive
#include "config.h"

#include "cc_tools.h"
#include <math.h>
#include <algorithm>
#include "cc_engine.h"
#include "cc_hilbert.h"
#include "cc_integerengine.h"
#include "cc_bitset.h"
#include "cc_crc32.h"
#include "barbecube.h"


std::vector<int> gc_iaAtomOrd;


CCCEngine::CCCEngine() {
}


CCCEngine::~CCCEngine() {

	int z;

	for (z=0;z<(int)m_oaInputAtomBuf.size();z++)
		delete m_oaInputAtomBuf[z];
	for (z=0;z<(int)m_oaInputCubeBuf.size();z++)
		delete m_oaInputCubeBuf[z];
	for (z=0;z<(int)m_oaOutputAtomBuf.size();z++)
		delete m_oaOutputAtomBuf[z];
	for (z=0;z<(int)m_oaOutputCubeBuf.size();z++)
		delete m_oaOutputCubeBuf[z];
	for (z=0;z<(int)m_oaOutput2AtomBuf.size();z++)
		delete m_oaOutput2AtomBuf[z];
	for (z=0;z<(int)m_oaOutput2CubeBuf.size();z++)
		delete m_oaOutput2CubeBuf[z];
}


#define AR_EPS (0.0001)
#define CR_EPS (0.0001)


void CCCEngine::ResetStatistics() {

	gc_iOutputSize = 0;
	gc_iOutputSizeAlphabet = 0;
	gc_iOutputSizeHuffmanTables = 0;
	gc_iOutputSizeTableSwitch = 0;
	gc_iOutputSizeHuffmanData = 0;
	gc_iCount = 0;

	gc_iCOutputSize = 0;
	gc_iCOutputSizeAlphabet = 0;
	gc_iCOutputSizeHuffmanTables = 0;
	gc_iCOutputSizeTableSwitch = 0;
	gc_iCOutputSizeHuffmanData = 0;
	gc_iCCount = 0;
}


void CCCEngine::BeginStatistics() {

	gc_iTempOutputSizeAlphabet = 0;
	gc_iTempOutputSizeHuffmanTables = 0;
	gc_iTempOutputSizeTableSwitch = 0;
	gc_iTempOutputSizeHuffmanData = 0;
}


void CCCEngine::EndStatistics(bool n, bool c) {

	gc_iOutputSizeAlphabet += gc_iTempOutputSizeAlphabet;
	gc_iOutputSizeHuffmanTables += gc_iTempOutputSizeHuffmanTables;
	gc_iOutputSizeTableSwitch += gc_iTempOutputSizeTableSwitch;
	gc_iOutputSizeHuffmanData += gc_iTempOutputSizeHuffmanData;
	if (n)
		gc_iCount++;

	if (c) {
		gc_iCOutputSizeAlphabet += gc_iTempOutputSizeAlphabet;
		gc_iCOutputSizeHuffmanTables += gc_iTempOutputSizeHuffmanTables;
		gc_iCOutputSizeTableSwitch += gc_iTempOutputSizeTableSwitch;
		gc_iCOutputSizeHuffmanData += gc_iTempOutputSizeHuffmanData;
		if (n)
			gc_iCCount++;
	}
}


void CCCEngine::BuildHilbertIdx(int resx, int resy, int resz, bool verbose) {

	int b, bt;
	int resxyz, resyz, ti;
	bitmask_t mi, index, coords[3];

	if (verbose)
		mprintf("\nCreating Hilbert curve index table...\n");
//	else
//		mprintf("      Creating Hilbert curve index table...\n");

	b = (int)(ceil(mylog2(resx))+0.5);
	bt = (int)(ceil(mylog2(resy))+0.5);
	if (bt > b)
		b = bt;
	bt = (int)(ceil(mylog2(resz))+0.5);
	if (bt > b)
		b = bt;
	mi = 2<<(3*b-1);

	resxyz = resx*resy*resz;
	resyz = resy*resz;

	if (verbose) {
		mprintf("Resolution %d | %d | %d --> Bits %d\n",resx,resy,resz,b);
		mprintf("Requires %lu indices, %.3f MiB of RAM.\n",mi,resxyz*sizeof(int)/1024.0/1024.0);
		mprintf("Running...\n");
		mprintf("        [");
		fflush(stdout);
	}

	m_iaHilbertIdx.resize(resxyz);
	ti = 0;
	for (index=0;index<mi;index++) {
		if (verbose)
			if (fmod(index,mi/60.0) < 1.0) {
				mprintf("#");
				fflush(stdout);
			}
		hilbert_i2c(3,b,index,coords);
		if (((long)coords[2] >= resx) || ((long)coords[1] >= resy) || ((long)coords[0] >= resz))
			continue;
		m_iaHilbertIdx[ti] = coords[2]*resyz+coords[1]*resy+coords[0];
		ti++;
	}
	if (verbose) {
		mprintf("]\n");
		mprintf("Finished. %lu entries.\n\n",m_iaHilbertIdx.size());
	}
}


bool CCCEngine::CubeToIntegerArray(std::vector<int> &outp, int order, int eps, int signi, int split, bool hilbert, bool verbose) {

	int z, z2, ti, tiov, tiuf, tiuf2, zx, zy, zz, zi, ii, ex, res, mantis;
	CCubeFrame *cfr;
	double tf, tf2, tf3, tfa2;
	int maxsymb;
	bool uf, of;

	if (verbose)
		mprintf("** CubeToIntegerArray order=%d ***\n",order);
	
	cfr = GetInputCubeFrame(0);

	m_faTempPred.resize(cfr->m_iResXYZ);
	m_faTempPred2.resize(cfr->m_iResXYZ);
	for (z=0;z<cfr->m_iResXYZ;z++)
		m_faTempPred[z] = 0;

	if (verbose)
		mprintf("Polynomial extrapolation...\n");

	for (z=1;z<order+1;z++) {
		cfr = GetInputCubeFrame(z);
		tf = 1.0;
		for (z2=1;z2<=order;z2++) { 
			if (z2 == z)
				continue;
			tf *= z2 / ((double)z2 - z);
		}
		for (z2=0;z2<cfr->m_iResXYZ;z2++)
			m_faTempPred[z2] += tf * cfr->m_iaMantis[z2] * pow10(cfr->m_iaExpo[z2]);
	}

	cfr = GetInputCubeFrame(0);

	maxsymb = pow10i(signi)-1;

	if (verbose)
		mprintf("Neighborhood extrapolation...\n");

	zi = 0;
	for (zx=0;zx<cfr->m_iRes[0];zx++) {
		for (zy=0;zy<cfr->m_iRes[1];zy++) {
			for (zz=0;zz<cfr->m_iRes[2];zz++) {

				tf2 = 0;
				tf3 = 0;

				if (zz > 0) {
					tf2++;
					tf3 += ((double)cfr->m_iaMantis[zi-1]*pow10(cfr->m_iaExpo[zi-1])) - m_faTempPred[zi-1];
				}

				if (zy > 0) {
					tf2++;
					tf3 += ((double)cfr->m_iaMantis[zi-cfr->m_iRes[2]]*pow10(cfr->m_iaExpo[zi-cfr->m_iRes[2]])) - m_faTempPred[zi-cfr->m_iRes[2]];
				}

				if (zx > 0) {
					tf2++;
					tf3 += ((double)cfr->m_iaMantis[zi-cfr->m_iResYZ]*pow10(cfr->m_iaExpo[zi-cfr->m_iResYZ])) - m_faTempPred[zi-cfr->m_iResYZ];
				}

				if (tf2 != 0)
					tf3 /= tf2 * 1.075;

				tf3 += m_faTempPred[zi];

				m_faTempPred2[zi] = tf3;

				zi++;
			}
		}
	}

	if (verbose)
		mprintf("Discretizing and serializing symbols...\n");

	tiov = 0;
	tiuf = 0;
	tiuf2 = 0;
	tfa2 = 0;
	for (zi=0;zi<cfr->m_iResXYZ;zi++) {

		if (hilbert)
			ii = m_iaHilbertIdx[zi];
		else
			ii = zi;

//		if (zi < 10)
//			mprintf("%d --> %d\n",zi,ii);

		if (m_faTempPred2[ii] == 0)
			ex = -eps;
		else
			ex = (int)floor(log10(fabs(m_faTempPred2[ii])))-signi+1;

		if (ex < -eps)
			ex = -eps;

		uf = false;
		of = false;

_again:
		mantis = (int)floor(cfr->m_faBin[ii]*pow10(-ex)+0.5);

		if (floor(m_faTempPred2[ii]*pow10(-ex)+0.5+CR_EPS) != floor(m_faTempPred2[ii]*pow10(-ex)+0.5-CR_EPS)) {
			ti = (int)floor(m_faTempPred2[ii]*pow10(-ex));
	//		mprintf("Sanitized rounding of %f.\n",m_faTempPred2[ii]*pow10(-ex));
		} else
			ti = (int)floor(m_faTempPred2[ii]*pow10(-ex)+0.5);

		res = mantis - ti;

		if (ExpMantisEqual(cfr->m_iaExpo[ii],cfr->m_iaMantis[ii],ex,mantis) && (abs(res) <= maxsymb)) {
			if (uf)
				outp.push_back(C_UNDERFLOW);
			if (of)
				outp.push_back(C_OVERFLOW);
			if (abs(res) >= split) {
	//			if (ii == 549755)
	//				mprintf("Equal with split: %d = %d & %d\n",res,res/split,abs(res)%split);
				outp.push_back(C_SPLIT);
				outp.push_back(res/split);
				if (res < 0)
					outp.push_back(-(abs(res)%split));
				else
					outp.push_back(res%split);
			} else {
				outp.push_back(res);
	//			if (ii == 549755)
	//				mprintf("Equal without split: %d\n",res);
			}
		} else if ((ex == cfr->m_iaExpo[ii]+1) && (ex > -eps)) {
			tiuf2++;
			ex--;
	//		tia.push_back(maxsymb+2);
			uf = true;
	//		if (ii == 549755)
	//			mprintf("Soft Underflow 2.\n");
			goto _again;
		} else if (ex == cfr->m_iaExpo[ii]-1) {
			tiuf2++;
			ex++;
			of = true;
	//		if (ii == 549755)
	//			mprintf("Soft Underflow 2.\n");
			goto _again;
		} else if (ex > cfr->m_iaExpo[ii]) {
			tiuf++;
			outp.push_back(C_FULLNUMBER);
			outp.push_back(cfr->m_iaExpo[ii]);
			outp.push_back(cfr->m_iaMantis[ii]/split);
			if (cfr->m_iaMantis[ii] < 0)
				outp.push_back(-(abs(cfr->m_iaMantis[ii])%split));
			else
				outp.push_back(cfr->m_iaMantis[ii]%split);
			if (verbose)
				mprintf("Underflow %d\n",ex-cfr->m_iaExpo[ii]);
	//		if (ii == 549755)
	//			mprintf("Underflow %d\n",ex-cfr->m_iaExpo[ii]);
		} else if (abs(res) > maxsymb) {
			tiov++;
			outp.push_back(C_FULLNUMBER);
	//		if (ii == 549755)
	//			mprintf("@0: %d\n",outp[outp.size()-1]);
			outp.push_back(cfr->m_iaExpo[ii]);
			outp.push_back(cfr->m_iaMantis[ii]/split);
			if (cfr->m_iaMantis[ii] < 0)
				outp.push_back(-(abs(cfr->m_iaMantis[ii])%split));
			else
				outp.push_back(cfr->m_iaMantis[ii]%split);
	//		if (ii == 549755)
	//			mprintf("Full Number: %d %d %d\n",cfr->m_iaExpo[ii],cfr->m_iaMantis[ii]/split,abs(cfr->m_iaMantis[ii])%split);
		} else {
			if (verbose)
				mprintf("UE: %d, val=%.10f, pred=%.10f, cexpo=%d, pexpo=%d, cmantis=%d, pmantis=%d, res=%d.\n",ii,cfr->m_faBin[ii],m_faTempPred2[ii],cfr->m_iaExpo[ii],ex,cfr->m_iaMantis[ii],mantis,res);
			outp.push_back(C_FULLNUMBER);
			outp.push_back(cfr->m_iaExpo[ii]);
			outp.push_back(cfr->m_iaMantis[ii]/split);
			if (cfr->m_iaMantis[ii] < 0)
				outp.push_back(-(abs(cfr->m_iaMantis[ii])%split));
			else
				outp.push_back(cfr->m_iaMantis[ii]%split);
		}

	//	if (ii == 549755)
	//		mprintf("UE: idx=%6d, val=%.10f, pred=%.22f, cexpo=%d, pexpo=%d, cmantis=%d, pmantis=%d, res=%d.\n",ii,cfr->m_faBin[ii],m_faTempPred2[ii],cfr->m_iaExpo[ii],ex,cfr->m_iaMantis[ii],mantis,res);

		tfa2 += res;
	}

	if (verbose) {
		tfa2 /= cfr->m_iResXYZ;
		mprintf("      tfa2 = %8.6f\n",tfa2);
		mprintf("      Overflow: %d, Underflow: %d, Soft Underflow: %d\n",tiov,tiuf,tiuf2);
		mprintf("      Produced %lu symbols.\n",outp.size());
	}

	return true;
}


bool CCCEngine::IntegerArrayToCube(std::vector<int> &inp, int order, int eps, int signi, int split, bool hilbert, bool verbose, bool second) {

	int z, z2, zx, zy, zz, zi, ii, ex;
	CCubeFrame *cfr, *ofr;
	double tf, tf2, tf3;
//	int maxsymb;
	std::vector<int> tresi, texpo, tmantis;

	if (verbose)
		mprintf("** IntegerArrayToCube order=%d ***\n",order);
	
	if (second)
		ofr = GetOutput2CubeFrame(0);
	else
		ofr = GetOutputCubeFrame(0);

	m_faTempPred.resize(ofr->m_iResXYZ);
	for (z=0;z<ofr->m_iResXYZ;z++)
		m_faTempPred[z] = 0;

	if (verbose)
		mprintf("Polynomial extrapolation...\n");

	for (z=1;z<order+1;z++) {
		if (second)
			cfr = GetOutput2CubeFrame(z);
		else
			cfr = GetOutputCubeFrame(z);
		if (cfr == NULL) {
			eprintf("CCCEngine::IntegerArrayToCube(): Error: Frame is of order %d, step history is too short (%d).\n",order,z);
			abort();
		}
		tf = 1.0;
		for (z2=1;z2<=order;z2++) { 
			if (z2 == z)
				continue;
			tf *= z2 / ((double)z2 - z);
		}
		for (z2=0;z2<ofr->m_iResXYZ;z2++)
			m_faTempPred[z2] += tf * cfr->m_iaMantis[z2] * pow10(cfr->m_iaExpo[z2]);
	}

//	maxsymb = pow10i(signi)-1;

	tresi.resize(ofr->m_iResXYZ);
	texpo.resize(ofr->m_iResXYZ);
	tmantis.resize(ofr->m_iResXYZ);

	if (verbose)
		mprintf("Decoding input...\n");

	zi = 0;
	z = 0;
	while (zi < (int)inp.size()) {
		if (hilbert)
			ii = m_iaHilbertIdx[z];
		else
			ii = z;
		if (inp[zi] == C_FULLNUMBER) { // Full Number
			texpo[ii] = inp[++zi];
			tmantis[ii] = inp[++zi]*split;
			tmantis[ii] += inp[zi+1];
	//		if (inp[zi+1] < 0)
	//			tmantis[ii] = -tmantis[ii];
//			if (ii == 549755)
//				mprintf("Full Number: %d %d %d --> %d %d\n",inp[zi-1],inp[zi],inp[zi+1],texpo[ii],tmantis[ii]);
			zi++;
		} else if (inp[zi] == C_UNDERFLOW) { // Soft Underflow
			if (inp[++zi] == C_SPLIT) {
//				if (ii == 549755)
//					mprintf("Soft Underflow with split: %d & %d = %d\n",inp[zi+1],inp[zi+2],inp[zi+1]*split+inp[zi+2]);
				tresi[ii] = inp[++zi]*split;
				tresi[ii] += inp[zi+1];
	//			if (inp[zi+1] < 0)
	//				tresi[ii] = -tresi[ii];
				zi++;
//				if (ii == 549755)
//					mprintf("--> %d\n",tresi[ii]);
			} else {
//				if (ii == 549755)
//					mprintf("Soft Underflow without split.\n");
				tresi[ii] = inp[zi];
			}
			texpo[ii] = -101;
		} else if (inp[zi] == C_OVERFLOW) { // Soft Overflow
			if (inp[++zi] == C_SPLIT) {
//				if (ii == 549755)
//					mprintf("Soft Underflow with split: %d & %d = %d\n",inp[zi+1],inp[zi+2],inp[zi+1]*split+inp[zi+2]);
				tresi[ii] = inp[++zi]*split;
				tresi[ii] += inp[zi+1];
	//			if (inp[zi+1] < 0)
	//				tresi[ii] = -tresi[ii];
				zi++;
//				if (ii == 549755)
//					mprintf("--> %d\n",tresi[ii]);
			} else {
//				if (ii == 549755)
//					mprintf("Soft Underflow without split.\n");
				tresi[ii] = inp[zi];
			}
			texpo[ii] = -102;
		} else if (inp[zi] == C_SPLIT) { // Splitted Residue
			tresi[ii] = inp[++zi]*split;
			tresi[ii] += inp[zi+1];
	//		if (inp[zi+1] < 0)
	//			tresi[ii] = -tresi[ii];
			zi++;
			texpo[ii] = -100;
//			if (ii == 549755)
//				mprintf("Splitted residue.\n");
		} else { // Standard Residue
			tresi[ii] = inp[zi];
			texpo[ii] = -100;
//			if (ii == 549755)
//				mprintf("Standard residue.\n");
		}
//		if (ii == 549755)
//			mprintf("resi=%d, expo=%d, mantis=%d\n",tresi[ii],texpo[ii],tmantis[ii]);
		zi++;
		z++;
	}

	if (verbose) {
		mprintf("Read %d symbols and %d grid points.\n",zi,z);
		mprintf("Neighborhood extrapolation...\n");
	}

	zi = 0;
	for (zx=0;zx<ofr->m_iRes[0];zx++) {
//		mprintf("zx=%d...\n",zx);
		for (zy=0;zy<ofr->m_iRes[1];zy++) {
//			mprintf("zy=%d...\n",zy);
			for (zz=0;zz<ofr->m_iRes[2];zz++) {
//				mprintf("zz=%d...\n",zz);

				if (texpo[zi] > -100) {
					ofr->m_iaMantis[zi] = tmantis[zi];
					ofr->m_iaExpo[zi] = texpo[zi];
					ofr->m_faBin[zi] = ofr->m_iaMantis[zi] * pow10(ofr->m_iaExpo[zi]);
					zi++;
					continue;
				}

				tf2 = 0;
				tf3 = 0;

				if (zz > 0) {
					tf2++;
					tf3 += ((double)ofr->m_iaMantis[zi-1]*pow10(ofr->m_iaExpo[zi-1])) - m_faTempPred[zi-1];
				}

				if (zy > 0) {
					tf2++;
					tf3 += ((double)ofr->m_iaMantis[zi-ofr->m_iRes[2]]*pow10(ofr->m_iaExpo[zi-ofr->m_iRes[2]])) - m_faTempPred[zi-ofr->m_iRes[2]];
				}

				if (zx > 0) {
					tf2++;
					tf3 += ((double)ofr->m_iaMantis[zi-ofr->m_iResYZ]*pow10(ofr->m_iaExpo[zi-ofr->m_iResYZ])) - m_faTempPred[zi-ofr->m_iResYZ];
				}

				if (tf2 != 0)
					tf3 /= tf2 * 1.075;

				tf3 += m_faTempPred[zi];

				if (tf3 == 0)
					ex = -eps;
				else
					ex = (int)floor(log10(fabs(tf3)))-signi+1;

				if (ex < -eps)
					ex = -eps;

				if (texpo[zi] == -101)
					ex--;
				else if (texpo[zi] == -102)
					ex++;

				if (floor(tf3*pow10(-ex)+0.5+CR_EPS) != floor(tf3*pow10(-ex)+0.5-CR_EPS)) {
					ofr->m_iaMantis[zi] = (int)floor(tf3*pow10(-ex));
	//				mprintf("Sanitized rounding of %f.\n",tf3*pow10(-ex));
				} else
					ofr->m_iaMantis[zi] = (int)floor(tf3*pow10(-ex)+0.5);

				ofr->m_iaMantis[zi] += tresi[zi];
				ofr->m_iaExpo[zi] = ex;
				while (ofr->m_iaMantis[zi] >= pow10i(signi)) {
//					if (zi == 549755)
//						mprintf("A %d %d %d\n",ofr->m_iaMantis[zi],ofr->m_iaExpo[zi],signi);
					ofr->m_iaMantis[zi] /= 10;
					ofr->m_iaExpo[zi]++;
				}
				while ((ofr->m_iaMantis[zi] < pow10i(signi-1)) && (ofr->m_iaExpo[zi] > -eps)) {
//					if (zi == 549755)
//						mprintf("B %d %d %d\n",ofr->m_iaMantis[zi],ofr->m_iaExpo[zi],signi);
					ofr->m_iaMantis[zi] *= 10;
					ofr->m_iaExpo[zi]--;
				}
				ofr->m_faBin[zi] = ofr->m_iaMantis[zi] * pow10(ofr->m_iaExpo[zi]);

//				if (zi == 549755)
//					mprintf("%d: Pred=%.22f, expo=%d, predmantis=%d, resi=%d, mantis=%d, result=%.10f\n",zi,tf3,ex,(int)floor(tf3*pow10(-ex)+0.5),tresi[zi],ofr->m_iaMantis[zi],ofr->m_faBin[zi]);

				zi++;
			}
		}
	}

	if (verbose)
		mprintf("All done.\n");

	return true;
}


void CCCEngine::ExportCubeHeader(CBitSet *bs, int order, bool verbose) {

	CCubeFrame *cfr;
	int i;

	if (verbose)
		mprintf("Writing cube header...\n");

	if (order != 0) {
		if (verbose)
			mprintf("Cube header already written before.\n");
		bs->WriteBit(0);
		return;
	} else
		bs->WriteBit(1);

	i = bs->GetLength();

	cfr = GetInputCubeFrame(0);

	bs->WriteBits(cfr->m_iRes[0],10);
	bs->WriteBits(cfr->m_iRes[1],10);
	bs->WriteBits(cfr->m_iRes[2],10);

	bs->WriteBits(cfr->m_iCenter[0],32);
	bs->WriteBits(cfr->m_iCenter[1],32);
	bs->WriteBits(cfr->m_iCenter[2],32);

	bs->WriteBits(cfr->m_iStride[0],32);
	bs->WriteBits(0,32);
	bs->WriteBits(0,32);

	bs->WriteBits(0,32);
	bs->WriteBits(cfr->m_iStride[1],32);
	bs->WriteBits(0,32);

	bs->WriteBits(0,32);
	bs->WriteBits(0,32);
	bs->WriteBits(cfr->m_iStride[2],32);

	if (verbose)
		mprintf("%d bytes written.\n",(bs->GetLength()-i)/8);
}


void CCCEngine::ImportCubeHeader(CBitSet *bs, bool verbose, bool second) {

	CCubeFrame *ofr, *ofr2;
	int z, z2, i;

	if (second)
		ofr = GetOutput2CubeFrame(0);
	else
		ofr = GetOutputCubeFrame(0);

	if (verbose)
		mprintf("Reading cube header...\n");

	i = bs->GetReadPos();

	if (!bs->ReadBit()) { // Header already found in previous frame
		if (second)
			ofr2 = GetOutput2CubeFrame(1);
		else
			ofr2 = GetOutputCubeFrame(1);
		if (ofr2 == NULL) {
			mprintf("CEngine::ImportCubeHeader(): Error: First frame does not contain header.\n");
			return;
		}
		if (verbose)
			mprintf("Cube header already read before.\n");
		for (z=0;z<3;z++) {
			ofr->m_iRes[z] = ofr2->m_iRes[z];
			ofr->m_iCenter[z] = ofr2->m_iCenter[z];
			ofr->m_iStride[z] = ofr2->m_iStride[z];
			ofr->m_fCenter[z] = ofr2->m_fCenter[z];
			ofr->m_fStride[z] = ofr2->m_fStride[z];
		}
	} else { // Header here

		for (z=0;z<3;z++)
			ofr->m_iRes[z] = bs->ReadBitsInteger(10);

		for (z=0;z<3;z++) {
			ofr->m_iCenter[z] = bs->ReadBitsInteger(32);
			ofr->m_fCenter[z] = FixedToFloat(ofr->m_iCenter[z],6);
		}

		for (z=0;z<3;z++) {
			ofr->m_iStride[z] = bs->ReadBitsInteger(32);
			ofr->m_fStride[z] = FixedToFloat(ofr->m_iStride[z],6);
			if (z != 2)
				for (z2=0;z2<3;z2++)
					bs->ReadBitsInteger(32);
		}
	}

	for (z=0;z<3;z++) {
		ofr->m_fMinVal[z] = ofr->m_fCenter[z];
		ofr->m_fMaxVal[z] = ofr->m_fCenter[z] + ofr->m_fStride[z] * ofr->m_iRes[z];
	}

	if (verbose) {
		mprintf("Resolution: %d x %d x %d\n",ofr->m_iRes[0],ofr->m_iRes[1],ofr->m_iRes[2]);
		mprintf("Center: %f | %f | %f\n",ofr->m_fCenter[0],ofr->m_fCenter[1],ofr->m_fCenter[2]);
		mprintf("Stride: %f | %f | %f\n",ofr->m_fStride[0],ofr->m_fStride[1],ofr->m_fStride[2]);
		mprintf("Range: X %f - %f, Y %f - %f, Z %f - %f\n",ofr->m_fMinVal[0],ofr->m_fMaxVal[0],ofr->m_fMinVal[1],ofr->m_fMaxVal[1],ofr->m_fMinVal[2],ofr->m_fMaxVal[2]);
		mprintf("%d bytes read.\n",(bs->GetReadPos()-i)/8);
	}

	ofr->m_iResXY = ofr->m_iRes[0] * ofr->m_iRes[1];
	ofr->m_iResYZ = ofr->m_iRes[1] * ofr->m_iRes[2];
	ofr->m_iResXYZ = ofr->m_iRes[0] * ofr->m_iRes[1] * ofr->m_iRes[2];

	ofr->m_faBin.resize(ofr->m_iResXYZ);
	ofr->m_iaExpo.resize(ofr->m_iResXYZ);
	ofr->m_iaMantis.resize(ofr->m_iResXYZ);
}


void CCCEngine::AtomsToIntegerArray(std::vector<int> &outp, int order, int asigni, int esplit, bool verbose) {

	int z, z2, i, zc, zi;
	std::vector<double> tpred;
	double compred;
	long mantis, ti, res;
	int split;
	double tf;
	CAtomSet *cfr;

	if (verbose)
		mprintf("\n*** AtomsToIntegerArray order=%d ***\n",order);

	i = (int)outp.size();

	cfr = GetInputAtomFrame(0);

	if (verbose)
		mprintf("Processing %lu atoms...\n",cfr->m_oaAtoms.size());

	tpred.resize(cfr->m_oaAtoms.size());

	split = 1<<esplit;

	for (zc=0;zc<3;zc++) {

		if (verbose)
			mprintf("*** Now coordinate %d/3...\n",zc+1);
	
		if (verbose)
			mprintf("Polynomial extrapolation...\n");

		for (z=0;z<(int)cfr->m_oaAtoms.size();z++)
			tpred[z] = 0;
		compred = 0;

		for (z=1;z<order+1;z++) {
			cfr = GetInputAtomFrame(z);
//			mprintf("#%d: %lu\n",z,cfr->m_oaAtoms.size());
			tf = 1.0;
			for (z2=1;z2<=order;z2++) { 
				if (z2 == z)
					continue;
				tf *= z2 / ((double)z2 - z);
			}
//			if (zc == 2)
//				mprintf("@%d: %f\n",z,tf);
			for (z2=0;z2<(int)cfr->m_oaAtoms.size();z2++)
				tpred[z2] += tf * cfr->m_oaAtoms[m_iaAtomSort[z2]]->m_fRelCoord[zc];
//			if (zc == 2)
//				mprintf("   + %.2f * %f\n",tf,cfr->m_oaAtoms[0]->m_fRelCoord[zc]);
			compred += tf * cfr->m_faCOM[zc];
		}
//		if (zc == 2)
//			mprintf("   = %f\n",tpred[0]);

		cfr = GetInputAtomFrame(0);

		if (verbose)
			mprintf("Discretizing and serializing symbols...\n");

		// COM component
		mantis = cfr->m_iaCOM[zc];
		if (floor(compred*pow10(asigni)+0.5+AR_EPS) != floor(compred*pow10(asigni)+0.5-AR_EPS)) {
			ti = (int)floor(compred*pow10(asigni));
//			mprintf("1A) Sanitized rounding of %f.\n",compred*pow10(asigni));
		} else
			ti = (int)floor(compred*pow10(asigni)+0.5);
		res = mantis - ti;
		if (abs(res) >= split) {
			outp.push_back(C_SPLIT);
			outp.push_back(res/split);
			if (res < 0)
				outp.push_back(-(abs(res)%split));
			else
				outp.push_back(res%split);
		} else {
			outp.push_back(res);
		}

//		if (zc == 2)
//			mprintf("COM[%d]: %.6f (%ld).\n",zc,cfr->m_faCOM[zc],cfr->m_iaCOM[zc]);

		for (zi=0;zi<(int)cfr->m_oaAtoms.size();zi++) {

			mantis = cfr->m_oaAtoms[m_iaAtomSort[zi]]->m_iRelCoord[zc];

			if (floor(tpred[zi]*pow10(asigni)+0.5+AR_EPS) != floor(tpred[zi]*pow10(asigni)+0.5-AR_EPS)) {
				ti = (int)floor(tpred[zi]*pow10(asigni));
//				mprintf("1B) %3d Sanitized rounding of %f.\n",zi,tpred[zi]*pow10(asigni));
			} else
				ti = (int)floor(tpred[zi]*pow10(asigni)+0.5);

			res = mantis - ti;

	//		if ((zc==2) && (m_iaAtomSort[zi] == 92)) {
	//			mprintf("Atom %d[%d]: fa=%.6f, ia=%ld, fr=%.6f, ir=%ld.\n",m_iaAtomSort[zi],zc,cfr->m_oaAtoms[m_iaAtomSort[zi]]->m_fCoord[zc],cfr->m_oaAtoms[m_iaAtomSort[zi]]->m_iCoord[zc],cfr->m_oaAtoms[m_iaAtomSort[zi]]->m_fRelCoord[zc],cfr->m_oaAtoms[m_iaAtomSort[zi]]->m_iRelCoord[zc]);
	//			mprintf("enc %2d: pred %d, mantis %d, res %d\n",m_iaAtomSort[zi],ti,mantis,res);
	//		}

			if (abs(res) >= split) {
				outp.push_back(C_SPLIT);
				outp.push_back(res/split);
				if (res < 0) 
					outp.push_back(-(abs(res)%split));
				else
					outp.push_back(res%split);
		//		if ((zc==2) && (m_iaAtomSort[zi] == 92))
		//			mprintf("push %d --> %d %d\n",res,res/split,res%split);
//				mprintf("    @%3d: Pred %10.6f, Val %10.6f --> Res %6d / %6d\n",zi,tpred[zi],atoms->m_oaAtoms[zi]->m_fRelCoord[zc],res/split,res%split);
			} else {
				outp.push_back(res);
		//		if ((zc==2) && (m_iaAtomSort[zi] == 92))
		//			mprintf("push %d\n",res);
//				mprintf("    @%3d: Pred %10.6f, Val %10.6f --> Res %6d\n",zi,tpred[zi],atoms->m_oaAtoms[zi]->m_fRelCoord[zc],res);
			}
		}
	}

	if (verbose)
		mprintf("*** AtomsToIntegerArray finished, %lu symbols written ***\n\n",outp.size()-i);
}


void CCCEngine::IntegerArrayToAtoms(const std::vector<int> &inp, int order, int asigni, int esplit, bool verbose, bool second) {

	int z, z2, i, zc, zi, ii;
	std::vector<double> tpred;
	double compred;
	long ti;
	int split;
	double tf;
	CAtomSet *ofr, *ofr0;

	ii = 0;

	if (verbose) {
		mprintf("\n*** IntegerArrayToAtoms ***\n");
		i = ii;
	}

	split = 1<<esplit;

	if (verbose)
		mprintf("Order=%d, asigni=%d, esplit=%d, split=%d\n",order,asigni,esplit,split);

	if (second)
		ofr0 = GetOutput2AtomFrame(0);
	else
		ofr0 = GetOutputAtomFrame(0);

	tpred.resize(ofr0->m_oaAtoms.size());

	for (zc=0;zc<3;zc++) {

		if (verbose)
			mprintf("*** Now coordinate %d/3...\n",zc+1);
	
		if (verbose)
			mprintf("Polynomial extrapolation...\n");

		for (z=0;z<(int)ofr0->m_oaAtoms.size();z++)
			tpred[z] = 0;
		compred = 0;

		for (z=1;z<order+1;z++) {
			if (second)
				ofr = GetOutput2AtomFrame(z);
			else
				ofr = GetOutputAtomFrame(z);
			if (ofr == NULL) {
				eprintf("CCCEngine::IntegerArrayToAtoms(): Error: Frame is of order %d, step history is too short (%d).\n",order,z);
				abort();
			}
			tf = 1.0;
			for (z2=1;z2<=order;z2++) { 
				if (z2 == z)
					continue;
				tf *= z2 / ((double)z2 - z);
			}
	//		if (zc == 2)
	//			mprintf("@%d: %f\n",z,tf);
			for (z2=0;z2<(int)ofr0->m_oaAtoms.size();z2++)
				tpred[z2] += tf * ofr->m_oaAtoms[m_iaAtomSort[z2]]->m_fRelCoord[zc];
	//		if (zc == 2)
	//			mprintf("   + %.2f * %f\n",tf,ofr->m_oaAtoms[0]->m_fRelCoord[zc]);
			compred += tf * ofr->m_faCOM[zc];
		}
	//	if (zc == 2)
	//		mprintf("   = %f\n",tpred[0]);

		if (second)
			ofr = GetOutput2AtomFrame(0);
		else
			ofr = GetOutputAtomFrame(0);

		if (verbose)
			mprintf("Decoding symbols...\n");

		// COM component
		if (floor(compred*pow10(asigni)+0.5+AR_EPS) != floor(compred*pow10(asigni)+0.5-AR_EPS)) {
			ti = (int)floor(compred*pow10(asigni));
//			mprintf("2A) Sanitized rounding of %f.\n",compred*pow10(asigni));
		} else
			ti = (int)floor(compred*pow10(asigni)+0.5);
		if (inp[ii] == C_SPLIT) {
			ii++;
			ofr0->m_iaCOM[zc] = ti + inp[ii++] * split;
			ofr0->m_iaCOM[zc] += inp[ii++];
		} else
			ofr0->m_iaCOM[zc] = ti + inp[ii++];
		ofr0->m_faCOM[zc] = FixedToFloat(ofr0->m_iaCOM[zc],asigni);
//		if ((order > 2) && (zc == 2))
//			mprintf("COM = %f\n",ofr0->m_faCOM[zc]);

//		if (zc == 2)
//			mprintf("COM[%d]: %.6f (%ld).\n",zc,ofr0->m_faCOM[zc],ofr0->m_iaCOM[zc]);

		for (zi=0;zi<(int)ofr0->m_oaAtoms.size();zi++) {

			if (floor(tpred[zi]*pow10(asigni)+0.5+AR_EPS) != floor(tpred[zi]*pow10(asigni)+0.5-AR_EPS)) {
				ti = (int)floor(tpred[zi]*pow10(asigni));
//				mprintf("2B) %3d Sanitized rounding of %f.\n",zi,tpred[zi]*pow10(asigni));
			} else
				ti = (int)floor(tpred[zi]*pow10(asigni)+0.5);

			if (inp[ii] == C_SPLIT) {
				ii++;
				ofr0->m_oaAtoms[m_iaAtomSort[zi]]->m_iRelCoord[zc] = ti + inp[ii++] * split;
				ofr0->m_oaAtoms[m_iaAtomSort[zi]]->m_iRelCoord[zc] += inp[ii++];
	//			if ((zc==2) && (m_iaAtomSort[zi] == 92))
	//				mprintf("pop %d %d --> %d\n",inp[ii-2],inp[ii-1],inp[ii-2]*split+inp[ii-1]);
			} else {
				ofr0->m_oaAtoms[m_iaAtomSort[zi]]->m_iRelCoord[zc] = ti + inp[ii++];
	//			if ((zc==2) && (m_iaAtomSort[zi] == 92))
	//				mprintf("pop %d\n",inp[ii-1]);
			}

			ofr0->m_oaAtoms[m_iaAtomSort[zi]]->m_iCoord[zc] = ofr0->m_oaAtoms[m_iaAtomSort[zi]]->m_iRelCoord[zc] + ofr0->m_iaCOM[zc];

			ofr0->m_oaAtoms[m_iaAtomSort[zi]]->m_fRelCoord[zc] = FixedToFloat(ofr0->m_oaAtoms[m_iaAtomSort[zi]]->m_iRelCoord[zc],asigni);
			ofr0->m_oaAtoms[m_iaAtomSort[zi]]->m_fCoord[zc] = FixedToFloat(ofr0->m_oaAtoms[m_iaAtomSort[zi]]->m_iCoord[zc],asigni);

	//		if ((zc==2) && (m_iaAtomSort[zi] == 92)) {
	//			mprintf("Atom %d[%d]: fa=%.6f, ia=%ld, fr=%.6f, ir=%ld.\n",m_iaAtomSort[zi],zc,ofr0->m_oaAtoms[m_iaAtomSort[zi]]->m_fCoord[zc],ofr0->m_oaAtoms[m_iaAtomSort[zi]]->m_iCoord[zc],ofr0->m_oaAtoms[m_iaAtomSort[zi]]->m_fRelCoord[zc],ofr0->m_oaAtoms[m_iaAtomSort[zi]]->m_iRelCoord[zc]);
	//			mprintf("dec %2d: pred %d, mantis %d, relcoord %f, res %d\n",m_iaAtomSort[zi],ti,ofr0->m_oaAtoms[m_iaAtomSort[zi]]->m_iRelCoord[zc],ofr0->m_oaAtoms[m_iaAtomSort[zi]]->m_fRelCoord[zc],inp[ii-1]);
	//		}
		}
	}

	if (verbose)
		mprintf("*** IntegerArrayToAtoms finished, %d symbols read ***\n\n",ii-i);
}



void CCCEngine::CompressString(const char *s, CBitSet *bs, bool verbose) {

	CIntegerEngine ie;
	std::vector<int> ia;
	int z, mi, ma, bits, co;
	CBitSet bs2;
	unsigned long tpos;

	tpos = bs->GetLength();

	if (verbose)
		mprintf(">>> CompressString >>>\n");

	for (z=0;z<(int)strlen(s);z++)
		ia.push_back(s[z]);

	mi = 255;
	ma = 0;
	for (z=0;z<(int)ia.size();z++) {
		if (ia[z] > ma)
			ma = ia[z];
		if (ia[z] < mi)
			mi = ia[z];
	}
	if (ia.size() == 0) {
		mi = 0;
		ma = 0;
	}
	bits = (int)ceil(mylog2(ma-mi+1));
	co = (int)ceil((bits*ia.size()+4.0)/8.0)+2;

	if (verbose)
		mprintf("CompressString: Trying conventional storage, length %lu (%lu), range %d - %d, %d bits.\n",strlen(s),ia.size(),mi,ma,bits);

	PushStatistics();

	if (ia.size() > 1)
		ie.CompressSingle(ia,&bs2,true,true,true,50,1,false,true,false,verbose);
//	ie.Compress(ia,&bs2,false,false,true,50,1,false,true,verbose);

	if (verbose)
		mprintf("CompressString: Source %lu Bytes, IntegerEngine %d Bytes, Conventional %d Bytes.\n",ia.size(),bs2.GetByteLength(),co);

	if ((ia.size() <= 1) || ((co <= bs2.GetByteLength()) && (ia.size() < 256))) {
		if (verbose)
			mprintf("--> Using conventional storage.\n");
		bs->WriteBit(0);
		bs->WriteBits((unsigned long)strlen(s),8);
//		bs->WriteProgressiveUnsignedChar(strlen(s));
		bs->WriteBits(mi,8);
//		bs->WriteProgressiveUnsignedChar(mi);
		bs->WriteBits(bits,4);
//		bs->WriteProgressiveUnsignedChar(bits);
		for (z=0;z<(int)ia.size();z++)
			bs->WriteBits(ia[z]-mi,bits);
	} else {
		if (verbose)
			mprintf("--> Using IntegerEngine storage.\n");
		bs->WriteBit(1);
		bs->WriteBits(&bs2);
	}

	PopStatistics();
	gc_oStatistics.m_lOverhead += bs->GetLength() - tpos;

	if (verbose)
		mprintf("<<< CompressString <<<\n");
}


void CCCEngine::DecompressString(CBitSet *bs, char **s, bool verbose) {

	CIntegerEngine ie;
	std::vector<int> ia;
	int z, mi, c, bits;

	if (verbose)
		mprintf(">>> DecompressString >>>\n");

	if (!bs->ReadBit()) {

		if (verbose)
			mprintf("Classical storage.\n");

		c = bs->ReadBitsInteger(8);
//		c = bs->ReadProgressiveUnsignedChar();
		mi = bs->ReadBitsInteger(8);
//		mi = bs->ReadProgressiveUnsignedChar();
		bits = bs->ReadBitsInteger(4);
//		bits = bs->ReadProgressiveUnsignedChar();

		for (z=0;z<c;z++)
			ia.push_back(mi+bs->ReadBitsInteger(bits));

	} else {

		if (verbose)
			mprintf("IntegerEngine was used.\n");

		ie.DecompressSingle(bs,ia,verbose);
	}

	*s = new char[ia.size()+1];

	for (z=0;z<(int)ia.size();z++)
		(*s)[z] = ia[z];

	(*s)[z] = 0;

	if (verbose)
		mprintf("<<< DecompressString <<<\n");
}


bool CCCEngine::CompressCubeFrame(CBitSet *bs, int corder, int eps, int csigni, int csplit, int cblock, int ctables, bool opttables, bool hilbert, bool coderun, bool bw, bool mtf, bool preopt, bool atominfo, int maxchunk, bool verbose) {

	(void)atominfo;

	CIntegerEngine ie;
	std::vector<int> ia;
	CCubeFrame *cfr;
	unsigned long tpos;

//	std::vector<int> ia2;
//	int z, mi, ma;

	cfr = GetInputCubeFrame(0);

	if (hilbert && ((int)m_iaHilbertIdx.size() != cfr->m_iResXYZ))
		BuildHilbertIdx(cfr->m_iRes[0],cfr->m_iRes[1],cfr->m_iRes[2],verbose);

	CubeToIntegerArray(ia,corder,eps,csigni,csplit,hilbert,verbose);

//	ia.resize(500000);

	tpos = bs->GetLength();

	bs->WriteBits(corder,4);
	bs->WriteBits(csigni,4);
	bs->WriteBits(eps,6);
	bs->WriteBits(csplit,6);
	bs->WriteBit(hilbert?1:0);

	ExportCubeHeader(bs,corder,verbose);

	gc_oStatistics.m_lOverhead += bs->GetLength() - tpos;


/*	CBitSet bs2;

	mi = 2e9;
	ma = -2e9;
	for (z=0;z<(int)ia.size();z++) {
		if (ia[z] > ma)
			ma = ia[z];
		if (ia[z] < mi)
			mi = ia[z];
	}
	mprintf("@@@ mi=%d, ma=%d @@@\n",mi,ma);*/

	ie.Compress(ia,bs,bw,mtf,coderun,cblock,ctables,opttables,false,preopt,maxchunk,verbose);

/*	mi = 2e9;
	ma = -2e9;
	for (z=0;z<(int)ia.size();z++) {
		if (ia[z] > ma)
			ma = ia[z];
		if (ia[z] < mi)
			mi = ia[z];
	}
	mprintf("@@@ mi=%d, ma=%d @@@\n",mi,ma);

	ie.Decompress(&bs2,ia2,true);

	mi = 2e9;
	ma = -2e9;
	for (z=0;z<(int)ia2.size();z++) {
		if (ia2[z] > ma)
			ma = ia2[z];
		if (ia2[z] < mi)
			mi = ia2[z];
	}
	mprintf("@@@ mi=%d, ma=%d @@@\n",mi,ma);

	mprintf("### Check ###\n");

	for (z=0;z<(int)ia.size();z++)
		if (ia[z] != ia2[z])
			mprintf("Error: %6d: %d vs %d\n",z,ia[z],ia2[z]);

	mprintf("### Check done ###\n");*/

	return true;
}


bool CCCEngine::DecompressCubeFrame(CBitSet *bs, bool verbose, bool second) {

	std::vector<int> ia;
	CIntegerEngine ie;
	int tco, tcsigni, teps, tcsplit;
	CCubeFrame *cfr;
	bool hilbert;

	tco = bs->ReadBitsInteger(4);
	tcsigni = bs->ReadBitsInteger(4);
	teps = bs->ReadBitsInteger(6);
	tcsplit = bs->ReadBitsInteger(6);
	hilbert = bs->ReadBit();

	if (second)
		cfr = GetOutput2CubeFrame(0);
	else
		cfr = GetOutputCubeFrame(0);

	cfr->m_iEps = teps;
	cfr->m_iSigni = tcsigni;

	ImportCubeHeader(bs,verbose,second);

	if (hilbert && ((int)m_iaHilbertIdx.size() != cfr->m_iResXYZ))
		BuildHilbertIdx(cfr->m_iRes[0],cfr->m_iRes[1],cfr->m_iRes[2],verbose);

	ie.Decompress(bs,ia,verbose);

	IntegerArrayToCube(ia,tco,teps,tcsigni,tcsplit,hilbert,verbose,second);

	return true;
}


bool CCCEngine::CompressAtomFrame(CBitSet *bs, int order, int signi, int split, int block, int tables, bool opttables, bool coderun, bool bw, bool mtf, bool preopt, bool atominfo, bool comment, int maxchunk, bool verbose) {

	CIntegerEngine ie;
	std::vector<int> ia;
	std::string ts;
	int bits, z;
	CAtomSet *as;
	unsigned long tpos;

	as = GetInputAtomFrame(0);

	tpos = bs->GetLength();
	PushStatistics();

	if (atominfo) {

		bs->WriteBit(1); // Atom Type information present
		bits = (int)ceil(mylog2((double)as->m_oaAtoms.size()+1));
		bs->WriteBits(bits,5);
		bs->WriteBits((unsigned long)as->m_oaAtoms.size(),bits);

		if (as->m_bOrd) {
			bs->WriteBit(1); // Ord numbers
			for (z=0;z<(int)as->m_oaAtoms.size();z++)
				bs->WriteBits(as->m_oaAtoms[z]->m_iOrd,8);
		} else
			bs->WriteBit(0); // No Ord numbers
		
		if (as->m_bLabels) {
			bs->WriteBit(1); // Atom label strings
			for (z=0;z<(int)as->m_oaAtoms.size();z++) {
				ts += as->m_oaAtoms[z]->m_sLabel;
				if (z+1 < (int)as->m_oaAtoms.size())
					ts += " ";
			}
			CompressString(ts.c_str(),bs,verbose);
		} else
			bs->WriteBit(0); // No Atom label strings

		BuildAtomSort(as);

	} else
		bs->WriteBit(0); // No atom Type information present

	if ((as->m_sComment != NULL) && comment) {
		bs->WriteBit(1);
		CompressString(as->m_sComment,bs,verbose);
	} else
		bs->WriteBit(0);

	bs->WriteBits(order,4);
	bs->WriteBits(signi,4);
	bs->WriteBits(split,6);

	AtomsToIntegerArray(ia,order,signi,split,verbose);

	PopStatistics();
	gc_oStatistics.m_lOverhead += bs->GetLength() - tpos;

	ie.Compress(ia,bs,bw,mtf,coderun,block,tables,opttables,false,preopt,maxchunk,verbose);

	return true;
}


bool CCCEngine::DecompressAtomFrame(CBitSet *bs, bool verbose, bool second) {

	std::vector<int> ia;
	CIntegerEngine ie;
	int z, bits, ti;
	int tao, tasigni, tasplit;
	CCCAtom *at;
	CAtomSet *as, *as2;
	char *tch, *p, *q;

	if (second)
		as = GetOutput2AtomFrame(0);
	else
		as = GetOutputAtomFrame(0);

	if (bs->ReadBit()) {

		if (verbose)
			mprintf("Reading atom type information...\n");
		bits = bs->ReadBitsInteger(5);
		ti = bs->ReadBitsInteger(bits);
		if (verbose)
			mprintf("Will read %d atoms.\n",ti);
		for (z=0;z<ti;z++) {
			at = new CCCAtom();
			as->m_oaAtoms.push_back(at);
		}

		if (bs->ReadBit()) { // Read Ord Numbers
			as->m_bOrd = true;
			for (z=0;z<ti;z++)
				as->m_oaAtoms[z]->m_iOrd = bs->ReadBitsInteger(8);
		}

		if (bs->ReadBit()) { // Read Atom label strings
			as->m_bLabels = true;
			DecompressString(bs,&tch,verbose);
			if (verbose)
				mprintf("Decompressed string to %lu characters.\n",strlen(tch));
			p = tch;
			for (z=0;z<ti;z++) {
				q = strchr(p,' ');
				if (q != NULL)
					*q = 0;
				as->m_oaAtoms[z]->m_sLabel = p;
				if (!as->m_bOrd)
					as->m_oaAtoms[z]->m_iOrd = GetAtomOrd(p);
				p = q+1;
			}
			delete[] tch;
		}

		BuildAtomSort(as);

	} else {

		if (verbose)
			mprintf("Atom type information already read before.\n");
		if (second)
			as2 = GetOutput2AtomFrame(1);
		else
			as2 = GetOutputAtomFrame(1);
		if (as2 == NULL) {
			mprintf("CEngine::DecompressAtomFrame(): Error: First frame does not contain atom type information.\n");
			return false;
		}
		for (z=0;z<(int)as2->m_oaAtoms.size();z++) {
			at = new CCCAtom();
			at->m_iOrd = as2->m_oaAtoms[z]->m_iOrd;
			at->m_sLabel = as2->m_oaAtoms[z]->m_sLabel;
			as->m_oaAtoms.push_back(at);
		}
		as->m_bLabels = as2->m_bLabels;
		as->m_bOrd = as2->m_bOrd;
	}

	if (bs->ReadBit())
		DecompressString(bs,&as->m_sComment,verbose);
	else
		as->m_sComment = NULL;

	tao = bs->ReadBitsInteger(4);
	tasigni = bs->ReadBitsInteger(4);
	tasplit = bs->ReadBitsInteger(6);

	as->m_iSigni = tasigni;

	ie.Decompress(bs,ia,verbose);

	IntegerArrayToAtoms(ia,tao,tasigni,tasplit,verbose,second);

	return true;
}


bool CCCEngine::CompressFile(const char *inp, const char *outp, int block, int tables, bool opttables, bool coderun, bool bw, bool mtf, bool preopt, int maxchunk, bool verbose) {

	FILE *a/*, *b*/;
	unsigned char buf[4096];
	std::vector<int> ia;
	CBitSet bs;
	CIntegerEngine ie;
	CBarbecubeFile bf;
	int i, z;


	if (verbose)
		mprintf(">>> CEngine::CompressFile >>>\n");

	gc_oStatistics.m_lSize = 0;
	gc_oStatistics.m_lOverhead = 0;
	gc_oStatistics.m_lAlphabet = 0;
	gc_oStatistics.m_lHuffmanTables = 0;
	gc_oStatistics.m_lTableSwitch = 0;
	gc_oStatistics.m_lHuffmanData = 0;

	a = fopen(inp,"rb");
	if (a == NULL) {
		mprintf("Error: Could not open file \"%s\" for reading.\n",inp);
		return false;
	}

	if (outp != NULL) {
/*		b = fopen(outp,"wb");
		if (b == NULL) {
			mprintf("Error: Could not open file \"%s\" for writing.\n",outp);
			return false;
		}*/
		if (!bf.OpenWriteReplace(outp)) {
			mprintf("Error: Could not open file \"%s\" for writing.\n",outp);
			fclose(a);
			return false;
		}
	} /*else
		b = NULL;*/

	while (!feof(a)) {
		i = (int)fread(buf,1,4096,a);
		for (z=0;z<i;z++)
			ia.push_back(buf[z]);
		if (i < 4096)
			break;
	}

	mprintf("%lu Bytes read from input file.\n",ia.size());

	if (!ie.Compress(ia,&bs,bw,mtf,coderun,block,tables,opttables,true,preopt,maxchunk,verbose)) {
		mprintf("IntegerEngine returned an error.\n");
		return false;
	}

	fclose(a);

	if (outp != NULL) {
		bf.CreateShortFrame(BARBECUBE_FRAMETYPE_COMPFILE,0,0);
		bf.PushPayload(bs.m_iaData);
		bf.FinalizeFrame();
		bf.Close();
		//bs.ExportToFile(b);
		//fclose(b);
	}

	mprintf("Compressed to %d Bytes (%.3f:1, %7.3f%%, %.3f Bits/Byte).\n\n",bs.GetByteLength(),(double)ia.size()/bs.GetByteLength(),(double)bs.GetByteLength()/ia.size()*100.0,(double)bs.GetLength()/ia.size());

	mprintf("%10.3f MiB (%12s Bytes) overhead.\n",double(gc_oStatistics.m_lOverhead)/1024.0/1024.0/8.0,(gc_oStatistics.m_lOverhead>>3).string());
	mprintf("%10.3f MiB (%12s Bytes) alphabet data.\n",double(gc_oStatistics.m_lAlphabet)/1024.0/1024.0/8.0,(gc_oStatistics.m_lAlphabet>>3).string());
	mprintf("%10.3f MiB (%12s Bytes) Huffman tables.\n",double(gc_oStatistics.m_lHuffmanTables)/1024.0/1024.0/8.0,(gc_oStatistics.m_lHuffmanTables>>3).string());
	mprintf("%10.3f MiB (%12s Bytes) table switching.\n",double(gc_oStatistics.m_lTableSwitch)/1024.0/1024.0/8.0,(gc_oStatistics.m_lTableSwitch>>3).string());
	mprintf("%10.3f MiB (%12s Bytes) payload data.\n",double(gc_oStatistics.m_lHuffmanData)/1024.0/1024.0/8.0,(gc_oStatistics.m_lHuffmanData>>3).string());
	mprintf("%10.3f MiB (%12s Bytes) in total.\n\n",double(gc_oStatistics.m_lOverhead+gc_oStatistics.m_lAlphabet+gc_oStatistics.m_lHuffmanTables+gc_oStatistics.m_lTableSwitch+gc_oStatistics.m_lHuffmanData)/1024.0/1024.0/8.0,((gc_oStatistics.m_lOverhead+gc_oStatistics.m_lAlphabet+gc_oStatistics.m_lHuffmanTables+gc_oStatistics.m_lTableSwitch+gc_oStatistics.m_lHuffmanData)>>3).string());

	if (verbose)
		mprintf("<<< CEngine::CompressFile <<<\n");

	return true;
}


bool CCCEngine::DecompressFile(const char *inp, const char *outp, const char *ref, bool verbose) {

	(void)ref;

	FILE /**a,*/ *b;
	unsigned char buf[4096];
	std::vector<int> ia;
	CBitSet bs;
	CIntegerEngine ie;
	int /*i,*/ z, z2;
	CBarbecubeFile bf;

	if (verbose)
		mprintf(">>> CEngine::DecompressFile >>>\n");

/*	a = fopen(inp,"rb");
	if (a == NULL) {
		mprintf("Error: Could not open file \"%s\" for reading.\n",inp);
		return false;
	}*/

	if (!bf.OpenRead(inp)) {
		mprintf("Error: Could not open file \"%s\" for reading.\n",inp);
		return false;
	}

	b = fopen(outp,"wb");
	if (b == NULL) {
		mprintf("Error: Could not open file \"%s\" for writing.\n",outp);
		return false;
	}

/*	while (!feof(a)) {
		i = fread(buf,1,4096,a);
		for (z=0;z<i;z++)
			bs.m_iaData.push_back(buf[z]);
		if (i < 4096)
			break;
	}*/

	bf.ReadFrame();

	if ((bf.GetFrameType() != BARBECUBE_FRAMETYPE_COMPFILE) || (bf.GetFrameTypeVersion() != 0)) {
		mprintf("Error: Unexpected frame type (expected %d v0, found %d v%d).\n",BARBECUBE_FRAMETYPE_COMPFILE,bf.GetFrameType(),bf.GetFrameTypeVersion());
		fclose(b);
		return false;
	}

	bs.m_iaData.assign(bf.GetFramePayload()->begin(),bf.GetFramePayload()->end());

	bf.Close();

	mprintf("%lu bytes of payload read from input file.\n",bs.m_iaData.size());

	if (!ie.Decompress(&bs,ia,verbose)) {
		mprintf("IntegerEngine returned an error.\n");
		fclose(b);
		return false;
	}

	if (verbose)
		mprintf("Decompressed to %lu Bytes.\n",ia.size());

	for (z=0;z<(int)ia.size()/4096;z++) {
		for (z2=0;z2<4096;z2++)
			buf[z2] = ia[z*4096+z2];
		fwrite(buf,4096,1,b);
	}

	if ((ia.size()%4096) != 0) {
		for (z2=0;z2<(int)ia.size()%4096;z2++)
			buf[z2] = ia[z*4096+z2];
		fwrite(buf,ia.size()%4096,1,b);
	}

	fclose(b);
//	fclose(a);

	if (verbose)
		mprintf("<<< CEngine::DecompressFile <<<\n");

	return true;
}


bool CCCEngine::CompressCube(
		const char *inp,
		const char *outp,
		const char *ref,
		int start,
		int steps,
		int stride,
		int corder,
		bool optcorder,
		int aorder,
		bool optaorder,
		int eps,
		int csigni,
		int asigni,
		int csplit,
		int asplit,
		int cblock,
		int ablock,
		int ctables,
		bool optctables,
		bool cpreopt,
		int cmaxchunk,
		int atables,
		bool optatables,
		bool hilbert,
		bool ccoderun,
		bool cbw,
		bool cmtf,
		bool acoderun,
		bool abw,
		bool amtf,
		bool apreopt,
		int amaxchunk,
		bool comment,
		bool compare,
		bool dummyread,
		bool verbose
		) {

	CCubeFrame *cfr, *cfr2;
	int z, z2, i, k, co, ao, lcsize, lasize, morder;
	double tb, tb2, tf, tf2;
	std::vector<int> ia, ia2;
	CBitSet bsat, bscu, bshe, bstemp;
	FILE *a, *fref;
	bool converged, err, to;
	unsigned long t0;
	CBarbecubeFile bf;


	t0 = (unsigned long)time(NULL);

	gc_oStatistics.m_lSize = 0;
	gc_oStatistics.m_lOverhead = 0;
	gc_oStatistics.m_lAlphabet = 0;
	gc_oStatistics.m_lHuffmanTables = 0;
	gc_oStatistics.m_lTableSwitch = 0;
	gc_oStatistics.m_lHuffmanData = 0;

	mprintf("Opening cube file \"%s\" ...\n",inp);
	a = fopen(inp,"rb");
	if (a == NULL) {
		mprintf("Error: Could not open file for reading.\n");
		return false;
	}

	if (!dummyread) {
		if (outp != NULL) {
			mprintf("Opening compressed file \"%s\" ...\n",outp);
			if (!bf.OpenWriteAppend(outp)) {
				mprintf("Error: Could not open file for writing.\n");
				fclose(a);
				return false;
			}
		}

		if (ref != NULL) {
			mprintf("Opening reference cube file \"%s\" ...\n",ref);
			fref = fopen(ref,"wb");
			if (fref == NULL) {
				mprintf("Error: Could not open file for writing.\n");
				fclose(a);
				return false;
			}
		} else
			fref = NULL;
	} else {
		ref = NULL;
		fref = NULL;
		outp = NULL;
		mprintf("Dummy Read Mode (no compression / output is performed).\n");
	}

	m_oaInputCubeBuf.resize(corder+1);
	m_oaOutputCubeBuf.resize(corder+1);
	for (z=0;z<corder+1;z++) {
		m_oaInputCubeBuf[z] = NULL;
		m_oaOutputCubeBuf[z] = NULL;
	}
	m_iInputCubeBufPos = 0;
	m_iOutputCubeBufPos = 0;

	m_oaInputAtomBuf.resize(aorder+1);
	m_oaOutputAtomBuf.resize(aorder+1);
	for (z=0;z<aorder+1;z++) {
		m_oaInputAtomBuf[z] = NULL;
		m_oaOutputAtomBuf[z] = NULL;
	}
	m_iInputAtomBufPos = 0;
	m_iOutputAtomBufPos = 0;

	mprintf("Starting process...\n");

	if (start != 0) {
		mprintf("Fast-forwarding to step %d: ",start+1);
		cfr = new CCubeFrame();
		for (z=0;z<start;z++) {
			if (!cfr->SkipFrame(a,verbose))
				return false;
			mprintf(".");
		}
		mprintf(" Done.\n");
		delete cfr;
	}

	morder = corder;
	if (aorder > morder)
		morder = aorder;

	ResetStatistics();

	converged = false;
	err = false;
	i = 0;
	k = 0;
	tb = 0;
	tb2 = 0;
	tf = 0;
	tf2 = 0;
	to = false;
	if (optcorder)
		lcsize = 1000000000;
	else
		lcsize = -1;
	if (optaorder)
		lasize = 1000000000;
	else
		lasize = -1;
	while (true) {

		if (verbose)
			mprintf("\n\n");

		if (i != 0)
			mprintf("    Step %6d ... (avg %5.1f s/step)\n",i+1,((double)(time(NULL)-t0))/i);
		else
			mprintf("    Step %6d ...\n",i+1);

		cfr = new CCubeFrame();

		if (!cfr->ReadFrame(a,eps,csigni,asigni,verbose))
			break;

		if (ref != NULL)
			cfr->WriteFrame(fref,verbose);

		for (z=0;z<stride-1;z++) {
			if (verbose)
				mprintf("Skipping frame...\n");
			if (!cfr->SkipFrame(a,verbose))
				break;
		}

		if (dummyread) { // Just benchmark the CUBE reading speed
			delete cfr;
			goto _dummy;
		}

		PushInputCubeFrame(cfr);

		PushInputAtomFrame(cfr->m_pAtoms);

_again:
		if (i < corder)
			co = i;
		else if (k < corder)
			co = k;
		else
			co = corder;

		if (i < aorder)
			ao = i;
		else if (k < aorder)
			ao = k;
		else
			ao = aorder;

_aagain:
		bsat.Clear();
		BeginStatistics();
		PushStatistics();

		CompressAtomFrame(&bsat,ao,asigni,asplit,ablock,atables,optatables,acoderun,abw,amtf,apreopt&&(i>=2),(i==0),comment,amaxchunk,verbose);

		mprintf("      Atoms: Order %d, output size %9.3f KiB.\n",ao,bsat.GetByteLength()/1024.0);

/*		if (lasize >= 0) {
			if ((ao != 0) && ((double)bsat.GetByteLength() >= (double)lasize*0.97)) {
				mprintf("      Size did not decrease any further with order %d. Limiting atom order to %d.\n",ao,ao-1);
				ao--;
				aorder = ao;
				lasize = -1;
				goto _aagain;
			} else if (ao == aorder)
				lasize = -1;
			else
				lasize = bsat.GetByteLength();
		}*/

		if (lasize >= 0) {
			if (to) {
				to = false;
				if (bsat.GetByteLength() <= bstemp.GetByteLength()) {
					mprintf("      Size is indeed smaller with lower order. Limiting atom order to %d.\n",ao);
					PopDiffStatistics();
					aorder = ao;
					lasize = -1;
				} else {
					mprintf("      Size was smaller with higher order. Not limiting.\n");
					PopStatistics();
					bsat = CBitSet(bstemp);
					lasize = bsat.GetByteLength();
				}
			} else {
				if ((ao != 0) && ((double)bsat.GetByteLength() >= (double)lasize*0.97)) {
					mprintf("      Size did not decrease any further with order %d. Trying order %d...\n",ao,ao-1);
					bstemp = CBitSet(bsat);
					to = true;
					ao--;
					goto _aagain;
				} else if (ao == aorder)
					lasize = -1;
				else
					lasize = bsat.GetByteLength();
			}
		}

		PopIgnoreStatistics();
		EndStatistics(true,converged);

_cagain:
		bscu.Clear();
		BeginStatistics();
		PushStatistics();

		CompressCubeFrame(&bscu,co,eps,csigni,csplit,cblock,ctables,optctables,hilbert,ccoderun,cbw,cmtf,cpreopt&&(i>=2),(i==0),cmaxchunk,verbose);

		mprintf("      Cube:  Order %d, output size %9.3f KiB.\n",co,bscu.GetByteLength()/1024.0);

/*		if (lcsize >= 0) {
			if ((co != 0) && ((double)bscu.GetByteLength() >= (double)lcsize*0.97)) {
				mprintf("      Size did not decrease any further with order %d. Limiting cube order to %d.\n",co,co-1);
				co--;
				corder = co;
				lcsize = -1;
				goto _cagain;
			} else if (co == corder)
				lcsize = -1;
			else
				lcsize = bscu.GetByteLength();
		}*/

		if (lcsize >= 0) {
			if (to) {
				to = false;
				if (bscu.GetByteLength() <= bstemp.GetByteLength()) {
					mprintf("      Size is indeed smaller with lower order. Limiting cube order to %d.\n",co);
					PopDiffStatistics();
					corder = co;
					lcsize = -1;
				} else {
					mprintf("      Size was smaller with higher order. Not limiting.\n");
					PopStatistics();
					bscu = CBitSet(bstemp);
					lcsize = bscu.GetByteLength();
				}
			} else {
				if ((co != 0) && ((double)bscu.GetByteLength() >= (double)lcsize*0.97)) {
					mprintf("      Size did not decrease any further with order %d. Trying order %d...\n",co,co-1);
					bstemp = CBitSet(bscu);
					to = true;
					co--;
					goto _cagain;
				} else if (co == corder)
					lcsize = -1;
				else
					lcsize = bscu.GetByteLength();
			}
		}

		PopIgnoreStatistics();
		EndStatistics(false,converged);

		if (MAX(aorder,corder) < morder)
			morder = MAX(aorder,corder);

/*		std::vector<int> ia;
		CBitSet bs2;
		CIntegerEngine ie;
		ia.clear();
		for (z=0;z<(int)bscu.m_iaData.size();z++)
			ia.push_back(bscu.m_iaData[z]);
		bs2.Clear();
		ie.Compress(ia,&bs2,true,true,true,50,31,true,verbose);
		mprintf("      Further Size reduction %.3f KiB --> %.3f KiB.\n",bscu.GetByteLength()/1024.0,bs2.GetByteLength()/1024.0);
*/

		//crc = crcengine.ComputeCRC32_Begin(bsat.m_iaData);
		//crc = crcengine.ComputeCRC32_Continue(bscu.m_iaData,crc);
		//crc = crcengine.ComputeCRC32_Finish(crc);

		//bshe.WriteBits(crc,32);

		if (compare) {

			cfr2 = new CCubeFrame();
			PushOutputCubeFrame(cfr2);
			cfr2->m_pAtoms = new CAtomSet();
			PushOutputAtomFrame(cfr2->m_pAtoms);

			DecompressAtomFrame(&bsat,verbose,false);

			DecompressCubeFrame(&bscu,verbose,false);

			if (verbose)
				mprintf("Comparing input and output...\n");

			for (z=0;z<(int)cfr2->m_pAtoms->m_oaAtoms.size();z++)
				for (z2=0;z2<3;z2++)
					if (cfr->m_pAtoms->m_oaAtoms[z]->m_iCoord[z2] != cfr2->m_pAtoms->m_oaAtoms[z]->m_iCoord[z2]) {
						eprintf("        Error in atom coordinate %d[%d]: %.6f (%ld) != %.6f (%ld)\n",z,z2,cfr->m_pAtoms->m_oaAtoms[z]->m_fCoord[z2],cfr->m_pAtoms->m_oaAtoms[z]->m_iCoord[z2],cfr2->m_pAtoms->m_oaAtoms[z]->m_fCoord[z2],cfr2->m_pAtoms->m_oaAtoms[z]->m_iCoord[z2]);
						err = true;
					}

			for (z=0;z<cfr->m_iResXYZ;z++)
				if (!ExpMantisEqual(cfr->m_iaExpo[z],cfr->m_iaMantis[z],cfr2->m_iaExpo[z],cfr2->m_iaMantis[z])) {
					eprintf("        Error in volumetric data element %7d: %.10G vs %.10G\n",z,cfr->m_faBin[z],cfr2->m_faBin[z]);
					err = true;
				}

			if (err) {
				if (k != 0) {
					eprintf("Errors occured. Compressing frame again with order zero.\n");
					err = false;
					k = 0;
					goto _again;
				} else {
					eprintf("Errors occured despite of order zero. Aborting.\n");
					goto _end;
				}
			}

			if (verbose)
				mprintf("Done.\n");
		}

		bshe.Clear();

		bshe.WriteBits(bsat.GetByteLength(),32);
		bshe.WriteBits(bscu.GetByteLength(),32);

		tb += bshe.GetByteLength();
		tb += bsat.GetByteLength();
		tb += bscu.GetByteLength();
		tf++;

		gc_oStatistics.m_lOverhead += bshe.GetLength();

		if ((bsat.GetLength()%8) != 0)
			gc_oStatistics.m_lOverhead += 8 - (bsat.GetLength()%8);

		if ((bscu.GetLength()%8) != 0)
			gc_oStatistics.m_lOverhead += 8 - (bscu.GetLength()%8);

		gc_iOutputSize += bshe.GetByteLength() + bsat.GetByteLength() + bscu.GetByteLength();
		if (converged)
			gc_iCOutputSize += bshe.GetByteLength() + bsat.GetByteLength() + bscu.GetByteLength();

		if (i >= morder) {
			tb2 += bshe.GetByteLength();
			tb2 += bsat.GetByteLength();
			tb2 += bscu.GetByteLength();
			tf2++;
		}

		if (outp != NULL) {
			bf.CreateShortFrame((k==0)?BARBECUBE_FRAMETYPE_COMPCUBESTART:BARBECUBE_FRAMETYPE_COMPCUBE,0,i+1);
			bf.PushPayload(bshe.m_iaData);
			bf.PushPayload(bsat.m_iaData);
			bf.PushPayload(bscu.m_iaData);
			bf.FinalizeFrame();
		/*	bshe.ExportToFile(b);
			bsat.ExportToFile(b);
			bscu.ExportToFile(b);*/
		}

		if ((ao == aorder) && (co == corder))
			converged = true;

_dummy:

		i++;
		k++;
		if (i == steps)
			break;
	}
_end:

	fclose(a);

	if (outp != NULL) {
		bf.WriteIndexFrame(true);
		bf.Close();
	}

	//if (b != NULL)
	//	fclose(b);

	if (ref != NULL)
		fclose(fref);

	if (err) {
		mprintf("Errors occurred while compressing the cube file.\n");
		return false;
	} else {
		mprintf("Finished compressing the cube file.\n\n");
//		mprintf("%10.3f MiB (%12.0f Bytes) written in total.\n",tb/1024.0/1024.0,tb);
		mprintf("%10.3f MiB (%12s Bytes) overhead.\n",double(gc_oStatistics.m_lOverhead)/1024.0/1024.0/8.0,(gc_oStatistics.m_lOverhead>>3).string());
		mprintf("%10.3f MiB (%12s Bytes) alphabet data.\n",double(gc_oStatistics.m_lAlphabet)/1024.0/1024.0/8.0,(gc_oStatistics.m_lAlphabet>>3).string());
		mprintf("%10.3f MiB (%12s Bytes) Huffman tables.\n",double(gc_oStatistics.m_lHuffmanTables)/1024.0/1024.0/8.0,(gc_oStatistics.m_lHuffmanTables>>3).string());
		mprintf("%10.3f MiB (%12s Bytes) table switching.\n",double(gc_oStatistics.m_lTableSwitch)/1024.0/1024.0/8.0,(gc_oStatistics.m_lTableSwitch>>3).string());
		mprintf("%10.3f MiB (%12s Bytes) payload data.\n",double(gc_oStatistics.m_lHuffmanData)/1024.0/1024.0/8.0,(gc_oStatistics.m_lHuffmanData>>3).string());
		mprintf("%10.3f MiB (%12s Bytes) in total.\n\n",double(gc_oStatistics.m_lOverhead+gc_oStatistics.m_lAlphabet+gc_oStatistics.m_lHuffmanTables+gc_oStatistics.m_lTableSwitch+gc_oStatistics.m_lHuffmanData)/1024.0/1024.0/8.0,((gc_oStatistics.m_lOverhead+gc_oStatistics.m_lAlphabet+gc_oStatistics.m_lHuffmanTables+gc_oStatistics.m_lTableSwitch+gc_oStatistics.m_lHuffmanData)>>3).string());
		if (tf > 0) {
			mprintf("%9.3f KiB per frame on average.\n",tb/1024.0/tf);
			mprintf("%.3f bits per bin entry on average.\n\n",tb/tf/cfr->m_iResXYZ*8.0);
		}
		if (tf2 > 0) {
			mprintf("%9.3f KiB per frame on average (starting from step %d).\n",tb2/1024.0/tf2,morder+1);
			mprintf("%.3f bits per bin entry on average (starting from step %d).\n\n",(tb2/tf2)/cfr->m_iResXYZ*8.0,morder+1);
		}
	}

	return true;
}


bool CCCEngine::DecompressCube(const char *inp, const char *outp, const char *readref, int steps, int stride, bool verbose) {

	(void)stride;

	CCubeFrame *cfr, *cfr2;
	int z, i, al, cl;
	//unsigned int crc, crc2;
	CBitSet bsat, bscu, bshe;
	FILE /**a,*/ *b, *fref;
	//CCRC32 crcengine;
	CBarbecubeFile bf;

	mprintf("Opening archive file \"%s\" ...\n",inp);
	if (!bf.OpenRead(inp)) {
		mprintf("Error: Could not open file for reading.\n");
		return false;
	}
/*	a = fopen(inp,"rb");
	if (a == NULL) {
		mprintf("Error: Could not open file for reading.\n");
		return false;
	}*/

	if (outp != NULL) {
		mprintf("Opening output cube file \"%s\" ...\n",outp);
		b = fopen(outp,"wb");
		if (b == NULL) {
			mprintf("Error: Could not open file for writing.\n");
			return false;
		}
	} else {
		b = NULL;
		mprintf("No output file specified; no output will be written.\n");
	}

	fref = NULL;
	if (readref != NULL) {
		mprintf("Opening reference cube file \"%s\" ...\n",readref);
		fref = fopen(readref,"rb");
		if (fref == NULL) {
			mprintf("Error: Could not open file for reading.\n");
			return false;
		}
	}

	m_oaOutputCubeBuf.resize(7);
	for (z=0;z<7;z++)
		m_oaOutputCubeBuf[z] = NULL;
	m_iOutputCubeBufPos = 0;

	m_oaOutputAtomBuf.resize(7);
	for (z=0;z<7;z++)
		m_oaOutputAtomBuf[z] = NULL;
	m_iOutputAtomBufPos = 0;

	mprintf("Starting process...\n");

	i = 0;
	while (true) {

		if (verbose)
			mprintf("\n\n");

		if (!bf.ReadFrame())
			break;

		if ((bf.GetFrameType() == BARBECUBE_FRAMETYPE_IDX) || (bf.GetFrameType() == BARBECUBE_FRAMETYPE_COMPIDX))
			continue;

		if (((bf.GetFrameType() != BARBECUBE_FRAMETYPE_COMPCUBE) && (bf.GetFrameType() != BARBECUBE_FRAMETYPE_COMPCUBESTART)) || (bf.GetFrameTypeVersion() != 0)) {
			mprintf("Error: Unexpected frame type (expected %d v0 or %d v0, found %d v%d).\n",BARBECUBE_FRAMETYPE_COMPCUBE,BARBECUBE_FRAMETYPE_COMPCUBESTART,bf.GetFrameType(),bf.GetFrameTypeVersion());
			return false;
		}

		bshe.Clear();
		bshe.m_iaData.assign(bf.GetFramePayload()->begin(),bf.GetFramePayload()->begin()+8);

		//if (!bshe.ImportFromFile(a,12))
		//	break;

		mprintf("    Step %d ...\n",i+1);

		al = bshe.ReadBitsInteger(32);
		cl = bshe.ReadBitsInteger(32);
		//crc = bshe.ReadBitsInteger(32);

		if (verbose)
			mprintf("Atom block %d bytes, Cube block %d bytes.\n",al,cl);

		cfr = new CCubeFrame();
		PushOutputCubeFrame(cfr);

		cfr->m_pAtoms = new CAtomSet();
		PushOutputAtomFrame(cfr->m_pAtoms);

		bsat.Clear();
		bsat.m_iaData.assign(bf.GetFramePayload()->begin()+8,bf.GetFramePayload()->begin()+8+al);

		bscu.Clear();
		bscu.m_iaData.assign(bf.GetFramePayload()->begin()+8+al,bf.GetFramePayload()->begin()+8+al+cl);

		//if (!bsat.ImportFromFile(a,al))
		//	break;

		//if (!bscu.ImportFromFile(a,cl))
		//	break;

		/*crc2 = crcengine.ComputeCRC32_Begin(bsat.m_iaData);
		crc2 = crcengine.ComputeCRC32_Continue(bscu.m_iaData,crc2);
		crc2 = crcengine.ComputeCRC32_Finish(crc2);

		if (crc != crc2) {
			mprintf("CRC error while reading archive file. %08X != %08X\n",crc,crc2);
			break;
		} else if (verbose)
			mprintf("CRC check passed.\n");*/

		if (!DecompressAtomFrame(&bsat,verbose,false))
			return false;

		if (!DecompressCubeFrame(&bscu,verbose,false))
			return false;

		if (readref != NULL) {

			mprintf("      Comparing to reference...\n");

			cfr2 = new CCubeFrame();

			if (!cfr2->ReadFrame(fref,cfr->m_iEps,cfr->m_iSigni,cfr->m_pAtoms->m_iSigni,verbose))
				break;

			for (z=0;z<cfr->m_iResXYZ;z++)
				if (!ExpMantisEqual(cfr->m_iaExpo[z],cfr->m_iaMantis[z],cfr2->m_iaExpo[z],cfr2->m_iaMantis[z])) {
					mprintf("        Error %7d: %.10G vs %.10G\n",z,cfr->m_faBin[z],cfr2->m_faBin[z]);
					break;
				}

			delete cfr2;

			if (verbose)
				mprintf("Done.\n");
		}

		mprintf("      %.3f KiB compressed data unpacked.\n",(al+cl)/1024.0);

		if (outp != NULL)
			cfr->WriteFrame(b,verbose);

		if (verbose)
			mprintf("Done.\n");

		i++;
		if (i == steps)
			break;
	}

	//fclose(a);

	bf.Close();

	if (outp != NULL)
		fclose(b);

	if (readref != NULL)
		fclose(fref);

	mprintf("Finished decompressing the cube file.\n\n");

	return true;
}


bool CCCEngine::PrepareDecompressCube(bool verbose) {

	int z;

	if (verbose)
		mprintf(">>> CCCEngine::PrepareDecompressCube() >>>\n");

	for (z=0;z<(int)m_oaOutputCubeBuf.size();z++)
		if (m_oaOutputCubeBuf[z] != NULL)
			delete m_oaOutputCubeBuf[z];

	for (z=0;z<(int)m_oaOutputAtomBuf.size();z++)
		if (m_oaOutputAtomBuf[z] != NULL)
			delete m_oaOutputAtomBuf[z];

	m_oaOutputCubeBuf.resize(10);
	for (z=0;z<7;z++)
		m_oaOutputCubeBuf[z] = NULL;
	m_iOutputCubeBufPos = 0;

	m_oaOutputAtomBuf.resize(10);
	for (z=0;z<7;z++)
		m_oaOutputAtomBuf[z] = NULL;
	m_iOutputAtomBufPos = 0;

	if (verbose)
		mprintf("<<< CCCEngine::PrepareDecompressCube() <<<\n");

	return true;
}


bool CCCEngine::DecompressCubeStep(const std::vector<unsigned char> *inp, bool verbose, bool skipvol) {

	CCubeFrame *cfr;
	int al, cl;
	CBitSet bsat, bscu, bshe;

	if (verbose)
		mprintf(">>> CCCEngine::DecompressCubeStep() >>>\n");

	bshe.Clear();
	bshe.m_iaData.assign(inp->begin(),inp->begin()+8);

	al = bshe.ReadBitsInteger(32);
	cl = bshe.ReadBitsInteger(32);

	cfr = new CCubeFrame();
	PushOutputCubeFrame(cfr);

	cfr->m_pAtoms = new CAtomSet();
	PushOutputAtomFrame(cfr->m_pAtoms);

	bsat.Clear();
	bsat.m_iaData.assign(inp->begin()+8,inp->begin()+8+al);

	bscu.Clear();
	bscu.m_iaData.assign(inp->begin()+8+al,inp->begin()+8+al+cl);

	if (!DecompressAtomFrame(&bsat,verbose,false))
		return false;

	if (!skipvol)
		if (!DecompressCubeFrame(&bscu,verbose,false))
			return false;

	if (verbose)
		mprintf("<<< CCCEngine::DecompressCubeStep() <<<\n");

	return true;
}


bool CCCEngine::CompressXYZ(
		const char *inp,
		const char *outp,
		const char *ref,
		int start,
		int steps,
		int stride,
		int order,
		bool optorder,
		int signi,
		int split,
		int block,
		int tables,
		bool opttables,
		bool coderun,
		bool bw,
		bool mtf,
		bool preopt,
		bool comment,
		bool compare,
		int maxchunk,
		bool verbose
	) {

	CAtomSet *as, *as2;
	int z, z2, i, o, lasize, morder;
	//unsigned int crc;
	double tb, tb2, tf, tf2;
	std::vector<int> ia, ia2;
	CBitSet bsat, bstemp/*, bshe*/;
	FILE *a, /**b,*/ *fref;
	//CCRC32 crcengine;
	bool err, to;
	CBarbecubeFile bf;


	gc_oStatistics.m_lSize = 0;
	gc_oStatistics.m_lOverhead = 0;
	gc_oStatistics.m_lAlphabet = 0;
	gc_oStatistics.m_lHuffmanTables = 0;
	gc_oStatistics.m_lTableSwitch = 0;
	gc_oStatistics.m_lHuffmanData = 0;

	mprintf("Opening XYZ file \"%s\" ...\n",inp);
	a = fopen(inp,"rb");
	if (a == NULL) {
		mprintf("Error: Could not open file for reading.\n");
		return false;
	}

	if (outp != NULL) {
		mprintf("Opening compressed output file \"%s\" ...\n",outp);
		if (!bf.OpenWriteAppend(outp)) {
			mprintf("Error: Could not open file for writing.\n");
			fclose(a);
			return false;
		}
		/*b = fopen(outp,"wb");
		if (b == NULL) {
			mprintf("Error: Could not open file for writing.\n");
			return false;
		}*/
	}/* else
		b = NULL;*/

	if (ref != NULL) {
		mprintf("Opening reference XYZ file \"%s\" ...\n",ref);
		fref = fopen(ref,"wb");
		if (fref == NULL) {
			mprintf("Error: Could not open file for writing.\n");
			fclose(a);
			return false;
		}
	} else
		fref = NULL;

	m_oaInputAtomBuf.resize(order+1);
	m_oaOutputAtomBuf.resize(order+1);
	for (z=0;z<order+1;z++) {
		m_oaInputAtomBuf[z] = NULL;
		m_oaOutputAtomBuf[z] = NULL;
	}
	m_iInputAtomBufPos = 0;
	m_iOutputAtomBufPos = 0;

	mprintf("Starting process...\n");

	if (start != 0) {
		mprintf("Fast-forwarding to step %d ...\n",start+1);
		as = new CAtomSet();
		for (z=0;z<start;z++)
			if (!as->SkipXYZ(a))
				return false;
		delete as;
	}

	morder = order;

	i = 0;
	tb = 0;
	tb2 = 0;
	tf = 0;
	tf2 = 0;
	to = false;
	if (optorder)
		lasize = 1000000000;
	else
		lasize = -1;

	err = false;

//	FILE *c;
//	c = fopen("sizes.txt","wt");

	while (true) {

		if (verbose)
			mprintf("\n\n");

		mprintf("    Step %d ...\n",i+1);

		as = new CAtomSet();

		if (!as->ReadXYZ(a,signi,fref))
			break;

		for (z=0;z<stride-1;z++) {
			if (verbose)
				mprintf("Skipping frame...\n");
			if (!as->SkipXYZ(a))
				break;
		}

		PushInputAtomFrame(as);

		if (i < order)
			o = i;
		else
			o = order;

_aagain:
		bsat.Clear();
		PushStatistics();

		CompressAtomFrame(&bsat,o,signi,split,block,tables,opttables,coderun,bw,mtf,preopt&&(i>=2),(i==0),comment,maxchunk,verbose);

		mprintf("      Order %d, output size %9.3f KiB.\n",o,bsat.GetByteLength()/1024.0);

/*		if (lasize >= 0) {
			if ((o != 0) && ((double)bsat.GetByteLength() >= (double)lasize*0.97)) {
				mprintf("      Size did not decrease any further with order %d. Limiting order to %d.\n",o,o-1);
				o--;
				order = o;
				lasize = -1;
				goto _aagain;
			} else if (o == order)
				lasize = -1;
			else
				lasize = bsat.GetByteLength();
		}*/

		if (lasize >= 0) {
			if (to) {
				to = false;
				if (bsat.GetByteLength() <= bstemp.GetByteLength()) {
					mprintf("    Size is indeed smaller with lower order. Limiting atom order to %d.\n",o);
					PopDiffStatistics();
					order = o;
					lasize = -1;
				} else {
					mprintf("    Size was smaller with higher order. Not limiting.\n");
					PopStatistics();
					bsat = CBitSet(bstemp);
					lasize = bsat.GetByteLength();
				}
			} else {
				if ((o != 0) && ((double)bsat.GetByteLength() >= (double)lasize*0.97)) {
					mprintf("    Size did not decrease any further with order %d. Trying order %d...\n",o,o-1);
					bstemp = CBitSet(bsat);
					to = true;
					o--;
					goto _aagain;
				} else if (o == order)
					lasize = -1;
				else
					lasize = bsat.GetByteLength();
			}
		}

		PopIgnoreStatistics();

//		fprintf(c,"%d\n",bsat.GetByteLength());

/*		std::vector<int> ia;
		CBitSet bs2;
		CIntegerEngine ie;
		ia.clear();
		for (z=0;z<(int)bsat.m_iaData.size();z++)
			ia.push_back(bsat.m_iaData[z]);
		bs2.Clear();
		ie.Compress(ia,&bs2,true,true,true,50,15,true,verbose);
		mprintf("      Further Size reduction %.3f KiB --> %.3f KiB.\n",bsat.GetByteLength()/1024.0,bs2.GetByteLength()/1024.0);
*/
		//bshe.Clear();

		//bshe.WriteBits(bsat.GetByteLength(),32);

		//crc = crcengine.ComputeCRC32(bsat.m_iaData);

		//bshe.WriteBits(crc,32);

		//tb += bshe.GetByteLength();
		tb += bsat.GetByteLength();
		tf++;

		if ((bsat.GetLength()%8) != 0)
			gc_oStatistics.m_lOverhead += 8 - (bsat.GetLength()%8);

		if (i >= morder) {
			//tb2 += bshe.GetByteLength();
			tb2 += bsat.GetByteLength();
			tf2++;
		}

		if (compare) {

			as2 = new CAtomSet();
			PushOutputAtomFrame(as2);

			DecompressAtomFrame(&bsat,verbose,false);

			if (verbose)
				mprintf("Comparing input and output...\n");

			for (z=0;z<(int)as2->m_oaAtoms.size();z++)
				for (z2=0;z2<3;z2++)
					if (as->m_oaAtoms[z]->m_iCoord[z2] != as2->m_oaAtoms[z]->m_iCoord[z2]) {
						eprintf("        Error %d[%d]: %.6f != %.6f\n",z,z2,as->m_oaAtoms[z]->m_fCoord[z2],as2->m_oaAtoms[z]->m_fCoord[z2]);
						err = true;
						goto _end;
					}
		
			if (verbose)
				mprintf("Done.\n");
		}

		if (outp != NULL) {
			bf.CreateShortFrame((i==0)?BARBECUBE_FRAMETYPE_COMPTRAJSTART:BARBECUBE_FRAMETYPE_COMPTRAJ,0,i+1);
			bf.PushPayload(bsat.m_iaData);
			bf.FinalizeFrame();
		}

		//if (b != NULL) {
		//	bshe.ExportToFile(b);
		//	bsat.ExportToFile(b);
		//}

		i++;
		if (i == steps)
			break;
	}
_end:

//	fclose(c);

	fclose(a);

	//if (b != NULL)
	//	fclose(b);

	if (outp != NULL) {
		bf.WriteIndexFrame(true);
		bf.Close();
	}

	if (fref != NULL)
		fclose(fref);

	if (err) {
		mprintf("Errors occurred while compressing the XYZ file.\n");
		return false;
	} else {
		mprintf("Finished compressing the XYZ file.\n");
//		mprintf("%9.3f MiB in total.\n",tb/1024.0/1024.0);
		mprintf("%10.3f MiB (%12s Bytes) overhead.\n",double(gc_oStatistics.m_lOverhead)/1024.0/1024.0/8.0,(gc_oStatistics.m_lOverhead>>3).string());
		mprintf("%10.3f MiB (%12s Bytes) alphabet data.\n",double(gc_oStatistics.m_lAlphabet)/1024.0/1024.0/8.0,(gc_oStatistics.m_lAlphabet>>3).string());
		mprintf("%10.3f MiB (%12s Bytes) Huffman tables.\n",double(gc_oStatistics.m_lHuffmanTables)/1024.0/1024.0/8.0,(gc_oStatistics.m_lHuffmanTables>>3).string());
		mprintf("%10.3f MiB (%12s Bytes) table switching.\n",double(gc_oStatistics.m_lTableSwitch)/1024.0/1024.0/8.0,(gc_oStatistics.m_lTableSwitch>>3).string());
		mprintf("%10.3f MiB (%12s Bytes) payload data.\n",double(gc_oStatistics.m_lHuffmanData)/1024.0/1024.0/8.0,(gc_oStatistics.m_lHuffmanData>>3).string());
		mprintf("%10.3f MiB (%12s Bytes) in total.\n\n",double(gc_oStatistics.m_lOverhead+gc_oStatistics.m_lAlphabet+gc_oStatistics.m_lHuffmanTables+gc_oStatistics.m_lTableSwitch+gc_oStatistics.m_lHuffmanData)/1024.0/1024.0/8.0,((gc_oStatistics.m_lOverhead+gc_oStatistics.m_lAlphabet+gc_oStatistics.m_lHuffmanTables+gc_oStatistics.m_lTableSwitch+gc_oStatistics.m_lHuffmanData)>>3).string());
//		mprintf("The sum of contributions is %s Bytes.\n\n",((gc_oStatistics.m_lOverhead+gc_oStatistics.m_lAlphabet+gc_oStatistics.m_lHuffmanTables+gc_oStatistics.m_lTableSwitch+gc_oStatistics.m_lHuffmanData)>>3).string());
		if (tf > 0)
			mprintf("%9.3f KiB per frame on average.\n",tb/1024.0/tf);
		if (tf2 > 0)
			mprintf("%9.3f KiB per frame on average (starting from step %d).\n\n",tb2/1024.0/tf2,morder+1);
	}

	return true;
}


bool CCCEngine::DecompressXYZ(const char *inp, const char *outp, const char *readref, int steps, int stride, bool verbose) {

	(void)stride;

	CAtomSet *as, *as2;
	int z, z2, i/*, al*/;
	//unsigned int crc, crc2;
	CBitSet bsat/*, bshe*/;
	FILE /**a,*/ *b, *fref;
	//CCRC32 crcengine;
	CBarbecubeFile bf;

	mprintf("Opening archive file \"%s\" ...\n",inp);
	if (!bf.OpenRead(inp)) {
		mprintf("Error: Could not open file for reading.\n");
		return false;
	}
/*	a = fopen(inp,"rb");
	if (a == NULL) {
		mprintf("Error: Could not open file for reading.\n");
		return false;
	}*/

	mprintf("Opening output XYZ file \"%s\" ...\n",outp);
	b = fopen(outp,"wb");
	if (b == NULL) {
		mprintf("Error: Could not open file for writing.\n");
		return false;
	}

	fref = NULL;
	if (readref != NULL) {
		mprintf("Opening reference XYZ file \"%s\" ...\n",readref);
		fref = fopen(readref,"rb");
		if (fref == NULL) {
			mprintf("Error: Could not open file for reading.\n");
			fclose(b);
			return false;
		}
	}

	m_oaOutputAtomBuf.resize(11);
	for (z=0;z<11;z++)
		m_oaOutputAtomBuf[z] = NULL;
	m_iOutputAtomBufPos = 0;

	mprintf("Starting process...\n");

	i = 0;
	while (true) {

		if (verbose)
			mprintf("\n\n");

		//if (!bshe.ImportFromFile(a,8))
		//	break;

		if (!bf.ReadFrame())
			break;

		if ((bf.GetFrameType() == BARBECUBE_FRAMETYPE_IDX) || (bf.GetFrameType() == BARBECUBE_FRAMETYPE_COMPIDX))
			continue;

		if (((bf.GetFrameType() != BARBECUBE_FRAMETYPE_COMPTRAJ) && (bf.GetFrameType() != BARBECUBE_FRAMETYPE_COMPTRAJSTART)) || (bf.GetFrameTypeVersion() != 0)) {
			mprintf("Error: Unexpected frame type (expected %d v0 or %d v0, found %d v%d).\n",BARBECUBE_FRAMETYPE_COMPTRAJ,BARBECUBE_FRAMETYPE_COMPTRAJSTART,bf.GetFrameType(),bf.GetFrameTypeVersion());
			return false;
		}

		mprintf("    Step %d ...\n",i+1);

		//al = bshe.ReadBitsInteger(32);
		//crc = bshe.ReadBitsInteger(32);

		if (verbose)
			mprintf("Atom block %lu bytes.\n",bf.GetFramePayload()->size());

		as = new CAtomSet();
		PushOutputAtomFrame(as);

		bsat.Clear();
		bsat.m_iaData.assign(bf.GetFramePayload()->begin(),bf.GetFramePayload()->end());

		//if (!bsat.ImportFromFile(a,al))
		//	break;

		/*crc2 = crcengine.ComputeCRC32(bsat.m_iaData);

		if (crc != crc2) {
			mprintf("CRC error while reading archive file. %08X != %08X\n",crc,crc2);
			break;
		} else if (verbose)
			mprintf("CRC check passed.\n");*/

		DecompressAtomFrame(&bsat,verbose,false);

		if (readref != NULL) {

			mprintf("      Comparing to reference...\n");

			as2 = new CAtomSet();

			if (!as2->ReadXYZ(fref,as->m_iSigni,NULL))
				break;

			for (z=0;z<(int)as2->m_oaAtoms.size();z++) {
				for (z2=0;z2<3;z2++)
					if (as->m_oaAtoms[z]->m_iCoord[z2] != as2->m_oaAtoms[z]->m_iCoord[z2])
						mprintf("        Error %d[%d]: %.6f != %.6f\n",z,z2,as->m_oaAtoms[z]->m_fCoord[z2],as2->m_oaAtoms[z]->m_fCoord[z2]);
			}

			delete as2;

			if (verbose)
				mprintf("Done.\n");
		}

		mprintf("      %.3f KiB compressed data unpacked.\n",bf.GetFramePayload()->size()/1024.0);

		as->WriteXYZ(b,as->m_iSigni);

		if (verbose)
			mprintf("Done.\n");

		i++;
		if (i == steps)
			break;
	}

	//fclose(a);
	bf.Close();

	fclose(b);

	if (readref != NULL)
		fclose(fref);

	mprintf("Finished decompressing the XYZ file.\n\n");

	return true;
}


void CCCEngine::PushInputCubeFrame(CCubeFrame *cfr) {

	m_iInputCubeBufPos++;
	if (m_iInputCubeBufPos == (int)m_oaInputCubeBuf.size())
		m_iInputCubeBufPos = 0;

	if (m_oaInputCubeBuf[m_iInputCubeBufPos] != NULL)
		delete m_oaInputCubeBuf[m_iInputCubeBufPos];
	m_oaInputCubeBuf[m_iInputCubeBufPos] = cfr;
}


void CCCEngine::PushInputCubeFrame_NoDelete(CCubeFrame *cfr) {

	m_iInputCubeBufPos++;
	if (m_iInputCubeBufPos == (int)m_oaInputCubeBuf.size())
		m_iInputCubeBufPos = 0;
	m_oaInputCubeBuf[m_iInputCubeBufPos] = cfr;
}


CCubeFrame* CCCEngine::GetInputCubeFrame(int depth) {

	int ti;

	if (depth >= (int)m_oaInputCubeBuf.size()) {
		eprintf("CCCEngine::GetInputCubeFrame(): Out of range error (%d/%lu).\n",depth,m_oaInputCubeBuf.size());
		abort();
	}

	ti = m_iInputCubeBufPos - depth;
	if (ti < 0)
		ti += (int)m_oaInputCubeBuf.size();

	return m_oaInputCubeBuf[ti];
}


void CCCEngine::PushOutputCubeFrame(CCubeFrame *cfr) {

	m_iOutputCubeBufPos++;
	if (m_iOutputCubeBufPos == (int)m_oaOutputCubeBuf.size())
		m_iOutputCubeBufPos = 0;

	if (m_oaOutputCubeBuf[m_iOutputCubeBufPos] != NULL)
		delete m_oaOutputCubeBuf[m_iOutputCubeBufPos];
	m_oaOutputCubeBuf[m_iOutputCubeBufPos] = cfr;
}


void CCCEngine::PushOutput2CubeFrame(CCubeFrame *cfr) {

	m_iOutput2CubeBufPos++;
	if (m_iOutput2CubeBufPos == (int)m_oaOutput2CubeBuf.size())
		m_iOutput2CubeBufPos = 0;

	if (m_oaOutput2CubeBuf[m_iOutput2CubeBufPos] != NULL)
		delete m_oaOutput2CubeBuf[m_iOutput2CubeBufPos];
	m_oaOutput2CubeBuf[m_iOutput2CubeBufPos] = cfr;
}


CCubeFrame* CCCEngine::GetOutputCubeFrame(int depth) {

	int ti;

	if (depth >= (int)m_oaOutputCubeBuf.size()) {
		eprintf("CCCEngine::GetOutputCubeFrame(): Out of bounds (%d/%d).\n",depth,(int)m_oaOutputCubeBuf.size());
		abort();
	}

	ti = m_iOutputCubeBufPos - depth;
	if (ti < 0)
		ti += (int)m_oaOutputCubeBuf.size();

	return m_oaOutputCubeBuf[ti];
}


CCubeFrame* CCCEngine::GetOutput2CubeFrame(int depth) {

	int ti;

	if (depth >= (int)m_oaOutput2CubeBuf.size()) {
		eprintf("CCCEngine::GetOutput2CubeFrame(): Out of bounds (%d/%d).\n",depth,(int)m_oaOutput2CubeBuf.size());
		abort();
	}

	ti = m_iOutput2CubeBufPos - depth;
	if (ti < 0)
		ti += (int)m_oaOutput2CubeBuf.size();

	return m_oaOutput2CubeBuf[ti];
}


void CCCEngine::PushInputAtomFrame(CAtomSet *cfr) {

	m_iInputAtomBufPos++;
	if (m_iInputAtomBufPos == (int)m_oaInputAtomBuf.size())
		m_iInputAtomBufPos = 0;

	if (m_oaInputAtomBuf[m_iInputAtomBufPos] != NULL)
		delete m_oaInputAtomBuf[m_iInputAtomBufPos];
	m_oaInputAtomBuf[m_iInputAtomBufPos] = cfr;
}


void CCCEngine::PushInputAtomFrame_NoDelete(CAtomSet *cfr) {

	m_iInputAtomBufPos++;
	if (m_iInputAtomBufPos == (int)m_oaInputAtomBuf.size())
		m_iInputAtomBufPos = 0;
	m_oaInputAtomBuf[m_iInputAtomBufPos] = cfr;
}


CAtomSet* CCCEngine::GetInputAtomFrame(int depth) {

	int ti;

	if (depth >= (int)m_oaInputAtomBuf.size()) {
		eprintf("CCCEngine::GetInputAtomFrame(): Out of range error (%d/%lu).\n",depth,m_oaInputAtomBuf.size());
		abort();
	}

	ti = m_iInputAtomBufPos - depth;
	if (ti < 0)
		ti += (int)m_oaInputAtomBuf.size();

	return m_oaInputAtomBuf[ti];
}


void CCCEngine::PushOutputAtomFrame(CAtomSet *cfr) {

	m_iOutputAtomBufPos++;
	if (m_iOutputAtomBufPos == (int)m_oaOutputAtomBuf.size())
		m_iOutputAtomBufPos = 0;

	if (m_oaOutputAtomBuf[m_iOutputAtomBufPos] != NULL)
		delete m_oaOutputAtomBuf[m_iOutputAtomBufPos];
	m_oaOutputAtomBuf[m_iOutputAtomBufPos] = cfr;
}


void CCCEngine::PushOutput2AtomFrame(CAtomSet *cfr) {

	m_iOutput2AtomBufPos++;
	if (m_iOutput2AtomBufPos == (int)m_oaOutput2AtomBuf.size())
		m_iOutput2AtomBufPos = 0;

	if (m_oaOutput2AtomBuf[m_iOutput2AtomBufPos] != NULL)
		delete m_oaOutput2AtomBuf[m_iOutput2AtomBufPos];
	m_oaOutput2AtomBuf[m_iOutput2AtomBufPos] = cfr;
}


CAtomSet* CCCEngine::GetOutputAtomFrame(int depth) {

	int ti;

	if (depth >= (int)m_oaOutputAtomBuf.size()) {
		eprintf("CCCEngine::GetOutputAtomFrame(): Out of bounds (%d/%d).\n",depth,(int)m_oaOutputAtomBuf.size());
		abort();
	}

	ti = m_iOutputAtomBufPos - depth;
	if (ti < 0)
		ti += (int)m_oaOutputAtomBuf.size();

	return m_oaOutputAtomBuf[ti];
}


CAtomSet* CCCEngine::GetOutput2AtomFrame(int depth) {

	int ti;

	if (depth >= (int)m_oaOutput2AtomBuf.size()) {
		eprintf("CCCEngine::GetOutput2AtomFrame(): Out of bounds (%d/%d).\n",depth,(int)m_oaOutput2AtomBuf.size());
		abort();
	}

	ti = m_iOutput2AtomBufPos - depth;
	if (ti < 0)
		ti += (int)m_oaOutput2AtomBuf.size();

	return m_oaOutput2AtomBuf[ti];
}


bool SORT_AtomOrd(int i1, int i2) {

	return gc_iaAtomOrd[i1] > gc_iaAtomOrd[i2];
}


void CCCEngine::BuildAtomSort(const CAtomSet *as) {

	int z;

//	if (m_bAtomSortBuilt)
//		return;

	m_iaAtomSort.resize(as->m_oaAtoms.size());
	gc_iaAtomOrd.resize(as->m_oaAtoms.size());
	for (z=0;z<(int)m_iaAtomSort.size();z++) {
		m_iaAtomSort[z] = z;
		gc_iaAtomOrd[z] = as->m_oaAtoms[z]->m_iOrd;
	}

	std::stable_sort(m_iaAtomSort.begin(),m_iaAtomSort.end(),SORT_AtomOrd);

//	m_bAtomSortBuilt = true;

/*	mprintf("### AtomSort:\n");
	for (z=0;z<(int)m_iaAtomSort.size();z++)
		mprintf("  %2d) Ord=%2d, Pos=%2d\n",z+1,gc_iaAtomOrd[m_iaAtomSort[z]],m_iaAtomSort[z]+1);*/
}


bool CCCEngine::MergeBQB(const char *outfile, const std::vector<const char*> &infile, bool verbose) {

	int z, i, zi, zo;
	CBarbecubeFile *pout;
	std::vector<CBarbecubeFile*> pin;

	SetBarbecubeVerbose(verbose);

	mprintf("\n");
	mprintf(WHITE,"    *****************************\n");
	mprintf(WHITE,"    ***   Merging BQB Files   ***\n");
	mprintf(WHITE,"    *****************************\n\n");

	mprintf("    Will merge the input files\n");
	for (z=0;z<(int)infile.size();z++)
		mprintf("      [%d] %s\n",z+1,infile[z]);
	mprintf("    into the output file %s.\n\n",outfile);

	mprintf("    Opening input files...\n");
	pin.resize(infile.size());
	for (z=0;z<(int)infile.size();z++) {
		mprintf("      [%d] %s... ",z+1,infile[z]);
		pin[z] = new CBarbecubeFile();
		if (!pin[z]->OpenRead(infile[z])) {
			eprintf("Error.\n");
			return false;
		}
		if ((i = pin[z]->GetTotalFrameCount()) != -1)
			mprintf("Ok, %d frames (index).",i);
		else
			mprintf("Ok, no index.");
		mprintf("\n");
	}

	mprintf("\n    Opening output file...\n");
	pout = new CBarbecubeFile();
	if (!pout->OpenWriteReplace(outfile)) {
		eprintf("Error.\n");
		return false;
	}

	zo = 0;
	for (z=0;z<(int)pin.size();z++) {

		mprintf("    Processing input file %d.\n",z+1);
		zi = 0;

		while (true) {
			if (!pin[z]->ReadFrame()) {
				mprintf("    Reached end of input file %d after reading %d frames.\n",z+1,zi);
				break;
			}
			zi++;
			mprintf("      Input file %d frame %6d: Type %2d.%d, Index %6d, Size %7.2f KiB",z+1,zi,pin[z]->GetFrameType(),pin[z]->GetFrameTypeVersion(),pin[z]->GetFrameIndex(),pin[z]->GetFramePayload()->size()/1024.0);
			if ((pin[z]->GetFrameType() == 2) || (pin[z]->GetFrameType() == 3)) {
				mprintf(" --> Skipping index frame.\n");
				continue;
			}
			mprintf(" --> Output frame %6d.\n",zo);
			pout->CreateShortFrame(pin[z]->GetFrameType(),pin[z]->GetFrameTypeVersion(),pin[z]->GetFrameIndex());
			pout->PushPayload(*pin[z]->GetFramePayload());
			pout->FinalizeFrame();
			zo++;
		}
	}

	mprintf("    Merged %d frames from %lu input files into output file.\n\n",zo,infile.size());

	mprintf("    Writing new index frame for output file...\n");
	pout->WriteIndexFrame(true);
	mprintf("    All done.\n\n");
	pout->Close();
	delete pout;
	for (z=0;z<(int)pin.size();z++) {
		pin[z]->Close();
		delete pin[z];
	}

	return true;
}


bool CCCEngine::SplitBQB(const char *infile, const char *outbase, int splitlength, int steps, bool verbose) {

	int z, z2, i, k, ofc, ofc2, al, cl;
	int ao, co, aorder, corder;
	int lcsize, lasize;
	bool err, to;
	CBarbecubeFile bqin, *bqout;
	CCubeFrame *cfr, *cfr2;
	CBitSet bshe, bsat, bscu, bstemp;
	CxString buf;


/*	CIntegerEngine ie;
	std::vector<int> iain, iaout;
	CBitSet bs;

	for (z=0;z<15;z++)
		iain.push_back(32);

	ie.CompressSingle(iain,&bs,false,true,true,50,1,false,false,false,true);
	
	ie.DecompressSingle(&bs,iaout,true);

	for (z=0;z<(int)iaout.size();z++)
		mprintf("@ %2d: %d\n",z+1,iaout[z]);

	return true;*/


	lcsize = 1000000000; // To avoid "use of uninitialized variable" warning
	lasize = 1000000000;
	ofc = 0;
	ofc2 = 0;

	mprintf("\n");
	mprintf(WHITE,"    ******************************\n");
	mprintf(WHITE,"    ***   Splitting BQB File   ***\n");
	mprintf(WHITE,"    ******************************\n\n");

	SetBarbecubeVerbose(verbose);

	if (!bqin.OpenRead(infile)) {
		eprintf("CCCEngine::SplitBQB(): Error: Could not open BQB file \"%s\" for reading.\n",infile);
		return false;
	}

	m_oaOutputCubeBuf.resize(9);
	m_oaOutput2CubeBuf.resize(9);
	m_oaInputCubeBuf.resize(9);
	m_oaInputAtomBuf.resize(9);
	m_oaOutputAtomBuf.resize(9);
	m_oaOutput2AtomBuf.resize(9);
	for (z=0;z<9;z++) {
		m_oaInputCubeBuf[z] = NULL;
		m_oaOutputCubeBuf[z] = NULL;
		m_oaOutput2CubeBuf[z] = NULL;
		m_oaInputAtomBuf[z] = NULL;
		m_oaOutputAtomBuf[z] = NULL;
		m_oaOutput2AtomBuf[z] = NULL;
	}
	m_iInputCubeBufPos = 0;
	m_iOutputCubeBufPos = 0;
	m_iOutput2CubeBufPos = 0;
	m_iInputAtomBufPos = 0;
	m_iOutputAtomBufPos = 0;
	m_iOutput2AtomBufPos = 0;

	i = 0;
	k = 0;
	aorder = 8;
	corder = 8;
	bqout = NULL;
	to = false;

	mprintf("\n");
	while (true) {

		mprintf("    Reading input frame %d...\n",i+1);
		if (!bqin.ReadFrame()) {
			mprintf("Could not read further frame from input file.\n");
			break;
		}
		if ((bqin.GetFrameType() == 2) || (bqin.GetFrameType() == 3)) {
			mprintf("      Skipping index frame.\n");
			goto _skip;
		}
		if ((bqin.GetFrameType() != 8) && (bqin.GetFrameType() != 9)) {
			eprintf("CCCEngine::SplitBQB(): Error: Splitting only implemented for compressed cube frames yet.\n");
			return false;
		}

		if (bqout == NULL) {
			buf.sprintf("%s%03d.bqb",outbase,k+1);
			mprintf("Creating next output file \"%s\".\n",(const char*)buf);
			bqout = new CBarbecubeFile();
			if (!bqout->OpenWriteReplace((const char*)buf)) {
				eprintf("CCCEngine::SplitBQB(): Error: Could not open BQB file \"%s\" for writing.\n",(const char*)buf);
				return false;
			}
			lcsize = 1000000000;
			lasize = 1000000000;
			aorder = 8;
			corder = 8;
			ofc = 0;
			ofc2 = 0;
		}

		mprintf("      Decompressing...\n");

		bshe.Clear();
		bshe.m_iaData.assign(bqin.GetFramePayload()->begin(),bqin.GetFramePayload()->begin()+8);

		al = bshe.ReadBitsInteger(32);
		cl = bshe.ReadBitsInteger(32);

		cfr = new CCubeFrame();
		PushOutputCubeFrame(cfr);
		PushInputCubeFrame_NoDelete(cfr);

		cfr->m_pAtoms = new CAtomSet();
		PushOutputAtomFrame(cfr->m_pAtoms);
		PushInputAtomFrame_NoDelete(cfr->m_pAtoms);

		bsat.Clear();
		bsat.m_iaData.assign(bqin.GetFramePayload()->begin()+8,bqin.GetFramePayload()->begin()+8+al);

		bscu.Clear();
		bscu.m_iaData.assign(bqin.GetFramePayload()->begin()+8+al,bqin.GetFramePayload()->begin()+8+al+cl);

		if (!DecompressAtomFrame(&bsat,verbose,false))
			return false;

		if (!DecompressCubeFrame(&bscu,verbose,false))
			return false;

		if (ofc >= 9) {

			mprintf("      Pass-through writing output file %d, frame %d...\n",k+1,ofc+1);

			bqout->CreateShortFrame((ofc==0)?BARBECUBE_FRAMETYPE_COMPCUBESTART:BARBECUBE_FRAMETYPE_COMPCUBE,0,i+1);
			bqout->PushPayload(*bqin.GetFramePayload());
			bqout->FinalizeFrame();

		} else {

			mprintf("      Re-compressing...\n");

_again:
			if (ofc2 < corder)
				co = ofc2;
			else if (ofc < corder)
				co = ofc;
			else
				co = corder;

			if (ofc2 < aorder)
				ao = ofc2;
			else if (ofc < aorder)
				ao = ofc;
			else
				ao = aorder;

_aagain:
			bsat.Clear();
			CompressAtomFrame(
				&bsat,
				ao,
				6,         // aprecision
				31,        // asplit
				40,        // ablock
				4,         // atables
				false,     // optatables
				true,      // acoderun
				true,      // abw
				true,      // amtf
				(ofc2>=2), // apreopt
				(ofc==0),  // atominfo
				true,      // keepcomment
				0,         // amaxchunk
				verbose
				);

			mprintf("        Atoms: Order %d, output size %9.3f KiB.\n",ao,bsat.GetByteLength()/1024.0);

			if (lasize >= 0) {
				if (to) {
					to = false;
					if (bsat.GetByteLength() <= bstemp.GetByteLength()) {
						mprintf("        Size is indeed smaller with lower order. Limiting atom order to %d.\n",ao);
						aorder = ao;
						lasize = -1;
					} else {
						mprintf("        Size was smaller with higher order. Not limiting.\n");
						bsat = CBitSet(bstemp);
						lasize = bsat.GetByteLength();
					}
				} else {
					if ((ao != 0) && ((double)bsat.GetByteLength() >= (double)lasize*0.97)) {
						mprintf("        Size did not decrease any further with order %d. Trying order %d...\n",ao,ao-1);
						bstemp = CBitSet(bsat);
						to = true;
						ao--;
						goto _aagain;
					} else if (ao == aorder)
						lasize = -1;
					else
						lasize = bsat.GetByteLength();
				}
			}

_cagain:
			bscu.Clear();
			CompressCubeFrame(
				&bscu,
				co,
				12,       // ceps
				5,        // csigni
				31,       // csplit
				40,       // cblock
				6,        // ctables
				false,    // copttables
				true,     // chilbert
				true,     // ccoderun
				false,    // cbw
				false,    // cmtf
				false,    // cpreopt
				(ofc==0), // atominfo
				0,        // cmaxchunk
				verbose
				);

			mprintf("        Cube:  Order %d, output size %9.3f KiB.\n",co,bscu.GetByteLength()/1024.0);

	/*		if (lcsize >= 0) {
				if ((co != 0) && ((double)bscu.GetByteLength() >= (double)lcsize*0.97)) {
					mprintf("        Size did not decrease any further with order %d. Limiting cube order to %d.\n",co,co-1);
					co--;
					corder = co;
					lcsize = -1;
					goto _cagain;
				} else if (co == corder)
					lcsize = -1;
				else
					lcsize = bscu.GetByteLength();
			}*/

			if (lcsize >= 0) {
				if (to) {
					to = false;
					if (bscu.GetByteLength() <= bstemp.GetByteLength()) {
						mprintf("        Size is indeed smaller with lower order. Limiting cube order to %d.\n",co);
						corder = co;
						lcsize = -1;
					} else {
						mprintf("        Size was smaller with higher order. Not limiting.\n");
						bscu = CBitSet(bstemp);
						lcsize = bscu.GetByteLength();
					}
				} else {
					if ((co != 0) && ((double)bscu.GetByteLength() >= (double)lcsize*0.97)) {
						mprintf("        Size did not decrease any further with order %d. Trying order %d...\n",co,co-1);
						bstemp = CBitSet(bscu);
						to = true;
						co--;
						goto _cagain;
					} else if (co == corder)
						lcsize = -1;
					else
						lcsize = bscu.GetByteLength();
				}
			}

			mprintf("        Comparing input and output...\n");

			cfr2 = new CCubeFrame();
			PushOutput2CubeFrame(cfr2);
			cfr2->m_pAtoms = new CAtomSet();
			PushOutput2AtomFrame(cfr2->m_pAtoms);

			DecompressAtomFrame(&bsat,verbose,true);

			DecompressCubeFrame(&bscu,verbose,true);

			err = false;
			for (z=0;z<(int)cfr2->m_pAtoms->m_oaAtoms.size();z++)
				for (z2=0;z2<3;z2++)
					if (cfr->m_pAtoms->m_oaAtoms[z]->m_iCoord[z2] != cfr2->m_pAtoms->m_oaAtoms[z]->m_iCoord[z2]) {
						eprintf("        Error in atom coordinate %d[%d]: %.6f (%ld) != %.6f (%ld)\n",z,z2,cfr->m_pAtoms->m_oaAtoms[z]->m_fCoord[z2],cfr->m_pAtoms->m_oaAtoms[z]->m_iCoord[z2],cfr2->m_pAtoms->m_oaAtoms[z]->m_fCoord[z2],cfr2->m_pAtoms->m_oaAtoms[z]->m_iCoord[z2]);
						err = true;
					}

			for (z=0;z<cfr->m_iResXYZ;z++)
				if (!ExpMantisEqual(cfr->m_iaExpo[z],cfr->m_iaMantis[z],cfr2->m_iaExpo[z],cfr2->m_iaMantis[z])) {
					eprintf("        Error in volumetric data element %7d: %.10G vs %.10G\n",z,cfr->m_faBin[z],cfr2->m_faBin[z]);
					err = true;
				}

			if (err) {
				if (ofc2 != 0) {
					eprintf("Errors occured. Compressing frame again with order zero.\n");
					err = false;
					ofc2 = 0;
					goto _again;
				} else {
					eprintf("Errors occured despite of order zero. Aborting.\n");
					return false;
				}
			}

			bshe.Clear();
			bshe.WriteBits(bsat.GetByteLength(),32);
			bshe.WriteBits(bscu.GetByteLength(),32);

			mprintf("      Writing output file %d, frame %d...\n",k+1,ofc+1);

			bqout->CreateShortFrame((ofc2==0)?BARBECUBE_FRAMETYPE_COMPCUBESTART:BARBECUBE_FRAMETYPE_COMPCUBE,0,i+1);
			bqout->PushPayload(bshe.m_iaData);
			bqout->PushPayload(bsat.m_iaData);
			bqout->PushPayload(bscu.m_iaData);
			bqout->FinalizeFrame();
		}

_skip:
		i++;
		ofc++;
		ofc2++;

		if ((i % splitlength) == 0) {
			mprintf("Closing output file \"%s\".\n",(const char*)buf);
			bqout->WriteIndexFrame(true);
			bqout->Close();
			delete bqout;
			bqout = NULL;
			k++;
		}

		if (i == steps) {
			mprintf("Reached step limit.\n\n");
			break;
		}
	}

	if (bqout != NULL) {
		mprintf("Closing output file \"%s\".\n\n",(const char*)buf);
		bqout->WriteIndexFrame(true);
		bqout->Close();
		delete bqout;
		bqout = NULL;
		k++;
	}

	mprintf("Processed %d frames, wrote %d output files.\n\n",i,k);

	bqin.Close();

	// Those are duplicates of the output buffers and deleted there
	m_oaInputAtomBuf.clear();
	m_oaInputCubeBuf.clear();

	mprintf("All done. Leaving.\n");

	return true;
}



