/*
 * tvflash - update firmware flash memory on Mellanox HCAs
 * Copyright (c) 2004 Topspin Communications.  All rights reserved.
 * Copyright (c) 2007 Cisco, Inc.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <sys/mman.h>
#include <termios.h>
#include <fcntl.h>
#include <pci/pci.h>
#include <sys/utsname.h>
#include <limits.h>
#include <ctype.h>
#include <time.h>
#include <netinet/in.h>

#include "firmware.h"

static struct tvdevice *tvdevices = NULL;
static struct tvdevice *tvdevices_tail = NULL;

static unsigned char force = 0;
static unsigned char verbose = 1;
static unsigned char use_config_space = 0;

static const char *argv0;

static unsigned short cols;

static const uint32_t hermon_fw_sig[] = {
	0x4d544657, 0x8cdfd000, 0xdead9270, 0x4154beef
};

/* Topspin OID */
static const uint8_t default_oid[] = { 0x00, 0x05, 0xad, 0x00 };

/* Tavor */
static struct board jaguar	   = { "Jaguar", CHIP_TAVOR, "HCA.Jaguar", 2 };
static struct board cougar	   = { "Cougar", CHIP_TAVOR, "HCA.Cougar", 2 };
static struct board cougarcub	   = { "Cougar Cub", CHIP_TAVOR, "HCA.CougarCub", 2 };

/* Arbel */
static struct board lioncub	   = { "Lion Cub", CHIP_ARBEL, "HCA.LionCub", 2 };
static struct board lioncub_revc   = { "Lion Cub", CHIP_ARBEL, "HCA.LionCub.RevC", 2 };
static struct board lioncub_ddr	   = { "Lion Cub DDR", CHIP_ARBEL, "HCA.LionCub-DDR", 2 };
static struct board glacier	   = { "DLGL", CHIP_ARBEL, "HCA.DLGL", 2 };
static struct board bc2		   = { "BC2 HSDC", CHIP_ARBEL, "HCA.HSDC", 2 };
static struct board lionmini	   = { "Lion Mini", CHIP_ARBEL, "HCA.LionMini", 2 };
static struct board lionmini_ddr   = { "Lion Mini DDR", CHIP_ARBEL, "HCA.LionMini-DDR", 2 };
static struct board genarbel	   = { "Generic Arbel", CHIP_ARBEL, NULL, 2 };

/* Sinai */
static struct board tiger	   = { "Tiger", CHIP_SINAI, "HCA.Tiger", 1 };
static struct board cheetah	   = { "Cheetah", CHIP_SINAI, "HCA.Cheetah", 1 };
static struct board cheetah_ddr	   = { "Cheetah DDR", CHIP_SINAI, "HCA.Cheetah-DDR", 1 };
static struct board gensinai	   = { "Generic Sinai", CHIP_SINAI, NULL, 1 };

/* Hermon */
static struct board eagle_ddr	   = { "Eagle DDR", CHIP_HERMON, "HCA.Eagle-DDR", 2 };
static struct board genhermon	   = { "Generic Hermon", CHIP_HERMON, NULL, 2 };

static struct board_vpd board_vpds[] = {
	{ "Cougar cub",		&cougarcub },
	{ "Lion cub",		&lioncub },
	{ "Lion cub DDR",	&lioncub_ddr },
	{ "Lion mini",		&lionmini },
	{ "Lion mini DDR",	&lionmini_ddr },
	{ "DLGL",		&glacier },
	{ "Tiger",		&tiger },
	{ "Cheetah",		&cheetah },
	{ "Cheetah DDR",	&cheetah_ddr },
	{ "BC2 HSDC",		&bc2 },
	{ "Eagle DDR",		&eagle_ddr },
};

static struct board *tavor_identify(struct tvdevice *);
static struct board *arbel_identify(struct tvdevice *);
static struct board *sinai_identify(struct tvdevice *);
static struct board *hermon_identify(struct tvdevice *);

static struct pciid pciids[] = {
	/* Tavor, only one has VPD */
	{ PCI_VENDOR_MELLANOX, PCI_DEVICE_MELLANOX_MT23108,
	  0, tavor_identify },
	{ PCI_VENDOR_TOPSPIN,	PCI_DEVICE_MELLANOX_MT23108,
	  0, tavor_identify },
	{ PCI_VENDOR_MELLANOX, PCI_DEVICE_MELLANOX_MT23108_PCICONF,
	  FLAG_RECOVERY, tavor_identify },
	{ PCI_VENDOR_TOPSPIN,	PCI_DEVICE_MELLANOX_MT23108_PCICONF,
	  FLAG_RECOVERY, tavor_identify },

	/* Arbel, all have VPDs */
	{ PCI_VENDOR_MELLANOX, PCI_DEVICE_MELLANOX_MT25208,
	  0, arbel_identify },
	{ PCI_VENDOR_TOPSPIN,	PCI_DEVICE_MELLANOX_MT25208,
	  0, arbel_identify },
	{ PCI_VENDOR_MELLANOX, PCI_DEVICE_MELLANOX_MT25208_COMPAT,
	  FLAG_TAVOR_COMPAT, arbel_identify },
	{ PCI_VENDOR_TOPSPIN,	PCI_DEVICE_MELLANOX_MT25208_COMPAT,
	  FLAG_TAVOR_COMPAT, arbel_identify },
	{ PCI_VENDOR_MELLANOX, PCI_DEVICE_MELLANOX_MT25208_PCICONF,
	  FLAG_RECOVERY, arbel_identify },
	{ PCI_VENDOR_TOPSPIN,	PCI_DEVICE_MELLANOX_MT25208_PCICONF,
	  FLAG_RECOVERY, arbel_identify },

	/* Sinai, all have VPDs */
	{ PCI_VENDOR_MELLANOX, PCI_DEVICE_MELLANOX_MT25204,
	  0, sinai_identify },
	{ PCI_VENDOR_TOPSPIN,	PCI_DEVICE_MELLANOX_MT25204,
	  0, sinai_identify },
	{ PCI_VENDOR_MELLANOX, PCI_DEVICE_MELLANOX_MT25204_PCICONF,
	  FLAG_RECOVERY, sinai_identify },
	{ PCI_VENDOR_TOPSPIN,	PCI_DEVICE_MELLANOX_MT25204_PCICONF,
	  FLAG_RECOVERY, sinai_identify },

	/* Some old Tiger and Cheetah boards use PCI device id 0x538x (24204) */
	{ PCI_VENDOR_MELLANOX, 0x5e8c,
	  0, sinai_identify },
	{ PCI_VENDOR_MELLANOX, 0x5e8d,
	  FLAG_RECOVERY, sinai_identify },

	/* Hermon, all have VPDs */
	{ PCI_VENDOR_MELLANOX, PCI_DEVICE_MELLANOX_MT25408_SDR,
	  0, hermon_identify },
	{ PCI_VENDOR_MELLANOX, PCI_DEVICE_MELLANOX_MT25408_DDR,
	  0, hermon_identify },
	{ PCI_VENDOR_MELLANOX, PCI_DEVICE_MELLANOX_MT25408_QDR,
	  0, hermon_identify },
	{ PCI_VENDOR_MELLANOX, PCI_DEVICE_MELLANOX_MT25408_DDR_PCIE2,
	  0, hermon_identify },
	{ PCI_VENDOR_MELLANOX, PCI_DEVICE_MELLANOX_MT25408_QDR_PCIE2,
	  0, hermon_identify },
};

/*
 * Simple utility functions
 */

/* Calculate the CRC16 of an image (list of bytes) */
static uint16_t flash_crc16(uint8_t *image, uint32_t size)
{
	uint32_t crc = 0xffff;
	uint32_t i, j;
	uint32_t word;

	for (j = 0; j < size / 4; j++) {
		word = (image[4 * j] << 24) | (image[4 * j + 1] << 16) |
			(image[4 * j + 2] << 8) | image[4 * j + 3];
		for (i = 0; i < 32; i++) {
			if (crc & 0x8000)
				crc = (((crc << 1) | (word >> 31)) ^ 0x100b) & 0xffff;
			else
				crc = ((crc << 1) | (word >> 31)) & 0xffff;
			word = word << 1;
		}
	}

	for (i = 0; i < 16; i++) {
		if (crc & 0x8000)
			crc = ((crc << 1) ^ 0x100b) & 0xffff;
		else
			crc = (crc << 1) & 0xffff;
	}

	crc = crc ^ 0xffff;

	return (crc & 0xffff);
}

/*
 * We need some LE to host macros and those vary depending on the architecture
 * and there are no standardized ones. So we mix the ntoh* and hton* functions
 * with an unconditional swab* to get the desired behavior. Not the most
 * efficient, but it works and is portable.
 */
static uint16_t le16_to_cpu(uint16_t x)
{
	uint16_t i = htons(x);

	return ((i >> 8) & 0xff) | ((i & 0xff) << 8);
}

#if 0
static uint16_t cpu_to_le16(uint16_t x)
{
	uint16_t i = ntohs(x);

	return ((i >> 8) & 0xff) | ((i & 0xff) << 8);
}
#endif

static uint32_t swab32(uint32_t x)
{
	return ((x >> 24) | ((x >> 8) & 0xff00) | ((x & 0xff00) << 8) | (x << 24));
}

#define be16_to_cpu ntohs
#define cpu_to_be16 htons
#define be32_to_cpu ntohl
#define cpu_to_be32 htonl

static int parse_hex_str(char *arg, unsigned char *hexstr, int octets)
{
	if (strchr(arg, ':') != NULL) {
		char *p;
		int octetspercolon;

		/* Determine the format of the GUID. Is it single or double octet? */
		p = strrchr(arg, ':');
		if (*(p + 1) && *(p + 2) && *(p + 3) && *(p + 4) && !*(p + 5))
			octetspercolon = 2;
		else if (*(p + 1) && *(p + 2) && !*(p + 3))
			octetspercolon = 1;
		else {
			fprintf(stderr, "Unknown hex string format\n");
			return -1;
		}

		/*
		 * We don't require a full GUID here. If the user doesn't
		 * specify a full GUID, we'll put the Topspin OID and 0 pad
		 * the rest and use as much of a GUID the user gives
		 */
		do {
			unsigned char value, hex;

			p = strrchr(arg, ':');
			if (p)
				*(p++) = 0;
			else
				p = arg;

			if (octetspercolon == 2) {
				/* There should be no more characters after this */
				if (*(p + 4) != 0)
					return -1;

				hex = toupper(*(p + 2));
				if (hex >= '0' && hex <= '9')
					value = (hex - '0') << 4;
				else if (hex >= 'A' && hex <= 'F')
					value = (hex - 'A' + 10) << 4;
				else
					return -1;

				hex = toupper(*(p + 3));
				if (hex >= '0' && hex <= '9')
					value |= hex - '0';
				else if (hex >= 'A' && hex <= 'F')
					value |= hex - 'A' + 10;
				else
					return -1;

				hexstr[--octets] = value;
			} else {
				/* There should be no more characters after this */
				if (*(p + 2) != 0)
					return -1;
			}

			hex = toupper(*p);
			if (hex >= '0' && hex <= '9')
				value = (hex - '0') << 4;
			else if (hex >= 'A' && hex <= 'F')
				value = (hex - 'A' + 10) << 4;
			else
				return -1;

			hex = toupper(*(p + 1));
			if (hex >= '0' && hex <= '9')
				value |= hex - '0';
			else if (hex >= 'A' && hex <= 'F')
				value |= hex - 'A' + 10;
			else
				return -1;

			hexstr[--octets] = value;
		} while (p > arg && octets > 0);

		/* Data left over */
		if (p > arg)
			return 1;
	} else {
		unsigned long low_guid = strtol(arg, NULL, 16);

		/* FIXME: Check the result of strtol, or do something more intelligent */

		hexstr[octets - 4] = (low_guid >> 24) & 0xff;
		hexstr[octets - 3] = (low_guid >> 16) & 0xff;
		hexstr[octets - 2] = (low_guid >> 8) & 0xff;
		hexstr[octets - 1] = low_guid & 0xff;

		octets -= 4;
	}

	return octets;
}

static int parse_guid(char *arg, unsigned char *new_guid)
{
	int octets;

	octets = parse_hex_str(arg, new_guid, GUID_LEN);

	if (octets < 0)
		return 1;

	if (octets > 0 && octets < 3) {
		fprintf(stderr, "GUID is not 8 octets, but longer than 4 maximum to automatically generate\n");
		return 1;
	}

	if (octets >= 3) {
		memcpy(new_guid, default_oid, 4);

		if (octets > 3)
			/* Zero out the non specified portions */
			memset(new_guid + 3, 0, octets - 3);
	}

	return 0;
}

static int parse_change_options(struct topspin_vsd *vsd, char *cmd)
{
	char *p = cmd, *end = strchr(cmd, 0);

	while (p && p < end) {
		char *pc, *pe;

		pc = strchr(p, ',');
		if (pc) {
			*pc = 0;
			pc++;
		}

		pe = strchr(p, '=');
		if (!pe) {
			fprintf(stderr, "Missing = in token '%s'\n", p);
			return 1;
		}

		*pe = 0;
		pe++;

		if (strcasecmp(p, "auto_upgrade") == 0) {
			if (strcasecmp(pe, "yes") == 0 ||
			    strcasecmp(pe, "on") == 0 ||
			    strcasecmp(pe, "1") == 0)
				vsd->flags |= cpu_to_be32(VSD_FLAG_AUTOUPGRADE);
			else if (strcasecmp(pe, "no") == 0 ||
				 strcasecmp(pe, "off") == 0 ||
				 strcasecmp(pe, "0") == 0)
				vsd->flags &= ~cpu_to_be32(VSD_FLAG_AUTOUPGRADE);
			else {
				fprintf(stderr, "Unknown boolean value '%s'\n", pe);
				return 1;
			}
		} else if (strcasecmp(p, "boot_enable_port_1") == 0) {
			if (strcasecmp(pe, "yes") == 0 ||
			    strcasecmp(pe, "on") == 0 ||
			    strcasecmp(pe, "1") == 0)
				vsd->flags |= cpu_to_be32(VSD_FLAG_BOOT_ENABLE_PORT_1);
			else if (strcasecmp(pe, "no") == 0 ||
				 strcasecmp(pe, "off") == 0 ||
				 strcasecmp(pe, "0") == 0)
				vsd->flags &= ~cpu_to_be32(VSD_FLAG_BOOT_ENABLE_PORT_1);
			else {
				fprintf(stderr, "Unknown boolean value '%s'\n", pe);
				return 1;
			}
		} else if (strcasecmp(p, "boot_enable_port_2") == 0) {
			if (strcasecmp(pe, "yes") == 0 ||
			    strcasecmp(pe, "on") == 0 ||
			    strcasecmp(pe, "1") == 0)
				vsd->flags |= cpu_to_be32(VSD_FLAG_BOOT_ENABLE_PORT_2);
			else if (strcasecmp(pe, "no") == 0 ||
				 strcasecmp(pe, "off") == 0 ||
				 strcasecmp(pe, "0") == 0)
				vsd->flags &= ~cpu_to_be32(VSD_FLAG_BOOT_ENABLE_PORT_2);
			else {
				fprintf(stderr, "Unknown boolean value '%s'\n", pe);
				return 1;
			}
		} else if (strcasecmp(p, "boot_service_scan") == 0) {
			if (strcasecmp(pe, "yes") == 0 ||
			    strcasecmp(pe, "on") == 0 ||
			    strcasecmp(pe, "1") == 0)
				vsd->flags |= cpu_to_be32(VSD_FLAG_BOOT_ENABLE_SCAN);
			else if (strcasecmp(pe, "no") == 0 ||
				 strcasecmp(pe, "off") == 0 ||
				 strcasecmp(pe, "0") == 0)
				vsd->flags &= ~cpu_to_be32(VSD_FLAG_BOOT_ENABLE_SCAN);
			else {
				fprintf(stderr, "Unknown boolean value '%s'\n", pe);
				return 1;
			}
		} else if (strcasecmp(p, "boot_wait_on_error") == 0) {
			if (strcasecmp(pe, "yes") == 0 ||
			    strcasecmp(pe, "on") == 0 ||
			    strcasecmp(pe, "1") == 0)
				vsd->flags |= cpu_to_be32(VSD_FLAG_BOOT_WAIT_ON_ERROR);
			else if (strcasecmp(pe, "no") == 0 ||
				 strcasecmp(pe, "off") == 0 ||
				 strcasecmp(pe, "0") == 0)
				vsd->flags &= ~cpu_to_be32(VSD_FLAG_BOOT_WAIT_ON_ERROR);
			else {
				fprintf(stderr, "Unknown boolean value '%s'\n", pe);
				return 1;
			}
		} else if (strcasecmp(p, "boot_try_forever") == 0) {
			if (strcasecmp(pe, "yes") == 0 ||
			    strcasecmp(pe, "on") == 0 ||
			    strcasecmp(pe, "1") == 0)
				vsd->flags |= cpu_to_be32(VSD_FLAG_BOOT_TRY_FOREVER);
			else if (strcasecmp(pe, "no") == 0 ||
				 strcasecmp(pe, "off") == 0 ||
				 strcasecmp(pe, "0") == 0)
				vsd->flags &= ~cpu_to_be32(VSD_FLAG_BOOT_TRY_FOREVER);
			else {
				fprintf(stderr, "Unknown boolean value '%s'\n", pe);
				return 1;
			}
		} else if (strcasecmp(p, "boot_type") == 0) {
			vsd->flags &= ~cpu_to_be32(VSD_FLAG_BOOT_TYPE);
			if (strcasecmp(pe, "well_known") == 0)
				vsd->flags |= cpu_to_be32(VSD_FLAG_BOOT_TYPE_WELL_KNOWN);
			else if (strcasecmp(pe, "saved") == 0)
				vsd->flags |= cpu_to_be32(VSD_FLAG_BOOT_TYPE_SAVED);
			else if (strcasecmp(pe, "pxe") == 0)
				vsd->flags |= cpu_to_be32(VSD_FLAG_BOOT_TYPE_PXE);
			else if (strcasecmp(pe, "disable") == 0)
				vsd->flags |= cpu_to_be32(VSD_FLAG_BOOT_TYPE_DISABLE);
			else {
				fprintf(stderr, "Unknown boot type '%s'\n", pe);
				return 1;
			}
		} else if (strcasecmp(p, "boot_saved_port") == 0) {
			unsigned long port;

			port = strtoul(pe, NULL, 10);
			if (port > 2) {
				fprintf(stderr, "Invalid port number '%s'\n", pe);
				return 1;
			}

			vsd->boot_port = port;
		} else if (strcasecmp(p, "boot_saved_ioc_num") == 0) {
			unsigned long ioc;

			ioc = strtoul(pe, NULL, 10);
			if (ioc == ULONG_MAX) {
				fprintf(stderr, "Invalid ioc number '%s'\n", pe);
				return 1;
			}

			vsd->boot_ioc_num = ioc;
		} else if (strcasecmp(p, "boot_saved_dgid") == 0) {
			if (parse_hex_str(pe, vsd->boot_dgid, 16) < 0) {
				fprintf(stderr, "Couldn't parse dgid\n");
				return 1;
			}
		} else if (strcasecmp(p, "boot_saved_service_name") == 0) {
			if (parse_hex_str(pe, vsd->boot_service_name, 8) < 0) {
				fprintf(stderr, "Couldn't parse service name\n");
				return 1;
			}
		} else if (strcasecmp(p, "boot_pxe_secs") == 0) {
			unsigned long secs;

			secs = strtoul(pe, NULL, 10);
			if (secs > 250) {
				fprintf(stderr, "Invalid number of seconds '%s'\n", pe);
				return 1;
			}

			vsd->boot_pxe_secs = secs;
		} else {
			fprintf(stderr, "Unknown option '%s'\n", p);
			return 1;
		}

		p = pc;
	}

	return 0;
}

static void inc_guid(unsigned char *guid, unsigned char *new_guid)
{
	int i, carry = 1;

	/* FIXME: Do we really want to try to increment the vendor oid too? */
	for (i = 7; i >= 0; i--) {
		if (carry && guid[i] == 0xFF) {
			new_guid[i] = 0;
			carry = 1;
		} else {
			new_guid[i] = guid[i] + carry;
			carry = 0;
		}
	}
}

unsigned int status_curpos, status_relpos, status_maxpos;
unsigned int status_curbar;
unsigned char status_prev;

static void status_start(unsigned int max)
{
	status_curpos = 0;
	status_relpos = 0;
	status_maxpos = max;

	status_curbar = 0;
	status_prev = 0;

	if (verbose)
		printf("Flashing - ");
}

static void status_update(unsigned char status, unsigned int curpos)
{
	unsigned int bar;

	if (!verbose)
		return;

	curpos += status_relpos;

	bar = (curpos * (cols - 12)) / status_maxpos;
	status_curpos = curpos;

	/* Force a character to be printed during a status change */
	if (status != status_prev && status_curbar >= bar)
		bar = status_curbar + 1;

	if (status_curbar < bar) {
		for (; status_curbar < bar; status_curbar++)
			printf("%c", status);

		fflush(stdout);
	}

	status_prev = status;
}

static void status_mark(void)
{
	status_relpos = status_curpos;
}

static void status_stop(void)
{
	if (verbose)
		printf("\n");
}

/*
 * HCA card management
 */

static int vpd_read(struct tvdevice *tvdev);

static int scan_devices(struct pci_access *pacc)
{
	struct pci_dev *pdev;
	struct tvdevice *tvdev;
	int pci_count = 0, pciconf_count = 0;
	unsigned short kernel;
	struct utsname utsname;

	/*
	 * libpci, depending on the kernel it's running under, will create the
	 * PCI device list in the correct order, or backwards. As a result, we
	 * need to create our list appropriately too so the order of the HCAs
	 * matches what the kernel sees the HCA list (ie ib0 is HCA #0, etc)
	 */
	if (uname(&utsname) < 0) {
		fprintf(stderr, "couldn't get uname information, assuming 2.4 kernel\n");
		kernel = 0x0204;
	} else {
		unsigned int major, minor;

		sscanf(utsname.release, "%d.%d", &major, &minor);
		kernel = ((major & 0xff) << 8) | (minor & 0xff);
	}

	/*
	 * Example of how various configurations appear on the PCI bus.
	 * A1 is first HCA, A0 is second HCA.
	 *
	 * Example of both A1 and A0 HCAs in normal mode:
	 *
	 * 07:00:0 5a44 00a0 (MT23108)
	 * 04:00:0 5a44 00a1 (MT23108)
	 * 06:01:0 5a46 00a0 (PCI bridge)
	 * 03:01:0 5a46 00a1 (PCI bridge)
	 *
	 * Example of A1 HCA in normal mode, A0 HCA in recovery mode:
	 *
	 * 06:01:0 5a45 00a0 (PCI conf)
	 * 04:00:0 5a44 00a1 (MT23108)
	 * 03:01:0 5a46 00a1 (PCI bridge)
	 *
	 * Example of A1 HCA in recovery mode, A0 HCA in normal mode:
	 *
	 * 06:00:0 5a44 00a0 (MT23108)
	 * 05:01:0 5a46 00a0 (PCI bridge)
	 * 03:01:0 5a45 00a1 (PCI conf)
	 */
	for (pdev = pacc->devices; pdev; pdev = pdev->next) {
		int i;

		pci_fill_info(pdev, PCI_FILL_IDENT);

		for (i = 0; i < sizeof(pciids) / sizeof(pciids[0]); i++) {
			if (pciids[i].vendor != pdev->vendor_id ||
					pciids[i].device != pdev->device_id)
				continue;

			tvdev = calloc(1, sizeof *tvdev);
			if (!tvdev) {
				fprintf(stderr, "couldn't allocate %zd bytes for device struct\n", sizeof *tvdev);
				return -1;
			}

			tvdev->pdev = pdev;
			tvdev->pciid = &pciids[i];
			tvdev->flags = pciids[i].flags;

			vpd_read(tvdev);

			tvdev->revision = pci_read_word(pdev, PCI_CLASS_REVISION) & 0xff;

			/* According to the documentation, a revision of 0x00 is an A0 */
			if (tvdev->revision == 0x00)
				tvdev->revision = 0xA0;
			else if (tvdev->revision == 0x01)
				tvdev->revision = 0xA1;

			if (kernel <= 0x0204) {
				tvdev->next = tvdevices;
				if (!tvdevices_tail)
					tvdevices_tail = tvdev;
				tvdevices = tvdev;
			} else {
				if (tvdevices_tail)
					tvdevices_tail->next = tvdev;
				else /* first device we found */
					tvdevices = tvdev;
				tvdevices_tail = tvdev;
			}
		}
	}

	for (tvdev = tvdevices; tvdev; tvdev = tvdev->next) {
		if (tvdev->flags & FLAG_RECOVERY)
			tvdev->num = pciconf_count++;
		else
			tvdev->num = pci_count++;
	}

	return pci_count + pciconf_count;
}

static struct tvdevice *find_device(int hca)
{
	struct tvdevice *tvdev;

	for (tvdev = tvdevices; tvdev && hca; tvdev = tvdev->next, hca--)
		;

	return tvdev;
}

/*
 * Low level flash functions (flash device, GPIO)
 */

/*
 * VPD code
 */

static int read_vpd_data(struct pci_dev *pdev, unsigned int vpd_off,
	unsigned int addr, unsigned int *data)
{
	int i;
	unsigned int val;
	struct timeval tstart, tcur;
	unsigned long elapsed;

	gettimeofday(&tstart, NULL);

	i = pci_write_long(pdev, vpd_off, addr << 16);
	do {
		val = pci_read_long(pdev, vpd_off);
		if ((val >> 31) & 0x01) {
			val = htonl(pci_read_long(pdev, vpd_off + 4));
			*data = val;
			return 0;
		}
		gettimeofday(&tcur, NULL);
		elapsed = ((tcur.tv_sec - tstart.tv_sec) * 1000L) +
				((tcur.tv_usec - tstart.tv_usec) / 1000L);
	} while (elapsed < 100);

	if (verbose > 1)
		fprintf(stderr, "read_vpd_data: timed out at off %x\n", addr);

	return 1;
}

static int vpd_exists(struct pci_dev *pdev, int *vpd_off)
{
	int cap_ptr, cap_id;
	int val;

	cap_ptr = pci_read_long(pdev, PCI_CAPABILITY_LIST) & 0xff;
	while (cap_ptr != 0) {
		val = pci_read_long(pdev, cap_ptr);
		cap_id = val & 0xff;
		if (cap_id == PCI_CAP_ID_VPD) {
			*vpd_off = cap_ptr;
			return 1;
		}
		cap_ptr = (val >> 8) & 0xff;
	}

	return 0;
}

static int vpd_read(struct tvdevice *tvdev)
{
	int vpd_off;

	tvdev->vpd_present = 0;

	if (vpd_exists(tvdev->pdev, &vpd_off)) {
		unsigned int *ptr;
		int i, swap_endian = 0;

		ptr = tvdev->vpd.vpd_int;

		if (read_vpd_data(tvdev->pdev, vpd_off, 0, ptr++) != 0)
			return 0;

		/* Check to see if we got the endianess swapped on us */
		if (tvdev->vpd.vpd_char[0] != 0x82 && tvdev->vpd.vpd_char[3] == 0x82)
			swap_endian = 1;

		for (i = 1; i < 64; i++) {
			if (read_vpd_data(tvdev->pdev, vpd_off, (i * 4), ptr++) != 0) {
				if (i < 32)
					return 0;
				else
					break;
			}
		}

		tvdev->vpd_present = 1;

		if (swap_endian) {
			ptr = tvdev->vpd.vpd_int;
			for (i = 0; i < 64; i++) {
				*ptr = swab32(*ptr);
				ptr++;
			}
		}
	}

	return tvdev->vpd_present;
}

/*
 * Read/Write from the configuration space
 */
static unsigned int read_cfg(struct tvdevice *tvdev, unsigned int addr)
{
	unsigned int val;

	switch (tvdev->method) {
	case METHOD_MMAP:
		/*
		 * Work around Hermon errate that allows config space
		 * writes to be passed by reads (so we avoid reading
		 * stale data and eg thinking a SPI command is already
		 * done because the BUSY bit isn't set).
		 */
		if (tvdev->flush_posted_write) {
			*(volatile uint32_t *)(tvdev->bar0 + 0xf0380) = 0;
			while (*(volatile uint32_t *)(tvdev->bar0 + 0xf0380))
				; /* nothing */
			tvdev->flush_posted_write = 0;
		}

		val = ntohl(*(uint32_t *)(tvdev->bar0 + addr));
		break;

	case METHOD_PCI_CFG:
		if (!pci_write_long(tvdev->pdev, 22 * 4, addr)) {
			fprintf(stderr, "read_cfg: pci_write_long failed\n");
			exit(1);
		}

		val = pci_read_long(tvdev->pdev, 23 * 4);
		break;
	default:
		abort();
	}

	return val;
}

static void write_cfg(struct tvdevice *tvdev, unsigned int addr,
	unsigned int data)
{
	switch (tvdev->method) {
	case METHOD_MMAP:
		*(uint32_t *)(tvdev->bar0 + addr) = htonl(data);

		/*
		 * This workaround is supposed to be needed only for
		 * A0 Hermon, but use it workaround for all Hermons
		 * for now:
		 */
		if (tvdev->board && tvdev->board->chip == CHIP_HERMON)
			tvdev->flush_posted_write = 1;
		break;

	case METHOD_PCI_CFG:
		if (!pci_write_long(tvdev->pdev, 22 * 4, addr) ||
				!pci_write_long(tvdev->pdev, 23 * 4, data)) {
			fprintf(stderr, "write_cfg: pci_write_long failed\n");
			exit(1);
		}
		break;
	default:
		abort();
	}
}

#define NUM_SPIS	4

#define FLASH_GW		0xF0400
#define FLASH_ADDR		0xF0404

#define SINAI_FLASH_DATA	0xF0408
#define SINAI_FLASH_CS		0xF0418

#define HERMON_FLASH_DATA	0xF0410

#define SPI_READ		(1 <<  0)
#define BUSY			(1 << 30)

#define SINAI_SPI_NO_DATA	(1 <<  4)
#define SINAI_SPI_NO_ADDR	(1 <<  5)
#define SINAI_SPI_SPECIAL	(1 <<  6)

#define HERMON_SPI_INSTR_PHASE	(1 <<  2)
#define HERMON_SPI_ADDR_PHASE	(1 <<  3)
#define HERMON_SPI_DATA_PHASE	(1 <<  4)
#define HERMON_SPI_FLASH_ENABLE	(1 << 13)

#define WIP				 (1 << 24)
#define WEL				 (1 << 25)

#define FC_SE		0xd8
#define FC_PP		0x02
#define FC_RD		0x03
#define FC_RDSR		0x05
#define FC_RDID		0x9f
#define FC_WREN		0x06

#define SPI_RDID_M25P80 0xffffffff
#define SPI_RDID_M25P16 0x20201500

#define WAIT_INTERVAL 10

static void flash_set_cs(struct tvdevice *tvdev, unsigned int bank)
{
	switch (tvdev->flash_command_set) {
	case CS_SPI_SINAI:
		write_cfg(tvdev, SINAI_FLASH_CS, bank << 30);
		break;
	case CS_SPI_HERMON:
		/* Nothing to do but save off bank */
		break;
	default:
		/* Set the appropriate data bits */
		write_cfg(tvdev, GPIO_DAT + 4, (read_cfg(tvdev, GPIO_DAT + 4) & ~(0x07 << 4)) | (bank << 4));
		break;
	}

	tvdev->flash_bank = bank;
}

uint32_t flash_spi_exec_special(struct tvdevice *tvdev, int op, int logsize)
{
	uint32_t status;

	switch (tvdev->flash_command_set) {
	case CS_SPI_SINAI:
		write_cfg(tvdev, FLASH_ADDR, op << 24);
		write_cfg(tvdev, FLASH_GW, SPI_READ | SINAI_SPI_NO_ADDR |
			  SINAI_SPI_SPECIAL | BUSY | logsize << 8);

		do {
			status = read_cfg(tvdev, FLASH_GW);
		} while (status & BUSY);
		return read_cfg(tvdev, SINAI_FLASH_DATA);

	case CS_SPI_HERMON:
		write_cfg(tvdev, FLASH_GW, SPI_READ | HERMON_SPI_INSTR_PHASE |
			  HERMON_SPI_DATA_PHASE | HERMON_SPI_FLASH_ENABLE |
			  tvdev->flash_bank << 11 | op << 16 | BUSY | logsize << 8);

		do {
			status = read_cfg(tvdev, FLASH_GW);
		} while (status & BUSY);

		return read_cfg(tvdev, HERMON_FLASH_DATA);

	default:
		abort();
	}

}

uint32_t flash_spi_rdsr(struct tvdevice *tvdev)
{
	return flash_spi_exec_special(tvdev, FC_RDSR, 2);
}

/* Discover the devices connected to the Sinai SPI interface */
void spi_discover_serial_proms(struct tvdevice *tvdev)
{
	unsigned int status, sp_num;

	tvdev->flash_num_spis = 0;

	do {
		status = read_cfg(tvdev, FLASH_GW);
	} while (status & BUSY);

	switch (tvdev->board->chip) {
	case CHIP_SINAI:  tvdev->flash_command_set = CS_SPI_SINAI; break;
	case CHIP_HERMON: tvdev->flash_command_set = CS_SPI_HERMON; break;
	default:	  tvdev->flash_command_set = CS_UNKNOWN; break;
	}

	for (sp_num = 0; sp_num < NUM_SPIS; sp_num++) {
		flash_set_cs(tvdev, sp_num);
		if (flash_spi_rdsr(tvdev) >> 24 == 0)
			tvdev->flash_num_spis++;
	}

	/* Discover the type of the first SPI device - it's assumed they are all the same */
	flash_set_cs(tvdev, 0);

	switch (flash_spi_exec_special(tvdev, FC_RDID, 2)) {
	case SPI_RDID_M25P80:
		tvdev->flash_spi_sp_type = SP_ST_M25P80;
		tvdev->flash_size = tvdev->flash_num_spis * 0x100000;
		tvdev->flash_bank_shift = 20;
		break;

	case SPI_RDID_M25P16:
		tvdev->flash_spi_sp_type = SP_ST_M25P16;
		tvdev->flash_size = tvdev->flash_num_spis * 0x200000;
		tvdev->flash_bank_shift = 21;
		break;

	default:
		tvdev->flash_spi_sp_type = SP_UNKNOWN;
		/* Assume some conservative settings */
		tvdev->flash_num_spis = 1;
		tvdev->flash_size = 0x100000;
		tvdev->flash_bank_shift = 20;
	}

	tvdev->flash_sector_sz = 0x10000;	/* 64KB */
}

static void flash_set_bank(struct tvdevice *tvdev, unsigned int addr)
{
	unsigned char bank = addr >> tvdev->flash_bank_shift;

	if (bank == tvdev->flash_bank)
		return;

	flash_set_cs(tvdev, bank);
}

/* read a dword from the flash, address must be 4 bytes aligned */
static unsigned int flash_read(struct tvdevice *tvdev, unsigned int addr)
{
	unsigned int cmd, status;

	flash_set_bank(tvdev, addr);

	switch (tvdev->flash_command_set) {
	case CS_SPI_SINAI:
		write_cfg(tvdev, FLASH_ADDR, (addr & tvdev->flash_bank_mask));
		write_cfg(tvdev, FLASH_GW, SPI_READ | BUSY | (2 << 8));

		do {
			status = read_cfg(tvdev, FLASH_GW);
		} while (status & BUSY);

		return read_cfg(tvdev, SINAI_FLASH_DATA);

	case CS_SPI_HERMON:
		write_cfg(tvdev, FLASH_ADDR, (addr & tvdev->flash_bank_mask));
		write_cfg(tvdev, FLASH_GW, SPI_READ | HERMON_SPI_INSTR_PHASE |
			  HERMON_SPI_ADDR_PHASE | HERMON_SPI_DATA_PHASE |
			  HERMON_SPI_FLASH_ENABLE | FC_RD << 16 |
			  tvdev->flash_bank << 11 | BUSY | (2 << 8));

		do {
			status = read_cfg(tvdev, FLASH_GW);
		} while (status & BUSY);

		return read_cfg(tvdev, HERMON_FLASH_DATA);

	default:
		write_cfg(tvdev, 0xF01A4, (addr & (tvdev->flash_bank_mask & ~3)) | (1 << 29));
		do {
			cmd = read_cfg(tvdev, 0xF01A4);
		} while (cmd & 0xE0000000);

		return read_cfg(tvdev, 0xF01A8);
	}
}

/* read a byte from the flash */
static unsigned char flash_byte_read(struct tvdevice *tvdev, unsigned int addr)
{
	unsigned int data;

	data = flash_read(tvdev, addr & ~3);
	return ((data >> ((3 - (addr & 3)) * 8)) & 0xFF);
}

/* writes a command to the flash */
static void flash_write_cmd(struct tvdevice *tvdev, unsigned int addr,
	unsigned char data)
{
	unsigned int cmd;

	write_cfg(tvdev, 0xF01A8, data << 24);
	write_cfg(tvdev, 0xF01A4, (addr & tvdev->flash_bank_mask) | (2 << 29));
	do {
		cmd = read_cfg(tvdev, 0xF01A4);
	} while (cmd & 0xE0000000);
}

static void flash_chip_reset(struct tvdevice *tvdev)
{
	/* Issue Flash Reset Command */
	switch (tvdev->flash_command_set) {
	case CS_INTEL:
		flash_write_cmd(tvdev, 0x555, 0xFF);
		break;
	case CS_AMD:
		flash_write_cmd(tvdev, 0x555, 0xF0);
		break;
	default:
		break;
	}
}

static void spi_wait_wip(struct tvdevice *tvdev)
{
	unsigned int status;
	unsigned int count = 0;

	do {
		status = read_cfg(tvdev, FLASH_GW);
	} while (status & BUSY);

	do {
		if (++count > 5000)
			usleep(WAIT_INTERVAL);

		status = flash_spi_rdsr(tvdev);
	} while (status & WIP);
}

static void flash_spi_write_enable(struct tvdevice *tvdev)
{
	uint32_t status;

	switch (tvdev->flash_command_set) {
	case CS_SPI_SINAI:
		write_cfg(tvdev, FLASH_ADDR, FC_WREN << 24);
		write_cfg(tvdev, FLASH_GW, SINAI_SPI_NO_ADDR | SINAI_SPI_NO_DATA |
			  SINAI_SPI_SPECIAL | BUSY);
		break;

	case CS_SPI_HERMON:
		write_cfg(tvdev, FLASH_GW, HERMON_SPI_INSTR_PHASE |
			  HERMON_SPI_FLASH_ENABLE | FC_WREN << 16 |
			  tvdev->flash_bank << 11 | BUSY);
		break;

	default:
		abort();
	}

	do {
		status = read_cfg(tvdev, FLASH_GW);
	} while (status & BUSY);
}

static void flash_erase_sector(struct tvdevice *tvdev, unsigned int addr)
{
	unsigned int status;

	flash_set_bank(tvdev, addr);

	/* Issue Flash Sector Erase Command */
	switch (tvdev->flash_command_set) {
	case CS_SPI_SINAI:
		flash_spi_write_enable(tvdev);

		/* Issue Sector Erase */
		write_cfg(tvdev, FLASH_ADDR, (FC_SE << 24) | (addr & tvdev->flash_bank_mask));
		write_cfg(tvdev, FLASH_GW, SINAI_SPI_NO_DATA | SINAI_SPI_SPECIAL | BUSY);

		/* Wait for sector erase completion */
		spi_wait_wip(tvdev);
		break;

	case CS_SPI_HERMON:
		flash_spi_write_enable(tvdev);

		/* Issue Sector Erase*/
		write_cfg(tvdev, FLASH_ADDR, addr & tvdev->flash_bank_mask);
		write_cfg(tvdev, FLASH_GW, HERMON_SPI_INSTR_PHASE |
			  HERMON_SPI_ADDR_PHASE | HERMON_SPI_FLASH_ENABLE |
			  FC_SE << 16 | tvdev->flash_bank << 11 | BUSY);

		/* Wait for sector erase completion */
		spi_wait_wip(tvdev);
		break;

	case CS_INTEL:
		flash_write_cmd(tvdev, addr & tvdev->flash_bank_mask, 0x20); /* Erase */
		flash_write_cmd(tvdev, addr & tvdev->flash_bank_mask, 0xD0); /* Confirm */

		/* Wait for sector erase completion */
		do {
			usleep(WAIT_INTERVAL);
			status = flash_read(tvdev, addr);
		} while (!(status & 0x80));
		break;

	case CS_AMD:
		flash_write_cmd(tvdev, 0x555, 0xAA);
		flash_write_cmd(tvdev, 0x2AA, 0x55);
		flash_write_cmd(tvdev, 0x555, 0x80);
		flash_write_cmd(tvdev, 0x555, 0xAA);
		flash_write_cmd(tvdev, 0x2AA, 0x55);
		flash_write_cmd(tvdev, addr & tvdev->flash_bank_mask, 0x30);

		/* Wait for sector erase completion */
		do {
			usleep(WAIT_INTERVAL);
			status = flash_read(tvdev, addr);
		} while (status != 0xFFFFFFFF);
		break;
	default:
		break;
	}

	flash_chip_reset(tvdev);
}

static int log2up(unsigned long in)
{
	unsigned int i;

	for (i = 0; i < 32; i++) {
		if (in <= (unsigned long)(1 << i))
			break;
	}

	return i;
}

static void flash_spi_write_block(struct tvdevice *tvdev, unsigned int addr,
				  unsigned char *sector, unsigned int len)
{
	int i;

	flash_spi_write_enable(tvdev);

	switch (tvdev->flash_command_set) {
	case CS_SPI_SINAI:
		write_cfg(tvdev, FLASH_ADDR, (FC_PP << 24) | (addr & tvdev->flash_bank_mask));

		for (i = 0; i < len; i += 4) {
			uint32_t dword = *((uint32_t *)(sector + i));

			write_cfg(tvdev, SINAI_FLASH_DATA + i, be32_to_cpu(dword));
		}

		write_cfg(tvdev, FLASH_GW, SINAI_SPI_SPECIAL | BUSY | (log2up(len) << 8));
		break;

	case CS_SPI_HERMON:
		write_cfg(tvdev, FLASH_ADDR, addr & tvdev->flash_bank_mask);

		for (i = 0; i < len; i += 4) {
			uint32_t dword = *((uint32_t *)(sector + i));

			write_cfg(tvdev, HERMON_FLASH_DATA + i, be32_to_cpu(dword));
		}

		write_cfg(tvdev, FLASH_GW, HERMON_SPI_INSTR_PHASE |
			  HERMON_SPI_ADDR_PHASE | HERMON_SPI_DATA_PHASE |
			  HERMON_SPI_FLASH_ENABLE | FC_PP << 16 |
			  tvdev->flash_bank << 11 | BUSY | (log2up(len) << 8));
		break;

	default:
		abort();
	}

	/* Wait for erase to complete */
	spi_wait_wip(tvdev);
}

static void flash_write_byte(struct tvdevice *tvdev, unsigned int addr,
			     unsigned char data)
{
	unsigned int status;

	switch (tvdev->flash_command_set) {
	case CS_INTEL:
		flash_write_cmd(tvdev, addr & tvdev->flash_bank_mask, 0x40);
		flash_write_cmd(tvdev, addr & tvdev->flash_bank_mask, data);

		/* Wait for the Byte Program to Complete (Verify Byte Program) */
		do {
			status = flash_read(tvdev, addr & ~3);
		} while (!(status & 0x80));
		break;
	case CS_AMD:
		/* Issue Byte Program Command */
		flash_write_cmd(tvdev, 0x555, 0xAA);
		flash_write_cmd(tvdev, 0x2AA, 0x55);
		flash_write_cmd(tvdev, 0x555, 0xA0);
		flash_write_cmd(tvdev, addr & tvdev->flash_bank_mask, data);

		/* Wait for the Byte Program to Complete (Verify Byte Program) */
		do {
			status = flash_read(tvdev, addr & ~3);
		} while (data != ((status >> ((3 - (addr & 3)) * 8)) & 0xFF));
		break;
	default:
		/* Not possible, but compiler complains */
		break;
	}
}

static int grab_gpio(struct tvdevice *tvdev)
{
	unsigned long elapsed_msec;
	struct timeval tstart, tcur;

	gettimeofday(&tstart, NULL);

	/* Grab the semaphore first */
	while (read_cfg(tvdev, 0xF03FC) == 1) {
		gettimeofday(&tcur, NULL);
		elapsed_msec = (tcur.tv_sec - tstart.tv_sec) * 1000L +
			(tcur.tv_usec - tstart.tv_usec) / 1000L;
		if (elapsed_msec > 10000) {
			if (force) {
				write_cfg(tvdev, 0xF03FC, 0);
				fprintf(stderr, "Forcing release of flash semaphore.\n");
				exit(1);
				break;
			}
			else
				return 1;
		}
	}

	write_cfg(tvdev, 0xF03FC, 1);

	/* Now copy out the values */
	tvdev->gpio_data[0] = read_cfg(tvdev, GPIO_DAT) & 0xFF;
	tvdev->gpio_data[1] = read_cfg(tvdev, GPIO_DAT + 4);

	tvdev->gpio_direction[0] = read_cfg(tvdev, GPIO_DIR) & 0xFF;
	tvdev->gpio_direction[1] = read_cfg(tvdev, GPIO_DIR + 4);

	tvdev->gpio_polarity[0] = read_cfg(tvdev, GPIO_POL) & 0xFF;
	tvdev->gpio_polarity[1] = read_cfg(tvdev, GPIO_POL + 4);

	tvdev->gpio_output_mode[0] = read_cfg(tvdev, GPIO_MOD) & 0xFF;
	tvdev->gpio_output_mode[1] = read_cfg(tvdev, GPIO_MOD + 4);

	return 0;
}

static void release_gpio(struct tvdevice *tvdev)
{
	if (tvdev->board->chip == CHIP_SINAI) {
		flash_set_bank(tvdev, 0);

		/* Unlock GPIO */
		write_cfg(tvdev, GPIO_LOCK, 0xAAAA);
	}

	/* Write back the saved values */
	write_cfg(tvdev, GPIO_DAT, (read_cfg(tvdev, GPIO_DAT) & ~0xFF) | tvdev->gpio_data[0]);
	write_cfg(tvdev, GPIO_DAT + 4, tvdev->gpio_data[1]);

	write_cfg(tvdev, GPIO_DIR, (read_cfg(tvdev, GPIO_DIR) & ~0xFF) | tvdev->gpio_direction[0]);
	write_cfg(tvdev, GPIO_DIR + 4, tvdev->gpio_direction[1]);

	write_cfg(tvdev, GPIO_POL, (read_cfg(tvdev, GPIO_POL) & ~0xFF) | tvdev->gpio_polarity[0]);
	write_cfg(tvdev, GPIO_POL + 4, tvdev->gpio_polarity[1]);

	write_cfg(tvdev, GPIO_MOD, (read_cfg(tvdev, GPIO_MOD) & ~0xFF) | tvdev->gpio_output_mode[0]);
	write_cfg(tvdev, GPIO_MOD + 4, tvdev->gpio_output_mode[1]);

	/* Release the semaphore */
	write_cfg(tvdev, 0xF03FC, 0);
}

static struct board *vpd_identify(struct tvdevice *tvdev)
{
	char str[MAX_STR];
	int i;

	if (tvdev->vpd_present) {
		if (tvdev->vpd.vpd_char[0] == 0x82) {
			unsigned short len16;
			char *p = (char *)&tvdev->vpd.vpd_char[3];

			memcpy(&len16, tvdev->vpd.vpd_char + 1, sizeof(len16));
			len16 = le16_to_cpu(len16);

			/* Skip leading whitespace */
			for (; len16 > 0 && isspace(*p); len16--, p++)
				;

			/* Make sure we don't copy too much */
			if (len16 > sizeof(str))
				len16 = sizeof(str);

			memcpy(str, p, len16);
			str[len16] = 0;

			/* Strip off any trailing whitespace */
			for (len16--; len16 > 0 && isspace(str[len16]); len16--)
				str[len16] = 0;
		} else
			fprintf(stderr, " \nError. String Tag not present (found tag %02x instead)\n",
				tvdev->vpd.vpd_char[0]);
	} else
		str[0] = 0;

	for (i = 0; i < sizeof(board_vpds) / sizeof(board_vpds[0]); i++) {
		if (strcasecmp(str, board_vpds[i].str) == 0)
			return board_vpds[i].board;
	}

	return NULL;
}

static struct board *arbel_identify(struct tvdevice *tvdev)
{
	struct board *board;

	board = vpd_identify(tvdev);
	if (board)
		return board;

	return &genarbel;
}

static struct board *sinai_identify(struct tvdevice *tvdev)
{
	struct board *board;

	board = vpd_identify(tvdev);
	if (board)
		return board;

	return &gensinai;
}

static struct board *hermon_identify(struct tvdevice *tvdev)
{
	struct board *board;

	board = vpd_identify(tvdev);
	if (board)
		return board;

	return &genhermon;
}

static struct board *tavor_identify(struct tvdevice *tvdev)
{
	struct board *board;

	/* Detect the difference between a Jaguar and Cougar via the GPIO pins */

	/* Set GPIO pin 12 to input and check the value */
	write_cfg(tvdev, GPIO_DIR + 4, read_cfg(tvdev, GPIO_DIR + 4) & ~(1 << 12));

	usleep(10);

	/* Check if board is a Jaguar by looking at GPIO pin 12 */
	if (!(read_cfg(tvdev, GPIO_DAT + 4) & (1 << 12)))
		return &jaguar;

	/* Check the rest by VPD */
	board = vpd_identify(tvdev);
	if (board)
		return board;

	/*
	 * FIXME: We shouldn't assume Cougar if the VPD is corrupt. The very
	 * fact that there is a VPD probably means it's not a Cougar board.
	 * It's only a Cougar if there is no VPD at all.
	 */

	return &cougar;
}

void convert_old_topspin_vsd(union vsd *vsd)
{
	struct topspin_vsd_old old_vsd;
	/* VSD Ascii prefix - prevents flint/mstflint from printing garbage */
	static const char vsd_prefix_str[VSD_PREFIX_LENGTH] =
		{ 'c', 's', 'i', 'C', '\0', '\0', '\0', 'o'};

	if (verbose)
		printf("  converting Topspin VSD format\n");

	/* Make a copy of the old VSD */
	memcpy(&old_vsd, vsd, sizeof old_vsd);

	/* Setup the new VSD based on the information from the old VSD */
	memset(vsd->raw, 0, sizeof(vsd->raw));

	memcpy(vsd->topspin.vsd_prefix, vsd_prefix_str, sizeof vsd_prefix_str);

	memcpy((void *) &vsd->topspin.signature1, &old_vsd, 
	       offsetof(struct topspin_vsd_old, unused));

	memcpy((void *) &vsd->topspin.boot_pxe_secs, &old_vsd.boot_pxe_secs,
	       sizeof old_vsd - offsetof(struct topspin_vsd_old, boot_pxe_secs));
}

static void fixup_vsd(union vsd *vsd)
{
	struct topspin_vsd *tsvsd;

	if (be16_to_cpu(vsd->topspin_old.signature1) == VSD_SIGNATURE_TOPSPIN &&
	    be16_to_cpu(vsd->topspin_old.signature2) == VSD_SIGNATURE_TOPSPIN)
		convert_old_topspin_vsd(vsd);

	if (be16_to_cpu(vsd->topspin.signature1) == VSD_SIGNATURE_TOPSPIN &&
	    be16_to_cpu(vsd->topspin.signature2) == VSD_SIGNATURE_TOPSPIN)
		/* Not Topspin VSD */
		return;

	tsvsd = &vsd->topspin;

	if (!(tsvsd->flags & cpu_to_be32(VSD_FLAG_ALIGN_FIXED))) {
		/*
		 * Some images have the hw_label field misaligned by one byte.
		 * If this is an old image (the flag isn't set), then move
		 * it back one character
		 */
		memmove(tsvsd->hw_label,
			tsvsd->hw_label + 1,
			sizeof(tsvsd->hw_label) - 1);
		tsvsd->flags |= cpu_to_be32(VSD_FLAG_ALIGN_FIXED);
	}

	if (!(tsvsd->flags & cpu_to_be32(VSD_FLAG_NEW_BUILD_NUM))) {
		/*
		 * build_num used to be an 8 bit field, but our build numbers
		 * quickly outgrew that limitation, so expand to 16 bits and
		 * a new location
		 */
		tsvsd->build_num = tsvsd->old_build_num;
		tsvsd->flags |= cpu_to_be32(VSD_FLAG_NEW_BUILD_NUM);
	}

	if ((tsvsd->flags & cpu_to_be32(VSD_FLAG_BOOT_OPTIONS)) &&
			((tsvsd->boot_version < 1) || (tsvsd->boot_version > BOOT_VERSION))) {
		/* Invalid, reset to defaults */
		tsvsd->boot_version = BOOT_VERSION;
		tsvsd->boot_port = 0;
		tsvsd->boot_ioc_num = 0;
		memset(tsvsd->boot_dgid, 0, sizeof(tsvsd->boot_dgid));
		memset(tsvsd->boot_service_name, 0, sizeof(tsvsd->boot_service_name));
		tsvsd->boot_pxe_secs = BOOT_PXE_SECS_DEFAULT;

		tsvsd->flags &= ~cpu_to_be32(VSD_FLAG_BOOT_OPTIONS_MASK);
		tsvsd->flags |= cpu_to_be32(VSD_FLAG_BOOT_ENABLE_PORT_1 |
					VSD_FLAG_BOOT_ENABLE_PORT_2 |
					VSD_FLAG_BOOT_ENABLE_SCAN |
					VSD_FLAG_BOOT_TYPE_WELL_KNOWN |
					VSD_FLAG_BOOT_GID_BIG);
	}

	if (!(tsvsd->flags & cpu_to_be32(VSD_FLAG_BOOT_GID_BIG))) {
		uint8_t dword[4];
		int i;

		for (i = 0; i < sizeof(tsvsd->boot_dgid); i += 4) {
			memcpy(dword, &tsvsd->boot_dgid[i], 4);
			tsvsd->boot_dgid[i + 0] = dword[3];
			tsvsd->boot_dgid[i + 1] = dword[2];
			tsvsd->boot_dgid[i + 2] = dword[1];
			tsvsd->boot_dgid[i + 3] = dword[0];
		}

		tsvsd->flags |= cpu_to_be32(VSD_FLAG_BOOT_GID_BIG);
	}

	if (tsvsd->boot_version < 2) {
		uint8_t dword[4];
		int i;

		tsvsd->boot_version = BOOT_VERSION;
		tsvsd->boot_pxe_secs = BOOT_PXE_SECS_DEFAULT;

		for (i = 0; i < sizeof(tsvsd->boot_service_name); i += 4) {
			memcpy(dword, &tsvsd->boot_service_name[i], 4);
			tsvsd->boot_service_name[i + 0] = dword[3];
			tsvsd->boot_service_name[i + 1] = dword[2];
			tsvsd->boot_service_name[i + 2] = dword[1];
			tsvsd->boot_service_name[i + 3] = dword[0];
		}
	}
}

static void parse_image_info(uint8_t *buf, union vsd **vsd)
{
	int off = 0;
	int size;

	while (buf[off] != IMAGE_INFO_END) {
		size = ntohl(*(uint32_t *) &buf[off]) & 0xffffff;

		switch (buf[off]) {
		case IMAGE_INFO_VSD:
			*vsd = (union vsd *) (buf + off + 4);
			break;
		}

		off += size + 4;
	}
}

/*
 * Mid level flash functions (firmware image)
 */

static int flash_image_read_from_file(char *fname,
				      unsigned char **isbuf, unsigned int *isbufsz,
				      unsigned char **psbuf, unsigned int *psbufsz,
				      unsigned char **ibuf, unsigned int *ibufsz,
				      union vsd **vsd)
{
	unsigned int sector_sz = 0;
	uint32_t signature;
	uint32_t ii_off;
	unsigned char *buf;
	struct stat imgstat;
	FILE *fimg;
	int i;

	/* Open and read image files */
	fimg = fopen(fname, "r");
	if (fimg == NULL) {
		fprintf(stderr, "tvflash: error opening file %s!\n", fname);
		return 1;
	}

	/* Get the size of the image */
	if (fstat(fileno(fimg), &imgstat) < 0) {
		fprintf(stderr, "tvflash: cannot do stat on firmware image!\n");
		return 1;
	}

	buf = malloc(imgstat.st_size);
	if (!buf) {
		fprintf(stderr, "tvflash: cannot allocate memory for the image buffer!\n");
		return 1;
	}

	if (fread(buf, imgstat.st_size, 1, fimg) != 1) {
		fprintf(stderr, "tvflash: error reading file %s!\n", fname);
		return 1;
	}
	fclose(fimg);

	*vsd = NULL;

	/* Check the signature on the IS */
	signature = be32_to_cpu(*(uint32_t *)(buf + 0x24));
	if (signature == 0x5a445a44) {
		unsigned int is_sz;

		is_sz = be32_to_cpu(*(uint32_t *)(buf + 0x2c));

		if (is_sz > 4 && is_sz < 1048576) {
			uint16_t crc, img_crc;

			/* is_sz is a count of dwords */
			is_sz = is_sz * 4;
			img_crc = be32_to_cpu(*(uint32_t *)(buf + 0x28 + is_sz + 12));

			/* Verify the CRC of the IS */
			crc = flash_crc16(buf + 0x28, is_sz + 16 - 4);
			if (crc == img_crc) {
				unsigned int sector_sz_ptr, log2_sector_sz;

				/* Then grab the sector size */
				sector_sz_ptr	= be16_to_cpu(*(uint16_t *)(buf + 0x16));
				log2_sector_sz = be16_to_cpu(*(uint16_t *)(buf + sector_sz_ptr + 0x32));

				/*
				 * Do some sanity checking of the result.
				 * Anything less than 4KB or more than
				 * 1MB is suspicious and thrown out
				 */
				if (log2_sector_sz >= 12 && log2_sector_sz <= 20)
					sector_sz = 1 << log2_sector_sz;
			}
		}
	}

	/* Check for PPS */
	if (sector_sz &&
	    be32_to_cpu(*(uint32_t *)(buf + sector_sz + 8)) == 0x5a445a44) {
		/* Failsafe firmware image */
		*isbuf = buf;
		*isbufsz = sector_sz;
		*psbuf = buf + sector_sz;
		*psbufsz = sector_sz;
		*ibuf = buf + be32_to_cpu(*(uint32_t *)(buf + sector_sz + 0));
		*ibufsz = be32_to_cpu(*(uint32_t *)(buf + sector_sz + 4));

		*vsd = (union vsd *) (buf + sector_sz + 0x20);
	} else {
		/* Non failsafe firmware image */
		*isbuf = *psbuf = NULL;
		*isbufsz = *psbufsz = 0;
		*ibuf = buf;
		*ibufsz = imgstat.st_size;

		for (i = 0; i < sizeof hermon_fw_sig / sizeof hermon_fw_sig[0]; ++i)
			if (ntohl(((uint32_t *) buf)[i]) != hermon_fw_sig[i])
				break;

		if (i == sizeof hermon_fw_sig / sizeof hermon_fw_sig[0]) {
			ii_off = ntohl(*(uint32_t *) (buf + HERMON_FLASH_IMAGE_INFO_OFF));

			i = (ii_off & 0xff) + (ii_off >> 8 & 0xff) +
				(ii_off >> 16 & 0xff) + (ii_off >> 24);
			if (i & 0xff)
				printf("  WARNING: firmware has image info pointer %08x "
				       "with checksum %x != 0\n",
				       ii_off, i & 0xff);
			else
				parse_image_info(buf + (ii_off & 0xffffff), vsd);
		}
	}

	if (*vsd)
		fixup_vsd(*vsd);

	return 0;
}

static int validate_xps(struct image *image, unsigned char *psbuf)
{
	/* Check for signature */
	image->valid = (be32_to_cpu(*(uint32_t *)(psbuf + 0x8)) == 0x5a445a44);
	if (image->valid) {
		image->addr = be32_to_cpu(*(uint32_t *)(psbuf + 0x0));
		image->size = be32_to_cpu(*(uint32_t *)(psbuf + 0x4));
		/* Check CRC for xPS */
		image->valid = (flash_crc16(psbuf, 0x104) == be16_to_cpu(*(uint16_t *)(psbuf + 0x106)));

		memcpy(&image->vsd, psbuf + 0x20, sizeof image->vsd);

		fixup_vsd(&image->vsd);
	}

	return image->valid;
}

static int tavor_flash_check_failsafe(struct tvdevice *tvdev)
{
	unsigned int sector_sz = 0;
	struct tavor_failsafe *failsafe = &tvdev->fw.tavor_failsafe;
	uint32_t signature;
	unsigned char *psbuf;
	int i;

	/* Check the signature on the IS */
	signature = flash_read(tvdev, 0x24);
	if (signature == 0x5a445a44) {
		unsigned char *is;
		unsigned int is_sz;

		is_sz = flash_read(tvdev, 0x2c);
		if (is_sz > 4 && is_sz < 1048576) {
			uint16_t crc, img_crc;

			/* is_sz is a count of dwords */
			is_sz = is_sz * 4;
			is = malloc(is_sz + 16);
			if (!is) {
				fprintf(stderr, "couldn't allocate %d bytes for sector buffer\n",
					is_sz);
				exit(1);
			}

			for (i = 0; i < is_sz + 16; i += 4)
				*(uint32_t *)(is + i) = be32_to_cpu(flash_read(tvdev, 0x28 + i));

			img_crc = flash_read(tvdev, 0x28 + is_sz + 12);

			/* Verify the CRC of the IS */
			crc = flash_crc16(is, is_sz + 16 - 4);
			if (crc == img_crc) {
				unsigned int sector_sz_ptr, log2_sector_sz;

				/* Then grab the sector size */
				sector_sz_ptr	= flash_read(tvdev, 0x14) & 0xffff;
				log2_sector_sz = flash_read(tvdev, sector_sz_ptr + 0x30) & 0xffff;

				/*
				 * Do some sanity checking of the result.
				 * Anything less than 4KB or more than
				 * 1MB is suspicious and thrown out
				 */
				if (log2_sector_sz >= 12 && log2_sector_sz <= 20)
					sector_sz = 1 << log2_sector_sz;
			}

			free(is);
		}
	}

	if (!sector_sz)
		return 0;

	psbuf = malloc(sector_sz);
	if (!psbuf) {
		fprintf(stderr, "couldn't allocate temp buffer for PPS/SPS (size = %d)\n",
			sector_sz);
		exit(1);
	}

	/* Check both PPS and SPS for valid signatures */

	/* Read out the PPS */
	for (i = 0; i < sector_sz; i += 4)
		*(uint32_t *)(psbuf + i) = be32_to_cpu(flash_read(tvdev, sector_sz + i));

	validate_xps(&failsafe->images[0], psbuf);

	/* Read out the SPS */
	for (i = 0; i < sector_sz; i += 4)
		*(uint32_t *)(psbuf + i) = be32_to_cpu(flash_read(tvdev, sector_sz * 2 + i));

	validate_xps(&failsafe->images[1], psbuf);

	/*
	 * Last sanity check. We can't say that the IS is programmed correctly
	 * if we don't see a valid PPS or SPS.
	 */
	failsafe->valid = (failsafe->images[0].valid || failsafe->images[1].valid);

	free(psbuf);

	return failsafe->valid;
}

static int hermon_flash_get_info(struct tvdevice *tvdev)
{
	struct hermon_fw_info *fw_info = &tvdev->fw.hermon_fw_info;
	static const int flash_base[] = {
		0x0 , 0x10000, 0x20000, 0x40000, 0x80000, 0x100000
	};
	int i, j;
	uint32_t chunk_info, ii_off;
	uint32_t *ii, ii_size;

	/* First look for a valid FW image */
	for (i = 0; i < sizeof flash_base / sizeof flash_base[0]; ++i) {
		for (j = 0; j < sizeof hermon_fw_sig / sizeof hermon_fw_sig[0]; ++j)
			if (flash_read(tvdev, flash_base[i] + j * 4) !=
			    hermon_fw_sig[j])
				break;

		if (j == sizeof hermon_fw_sig / sizeof hermon_fw_sig[0])
			goto found;
	}

	if (verbose > 1)
		printf("  No valid ConnectX FW found.\n");

	return 0;

found:
	fw_info->base = flash_base[i];

	chunk_info = flash_read(tvdev, fw_info->base + 0x28);
	i = (chunk_info & 0xff) + (chunk_info >> 8 & 0xff) +
		(chunk_info >> 16 & 0xff) + (chunk_info >> 24);
	if (i & 0xff)
		printf("  WARNING: firmware image at %x has chunk info word %08x "
		       "with checksum %x != 0\n",
		       fw_info->base, chunk_info, i & 0xff);

	fw_info->failsafe   = !!(chunk_info & (1 << 3));
	fw_info->chunk_size = 0x10000 << (chunk_info & 7);

	ii_off = flash_read(tvdev, fw_info->base + HERMON_FLASH_IMAGE_INFO_OFF);
	i = (ii_off & 0xff) + (ii_off >> 8 & 0xff) +
		(ii_off >> 16 & 0xff) + (ii_off >> 24);
	if (i & 0xff)
		printf("  WARNING: firmware image at %x has image info pointer %08x "
		       "with checksum %x != 0\n",
		       fw_info->base, ii_off, i & 0xff);
	ii_off &= 0xffffff;

	memset(&fw_info->vsd, 0, sizeof fw_info->vsd);

	ii_size = 256;
	ii = malloc(ii_size);
	if (!ii)
		goto out;

	i = 0;
	while (1) {
		ii[i] = flash_read(tvdev, fw_info->base + ii_off + i * 4);
		if (ii[i] >> 24 == IMAGE_INFO_END)
			break;

		if ((i + 1) * sizeof (uint32_t) + (ii[i] & 0xfffffff) >= ii_size) {
			ii_size *= 2;
			ii = realloc(ii, ii_size);
			if (!ii)
				goto out;
		}

		for (j = 1; j <= (ii[i] & 0xffffff) / 4; ++j)
			ii[i + j] = be32_to_cpu(flash_read(tvdev, fw_info->base + ii_off + (i + j) * 4));

		if (ii[i] >> 24 == IMAGE_INFO_VSD) {
			memcpy(&fw_info->vsd, &ii[i + 1], ii[i] & 0xffffff);
			fixup_vsd(&fw_info->vsd);
		}

		i += j;
	}

out:
	return fw_info->failsafe;
}

static int flash_get_info(struct tvdevice *tvdev)
{
	switch (tvdev->board->chip) {
	case CHIP_TAVOR:
	case CHIP_ARBEL:
	case CHIP_SINAI:
		return tavor_flash_check_failsafe(tvdev);

	case CHIP_HERMON:
		return hermon_flash_get_info(tvdev);

	default:
		return 0;
	}
}

static int guid_offset(struct tvdevice *tvdev)
{
	switch (tvdev->board->chip) {
	case CHIP_TAVOR:
	case CHIP_ARBEL:
	case CHIP_SINAI:
		return TAVOR_FLASH_GUID_OFF;

	case CHIP_HERMON:
		return HERMON_FLASH_GUID_OFF;

	default:
		return 0;
	}
}

static unsigned fw_image_addr(struct tvdevice *tvdev)
{
	switch (tvdev->board->chip) {
	case CHIP_TAVOR:
	case CHIP_ARBEL:
	case CHIP_SINAI:
		if (tvdev->fw.tavor_failsafe.valid) {
			if (tvdev->fw.tavor_failsafe.images[0].valid)
				return tvdev->fw.tavor_failsafe.images[0].addr;
			else
				return tvdev->fw.tavor_failsafe.images[1].addr;
		} else
			return 0;

	case CHIP_HERMON:
		return tvdev->fw.hermon_fw_info.base;

	default:
		return 0;
	}
}

/* Get the GUIDs currently programmed in the flash */
static int flash_get_curr_guids(struct tvdevice *tvdev, unsigned char *guid_node,
	unsigned char *guid_port1, unsigned char *guid_port2)
{
	unsigned int imageaddr = 0;
	unsigned char *up;
	unsigned int paddr;
	int bcnt;

	imageaddr = fw_image_addr(tvdev);

	paddr = flash_read(tvdev, imageaddr + guid_offset(tvdev)) + imageaddr;
	if (paddr > tvdev->flash_size) {
		fprintf(stderr, "GUID offset (0x%X) is larger than flash size (0x%X)!\n",
			paddr, tvdev->flash_size);
		return 1;
	}

	/* Read the Node GUID */
	up = guid_node;
	for (bcnt = 0; bcnt < GUID_LEN; bcnt++)
		*up++ = flash_byte_read(tvdev, paddr++);

	/* Read the Port1 GUID */
	up = guid_port1;
	for (bcnt = 0; bcnt < GUID_LEN; bcnt++)
		*up++ = flash_byte_read(tvdev, paddr++);

	/* Read the Port2 GUID */
	up = guid_port2;
	for (bcnt = 0; bcnt < GUID_LEN; bcnt++)
		*up++ = flash_byte_read(tvdev, paddr++);

	return 0;
}

static int guid_sect_crc_len(struct tvdevice *tvdev)
{
	switch (tvdev->board->chip) {
	case CHIP_TAVOR:
	case CHIP_ARBEL:
	case CHIP_SINAI:
		return TAVOR_FLASH_GUID_SECT_CRC_LEN;

	case CHIP_HERMON:
		return HERMON_FLASH_GUID_SECT_CRC_LEN;

	default:
		return 0;
	}
}

static int guid_sect_crc_offset(struct tvdevice *tvdev)
{
	switch (tvdev->board->chip) {
	case CHIP_TAVOR:
	case CHIP_ARBEL:
	case CHIP_SINAI:
		return TAVOR_FLASH_GUID_SECT_CRC_OFFSET;

	case CHIP_HERMON:
		return HERMON_FLASH_GUID_SECT_CRC_OFFSET;

	default:
		return 0;
	}
}

/* GUID update */
static int flash_guids_update(struct tvdevice *tvdev, unsigned char *ibuf,
			      unsigned int ibufsz, unsigned char *guid_node,
			      unsigned char *guid_port1, unsigned char *guid_port2)
{
	unsigned int paddr;
	void *gsp;

	/* Endianess problems suck */
	paddr = be32_to_cpu(*(uint32_t *)(ibuf + guid_offset(tvdev)));
	if (paddr > ibufsz) {
		fprintf(stderr, "GUID pointer (0x%X) is larger than image size (0x%X)!\n",
			paddr, ibufsz);
		return 1;
	}

	/* Blend in the new GUIDs */
	memcpy(&ibuf[paddr], guid_node, GUID_LEN);
	memcpy(&ibuf[paddr + GUID_LEN], guid_port1, GUID_LEN);
	if (tvdev->board->num_ports > 1)
		memcpy(&ibuf[paddr + GUID_LEN * 2], guid_port2, GUID_LEN);

	printf("New Node  GUID = %02x%02x%02x%02x%02x%02x%02x%02x\n",
		guid_node[0], guid_node[1], guid_node[2], guid_node[3],
		guid_node[4], guid_node[5], guid_node[6], guid_node[7]);
	printf("New Port1 GUID = %02x%02x%02x%02x%02x%02x%02x%02x\n",
		guid_port1[0], guid_port1[1], guid_port1[2], guid_port1[3],
		guid_port1[4], guid_port1[5], guid_port1[6], guid_port1[7]);
	if (tvdev->board->num_ports > 1)
		printf("New Port2 GUID = %02x%02x%02x%02x%02x%02x%02x%02x\n",
			guid_port2[0], guid_port2[1], guid_port2[2], guid_port2[3],
			guid_port2[4], guid_port2[5], guid_port2[6], guid_port2[7]);

	/* Get pointer to the GUID section in the image buffer */
	gsp = ibuf + paddr - FLASH_GUID_SECT_FW_RESERVED;

	/* Recalculate GUID section CRC-16 - FW_Reserver_2 and CRC itself excluded */
	*(uint16_t *) (gsp + guid_sect_crc_offset(tvdev)) =
		cpu_to_be16(flash_crc16(gsp, guid_sect_crc_len(tvdev)));

	return 0;
}

/*
 * High level flash functions (information, read/write firmware images)
 */

/* Identify */
static int create_ver_str(union vsd *vsd, char *buf, unsigned int buflen)
{
	if (vsd->topspin.revision_ver) {
		if (vsd->topspin.revision_ver & 0x80)
			snprintf(buf, buflen, "%d.%d.%03d-rc%d",
				vsd->topspin.major_ver,
				vsd->topspin.minor_ver,
				be16_to_cpu(vsd->topspin.micro_ver),
				vsd->topspin.revision_ver & ~(-1 << 7));
		else
			snprintf(buf, buflen, "%d.%d.%03d_%d",
				vsd->topspin.major_ver,
				vsd->topspin.minor_ver,
				be16_to_cpu(vsd->topspin.micro_ver),
				vsd->topspin.revision_ver);
	} else
		snprintf(buf, buflen, "%d.%d.%03d",
			vsd->topspin.major_ver,
			vsd->topspin.minor_ver,
			be16_to_cpu(vsd->topspin.micro_ver));

	return strlen(buf);
}

static void print_vpd_info(struct tvdevice *tvdev)
{
	char str[MAX_STR];
	unsigned short i = 0, rlen;
	unsigned char *ptr = tvdev->vpd.vpd_char;
	unsigned short len16;
	unsigned char len8 = 0, cksum = 0;
	int len = 0;
	int pn = 0, ec = 0, sn = 0, fp = 0, dc = 0, rv = 0;

	printf("\n  Vital Product Data\n");

	/* First field is string tag */
	if (ptr[0] != 0x82) {
		fprintf(stderr, "   Error. String Tag not present (found tag %02x instead)\n",
			ptr[0]);
		return;
	}

	/* print out String Tag */
	memcpy(&len16, ptr + 1, sizeof(len16));
	len16 = le16_to_cpu(len16);
	memcpy(str, &ptr[3], len16);
	str[len16] = 0;
	printf("    Product Name: %s\n", str);

	/* string len + 2 bytes of length field + 1 byte of tag */
	ptr = &ptr[len16 + 3];

	/* check for Read only tag */
	if (ptr[0] != 0x90) {
		fprintf(stderr, "    Error. R Tag not present\n");
		return;
	}
	memcpy(&rlen, ptr + 1, sizeof(rlen));	/* Read only resource len */
	rlen = le16_to_cpu(rlen);

	/* parse and print out each field, till we hit end tag */
	ptr += 3;		 /* 1 byte rtag + 2 bytes length field */
	while (i < rlen) {
		if ((ptr[0] == 'P') && (ptr[1] == 'N')) {
			len8 = ptr[2];
			memcpy(str, &ptr[3], len8);
			str[len8] = 0;
			printf("    P/N: %s\n", str);
			pn = 1;
		} else if ((ptr[0] == 'E') && (ptr[1] == 'C')) {
			len8 = ptr[2];
			memcpy(str, &ptr[3], len8);
			str[len8] = 0;
			printf("    E/C: %s\n", str);
			ec = 1;
		} else if ((ptr[0] == 'S') && (ptr[1] == 'N')) {
			len8 = ptr[2];
			memcpy(str, &ptr[3], len8);
			str[len8] = 0;
			printf("    S/N: %s\n", str);
			sn = 1;
		} else if ((ptr[0] == 'V') && (ptr[1] == '0')) {
			len8 = ptr[2];
			memcpy(str, &ptr[3], len8);
			str[len8] = 0;
			printf("    Freq/Power: %s\n", str);
			fp = 1;
		} else if ((ptr[0] == 'V') && (ptr[1] == '2')) {
			len8 = ptr[2];
			memcpy(str, &ptr[3], len8);
			str[len8] = 0;
			printf("    Date Code: %s\n", str);
			dc = 1;
		} else if ((ptr[0] == 'R') && (ptr[1] == 'V')) {
			len = &ptr[2] - tvdev->vpd.vpd_char + 1;

			while (len >= 0) {
				cksum += tvdev->vpd.vpd_char[len];
				len--;
			}

			if (cksum == 0)
				printf("    Checksum: Ok\n");
			else
				printf("    Checksum: Incorrect\n");

			rv = 1;
		} else if (ptr[0] == 0x78)	/* End Tag */
			 break;

		i += (len8 + 3);
		ptr += (len8 + 3);
	}
	if (!pn)
		printf("    P/N: N/A\n");
	if (!ec)
		printf("    E/C: N/A\n");
	if (!sn)
		printf("    S/N: N/A\n");
	if (!dc)
		printf("    Date Code: N/A\n");
	if (!fp)
		printf("    Freq/Power: N/A\n");
	if (!rv)
		printf("    Checksum: N/A\n");
}

static int open_device(struct tvdevice *tvdev)
{
	unsigned short command;

	/* Enable memory regions if it's disabled */
	command = pci_read_word(tvdev->pdev, PCI_COMMAND);
	if (!(command & PCI_COMMAND_MEMORY)) {
		if (verbose)
			printf("INFO: enabling PCI memory transactions\n");
		pci_write_word(tvdev->pdev, PCI_COMMAND,
			       command | PCI_COMMAND_MEMORY);
	}

	if (!use_config_space && !(tvdev->flags & FLAG_RECOVERY)) {
		int fd;

		fd = open("/dev/mem", O_RDWR, O_SYNC);
		if (fd >= 0) {
			tvdev->bar0 = mmap64(NULL, 1 << 20, PROT_READ | PROT_WRITE, MAP_SHARED,
					 fd, tvdev->pdev->base_addr[0] & PCI_ADDR_MEM_MASK);
			close(fd);
			if (tvdev->bar0 == MAP_FAILED)
				tvdev->bar0 = NULL;
		} else
			tvdev->bar0 = NULL;
	} else
		tvdev->bar0 = NULL;

	if (tvdev->bar0) {
		tvdev->method = METHOD_MMAP;
		return 0;
	}

	/* FIXME: Test that we can do this first */
	tvdev->method = METHOD_PCI_CFG;

	return 0;
}

static void identify_flash_device(struct tvdevice *tvdev)
{
	char cfiq[1024], str[4];
	int i, width = 0;

	switch (tvdev->board->chip) {
	case CHIP_SINAI:
	case CHIP_HERMON:
		/* SPI device */
		spi_discover_serial_proms(tvdev);
		break;

	default:
		tvdev->flash_size = 0x400000;	/* 4MB */
		tvdev->flash_bank_shift = 19;
		tvdev->flash_bank_mask = (1 << tvdev->flash_bank_shift) - 1;

		/* Parallel flash chip */

		/* Get CFI info for flash device */
		flash_write_cmd(tvdev, 0x55, 0xFF);	/* Reset */
		flash_write_cmd(tvdev, 0x55, 0x98);	/* CFI Query */

		str[0] = flash_byte_read(tvdev, 0x10);
		str[1] = flash_byte_read(tvdev, 0x11);
		str[2] = flash_byte_read(tvdev, 0x12);
		str[3] = 0;

		if (strcmp(str, "QRY") == 0)
			width = 1;
		else {
			str[0] = flash_byte_read(tvdev, 0x20);
			str[1] = flash_byte_read(tvdev, 0x22);
			str[2] = flash_byte_read(tvdev, 0x24);
			str[3] = 0;

			if (strcmp(str, "QRY") == 0)
				width = 2;
		}

		if (!width) {
			fprintf(stderr, "CFI query failed. Unknown flash device.\n");
			exit(1);
		}

		for (i = 0; i < sizeof(cfiq); i++)
			cfiq[i] = flash_byte_read(tvdev, i * width);

		tvdev->flash_command_set = cfiq[0x13];

		/*
		 * FIXME: This is a hack for now. We should really get this
		 * from the CFI query. Careful, MX chips have buggy
		 * information.
		 */
		switch (tvdev->flash_command_set) {
		case CS_INTEL:
			tvdev->flash_sector_sz = 0x20000;	/* 128KB */
			break;
		case CS_AMD:
			tvdev->flash_sector_sz = 0x10000;	/* 64KB */
			break;
		default:
			printf("Unknown flash command set.\n");
			exit(1);
		}

		break;
	}

	tvdev->flash_bank_mask = (1 << tvdev->flash_bank_shift) - 1;

	flash_chip_reset(tvdev);

	tvdev->flash_bank = -1;
	flash_set_bank(tvdev, 0);
}

static int open_hca(struct tvdevice *tvdev)
{
	int ret;

	ret = open_device(tvdev);
	if (ret)
		return ret;

	read_cfg(tvdev, 0xF0150);
	write_cfg(tvdev, 0xF0150, 1 << 30);

	if (grab_gpio(tvdev)) {
		fprintf(stderr, "unable to acquire flash semaphore\n");
		return 1;
	}

	tvdev->board = tvdev->pciid->identify(tvdev);
	if (!tvdev->board) {
		fprintf(stderr, "unable to identify board\n");
		return 1;
	}

	switch (tvdev->board->chip) {
	case CHIP_SINAI:
		/* SPI flash */

		/* Unlock GPIO */
		write_cfg(tvdev, GPIO_LOCK, 0xAAAA);

#define SPI_ENABLE	((1 << (NUM_SPIS - 1)) - 1)
		write_cfg(tvdev, GPIO_DIR + 4, SPI_ENABLE << 5);
		write_cfg(tvdev, GPIO_POL + 4, (SPI_ENABLE ^ 0x07) << 5);
		write_cfg(tvdev, GPIO_MOD + 4, (SPI_ENABLE ^ 0x07) << 5);
		break;
	default:
		/* Parallel flash */

		/* Set the direction of the flash pins to output */
		write_cfg(tvdev, GPIO_DIR + 4, read_cfg(tvdev, GPIO_DIR + 4) | (0x07 << 4));

		/* Clear the data for the flash pins to start at bank 0 */
		write_cfg(tvdev, 0xF0080 + 0x54, 0x07 << 4);

		/* Clear the polarity for the flash pins */
		write_cfg(tvdev, GPIO_POL + 4, read_cfg(tvdev, GPIO_POL + 4) & ~(0x07 << 4));

		/* Clear the output mode for the flash pins */
		write_cfg(tvdev, GPIO_MOD + 4, read_cfg(tvdev, GPIO_MOD + 4) & ~(0x07 << 4));
		break;
	}

	identify_flash_device(tvdev);

	if (tvdev->flash_command_set == CS_UNKNOWN) {
		printf("Unknown flash device, cannot continue\n");
		return 1;
	}

	/* RevC LionCub boards use Intel flash chips and use a different firmware */
	if (tvdev->board == &lioncub && tvdev->flash_command_set == CS_INTEL)
		tvdev->board = &lioncub_revc;

	return 0;
}

static void close_hca(struct tvdevice *tvdev)
{
	release_gpio(tvdev);

	if (tvdev->bar0)
		munmap(tvdev->bar0, 1 << 20);
}

static void identify_fw_extended(const char *name, union vsd *vsd)
{
	if (vsd) {
		if (be16_to_cpu(vsd->topspin.signature1) == VSD_SIGNATURE_TOPSPIN &&
		    be16_to_cpu(vsd->topspin.signature2) == VSD_SIGNATURE_TOPSPIN) {
			char ver_str[40];

			create_ver_str(vsd, ver_str, sizeof(ver_str));
			if (vsd->topspin.build_rev[0])
				printf("  %s is v%s build %s.%d, with label '%s'\n",
				       name, ver_str,
				       vsd->topspin.build_rev,
				       le16_to_cpu(vsd->topspin.build_num),
				       vsd->topspin.hw_label);
			else
				printf("  %s is v%s, with label '%s'\n",
				       name, ver_str,
				       vsd->topspin.hw_label);
		} else
			printf("  %s is valid, unknown source\n", name);
	} else
		printf("  %s is NOT valid\n", name);
}

static void identify_hca_extended(int num, struct tvdevice *tvdev, int *unreliable)
{
	printf("HCA #%d: ", num);
	switch (tvdev->board->chip) {
	case CHIP_TAVOR:
		printf("MT23108");
		break;
	case CHIP_ARBEL:
		printf("MT25208");
		if (tvdev->flags & FLAG_TAVOR_COMPAT)
			printf(" Tavor Compat");
		break;
	case CHIP_SINAI:
		printf("MT25204");
		break;
	case CHIP_HERMON:
		printf("MT25408");
		break;
	default:
		printf("Unknown");
		break;
	}

	if (tvdev->flags & FLAG_RECOVERY)
		printf(" (recovery mode)");

	printf(", %s", tvdev->board->name);

	if (tvdev->flags & FLAG_RECOVERY) {
		printf("(*)");
		*unreliable = 1;
	}

	printf(", revision %02X\n", tvdev->revision);

	if (tvdev->board->chip == CHIP_HERMON) {
		identify_fw_extended("Firmware image", &tvdev->fw.hermon_fw_info.vsd);
	} else if (tvdev->fw.tavor_failsafe.valid) {
		identify_fw_extended("Primary image",
				     tvdev->fw.tavor_failsafe.images[0].valid ?
				     &tvdev->fw.tavor_failsafe.images[0].vsd : NULL);
		identify_fw_extended("Secondary image",
				     tvdev->fw.tavor_failsafe.images[1].valid ?
				     &tvdev->fw.tavor_failsafe.images[1].vsd : NULL);
	} else
		printf("  Firmware is NOT installed in failsafe mode\n");

	/* If vpd is present, read it and print it */
	if (tvdev->vpd_present)
		print_vpd_info(tvdev);
}

static void identify_hca_primary_fw(struct tvdevice *tvdev)
{
	union vsd *vsd;

	if (!tvdev->fw.tavor_failsafe.valid ||
	    !tvdev->fw.tavor_failsafe.images[0].valid) {
		printf("Invalid\n");
		return;
	}

	vsd = &tvdev->fw.tavor_failsafe.images[0].vsd;
	if (be16_to_cpu(vsd->topspin.signature1) == VSD_SIGNATURE_TOPSPIN &&
	    be16_to_cpu(vsd->topspin.signature2) == VSD_SIGNATURE_TOPSPIN) {
		unsigned int build_major = 0, build_minor = 0, build_micro = 0;

		sscanf(vsd->topspin.build_rev, "%u.%u.%u",
		       &build_major, &build_minor, &build_micro);

		printf("%s:%d:%d:%d:%d:%d:%d:%d:%d:%d\n",
		       vsd->topspin.hw_label,
		       (vsd->topspin.flags & cpu_to_be32(VSD_FLAG_AUTOUPGRADE)) ? 1 : 0,
		       vsd->topspin.major_ver,
		       vsd->topspin.minor_ver,
		       be16_to_cpu(vsd->topspin.micro_ver),
		       vsd->topspin.revision_ver,
		       build_major, build_minor, build_micro,
		       le16_to_cpu(vsd->topspin.build_num));
	} else
		printf("Unknown\n");
}

static int identify_hca(int num, struct tvdevice *tvdev,
			enum identify_mode identify_mode, int *unreliable)
{
	if (open_hca(tvdev)) {
		fprintf(stderr, "couldn't open hca %d\n", num);
		return 1;
	}

	flash_get_info(tvdev);

	switch (identify_mode) {
	case IDENTIFY_EXTENDED:
		identify_hca_extended(num, tvdev, unreliable);
		break;
	case IDENTIFY_PRIMARY_FIRMWARE_LABEL:
		identify_hca_primary_fw(tvdev);
		break;
	case IDENTIFY_HARDWARE_LABEL:
		/* Print out what we think the firmware label should be */
		if (tvdev->board->fwlabel)
			printf("%s.%02X",
				tvdev->board->fwlabel, tvdev->revision);
		else
			printf("Unknown.%02X", tvdev->revision);
		break;
	}

	close_hca(tvdev);

	return 0;
}

static int identify_hcas(int hca, enum identify_mode identify_mode)
{
	struct tvdevice *tvdev;
	int ret = 0, count = 0, unreliable = 0;

	if (hca >= 0) {
		tvdev = find_device(hca);
		if (!tvdev) {
			fprintf(stderr, "couldn't find HCA #%d on the PCI bus\n",
				hca);
			return 1;
		}

		ret = identify_hca(hca, tvdev, identify_mode, &unreliable);
	} else {
		for (tvdev = tvdevices; tvdev; tvdev = tvdev->next) {
			ret = identify_hca(count++, tvdev, identify_mode, &unreliable);
			if (ret)
				break;
		}
	}

	if (unreliable)
		printf("\n(*) Unreliable in recovery mode\n");

	return ret;
}

static int identify_firmware(char *ifname, enum identify_mode identify_mode)
{
	unsigned char *isbuf, *psbuf = NULL, *ibuf;
	unsigned int isbufsz, psbufsz, ibufsz;
	union vsd *vsd = NULL;

	/* Read the flash image into the memory */
	if (flash_image_read_from_file(ifname, &isbuf, &isbufsz, &psbuf,
				       &psbufsz, &ibuf, &ibufsz, &vsd))
		return 1;

	switch (identify_mode) {
	case IDENTIFY_EXTENDED:
		printf("Firmware image %s", ifname);

		if (vsd && vsd->topspin.flags & cpu_to_be32(VSD_FLAG_AUTOUPGRADE))
			printf(" (firmware autoupgrade)");

		printf("\n");

		if (vsd) {
			if (be16_to_cpu(vsd->topspin.signature1) == VSD_SIGNATURE_TOPSPIN &&
			    be16_to_cpu(vsd->topspin.signature2) == VSD_SIGNATURE_TOPSPIN) {
				char ver_str[40];

				create_ver_str(vsd, ver_str, sizeof(ver_str));
				if (vsd->topspin.build_rev[0])
					printf("  Image is v%s build %s.%d, with label '%s'\n",
						ver_str, vsd->topspin.build_rev,
						le16_to_cpu(vsd->topspin.build_num),
						vsd->topspin.hw_label);
				else
					printf("  Image is v%s, with label '%s'\n",
						ver_str,
						vsd->topspin.hw_label);
			} else
				printf("  Image is valid, unknown source\n");
		} else
			printf("  Image is NOT valid\n");

		break;
	case IDENTIFY_PRIMARY_FIRMWARE_LABEL:
		if (vsd) {
			if (be16_to_cpu(vsd->topspin.signature1) == VSD_SIGNATURE_TOPSPIN &&
			    be16_to_cpu(vsd->topspin.signature2) == VSD_SIGNATURE_TOPSPIN) {
				unsigned int build_major = 0, build_minor = 0, build_micro = 0;

				sscanf(vsd->topspin.build_rev, "%u.%u.%u",
					 &build_major, &build_minor, &build_micro);

				printf("%s:%d:%d:%d:%d:%d:%d:%d:%d:%d\n",
					 vsd->topspin.hw_label,
					 (vsd->topspin.flags & cpu_to_be32(VSD_FLAG_AUTOUPGRADE)) ? 1 : 0,
					 vsd->topspin.major_ver,
					 vsd->topspin.minor_ver,
					 be16_to_cpu(vsd->topspin.micro_ver),
					 vsd->topspin.revision_ver,
					 build_major, build_minor, build_micro,
					 le16_to_cpu(vsd->topspin.build_num));
			} else
				printf("Unknown\n");
		} else
			printf("Invalid\n");

		break;

	default:
		printf("Invalid mode\n");
		break;
	}

	return 0;
}

static int print_hca_guids(int num, struct tvdevice *tvdev)
{
	unsigned char guid_node[GUID_LEN];
	unsigned char guid_port1[GUID_LEN];
	unsigned char guid_port2[GUID_LEN];

	if (open_hca(tvdev)) {
		fprintf(stderr, "couldn't open hca %d\n", num);
		return 1;
	}

	flash_get_info(tvdev);

	if (flash_get_curr_guids(tvdev, guid_node, guid_port1, guid_port2)) {
		fprintf(stderr, "Cannot determine previous GUID. Corrupted flash?\n");
		return 1;
	}

	printf("HCA #%d\n", num);
	printf("Node  GUID = %02x%02x%02x%02x%02x%02x%02x%02x\n",
		guid_node[0], guid_node[1], guid_node[2], guid_node[3],
		guid_node[4], guid_node[5], guid_node[6], guid_node[7]);
	printf("Port1 GUID = %02x%02x%02x%02x%02x%02x%02x%02x\n",
		guid_port1[0], guid_port1[1], guid_port1[2], guid_port1[3],
		guid_port1[4], guid_port1[5], guid_port1[6], guid_port1[7]);
	if (tvdev->board->num_ports > 1)
		printf("Port2 GUID = %02x%02x%02x%02x%02x%02x%02x%02x\n",
			guid_port2[0], guid_port2[1], guid_port2[2], guid_port2[3],
			guid_port2[4], guid_port2[5], guid_port2[6], guid_port2[7]);

	close_hca(tvdev);

	return 0;
}

static int print_guids(int hca)
{
	struct tvdevice *tvdev;
	int ret = 0, count = 0;

	if (hca >= 0) {
		tvdev = find_device(hca);
		if (!tvdev) {
			fprintf(stderr, "couldn't find HCA #%d on the PCI bus\n", hca);
			return 1;
		}

		ret = print_hca_guids(hca, tvdev);
	} else {
		for (tvdev = tvdevices; tvdev; tvdev = tvdev->next) {
			ret = print_hca_guids(count++, tvdev);
			if (ret)
				break;
		}
	}

	return ret;
}

/* Download (from HCA to host) */
static int flash_image_write_to_file(struct tvdevice *tvdev, char *fname)
{
	char *buffer;
	int i, fd;
	unsigned int offset;

	buffer = malloc(tvdev->flash_sector_sz);
	if (!buffer) {
		fprintf(stderr, "couldn't allocated %d bytes of memory for buffer\n",
			tvdev->flash_sector_sz);
		return 1;
	}

	if (strcmp(fname, "-") == 0)
		fd = fileno(stdout);
	else
		fd = creat(fname, 0644);

	if (fd < 0) {
		fprintf(stderr, "couldn't open %s to save firmware: %m\n", fname);
		return 1;
	}

	offset = 0;
	while (offset < tvdev->flash_size) {
		for (i = 0; i < tvdev->flash_sector_sz; i += 4) {
			unsigned int data;

			data = flash_read(tvdev, offset + i);

			buffer[i + 0] = (data >> 24) & 0xff;
			buffer[i + 1] = (data >> 16) & 0xff;
			buffer[i + 2] = (data >> 8) & 0xff;
			buffer[i + 3] = data & 0xff;
		}

		write(fd, buffer, tvdev->flash_sector_sz);

		offset += tvdev->flash_sector_sz;
	}

	close(fd);

	return 0;
}

static int download_firmware(int hca, char *ofname)
{
	struct tvdevice *tvdev;
	int ret;

	tvdev = find_device(hca);
	if (!tvdev) {
		fprintf(stderr, "couldn't find HCA #%d on the PCI bus\n", hca);
		return 1;
	}

	if (open_hca(tvdev)) {
		fprintf(stderr, "couldn't open hca %d\n", hca);
		return 1;
	}

	flash_get_info(tvdev);

	ret = flash_image_write_to_file(tvdev, ofname);

	close_hca(tvdev);

	return ret;
}

static void flash_write_block(struct tvdevice *tvdev, unsigned char status,
	unsigned int addr, unsigned char *buffer, unsigned int buflen)
{
	unsigned int pos = 0, bufsz = buflen;

	while (bufsz > 0) {
		unsigned int len = 16;
		int i;

		flash_set_bank(tvdev, addr);

		if (len > bufsz)
			bufsz = len;

		switch (tvdev->flash_command_set) {
		case CS_SPI_SINAI:
		case CS_SPI_HERMON:
			flash_spi_write_block(tvdev, addr, buffer + pos, len);
			break;

		default:
			for (i = 0; i < len; i++)
				flash_write_byte(tvdev, addr + i, buffer[pos + i]);
			break;
		}
		addr += len;
		pos += len;
		bufsz -= len;

		if (status)
			status_update(status, pos);
	}
}

/* Upload (from host to HCA) */
static int flash_compare_invariant_sector(struct tvdevice *tvdev, unsigned char *isbuf,
	unsigned int isbufsz)
{
	unsigned int addr, data;

	flash_chip_reset(tvdev);

	for (addr = 0; addr < isbufsz; addr += 4) {
		int i;

		data = flash_read(tvdev, addr);
		for (i = 0; i < 4; i++) {
			if (isbuf[addr + i] != ((data >> ((3 - i) * 8)) & 0xFF))
				return 1;
		}
	}

	return 0;
}

static void flash_move_pps_to_sps(struct tvdevice *tvdev)
{
	unsigned int addr;
	unsigned char *sector;

	flash_chip_reset(tvdev);

	sector = malloc(tvdev->flash_sector_sz);
	if (!sector) {
		fprintf(stderr, "Unable to allocate %d bytes of memory for buffer\n",
			tvdev->flash_sector_sz);
		exit(1);
	}

	for (addr = 0; addr < tvdev->flash_sector_sz; addr += 4) {
		uint32_t data;
		int i;

		data = flash_read(tvdev, tvdev->flash_sector_sz + addr);
		for (i = 0; i < 4; i++)
			sector[addr + i] = ((data >> ((3 - i) * 8)) & 0xFF);
	}

	/* Erase the SPS sector */
	status_update('E', 0);
	flash_erase_sector(tvdev, tvdev->flash_sector_sz * 2);

	/* Write the old PPS as the SPS */
	flash_write_block(tvdev, 'F', tvdev->flash_sector_sz * 2, sector, tvdev->flash_sector_sz);
	status_mark();

	flash_chip_reset(tvdev);

	free(sector);
}

static void flash_write_invariant_sector(struct tvdevice *tvdev, unsigned char *isbuf,
	unsigned int isbufsz)
{
	flash_chip_reset(tvdev);

	status_update('E', 0);
	flash_erase_sector(tvdev, 0);
	flash_write_block(tvdev, 'I', 0, isbuf, isbufsz);
	status_mark();

	flash_chip_reset(tvdev);
}

static void flash_write_pps(struct tvdevice *tvdev, unsigned int psbufoff,
	unsigned char *psbuf, unsigned int psbufsz, unsigned int ibufoff,
	unsigned int ibufsz)
{
	/* Set the image offset and size */
	*(uint32_t *)(psbuf + 0x0) = cpu_to_be32(ibufoff);
	*(uint32_t *)(psbuf + 0x4) = cpu_to_be32(ibufsz);

	/* Set signature and CRC to all-ones for now */
	*(uint32_t *)(psbuf + 0x8) = 0xFFFFFFFF;
	*(uint16_t *)(psbuf + 0x106) = 0xFFFF;

	/* Erase then write the PPS */
	flash_chip_reset(tvdev);

	status_update('E', 0);
	flash_erase_sector(tvdev, psbufoff);
	flash_write_block(tvdev, 'P', psbufoff, psbuf, psbufsz);
	status_mark();

	flash_chip_reset(tvdev);
}

static void flash_finish_failsafe(struct tvdevice *tvdev, unsigned int psbufoff,
	unsigned char *psbuf, unsigned int psbufsz)
{
	/* Set signature and CRC to something valid */
	*(uint32_t *)(psbuf + 0x8) = cpu_to_be32(0x5a445a44);
	*(uint16_t *)(psbuf + 0x106) = cpu_to_be16(flash_crc16(psbuf, 0x104));

	/* Write CRC and signature */
	flash_chip_reset(tvdev);

	flash_write_block(tvdev, 0, psbufoff + 0x106, psbuf + 0x106, sizeof(uint16_t));
	flash_write_block(tvdev, 0, psbufoff + 0x8, psbuf + 0x8, sizeof(uint32_t));

	flash_chip_reset(tvdev);
}

static void flash_write_image(struct tvdevice *tvdev, unsigned int ibufoff,
	unsigned char *ibuf, unsigned int ibufsz)
{
	unsigned int addr;

	/* Erase then write the firmware image */
	for (addr = 0; addr < ibufsz; addr += tvdev->flash_sector_sz) {
		unsigned int len = tvdev->flash_sector_sz;

		if (len > ibufsz - addr)
			len = ibufsz - addr;

		status_update('E', 0);
		flash_erase_sector(tvdev, ibufoff + addr);
		flash_write_block(tvdev, 'W', ibufoff + addr, ibuf + addr, len);
		status_mark();
	}

	flash_chip_reset(tvdev);
}

static unsigned int flash_verify_image(struct tvdevice *tvdev, unsigned int off,
	unsigned char *buf, unsigned int bufsz)
{
	unsigned int addr, data;

	flash_chip_reset(tvdev);

	for (addr = 0; addr < bufsz; addr += 4) {
		int i;

		status_update('V', addr);

		data = flash_read(tvdev, off + addr);
		for (i = 0; i < 4; i++) {
			if (buf[addr + i] != ((data >> ((3 - i) * 8)) & 0xFF))
				return addr + i;
		}
	}
	status_mark();

	return addr;
}

static void usage(void)
{
	fprintf(stderr, "Usage: %s [OPTION]...\n", argv0);
	fprintf(stderr, "  -q\t\tProduce less output (quieter)\n");
	fprintf(stderr, "  -v\t\tProduce more output (verbose)\n");
	fprintf(stderr, "  -f\t\tForce operation (use cautiously)\n");
	fprintf(stderr, "  -p\t\tUser PCI config space access instead of memory mapping\n");
	fprintf(stderr, "  -h <num>\tSelect HCA to use for operation (default: 0)\n");
	fprintf(stderr, "  -o [<cmd>]\tChange options command (no command shows current values)\n");
	fprintf(stderr, "    The following commands and values are accepted:\n");
	fprintf(stderr, "      auto_upgrade              [yes|no]\n");
	fprintf(stderr, "      boot_enable_port1         [yes|no]\n");
	fprintf(stderr, "      boot_enable_port2         [yes|no]\n");
	fprintf(stderr, "      boot_wait_on_error        [yes|no]\n");
	fprintf(stderr, "      boot_try_forever          [yes|no]\n");
	fprintf(stderr, "      boot_type                 [well_known|saved|pxe|disable]\n");
	fprintf(stderr, "      boot_saved_port           [0|1|2]\n");
	fprintf(stderr, "      boot_saved_ioc_num        ##\n");
	fprintf(stderr, "      boot_saved_dgid           ####:####:####:####:####:####:####:####\n");
	fprintf(stderr, "      boot_saved_service_name   ####:####:####:####\n");
	fprintf(stderr, "      boot_pxe_secs             ##\n");
	fprintf(stderr, "  -u\t\tDisable autoupgrade of firmware (-o auto_upgrade=no)\n\n");

	fprintf(stderr, "One operation mode must be specified:\n");
	fprintf(stderr, "  -i\t\tIdentify HCAs\n");
	fprintf(stderr, "  -g\t\tPrint GUIDs programmed into firmware\n");
	fprintf(stderr, "  -s <filename>\n\t\tSave firmware programmed on flash device to local file\n");
	fprintf(stderr, "  [-d] <filename> [<Node GUID>]\n\t\tDownload firmware to flash device from local file\n\n");

	fprintf(stderr, "When specifying a GUID, use only the lower 24 bits in hexadecimal format.\n");

	exit(1);
}

static int upload_firmware(int hca, char *ifname, char *guid, char *change_option_cmd)
{
	struct tvdevice *tvdev;
	unsigned char *isbuf, *psbuf, *ibuf;
	unsigned int isbufsz, psbufsz, ibufsz, addr, status_count;
	unsigned char new_guid_node[GUID_LEN], new_guid_port1[GUID_LEN],
	new_guid_port2[GUID_LEN];
	union vsd *vsd;
	int ret = 1, sector = 0, program_is = 0;

	tvdev = find_device(hca);
	if (!tvdev) {
		fprintf(stderr, "couldn't find HCA #%d on the PCI bus\n", hca);
		return 1;
	}

	if (open_hca(tvdev)) {
		fprintf(stderr, "couldn't open hca %d\n", hca);
		return 1;
	}

	flash_get_info(tvdev);

	if (guid) {
		/* GUIDs given to us on command line */
		if (parse_guid(guid, new_guid_node)) {
			fprintf(stderr, "Unable to parse GUID from command line\n\n");
			usage();
		}

		inc_guid(new_guid_node, new_guid_port1);
		if (tvdev->board->num_ports > 1)
			inc_guid(new_guid_port1, new_guid_port2);
	} else {
		/* Preserve the current GUIDs */
		if (flash_get_curr_guids(tvdev, new_guid_node, new_guid_port1,
					 new_guid_port2)) {
			fprintf(stderr, "Cannot determine previous GUID. Corrupted flash?\n");
			fprintf(stderr, "GUID must be specified on command line.\n\n");
			usage();
		}

		/*
		 * Now we check the GUIDs to make sure they are valid or
		 * could possibly cause some confusion (the default Mellanox
		 * GUID)
		 */
		if (memcmp(new_guid_node,  "\x00\x00\x00\x00\x00\x00\x00\x00", 8) == 0 ||
		    memcmp(new_guid_port1, "\x00\x00\x00\x00\x00\x00\x00\x00", 8) == 0 ||
		    (tvdev->board->num_ports > 1 &&
		     memcmp(new_guid_port2, "\x00\x00\x00\x00\x00\x00\x00\x00", 8) == 0)) {
			fprintf(stderr, "GUIDs read from flash are all zeros. Corrupted flash?\n");
			fprintf(stderr, "GUID must be specified on command line.\n\n");
			usage();
		}

		if (memcmp(new_guid_node,  "\xff\xff\xff\xff\xff\xff\xff\xff", 8) == 0 ||
		    memcmp(new_guid_port1, "\xff\xff\xff\xff\xff\xff\xff\xff", 8) == 0 ||
		    (tvdev->board->num_ports > 1 &&
		     memcmp(new_guid_port2, "\xff\xff\xff\xff\xff\xff\xff\xff", 8) == 0)) {
			fprintf(stderr, "GUIDs read from flash are all ones. Device access error?\n");
			fprintf(stderr, "GUID must be specified on command line.\n\n");
			usage();
		}

		if ((memcmp(new_guid_node,  "\x00\x02\xc9\x00\x01\x00\xd0\x50", 8) == 0 ||
		     memcmp(new_guid_port1, "\x00\x02\xc9\x00\x01\x00\xd0\x51", 8) == 0 ||
		     (tvdev->board->num_ports > 1 &&
		      memcmp(new_guid_port2, "\x00\x02\xc9\x00\x01\x00\xd0\x52", 8) == 0)) &&
		    !force) {
			fprintf(stderr, "GUIDs read from flash are vendor default GUIDs. It is not recommended\n");
			fprintf(stderr, "that these GUIDs be used because of a high chance of GUID conflict.\n");
			fprintf(stderr, "Please specify new GUID on command line, or use -f option.\n");
			usage();
		}
	}

	/* Read the flash image into the memory */
	if (flash_image_read_from_file(ifname, &isbuf, &isbufsz, &psbuf,
				       &psbufsz, &ibuf, &ibufsz, &vsd))
		return 1;

	if (psbuf) {
		if (be16_to_cpu(vsd->topspin.signature1) == VSD_SIGNATURE_TOPSPIN &&
		    be16_to_cpu(vsd->topspin.signature2) == VSD_SIGNATURE_TOPSPIN) {
			union vsd *fvsd = &tvdev->fw.tavor_failsafe.images[0].vsd;

			/* Update the time the image was flashed */
			vsd->topspin.flashtime = cpu_to_be32(time(NULL));

			/* Set default value, we'll overwrite this if we need to */
			vsd->topspin.flags |= cpu_to_be32(VSD_FLAG_AUTOUPGRADE);

			/* Copy over boot ROM options from old VSD */
			vsd->topspin.flags &= ~cpu_to_be32(VSD_FLAG_BOOT_OPTIONS);
			vsd->topspin.flags |= (fvsd->topspin.flags & cpu_to_be32(VSD_FLAG_BOOT_OPTIONS));

			if (vsd->topspin.flags & cpu_to_be32(VSD_FLAG_BOOT_OPTIONS)) {
				vsd->topspin.flags &= ~cpu_to_be32(VSD_FLAG_BOOT_OPTIONS_MASK);
				vsd->topspin.flags |= (fvsd->topspin.flags & cpu_to_be32(VSD_FLAG_BOOT_OPTIONS_MASK));

				vsd->topspin.boot_version = fvsd->topspin.boot_version;
				vsd->topspin.boot_port = fvsd->topspin.boot_port;
				vsd->topspin.boot_ioc_num = fvsd->topspin.boot_ioc_num;
				memcpy(vsd->topspin.boot_dgid,
					fvsd->topspin.boot_dgid,
					sizeof(vsd->topspin.boot_dgid));
				memcpy(vsd->topspin.boot_service_name,
					fvsd->topspin.boot_service_name,
					sizeof(vsd->topspin.boot_service_name));
				vsd->topspin.boot_pxe_secs = fvsd->topspin.boot_pxe_secs;
			} else {
				/* Set some sensible defaults */
				vsd->topspin.flags |= cpu_to_be32(VSD_FLAG_BOOT_OPTIONS |
					 VSD_FLAG_BOOT_ENABLE_PORT_1 |
					 VSD_FLAG_BOOT_ENABLE_PORT_2 |
					 VSD_FLAG_BOOT_ENABLE_SCAN |
					 VSD_FLAG_BOOT_TYPE_WELL_KNOWN);
				vsd->topspin.boot_version = BOOT_VERSION;
				vsd->topspin.boot_pxe_secs = BOOT_PXE_SECS_DEFAULT;
			}

			if (change_option_cmd &&
			    parse_change_options(&vsd->topspin, change_option_cmd))
				return 1;

			/* Calculate the VSD checksum */
			vsd->topspin.checksum = 0;
			vsd->topspin.checksum = cpu_to_be16(flash_crc16(vsd->raw, sizeof(vsd->raw)));
		}

		/* PPS checksum will be updated as a final step */

		if (be16_to_cpu(vsd->topspin.signature1) == VSD_SIGNATURE_TOPSPIN &&
		    be16_to_cpu(vsd->topspin.signature2) == VSD_SIGNATURE_TOPSPIN &&
		    !force) {
			if (tvdev->board->fwlabel) {
				char hwlabel[64];

				/*
				 * Check to make sure this file is correct for
				 * the hardware. We check only as far as the
				 * hwlabel since boot firmware will have extra
				 * information appended to the label.
				 */
				snprintf(hwlabel, sizeof(hwlabel), "%s.%02X",
					tvdev->board->fwlabel, tvdev->revision);
				if (strncasecmp(vsd->topspin.hw_label, hwlabel, strlen(hwlabel)) != 0) {
					fprintf(stderr, "Firmware image specified has hardware label '%s', but hardware\n",
						vsd->topspin.hw_label);
					fprintf(stderr, "is of type '%s'. Please confirm you are using the correct\n",
						hwlabel);
					fprintf(stderr, "firmware or use the force option (-f).\n");
					return 1;
				}
			} else {
				if (tvdev->flags & FLAG_RECOVERY) {
					fprintf(stderr, "WARNING: Unable to verify firmware image is appropriate for hardware when\n");
					fprintf(stderr, "hardware is in flash recovery mode.\n");
				} else
					fprintf(stderr, "WARNING: Unable to verify firmware image is appropriate for unknown hardware.\n");

				fprintf(stderr, "Will upload flash image in 20 seconds or hit Ctrl-C to exit.\n");
				sleep(20);
			}
		}

		if (tvdev->fw.tavor_failsafe.valid) {
			if ((tvdev->flash_command_set != CS_SPI_SINAI &&
			     tvdev->flash_command_set != CS_SPI_HERMON) ||
			    tvdev->flash_spi_sp_type == SP_ST_M25P80) {
			/*
			 * The mainstream case - We'll always use the
			 * space opposite of the PPS.
			 */
				if (tvdev->fw.tavor_failsafe.images[0].addr + tvdev->fw.tavor_failsafe.images[0].size < tvdev->flash_size / 2)
					sector = (tvdev->flash_size / 2) / tvdev->flash_sector_sz;
				else if (tvdev->fw.tavor_failsafe.images[0].addr >= tvdev->flash_sector_sz * 3 + ibufsz)
					sector = 3;
				else {
					fprintf(stderr, "Unable to fit new image (size 0x%x) on flash in Failsafe mode\n", ibufsz);
					exit(1);
				}
			} else {
			/*
			 * Sinai with 2MB single flash prom has issue
			 * if the PPS points above 1MB
			 */
				if ((tvdev->fw.tavor_failsafe.images[0].addr / tvdev->flash_sector_sz) == 3)
					sector = ((tvdev->flash_sector_sz * 3 + tvdev->fw.tavor_failsafe.images[1].size) / tvdev->flash_sector_sz) + 1;
				else if (tvdev->fw.tavor_failsafe.images[0].addr > tvdev->flash_sector_sz * 3)
					sector = 3;
				else {
					fprintf(stderr, "Unable to fit new image (size 0x%x) on flash in Failsafe mode\n", ibufsz);
					exit(1);
				}
			}
		} else
			/* No Invariant Sector yet (will be programmed below) */
			sector = 3;
	} else if (ibufsz > tvdev->flash_size) {
		fprintf(stderr, "Image size is larger than size of flash (0x%X)\n",
			tvdev->flash_size);
		return 1;
	}

	if (tvdev->board->chip != CHIP_HERMON && tvdev->fw.tavor_failsafe.valid && !psbuf) {
		if (!force) {
			/* Flash is in Failsafe mode, but image doesn't have Failsafe information */
			printf("Flash is in failsafe mode, but image is not a failsafe firmware image.\n");
			printf("Please use a failsafe firmware image, or use the force option if you\n");
			printf("are sure you know what you are doing.\n");
			exit(1);
		} else {
			printf("WARNING: Downgrading flash from Failsafe to non Failsafe. Please hit Ctrl-C\n");
			printf("now if that is not what you wanted.\n");
			sleep(5);
		}
	}

	if (flash_guids_update(tvdev, ibuf, ibufsz, new_guid_node,
			       new_guid_port1, new_guid_port2))
		return 1;

	/* Compare the currently programmed IS with the IS from the file */
	if (isbuf) {
		if (force > 1) {
			/* User specified -ff, always upgrade IS */
			if (tvdev->fw.tavor_failsafe.valid) {
				/* But warn if the firmware is already in failsafe mode, just in case */
				printf("WARNING: Flash reprogramming won't be failsafe, continuing in 10 seconds\n");
				sleep(10);
			}

			program_is = 1;
		} else if (flash_compare_invariant_sector(tvdev, isbuf, isbufsz) != 0) {
			/* IS is out-of-date, so we need to upgrade it */
			printf("WARNING: Out-of-date Invariant sector found. Flash reprogramming won't be\n");
			printf("failsafe, continuing in 10 seconds\n");
			sleep(10);
			program_is = 1;
		} else if (!tvdev->fw.tavor_failsafe.valid)
			/* Not in failsafe, so it's safe to silently program the IS */
			program_is = 1;
	}

	printf("Programming HCA firmware... Flash Image Size = %d\n", ibufsz);

	if (psbuf) {
		status_count = psbufsz + ibufsz + psbufsz + ibufsz;
		if (tvdev->fw.tavor_failsafe.images[0].valid)
			status_count += tvdev->flash_sector_sz;
	} else
		status_count = ibufsz + ibufsz;

	if (program_is) {
		/* Program the Invariant Sector first */
		status_start(status_count + isbufsz + isbufsz);
		flash_write_invariant_sector(tvdev, isbuf, isbufsz);

		addr = flash_verify_image(tvdev, 0, isbuf, isbufsz);
		if (addr < isbufsz) {
			status_stop();
			printf("Flash verify of IS FAILED @ %d (%d total)\n",
				addr, isbufsz);
			ret = 1;
			goto verify_failed;
		}
	} else
		status_start(status_count);

	if (psbuf) {
		if (tvdev->fw.tavor_failsafe.images[0].valid)
			flash_move_pps_to_sps(tvdev);

		/* Write the PPS and image */
		flash_write_pps(tvdev, tvdev->flash_sector_sz, psbuf, psbufsz,
		sector * tvdev->flash_sector_sz, ibufsz);
		flash_write_image(tvdev, sector * tvdev->flash_sector_sz, ibuf, ibufsz);

		/* Verify the PPS and image */
		addr = flash_verify_image(tvdev, tvdev->flash_sector_sz, psbuf, psbufsz);
		if (addr < psbufsz) {
			status_stop();
			printf("Flash verify of PPS FAILED @ %d (%d total)\n",
				addr, psbufsz);
			ret = 1;
			goto verify_failed;
		}

		addr = flash_verify_image(tvdev, sector * tvdev->flash_sector_sz, ibuf, ibufsz);
		if (addr < ibufsz) {
			status_stop();
			printf("Flash verify of image FAILED @ %d (%d total)\n",
				addr, ibufsz);
			ret = 1;
			goto verify_failed;
		}

		/* Finish writing the PPS */
		flash_finish_failsafe(tvdev, tvdev->flash_sector_sz, psbuf, psbufsz);
	} else {
		flash_write_image(tvdev, 0, ibuf, ibufsz);
		addr = flash_verify_image(tvdev, 0, ibuf, ibufsz);
		if (addr < ibufsz) {
			status_stop();
			printf("Flash verify of image FAILED @ %d (%d total)\n",
				addr, ibufsz);
			ret = 1;
			goto verify_failed;
		}
	}

	status_stop();

	ret = 0;

	printf("Flash verify passed!\n");

verify_failed:
	close_hca(tvdev);

	return ret;
}

static int modify_options(int hca, char *change_option_cmd)
{
	struct tvdevice *tvdev;
	union vsd *vsd;
	unsigned char *psbuf;
	int ret;
	int i;

	tvdev = find_device(hca);
	if (!tvdev) {
		fprintf(stderr, "couldn't find HCA #%d on the PCI bus\n", hca);
		return 1;
	}

	if (open_hca(tvdev)) {
		fprintf(stderr, "couldn't open hca %d\n", hca);
		return 1;
	}

	flash_get_info(tvdev);

	psbuf = malloc(tvdev->flash_sector_sz);
	if (!psbuf) {
		fprintf(stderr, "couldn't allocate temp buffer for PPS/SPS (size = %d)\n",
			tvdev->flash_sector_sz);
		exit(1);
	}

	for (i = 0; i < tvdev->flash_sector_sz; i += 4)
		*(uint32_t *)(psbuf + i) = be32_to_cpu(flash_read(tvdev, tvdev->flash_sector_sz + i));

	vsd = (union vsd *)(psbuf + 0x20);

	fixup_vsd(vsd);

	if (be16_to_cpu(vsd->topspin.signature1) != VSD_SIGNATURE_TOPSPIN ||
	    be16_to_cpu(vsd->topspin.signature2) != VSD_SIGNATURE_TOPSPIN) {
		/* Not our VSD format, so nothing to print */
		printf("error: VSD not in Topspin format\n");
		ret = 1;
		goto out;
	}

	if (change_option_cmd[0] == 0) {
		if (vsd->topspin.flags & cpu_to_be32(VSD_FLAG_AUTOUPGRADE))
			printf("  auto_upgrade=yes\n");
		else
			printf("  auto_upgrade=no\n");

		if (vsd->topspin.flags & cpu_to_be32(VSD_FLAG_BOOT_ENABLE_PORT_1))
			printf("  boot_enable_port_1=yes\n");
		else
			printf("  boot_enable_port_1=no\n");

		if (vsd->topspin.flags & cpu_to_be32(VSD_FLAG_BOOT_ENABLE_PORT_2))
			printf("  boot_enable_port_2=yes\n");
		else
			printf("  boot_enable_port_2=no\n");

		if (!(vsd->topspin.flags & cpu_to_be32(VSD_FLAG_BOOT_ENABLE_SCAN)))
			printf("  boot_service_scan=no\n");

		if (vsd->topspin.flags & cpu_to_be32(VSD_FLAG_BOOT_WAIT_ON_ERROR))
			printf("  boot_wait_on_error=yes\n");
		else
			printf("  boot_wait_on_error=no\n");

		if (vsd->topspin.flags & cpu_to_be32(VSD_FLAG_BOOT_TRY_FOREVER))
			printf("  boot_try_forever=yes\n");
		else
			printf("  boot_try_forever=no\n");

		switch (be32_to_cpu(vsd->topspin.flags) & VSD_FLAG_BOOT_TYPE) {
		case VSD_FLAG_BOOT_TYPE_WELL_KNOWN:
			printf("  boot_type=well_known\n");
			break;
		case VSD_FLAG_BOOT_TYPE_SAVED:
			printf("  boot_type=saved\n");
			break;
		case VSD_FLAG_BOOT_TYPE_PXE:
			printf("  boot_type=pxe\n");
			break;
		case VSD_FLAG_BOOT_TYPE_DISABLE:
			printf("  boot_type=disable\n");
			break;
		default:
			printf("  boot_type=unknown\n");
			break;
		}

		printf("  boot_saved_port=%d\n", vsd->topspin.boot_port);
		printf("  boot_saved_ioc_num=%d\n", vsd->topspin.boot_ioc_num);
		printf("  boot_saved_dgid=%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x\n",
			vsd->topspin.boot_dgid[0],  vsd->topspin.boot_dgid[1],
			vsd->topspin.boot_dgid[2],  vsd->topspin.boot_dgid[3],
			vsd->topspin.boot_dgid[4],  vsd->topspin.boot_dgid[5],
			vsd->topspin.boot_dgid[6],  vsd->topspin.boot_dgid[7],
			vsd->topspin.boot_dgid[8],  vsd->topspin.boot_dgid[9],
			vsd->topspin.boot_dgid[10], vsd->topspin.boot_dgid[11],
			vsd->topspin.boot_dgid[12], vsd->topspin.boot_dgid[13],
			vsd->topspin.boot_dgid[14], vsd->topspin.boot_dgid[15]);
		printf("  boot_saved_service_name=%02x%02x:%02x%02x:%02x%02x:%02x%02x\n",
			vsd->topspin.boot_service_name[0],
			vsd->topspin.boot_service_name[1],
			vsd->topspin.boot_service_name[2],
			vsd->topspin.boot_service_name[3],
			vsd->topspin.boot_service_name[4],
			vsd->topspin.boot_service_name[5],
			vsd->topspin.boot_service_name[6],
			vsd->topspin.boot_service_name[7]);
		printf("  boot_pxe_secs=%d\n", vsd->topspin.boot_pxe_secs);
	} else {
		if (parse_change_options(&vsd->topspin, change_option_cmd)) {
			ret = 1;
			goto out;
		}

		/* Calculate the VSD checksum */
		vsd->topspin.checksum = 0;
		vsd->topspin.checksum = cpu_to_be16(flash_crc16(vsd->raw, sizeof(vsd->raw)));

		/* Set signature and CRC to all-ones for now */
		*(uint32_t *)(psbuf + 0x8) = 0xFFFFFFFF;
		*(uint16_t *)(psbuf + 0x106) = 0xFFFF;

		flash_chip_reset(tvdev);

		status_start(tvdev->flash_sector_sz);

		status_update('E', 0);
		flash_erase_sector(tvdev, tvdev->flash_sector_sz);
		flash_write_block(tvdev, 'F', tvdev->flash_sector_sz, psbuf, tvdev->flash_sector_sz);
		status_mark();

		flash_chip_reset(tvdev);

		flash_finish_failsafe(tvdev, tvdev->flash_sector_sz, psbuf, tvdev->flash_sector_sz);

		status_stop();
	}

	free(psbuf);

	ret = 0;

out:
	close_hca(tvdev);

	return ret;
}

int main(int argc, char *argv[])
{
	char *filename = NULL, *guid = NULL, *change_option_cmd = NULL;
	struct winsize winsize;
	struct pci_access *pacc;
	enum operation operation = OP_NONE;
	enum identify_mode identify_mode = IDENTIFY_EXTENDED;
	int ret, autoupgrade = 1, hca = -1;

	/* Check the size is correct. The compiler will optimize this away */
	{
		union vsd vsd;

		if (sizeof(vsd.topspin) != sizeof(vsd.raw)) {
			extern void __sizeof_vsd_data_doesnt_equal_sizeof_vsd_raw();

			__sizeof_vsd_data_doesnt_equal_sizeof_vsd_raw();
		}
	}

	/* Figure out the size of the window, but fall back if we can't */
	if (ioctl(0, TIOCGWINSZ, &winsize) < 0)
		cols = 80;
	else
		cols = winsize.ws_col;

	argv0 = argv[0];

	while (1) {
		int c;

		c = getopt(argc, argv, "qvfupgh:i::s:d:o::");
		if (c == -1)
			break;

		switch (c) {
		case 'q':
			verbose--;
			break;
		case 'v':
			verbose++;
			break;
		case 'f':
			force++;
			break;
		case 'u':
			autoupgrade = 0;
			break;
		case 'p':
			use_config_space = 1;
			break;
		case 'h': {
			char *p;

			ret = strtoul(optarg, &p, 0);
			if (ret == ULONG_MAX || *p) {
				fprintf(stderr, "parameter to -h option is invalid\n");
				return 1;
			}

			hca = (unsigned int)ret;
			break;
			}
		case 'i':
			if (operation != OP_NONE) {
				fprintf(stderr, "multiple operations specified on command line\n");
				usage();
			}

			operation = OP_IDENTIFY;
			if (optarg) {
				if (strcasecmp(optarg, "extended") == 0)
					identify_mode = IDENTIFY_EXTENDED;
				else if (strcasecmp(optarg, "pfwlabel") == 0)
					identify_mode = IDENTIFY_PRIMARY_FIRMWARE_LABEL;
				else if (strcasecmp(optarg, "hwlabel") == 0)
					identify_mode = IDENTIFY_HARDWARE_LABEL;
				else {
					fprintf(stderr, "unknown identify mode '%s'\n",
						optarg);
					usage();
				}
			} else
				identify_mode = IDENTIFY_EXTENDED;
			break;
		case 'g':
			if (operation != OP_NONE) {
				fprintf(stderr, "multiple operations specified on command line\n");
				usage();
			}

			operation = OP_PRINT_GUIDS;
			break;
		case 's':
			if (operation != OP_NONE) {
				fprintf(stderr, "multiple operations specified on command line\n");
				usage();
			}

			operation = OP_DOWNLOAD;	/* Save firmware from HCA to disk */
			filename = optarg;
			break;
		case 'd':
			if (operation != OP_NONE && operation != OP_MODIFY_OPTIONS) {
				fprintf(stderr, "multiple operations specified on command line\n");
				usage();
			}

			operation = OP_UPLOAD;	/* Download firmware to HCA */
			filename = optarg;
			break;
		case 'o':
			if (operation != OP_NONE && operation != OP_UPLOAD) {
				fprintf(stderr, "multiple operations specified on command line\n");
				usage();
			}

			if (optarg)
				change_option_cmd = optarg;
			else
				change_option_cmd = "";

			/* operation will be set below */
			break;
		}
	}

	if (operation == OP_NONE) {
		if (change_option_cmd) {
			operation = OP_MODIFY_OPTIONS;
		} else {
			/* Default to OP_UPLOAD, but we need a filename argument */
			if (optind >= argc)
				usage();

			operation = OP_UPLOAD;	/* Download firmware to HCA */
			filename = argv[optind++];
		}
	}

	/* The upload operation can take an optional GUID argument too */
	if (optind < argc && operation == OP_UPLOAD)
		guid = argv[optind++];

	/* Identify operation can take an optional filename argument here */
	if (optind < argc && operation == OP_IDENTIFY) {
		filename = argv[optind++];
		if (filename && identify_mode == IDENTIFY_HARDWARE_LABEL) {
			fprintf(stderr, "cannot print hardware label for firmware image file\n");
			usage();
		}
	}

	/* Change option command can take an optional filename argument here */
	if (optind < argc && change_option_cmd && change_option_cmd[0] == 0)
		change_option_cmd = argv[optind++];

	/* Set change_option_cmd correctly */
	if (!autoupgrade) {
		if (change_option_cmd && change_option_cmd[0] != 0) {
			size_t len = strlen(change_option_cmd) + strlen("auto_upgrade=off") + 1;
			char *p;

			p = malloc(len);
			if (!p) {
				fprintf(stderr, "Unable to allocate %Zd bytes for change_option_cmd\n",
					len);
				exit(1);
			}

			snprintf(p, len - 1, "%s,auto_upgrade=off",
				change_option_cmd);
			change_option_cmd = p;
		} else
			change_option_cmd = "auto_upgrade=off";
	}

	pacc = pci_alloc();
	pci_init(pacc);
	pci_scan_bus(pacc);

	switch (operation) {
	case OP_IDENTIFY:
		if (filename)
			ret = identify_firmware(filename, identify_mode);
		else {
			scan_devices(pacc);
			ret = identify_hcas(hca, identify_mode);
		}
		break;
	case OP_PRINT_GUIDS:
		scan_devices(pacc);
		ret = print_guids(hca);
		break;
	case OP_DOWNLOAD:
		scan_devices(pacc);
		ret = download_firmware(hca >= 0 ? hca : 0, filename);
		break;
	case OP_UPLOAD:
		scan_devices(pacc);
		ret = upload_firmware(hca >= 0 ? hca : 0, filename, guid, change_option_cmd);
		break;
	case OP_MODIFY_OPTIONS:
		scan_devices(pacc);
		ret = modify_options(hca >= 0 ? hca : 0, change_option_cmd);
		break;
	default:
		fprintf(stderr, "ERROR: Invalid operation %d\n", operation);
		ret = 1;
		break;
	}

	pci_cleanup(pacc);

	return ret;
}
