/*
 * Copyright (C) 2023, KylinSoft Co., Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
**/
#include "wallpaperinterface.h"

WallpaperInterface::WallpaperInterface()
{
    WorkerObject *workThread = new WorkerObject(this);
    connect(workThread, &WorkerObject::wallpaperReaded, this, [=](QStringList previewFiles, QStringList sourceFiles) {
        previewWallpaperList = previewFiles;
        sourceWallpaperList  = sourceFiles;
    });
    workThread->start();

    mWallpaperMode.clear();
    QString filename = UniversalInterface::self()->mateBackgroundGsettings()->get("picture-filename").toString();
    if (filename.isEmpty()) {
        mWallpaperMode = "color";
    } else {
        mWallpaperMode = "picture";
    }

    mDisplayType.clear();
    mDisplayType = UniversalInterface::self()->mateBackgroundGsettings()->get("picture-options").toString();

    mWallpaper.clear();
    if (mWallpaperMode == "color") {
        mWallpaper = UniversalInterface::self()->mateBackgroundGsettings()->get("primary-color").toString();
    } else {
        mWallpaper = UniversalInterface::self()->accountUserDbus()->property("BackgroundFile").toString();
        if (mWallpaper.isEmpty() || !QFile::exists(mWallpaper) ||
            mWallpaper == "/usr/share/backgrounds/ubuntukylin-default-settings.jpg") {
            mWallpaper = "/usr/share/backgrounds/1-warty-final-ubuntukylin.jpg";
        }
    }
    connect(UniversalInterface::self()->mateBackgroundGsettings(), &QGSettings::changed, this, [=](QString key){
        QString filename = UniversalInterface::self()->mateBackgroundGsettings()->get("picture-filename").toString();
        if (filename.isEmpty()) {
            mWallpaperMode = "color";
        } else {
            mWallpaperMode = "picture";
        }
        if (key == "pictureOptions") {
            mDisplayType = UniversalInterface::self()->mateBackgroundGsettings()->get("picture-options").toString();
            Q_EMIT changed("displayType");
        } else if(key == "pictureFilename") {
            if (mWallpaperMode == "color") {
                mWallpaper = UniversalInterface::self()->mateBackgroundGsettings()->get("primary-color").toString();
            } else {
                mWallpaper = UniversalInterface::self()->accountUserDbus()->property("BackgroundFile").toString();
                if (mWallpaper.isEmpty() || !QFile::exists(mWallpaper) ||
                    mWallpaper == "/usr/share/backgrounds/ubuntukylin-default-settings.jpg") {
                    mWallpaper = "/usr/share/backgrounds/1-warty-final-ubuntukylin.jpg";
                }
            }
            Q_EMIT changed("wallpaper");
        }
    });
}

WallpaperInterface::~WallpaperInterface()
{

}

QString WallpaperInterface::getWallpaperMode()
{
    return mWallpaperMode;
}

QString WallpaperInterface::getDisplayType()
{
    return mDisplayType;
}

QString WallpaperInterface::getWallpaper()
{
    return mWallpaper;
}

QStringList WallpaperInterface::getPreviewWallpapers()
{
    return previewWallpaperList;
}

QStringList WallpaperInterface::getSourceWallpapers()
{
    return sourceWallpaperList;
}

void WallpaperInterface::setDisplayType(QString type)
{
    UniversalInterface::self()->mateBackgroundGsettings()->set("picture-options", type);
    Q_EMIT changed("displayType");
}

void WallpaperInterface::setWallpaper(QString mode, QString data)
{
    changeGlobalTheme();
    QString name = qgetenv("USER");
    if (name.isEmpty()) {
        name = qgetenv("USERNAME");
    }
    QString confFilename = "/var/lib/lightdm-data/" + name + "/ukui-greeter.conf";
    QSettings qsetting(confFilename, QSettings::IniFormat);
    qsetting.beginGroup("greeter");
    if (mode == "color") {
        qsetting.setValue("color", data);
        UniversalInterface::self()->mateBackgroundGsettings()->set("picture-filename", "");
        UniversalInterface::self()->mateBackgroundGsettings()->set("primary-color", QVariant(data));
        UniversalInterface::self()->mateBackgroundGsettings()->set("secondary-color", QVariant(data));
    } else {
        qsetting.setValue("color", "");
        UniversalInterface::self()->accountUserDbus()->call("SetBackgroundFile", data);
        UniversalInterface::self()->mateBackgroundGsettings()->set("picture-filename", data);
    }
    qsetting.endGroup();
    Q_EMIT changed("wallpaper");
}

void WallpaperInterface::resetDefault()
{
    UniversalInterface::self()->mateBackgroundGsettings()->reset("picture-filename");
    setWallpaper("picture", UniversalInterface::self()->mateBackgroundGsettings()->get("picture-filename").toString());
}

void WallpaperInterface::changeGlobalTheme()
{
    if (QGSettings::isSchemaInstalled(QByteArray("org.ukui.globaltheme.settings"))) {
        QGSettings globalTheme("org.ukui.globaltheme.settings");
        globalTheme.set("global-theme-name", "custom");
    }
}
