/*
    utils.c - tools for The WebDruid

    Copyright (C) 2003-2004 Fabien Chevalier (fabien@juliana-multimedia.com)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
*/

#include "config.h"

/* local includes */

/* ensure sys/types */
#ifndef _SYS_TYPES_H
#include <sys/types.h>
#endif

/* some systems need this */
#ifdef HAVE_MATH_H
#include <math.h>
#endif

/* SunOS 4.x Fix */
#ifndef CLK_TCK
#define CLK_TCK _SC_CLK_TCK
#endif

#include <stdio.h>
#include <stdlib.h> /* atexit */
#include <ctype.h> /* isspace */
#include <string.h>
#include <errno.h>
#include <sys/wait.h>
#include <signal.h>
#include <langinfo.h> /* nl_langinfo */
#include <iconv.h>    /* iconv_xxxxx */
#include <sys/stat.h> /* fstat       */
#include <unistd.h>   /* fstat       */

#include "webdruid.h"
#include "lang.h"

/*********************************************/
/* strtolower_utf8                            */
/*********************************************/

/*
   convert an uft8 string to lower case

   This is not as simple as it should be.
   I do not know any solution for generic
   UNIX systems to do this.
   For systems with GNU glibc >= 2.3, solution
   might be to use the following GNU extensions:
     - newlocale (locale.h)
     - tolower_l (ctype.h)
     - freelocale (locale.h)

   -- I would require more advice. For now, i will
   only deal with ascii characters.

   Fabien. 20 aug 2003

*/

void strtolower_utf8(char *utf8_str)
{
   if(utf8_str)
      while(*utf8_str)
      {
         if(*utf8_str >= 'A' && *utf8_str <= 'Z')
            *utf8_str += 'a' - 'A';
         utf8_str++;
      }
}


/*********************************************/
/* strip_spaces                              */
/*********************************************/

/*
  removes a string's trailing and leading spaces
*/

void strip_spaces(char** str)
{
   char *end;

   /* strip heading spaces */
   while (isspace(**str)) (*str)++;
   end = *str + strlen(*str) - 1;
   /* strip trailing spaces */
   while((*str != end) && isspace(*end)) end--[0] = 0;
}

/*********************************************/
/* HASH - return hash value for string       */
/*********************************************/

/*
  note thath the value is not constrained
*/

u_long hash(char *str)
{
   u_long hashval;
   for (hashval = 0; *str != '\0'; str++)
      hashval = *str + 31 * hashval;
   return hashval;
}

/*********************************************/
/* PROGRESSIVE_HASH                          */
/*********************************************/

/*
  Calculates hash on a set of strings
  progressively.
  To be used like this:
    u_long hash;
    hash = progrssive_hash(str1, 0);
    hash = progressive_hash(str2, hash);char *tag
    ...
    hash = progressive_hash(strn, hash);
*/

u_long progressive_hash(char *str, u_long hashval)
{
   for (; *str != '\0'; str++)
      hashval = *str + 31 * hashval;
   return hashval;
}

/*********************************************/
/* OPEN_OUT_FILE - Open file for output      */
/*********************************************/

FILE *open_out_file(char *filename)
{
   FILE *out_fp;

   /* open the file... */
   if ( (out_fp=fopen(filename,"w")) == NULL)
   {
      if (verbose)
      fprintf(stderr,"%s %s!\n",_("Error: Unable to open file"),filename);
      return NULL;
   }
   return out_fp;
}

/*********************************************/
/* OPEN_OUT_FILE_IN_DIR - does what it says  */
/*********************************************/

FILE *open_out_file_in_dir(const char *dir,const char *filename)
{
   FILE *out_fp;
   struct stat st;
   char buff[256];

   if(stat("xml", &st) == 0)
   {
      if(!S_ISDIR(st.st_mode))
      {
         if (verbose)
            fprintf(stderr,"%s %s!\n",_("Error: not a directory! :"),dir);
         return NULL;
      }
   }
   else
   {
      if(mkdir(dir, S_IRWXU | S_IRWXG | S_IRWXO) != 0)
      {
         if (verbose)
            fprintf(stderr,_("Error: Could not create subdirectory %s in output dir: %s"), dir, strerror(errno));
         return NULL;
      }
   }

   snprintf(buff, sizeof(buff), "%s/%s", dir, filename);

   /* open the file... */
   if ( (out_fp=fopen(buff,"w")) == NULL)
   {
      if (verbose)
         fprintf(stderr,"%s %s!\n",_("Error: Unable to open file"),buff);
      return NULL;
   }
   return out_fp;
}



/*********************************************/
/* DUMP_LOGREC - debugging purpose           */
/*********************************************/

   /*snprintf(log_record->datetime,sizeof(log_record->datetime),
           "[%02d/%s/%4d:%s -0000]",i,cpx,j,cpy);*/

void dump_logrec(FILE * stream)
{
   fprintf(stream, "hostname: %s\n", log_rec.hostname);
   fprintf(stream, "datetime: [%02d/%02d/%4d:%02d:%02d:%02d -0000]\n",
       log_rec.year, log_rec.month, log_rec.day, log_rec.hour, log_rec.min, log_rec.sec);
   fprintf(stream, "url: %s\n", log_rec.url);
   fprintf(stream, "resp_code: %d\n", log_rec.resp_code);
   fprintf(stream, "xfer_size: %lu\n", log_rec.xfer_size);
   fprintf(stream, "referrer: %s\n", log_rec.refer);
   fprintf(stream, "srch_string: %s\n", log_rec.srchstr);
}

/*****************************************************************/
/*                                                               */
/* JDATE  - Julian date calculator                               */
/*                                                               */
/* Calculates the number of days since Jan 1, 0000.              */
/*                                                               */
/* Originally written by Bradford L. Barrett (03/17/1988)        */
/* Returns an unsigned long value representing the number of     */
/* days since January 1, 0000.                                   */
/*                                                               */
/* Note: Due to the changes made by Pope Gregory XIII in the     */
/*       16th Centyry (Feb 24, 1582), dates before 1583 will     */
/*       not return a truely accurate number (will be at least   */
/*       10 days off).  Somehow, I don't think this will         */
/*       present much of a problem for most situations :)        */
/*                                                               */
/* Usage: days = jdate(day, month, year)                         */
/*                                                               */
/* The number returned is adjusted by 5 to facilitate day of     */
/* week calculations.  The mod of the returned value gives the   */
/* day of the week the date is.  (ie: dow = days % 7 ) where     */
/* dow will return 0=Sunday, 1=Monday, 2=Tuesday, etc...         */
/*                                                               */
/*****************************************************************/

u_long jdate( int day, int month, int year )
{
   u_long days;                      /* value returned */
   int mtable[] = {0,31,59,90,120,151,181,212,243,273,304,334};

   /* First, calculate base number including leap and Centenial year stuff */

   days=(((u_long)year*365)+day+mtable[month-1]+
           ((year+4)/4) - ((year/100)-(year/400)));

   /* now adjust for leap year before March 1st */

   if ((year % 4 == 0) && !((year % 100 == 0) &&
       (year % 400 != 0)) && (month < 3))
   --days;

   /* done, return with calculated value */

   return(days+5);
}

/*****************************************************************/
/* _2UTF8 converts given string from current charset to utf8     */
/*****************************************************************/

/*
   Use is rather simple.

   For a string that you would write in your program as _("Foo string"),
   You just have to replace this by _2utf8(_("Foo string"))

   Beware though that this function use static buffers to store its convertions.
   The buffers are used one after the other, and when the last one has been
   used, the first one is used again -- this is a circular use of buffers.

   The maximum number of buffers limits the number of times you can use this function
   as result for the same function call.

   whith 3 buffers, you can do:

     printf("%s %s %s", _2utf8(_("String 1")), _2utf8(_("String 2")),
        _2utf8(_("String 3")))

   but you cannot do:

     printf("%s %s %s %s", _2utf8(_("String 1")), _2utf8(_("String 2")),
        _2utf8(_("String 3")), _2utf8(_("String 4")))
*/


/* number of buffers */
#define NBUFFS 5

/* buffers len */
#define BUFFSLEN 1024

/* internal iconv handle */
iconv_t utf8_iconv_h = 0;

/* internal free function */
static void free_utf8_iconv_h()
{
   iconv_close(utf8_iconv_h);
   utf8_iconv_h = 0;
}

char * _2utf8(char * from)
{
   static char bufs[NBUFFS][BUFFSLEN];
   static int curbuf = 0;
   static char * charset = NULL;
   size_t in_size;
   size_t out_size;
   char *in, *out;

   if(charset == NULL)
      charset = nl_langinfo(CODESET);

   if(strcasecmp("UTF-8", charset) != 0)
   {
      if(utf8_iconv_h == 0)
      {
         utf8_iconv_h = iconv_open("UTF-8", charset);
         if(utf8_iconv_h == 0)
         {
            fprintf(stderr, _("_2utf8: cannot open iconv handle %s->UTF8: %s\n"), charset, strerror(errno));
            /* do for best */
            return from;
         }
         else
            atexit(free_utf8_iconv_h);
      }

      curbuf = (curbuf + 1) % NBUFFS;
      in_size = strlen(from) + 1;
      out_size = BUFFSLEN;
      in = from;
      out = bufs[curbuf];

      if(iconv(utf8_iconv_h, &in, &in_size, &out, &out_size) == -1)
      {
         fprintf(stderr, _("_2utf8: cannot convert string %s from %s to UTF8: %s\n"), from, charset, strerror(errno));
         /* reset iconv state -- cf man 3 iconv */
         in = NULL; out = NULL;
         iconv(utf8_iconv_h, &in, &in_size, &out, &out_size);
         return from;
      }
      else
         return bufs[curbuf];
   }
   else
      return from;
}

/*********************************************/
/* SAFE_RUN                                  */
/*********************************************/

/* runs given executable savely -
   this will kill the child after timeout seconds
   if it hasn't finished yet

   if timeout equals 0, timer is disabled

   returns:
     - 0 if Ok
     - 1 if sth went wrong
*/

/* local variable */
static pid_t safe_run_pid;

void do_timeout(int value)
{
   kill(safe_run_pid, SIGKILL);
   fprintf(stderr,"[safe_run] %s \n",_("Error: killed child, timeout elapsed"));
}

int safe_run(const char *path, char *const argv[], int timeout)
{
   safe_run_pid = fork();

   if(safe_run_pid < 0)
   {
      /* error */
      fprintf(stderr,"[safe_run] %s : %s\n",_("Error: could not fork :"), strerror(errno));
      return 1;
   }
   else if(safe_run_pid == 0)
   {
      /* son */
      execv(path, argv);
      /* should never get there */
      fprintf(stderr,"[safe_run] %s : %s : %s\n",_("Error: could not exec"),
                path, strerror(errno));
      return 1;

   }
   else /* safe_run_pid > 0 */
   {
      /* father */
      int status;

      signal(SIGALRM, &do_timeout);
      alarm(timeout);
      waitpid(safe_run_pid, &status, 0);
      alarm(0);
      signal(SIGALRM, SIG_DFL);

      if(WIFSIGNALED(status))
         return 1;
      else
         return WEXITSTATUS(status);
   }
}
