/*
 *========================================================================
 * $Id: llists.c 87 2004-09-28 17:31:03Z rgb $
 *
 * See copyright in copyright.h and the accompanying file COPYING
 *========================================================================
 */


#include <wulfware/libwulf_llist.h>

/*
 * This is the working/public interface to manipulation of the
 * linked lists used in xmlsysd.  The routines are pretty simple.
 *
 * List *new_list();
 *   allocates a List struct at the list address.
 * void add_list(List *list,void *data,int size)
 *   allocates a list element (including room for the data element
 *   it must hold) and adds it to the tail of newlist. Updates list->size;
 * void rm_list(List *list,void *data,int size);
 *   searches the list for an element containing data, puts its
 *   tail address into the tail address of the preceding element,
 *   frees the data memory, then frees the element memory.
 * delete_list(List *list);
 *   walks the list from the tail to the head, removing each element
 *   it finds.  Does NOT deallocate the list -- this must be done
 *   with free if desired.
 *
 * We can easily add commands but this is probably enough four our
 * purposes.
 */

List *newlist()
{

 List *ret;

 ret = (List *)malloc((size_t) sizeof(List));
 if(ret == NULL){
   fprintf(stderr,"Error: Out of memory in newlist()\n");
   exit(0);
 }
 /* zero the list contents */
 ret->size = 0;
 ret->head =  NULL;
 ret->tail =  NULL;

 return ret;

}

void add_list(List *list,void *data,int size)
{

 ListElement *new,*cur;

 /*
  * First allocate a new element.
  */
 new = (ListElement *) malloc((size_t) sizeof(ListElement));
 if(new == NULL){
   fprintf(stderr,"Error: Out of memory in add_list()\n");
   exit(0);
 }

 /*
  * Allocate room for data in the new element.  Include room for
  * a terminating null (always).
  */
 new->data = (void *)malloc((size_t) size);
 if(new->data == NULL){
   fprintf(stderr,"Error: Out of memory in add_list()\n");
   exit(0);
 }
 memcpy(new->data,data,size);

 /*
  * new now is an element containing the data.  Time to link it into
  * the list.
  */

 /*
  * Is the list empty?  If so we put it at the head (and tail)
  * and cause the element's prev and next to point to NULL.
  */
 if(list->size == 0){
  list->head = new;
  list->tail = new;
  list->size = 1;
  new->prev = NULL;
  new->next = NULL;
 } else {
   /*
    * Otherwise we simply add the new element at the end.
    */
   list->tail->next = new;	/* next of prev element */
   new->prev = list->tail;	/* prev of new element */
   list->tail = new;		/* tail of list */
   new->next = NULL;		/* next of new element */
   list->size++;			/* one more element all done. */
 }

}


/*
 * We want to remove only ONE instance of any given entry from the
 * list, I think -- if we can add more than one in the first place
 * (one at a time) we should have to remove them one at a time as well,
 * or we should uniqueify the additions.  We'll therefore break out
 * of the remove process instead of completing the loop.
 */
void rm_list(List *list,void *data,int size)
{

 int i;
 ListElement *element;

 i = 0;
 element = list->head;
 while (element != NULL) {
   /* 
    * Have to figure out where we are going before dumping element.
    * We do this here as element might not exist at the bottom.
    */
   if(strncmp((char *)element->data,(char *)data,size) == 0){
     /* 
      * They match! Remove this element from the list.  First free
      * the data.
      */
     free(element->data);
     /*
      * We now have a nice little logic problem.  The element we're
      * freeing could be the only element in the list (so its next and
      * prev are both null and it is both head and tail).  It could
      * be head (prev is null).  It could be tail (next is null).
      * OR it could be in the middle: neither head nor tail, with
      * a non-null prev and next.
      *
      * Each of these has its own peculiar set of actions we have to
      * take to not break the linkage on removal.
      */

     /* only element left -- NULL list head and tail */
     if(element == list->head && element == list->tail) {
       list->head = NULL;
       list->tail = NULL;
     } else
     /* head of non-empty list -- set head to next, set prev of next to null */
     if(element == list->head){
       list->head = element->next;
       element->next->prev = NULL;
     } else
     /* tail of non-empty list -- set tail to prev, set next of prev to null */
     if(element == list->tail){
       list->tail = element->prev;
       element->prev->next = NULL;
     } else {
       /* 
        * If we get here, we're not the head OR the tail OR the
        * head AND tail, so we must be a mid-element of non-empty 
        * list -- we therefore join prev->next and next->prev 
        */
       element->prev->next = element->next;
       element->next->prev = element->prev;
     }
     /* Decrement the count of list */
     list->size--;
     /* FINALLY, free the element */
     free(element);
     break;
   }
   /* Go on to the next element */
   element = element->next;
   i++;
 }

}

/*
 * This doesn't de-initialize the list, only clear all its elements
 */
void delete_list(List *list)
{

 int i;
 ListElement *element,*previous_element;

 /*
  * We'll start at the end and work backwards -- why not?
  */
 i = list->size;
 element = list->tail;
 while (element != NULL) {
   free(element->data);
   previous_element = element->prev;
   free(element);
   /* Go on to the next element */
   element = previous_element;
   i--;
 }
 /* Now reset the list itself to be properly empty */
 list->size = 0;
 list->head = NULL;
 list->tail = NULL;

}

