# This file is part of the YAZ toolkit
# Copyright (C) Index Data
# See the file LICENSE for details.
#
#
# Converts a CSV file with diagnostics to C+H file for easy
# maintenance

proc csvtodiag {ifiles name alias} {
    set uname [string toupper $name]
    set funcproto "const char *yaz_diag_${name}_str(int code)"
    if {[string length $alias]} {
	set funcalias "const char *${alias}(int code)"
    }
    set csv [open [lindex $ifiles 0] r]
    set cfile [open [lindex $ifiles 1] w]
    set hfile [open [lindex $ifiles 2] w]
    set lineno 0
    puts $cfile "/** \\file [lindex $ifiles 1]"
    puts $hfile "/** \\file [lindex $ifiles 2]"
    set preamble "    \\brief Diagnostics: Generated by csvtodiag.tcl from [lindex $ifiles 0] */"
    puts $cfile $preamble
    puts $cfile "\#ifdef HAVE_CONFIG_H"
    puts $cfile "\#include <config.h>"
    puts $cfile "\#endif"

    puts $cfile "
#include <diag-entry.h>
\#include \"[lindex $ifiles 2]\"
struct yaz_diag_entry yaz_diag_${name}_tab\[\] = \{
"
    puts $hfile $preamble
    puts $hfile "
\#include <yaz/yconfig.h>

\#ifndef YAZ_DIAG_${name}_H
\#define YAZ_DIAG_${name}_H
YAZ_BEGIN_CDECL
YAZ_EXPORT $funcproto;"

    if {[info exists funcalias]} {
         puts $hfile "YAZ_EXPORT $funcalias;"
}
    while {1} {
	incr lineno
	set cnt [gets $csv line]
	if {$cnt < 0} {
	    break
	}
	if {[regexp {([0-9]+)[^\"]*"([^\"]*)"} $line s code msg]} {
	    puts $cfile "\{$code, \"$msg\"\},"

	    set m [string toupper $msg]
	    regsub -all {DUPLICATE} $m {DUP} m
	    regsub -all {SECURITY CHALLENGE} $m {SEC_CHAL} m
	    regsub -all {COULD NOT} $m {COULDNT} m
	    regsub -all {COULD NOT} $m {COULDNT} m
	    regsub -all {NOT SUPPORTED} $m {UNSUPP} m
	    regsub -all {UNSUPPORTED} $m {UNSUPP} m
	    regsub -all {COMBINATION} $m {COMBI} m
	    regsub -all {PROXIMITY} $m {PROX} m
	    regsub -all {CHARACTERS} $m {CHARS} m
	    regsub -all {CHARACTER} $m {CHAR} m
	    regsub -all {[-/,:;."' \{\}()]} $m _ m
            set m [string map {___ _ __ _} $m]
	    if {[string length $m] > 55} {
	    	set m [string range $m 0 55]
		set p [string last _ $m]
		if {$p > 30} {
		    set m [string range $m 0 $p]
		}
	    }
            puts $hfile "\#define YAZ_${uname}_${m} $code"
	}
    }
    puts $cfile "\{0, 0\}\}\;"
    puts $cfile $funcproto
    puts $cfile "\{"
    puts $cfile "    return yaz_diag_to_str(yaz_diag_${name}_tab, code);"
    puts $cfile "\}"

    if {[info exists funcalias]} {
        puts $cfile $funcalias
        puts $cfile "\{"
        puts $cfile "    return yaz_diag_to_str(yaz_diag_${name}_tab, code);"
        puts $cfile "\}"
    }
    puts $hfile "
YAZ_END_CDECL
\#endif
"
    close $csv
    close $cfile
    close $hfile
}

proc csvto_xml {csvfname xmlfname id} {
    set xfile [open $xmlfname w]
    set csv [open $csvfname r]

    puts $xfile "<!-- Generated by csvtodiag.tcl from $csvfname -->"
    puts $xfile "<informaltable id=\"${id}\">"
    puts $xfile {<tgroup cols="2">}
    puts $xfile {<colspec colwidth="1*" colname="code"></colspec>}
    puts $xfile {<colspec colwidth="4*" colname="text"></colspec>}
    puts $xfile {<thead>}
    puts $xfile {<row>}
    puts $xfile {<entry>Code</entry>}
    puts $xfile {<entry>Text</entry>}
    puts $xfile {</row>}
    puts $xfile {</thead>}
    puts $xfile {<tbody>}

    set lineno 0
    while {1} {
	incr lineno
	set cnt [gets $csv line]
	if {$cnt < 0} {
	    break
	}
	if {[regexp {([0-9]+)[^\"]*"([^\"]*)"} $line s code msg]} {
	    puts $xfile {<row>}
	    puts $xfile {<entry>}
	    puts $xfile $code
	    puts $xfile {</entry><entry>}
	    puts $xfile $msg
	    puts $xfile {</entry>}
	    puts $xfile {</row>}
	}
    }
    puts $xfile {</tbody>}
    puts $xfile {</tgroup>}
    puts $xfile {</informaltable>}

    close $xfile
    close $csv
}

if {[llength $argv] >= 4} {
    set alias {}
    if {[llength $argv] >= 5} {
	set alias [lindex $argv 4]
    }
    csvtodiag [list [lindex $argv 0] [lindex $argv 1] [lindex $argv 2]] \
	[lindex $argv 3] $alias
} elseif {[llength $argv] == 3} {
    csvto_xml [lindex $argv 0] [lindex $argv 1] [lindex $argv 2]
}
