/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.sting.utils;

import com.google.java.contract.Ensures;
import com.google.java.contract.Requires;
import com.google.java.contract.ThrowEnsures;
import net.sf.picard.reference.ReferenceSequenceFile;
import net.sf.samtools.SAMRecord;
import net.sf.samtools.SAMSequenceDictionary;
import net.sf.samtools.SAMSequenceRecord;
import org.apache.log4j.Logger;
import org.broad.tribble.Feature;
import org.broadinstitute.sting.utils.GenomeLoc;
import org.broadinstitute.sting.utils.exceptions.ReviewedStingException;
import org.broadinstitute.sting.utils.exceptions.UserException;
import org.broadinstitute.sting.utils.variantcontext.VariantContext;

public final class GenomeLocParser {
    private static Logger logger = Logger.getLogger(GenomeLocParser.class);
    private final SAMSequenceDictionary SINGLE_MASTER_SEQUENCE_DICTIONARY;
    private final ThreadLocal<CachingSequenceDictionary> contigInfoPerThread = new ThreadLocal();

    private CachingSequenceDictionary getContigInfo() {
        if (this.contigInfoPerThread.get() == null) {
            this.contigInfoPerThread.set(new CachingSequenceDictionary(this.SINGLE_MASTER_SEQUENCE_DICTIONARY));
        }
        assert (this.contigInfoPerThread.get() != null);
        return this.contigInfoPerThread.get();
    }

    @Requires(value={"refFile != null"})
    public GenomeLocParser(ReferenceSequenceFile refFile) {
        this(refFile.getSequenceDictionary());
    }

    public GenomeLocParser(SAMSequenceDictionary seqDict) {
        if (seqDict == null) {
            throw new UserException.CommandLineException("Failed to load reference dictionary");
        }
        this.SINGLE_MASTER_SEQUENCE_DICTIONARY = seqDict;
        logger.debug((Object)String.format("Prepared reference sequence contig dictionary", new Object[0]));
        for (SAMSequenceRecord contig : seqDict.getSequences()) {
            logger.debug((Object)String.format(" %s (%d bp)", contig.getSequenceName(), contig.getSequenceLength()));
        }
    }

    public final boolean contigIsInDictionary(String contig) {
        return contig != null && this.getContigInfo().hasContig(contig);
    }

    public final boolean indexIsInDictionary(int index) {
        return index >= 0 && this.getContigInfo().hasContig(index);
    }

    @Ensures(value={"result != null"})
    @ThrowEnsures(value={"UserException.MalformedGenomeLoc", "!contigIsInDictionary(contig) || contig == null"})
    public final SAMSequenceRecord getContigInfo(String contig) {
        if (contig == null || !this.contigIsInDictionary(contig)) {
            throw new UserException.MalformedGenomeLoc(String.format("Contig %s given as location, but this contig isn't present in the Fasta sequence dictionary", contig));
        }
        return this.getContigInfo().getSequence(contig);
    }

    @Ensures(value={"result >= 0"})
    @ThrowEnsures(value={"UserException.MalformedGenomeLoc", "!contigIsInDictionary(contig) || contig == null"})
    public final int getContigIndex(String contig) {
        return this.getContigInfo(contig).getSequenceIndex();
    }

    @Requires(value={"contig != null"})
    protected int getContigIndexWithoutException(String contig) {
        if (contig == null || !this.getContigInfo().hasContig(contig)) {
            return -1;
        }
        return this.getContigInfo().getSequenceIndex(contig);
    }

    public final SAMSequenceDictionary getContigs() {
        return this.getContigInfo().dict;
    }

    @Ensures(value={"result != null"})
    @ThrowEnsures(value={"UserException.MalformedGenomeLoc", "!isValidGenomeLoc(contig, start, stop)"})
    public GenomeLoc createGenomeLoc(String contig, int start, int stop) {
        return this.createGenomeLoc(contig, this.getContigIndex(contig), start, stop);
    }

    public GenomeLoc createGenomeLoc(String contig, int start, int stop, boolean mustBeOnReference) {
        return this.createGenomeLoc(contig, this.getContigIndex(contig), start, stop, mustBeOnReference);
    }

    @ThrowEnsures(value={"UserException.MalformedGenomeLoc", "!isValidGenomeLoc(contig, start, stop, false)"})
    public GenomeLoc createGenomeLoc(String contig, int index, int start, int stop) {
        return this.createGenomeLoc(contig, index, start, stop, false);
    }

    @ThrowEnsures(value={"UserException.MalformedGenomeLoc", "!isValidGenomeLoc(contig, start, stop,mustBeOnReference)"})
    public GenomeLoc createGenomeLoc(String contig, int index, int start, int stop, boolean mustBeOnReference) {
        this.validateGenomeLoc(contig, index, start, stop, mustBeOnReference, true);
        return new GenomeLoc(contig, index, start, stop);
    }

    private boolean validateGenomeLoc(String contig, int contigIndex, int start, int stop, boolean mustBeOnReference, boolean exceptOnError) {
        if (!this.getContigInfo().hasContig(contig)) {
            return this.vglHelper(exceptOnError, String.format("Unknown contig %s", contig));
        }
        if (stop < start) {
            return this.vglHelper(exceptOnError, String.format("The stop position %d is less than start %d in contig %s", stop, start, contig));
        }
        if (contigIndex < 0) {
            return this.vglHelper(exceptOnError, String.format("The contig index %d is less than 0", contigIndex));
        }
        if (contigIndex >= this.getContigInfo().getNSequences()) {
            return this.vglHelper(exceptOnError, String.format("The contig index %d is greater than the stored sequence count (%d)", contigIndex, this.getContigInfo().getNSequences()));
        }
        if (mustBeOnReference) {
            if (start < 1) {
                return this.vglHelper(exceptOnError, String.format("The start position %d is less than 1", start));
            }
            if (stop < 1) {
                return this.vglHelper(exceptOnError, String.format("The stop position %d is less than 1", stop));
            }
            int contigSize = this.getContigInfo().getSequence(contigIndex).getSequenceLength();
            if (start > contigSize || stop > contigSize) {
                return this.vglHelper(exceptOnError, String.format("The genome loc coordinates %d-%d exceed the contig size (%d)", start, stop, contigSize));
            }
        }
        return true;
    }

    public boolean isValidGenomeLoc(String contig, int start, int stop, boolean mustBeOnReference) {
        return this.validateGenomeLoc(contig, this.getContigIndexWithoutException(contig), start, stop, mustBeOnReference, false);
    }

    public boolean isValidGenomeLoc(String contig, int start, int stop) {
        return this.validateGenomeLoc(contig, this.getContigIndexWithoutException(contig), start, stop, true, false);
    }

    private boolean vglHelper(boolean exceptOnError, String msg) {
        if (exceptOnError) {
            throw new UserException.MalformedGenomeLoc("Parameters to GenomeLocParser are incorrect:" + msg);
        }
        return false;
    }

    @Requires(value={"str != null"})
    @Ensures(value={"result != null"})
    public GenomeLoc parseGenomeLoc(String str) {
        String contig = null;
        int start = 1;
        int stop = -1;
        int colonIndex = str.indexOf(":");
        if (colonIndex == -1) {
            contig = str.substring(0, str.length());
            stop = Integer.MAX_VALUE;
        } else {
            contig = str.substring(0, colonIndex);
            int dashIndex = str.indexOf(45, colonIndex);
            try {
                if (dashIndex == -1) {
                    if (str.charAt(str.length() - 1) == '+') {
                        start = this.parsePosition(str.substring(colonIndex + 1, str.length() - 1));
                        stop = Integer.MAX_VALUE;
                    } else {
                        stop = start = this.parsePosition(str.substring(colonIndex + 1));
                    }
                } else {
                    start = this.parsePosition(str.substring(colonIndex + 1, dashIndex));
                    stop = this.parsePosition(str.substring(dashIndex + 1));
                }
            }
            catch (Exception e) {
                throw new UserException("Failed to parse Genome Location string: " + str, e);
            }
        }
        if (!this.contigIsInDictionary(contig)) {
            throw new UserException.MalformedGenomeLoc("Contig '" + contig + "' does not match any contig in the GATK sequence dictionary derived from the reference; are you sure you are using the correct reference fasta file?");
        }
        if (stop == Integer.MAX_VALUE) {
            stop = this.getContigInfo(contig).getSequenceLength();
        }
        return this.createGenomeLoc(contig, this.getContigIndex(contig), start, stop, true);
    }

    @Requires(value={"pos != null"})
    @Ensures(value={"result >= 0"})
    private int parsePosition(String pos) {
        if (pos.indexOf(45) != -1) {
            throw new NumberFormatException("Position: '" + pos + "' can't contain '-'.");
        }
        if (pos.indexOf(44) != -1) {
            StringBuilder buffer = new StringBuilder();
            for (int i = 0; i < pos.length(); ++i) {
                char c = pos.charAt(i);
                if (c == ',') continue;
                if (c < '0' || c > '9') {
                    throw new NumberFormatException("Position: '" + pos + "' contains invalid chars.");
                }
                buffer.append(c);
            }
            return Integer.parseInt(buffer.toString());
        }
        return Integer.parseInt(pos);
    }

    @Requires(value={"read != null"})
    @Ensures(value={"result != null"})
    public GenomeLoc createGenomeLoc(SAMRecord read) {
        if (read.getReadUnmappedFlag() && read.getReferenceIndex() == -1) {
            return GenomeLoc.UNMAPPED;
        }
        int end = read.getReadUnmappedFlag() ? read.getAlignmentStart() : Math.max(read.getAlignmentEnd(), read.getAlignmentStart());
        return this.createGenomeLoc(read.getReferenceName(), read.getReferenceIndex(), read.getAlignmentStart(), end, false);
    }

    public GenomeLoc createGenomeLoc(Feature feature) {
        return this.createGenomeLoc(feature.getChr(), feature.getStart(), feature.getEnd());
    }

    public GenomeLoc createGenomeLoc(VariantContext vc, boolean includeSymbolicEndIfPossible) {
        if (includeSymbolicEndIfPossible && vc.isSymbolic()) {
            int end = vc.getAttributeAsInt("END", vc.getEnd());
            return this.createGenomeLoc(vc.getChr(), vc.getStart(), end);
        }
        return this.createGenomeLoc(vc.getChr(), vc.getStart(), vc.getEnd());
    }

    public GenomeLoc createGenomeLoc(VariantContext vc) {
        return this.createGenomeLoc(vc, false);
    }

    @Ensures(value={"result != null"})
    @ThrowEnsures(value={"UserException.MalformedGenomeLoc", "!isValidGenomeLoc(contig, pos, pos, true)"})
    public GenomeLoc createGenomeLoc(String contig, int pos) {
        return this.createGenomeLoc(contig, this.getContigIndex(contig), pos, pos);
    }

    public GenomeLoc setStart(GenomeLoc loc, int start) {
        return this.createGenomeLoc(loc.getContig(), loc.getContigIndex(), start, loc.getStop());
    }

    public GenomeLoc setStop(GenomeLoc loc, int stop) {
        return this.createGenomeLoc(loc.getContig(), loc.getContigIndex(), loc.start, stop);
    }

    public GenomeLoc incPos(GenomeLoc loc) {
        return this.incPos(loc, 1);
    }

    public GenomeLoc incPos(GenomeLoc loc, int by) {
        return this.createGenomeLoc(loc.getContig(), loc.getContigIndex(), loc.start + by, loc.stop + by);
    }

    @Requires(value={"contigName != null"})
    @Ensures(value={"result != null"})
    public GenomeLoc createOverEntireContig(String contigName) {
        SAMSequenceRecord contig = this.getContigInfo().getSequence(contigName);
        return this.createGenomeLoc(contigName, contig.getSequenceIndex(), 1, contig.getSequenceLength(), true);
    }

    @Requires(value={"loc != null", "maxBasePairs > 0"})
    public GenomeLoc createGenomeLocAtStart(GenomeLoc loc, int maxBasePairs) {
        if (GenomeLoc.isUnmapped(loc)) {
            return null;
        }
        String contigName = loc.getContig();
        SAMSequenceRecord contig = this.getContigInfo().getSequence(contigName);
        int contigIndex = contig.getSequenceIndex();
        int start = loc.getStart() - maxBasePairs;
        int stop = loc.getStart() - 1;
        if (start < 1) {
            start = 1;
        }
        if (stop < 1) {
            return null;
        }
        return this.createGenomeLoc(contigName, contigIndex, start, stop, true);
    }

    @Requires(value={"loc != null", "padding > 0"})
    public GenomeLoc createPaddedGenomeLoc(GenomeLoc loc, int padding) {
        if (GenomeLoc.isUnmapped(loc)) {
            return loc;
        }
        String contigName = loc.getContig();
        SAMSequenceRecord contig = this.getContigInfo().getSequence(contigName);
        int contigIndex = contig.getSequenceIndex();
        int contigLength = contig.getSequenceLength();
        int start = Math.max(1, loc.getStart() - padding);
        int stop = Math.min(contigLength, loc.getStop() + padding);
        return this.createGenomeLoc(contigName, contigIndex, start, stop, true);
    }

    @Requires(value={"loc != null", "maxBasePairs > 0"})
    public GenomeLoc createGenomeLocAtStop(GenomeLoc loc, int maxBasePairs) {
        if (GenomeLoc.isUnmapped(loc)) {
            return null;
        }
        String contigName = loc.getContig();
        SAMSequenceRecord contig = this.getContigInfo().getSequence(contigName);
        int contigIndex = contig.getSequenceIndex();
        int contigLength = contig.getSequenceLength();
        int start = loc.getStop() + 1;
        int stop = loc.getStop() + maxBasePairs;
        if (start > contigLength) {
            return null;
        }
        if (stop > contigLength) {
            stop = contigLength;
        }
        return this.createGenomeLoc(contigName, contigIndex, start, stop, true);
    }

    private final class CachingSequenceDictionary {
        private final SAMSequenceDictionary dict;
        SAMSequenceRecord lastSSR = null;
        String lastContig = "";
        int lastIndex = -1;

        @Requires(value={"dict != null", "dict.size() > 0"})
        public CachingSequenceDictionary(SAMSequenceDictionary dict) {
            this.dict = dict;
        }

        @Ensures(value={"result > 0"})
        public final int getNSequences() {
            return this.dict.size();
        }

        @Requires(value={"contig != null"})
        public final synchronized boolean hasContig(String contig) {
            return contig.equals(this.lastContig) || this.dict.getSequence(contig) != null;
        }

        @Requires(value={"index >= 0"})
        public final synchronized boolean hasContig(int index) {
            return this.lastIndex == index || this.dict.getSequence(index) != null;
        }

        @Requires(value={"contig != null"})
        @Ensures(value={"result != null"})
        public final synchronized SAMSequenceRecord getSequence(String contig) {
            if (this.isCached(contig)) {
                return this.lastSSR;
            }
            return this.updateCache(contig, -1);
        }

        @Requires(value={"index >= 0"})
        @Ensures(value={"result != null"})
        public final synchronized SAMSequenceRecord getSequence(int index) {
            if (this.isCached(index)) {
                return this.lastSSR;
            }
            return this.updateCache(null, index);
        }

        @Requires(value={"contig != null"})
        @Ensures(value={"result >= 0"})
        public final synchronized int getSequenceIndex(String contig) {
            if (!this.isCached(contig)) {
                this.updateCache(contig, -1);
            }
            return this.lastIndex;
        }

        @Requires(value={"contig != null", "lastContig != null"})
        private synchronized boolean isCached(String contig) {
            return this.lastContig.equals(contig);
        }

        @Requires(value={"lastIndex != -1", "index >= 0"})
        private synchronized boolean isCached(int index) {
            return this.lastIndex == index;
        }

        @Requires(value={"contig != null || index >= 0"})
        @Ensures(value={"result != null"})
        private synchronized SAMSequenceRecord updateCache(String contig, int index) {
            SAMSequenceRecord rec;
            SAMSequenceRecord sAMSequenceRecord = rec = contig == null ? this.dict.getSequence(index) : this.dict.getSequence(contig);
            if (rec == null) {
                throw new ReviewedStingException("BUG: requested unknown contig=" + contig + " index=" + index);
            }
            this.lastSSR = rec;
            this.lastContig = rec.getSequenceName();
            this.lastIndex = rec.getSequenceIndex();
            return rec;
        }
    }
}

