// Copyright (c) Microsoft Open Technologies, Inc. All rights reserved. See License.txt in the project root for license information.
#pragma warning disable 414

//#####################################################################
// namespace: System.Data.Entity.Core.Common.EntitySql
//#####################################################################

namespace System.Data.Entity.Core.Common.EntitySql
{
    using System.Data.Entity.Core.Common.EntitySql.AST;
    using System.Data.Entity.Resources;
    using System.Diagnostics;

//#####################################################################
// class: CqlParser
// does : encapsulates yacc() parser functionality in a C# class
//#####################################################################
    internal partial class CqlParser
    {
        private readonly Boolean yydebug; //do I want debug output?
        private static int YYMAJOR = 1;
        private static int YYMINOR = 9;
        private int yynerrs; //number of errors so far
        private int yyerrflag; //was there an error?
        private int yychar; //the current working character

//########## MESSAGES ##########
//###############################################################
// method: debug
//###############################################################
        private void debug(String msg)
        {
            if (yydebug)
            {
                Console.WriteLine(msg);
            }
        }

//########## STATE STACK ##########
        private static int YYSTACKSIZE = 500; //maximum stack size
        private int[] statestk;
        private int stateptr; //state stack
//###############################################################
// methods: state stack push,pop,drop,peek
//###############################################################
        private void state_push(int state)
        {
            if (stateptr >= YYSTACKSIZE - 1) //overflowed?
            {
                yyerror_stackoverflow();
            }
            statestk[++stateptr] = state;
        }

        private int state_pop()
        {
            if (stateptr < 0) //underflowed?
            {
                return -1;
            }
            return statestk[stateptr--];
        }

        private void state_drop(int cnt)
        {
            int ptr;
            ptr = stateptr - cnt;
            if (ptr < 0)
            {
                return;
            }
            stateptr = ptr;
        }

        private int state_peek(int relative)
        {
            int ptr;
            ptr = stateptr - relative;
            if (ptr < 0)
            {
                return -1;
            }
            return statestk[ptr];
        }

//###############################################################
// method: init_stacks : allocate and prepare stacks
//###############################################################
        private Boolean init_stacks()
        {
            statestk = new int[YYSTACKSIZE];
            stateptr = -1;
            val_init();
            return true;
        }

//###############################################################
// method: dump_stacks : show n levels of the stacks
//###############################################################
        private void dump_stacks(int count)
        {
            int i;
            Console.WriteLine("=index==state====value=     s:" + stateptr + "  v:" + valptr);
            for (i = 0; i < count; i++)
            {
                Console.WriteLine(" " + i + "    " + statestk[i] + "      " + valstk[i]);
            }
            Console.WriteLine("======================");
        }

//########## SEMANTIC VALUES ##########
//## **default:object
//String   yytext;//user variable to return contextual strings
        private object yyval; //used to return semantic vals from action routines
        private object yylval; //the 'lval' (result) I got from yylex()
        private object[] valstk;
        private int valptr;
//###############################################################
// methods: value stack push,pop,drop,peek.
//###############################################################
        private void val_init()
        {
            valstk = new object[YYSTACKSIZE];
            yyval = 0;
            yylval = 0;
            valptr = -1;
        }

        private void val_push(object val)
        {
            if (valptr >= YYSTACKSIZE)
            {
                return;
            }
            valstk[++valptr] = val;
        }

        private object val_pop()
        {
            if (valptr < 0)
            {
                return -1;
            }
            return valstk[valptr--];
        }

        private void val_drop(int cnt)
        {
            int ptr;
            ptr = valptr - cnt;
            if (ptr < 0)
            {
                return;
            }
            valptr = ptr;
        }

        private object val_peek(int relative)
        {
            int ptr;
            ptr = valptr - relative;
            if (ptr < 0)
            {
                return -1;
            }
            return valstk[ptr];
        }

//#### end semantic value section ####
        public static short IDENTIFIER = 257;
        public static short ESCAPED_IDENTIFIER = 258;
        public static short PARAMETER = 259;
        public static short LITERAL = 260;
        public static short ALL = 261;
        public static short AND = 262;
        public static short ANYELEMENT = 263;
        public static short APPLY = 264;
        public static short AS = 265;
        public static short ASC = 266;
        public static short BETWEEN = 267;
        public static short BY = 268;
        public static short CASE = 269;
        public static short CAST = 270;
        public static short COLLATE = 271;
        public static short COLLECTION = 272;
        public static short CROSS = 273;
        public static short CREATEREF = 274;
        public static short DEREF = 275;
        public static short DESC = 276;
        public static short DISTINCT = 277;
        public static short ELEMENT = 278;
        public static short ELSE = 279;
        public static short END = 280;
        public static short EXCEPT = 281;
        public static short EXISTS = 282;
        public static short ESCAPE = 283;
        public static short FLATTEN = 284;
        public static short FROM = 285;
        public static short FULL = 286;
        public static short FUNCTION = 287;
        public static short GROUP = 288;
        public static short GROUPPARTITION = 289;
        public static short HAVING = 290;
        public static short IN = 291;
        public static short INNER = 292;
        public static short INTERSECT = 293;
        public static short IS = 294;
        public static short JOIN = 295;
        public static short KEY = 296;
        public static short LEFT = 297;
        public static short LIKE = 298;
        public static short LIMIT = 299;
        public static short MULTISET = 300;
        public static short NAVIGATE = 301;
        public static short NOT = 302;
        public static short NULL = 303;
        public static short OF = 304;
        public static short OFTYPE = 305;
        public static short ON = 306;
        public static short OR = 307;
        public static short ORDER = 308;
        public static short OUTER = 309;
        public static short OVERLAPS = 310;
        public static short ONLY = 311;
        public static short QMARK = 312;
        public static short REF = 313;
        public static short RELATIONSHIP = 314;
        public static short RIGHT = 315;
        public static short ROW = 316;
        public static short SELECT = 317;
        public static short SET = 318;
        public static short SKIP = 319;
        public static short THEN = 320;
        public static short TOP = 321;
        public static short TREAT = 322;
        public static short UNION = 323;
        public static short USING = 324;
        public static short VALUE = 325;
        public static short WHEN = 326;
        public static short WHERE = 327;
        public static short WITH = 328;
        public static short COMMA = 329;
        public static short COLON = 330;
        public static short SCOLON = 331;
        public static short DOT = 332;
        public static short EQUAL = 333;
        public static short L_PAREN = 334;
        public static short R_PAREN = 335;
        public static short L_BRACE = 336;
        public static short R_BRACE = 337;
        public static short L_CURLY = 338;
        public static short R_CURLY = 339;
        public static short PLUS = 340;
        public static short MINUS = 341;
        public static short STAR = 342;
        public static short FSLASH = 343;
        public static short PERCENT = 344;
        public static short OP_EQ = 345;
        public static short OP_NEQ = 346;
        public static short OP_LT = 347;
        public static short OP_LE = 348;
        public static short OP_GT = 349;
        public static short OP_GE = 350;
        public static short UNARYPLUS = 351;
        public static short UNARYMINUS = 352;
        public static short YYERRCODE = 256;

        private static readonly short[] yylhs = {
            -1,
            0, 0, 1, 2, 2, 4, 4, 5, 5, 5,
            3, 9, 9, 12, 12, 13, 14, 14, 15, 15,
            16, 10, 10, 11, 11, 18, 27, 20, 30, 20,
            26, 26, 26, 28, 28, 21, 31, 31, 32, 32,
            32, 32, 32, 34, 34, 35, 36, 36, 36, 36,
            36, 36, 36, 36, 36, 36, 37, 37, 22, 22,
            38, 23, 23, 39, 24, 24, 41, 40, 25, 25,
            44, 42, 45, 45, 46, 46, 43, 43, 47, 47,
            48, 48, 48, 50, 50, 19, 19, 19, 19, 19,
            19, 19, 19, 19, 19, 19, 19, 19, 19, 51,
            62, 63, 52, 52, 52, 52, 52, 52, 52, 52,
            52, 52, 52, 52, 52, 52, 52, 52, 52, 52,
            52, 52, 52, 52, 52, 52, 52, 52, 52, 52,
            52, 52, 52, 52, 52, 52, 52, 52, 52, 52,
            52, 52, 52, 52, 52, 52, 52, 66, 66, 8,
            67, 33, 33, 29, 29, 64, 64, 68, 68, 69,
            58, 58, 58, 7, 53, 59, 54, 54, 55, 56,
            57, 57, 57, 57, 57, 57, 60, 60, 60, 70,
            70, 71, 71, 72, 72, 72, 65, 65, 65, 65,
            65, 73, 74, 74, 74, 74, 6, 6, 49, 61,
            61, 17, 17, 17, 17, 75, 76, 77, 78, 78,
            79,
        };

        private static readonly short[] yylen = {
            2,
            0, 1, 2, 0, 1, 1, 2, 3, 3, 3,
            3, 0, 1, 1, 2, 7, 2, 3, 1, 3,
            2, 1, 1, 0, 1, 6, 0, 5, 0, 6,
            0, 1, 1, 0, 4, 2, 1, 3, 1, 3,
            1, 3, 1, 3, 5, 3, 2, 3, 2, 3,
            2, 1, 2, 2, 3, 2, 2, 2, 0, 1,
            2, 0, 1, 3, 0, 1, 0, 3, 0, 1,
            0, 6, 0, 2, 0, 2, 1, 3, 2, 4,
            0, 1, 1, 1, 3, 1, 1, 1, 1, 1,
            1, 1, 1, 1, 1, 1, 1, 1, 1, 3,
            3, 4, 3, 3, 3, 3, 3, 2, 2, 3,
            3, 3, 3, 3, 3, 3, 4, 3, 3, 3,
            4, 4, 4, 4, 4, 4, 3, 4, 1, 6,
            6, 6, 7, 6, 7, 7, 8, 3, 4, 5,
            6, 3, 3, 3, 2, 3, 1, 1, 1, 3,
            3, 3, 1, 1, 3, 3, 4, 4, 5, 2,
            4, 4, 3, 3, 4, 4, 6, 8, 4, 5,
            3, 6, 6, 3, 6, 6, 6, 8, 10, 0,
            1, 2, 2, 6, 8, 10, 1, 1, 2, 2,
            1, 3, 3, 4, 3, 4, 1, 1, 1, 1,
            1, 1, 1, 1, 1, 4, 4, 4, 1, 3,
            2,
        };

        private static readonly short[] yydefred = {
            0,
            0, 0, 2, 0, 0, 6, 199, 197, 87, 200,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 201, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 198, 86, 89,
            91, 92, 93, 94, 95, 96, 97, 98, 99, 0,
            0, 129, 147, 149, 0, 3, 0, 0, 14, 7,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 148, 0, 0, 0, 0,
            0, 0, 0, 0, 22, 0, 0, 0, 0, 0,
            0, 8, 0, 9, 0, 10, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 15, 0, 0, 0, 156, 0,
            0, 0, 0, 0, 0, 0, 0, 32, 33, 0,
            0, 0, 0, 0, 0, 0, 0, 154, 0, 0,
            0, 0, 100, 0, 0, 0, 163, 174, 0, 171,
            0, 0, 0, 0, 0, 0, 0, 127, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 164, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 25, 11, 123, 0, 0, 0,
            157, 0, 0, 166, 124, 122, 125, 0, 169, 162,
            0, 0, 165, 0, 0, 161, 126, 0, 0, 0,
            0, 0, 0, 39, 41, 43, 0, 0, 60, 0,
            0, 0, 0, 0, 128, 0, 0, 0, 0, 0,
            0, 0, 17, 0, 0, 19, 0, 0, 0, 0,
            0, 0, 191, 0, 170, 0, 0, 0, 152, 155,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 52, 0, 0, 0, 0, 0, 0, 0,
            0, 63, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 21, 0, 203, 204, 205, 0,
            18, 0, 0, 189, 0, 0, 131, 190, 0, 0,
            167, 0, 177, 0, 132, 130, 0, 0, 0, 40,
            42, 0, 57, 47, 54, 0, 53, 49, 0, 58,
            51, 0, 0, 46, 0, 67, 0, 66, 0, 176,
            0, 175, 173, 172, 0, 0, 0, 134, 0, 0,
            0, 0, 20, 0, 195, 0, 192, 193, 0, 0,
            0, 133, 0, 0, 55, 48, 50, 0, 0, 0,
            0, 26, 70, 0, 182, 183, 0, 135, 136, 0,
            0, 0, 0, 209, 16, 196, 194, 168, 0, 178,
            35, 0, 0, 71, 0, 137, 206, 207, 211, 0,
            208, 0, 0, 0, 210, 179, 0, 0, 77, 0,
            82, 0, 83, 79, 0, 0, 0, 0, 0, 0,
            78, 0, 72, 0, 184, 80, 0, 0, 0, 185,
            0, 186,
        };

        private static readonly short[] yydgoto = {
            2,
            3, 4, 56, 5, 6, 74, 75, 76, 57, 84,
            196, 58, 59, 194, 245, 246, 295, 85, 86, 87,
            155, 228, 281, 337, 372, 140, 151, 263, 147, 152,
            222, 223, 224, 225, 226, 277, 278, 229, 282, 338,
            370, 373, 408, 403, 417, 423, 409, 414, 38, 89,
            39, 40, 41, 42, 43, 44, 45, 46, 47, 48,
            49, 50, 51, 52, 296, 53, 54, 63, 131, 340,
            341, 375, 252, 253, 297, 298, 299, 383, 384,
        };

        private static readonly short[] yysindex = {
            -321,
            4723, 0, 0, -269, -321, 0, 0, 0, 0, 0,
            -309, -271, -284, -204, -176, -162, -155, -131, -118, -94,
            -84, -75, 4723, 0, -73, -51, -46, -42, -37, 4054,
            4723, 4723, 4723, -182, -181, -211, 3614, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, -111,
            14, 0, 0, 0, -171, 0, 4054, -269, 0, 0,
            4054, 4723, -207, 4723, 4723, 4054, 4054, 4054, 4054, -225,
            4054, 4723, 4723, -36, -28, 0, 5025, 4723, 4054, 4723,
            4054, 4723, 0, -13, 0, 3614, 22, 3614, -291, -8,
            -8, 0, -232, 0, -226, 0, 4723, 4723, 4723, 4723,
            4723, 174, 4723, -186, 4723, 4723, 4139, -171, 4723, 4723,
            4723, 4723, 4723, 4723, 4723, 4723, 4723, 4723, 4723, 4723,
            4723, 4723, -6, -5, 0, -3, 3098, 4723, 0, 4723,
            53, 3170, 3198, 3, 13, 18, 19, 0, 0, 4054,
            41, -213, 3270, 3298, 42, 3370, -159, 0, 47, 3442,
            -225, 62, 0, 4796, 68, 4723, 0, 0, 4054, 0,
            4054, 5025, 4955, 5124, 5053, 3413, -141, 0, 59, 2998,
            4723, 4723, 4723, 3814, 2926, 4723, 3341, 0, 4037, -177,
            -177, -8, -8, -8, 4037, 4037, 1461, 1461, 1461, 1461,
            5025, 5025, -256, 160, 0, 0, 0, 4723, 3614, 3514,
            0, -171, 4723, 0, 0, 0, 0, 99, 0, 0,
            -171, -241, 0, -171, 4723, 0, 0, -171, 124, -225,
            4212, 122, -58, 0, 0, 0, 4723, 167, 0, 3614,
            125, -110, 127, -81, 0, 123, -194, 4723, 4955, 5053,
            3786, 3341, 0, -216, -69, 0, 135, 3614, 4723, -254,
            -161, -245, 0, 3026, 0, -214, -171, -128, 0, 0,
            -127, 139, 4723, 124, 3370, -58, 149, 151, 4796, -249,
            -272, 179, 0, -265, 223, -248, 4796, 4796, 3614, 226,
            207, 0, 170, 170, 170, 170, -183, -171, -112, 5096,
            4723, 161, 175, 176, 0, 169, 0, 0, 0, -171,
            0, 4054, 3614, 0, 4285, -171, 0, 0, 4358, -171,
            0, -171, 0, -102, 0, 0, 4054, 178, 4723, 0,
            0, -58, 0, 0, 0, 216, 0, 0, 230, 0,
            0, 236, 228, 0, 4723, 0, 229, 0, 222, 0,
            222, 0, 0, 0, -171, -100, -39, 0, 5096, -216,
            -171, -171, 0, 204, 0, -66, 0, 0, -61, -20,
            -57, 0, 209, 178, 0, 0, 0, 4723, 178, 4723,
            272, 0, 0, 212, 0, 0, 73, 0, 0, 214,
            115, -216, -56, 0, 0, 0, 0, 0, -171, 0,
            0, 3614, 3614, 0, 4723, 0, 0, 0, 0, -171,
            0, 227, 4723, 3542, 0, 0, 2954, -276, 0, -171,
            0, 294, 0, 0, 4723, 4723, 259, -151, -199, 3614,
            0, 4723, 0, -171, 0, 0, 3614, -45, -171, 0,
            231, 0,
        };

        private static readonly short[] yyrindex = {
            1510,
            0, 0, 0, 4431, 3981, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 3642, 3714, 3742, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 4504, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 4577,
            0, 0, 0, 173, 262, 0, 2591, 0, 0, 0,
            0, 0, 3908, 0, 0, 76, 0, -117, 0, 532,
            621, 0, 4577, 0, 4577, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 563, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 28, 0, 0, 0, 0,
            4650, 0, 0, 0, 39, 0, 0, 0, 0, 0,
            0, 2662, 303, 2023, 2307, 1576, 0, 0, 0, 2378,
            0, 0, 0, 2861, 2094, 0, 1876, 0, 1647, 979,
            1068, 711, 800, 889, 1723, 1798, 1157, 1242, 1331, 1420,
            2733, 2804, 0, 0, 0, 0, 0, 0, 287, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 4869, 4650,
            0, 1185, 31, 0, 0, 0, 0, 82, 0, -108,
            0, 0, 0, 0, 0, 0, 0, 0, 311, 2449,
            2520, 1947, 0, 0, 0, 0, 0, -169, 0, -139,
            0, -83, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 4869, -109, 0, -15, 1294, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 96, 0,
            33, 0, 352, 352, 352, 352, 0, 0, 0, 2165,
            0, 0, 0, 0, 0, -44, 0, 0, 0, 0,
            0, 0, -166, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 289, 0, 0,
            0, 95, 0, 0, 0, 4942, 0, 0, 0, 0,
            0, 0, 2909, 0, 0, 0, 84, 0, 0, 0,
            443, 0, 0, 0, 0, 0, 0, 0, 2236, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 290, 0, 0, 0, 0, 195, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 83, 71, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 11, 66, 0, 0,
            0, 0, 0, 0, 0, 0, 85, 0, 11, 101,
            0, 0, 0, 0, 0, 0, 102, 0, 0, 0,
            0, 0,
        };

        private static readonly short[] yygindex = {
            0,
            0, 0, 0, 0, 571, -1, 577, 578, 0, -47,
            0, 0, 522, 0, 0, 283, -342, 30, 26, 0,
            0, 0, 0, 0, 0, -86, 0, 320, -257, 0,
            0, -209, -54, 365, 366, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 180, 171, 186, -67,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, -43, 0, 0, 0, 0, 205,
            0, 251, 0, 0, 0, 0, 0, 0, 199,
        };

        private static int YYTABLESIZE = 5474;

        private static readonly short[] yytable = {
            34,
            7, 8, 1, 304, 142, 318, 159, 380, 161, 124,
            81, 266, 308, 126, 323, 7, 8, 55, 134, 135,
            136, 137, 325, 141, 61, 148, 37, 153, 138, 328,
            37, 145, 65, 149, 138, 138, 326, 156, 59, 399,
            7, 8, 415, 329, 139, 324, 331, 157, 77, 64,
            139, 139, 416, 123, 62, 292, 88, 90, 91, 322,
            332, 364, 7, 8, 219, 73, 411, 333, 334, 257,
            68, 128, 129, 7, 8, 23, 413, 369, 243, 305,
            171, 62, 45, 69, 75, 7, 8, 127, 309, 132,
            133, 232, 208, 234, 38, 61, 293, 88, 143, 294,
            74, 76, 158, 144, 172, 146, 178, 150, 160, 158,
            158, 173, 159, 159, 312, 156, 288, 306, 130, 96,
            313, 210, 162, 163, 164, 165, 166, 345, 170, 65,
            174, 175, 177, 264, 179, 180, 181, 182, 183, 184,
            185, 186, 187, 188, 189, 190, 191, 192, 92, 94,
            121, 93, 95, 199, 108, 200, 158, 66, 251, 159,
            260, 235, 236, 153, 112, 113, 114, 256, 258, 215,
            306, 67, 88, 307, 261, 216, 153, 424, 68, 146,
            306, 230, 153, 425, 88, 153, 88, 153, 231, 187,
            233, 244, 187, 289, 64, 187, 239, 240, 241, 153,
            250, 242, 69, 306, 306, 153, 315, 316, 148, 250,
            250, 84, 259, 314, 270, 70, 250, 84, 156, 306,
            85, 84, 348, 248, 284, 23, 85, 271, 254, 306,
            85, 306, 362, 272, 378, 250, 273, 356, 274, 71,
            146, 359, 250, 346, 347, 188, 265, 156, 188, 72,
            275, 188, 279, 286, 354, 250, 276, 41, 73, 300,
            78, 90, 156, 290, 148, 301, 360, 156, 386, 363,
            41, 389, 400, 387, 303, 122, 41, 390, 401, 41,
            148, 41, 79, 429, 202, 250, 250, 80, 146, 430,
            202, 81, 306, 41, 146, 379, 82, 93, 244, 41,
            153, 377, 146, 146, 357, 95, 154, 381, 250, 81,
            361, 306, 153, 153, 388, 153, 349, 153, 37, 153,
            37, 153, 153, 108, 153, 195, 59, 193, 59, 81,
            88, 197, 201, 153, 88, 153, 153, 204, 37, 81,
            65, 81, 153, 250, 146, 81, 59, 205, 250, 250,
            382, 180, 206, 207, 153, 45, 153, 37, 153, 37,
            146, 37, 153, 65, 73, 37, 418, 65, 45, 59,
            45, 62, 45, 59, 45, 209, 213, 45, 68, 45,
            250, 217, 38, 61, 38, 61, 220, 402, 45, 62,
            45, 45, 237, 392, 227, 393, 73, 45, 382, 74,
            73, 68, 38, 61, 306, 68, 23, 396, 250, 45,
            23, 45, 62, 45, 69, 75, 62, 45, 69, 75,
            404, 38, 428, 38, 247, 38, 61, 431, 407, 38,
            61, 74, 76, 255, 88, 74, 76, 88, 88, 88,
            420, 407, 181, 88, 262, 88, 306, 427, 88, 398,
            269, 88, 88, 88, 280, 88, 287, 88, 88, 283,
            88, 285, 88, 88, 88, 88, 88, 88, 302, 88,
            88, 88, 317, 327, 88, 167, 168, 169, 88, 88,
            88, 88, 88, 320, 64, 321, 330, 88, 342, 343,
            344, 88, 88, 335, 350, 88, 336, 339, 88, 88,
            306, 88, 64, 88, 88, 88, 215, 88, 351, 352,
            365, 88, 88, 88, 88, 88, 88, 88, 88, 88,
            88, 88, 88, 90, 366, 64, 90, 90, 90, 64,
            367, 109, 90, 368, 90, 374, 371, 90, 385, 394,
            90, 90, 90, 391, 90, 395, 90, 90, 397, 90,
            7, 90, 90, 90, 90, 90, 90, 422, 90, 90,
            90, 406, 24, 90, 101, 432, 160, 90, 90, 90,
            90, 90, 102, 28, 30, 60, 90, 35, 36, 125,
            90, 90, 353, 319, 90, 267, 268, 90, 90, 426,
            90, 376, 90, 90, 90, 421, 90, 419, 405, 0,
            90, 90, 90, 90, 90, 90, 90, 90, 90, 90,
            90, 90, 0, 180, 0, 0, 180, 180, 180, 0,
            108, 0, 180, 0, 180, 0, 0, 180, 0, 0,
            180, 180, 180, 0, 180, 0, 180, 180, 0, 180,
            0, 180, 180, 180, 180, 180, 180, 0, 180, 180,
            180, 0, 0, 180, 0, 0, 0, 180, 180, 180,
            180, 180, 0, 0, 0, 0, 180, 0, 0, 0,
            180, 180, 0, 0, 180, 0, 0, 180, 180, 0,
            180, 0, 180, 180, 180, 0, 180, 0, 0, 0,
            180, 180, 180, 180, 180, 180, 180, 180, 180, 180,
            180, 180, 0, 0, 181, 0, 0, 181, 181, 181,
            105, 0, 0, 181, 0, 181, 0, 0, 181, 0,
            0, 181, 181, 181, 0, 181, 0, 181, 181, 0,
            181, 0, 181, 181, 181, 181, 181, 181, 0, 181,
            181, 181, 0, 0, 181, 0, 0, 0, 181, 181,
            181, 181, 181, 0, 0, 0, 0, 181, 0, 0,
            0, 181, 181, 0, 0, 181, 0, 0, 181, 181,
            0, 181, 0, 181, 181, 181, 0, 181, 0, 0,
            0, 181, 181, 181, 181, 181, 181, 181, 181, 181,
            181, 181, 181, 109, 0, 0, 109, 109, 109, 106,
            0, 0, 109, 0, 109, 0, 0, 109, 0, 0,
            109, 109, 109, 0, 109, 0, 109, 109, 0, 109,
            0, 109, 109, 109, 109, 109, 109, 0, 109, 109,
            109, 0, 0, 109, 0, 0, 0, 109, 109, 109,
            109, 109, 0, 0, 0, 0, 109, 0, 0, 0,
            109, 109, 0, 0, 109, 0, 0, 109, 109, 0,
            109, 0, 109, 0, 109, 0, 109, 0, 0, 0,
            109, 109, 109, 109, 109, 109, 109, 109, 109, 109,
            109, 109, 108, 0, 0, 108, 108, 108, 107, 0,
            0, 108, 0, 108, 0, 0, 108, 0, 0, 108,
            108, 108, 0, 108, 0, 108, 108, 0, 108, 0,
            108, 108, 108, 108, 108, 108, 0, 108, 108, 108,
            0, 0, 108, 0, 0, 0, 108, 108, 108, 108,
            108, 0, 0, 0, 0, 108, 0, 0, 0, 108,
            108, 0, 0, 108, 0, 0, 108, 108, 0, 108,
            0, 108, 0, 108, 0, 108, 0, 0, 0, 108,
            108, 108, 108, 108, 108, 108, 108, 108, 108, 108,
            108, 0, 105, 0, 0, 105, 105, 105, 103, 0,
            0, 105, 0, 105, 0, 0, 105, 0, 0, 105,
            105, 105, 0, 105, 0, 105, 105, 0, 105, 0,
            105, 105, 105, 105, 105, 105, 0, 105, 105, 105,
            0, 0, 105, 0, 0, 0, 105, 105, 105, 105,
            105, 0, 0, 0, 0, 105, 0, 0, 0, 105,
            105, 0, 0, 105, 0, 0, 105, 105, 0, 105,
            0, 105, 0, 105, 0, 105, 0, 0, 0, 105,
            105, 105, 105, 105, 105, 105, 105, 105, 105, 105,
            105, 106, 0, 0, 106, 106, 106, 104, 0, 0,
            106, 0, 106, 0, 0, 106, 0, 0, 106, 106,
            106, 0, 106, 0, 106, 106, 0, 106, 0, 106,
            106, 106, 106, 106, 106, 0, 106, 106, 106, 0,
            0, 106, 0, 0, 0, 106, 106, 106, 106, 106,
            0, 0, 0, 0, 106, 0, 0, 0, 106, 106,
            0, 0, 106, 0, 0, 106, 106, 0, 106, 0,
            106, 0, 106, 0, 106, 0, 0, 0, 106, 106,
            106, 106, 106, 106, 106, 106, 106, 106, 106, 106,
            107, 0, 0, 107, 107, 107, 113, 0, 0, 107,
            0, 107, 0, 0, 107, 0, 0, 107, 107, 107,
            0, 107, 0, 107, 107, 0, 107, 0, 107, 107,
            107, 107, 107, 107, 36, 107, 107, 107, 0, 0,
            107, 0, 0, 0, 107, 107, 107, 107, 107, 0,
            0, 0, 0, 107, 0, 0, 0, 107, 107, 0,
            0, 107, 0, 0, 107, 107, 0, 107, 0, 107,
            0, 107, 0, 107, 0, 0, 0, 107, 107, 107,
            107, 107, 107, 107, 107, 107, 107, 107, 107, 0,
            103, 114, 0, 103, 103, 103, 0, 0, 0, 103,
            0, 103, 0, 0, 103, 0, 0, 103, 103, 103,
            0, 103, 0, 103, 103, 0, 103, 0, 103, 103,
            103, 103, 103, 103, 0, 103, 103, 103, 0, 0,
            103, 0, 0, 0, 103, 103, 103, 103, 103, 0,
            0, 0, 0, 103, 0, 0, 0, 103, 103, 0,
            0, 103, 0, 0, 103, 103, 0, 103, 0, 103,
            0, 103, 0, 103, 0, 0, 0, 103, 103, 103,
            0, 0, 0, 103, 103, 103, 103, 103, 103, 104,
            111, 0, 104, 104, 104, 0, 0, 0, 104, 0,
            104, 0, 0, 104, 0, 0, 104, 104, 104, 0,
            104, 0, 104, 104, 0, 104, 0, 104, 104, 104,
            104, 104, 104, 0, 104, 104, 104, 0, 0, 104,
            0, 0, 0, 104, 104, 104, 104, 104, 0, 0,
            0, 0, 104, 0, 0, 0, 104, 104, 0, 0,
            104, 0, 0, 104, 104, 0, 104, 0, 104, 0,
            104, 0, 104, 0, 0, 0, 104, 104, 104, 0,
            0, 0, 104, 104, 104, 104, 104, 104, 113, 112,
            0, 113, 113, 113, 0, 0, 0, 113, 0, 113,
            0, 0, 113, 0, 0, 113, 113, 113, 0, 113,
            0, 113, 113, 0, 113, 0, 113, 113, 113, 113,
            113, 113, 0, 113, 113, 113, 0, 0, 113, 0,
            0, 0, 113, 113, 113, 113, 113, 0, 0, 0,
            0, 113, 36, 0, 36, 113, 113, 0, 0, 113,
            0, 0, 113, 113, 0, 113, 0, 113, 0, 113,
            0, 113, 36, 0, 0, 113, 0, 0, 0, 0,
            0, 113, 113, 114, 0, 0, 114, 114, 114, 1,
            0, 36, 114, 0, 114, 36, 0, 114, 0, 36,
            114, 114, 114, 0, 114, 0, 114, 114, 0, 114,
            0, 114, 114, 114, 114, 114, 114, 0, 114, 114,
            114, 0, 0, 114, 0, 0, 0, 114, 114, 114,
            114, 114, 0, 0, 0, 0, 114, 0, 0, 0,
            114, 114, 0, 0, 114, 0, 43, 114, 114, 0,
            114, 0, 114, 0, 114, 115, 114, 0, 0, 43,
            114, 0, 0, 0, 0, 43, 114, 114, 43, 0,
            43, 0, 111, 0, 0, 111, 111, 111, 0, 0,
            0, 111, 43, 111, 0, 0, 111, 0, 43, 111,
            111, 111, 0, 111, 0, 111, 111, 0, 111, 0,
            111, 111, 111, 111, 111, 111, 0, 111, 111, 111,
            0, 0, 111, 0, 0, 0, 111, 111, 111, 111,
            111, 0, 0, 0, 0, 111, 150, 0, 0, 111,
            111, 0, 0, 111, 0, 0, 111, 111, 0, 111,
            0, 111, 0, 111, 0, 111, 0, 0, 0, 111,
            0, 0, 0, 0, 0, 111, 111, 0, 0, 0,
            0, 112, 0, 0, 112, 112, 112, 0, 0, 0,
            112, 0, 112, 0, 0, 112, 0, 0, 112, 112,
            112, 0, 112, 0, 112, 112, 0, 112, 0, 112,
            112, 112, 112, 112, 112, 0, 112, 112, 112, 0,
            0, 112, 151, 0, 0, 112, 112, 112, 112, 112,
            0, 0, 0, 0, 112, 0, 0, 0, 112, 112,
            0, 0, 112, 0, 0, 112, 112, 0, 112, 0,
            112, 0, 112, 0, 112, 0, 0, 0, 112, 0,
            0, 0, 0, 0, 112, 112, 4, 4, 4, 4,
            0, 0, 4, 0, 0, 0, 0, 0, 4, 4,
            0, 0, 0, 4, 4, 0, 0, 4, 0, 0,
            0, 4, 108, 4, 0, 0, 4, 110, 4, 0,
            110, 111, 112, 113, 114, 4, 0, 0, 0, 4,
            4, 4, 4, 0, 4, 0, 0, 0, 0, 0,
            0, 0, 4, 0, 0, 4, 4, 4, 0, 0,
            0, 4, 0, 0, 0, 0, 0, 115, 0, 0,
            115, 115, 115, 4, 0, 0, 115, 4, 115, 4,
            4, 115, 0, 0, 115, 115, 115, 0, 115, 0,
            115, 115, 0, 115, 0, 115, 115, 115, 115, 115,
            115, 0, 115, 115, 115, 116, 0, 115, 0, 0,
            0, 115, 115, 115, 115, 115, 0, 0, 0, 0,
            115, 0, 0, 0, 115, 115, 0, 0, 115, 0,
            0, 115, 115, 0, 115, 0, 115, 0, 150, 0,
            115, 150, 150, 150, 115, 0, 0, 150, 0, 150,
            0, 0, 150, 0, 0, 150, 150, 150, 0, 150,
            0, 150, 150, 0, 150, 0, 150, 150, 150, 150,
            150, 150, 0, 150, 150, 150, 117, 0, 150, 0,
            0, 0, 150, 150, 150, 150, 150, 0, 0, 0,
            0, 150, 0, 0, 0, 150, 150, 0, 0, 150,
            0, 0, 150, 150, 0, 150, 0, 150, 0, 0,
            0, 150, 0, 0, 151, 150, 0, 151, 151, 151,
            0, 0, 0, 151, 0, 151, 0, 0, 151, 0,
            0, 151, 151, 151, 0, 151, 0, 151, 151, 0,
            151, 0, 151, 151, 151, 151, 151, 151, 0, 151,
            151, 151, 118, 0, 151, 0, 0, 0, 151, 151,
            151, 151, 151, 0, 0, 0, 0, 151, 0, 0,
            0, 151, 151, 0, 0, 151, 0, 0, 151, 151,
            0, 151, 0, 151, 0, 0, 0, 151, 0, 110,
            0, 151, 110, 110, 110, 0, 0, 0, 110, 0,
            110, 0, 0, 110, 0, 0, 110, 110, 110, 0,
            110, 0, 110, 110, 0, 110, 0, 110, 110, 110,
            110, 110, 110, 119, 110, 110, 110, 0, 0, 110,
            0, 0, 0, 110, 110, 110, 110, 110, 0, 0,
            0, 0, 110, 0, 0, 0, 110, 110, 0, 0,
            110, 0, 0, 110, 110, 0, 110, 0, 110, 0,
            0, 0, 110, 0, 0, 0, 110, 116, 0, 0,
            116, 116, 116, 0, 0, 0, 116, 0, 116, 0,
            0, 116, 0, 0, 116, 116, 116, 0, 116, 0,
            116, 116, 0, 116, 140, 116, 116, 116, 0, 116,
            116, 0, 116, 116, 116, 0, 0, 116, 0, 0,
            0, 116, 116, 116, 116, 116, 0, 0, 0, 0,
            116, 0, 0, 0, 116, 116, 0, 0, 116, 0,
            0, 116, 116, 0, 116, 0, 116, 0, 117, 0,
            116, 117, 117, 117, 116, 0, 0, 117, 0, 117,
            0, 0, 117, 0, 0, 117, 117, 117, 0, 117,
            0, 117, 117, 0, 117, 141, 117, 117, 117, 0,
            117, 117, 0, 117, 117, 117, 0, 0, 117, 0,
            0, 0, 117, 117, 117, 117, 117, 0, 0, 0,
            0, 117, 0, 0, 0, 117, 117, 0, 0, 117,
            0, 0, 117, 117, 0, 117, 0, 117, 0, 0,
            0, 117, 0, 0, 118, 117, 0, 118, 118, 118,
            0, 0, 0, 118, 0, 118, 0, 0, 118, 0,
            0, 118, 118, 118, 0, 118, 120, 118, 118, 0,
            118, 0, 118, 118, 118, 0, 118, 118, 0, 118,
            118, 118, 0, 0, 118, 0, 0, 0, 118, 118,
            118, 118, 118, 0, 0, 0, 0, 118, 0, 0,
            0, 118, 118, 0, 0, 0, 0, 0, 118, 118,
            0, 118, 0, 118, 0, 119, 0, 118, 119, 119,
            119, 118, 0, 0, 119, 0, 119, 0, 0, 119,
            0, 0, 119, 119, 0, 0, 119, 138, 119, 119,
            0, 119, 0, 119, 119, 119, 0, 119, 119, 0,
            119, 119, 119, 0, 0, 119, 0, 0, 0, 119,
            119, 119, 119, 119, 0, 0, 0, 0, 119, 0,
            0, 0, 119, 119, 0, 0, 0, 0, 0, 119,
            119, 0, 119, 0, 119, 0, 140, 0, 119, 140,
            140, 140, 119, 0, 0, 140, 0, 140, 0, 0,
            140, 0, 0, 140, 140, 0, 0, 140, 121, 140,
            140, 0, 140, 0, 140, 140, 140, 0, 140, 140,
            0, 140, 140, 140, 0, 0, 140, 0, 0, 0,
            140, 140, 140, 140, 0, 0, 0, 0, 0, 140,
            0, 0, 0, 140, 140, 0, 0, 0, 0, 0,
            140, 140, 0, 140, 0, 140, 0, 141, 0, 140,
            141, 141, 141, 140, 0, 0, 141, 0, 141, 0,
            0, 141, 0, 0, 141, 141, 0, 0, 141, 139,
            141, 141, 0, 141, 0, 141, 141, 141, 0, 141,
            141, 0, 141, 141, 141, 0, 0, 141, 0, 0,
            0, 141, 141, 141, 141, 0, 0, 0, 0, 0,
            141, 0, 0, 0, 141, 141, 0, 0, 0, 0,
            0, 141, 141, 0, 141, 0, 141, 0, 120, 0,
            141, 120, 120, 120, 141, 0, 0, 120, 0, 120,
            0, 0, 120, 0, 0, 120, 120, 0, 0, 120,
            145, 120, 120, 0, 120, 0, 120, 0, 120, 0,
            120, 120, 0, 120, 0, 120, 0, 0, 120, 0,
            0, 0, 120, 120, 120, 120, 0, 0, 0, 0,
            0, 120, 0, 0, 0, 120, 120, 0, 0, 0,
            0, 0, 120, 120, 0, 120, 0, 120, 0, 138,
            0, 120, 138, 138, 138, 120, 0, 0, 138, 0,
            138, 0, 0, 138, 0, 0, 138, 138, 0, 0,
            0, 146, 138, 138, 0, 138, 0, 138, 138, 138,
            0, 138, 138, 0, 138, 0, 138, 0, 0, 138,
            0, 0, 0, 138, 138, 138, 138, 0, 0, 0,
            0, 0, 138, 0, 0, 0, 138, 138, 0, 0,
            0, 0, 0, 138, 138, 0, 138, 0, 138, 0,
            121, 0, 138, 121, 121, 121, 138, 0, 0, 121,
            0, 121, 0, 0, 121, 0, 0, 121, 121, 0,
            0, 121, 142, 121, 121, 0, 121, 0, 121, 0,
            121, 0, 121, 121, 0, 121, 0, 121, 0, 0,
            121, 0, 0, 0, 121, 121, 121, 121, 0, 0,
            0, 0, 0, 121, 0, 0, 0, 121, 121, 0,
            0, 0, 0, 0, 121, 121, 0, 121, 0, 121,
            0, 139, 0, 121, 139, 139, 139, 121, 0, 0,
            139, 0, 139, 0, 0, 139, 0, 0, 139, 139,
            0, 0, 0, 143, 139, 139, 0, 139, 0, 139,
            139, 139, 0, 139, 139, 0, 139, 0, 139, 0,
            0, 139, 0, 0, 0, 139, 139, 139, 139, 0,
            0, 0, 0, 0, 139, 0, 0, 0, 139, 139,
            0, 0, 0, 0, 0, 139, 139, 0, 139, 0,
            139, 0, 145, 0, 139, 145, 145, 0, 139, 0,
            144, 145, 0, 145, 0, 0, 145, 0, 0, 145,
            145, 0, 0, 145, 0, 145, 145, 0, 145, 0,
            145, 0, 145, 0, 0, 145, 0, 145, 0, 145,
            0, 0, 0, 0, 0, 0, 145, 145, 145, 145,
            0, 0, 0, 0, 0, 145, 0, 0, 44, 145,
            145, 0, 0, 0, 0, 0, 145, 145, 0, 145,
            0, 145, 0, 146, 0, 145, 146, 146, 0, 145,
            0, 0, 146, 0, 146, 0, 0, 146, 0, 0,
            146, 146, 0, 0, 146, 0, 146, 146, 0, 146,
            0, 146, 0, 146, 0, 0, 146, 0, 146, 0,
            146, 0, 0, 0, 0, 0, 0, 146, 146, 146,
            146, 0, 0, 0, 0, 0, 146, 0, 0, 0,
            146, 146, 0, 0, 0, 0, 0, 146, 146, 0,
            146, 0, 146, 0, 142, 0, 146, 142, 142, 0,
            146, 0, 0, 142, 0, 142, 0, 0, 142, 0,
            0, 142, 142, 0, 0, 142, 0, 142, 142, 0,
            142, 0, 142, 0, 142, 0, 0, 142, 0, 142,
            0, 142, 0, 0, 0, 0, 0, 0, 142, 142,
            142, 142, 0, 0, 0, 0, 0, 142, 0, 0,
            0, 142, 142, 0, 0, 0, 0, 0, 142, 142,
            0, 142, 0, 142, 0, 143, 0, 142, 143, 143,
            0, 142, 0, 0, 143, 0, 143, 0, 0, 143,
            0, 0, 143, 143, 0, 0, 143, 0, 143, 143,
            0, 143, 0, 143, 0, 143, 0, 0, 143, 0,
            143, 0, 143, 0, 0, 0, 0, 0, 0, 143,
            143, 143, 143, 0, 0, 0, 0, 0, 143, 0,
            0, 0, 143, 143, 0, 144, 144, 0, 0, 143,
            143, 144, 143, 144, 143, 0, 144, 0, 143, 144,
            144, 0, 143, 144, 0, 144, 144, 0, 144, 0,
            144, 0, 144, 0, 0, 144, 0, 144, 0, 144,
            0, 0, 0, 0, 0, 0, 144, 144, 144, 144,
            0, 0, 0, 0, 0, 144, 0, 0, 0, 144,
            144, 44, 0, 0, 0, 0, 144, 144, 0, 144,
            0, 144, 0, 0, 44, 144, 44, 0, 44, 144,
            44, 0, 0, 44, 0, 44, 99, 0, 0, 0,
            0, 0, 0, 0, 0, 97, 44, 44, 101, 411,
            98, 0, 0, 44, 412, 0, 0, 0, 0, 413,
            0, 0, 0, 0, 99, 44, 0, 44, 0, 44,
            0, 0, 0, 44, 100, 0, 101, 102, 107, 0,
            0, 103, 0, 0, 0, 104, 0, 108, 109, 0,
            105, 0, 0, 106, 0, 110, 111, 112, 113, 114,
            115, 116, 117, 118, 119, 120, 107, 0, 99, 0,
            238, 0, 0, 0, 0, 108, 109, 97, 0, 0,
            101, 0, 98, 110, 111, 112, 113, 114, 115, 116,
            117, 118, 119, 120, 0, 0, 99, 106, 0, 0,
            0, 0, 0, 0, 0, 0, 100, 0, 101, 102,
            107, 0, 0, 103, 0, 0, 0, 104, 0, 108,
            109, 0, 105, 0, 0, 106, 0, 110, 111, 112,
            113, 114, 115, 116, 117, 118, 119, 120, 107, 0,
            0, 0, 0, 0, 310, 0, 0, 108, 109, 97,
            311, 0, 0, 0, 98, 110, 111, 112, 113, 114,
            115, 116, 117, 118, 119, 120, 0, 0, 99, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 100, 0,
            101, 102, 0, 0, 0, 103, 0, 0, 0, 104,
            0, 0, 0, 0, 105, 0, 0, 106, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 198, 0, 0,
            107, 0, 0, 0, 0, 0, 0, 0, 0, 108,
            109, 97, 0, 0, 202, 0, 98, 110, 111, 112,
            113, 114, 115, 116, 117, 118, 119, 120, 0, 0,
            99, 0, 0, 0, 0, 0, 0, 0, 0, 97,
            100, 0, 101, 102, 98, 0, 0, 103, 0, 0,
            0, 104, 0, 0, 0, 0, 105, 0, 99, 106,
            0, 0, 0, 0, 0, 0, 0, 0, 100, 0,
            101, 102, 107, 0, 0, 103, 0, 0, 0, 104,
            0, 108, 109, 0, 105, 0, 0, 106, 0, 110,
            111, 112, 113, 114, 115, 116, 117, 118, 119, 120,
            107, 0, 0, 0, 0, 0, 203, 0, 0, 108,
            109, 97, 0, 0, 0, 0, 98, 110, 111, 112,
            113, 114, 115, 116, 117, 118, 119, 120, 0, 0,
            99, 0, 0, 0, 0, 0, 0, 0, 0, 97,
            100, 0, 101, 102, 98, 0, 0, 103, 0, 0,
            0, 104, 0, 0, 0, 0, 105, 0, 99, 106,
            0, 0, 0, 0, 0, 0, 0, 0, 100, 0,
            101, 102, 107, 0, 0, 103, 0, 0, 211, 104,
            0, 108, 109, 0, 105, 0, 0, 106, 0, 110,
            111, 112, 113, 114, 115, 116, 117, 118, 119, 120,
            107, 0, 0, 0, 0, 0, 212, 0, 0, 108,
            109, 97, 0, 101, 214, 0, 98, 110, 111, 112,
            113, 114, 115, 116, 117, 118, 119, 120, 0, 0,
            99, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            100, 0, 101, 102, 0, 0, 0, 103, 0, 0,
            0, 104, 108, 109, 0, 0, 105, 0, 0, 106,
            110, 111, 112, 113, 114, 115, 116, 117, 118, 119,
            120, 0, 107, 0, 0, 0, 0, 0, 0, 0,
            0, 108, 109, 97, 0, 0, 218, 0, 98, 110,
            111, 112, 113, 114, 115, 116, 117, 118, 119, 120,
            0, 0, 99, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 100, 0, 101, 102, 0, 0, 0, 103,
            0, 0, 0, 104, 108, 109, 0, 0, 105, 0,
            0, 106, 110, 111, 112, 113, 114, 115, 116, 117,
            118, 119, 120, 0, 107, 0, 0, 0, 0, 0,
            0, 0, 0, 108, 109, 97, 0, 0, 0, 0,
            98, 110, 111, 112, 113, 114, 115, 116, 117, 118,
            119, 120, 0, 0, 99, 0, 0, 0, 0, 0,
            0, 0, 0, 97, 100, 0, 101, 102, 98, 0,
            0, 103, 0, 0, 0, 104, 0, 0, 0, 0,
            105, 0, 99, 106, 0, 0, 0, 0, 0, 0,
            0, 0, 100, 249, 101, 102, 107, 0, 0, 103,
            0, 0, 0, 104, 0, 108, 109, 0, 105, 0,
            0, 106, 0, 110, 111, 112, 113, 114, 115, 116,
            117, 118, 119, 120, 107, 0, 0, 0, 0, 0,
            410, 0, 0, 108, 109, 97, 0, 0, 0, 0,
            98, 110, 111, 112, 113, 114, 115, 116, 117, 118,
            119, 120, 0, 0, 99, 0, 0, 0, 0, 0,
            0, 0, 0, 88, 100, 0, 101, 102, 88, 0,
            0, 103, 0, 0, 0, 104, 0, 0, 0, 0,
            105, 0, 88, 106, 0, 0, 0, 0, 0, 0,
            0, 0, 88, 0, 88, 88, 107, 0, 0, 88,
            0, 0, 0, 88, 0, 108, 109, 0, 88, 0,
            0, 88, 0, 110, 111, 112, 113, 114, 115, 116,
            117, 118, 119, 120, 88, 0, 0, 0, 0, 0,
            0, 0, 0, 88, 88, 90, 0, 0, 0, 0,
            90, 88, 88, 88, 88, 88, 88, 88, 88, 88,
            88, 88, 0, 0, 90, 0, 0, 0, 0, 0,
            0, 0, 0, 148, 90, 0, 90, 90, 148, 0,
            0, 90, 0, 0, 0, 90, 0, 0, 0, 0,
            90, 0, 148, 90, 0, 0, 0, 0, 0, 0,
            0, 0, 148, 0, 148, 148, 90, 0, 0, 148,
            0, 0, 0, 148, 0, 90, 90, 0, 148, 0,
            0, 148, 0, 90, 90, 90, 90, 90, 90, 90,
            90, 90, 90, 90, 148, 0, 99, 0, 291, 0,
            0, 0, 0, 148, 148, 97, 0, 0, 101, 0,
            98, 148, 148, 148, 148, 148, 148, 148, 148, 148,
            148, 148, 0, 0, 99, 106, 0, 0, 0, 0,
            0, 0, 0, 0, 100, 0, 101, 102, 107, 0,
            0, 103, 0, 0, 0, 104, 0, 108, 109, 0,
            0, 0, 0, 106, 0, 110, 111, 112, 113, 114,
            115, 116, 117, 118, 119, 120, 107, 0, 0, 0,
            0, 0, 0, 0, 0, 108, 109, 0, 0, 0,
            0, 0, 0, 110, 111, 112, 113, 114, 115, 116,
            117, 118, 119, 120, 27, 27, 27, 27, 27, 0,
            27, 0, 0, 0, 0, 0, 27, 27, 0, 0,
            0, 27, 27, 0, 27, 27, 0, 0, 0, 27,
            0, 27, 0, 0, 0, 0, 27, 0, 0, 0,
            0, 0, 0, 27, 0, 0, 0, 27, 27, 27,
            27, 0, 27, 0, 0, 0, 0, 0, 0, 0,
            27, 0, 0, 27, 0, 27, 0, 0, 27, 27,
            0, 0, 29, 0, 0, 0, 0, 5, 5, 5,
            5, 27, 0, 5, 0, 27, 0, 27, 27, 5,
            5, 0, 0, 0, 5, 5, 0, 0, 5, 0,
            0, 0, 5, 0, 5, 0, 0, 5, 0, 5,
            0, 0, 0, 0, 0, 0, 5, 0, 0, 0,
            5, 5, 5, 5, 0, 5, 0, 0, 0, 0,
            0, 0, 0, 5, 0, 0, 5, 5, 5, 0,
            0, 0, 5, 0, 0, 0, 0, 0, 0, 0,
            7, 8, 9, 10, 5, 0, 11, 0, 5, 0,
            5, 5, 12, 13, 0, 0, 0, 14, 15, 0,
            0, 16, 0, 0, 0, 17, 0, 18, 0, 0,
            0, 0, 19, 0, 0, 0, 0, 0, 0, 20,
            0, 0, 0, 21, 22, 23, 24, 0, 25, 0,
            0, 0, 0, 0, 0, 0, 26, 0, 108, 27,
            83, 28, 0, 0, 0, 29, 110, 111, 112, 113,
            114, 0, 0, 117, 118, 119, 120, 30, 0, 0,
            0, 31, 0, 32, 33, 7, 8, 9, 10, 176,
            0, 11, 0, 0, 0, 0, 0, 12, 13, 0,
            0, 0, 14, 15, 0, 0, 16, 0, 0, 0,
            17, 0, 18, 0, 0, 0, 0, 19, 0, 0,
            0, 0, 0, 0, 20, 0, 0, 0, 21, 22,
            23, 24, 0, 25, 0, 0, 0, 0, 0, 0,
            0, 26, 0, 0, 27, 0, 28, 0, 0, 0,
            29, 0, 0, 0, 0, 0, 0, 0, 7, 8,
            9, 10, 30, 0, 11, 0, 31, 0, 32, 33,
            12, 13, 0, 0, 0, 14, 15, 0, 0, 16,
            0, 0, 0, 17, 0, 18, 0, 0, 0, 0,
            19, 0, 0, 0, 0, 0, 0, 20, 0, 0,
            0, 21, 22, 23, 24, 0, 25, 0, 0, 0,
            0, 0, 0, 0, 26, 0, 0, 27, 83, 28,
            0, 0, 0, 29, 0, 0, 0, 0, 0, 0,
            0, 7, 8, 9, 10, 221, 0, 11, 0, 31,
            0, 32, 33, 12, 13, 0, 0, 0, 14, 15,
            0, 0, 16, 0, 0, 0, 17, 0, 18, 0,
            0, 0, 0, 19, 0, 0, 0, 0, 0, 0,
            20, 0, 0, 0, 21, 22, 23, 24, 0, 25,
            0, 0, 0, 0, 0, 0, 0, 26, 0, 0,
            27, 0, 28, 0, 0, 0, 29, 0, 0, 0,
            0, 0, 0, 0, 7, 8, 9, 10, 30, 355,
            11, 0, 31, 0, 32, 33, 12, 13, 0, 0,
            0, 14, 15, 0, 0, 16, 0, 0, 0, 17,
            0, 18, 0, 0, 0, 0, 19, 0, 0, 0,
            0, 0, 0, 20, 0, 0, 0, 21, 22, 23,
            24, 0, 25, 0, 0, 0, 0, 0, 0, 0,
            26, 0, 0, 27, 0, 28, 0, 0, 0, 29,
            0, 0, 0, 0, 0, 0, 0, 12, 12, 12,
            12, 30, 358, 12, 0, 31, 0, 32, 33, 12,
            12, 0, 0, 0, 12, 12, 0, 0, 12, 0,
            0, 0, 12, 0, 12, 0, 0, 0, 0, 12,
            0, 0, 0, 0, 0, 0, 12, 0, 0, 0,
            12, 12, 12, 12, 0, 12, 0, 0, 0, 0,
            0, 0, 0, 12, 0, 0, 12, 12, 12, 0,
            0, 0, 12, 0, 0, 0, 0, 0, 0, 0,
            13, 13, 13, 13, 12, 0, 13, 0, 12, 0,
            12, 12, 13, 13, 0, 0, 0, 13, 13, 0,
            0, 13, 0, 0, 0, 13, 0, 13, 0, 0,
            0, 0, 13, 0, 0, 0, 0, 0, 0, 13,
            0, 0, 0, 13, 13, 13, 13, 0, 13, 0,
            0, 0, 0, 0, 0, 0, 13, 0, 0, 13,
            13, 13, 0, 0, 0, 13, 0, 0, 0, 0,
            0, 0, 0, 31, 31, 31, 31, 13, 0, 31,
            0, 13, 0, 13, 13, 31, 31, 0, 0, 0,
            31, 31, 0, 0, 31, 0, 0, 0, 31, 0,
            31, 0, 0, 0, 0, 31, 0, 0, 0, 0,
            0, 0, 31, 0, 0, 0, 31, 31, 31, 31,
            0, 31, 0, 0, 0, 0, 0, 0, 0, 31,
            0, 0, 31, 31, 31, 0, 0, 0, 31, 0,
            0, 0, 0, 0, 0, 0, 31, 31, 31, 31,
            31, 0, 31, 0, 31, 0, 31, 31, 31, 31,
            0, 0, 0, 31, 31, 0, 0, 31, 0, 0,
            0, 31, 0, 31, 0, 0, 0, 0, 31, 0,
            0, 0, 0, 0, 0, 31, 0, 0, 0, 31,
            31, 31, 31, 0, 31, 0, 0, 0, 0, 0,
            0, 0, 31, 0, 0, 31, 0, 31, 0, 0,
            31, 31, 0, 0, 0, 0, 0, 0, 0, 7,
            8, 9, 10, 31, 0, 11, 0, 31, 0, 31,
            31, 12, 13, 0, 0, 0, 14, 15, 0, 0,
            16, 0, 0, 0, 17, 0, 18, 0, 0, 0,
            0, 19, 0, 0, 0, 0, 0, 0, 20, 0,
            0, 0, 21, 22, 23, 24, 0, 25, 0, 0,
            0, 0, 0, 0, 0, 26, 0, 0, 27, 0,
            28, 0, 0, 0, 29, 0, 0, 0, 0, 0,
            0, 0, 7, 8, 9, 10, 30, 0, 11, 0,
            31, 0, 32, 33, 12, 13, 0, 0, 0, 14,
            15, 0, 0, 16, 0, 0, 0, 17, 0, 18,
            0, 0, 0, 0, 19, 0, 0, 0, 0, 0,
            0, 20, 0, 0, 0, 21, 22, 23, 24, 0,
            25, 0, 0, 0, 0, 0, 0, 0, 26, 0,
            0, 27, 0, 28, 0, 0, 0, 29, 0, 0,
            0, 0, 0, 0, 0, 34, 34, 34, 34, 221,
            0, 34, 0, 31, 0, 32, 33, 34, 34, 0,
            0, 0, 34, 34, 0, 0, 34, 0, 0, 0,
            34, 0, 34, 0, 0, 0, 0, 34, 0, 0,
            0, 0, 0, 0, 34, 0, 0, 0, 34, 34,
            34, 34, 0, 34, 0, 0, 0, 0, 0, 0,
            0, 34, 0, 0, 34, 0, 34, 0, 0, 0,
            34, 0, 0, 0, 0, 0, 0, 0, 56, 56,
            56, 56, 34, 0, 56, 0, 34, 0, 34, 34,
            56, 56, 0, 0, 0, 56, 56, 0, 0, 56,
            0, 98, 0, 56, 0, 56, 0, 0, 0, 0,
            56, 0, 0, 0, 0, 99, 0, 56, 0, 0,
            0, 56, 56, 56, 56, 100, 56, 101, 102, 0,
            0, 0, 103, 0, 56, 0, 104, 56, 0, 56,
            0, 105, 0, 56, 106, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 56, 0, 107, 0, 56,
            0, 56, 56, 0, 0, 0, 108, 109, 0, 0,
            0, 98, 0, 0, 110, 111, 112, 113, 114, 115,
            116, 117, 118, 119, 120, 99, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 100, 0, 101, 102, 0,
            0, 0, 103, 0, 0, 0, 104, 0, 0, 0,
            0, 0, 0, 99, 106, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 101, 0, 107, 0, 0,
            103, 0, 0, 0, 0, 0, 108, 109, 0, 0,
            0, 0, 106, 0, 110, 111, 112, 113, 114, 115,
            116, 117, 118, 119, 120, 107, 99, 0, 0, 0,
            0, 0, 0, 0, 108, 109, 0, 0, 101, 0,
            0, 0, 110, 111, 112, 113, 114, 115, 116, 117,
            118, 119, 120, 0, 0, 106, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 101, 0, 107, 0,
            0, 0, 0, 0, 0, 0, 0, 108, 109, 0,
            0, 0, 0, 0, 0, 110, 111, 112, 113, 114,
            115, 116, 117, 118, 119, 120, 107, 0, 0, 0,
            0, 0, 0, 0, 0, 108, 109, 0, 0, 0,
            0, 0, 0, 110, 111, 112, 113, 114, 115, 116,
            117, 118, 119, 120,
        };

        private static readonly short[] yycheck = {
            1,
            257, 258, 324, 258, 72, 263, 93, 350, 95, 57,
            0, 221, 258, 61, 264, 257, 258, 287, 66, 67,
            68, 69, 295, 71, 334, 80, 1, 0, 261, 295,
            0, 79, 0, 81, 261, 261, 309, 329, 0, 382,
            257, 258, 319, 309, 277, 295, 295, 339, 23, 334,
            277, 277, 329, 55, 326, 272, 31, 32, 33, 269,
            309, 319, 257, 258, 151, 0, 266, 277, 278, 311,
            0, 279, 280, 257, 258, 0, 276, 335, 335, 334,
            267, 0, 0, 0, 0, 257, 258, 62, 334, 64,
            65, 159, 140, 161, 0, 0, 313, 72, 73, 316,
            0, 0, 335, 78, 291, 80, 108, 82, 335, 279,
            280, 298, 279, 280, 329, 329, 311, 332, 326, 331,
            335, 335, 97, 98, 99, 100, 101, 311, 103, 334,
            105, 106, 107, 220, 109, 110, 111, 112, 113, 114,
            115, 116, 117, 118, 119, 120, 121, 122, 331, 331,
            262, 334, 334, 128, 332, 130, 326, 334, 202, 326,
            215, 303, 304, 273, 342, 343, 344, 211, 212, 329,
            332, 334, 0, 335, 218, 335, 286, 329, 334, 154,
            332, 156, 292, 335, 159, 295, 161, 297, 159, 329,
            161, 193, 332, 237, 0, 335, 171, 172, 173, 309,
            202, 176, 334, 332, 332, 315, 335, 335, 263, 211,
            212, 329, 214, 257, 273, 334, 218, 335, 329, 332,
            329, 339, 335, 198, 335, 335, 335, 286, 203, 332,
            339, 332, 335, 292, 335, 237, 295, 305, 297, 334,
            215, 309, 244, 287, 288, 329, 221, 329, 332, 334,
            309, 335, 227, 335, 302, 257, 315, 273, 334, 329,
            334, 0, 329, 238, 319, 335, 310, 329, 335, 317,
            286, 329, 329, 335, 249, 262, 292, 335, 335, 295,
            335, 297, 334, 329, 329, 287, 288, 334, 263, 335,
            335, 334, 332, 309, 269, 335, 334, 334, 300, 315,
            273, 345, 277, 278, 306, 334, 285, 351, 310, 299,
            312, 332, 285, 286, 335, 288, 291, 290, 288, 292,
            290, 335, 295, 332, 297, 331, 288, 334, 290, 319,
            305, 335, 280, 306, 309, 308, 309, 335, 308, 329,
            308, 331, 315, 345, 319, 335, 308, 335, 350, 351,
            352, 0, 335, 335, 327, 273, 329, 327, 331, 329,
            335, 331, 335, 331, 299, 335, 410, 335, 286, 331,
            288, 290, 290, 335, 292, 335, 335, 295, 308, 297,
            382, 335, 288, 288, 290, 290, 325, 389, 306, 308,
            308, 309, 334, 368, 327, 370, 331, 315, 400, 299,
            335, 331, 308, 308, 332, 335, 331, 335, 410, 327,
            335, 329, 331, 331, 331, 331, 335, 335, 335, 335,
            395, 327, 424, 329, 265, 331, 331, 429, 403, 335,
            335, 331, 331, 335, 262, 335, 335, 265, 266, 267,
            415, 416, 0, 271, 321, 273, 332, 422, 276, 335,
            329, 279, 280, 281, 288, 283, 334, 285, 286, 335,
            288, 335, 290, 291, 292, 293, 294, 295, 334, 297,
            298, 299, 334, 295, 302, 302, 303, 304, 306, 307,
            308, 309, 310, 335, 290, 335, 264, 315, 284, 285,
            286, 319, 320, 268, 334, 323, 290, 328, 326, 327,
            332, 329, 308, 331, 332, 333, 329, 335, 334, 334,
            295, 339, 340, 341, 342, 343, 344, 345, 346, 347,
            348, 349, 350, 262, 295, 331, 265, 266, 267, 335,
            295, 0, 271, 306, 273, 314, 308, 276, 335, 268,
            279, 280, 281, 335, 283, 334, 285, 286, 335, 288,
            257, 290, 291, 292, 293, 294, 295, 299, 297, 298,
            299, 335, 0, 302, 262, 335, 280, 306, 307, 308,
            309, 310, 262, 285, 285, 5, 315, 1, 1, 58,
            319, 320, 300, 264, 323, 221, 221, 326, 327, 419,
            329, 341, 331, 332, 333, 416, 335, 412, 400, -1,
            339, 340, 341, 342, 343, 344, 345, 346, 347, 348,
            349, 350, -1, 262, -1, -1, 265, 266, 267, -1,
            0, -1, 271, -1, 273, -1, -1, 276, -1, -1,
            279, 280, 281, -1, 283, -1, 285, 286, -1, 288,
            -1, 290, 291, 292, 293, 294, 295, -1, 297, 298,
            299, -1, -1, 302, -1, -1, -1, 306, 307, 308,
            309, 310, -1, -1, -1, -1, 315, -1, -1, -1,
            319, 320, -1, -1, 323, -1, -1, 326, 327, -1,
            329, -1, 331, 332, 333, -1, 335, -1, -1, -1,
            339, 340, 341, 342, 343, 344, 345, 346, 347, 348,
            349, 350, -1, -1, 262, -1, -1, 265, 266, 267,
            0, -1, -1, 271, -1, 273, -1, -1, 276, -1,
            -1, 279, 280, 281, -1, 283, -1, 285, 286, -1,
            288, -1, 290, 291, 292, 293, 294, 295, -1, 297,
            298, 299, -1, -1, 302, -1, -1, -1, 306, 307,
            308, 309, 310, -1, -1, -1, -1, 315, -1, -1,
            -1, 319, 320, -1, -1, 323, -1, -1, 326, 327,
            -1, 329, -1, 331, 332, 333, -1, 335, -1, -1,
            -1, 339, 340, 341, 342, 343, 344, 345, 346, 347,
            348, 349, 350, 262, -1, -1, 265, 266, 267, 0,
            -1, -1, 271, -1, 273, -1, -1, 276, -1, -1,
            279, 280, 281, -1, 283, -1, 285, 286, -1, 288,
            -1, 290, 291, 292, 293, 294, 295, -1, 297, 298,
            299, -1, -1, 302, -1, -1, -1, 306, 307, 308,
            309, 310, -1, -1, -1, -1, 315, -1, -1, -1,
            319, 320, -1, -1, 323, -1, -1, 326, 327, -1,
            329, -1, 331, -1, 333, -1, 335, -1, -1, -1,
            339, 340, 341, 342, 343, 344, 345, 346, 347, 348,
            349, 350, 262, -1, -1, 265, 266, 267, 0, -1,
            -1, 271, -1, 273, -1, -1, 276, -1, -1, 279,
            280, 281, -1, 283, -1, 285, 286, -1, 288, -1,
            290, 291, 292, 293, 294, 295, -1, 297, 298, 299,
            -1, -1, 302, -1, -1, -1, 306, 307, 308, 309,
            310, -1, -1, -1, -1, 315, -1, -1, -1, 319,
            320, -1, -1, 323, -1, -1, 326, 327, -1, 329,
            -1, 331, -1, 333, -1, 335, -1, -1, -1, 339,
            340, 341, 342, 343, 344, 345, 346, 347, 348, 349,
            350, -1, 262, -1, -1, 265, 266, 267, 0, -1,
            -1, 271, -1, 273, -1, -1, 276, -1, -1, 279,
            280, 281, -1, 283, -1, 285, 286, -1, 288, -1,
            290, 291, 292, 293, 294, 295, -1, 297, 298, 299,
            -1, -1, 302, -1, -1, -1, 306, 307, 308, 309,
            310, -1, -1, -1, -1, 315, -1, -1, -1, 319,
            320, -1, -1, 323, -1, -1, 326, 327, -1, 329,
            -1, 331, -1, 333, -1, 335, -1, -1, -1, 339,
            340, 341, 342, 343, 344, 345, 346, 347, 348, 349,
            350, 262, -1, -1, 265, 266, 267, 0, -1, -1,
            271, -1, 273, -1, -1, 276, -1, -1, 279, 280,
            281, -1, 283, -1, 285, 286, -1, 288, -1, 290,
            291, 292, 293, 294, 295, -1, 297, 298, 299, -1,
            -1, 302, -1, -1, -1, 306, 307, 308, 309, 310,
            -1, -1, -1, -1, 315, -1, -1, -1, 319, 320,
            -1, -1, 323, -1, -1, 326, 327, -1, 329, -1,
            331, -1, 333, -1, 335, -1, -1, -1, 339, 340,
            341, 342, 343, 344, 345, 346, 347, 348, 349, 350,
            262, -1, -1, 265, 266, 267, 0, -1, -1, 271,
            -1, 273, -1, -1, 276, -1, -1, 279, 280, 281,
            -1, 283, -1, 285, 286, -1, 288, -1, 290, 291,
            292, 293, 294, 295, 0, 297, 298, 299, -1, -1,
            302, -1, -1, -1, 306, 307, 308, 309, 310, -1,
            -1, -1, -1, 315, -1, -1, -1, 319, 320, -1,
            -1, 323, -1, -1, 326, 327, -1, 329, -1, 331,
            -1, 333, -1, 335, -1, -1, -1, 339, 340, 341,
            342, 343, 344, 345, 346, 347, 348, 349, 350, -1,
            262, 0, -1, 265, 266, 267, -1, -1, -1, 271,
            -1, 273, -1, -1, 276, -1, -1, 279, 280, 281,
            -1, 283, -1, 285, 286, -1, 288, -1, 290, 291,
            292, 293, 294, 295, -1, 297, 298, 299, -1, -1,
            302, -1, -1, -1, 306, 307, 308, 309, 310, -1,
            -1, -1, -1, 315, -1, -1, -1, 319, 320, -1,
            -1, 323, -1, -1, 326, 327, -1, 329, -1, 331,
            -1, 333, -1, 335, -1, -1, -1, 339, 340, 341,
            -1, -1, -1, 345, 346, 347, 348, 349, 350, 262,
            0, -1, 265, 266, 267, -1, -1, -1, 271, -1,
            273, -1, -1, 276, -1, -1, 279, 280, 281, -1,
            283, -1, 285, 286, -1, 288, -1, 290, 291, 292,
            293, 294, 295, -1, 297, 298, 299, -1, -1, 302,
            -1, -1, -1, 306, 307, 308, 309, 310, -1, -1,
            -1, -1, 315, -1, -1, -1, 319, 320, -1, -1,
            323, -1, -1, 326, 327, -1, 329, -1, 331, -1,
            333, -1, 335, -1, -1, -1, 339, 340, 341, -1,
            -1, -1, 345, 346, 347, 348, 349, 350, 262, 0,
            -1, 265, 266, 267, -1, -1, -1, 271, -1, 273,
            -1, -1, 276, -1, -1, 279, 280, 281, -1, 283,
            -1, 285, 286, -1, 288, -1, 290, 291, 292, 293,
            294, 295, -1, 297, 298, 299, -1, -1, 302, -1,
            -1, -1, 306, 307, 308, 309, 310, -1, -1, -1,
            -1, 315, 288, -1, 290, 319, 320, -1, -1, 323,
            -1, -1, 326, 327, -1, 329, -1, 331, -1, 333,
            -1, 335, 308, -1, -1, 339, -1, -1, -1, -1,
            -1, 345, 346, 262, -1, -1, 265, 266, 267, 0,
            -1, 327, 271, -1, 273, 331, -1, 276, -1, 335,
            279, 280, 281, -1, 283, -1, 285, 286, -1, 288,
            -1, 290, 291, 292, 293, 294, 295, -1, 297, 298,
            299, -1, -1, 302, -1, -1, -1, 306, 307, 308,
            309, 310, -1, -1, -1, -1, 315, -1, -1, -1,
            319, 320, -1, -1, 323, -1, 273, 326, 327, -1,
            329, -1, 331, -1, 333, 0, 335, -1, -1, 286,
            339, -1, -1, -1, -1, 292, 345, 346, 295, -1,
            297, -1, 262, -1, -1, 265, 266, 267, -1, -1,
            -1, 271, 309, 273, -1, -1, 276, -1, 315, 279,
            280, 281, -1, 283, -1, 285, 286, -1, 288, -1,
            290, 291, 292, 293, 294, 295, -1, 297, 298, 299,
            -1, -1, 302, -1, -1, -1, 306, 307, 308, 309,
            310, -1, -1, -1, -1, 315, 0, -1, -1, 319,
            320, -1, -1, 323, -1, -1, 326, 327, -1, 329,
            -1, 331, -1, 333, -1, 335, -1, -1, -1, 339,
            -1, -1, -1, -1, -1, 345, 346, -1, -1, -1,
            -1, 262, -1, -1, 265, 266, 267, -1, -1, -1,
            271, -1, 273, -1, -1, 276, -1, -1, 279, 280,
            281, -1, 283, -1, 285, 286, -1, 288, -1, 290,
            291, 292, 293, 294, 295, -1, 297, 298, 299, -1,
            -1, 302, 0, -1, -1, 306, 307, 308, 309, 310,
            -1, -1, -1, -1, 315, -1, -1, -1, 319, 320,
            -1, -1, 323, -1, -1, 326, 327, -1, 329, -1,
            331, -1, 333, -1, 335, -1, -1, -1, 339, -1,
            -1, -1, -1, -1, 345, 346, 257, 258, 259, 260,
            -1, -1, 263, -1, -1, -1, -1, -1, 269, 270,
            -1, -1, -1, 274, 275, -1, -1, 278, -1, -1,
            -1, 282, 332, 284, -1, -1, 287, 0, 289, -1,
            340, 341, 342, 343, 344, 296, -1, -1, -1, 300,
            301, 302, 303, -1, 305, -1, -1, -1, -1, -1,
            -1, -1, 313, -1, -1, 316, 317, 318, -1, -1,
            -1, 322, -1, -1, -1, -1, -1, 262, -1, -1,
            265, 266, 267, 334, -1, -1, 271, 338, 273, 340,
            341, 276, -1, -1, 279, 280, 281, -1, 283, -1,
            285, 286, -1, 288, -1, 290, 291, 292, 293, 294,
            295, -1, 297, 298, 299, 0, -1, 302, -1, -1,
            -1, 306, 307, 308, 309, 310, -1, -1, -1, -1,
            315, -1, -1, -1, 319, 320, -1, -1, 323, -1,
            -1, 326, 327, -1, 329, -1, 331, -1, 262, -1,
            335, 265, 266, 267, 339, -1, -1, 271, -1, 273,
            -1, -1, 276, -1, -1, 279, 280, 281, -1, 283,
            -1, 285, 286, -1, 288, -1, 290, 291, 292, 293,
            294, 295, -1, 297, 298, 299, 0, -1, 302, -1,
            -1, -1, 306, 307, 308, 309, 310, -1, -1, -1,
            -1, 315, -1, -1, -1, 319, 320, -1, -1, 323,
            -1, -1, 326, 327, -1, 329, -1, 331, -1, -1,
            -1, 335, -1, -1, 262, 339, -1, 265, 266, 267,
            -1, -1, -1, 271, -1, 273, -1, -1, 276, -1,
            -1, 279, 280, 281, -1, 283, -1, 285, 286, -1,
            288, -1, 290, 291, 292, 293, 294, 295, -1, 297,
            298, 299, 0, -1, 302, -1, -1, -1, 306, 307,
            308, 309, 310, -1, -1, -1, -1, 315, -1, -1,
            -1, 319, 320, -1, -1, 323, -1, -1, 326, 327,
            -1, 329, -1, 331, -1, -1, -1, 335, -1, 262,
            -1, 339, 265, 266, 267, -1, -1, -1, 271, -1,
            273, -1, -1, 276, -1, -1, 279, 280, 281, -1,
            283, -1, 285, 286, -1, 288, -1, 290, 291, 292,
            293, 294, 295, 0, 297, 298, 299, -1, -1, 302,
            -1, -1, -1, 306, 307, 308, 309, 310, -1, -1,
            -1, -1, 315, -1, -1, -1, 319, 320, -1, -1,
            323, -1, -1, 326, 327, -1, 329, -1, 331, -1,
            -1, -1, 335, -1, -1, -1, 339, 262, -1, -1,
            265, 266, 267, -1, -1, -1, 271, -1, 273, -1,
            -1, 276, -1, -1, 279, 280, 281, -1, 283, -1,
            285, 286, -1, 288, 0, 290, 291, 292, -1, 294,
            295, -1, 297, 298, 299, -1, -1, 302, -1, -1,
            -1, 306, 307, 308, 309, 310, -1, -1, -1, -1,
            315, -1, -1, -1, 319, 320, -1, -1, 323, -1,
            -1, 326, 327, -1, 329, -1, 331, -1, 262, -1,
            335, 265, 266, 267, 339, -1, -1, 271, -1, 273,
            -1, -1, 276, -1, -1, 279, 280, 281, -1, 283,
            -1, 285, 286, -1, 288, 0, 290, 291, 292, -1,
            294, 295, -1, 297, 298, 299, -1, -1, 302, -1,
            -1, -1, 306, 307, 308, 309, 310, -1, -1, -1,
            -1, 315, -1, -1, -1, 319, 320, -1, -1, 323,
            -1, -1, 326, 327, -1, 329, -1, 331, -1, -1,
            -1, 335, -1, -1, 262, 339, -1, 265, 266, 267,
            -1, -1, -1, 271, -1, 273, -1, -1, 276, -1,
            -1, 279, 280, 281, -1, 283, 0, 285, 286, -1,
            288, -1, 290, 291, 292, -1, 294, 295, -1, 297,
            298, 299, -1, -1, 302, -1, -1, -1, 306, 307,
            308, 309, 310, -1, -1, -1, -1, 315, -1, -1,
            -1, 319, 320, -1, -1, -1, -1, -1, 326, 327,
            -1, 329, -1, 331, -1, 262, -1, 335, 265, 266,
            267, 339, -1, -1, 271, -1, 273, -1, -1, 276,
            -1, -1, 279, 280, -1, -1, 283, 0, 285, 286,
            -1, 288, -1, 290, 291, 292, -1, 294, 295, -1,
            297, 298, 299, -1, -1, 302, -1, -1, -1, 306,
            307, 308, 309, 310, -1, -1, -1, -1, 315, -1,
            -1, -1, 319, 320, -1, -1, -1, -1, -1, 326,
            327, -1, 329, -1, 331, -1, 262, -1, 335, 265,
            266, 267, 339, -1, -1, 271, -1, 273, -1, -1,
            276, -1, -1, 279, 280, -1, -1, 283, 0, 285,
            286, -1, 288, -1, 290, 291, 292, -1, 294, 295,
            -1, 297, 298, 299, -1, -1, 302, -1, -1, -1,
            306, 307, 308, 309, -1, -1, -1, -1, -1, 315,
            -1, -1, -1, 319, 320, -1, -1, -1, -1, -1,
            326, 327, -1, 329, -1, 331, -1, 262, -1, 335,
            265, 266, 267, 339, -1, -1, 271, -1, 273, -1,
            -1, 276, -1, -1, 279, 280, -1, -1, 283, 0,
            285, 286, -1, 288, -1, 290, 291, 292, -1, 294,
            295, -1, 297, 298, 299, -1, -1, 302, -1, -1,
            -1, 306, 307, 308, 309, -1, -1, -1, -1, -1,
            315, -1, -1, -1, 319, 320, -1, -1, -1, -1,
            -1, 326, 327, -1, 329, -1, 331, -1, 262, -1,
            335, 265, 266, 267, 339, -1, -1, 271, -1, 273,
            -1, -1, 276, -1, -1, 279, 280, -1, -1, 283,
            0, 285, 286, -1, 288, -1, 290, -1, 292, -1,
            294, 295, -1, 297, -1, 299, -1, -1, 302, -1,
            -1, -1, 306, 307, 308, 309, -1, -1, -1, -1,
            -1, 315, -1, -1, -1, 319, 320, -1, -1, -1,
            -1, -1, 326, 327, -1, 329, -1, 331, -1, 262,
            -1, 335, 265, 266, 267, 339, -1, -1, 271, -1,
            273, -1, -1, 276, -1, -1, 279, 280, -1, -1,
            -1, 0, 285, 286, -1, 288, -1, 290, 291, 292,
            -1, 294, 295, -1, 297, -1, 299, -1, -1, 302,
            -1, -1, -1, 306, 307, 308, 309, -1, -1, -1,
            -1, -1, 315, -1, -1, -1, 319, 320, -1, -1,
            -1, -1, -1, 326, 327, -1, 329, -1, 331, -1,
            262, -1, 335, 265, 266, 267, 339, -1, -1, 271,
            -1, 273, -1, -1, 276, -1, -1, 279, 280, -1,
            -1, 283, 0, 285, 286, -1, 288, -1, 290, -1,
            292, -1, 294, 295, -1, 297, -1, 299, -1, -1,
            302, -1, -1, -1, 306, 307, 308, 309, -1, -1,
            -1, -1, -1, 315, -1, -1, -1, 319, 320, -1,
            -1, -1, -1, -1, 326, 327, -1, 329, -1, 331,
            -1, 262, -1, 335, 265, 266, 267, 339, -1, -1,
            271, -1, 273, -1, -1, 276, -1, -1, 279, 280,
            -1, -1, -1, 0, 285, 286, -1, 288, -1, 290,
            291, 292, -1, 294, 295, -1, 297, -1, 299, -1,
            -1, 302, -1, -1, -1, 306, 307, 308, 309, -1,
            -1, -1, -1, -1, 315, -1, -1, -1, 319, 320,
            -1, -1, -1, -1, -1, 326, 327, -1, 329, -1,
            331, -1, 262, -1, 335, 265, 266, -1, 339, -1,
            0, 271, -1, 273, -1, -1, 276, -1, -1, 279,
            280, -1, -1, 283, -1, 285, 286, -1, 288, -1,
            290, -1, 292, -1, -1, 295, -1, 297, -1, 299,
            -1, -1, -1, -1, -1, -1, 306, 307, 308, 309,
            -1, -1, -1, -1, -1, 315, -1, -1, 0, 319,
            320, -1, -1, -1, -1, -1, 326, 327, -1, 329,
            -1, 331, -1, 262, -1, 335, 265, 266, -1, 339,
            -1, -1, 271, -1, 273, -1, -1, 276, -1, -1,
            279, 280, -1, -1, 283, -1, 285, 286, -1, 288,
            -1, 290, -1, 292, -1, -1, 295, -1, 297, -1,
            299, -1, -1, -1, -1, -1, -1, 306, 307, 308,
            309, -1, -1, -1, -1, -1, 315, -1, -1, -1,
            319, 320, -1, -1, -1, -1, -1, 326, 327, -1,
            329, -1, 331, -1, 262, -1, 335, 265, 266, -1,
            339, -1, -1, 271, -1, 273, -1, -1, 276, -1,
            -1, 279, 280, -1, -1, 283, -1, 285, 286, -1,
            288, -1, 290, -1, 292, -1, -1, 295, -1, 297,
            -1, 299, -1, -1, -1, -1, -1, -1, 306, 307,
            308, 309, -1, -1, -1, -1, -1, 315, -1, -1,
            -1, 319, 320, -1, -1, -1, -1, -1, 326, 327,
            -1, 329, -1, 331, -1, 262, -1, 335, 265, 266,
            -1, 339, -1, -1, 271, -1, 273, -1, -1, 276,
            -1, -1, 279, 280, -1, -1, 283, -1, 285, 286,
            -1, 288, -1, 290, -1, 292, -1, -1, 295, -1,
            297, -1, 299, -1, -1, -1, -1, -1, -1, 306,
            307, 308, 309, -1, -1, -1, -1, -1, 315, -1,
            -1, -1, 319, 320, -1, 265, 266, -1, -1, 326,
            327, 271, 329, 273, 331, -1, 276, -1, 335, 279,
            280, -1, 339, 283, -1, 285, 286, -1, 288, -1,
            290, -1, 292, -1, -1, 295, -1, 297, -1, 299,
            -1, -1, -1, -1, -1, -1, 306, 307, 308, 309,
            -1, -1, -1, -1, -1, 315, -1, -1, -1, 319,
            320, 273, -1, -1, -1, -1, 326, 327, -1, 329,
            -1, 331, -1, -1, 286, 335, 288, -1, 290, 339,
            292, -1, -1, 295, -1, 297, 281, -1, -1, -1,
            -1, -1, -1, -1, -1, 262, 308, 309, 293, 266,
            267, -1, -1, 315, 271, -1, -1, -1, -1, 276,
            -1, -1, -1, -1, 281, 327, -1, 329, -1, 331,
            -1, -1, -1, 335, 291, -1, 293, 294, 323, -1,
            -1, 298, -1, -1, -1, 302, -1, 332, 333, -1,
            307, -1, -1, 310, -1, 340, 341, 342, 343, 344,
            345, 346, 347, 348, 349, 350, 323, -1, 281, -1,
            283, -1, -1, -1, -1, 332, 333, 262, -1, -1,
            293, -1, 267, 340, 341, 342, 343, 344, 345, 346,
            347, 348, 349, 350, -1, -1, 281, 310, -1, -1,
            -1, -1, -1, -1, -1, -1, 291, -1, 293, 294,
            323, -1, -1, 298, -1, -1, -1, 302, -1, 332,
            333, -1, 307, -1, -1, 310, -1, 340, 341, 342,
            343, 344, 345, 346, 347, 348, 349, 350, 323, -1,
            -1, -1, -1, -1, 329, -1, -1, 332, 333, 262,
            335, -1, -1, -1, 267, 340, 341, 342, 343, 344,
            345, 346, 347, 348, 349, 350, -1, -1, 281, -1,
            -1, -1, -1, -1, -1, -1, -1, -1, 291, -1,
            293, 294, -1, -1, -1, 298, -1, -1, -1, 302,
            -1, -1, -1, -1, 307, -1, -1, 310, -1, -1,
            -1, -1, -1, -1, -1, -1, -1, 320, -1, -1,
            323, -1, -1, -1, -1, -1, -1, -1, -1, 332,
            333, 262, -1, -1, 265, -1, 267, 340, 341, 342,
            343, 344, 345, 346, 347, 348, 349, 350, -1, -1,
            281, -1, -1, -1, -1, -1, -1, -1, -1, 262,
            291, -1, 293, 294, 267, -1, -1, 298, -1, -1,
            -1, 302, -1, -1, -1, -1, 307, -1, 281, 310,
            -1, -1, -1, -1, -1, -1, -1, -1, 291, -1,
            293, 294, 323, -1, -1, 298, -1, -1, -1, 302,
            -1, 332, 333, -1, 307, -1, -1, 310, -1, 340,
            341, 342, 343, 344, 345, 346, 347, 348, 349, 350,
            323, -1, -1, -1, -1, -1, 329, -1, -1, 332,
            333, 262, -1, -1, -1, -1, 267, 340, 341, 342,
            343, 344, 345, 346, 347, 348, 349, 350, -1, -1,
            281, -1, -1, -1, -1, -1, -1, -1, -1, 262,
            291, -1, 293, 294, 267, -1, -1, 298, -1, -1,
            -1, 302, -1, -1, -1, -1, 307, -1, 281, 310,
            -1, -1, -1, -1, -1, -1, -1, -1, 291, -1,
            293, 294, 323, -1, -1, 298, -1, -1, 329, 302,
            -1, 332, 333, -1, 307, -1, -1, 310, -1, 340,
            341, 342, 343, 344, 345, 346, 347, 348, 349, 350,
            323, -1, -1, -1, -1, -1, 329, -1, -1, 332,
            333, 262, -1, 293, 265, -1, 267, 340, 341, 342,
            343, 344, 345, 346, 347, 348, 349, 350, -1, -1,
            281, -1, -1, -1, -1, -1, -1, -1, -1, -1,
            291, -1, 293, 294, -1, -1, -1, 298, -1, -1,
            -1, 302, 332, 333, -1, -1, 307, -1, -1, 310,
            340, 341, 342, 343, 344, 345, 346, 347, 348, 349,
            350, -1, 323, -1, -1, -1, -1, -1, -1, -1,
            -1, 332, 333, 262, -1, -1, 265, -1, 267, 340,
            341, 342, 343, 344, 345, 346, 347, 348, 349, 350,
            -1, -1, 281, -1, -1, -1, -1, -1, -1, -1,
            -1, -1, 291, -1, 293, 294, -1, -1, -1, 298,
            -1, -1, -1, 302, 332, 333, -1, -1, 307, -1,
            -1, 310, 340, 341, 342, 343, 344, 345, 346, 347,
            348, 349, 350, -1, 323, -1, -1, -1, -1, -1,
            -1, -1, -1, 332, 333, 262, -1, -1, -1, -1,
            267, 340, 341, 342, 343, 344, 345, 346, 347, 348,
            349, 350, -1, -1, 281, -1, -1, -1, -1, -1,
            -1, -1, -1, 262, 291, -1, 293, 294, 267, -1,
            -1, 298, -1, -1, -1, 302, -1, -1, -1, -1,
            307, -1, 281, 310, -1, -1, -1, -1, -1, -1,
            -1, -1, 291, 320, 293, 294, 323, -1, -1, 298,
            -1, -1, -1, 302, -1, 332, 333, -1, 307, -1,
            -1, 310, -1, 340, 341, 342, 343, 344, 345, 346,
            347, 348, 349, 350, 323, -1, -1, -1, -1, -1,
            329, -1, -1, 332, 333, 262, -1, -1, -1, -1,
            267, 340, 341, 342, 343, 344, 345, 346, 347, 348,
            349, 350, -1, -1, 281, -1, -1, -1, -1, -1,
            -1, -1, -1, 262, 291, -1, 293, 294, 267, -1,
            -1, 298, -1, -1, -1, 302, -1, -1, -1, -1,
            307, -1, 281, 310, -1, -1, -1, -1, -1, -1,
            -1, -1, 291, -1, 293, 294, 323, -1, -1, 298,
            -1, -1, -1, 302, -1, 332, 333, -1, 307, -1,
            -1, 310, -1, 340, 341, 342, 343, 344, 345, 346,
            347, 348, 349, 350, 323, -1, -1, -1, -1, -1,
            -1, -1, -1, 332, 333, 262, -1, -1, -1, -1,
            267, 340, 341, 342, 343, 344, 345, 346, 347, 348,
            349, 350, -1, -1, 281, -1, -1, -1, -1, -1,
            -1, -1, -1, 262, 291, -1, 293, 294, 267, -1,
            -1, 298, -1, -1, -1, 302, -1, -1, -1, -1,
            307, -1, 281, 310, -1, -1, -1, -1, -1, -1,
            -1, -1, 291, -1, 293, 294, 323, -1, -1, 298,
            -1, -1, -1, 302, -1, 332, 333, -1, 307, -1,
            -1, 310, -1, 340, 341, 342, 343, 344, 345, 346,
            347, 348, 349, 350, 323, -1, 281, -1, 283, -1,
            -1, -1, -1, 332, 333, 262, -1, -1, 293, -1,
            267, 340, 341, 342, 343, 344, 345, 346, 347, 348,
            349, 350, -1, -1, 281, 310, -1, -1, -1, -1,
            -1, -1, -1, -1, 291, -1, 293, 294, 323, -1,
            -1, 298, -1, -1, -1, 302, -1, 332, 333, -1,
            -1, -1, -1, 310, -1, 340, 341, 342, 343, 344,
            345, 346, 347, 348, 349, 350, 323, -1, -1, -1,
            -1, -1, -1, -1, -1, 332, 333, -1, -1, -1,
            -1, -1, -1, 340, 341, 342, 343, 344, 345, 346,
            347, 348, 349, 350, 257, 258, 259, 260, 261, -1,
            263, -1, -1, -1, -1, -1, 269, 270, -1, -1,
            -1, 274, 275, -1, 277, 278, -1, -1, -1, 282,
            -1, 284, -1, -1, -1, -1, 289, -1, -1, -1,
            -1, -1, -1, 296, -1, -1, -1, 300, 301, 302,
            303, -1, 305, -1, -1, -1, -1, -1, -1, -1,
            313, -1, -1, 316, -1, 318, -1, -1, 321, 322,
            -1, -1, 325, -1, -1, -1, -1, 257, 258, 259,
            260, 334, -1, 263, -1, 338, -1, 340, 341, 269,
            270, -1, -1, -1, 274, 275, -1, -1, 278, -1,
            -1, -1, 282, -1, 284, -1, -1, 287, -1, 289,
            -1, -1, -1, -1, -1, -1, 296, -1, -1, -1,
            300, 301, 302, 303, -1, 305, -1, -1, -1, -1,
            -1, -1, -1, 313, -1, -1, 316, 317, 318, -1,
            -1, -1, 322, -1, -1, -1, -1, -1, -1, -1,
            257, 258, 259, 260, 334, -1, 263, -1, 338, -1,
            340, 341, 269, 270, -1, -1, -1, 274, 275, -1,
            -1, 278, -1, -1, -1, 282, -1, 284, -1, -1,
            -1, -1, 289, -1, -1, -1, -1, -1, -1, 296,
            -1, -1, -1, 300, 301, 302, 303, -1, 305, -1,
            -1, -1, -1, -1, -1, -1, 313, -1, 332, 316,
            317, 318, -1, -1, -1, 322, 340, 341, 342, 343,
            344, -1, -1, 347, 348, 349, 350, 334, -1, -1,
            -1, 338, -1, 340, 341, 257, 258, 259, 260, 261,
            -1, 263, -1, -1, -1, -1, -1, 269, 270, -1,
            -1, -1, 274, 275, -1, -1, 278, -1, -1, -1,
            282, -1, 284, -1, -1, -1, -1, 289, -1, -1,
            -1, -1, -1, -1, 296, -1, -1, -1, 300, 301,
            302, 303, -1, 305, -1, -1, -1, -1, -1, -1,
            -1, 313, -1, -1, 316, -1, 318, -1, -1, -1,
            322, -1, -1, -1, -1, -1, -1, -1, 257, 258,
            259, 260, 334, -1, 263, -1, 338, -1, 340, 341,
            269, 270, -1, -1, -1, 274, 275, -1, -1, 278,
            -1, -1, -1, 282, -1, 284, -1, -1, -1, -1,
            289, -1, -1, -1, -1, -1, -1, 296, -1, -1,
            -1, 300, 301, 302, 303, -1, 305, -1, -1, -1,
            -1, -1, -1, -1, 313, -1, -1, 316, 317, 318,
            -1, -1, -1, 322, -1, -1, -1, -1, -1, -1,
            -1, 257, 258, 259, 260, 334, -1, 263, -1, 338,
            -1, 340, 341, 269, 270, -1, -1, -1, 274, 275,
            -1, -1, 278, -1, -1, -1, 282, -1, 284, -1,
            -1, -1, -1, 289, -1, -1, -1, -1, -1, -1,
            296, -1, -1, -1, 300, 301, 302, 303, -1, 305,
            -1, -1, -1, -1, -1, -1, -1, 313, -1, -1,
            316, -1, 318, -1, -1, -1, 322, -1, -1, -1,
            -1, -1, -1, -1, 257, 258, 259, 260, 334, 335,
            263, -1, 338, -1, 340, 341, 269, 270, -1, -1,
            -1, 274, 275, -1, -1, 278, -1, -1, -1, 282,
            -1, 284, -1, -1, -1, -1, 289, -1, -1, -1,
            -1, -1, -1, 296, -1, -1, -1, 300, 301, 302,
            303, -1, 305, -1, -1, -1, -1, -1, -1, -1,
            313, -1, -1, 316, -1, 318, -1, -1, -1, 322,
            -1, -1, -1, -1, -1, -1, -1, 257, 258, 259,
            260, 334, 335, 263, -1, 338, -1, 340, 341, 269,
            270, -1, -1, -1, 274, 275, -1, -1, 278, -1,
            -1, -1, 282, -1, 284, -1, -1, -1, -1, 289,
            -1, -1, -1, -1, -1, -1, 296, -1, -1, -1,
            300, 301, 302, 303, -1, 305, -1, -1, -1, -1,
            -1, -1, -1, 313, -1, -1, 316, 317, 318, -1,
            -1, -1, 322, -1, -1, -1, -1, -1, -1, -1,
            257, 258, 259, 260, 334, -1, 263, -1, 338, -1,
            340, 341, 269, 270, -1, -1, -1, 274, 275, -1,
            -1, 278, -1, -1, -1, 282, -1, 284, -1, -1,
            -1, -1, 289, -1, -1, -1, -1, -1, -1, 296,
            -1, -1, -1, 300, 301, 302, 303, -1, 305, -1,
            -1, -1, -1, -1, -1, -1, 313, -1, -1, 316,
            317, 318, -1, -1, -1, 322, -1, -1, -1, -1,
            -1, -1, -1, 257, 258, 259, 260, 334, -1, 263,
            -1, 338, -1, 340, 341, 269, 270, -1, -1, -1,
            274, 275, -1, -1, 278, -1, -1, -1, 282, -1,
            284, -1, -1, -1, -1, 289, -1, -1, -1, -1,
            -1, -1, 296, -1, -1, -1, 300, 301, 302, 303,
            -1, 305, -1, -1, -1, -1, -1, -1, -1, 313,
            -1, -1, 316, 317, 318, -1, -1, -1, 322, -1,
            -1, -1, -1, -1, -1, -1, 257, 258, 259, 260,
            334, -1, 263, -1, 338, -1, 340, 341, 269, 270,
            -1, -1, -1, 274, 275, -1, -1, 278, -1, -1,
            -1, 282, -1, 284, -1, -1, -1, -1, 289, -1,
            -1, -1, -1, -1, -1, 296, -1, -1, -1, 300,
            301, 302, 303, -1, 305, -1, -1, -1, -1, -1,
            -1, -1, 313, -1, -1, 316, -1, 318, -1, -1,
            321, 322, -1, -1, -1, -1, -1, -1, -1, 257,
            258, 259, 260, 334, -1, 263, -1, 338, -1, 340,
            341, 269, 270, -1, -1, -1, 274, 275, -1, -1,
            278, -1, -1, -1, 282, -1, 284, -1, -1, -1,
            -1, 289, -1, -1, -1, -1, -1, -1, 296, -1,
            -1, -1, 300, 301, 302, 303, -1, 305, -1, -1,
            -1, -1, -1, -1, -1, 313, -1, -1, 316, -1,
            318, -1, -1, -1, 322, -1, -1, -1, -1, -1,
            -1, -1, 257, 258, 259, 260, 334, -1, 263, -1,
            338, -1, 340, 341, 269, 270, -1, -1, -1, 274,
            275, -1, -1, 278, -1, -1, -1, 282, -1, 284,
            -1, -1, -1, -1, 289, -1, -1, -1, -1, -1,
            -1, 296, -1, -1, -1, 300, 301, 302, 303, -1,
            305, -1, -1, -1, -1, -1, -1, -1, 313, -1,
            -1, 316, -1, 318, -1, -1, -1, 322, -1, -1,
            -1, -1, -1, -1, -1, 257, 258, 259, 260, 334,
            -1, 263, -1, 338, -1, 340, 341, 269, 270, -1,
            -1, -1, 274, 275, -1, -1, 278, -1, -1, -1,
            282, -1, 284, -1, -1, -1, -1, 289, -1, -1,
            -1, -1, -1, -1, 296, -1, -1, -1, 300, 301,
            302, 303, -1, 305, -1, -1, -1, -1, -1, -1,
            -1, 313, -1, -1, 316, -1, 318, -1, -1, -1,
            322, -1, -1, -1, -1, -1, -1, -1, 257, 258,
            259, 260, 334, -1, 263, -1, 338, -1, 340, 341,
            269, 270, -1, -1, -1, 274, 275, -1, -1, 278,
            -1, 267, -1, 282, -1, 284, -1, -1, -1, -1,
            289, -1, -1, -1, -1, 281, -1, 296, -1, -1,
            -1, 300, 301, 302, 303, 291, 305, 293, 294, -1,
            -1, -1, 298, -1, 313, -1, 302, 316, -1, 318,
            -1, 307, -1, 322, 310, -1, -1, -1, -1, -1,
            -1, -1, -1, -1, -1, 334, -1, 323, -1, 338,
            -1, 340, 341, -1, -1, -1, 332, 333, -1, -1,
            -1, 267, -1, -1, 340, 341, 342, 343, 344, 345,
            346, 347, 348, 349, 350, 281, -1, -1, -1, -1,
            -1, -1, -1, -1, -1, 291, -1, 293, 294, -1,
            -1, -1, 298, -1, -1, -1, 302, -1, -1, -1,
            -1, -1, -1, 281, 310, -1, -1, -1, -1, -1,
            -1, -1, -1, -1, -1, 293, -1, 323, -1, -1,
            298, -1, -1, -1, -1, -1, 332, 333, -1, -1,
            -1, -1, 310, -1, 340, 341, 342, 343, 344, 345,
            346, 347, 348, 349, 350, 323, 281, -1, -1, -1,
            -1, -1, -1, -1, 332, 333, -1, -1, 293, -1,
            -1, -1, 340, 341, 342, 343, 344, 345, 346, 347,
            348, 349, 350, -1, -1, 310, -1, -1, -1, -1,
            -1, -1, -1, -1, -1, -1, 293, -1, 323, -1,
            -1, -1, -1, -1, -1, -1, -1, 332, 333, -1,
            -1, -1, -1, -1, -1, 340, 341, 342, 343, 344,
            345, 346, 347, 348, 349, 350, 323, -1, -1, -1,
            -1, -1, -1, -1, -1, 332, 333, -1, -1, -1,
            -1, -1, -1, 340, 341, 342, 343, 344, 345, 346,
            347, 348, 349, 350,
        };

        private static short YYFINAL = 2;
        private static short YYMAXTOKEN = 352;

        private static readonly String[] yyname = {
            "end-of-file", null, null, null, null, null, null, null, null, null, null, null, null,
            null,
            null, null, null, null, null, null, null, null, null, null, null, null, null, null,
            null, null,
            null, null, null, null, null, null, null, null, null, null, null, null, null, null,
            null, null,
            null, null, null, null, null, null, null, null, null, null, null, null, null, null,
            null, null,
            null, null, null, null, null, null, null, null, null, null, null, null, null, null,
            null, null,
            null, null, null, null, null, null, null, null, null, null, null, null, null, null,
            null, null,
            null, null, null, null, null, null, null, null, null, null, null, null, null, null,
            null, null,
            null, null, null, null, null, null, null, null, null, null, null, null, null, null,
            null, null,
            null, null, null, null, null, null, null, null, null, null, null, null, null, null,
            null, null,
            null, null, null, null, null, null, null, null, null, null, null, null, null, null,
            null, null,
            null, null, null, null, null, null, null, null, null, null, null, null, null, null,
            null, null,
            null, null, null, null, null, null, null, null, null, null, null, null, null, null,
            null, null,
            null, null, null, null, null, null, null, null, null, null, null, null, null, null,
            null, null,
            null, null, null, null, null, null, null, null, null, null, null, null, null, null,
            null, null,
            null, null, null, null, null, null, null, null, null, null, null, null, null, null,
            null, null,
            null, null, null, null, null, null, null, null, null, null, null, null, null, null,
            null, null,
            null, null, null, "IDENTIFIER", "ESCAPED_IDENTIFIER", "PARAMETER", "LITERAL", "ALL",
            "AND", "ANYELEMENT", "APPLY", "AS", "ASC", "BETWEEN", "BY", "CASE", "CAST", "COLLATE",
            "COLLECTION", "CROSS", "CREATEREF", "DEREF", "DESC", "DISTINCT", "ELEMENT", "ELSE",
            "END", "EXCEPT", "EXISTS", "ESCAPE", "FLATTEN", "FROM", "FULL", "FUNCTION", "GROUP",
            "GROUPPARTITION", "HAVING", "IN", "INNER", "INTERSECT", "IS", "JOIN", "KEY", "LEFT",
            "LIKE", "LIMIT", "MULTISET", "NAVIGATE", "NOT", "NULL", "OF", "OFTYPE", "ON", "OR",
            "ORDER", "OUTER", "OVERLAPS", "ONLY", "QMARK", "REF", "RELATIONSHIP", "RIGHT", "ROW",
            "SELECT", "SET", "SKIP", "THEN", "TOP", "TREAT", "UNION", "USING", "VALUE", "WHEN",
            "WHERE", "WITH", "COMMA", "COLON", "SCOLON", "DOT", "EQUAL", "L_PAREN", "R_PAREN",
            "L_BRACE", "R_BRACE", "L_CURLY", "R_CURLY", "PLUS", "MINUS", "STAR", "FSLASH",
            "PERCENT", "OP_EQ", "OP_NEQ", "OP_LT", "OP_LE", "OP_GT", "OP_GE", "UNARYPLUS",
            "UNARYMINUS",
        };

        private static String[] yyrule = {
            "$accept : commandStart",
            "commandStart :",
            "commandStart : command",
            "command : optNamespaceImportList queryStatement",
            "optNamespaceImportList :",
            "optNamespaceImportList : namespaceImportList",
            "namespaceImportList : namespaceImport",
            "namespaceImportList : namespaceImportList namespaceImport",
            "namespaceImport : USING identifier SCOLON",
            "namespaceImport : USING dotExpr SCOLON",
            "namespaceImport : USING assignExpr SCOLON",
            "queryStatement : optQueryDefList generalExpr optSemiColon",
            "optQueryDefList :",
            "optQueryDefList : functionDefList",
            "functionDefList : functionDef",
            "functionDefList : functionDefList functionDef",
            "functionDef : FUNCTION identifier functionParamsDef AS L_PAREN generalExpr R_PAREN",
            "functionParamsDef : L_PAREN R_PAREN",
            "functionParamsDef : L_PAREN functionParamDefList R_PAREN",
            "functionParamDefList : functionParamDef",
            "functionParamDefList : functionParamDefList COMMA functionParamDef",
            "functionParamDef : identifier typeDef",
            "generalExpr : queryExpr",
            "generalExpr : Expr",
            "optSemiColon :",
            "optSemiColon : SCOLON",
            "queryExpr : selectClause fromClause optWhereClause optGroupByClause optHavingClause optOrderByClause"
            ,
            "$$1 :",
            "selectClause : SELECT $$1 optAllOrDistinct optTopClause aliasExprList",
            "$$2 :",
            "selectClause : SELECT $$2 VALUE optAllOrDistinct optTopClause aliasExprList",
            "optAllOrDistinct :",
            "optAllOrDistinct : ALL",
            "optAllOrDistinct : DISTINCT",
            "optTopClause :",
            "optTopClause : TOP L_PAREN generalExpr R_PAREN",
            "fromClause : FROM fromClauseList",
            "fromClauseList : fromClauseItem",
            "fromClauseList : fromClauseList COMMA fromClauseItem",
            "fromClauseItem : aliasExpr",
            "fromClauseItem : L_PAREN joinClauseItem R_PAREN",
            "fromClauseItem : joinClauseItem",
            "fromClauseItem : L_PAREN applyClauseItem R_PAREN",
            "fromClauseItem : applyClauseItem",
            "joinClauseItem : fromClauseItem joinType fromClauseItem",
            "joinClauseItem : fromClauseItem joinType fromClauseItem ON Expr",
            "applyClauseItem : fromClauseItem applyType fromClauseItem",
            "joinType : CROSS JOIN",
            "joinType : LEFT OUTER JOIN",
            "joinType : LEFT JOIN",
            "joinType : RIGHT OUTER JOIN",
            "joinType : RIGHT JOIN",
            "joinType : JOIN",
            "joinType : INNER JOIN",
            "joinType : FULL JOIN",
            "joinType : FULL OUTER JOIN",
            "joinType : FULL OUTER",
            "applyType : CROSS APPLY",
            "applyType : OUTER APPLY",
            "optWhereClause :",
            "optWhereClause : whereClause",
            "whereClause : WHERE Expr",
            "optGroupByClause :",
            "optGroupByClause : groupByClause",
            "groupByClause : GROUP BY aliasExprList",
            "optHavingClause :",
            "optHavingClause : havingClause",
            "$$3 :",
            "havingClause : HAVING $$3 Expr",
            "optOrderByClause :",
            "optOrderByClause : orderByClause",
            "$$4 :",
            "orderByClause : ORDER BY $$4 orderByItemList optSkipSubClause optLimitSubClause",
            "optSkipSubClause :",
            "optSkipSubClause : SKIP Expr",
            "optLimitSubClause :",
            "optLimitSubClause : LIMIT Expr",
            "orderByItemList : orderByClauseItem",
            "orderByItemList : orderByItemList COMMA orderByClauseItem",
            "orderByClauseItem : Expr optAscDesc",
            "orderByClauseItem : Expr COLLATE simpleIdentifier optAscDesc",
            "optAscDesc :",
            "optAscDesc : ASC",
            "optAscDesc : DESC",
            "exprList : Expr",
            "exprList : exprList COMMA Expr",
            "Expr : parenExpr",
            "Expr : PARAMETER",
            "Expr : identifier",
            "Expr : builtInExpr",
            "Expr : dotExpr",
            "Expr : refExpr",
            "Expr : createRefExpr",
            "Expr : keyExpr",
            "Expr : groupPartitionExpr",
            "Expr : methodExpr",
            "Expr : ctorExpr",
            "Expr : derefExpr",
            "Expr : navigateExpr",
            "Expr : literalExpr",
            "parenExpr : L_PAREN generalExpr R_PAREN",
            "betweenPrefix : Expr BETWEEN Expr",
            "notBetweenPrefix : Expr NOT BETWEEN Expr",
            "builtInExpr : Expr PLUS Expr",
            "builtInExpr : Expr MINUS Expr",
            "builtInExpr : Expr STAR Expr",
            "builtInExpr : Expr FSLASH Expr",
            "builtInExpr : Expr PERCENT Expr",
            "builtInExpr : MINUS Expr",
            "builtInExpr : PLUS Expr",
            "builtInExpr : Expr OP_NEQ Expr",
            "builtInExpr : Expr OP_GT Expr",
            "builtInExpr : Expr OP_GE Expr",
            "builtInExpr : Expr OP_LT Expr",
            "builtInExpr : Expr OP_LE Expr",
            "builtInExpr : Expr INTERSECT Expr",
            "builtInExpr : Expr UNION Expr",
            "builtInExpr : Expr UNION ALL Expr",
            "builtInExpr : Expr EXCEPT Expr",
            "builtInExpr : Expr OVERLAPS Expr",
            "builtInExpr : Expr IN Expr",
            "builtInExpr : Expr NOT IN Expr",
            "builtInExpr : EXISTS L_PAREN generalExpr R_PAREN",
            "builtInExpr : ANYELEMENT L_PAREN generalExpr R_PAREN",
            "builtInExpr : ELEMENT L_PAREN generalExpr R_PAREN",
            "builtInExpr : FLATTEN L_PAREN generalExpr R_PAREN",
            "builtInExpr : SET L_PAREN generalExpr R_PAREN",
            "builtInExpr : Expr IS NULL",
            "builtInExpr : Expr IS NOT NULL",
            "builtInExpr : searchedCaseExpr",
            "builtInExpr : TREAT L_PAREN Expr AS typeName R_PAREN",
            "builtInExpr : CAST L_PAREN Expr AS typeName R_PAREN",
            "builtInExpr : OFTYPE L_PAREN Expr COMMA typeName R_PAREN",
            "builtInExpr : OFTYPE L_PAREN Expr COMMA ONLY typeName R_PAREN",
            "builtInExpr : Expr IS OF L_PAREN typeName R_PAREN",
            "builtInExpr : Expr IS NOT OF L_PAREN typeName R_PAREN",
            "builtInExpr : Expr IS OF L_PAREN ONLY typeName R_PAREN",
            "builtInExpr : Expr IS NOT OF L_PAREN ONLY typeName R_PAREN",
            "builtInExpr : Expr LIKE Expr",
            "builtInExpr : Expr NOT LIKE Expr",
            "builtInExpr : Expr LIKE Expr ESCAPE Expr",
            "builtInExpr : Expr NOT LIKE Expr ESCAPE Expr",
            "builtInExpr : betweenPrefix AND Expr",
            "builtInExpr : notBetweenPrefix AND Expr",
            "builtInExpr : Expr OR Expr",
            "builtInExpr : NOT Expr",
            "builtInExpr : Expr AND Expr",
            "builtInExpr : equalsOrAssignExpr",
            "equalsOrAssignExpr : assignExpr",
            "equalsOrAssignExpr : equalsExpr",
            "assignExpr : Expr EQUAL Expr",
            "equalsExpr : Expr OP_EQ Expr",
            "aliasExpr : Expr AS identifier",
            "aliasExpr : Expr",
            "aliasExprList : aliasExpr",
            "aliasExprList : aliasExprList COMMA aliasExpr",
            "searchedCaseExpr : CASE whenThenExprList END",
            "searchedCaseExpr : CASE whenThenExprList caseElseExpr END",
            "whenThenExprList : WHEN Expr THEN Expr",
            "whenThenExprList : whenThenExprList WHEN Expr THEN Expr",
            "caseElseExpr : ELSE Expr",
            "ctorExpr : ROW L_PAREN aliasExprList R_PAREN",
            "ctorExpr : MULTISET L_PAREN exprList R_PAREN",
            "ctorExpr : L_CURLY exprList R_CURLY",
            "dotExpr : Expr DOT identifier",
            "refExpr : REF L_PAREN generalExpr R_PAREN",
            "derefExpr : DEREF L_PAREN generalExpr R_PAREN",
            "createRefExpr : CREATEREF L_PAREN Expr COMMA Expr R_PAREN",
            "createRefExpr : CREATEREF L_PAREN Expr COMMA Expr COMMA typeName R_PAREN",
            "keyExpr : KEY L_PAREN generalExpr R_PAREN",
            "groupPartitionExpr : GROUPPARTITION L_PAREN optAllOrDistinct generalExpr R_PAREN",
            "methodExpr : dotExpr L_PAREN R_PAREN",
            "methodExpr : dotExpr L_PAREN optAllOrDistinct exprList R_PAREN optWithRelationship",
            "methodExpr : dotExpr L_PAREN optAllOrDistinct queryExpr R_PAREN optWithRelationship",
            "methodExpr : identifier L_PAREN R_PAREN",
            "methodExpr : identifier L_PAREN optAllOrDistinct exprList R_PAREN optWithRelationship",
            "methodExpr : identifier L_PAREN optAllOrDistinct queryExpr R_PAREN optWithRelationship",
            "navigateExpr : NAVIGATE L_PAREN Expr COMMA typeName R_PAREN",
            "navigateExpr : NAVIGATE L_PAREN Expr COMMA typeName COMMA identifier R_PAREN",
            "navigateExpr : NAVIGATE L_PAREN Expr COMMA typeName COMMA identifier COMMA identifier R_PAREN"
            ,
            "optWithRelationship :",
            "optWithRelationship : relationshipList",
            "relationshipList : WITH relationshipExpr",
            "relationshipList : relationshipList relationshipExpr",
            "relationshipExpr : RELATIONSHIP L_PAREN Expr COMMA typeName R_PAREN",
            "relationshipExpr : RELATIONSHIP L_PAREN Expr COMMA typeName COMMA identifier R_PAREN",
            "relationshipExpr : RELATIONSHIP L_PAREN Expr COMMA typeName COMMA identifier COMMA identifier R_PAREN"
            ,
            "typeName : identifier",
            "typeName : qualifiedTypeName",
            "typeName : identifier ESCAPED_IDENTIFIER",
            "typeName : qualifiedTypeName ESCAPED_IDENTIFIER",
            "typeName : typeNameWithTypeSpec",
            "qualifiedTypeName : typeName DOT identifier",
            "typeNameWithTypeSpec : qualifiedTypeName L_PAREN R_PAREN",
            "typeNameWithTypeSpec : qualifiedTypeName L_PAREN exprList R_PAREN",
            "typeNameWithTypeSpec : identifier L_PAREN R_PAREN",
            "typeNameWithTypeSpec : identifier L_PAREN exprList R_PAREN",
            "identifier : ESCAPED_IDENTIFIER",
            "identifier : simpleIdentifier",
            "simpleIdentifier : IDENTIFIER",
            "literalExpr : LITERAL",
            "literalExpr : NULL",
            "typeDef : typeName",
            "typeDef : collectionTypeDef",
            "typeDef : refTypeDef",
            "typeDef : rowTypeDef",
            "collectionTypeDef : COLLECTION L_PAREN typeDef R_PAREN",
            "refTypeDef : REF L_PAREN typeName R_PAREN",
            "rowTypeDef : ROW L_PAREN propertyDefList R_PAREN",
            "propertyDefList : propertyDef",
            "propertyDefList : propertyDefList COMMA propertyDef",
            "propertyDef : identifier typeDef",
        };

//#line 1415 "CqlGrammar.y"

#pragma warning restore 414
//###############################################################
// method: yylexdebug : check lexer state
//###############################################################
        private void yylexdebug(int state, int ch)
        {
            String s = null;
            if (ch < 0)
            {
                ch = 0;
            }
            if (ch <= YYMAXTOKEN) //check index bounds
            {
                s = yyname[ch]; //now get it
            }
            if (s == null)
            {
                s = "illegal-symbol";
            }
            debug("state " + state + ", reading " + ch + " (" + s + ")");
        }

//###############################################################
// method: yyparse : parse input and execute indicated items
//###############################################################
        private int yyparse()
        {
            int yyn; //next thing to do
            int yym; //
            int yystate; //current parsing state from state table
#if YYDEBUG
    String yys;    //current token string
#endif
            init_stacks();
            yynerrs = 0;
            yyerrflag = 0;
            yychar = (-1);

            yystate = 0;
            state_push(yystate);

            yyloop:
#if YYDEBUG
    debug("yyloop"); 
#endif
            yyn = yydefred[yystate];
            if (yyn != 0)
            {
                goto yyreduce;
            }
#if YYDEBUG
	debug("yyn:"+yyn+"  state:"+yystate+"  char:"+yychar);
#endif
            if (yychar < 0)
            {
                yychar = yylex();
                if (yychar < 0)
                {
                    yychar = 0;
                }
            }
            yyn = yysindex[yystate];
            if ((yyn != 0) && (yyn += yychar) >= 0 &&
                yyn <= YYTABLESIZE
                && yycheck[yyn] == yychar)
            {
#if YYDEBUG
        debug("state "+yystate+", shifting to state "+yytable[yyn]+"");
#endif
                yystate = yytable[yyn];
                state_push(yystate);
                val_push(yylval);
                yychar = (-1);
                if (yyerrflag > 0)
                {
                    --yyerrflag;
                }
                goto yyloop;
            }
            yyn = yyrindex[yystate];
            if ((yyn != 0) && (yyn += yychar) >= 0 &&
                yyn <= YYTABLESIZE
                && yycheck[yyn] == yychar)
            {
                yyn = yytable[yyn];
#if YYDEBUG
	     debug("reduce");
#endif
                goto yyreduce;
            }
            if (yyerrflag != 0)
            {
                goto yyinrecovery;
            }
            yyerror("syntax error");

            ++yynerrs;
            yyinrecovery:
            if (yyerrflag < 3)
            {
                yyerrflag = 3;
                for (;;)
                {
                    if (stateptr < 0)
                    {
                        goto yyoverflow;
                    }
                    yyn = yysindex[state_peek(0)];
                    if ((yyn != 0) && (yyn += YYERRCODE) >= 0 &&
                        yyn <= YYTABLESIZE
                        && yycheck[yyn] == YYERRCODE)
                    {
                        if (stateptr < 0)
                        {
                            goto yyoverflow;
                        }
#if YYDEBUG
                debug("state "+state_peek(0)+", error recovery shifting to state "+yytable[yyn]+" ");
#endif
                        yystate = yytable[yyn];
                        state_push(yystate);
                        val_push(yylval);
                        goto yyloop;
                    }
                    else
                    {
                        if (stateptr < 0)
                        {
                            goto yyoverflow;
                        }
#if YYDEBUG
                debug("error recovery discarding state "+state_peek(0)+" ");
#endif
                        state_pop();
                        val_pop();
                    }
                }
            }
            else
            {
                if (yychar == 0)
                {
                    goto yyabort;
                }
#if YYDEBUG
        if (yydebug)
        {
            yys = null;
            if (yychar <= YYMAXTOKEN) yys = yyname[yychar];
            if (yys == null) yys = "illegal-symbol";
            debug("state "+yystate+", error recovery discards token "+yychar+" ("+yys+")");
        }
#endif
                yychar = (-1);
                goto yyloop;
            }
            yyreduce:
            yym = yylen[yyn];
#if YYDEBUG
      debug("state "+yystate+", reducing "+yym+" by rule "+yyn+" ("+yyrule[yyn]+")");
#endif
            yyval = val_peek(yym - 1);
            switch (yyn)
            {
//########## USER-SUPPLIED ACTIONS ##########
                case 1:
//#line 108 "CqlGrammar.y"
                    {
                        yyval = _parsedTree = null;
                    }
                    break;
                case 2:
//#line 112 "CqlGrammar.y"
                    {
                        yyval = _parsedTree = (Node)val_peek(0);
                    }
                    break;
                case 3:
//#line 118 "CqlGrammar.y"
                    {
                        yyval = new Command(ToNodeList<NamespaceImport>(val_peek(1)), (Statement)val_peek(0));
                        SetErrCtx(
                            AstNode(yyval), (val_peek(1) != null) ? AstNodePos(val_peek(1)) : AstNodePos(val_peek(0)),
                            EntityRes.CtxCommandExpression);
                    }
                    break;
                case 4:
//#line 129 "CqlGrammar.y"
                    {
                        yyval = null;
                    }
                    break;
                case 5:
//#line 133 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 6:
//#line 139 "CqlGrammar.y"
                    {
                        yyval = new NodeList<NamespaceImport>((NamespaceImport)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(0)), EntityRes.CtxNamespaceImportList);
                    }
                    break;
                case 7:
//#line 144 "CqlGrammar.y"
                    {
                        yyval = ToNodeList<NamespaceImport>(val_peek(1)).Add((NamespaceImport)val_peek(0));
                    }
                    break;
                case 8:
//#line 150 "CqlGrammar.y"
                    {
                        yyval = new NamespaceImport((Identifier)val_peek(1));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(2)), EntityRes.CtxNamespaceImport);
                    }
                    break;
                case 9:
//#line 155 "CqlGrammar.y"
                    {
                        yyval = new NamespaceImport((DotExpr)val_peek(1));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(2)), EntityRes.CtxNamespaceImport);
                    }
                    break;
                case 10:
//#line 160 "CqlGrammar.y"
                    {
                        yyval = new NamespaceImport((BuiltInExpr)val_peek(1));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(2)), EntityRes.CtxAliasedNamespaceImport);
                    }
                    break;
                case 11:
//#line 171 "CqlGrammar.y"
                    {
                        yyval = new QueryStatement(ToNodeList<AST.FunctionDefinition>(val_peek(2)), (Node)val_peek(1));
                        SetErrCtx(
                            AstNode(yyval), (val_peek(2) != null) ? AstNodePos(val_peek(2)) : AstNodePos(val_peek(1)),
                            EntityRes.CtxQueryStatement);
                    }
                    break;
                case 12:
//#line 182 "CqlGrammar.y"
                    {
                        yyval = null;
                    }
                    break;
                case 13:
//#line 186 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 14:
//#line 192 "CqlGrammar.y"
                    {
                        yyval = new NodeList<AST.FunctionDefinition>((AST.FunctionDefinition)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(0)), AstNode(val_peek(0)).ErrCtx.ErrorContextInfo);
                    }
                    break;
                case 15:
//#line 197 "CqlGrammar.y"
                    {
                        yyval = ToNodeList<AST.FunctionDefinition>(val_peek(1)).Add((AST.FunctionDefinition)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(yyval), AstNode(val_peek(0)).ErrCtx.ErrorContextInfo);
                    }
                    break;
                case 16:
//#line 204 "CqlGrammar.y"
                    {
                        yyval = new AST.FunctionDefinition(
                            (Identifier)val_peek(5), ToNodeList<PropDefinition>(val_peek(4)), (Node)val_peek(1), Terminal(val_peek(6)).IPos,
                            Terminal(val_peek(0)).IPos);
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(6)), EntityRes.CtxFunctionDefinition);
                    }
                    break;
                case 17:
//#line 211 "CqlGrammar.y"
                    {
                        yyval = null;
                    }
                    break;
                case 18:
//#line 215 "CqlGrammar.y"
                    {
                        yyval = val_peek(1);
                    }
                    break;
                case 19:
//#line 221 "CqlGrammar.y"
                    {
                        yyval = new NodeList<PropDefinition>((PropDefinition)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(0)), AstNode(val_peek(0)).ErrCtx.ErrorContextInfo);
                    }
                    break;
                case 20:
//#line 226 "CqlGrammar.y"
                    {
                        yyval = ToNodeList<PropDefinition>(val_peek(2)).Add((PropDefinition)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(yyval), AstNode(val_peek(0)).ErrCtx.ErrorContextInfo);
                    }
                    break;
                case 21:
//#line 233 "CqlGrammar.y"
                    {
                        yyval = new PropDefinition((Identifier)val_peek(1), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(1)), EntityRes.CtxFunctionDefinition);
                    }
                    break;
                case 22:
//#line 244 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 23:
//#line 248 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 24:
//#line 254 "CqlGrammar.y"
                    {
                        yyval = null;
                    }
                    break;
                case 25:
//#line 258 "CqlGrammar.y"
                    {
                        yyval = null;
                    }
                    break;
                case 26:
//#line 268 "CqlGrammar.y"
                    {
                        yyval = new QueryExpr(
                            (SelectClause)val_peek(5),
                            (FromClause)val_peek(4),
                            (Node)val_peek(3),
                            (GroupByClause)val_peek(2),
                            (HavingClause)val_peek(1),
                            (OrderByClause)val_peek(0));

                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(5)), EntityRes.CtxQueryExpression);
                    }
                    break;
                case 27:
//#line 281 "CqlGrammar.y"
                    {
                        StartMethodExprCounting();
                    }
                    break;
                case 28:
//#line 287 "CqlGrammar.y"
                    {
                        yyval = new SelectClause(
                            ToNodeList<AliasedExpr>(val_peek(0)), SelectKind.Row, (DistinctKind)val_peek(2), (Node)val_peek(1),
                            EndMethodExprCounting());
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(4)), EntityRes.CtxSelectRowClause);
                    }
                    break;
                case 29:
//#line 292 "CqlGrammar.y"
                    {
                        StartMethodExprCounting();
                    }
                    break;
                case 30:
//#line 299 "CqlGrammar.y"
                    {
                        yyval = new SelectClause(
                            ToNodeList<AliasedExpr>(val_peek(0)), SelectKind.Value, (DistinctKind)val_peek(2), (Node)val_peek(1),
                            EndMethodExprCounting());
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(5)), EntityRes.CtxSelectValueClause);
                    }
                    break;
                case 31:
//#line 306 "CqlGrammar.y"
                    {
                        yyval = DistinctKind.None;
                    }
                    break;
                case 32:
//#line 310 "CqlGrammar.y"
                    {
                        yyval = DistinctKind.All;
                    }
                    break;
                case 33:
//#line 314 "CqlGrammar.y"
                    {
                        yyval = DistinctKind.Distinct;
                    }
                    break;
                case 34:
//#line 320 "CqlGrammar.y"
                    {
                        yyval = null;
                    }
                    break;
                case 35:
//#line 324 "CqlGrammar.y"
                    {
                        yyval = val_peek(1);
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(3)), EntityRes.CtxTopSubClause);
                    }
                    break;
                case 36:
//#line 331 "CqlGrammar.y"
                    {
                        yyval = new FromClause(ToNodeList<FromClauseItem>(val_peek(0)));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxFromClause);
                    }
                    break;
                case 37:
//#line 338 "CqlGrammar.y"
                    {
                        yyval = new NodeList<FromClauseItem>((FromClauseItem)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(0)), AstNode(val_peek(0)).ErrCtx.ErrorContextInfo);
                    }
                    break;
                case 38:
//#line 343 "CqlGrammar.y"
                    {
                        yyval = ToNodeList<FromClauseItem>(val_peek(2)).Add((FromClauseItem)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(2)), EntityRes.CtxFromClauseList);
                    }
                    break;
                case 39:
//#line 350 "CqlGrammar.y"
                    {
                        yyval = new FromClauseItem((AliasedExpr)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(0)), EntityRes.CtxFromClauseItem);
                    }
                    break;
                case 40:
//#line 355 "CqlGrammar.y"
                    {
                        yyval = new FromClauseItem((JoinClauseItem)val_peek(1));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(1)), EntityRes.CtxFromJoinClause);
                    }
                    break;
                case 41:
//#line 360 "CqlGrammar.y"
                    {
                        yyval = new FromClauseItem((JoinClauseItem)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(0)), EntityRes.CtxFromJoinClause);
                    }
                    break;
                case 42:
//#line 365 "CqlGrammar.y"
                    {
                        yyval = new FromClauseItem((ApplyClauseItem)val_peek(1));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(1)), EntityRes.CtxFromApplyClause);
                    }
                    break;
                case 43:
//#line 370 "CqlGrammar.y"
                    {
                        yyval = new FromClauseItem((ApplyClauseItem)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(0)), EntityRes.CtxFromApplyClause);
                    }
                    break;
                case 44:
//#line 377 "CqlGrammar.y"
                    {
                        yyval = new JoinClauseItem((FromClauseItem)val_peek(2), (FromClauseItem)val_peek(0), (JoinKind)val_peek(1));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(2)), EntityRes.CtxJoinClause);
                    }
                    break;
                case 45:
//#line 382 "CqlGrammar.y"
                    {
                        yyval = new JoinClauseItem(
                            (FromClauseItem)val_peek(4), (FromClauseItem)val_peek(2), (JoinKind)val_peek(3), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(4)), EntityRes.CtxJoinOnClause);
                    }
                    break;
                case 46:
//#line 389 "CqlGrammar.y"
                    {
                        yyval = new ApplyClauseItem((FromClauseItem)val_peek(2), (FromClauseItem)val_peek(0), (ApplyKind)val_peek(1));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(2)), EntityRes.CtxApplyClause);
                    }
                    break;
                case 47:
//#line 396 "CqlGrammar.y"
                    {
                        yyval = JoinKind.Cross;
                    }
                    break;
                case 48:
//#line 400 "CqlGrammar.y"
                    {
                        yyval = JoinKind.LeftOuter;
                    }
                    break;
                case 49:
//#line 404 "CqlGrammar.y"
                    {
                        yyval = JoinKind.LeftOuter;
                    }
                    break;
                case 50:
//#line 408 "CqlGrammar.y"
                    {
                        yyval = JoinKind.RightOuter;
                    }
                    break;
                case 51:
//#line 412 "CqlGrammar.y"
                    {
                        yyval = JoinKind.RightOuter;
                    }
                    break;
                case 52:
//#line 416 "CqlGrammar.y"
                    {
                        yyval = JoinKind.Inner;
                    }
                    break;
                case 53:
//#line 420 "CqlGrammar.y"
                    {
                        yyval = JoinKind.Inner;
                    }
                    break;
                case 54:
//#line 424 "CqlGrammar.y"
                    {
                        yyval = JoinKind.FullOuter;
                    }
                    break;
                case 55:
//#line 428 "CqlGrammar.y"
                    {
                        yyval = JoinKind.FullOuter;
                    }
                    break;
                case 56:
//#line 432 "CqlGrammar.y"
                    {
                        yyval = JoinKind.FullOuter;
                    }
                    break;
                case 57:
//#line 438 "CqlGrammar.y"
                    {
                        yyval = ApplyKind.Cross;
                    }
                    break;
                case 58:
//#line 442 "CqlGrammar.y"
                    {
                        yyval = ApplyKind.Outer;
                    }
                    break;
                case 59:
//#line 447 "CqlGrammar.y"
                    {
                        yyval = null;
                    }
                    break;
                case 60:
//#line 451 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 61:
//#line 457 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(0)), EntityRes.CtxWhereClause);
                    }
                    break;
                case 62:
//#line 464 "CqlGrammar.y"
                    {
                        yyval = null;
                    }
                    break;
                case 63:
//#line 468 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 64:
//#line 474 "CqlGrammar.y"
                    {
                        yyval = new GroupByClause(ToNodeList<AliasedExpr>(val_peek(0)));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(2)), EntityRes.CtxGroupByClause);
                    }
                    break;
                case 65:
//#line 481 "CqlGrammar.y"
                    {
                        yyval = null;
                    }
                    break;
                case 66:
//#line 485 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 67:
//#line 491 "CqlGrammar.y"
                    {
                        StartMethodExprCounting();
                    }
                    break;
                case 68:
//#line 495 "CqlGrammar.y"
                    {
                        yyval = new HavingClause((Node)val_peek(0), EndMethodExprCounting());
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(0)), EntityRes.CtxHavingClause);
                    }
                    break;
                case 69:
//#line 502 "CqlGrammar.y"
                    {
                        yyval = null;
                    }
                    break;
                case 70:
//#line 506 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 71:
//#line 512 "CqlGrammar.y"
                    {
                        StartMethodExprCounting();
                    }
                    break;
                case 72:
//#line 518 "CqlGrammar.y"
                    {
                        yyval = new OrderByClause(
                            ToNodeList<OrderByClauseItem>(val_peek(2)), (Node)val_peek(1), (Node)val_peek(0), EndMethodExprCounting());
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(5)), EntityRes.CtxOrderByClauseItem);
                    }
                    break;
                case 73:
//#line 525 "CqlGrammar.y"
                    {
                        yyval = null;
                    }
                    break;
                case 74:
//#line 529 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxSkipSubClause);
                    }
                    break;
                case 75:
//#line 536 "CqlGrammar.y"
                    {
                        yyval = null;
                    }
                    break;
                case 76:
//#line 540 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxLimitSubClause);
                    }
                    break;
                case 77:
//#line 547 "CqlGrammar.y"
                    {
                        yyval = new NodeList<OrderByClauseItem>((OrderByClauseItem)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(0)), AstNode(val_peek(0)).ErrCtx.ErrorContextInfo);
                    }
                    break;
                case 78:
//#line 552 "CqlGrammar.y"
                    {
                        yyval = ToNodeList<OrderByClauseItem>(val_peek(2)).Add((OrderByClauseItem)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(2)), EntityRes.CtxOrderByClause);
                    }
                    break;
                case 79:
//#line 559 "CqlGrammar.y"
                    {
                        yyval = new OrderByClauseItem((Node)val_peek(1), (OrderKind)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(1)), EntityRes.CtxOrderByClauseItem);
                    }
                    break;
                case 80:
//#line 564 "CqlGrammar.y"
                    {
                        yyval = new OrderByClauseItem((Node)val_peek(3), (OrderKind)val_peek(0), (Identifier)val_peek(1));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(3)), EntityRes.CtxCollatedOrderByClauseItem);
                    }
                    break;
                case 81:
//#line 571 "CqlGrammar.y"
                    {
                        yyval = OrderKind.None;
                    }
                    break;
                case 82:
//#line 575 "CqlGrammar.y"
                    {
                        yyval = OrderKind.Asc;
                    }
                    break;
                case 83:
//#line 579 "CqlGrammar.y"
                    {
                        yyval = OrderKind.Desc;
                    }
                    break;
                case 84:
//#line 588 "CqlGrammar.y"
                    {
                        yyval = new NodeList<Node>((Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(0)), AstNode(val_peek(0)).ErrCtx.ErrorContextInfo);
                    }
                    break;
                case 85:
//#line 593 "CqlGrammar.y"
                    {
                        yyval = ToNodeList<Node>(val_peek(2)).Add((Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(2)), EntityRes.CtxExpressionList);
                    }
                    break;
                case 86:
//#line 600 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 87:
//#line 604 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 88:
//#line 608 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 89:
//#line 612 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 90:
//#line 616 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 91:
//#line 620 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 92:
//#line 624 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 93:
//#line 628 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 94:
//#line 632 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                        IncrementMethodExprCount();
                    }
                    break;
                case 95:
//#line 637 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                        IncrementMethodExprCount();
                    }
                    break;
                case 96:
//#line 642 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 97:
//#line 646 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 98:
//#line 650 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 99:
//#line 654 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 100:
//#line 660 "CqlGrammar.y"
                    {
                        yyval = new ParenExpr((Node)val_peek(1));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(2)), EntityRes.CtxParen);
                    }
                    break;
                case 101:
//#line 667 "CqlGrammar.y"
                    {
                        yyval = new NodeList<Node>((Node)val_peek(2)).Add((Node)val_peek(0));
                    }
                    break;
                case 102:
//#line 673 "CqlGrammar.y"
                    {
                        yyval = new NodeList<Node>((Node)val_peek(3)).Add((Node)val_peek(0));
                    }
                    break;
                case 103:
//#line 682 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.Plus, Terminal(val_peek(1)).Token, (Node)val_peek(2), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxPlus);
                    }
                    break;
                case 104:
//#line 687 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.Minus, Terminal(val_peek(1)).Token, (Node)val_peek(2), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxMinus);
                    }
                    break;
                case 105:
//#line 692 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.Multiply, Terminal(val_peek(1)).Token, (Node)val_peek(2), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxMultiply);
                    }
                    break;
                case 106:
//#line 697 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.Divide, Terminal(val_peek(1)).Token, (Node)val_peek(2), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxDivide);
                    }
                    break;
                case 107:
//#line 702 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.Modulus, Terminal(val_peek(1)).Token, (Node)val_peek(2), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxModulus);
                    }
                    break;
                case 108:
//#line 707 "CqlGrammar.y"
                    {
                        var literal = val_peek(0) as Literal;
                        if (literal != null && literal.IsNumber
                            && !literal.IsSignedNumber)
                        {
                            literal.PrefixSign(Terminal(val_peek(1)).Token);
                            yyval = val_peek(0);
                        }
                        else
                        {
                            yyval = new BuiltInExpr(BuiltInKind.UnaryMinus, Terminal(val_peek(1)).Token, (Node)val_peek(0));
                            SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxUnaryMinus);
                        }
                    }
                    break;
                case 109:
//#line 722 "CqlGrammar.y"
                    {
                        var literal = val_peek(0) as Literal;
                        if (null != literal && literal.IsNumber
                            && !literal.IsSignedNumber)
                        {
                            literal.PrefixSign(Terminal(val_peek(1)).Token);
                            yyval = val_peek(0);
                        }
                        else
                        {
                            yyval = new BuiltInExpr(BuiltInKind.UnaryPlus, Terminal(val_peek(1)).Token, (Node)val_peek(0));
                            SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxUnaryPlus);
                        }
                    }
                    break;
                case 110:
//#line 739 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.NotEqual, Terminal(val_peek(1)).Token, (Node)val_peek(2), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxNotEqual);
                    }
                    break;
                case 111:
//#line 744 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.GreaterThan, Terminal(val_peek(1)).Token, (Node)val_peek(2), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxGreaterThan);
                    }
                    break;
                case 112:
//#line 749 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.GreaterEqual, Terminal(val_peek(1)).Token, (Node)val_peek(2), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxGreaterThanEqual);
                    }
                    break;
                case 113:
//#line 754 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.LessThan, Terminal(val_peek(1)).Token, (Node)val_peek(2), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxLessThan);
                    }
                    break;
                case 114:
//#line 759 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.LessEqual, Terminal(val_peek(1)).Token, (Node)val_peek(2), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxLessThanEqual);
                    }
                    break;
                case 115:
//#line 767 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.Intersect, Terminal(val_peek(1)).Token, (Node)val_peek(2), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxIntersect);
                    }
                    break;
                case 116:
//#line 772 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.Union, Terminal(val_peek(1)).Token, (Node)val_peek(2), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxUnion);
                    }
                    break;
                case 117:
//#line 777 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.UnionAll, Terminal(val_peek(2)).Token, (Node)val_peek(3), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(2)), EntityRes.CtxUnionAll);
                    }
                    break;
                case 118:
//#line 782 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.Except, Terminal(val_peek(1)).Token, (Node)val_peek(2), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxExcept);
                    }
                    break;
                case 119:
//#line 787 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.Overlaps, Terminal(val_peek(1)).Token, (Node)val_peek(2), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxOverlaps);
                    }
                    break;
                case 120:
//#line 792 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.In, Terminal(val_peek(1)).Token, (Node)val_peek(2), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxIn);
                    }
                    break;
                case 121:
//#line 797 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.NotIn, Terminal(val_peek(2)).Token, (Node)val_peek(3), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(2)), EntityRes.CtxNotIn);
                    }
                    break;
                case 122:
//#line 802 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.Exists, Terminal(val_peek(3)).Token, (Node)val_peek(1));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(3)), EntityRes.CtxExists);
                    }
                    break;
                case 123:
//#line 807 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.AnyElement, Terminal(val_peek(3)).Token, (Node)val_peek(1));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(3)), EntityRes.CtxAnyElement);
                    }
                    break;
                case 124:
//#line 812 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.Element, Terminal(val_peek(3)).Token, (Node)val_peek(1));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(3)), EntityRes.CtxElement);
                    }
                    break;
                case 125:
//#line 817 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.Flatten, Terminal(val_peek(3)).Token, (Node)val_peek(1));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(3)), EntityRes.CtxFlatten);
                    }
                    break;
                case 126:
//#line 822 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.Distinct, Terminal(val_peek(3)).Token, (Node)val_peek(1));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(3)), EntityRes.CtxSet);
                    }
                    break;
                case 127:
//#line 830 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.IsNull, "IsNull", (Node)val_peek(2));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxIsNull);
                    }
                    break;
                case 128:
//#line 835 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.IsNotNull, "IsNotNull", (Node)val_peek(3));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(2)), EntityRes.CtxIsNotNull);
                    }
                    break;
                case 129:
//#line 843 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 130:
//#line 850 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.Treat, Terminal(val_peek(5)).Token, (Node)val_peek(3), (Node)val_peek(1));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(5)), EntityRes.CtxTreat);
                    }
                    break;
                case 131:
//#line 855 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.Cast, Terminal(val_peek(5)).Token, (Node)val_peek(3), (Node)val_peek(1));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(5)), EntityRes.CtxCast);
                    }
                    break;
                case 132:
//#line 864 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(
                            BuiltInKind.OfType,
                            Terminal(val_peek(5)).Token,
                            (Node)val_peek(3),
                            (Node)val_peek(1),
                            Literal.NewBooleanLiteral(false) /* only */);

                        SetErrCtx(AstNode(yyval), Terminal(val_peek(5)), EntityRes.CtxOfType);
                    }
                    break;
                case 133:
//#line 874 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(
                            BuiltInKind.OfType,
                            "OFTYPE ONLY",
                            (Node)val_peek(4),
                            (Node)val_peek(1),
                            Literal.NewBooleanLiteral(true) /* only */);

                        SetErrCtx(AstNode(yyval), Terminal(val_peek(6)), EntityRes.CtxOfTypeOnly);
                    }
                    break;
                case 134:
//#line 887 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(
                            BuiltInKind.IsOf,
                            "IS OF",
                            (Node)val_peek(5),
                            (Node)val_peek(1),
                            Literal.NewBooleanLiteral(false), /* only */
                            Literal.NewBooleanLiteral(false) /* not  */
                            );

                        SetErrCtx(AstNode(yyval), Terminal(val_peek(4)), EntityRes.CtxIsOf);
                    }
                    break;
                case 135:
//#line 899 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(
                            BuiltInKind.IsOf,
                            "IS NOT OF",
                            (Node)val_peek(6), /* instance */
                            (Node)val_peek(1), /* type */
                            Literal.NewBooleanLiteral(false), /* only */
                            Literal.NewBooleanLiteral(true) /* not  */
                            );

                        SetErrCtx(AstNode(yyval), Terminal(val_peek(5)), EntityRes.CtxIsNotOf);
                    }
                    break;
                case 136:
//#line 911 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(
                            BuiltInKind.IsOf,
                            "IS OF ONLY",
                            (Node)val_peek(6), /* instance */
                            (Node)val_peek(1), /* type */
                            Literal.NewBooleanLiteral(true), /* only */
                            Literal.NewBooleanLiteral(false) /* not  */
                            );

                        SetErrCtx(AstNode(yyval), Terminal(val_peek(5)), EntityRes.CtxIsOf);
                    }
                    break;
                case 137:
//#line 923 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(
                            BuiltInKind.IsOf,
                            "IS NOT OF ONLY",
                            (Node)val_peek(7), /* instance */
                            (Node)val_peek(1), /* type */
                            Literal.NewBooleanLiteral(true), /* only */
                            Literal.NewBooleanLiteral(true) /* not  */
                            );

                        SetErrCtx(AstNode(yyval), Terminal(val_peek(6)), EntityRes.CtxIsNotOf);
                    }
                    break;
                case 138:
//#line 938 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.Like, Terminal(val_peek(1)).Token, (Node)val_peek(2), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxLike);
                    }
                    break;
                case 139:
//#line 943 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(
                            BuiltInKind.Not,
                            Terminal(val_peek(2)).Token,
                            new BuiltInExpr(BuiltInKind.Like, Terminal(val_peek(1)).Token, (Node)val_peek(3), (Node)val_peek(0)));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(2)), EntityRes.CtxNotLike);
                    }
                    break;
                case 140:
//#line 950 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(
                            BuiltInKind.Like, Terminal(val_peek(3)).Token, (Node)val_peek(4), (Node)val_peek(2), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(3)), EntityRes.CtxLike);
                    }
                    break;
                case 141:
//#line 955 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(
                            BuiltInKind.Not,
                            Terminal(val_peek(4)).Token,
                            new BuiltInExpr(
                                BuiltInKind.Like, Terminal(val_peek(3)).Token, (Node)val_peek(5), (Node)val_peek(2), (Node)val_peek(0)));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(4)), EntityRes.CtxNotLike);
                    }
                    break;
                case 142:
//#line 965 "CqlGrammar.y"
                    {
                        var elist = (NodeList<Node>)val_peek(2);
                        Debug.Assert(elist.Count == 2);
                        yyval = new BuiltInExpr(BuiltInKind.Between, "between", elist[0], elist[1], (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxBetween);
                    }
                    break;
                case 143:
//#line 975 "CqlGrammar.y"
                    {
                        var elist = (NodeList<Node>)val_peek(2);
                        Debug.Assert(elist.Count == 2);
                        yyval = new BuiltInExpr(BuiltInKind.NotBetween, "notbetween", elist[0], elist[1], (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxNotBetween);
                    }
                    break;
                case 144:
//#line 985 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.Or, "or", (Node)val_peek(2), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxOr);
                    }
                    break;
                case 145:
//#line 990 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.Not, "not", (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxNot);
                    }
                    break;
                case 146:
//#line 995 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.And, "and", (Node)val_peek(2), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxAnd);
                    }
                    break;
                case 147:
//#line 1000 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 148:
//#line 1006 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 149:
//#line 1010 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 150:
//#line 1016 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.Equal, Terminal(val_peek(1)).Token, (Node)val_peek(2), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxEquals);
                    }
                    break;
                case 151:
//#line 1023 "CqlGrammar.y"
                    {
                        yyval = new BuiltInExpr(BuiltInKind.Equal, Terminal(val_peek(1)).Token, (Node)val_peek(2), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxEquals);
                    }
                    break;
                case 152:
//#line 1030 "CqlGrammar.y"
                    {
                        yyval = new AliasedExpr((Node)val_peek(2), (Identifier)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxAlias);
                    }
                    break;
                case 153:
//#line 1035 "CqlGrammar.y"
                    {
                        yyval = new AliasedExpr((Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(0)), AstNode(val_peek(0)).ErrCtx.ErrorContextInfo);
                    }
                    break;
                case 154:
//#line 1042 "CqlGrammar.y"
                    {
                        yyval = new NodeList<AliasedExpr>((AliasedExpr)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(0)), AstNode(val_peek(0)).ErrCtx.ErrorContextInfo);
                    }
                    break;
                case 155:
//#line 1047 "CqlGrammar.y"
                    {
                        yyval = ToNodeList<AliasedExpr>(val_peek(2)).Add((AliasedExpr)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(yyval), EntityRes.CtxExpressionList);
                    }
                    break;
                case 156:
//#line 1054 "CqlGrammar.y"
                    {
                        yyval = new CaseExpr(ToNodeList<WhenThenExpr>(val_peek(1)));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(2)), EntityRes.CtxCase);
                    }
                    break;
                case 157:
//#line 1059 "CqlGrammar.y"
                    {
                        yyval = new CaseExpr(ToNodeList<WhenThenExpr>(val_peek(2)), (Node)val_peek(1));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(3)), EntityRes.CtxCase);
                    }
                    break;
                case 158:
//#line 1066 "CqlGrammar.y"
                    {
                        yyval = new NodeList<WhenThenExpr>(new WhenThenExpr((Node)val_peek(2), (Node)val_peek(0)));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(3)), EntityRes.CtxCaseWhenThen);
                    }
                    break;
                case 159:
//#line 1071 "CqlGrammar.y"
                    {
                        yyval = ToNodeList<WhenThenExpr>(val_peek(4)).Add(new WhenThenExpr((Node)val_peek(2), (Node)val_peek(0)));
                    }
                    break;
                case 160:
//#line 1077 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxCaseElse);
                    }
                    break;
                case 161:
//#line 1084 "CqlGrammar.y"
                    {
                        yyval = new RowConstructorExpr(ToNodeList<AliasedExpr>(val_peek(1)));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(3)), EntityRes.CtxRowCtor);
                    }
                    break;
                case 162:
//#line 1089 "CqlGrammar.y"
                    {
                        yyval = new MultisetConstructorExpr(ToNodeList<Node>(val_peek(1)));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(3)), EntityRes.CtxMultisetCtor);
                    }
                    break;
                case 163:
//#line 1094 "CqlGrammar.y"
                    {
                        yyval = new MultisetConstructorExpr(ToNodeList<Node>(val_peek(1)));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(2)), EntityRes.CtxMultisetCtor);
                    }
                    break;
                case 164:
//#line 1101 "CqlGrammar.y"
                    {
                        yyval = new DotExpr((Node)val_peek(2), (Identifier)val_peek(0));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(1)), EntityRes.CtxMemberAccess);
                    }
                    break;
                case 165:
//#line 1108 "CqlGrammar.y"
                    {
                        yyval = new RefExpr((Node)val_peek(1));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(3)), EntityRes.CtxRef);
                    }
                    break;
                case 166:
//#line 1115 "CqlGrammar.y"
                    {
                        yyval = new DerefExpr((Node)val_peek(1));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(3)), EntityRes.CtxDeref);
                    }
                    break;
                case 167:
//#line 1122 "CqlGrammar.y"
                    {
                        yyval = new CreateRefExpr((Node)val_peek(3), (Node)val_peek(1));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(5)), EntityRes.CtxCreateRef);
                    }
                    break;
                case 168:
//#line 1127 "CqlGrammar.y"
                    {
                        yyval = new CreateRefExpr((Node)val_peek(5), (Node)val_peek(3), (Node)val_peek(1));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(7)), EntityRes.CtxCreateRef);
                    }
                    break;
                case 169:
//#line 1134 "CqlGrammar.y"
                    {
                        yyval = new KeyExpr((Node)val_peek(1));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(3)), EntityRes.CtxKey);
                    }
                    break;
                case 170:
//#line 1141 "CqlGrammar.y"
                    {
                        yyval = new GroupPartitionExpr((DistinctKind)val_peek(2), (Node)val_peek(1));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(4)), EntityRes.CtxGroupPartition);
                    }
                    break;
                case 171:
//#line 1148 "CqlGrammar.y"
                    {
                        yyval = new MethodExpr((Node)val_peek(2), DistinctKind.None, null);
                        SetErrCtx(AstNode(yyval), AstNodePos(((DotExpr)val_peek(2)).Identifier), EntityRes.CtxMethod);
                    }
                    break;
                case 172:
//#line 1153 "CqlGrammar.y"
                    {
                        yyval = new MethodExpr(
                            (Node)val_peek(5), (DistinctKind)val_peek(3), ToNodeList<Node>(val_peek(2)),
                            ToNodeList<RelshipNavigationExpr>(val_peek(0)));
                        SetErrCtx(AstNode(yyval), AstNodePos(((DotExpr)val_peek(5)).Identifier), EntityRes.CtxMethod);
                    }
                    break;
                case 173:
//#line 1158 "CqlGrammar.y"
                    {
                        yyval = new MethodExpr(
                            (Node)val_peek(5), (DistinctKind)val_peek(3), new NodeList<Node>((Node)val_peek(2)),
                            ToNodeList<RelshipNavigationExpr>(val_peek(0)));
                        SetErrCtx(AstNode(yyval), AstNodePos(((DotExpr)val_peek(5)).Identifier), EntityRes.CtxMethod);
                    }
                    break;
                case 174:
//#line 1163 "CqlGrammar.y"
                    {
                        yyval = new MethodExpr((Identifier)val_peek(2), DistinctKind.None, null);
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(2)), EntityRes.CtxMethod);
                    }
                    break;
                case 175:
//#line 1168 "CqlGrammar.y"
                    {
                        yyval = new MethodExpr(
                            (Identifier)val_peek(5), (DistinctKind)val_peek(3), ToNodeList<Node>(val_peek(2)),
                            ToNodeList<RelshipNavigationExpr>(val_peek(0)));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(5)), EntityRes.CtxMethod);
                    }
                    break;
                case 176:
//#line 1173 "CqlGrammar.y"
                    {
                        yyval = new MethodExpr(
                            (Identifier)val_peek(5), (DistinctKind)val_peek(3), new NodeList<Node>((Node)val_peek(2)),
                            ToNodeList<RelshipNavigationExpr>(val_peek(0)));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(5)), EntityRes.CtxMethod);
                    }
                    break;
                case 177:
//#line 1183 "CqlGrammar.y"
                    {
                        yyval = new RelshipNavigationExpr((Node)val_peek(3), (Node)val_peek(1), null, null);
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(5)), EntityRes.CtxNavigate);
                    }
                    break;
                case 178:
//#line 1191 "CqlGrammar.y"
                    {
                        yyval = new RelshipNavigationExpr((Node)val_peek(5), (Node)val_peek(3), (Identifier)val_peek(1), null);
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(7)), EntityRes.CtxNavigate);
                    }
                    break;
                case 179:
//#line 1199 "CqlGrammar.y"
                    {
                        yyval = new RelshipNavigationExpr(
                            (Node)val_peek(7), (Node)val_peek(5), (Identifier)val_peek(3), (Identifier)val_peek(1));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(9)), EntityRes.CtxNavigate);
                    }
                    break;
                case 180:
//#line 1206 "CqlGrammar.y"
                    {
                        yyval = null;
                    }
                    break;
                case 181:
//#line 1210 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 182:
//#line 1216 "CqlGrammar.y"
                    {
                        yyval = new NodeList<RelshipNavigationExpr>((RelshipNavigationExpr)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(0)), EntityRes.CtxRelationshipList);
                    }
                    break;
                case 183:
//#line 1221 "CqlGrammar.y"
                    {
                        yyval = ToNodeList<RelshipNavigationExpr>(val_peek(1)).Add((RelshipNavigationExpr)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(1)), EntityRes.CtxRelationshipList);
                    }
                    break;
                case 184:
//#line 1231 "CqlGrammar.y"
                    {
                        yyval = new RelshipNavigationExpr((Node)val_peek(3), (Node)val_peek(1), null, null);
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(5)), EntityRes.CtxRelationship);
                    }
                    break;
                case 185:
//#line 1239 "CqlGrammar.y"
                    {
                        yyval = new RelshipNavigationExpr((Node)val_peek(5), (Node)val_peek(3), null, (Identifier)val_peek(1));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(7)), EntityRes.CtxRelationship);
                    }
                    break;
                case 186:
//#line 1247 "CqlGrammar.y"
                    {
                        yyval = new RelshipNavigationExpr(
                            (Node)val_peek(7), (Node)val_peek(5), (Identifier)val_peek(1), (Identifier)val_peek(3));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(9)), EntityRes.CtxRelationship);
                    }
                    break;
                case 187:
//#line 1254 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(0)), EntityRes.CtxTypeName);
                    }
                    break;
                case 188:
//#line 1259 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(0)), EntityRes.CtxTypeName);
                    }
                    break;
                case 189:
//#line 1264 "CqlGrammar.y"
                    {
                        var identifier = (Identifier)val_peek(1);
                        var escapedIdentifier = (Identifier)val_peek(0);
                        if (identifier.IsEscaped
                            || escapedIdentifier.Name.Length > 0)
                        {
                            var errCtx = identifier.ErrCtx;
                            var message = Strings.InvalidMetadataMemberName;
                            throw EntitySqlException.Create(errCtx, message, null);
                        }
                        yyval = new Identifier(identifier.Name + "[]", /*isEscaped*/false, _query, AstNodePos(val_peek(1)));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(1)), EntityRes.CtxTypeName);
                    }
                    break;
                case 190:
//#line 1275 "CqlGrammar.y"
                    {
                        var dotExpr = (DotExpr)val_peek(1);
                        var identifier = dotExpr.Identifier;
                        var escapedIdentifier = (Identifier)val_peek(0);
                        if (identifier.IsEscaped
                            || escapedIdentifier.Name.Length > 0)
                        {
                            var errCtx = identifier.ErrCtx;
                            var message = Strings.InvalidMetadataMemberName;
                            throw EntitySqlException.Create(errCtx, message, null);
                        }
                        yyval = new DotExpr(
                            dotExpr.Left, new Identifier(identifier.Name + "[]", /*isEscaped*/false, _query, AstNodePos(val_peek(1))));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(1)), EntityRes.CtxTypeName);
                    }
                    break;
                case 191:
//#line 1287 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(0)), EntityRes.CtxTypeName);
                    }
                    break;
                case 192:
//#line 1294 "CqlGrammar.y"
                    {
                        yyval = new DotExpr((Node)val_peek(2), (Identifier)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(2)), EntityRes.CtxTypeName);
                    }
                    break;
                case 193:
//#line 1301 "CqlGrammar.y"
                    {
                        yyval = new MethodExpr((Node)val_peek(2), DistinctKind.None, null);
                        SetErrCtx(AstNode(yyval), AstNodePos(((DotExpr)val_peek(2)).Identifier), EntityRes.CtxTypeNameWithTypeSpec);
                    }
                    break;
                case 194:
//#line 1306 "CqlGrammar.y"
                    {
                        yyval = new MethodExpr((Node)val_peek(3), DistinctKind.None, ToNodeList<Node>(val_peek(1)));
                        SetErrCtx(AstNode(yyval), AstNodePos(((DotExpr)val_peek(3)).Identifier), EntityRes.CtxTypeNameWithTypeSpec);
                    }
                    break;
                case 195:
//#line 1311 "CqlGrammar.y"
                    {
                        yyval = new MethodExpr((Identifier)val_peek(2), DistinctKind.None, null);
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(2)), EntityRes.CtxTypeNameWithTypeSpec);
                    }
                    break;
                case 196:
//#line 1316 "CqlGrammar.y"
                    {
                        yyval = new MethodExpr((Identifier)val_peek(3), DistinctKind.None, ToNodeList<Node>(val_peek(1)));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(3)), EntityRes.CtxTypeNameWithTypeSpec);
                    }
                    break;
                case 197:
//#line 1323 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(0)), EntityRes.CtxEscapedIdentifier);
                    }
                    break;
                case 198:
//#line 1328 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(0)), EntityRes.CtxSimpleIdentifier);
                    }
                    break;
                case 199:
//#line 1335 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 200:
//#line 1341 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(0)), EntityRes.CtxLiteral);
                    }
                    break;
                case 201:
//#line 1346 "CqlGrammar.y"
                    {
                        yyval = new Literal(null, LiteralKind.Null, _query, TerminalPos(val_peek(0)));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(0)), EntityRes.CtxNullLiteral);
                    }
                    break;
                case 202:
//#line 1357 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 203:
//#line 1361 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 204:
//#line 1365 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 205:
//#line 1369 "CqlGrammar.y"
                    {
                        yyval = val_peek(0);
                    }
                    break;
                case 206:
//#line 1375 "CqlGrammar.y"
                    {
                        yyval = new CollectionTypeDefinition((Node)val_peek(1));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(3)), EntityRes.CtxCollectionTypeDefinition);
                    }
                    break;
                case 207:
//#line 1382 "CqlGrammar.y"
                    {
                        yyval = new RefTypeDefinition((Node)val_peek(1));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(3)), EntityRes.CtxRefTypeDefinition);
                    }
                    break;
                case 208:
//#line 1389 "CqlGrammar.y"
                    {
                        yyval = new RowTypeDefinition(ToNodeList<PropDefinition>(val_peek(1)));
                        SetErrCtx(AstNode(yyval), Terminal(val_peek(3)), EntityRes.CtxRowTypeDefinition);
                    }
                    break;
                case 209:
//#line 1396 "CqlGrammar.y"
                    {
                        yyval = new NodeList<PropDefinition>((PropDefinition)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(0)), AstNode(val_peek(0)).ErrCtx.ErrorContextInfo);
                    }
                    break;
                case 210:
//#line 1401 "CqlGrammar.y"
                    {
                        yyval = ToNodeList<PropDefinition>(val_peek(2)).Add((PropDefinition)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(yyval), AstNode(val_peek(0)).ErrCtx.ErrorContextInfo);
                    }
                    break;
                case 211:
//#line 1408 "CqlGrammar.y"
                    {
                        yyval = new PropDefinition((Identifier)val_peek(1), (Node)val_peek(0));
                        SetErrCtx(AstNode(yyval), AstNodePos(val_peek(1)), EntityRes.CtxRowTypeDefinition);
                    }
                    break;

//########## END OF USER-SUPPLIED ACTIONS ##########
            }
#if YYDEBUG
    debug("reduce");
#endif
            state_drop(yym);
            yystate = state_peek(0);
            val_drop(yym);
            yym = yylhs[yyn];
            if (yystate == 0
                && yym == 0)
            {
#if YYDEBUG
        debug("After reduction, shifting from state 0 to state "+YYFINAL+"");
#endif
                yystate = YYFINAL;
                state_push(YYFINAL);
                val_push(yyval);
                if (yychar < 0)
                {
                    yychar = yylex();
                    if (yychar < 0)
                    {
                        yychar = 0;
                    }
#if YYDEBUG
            if (yydebug)
            yylexdebug(yystate,yychar);
#endif
                }
                if (yychar == 0)
                {
                    goto yyaccept;
                }
                goto yyloop;
            }
            yyn = yygindex[yym];
            if ((yyn != 0) && (yyn += yystate) >= 0 &&
                yyn <= YYTABLESIZE
                && yycheck[yyn] == yystate)
            {
                yystate = yytable[yyn];
            }
            else
            {
                yystate = yydgoto[yym];
            }
            if (stateptr < 0)
            {
                goto yyoverflow;
            }
#if YYDEBUG
    debug("after reduction, shifting from state "+state_peek(0)+" to state "+yystate+"");
#endif
            state_push(yystate);
            val_push(yyval);
            goto yyloop;
            yyoverflow:
            yyerror("yacc stack overflow");
            yyabort:
            return (1);
            yyaccept:
            return (0);
        }

//## end of method parse() ######################################
    }

//################### END OF CLASS ######################
}

//################### END OF NAMESPACE ######################
