/* Copyright (C) 2016 Canonical Ltd.
 * Author: Luke Yelavich <luke.yelavich@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

private void
print_profile_list(bool detailed)
{
	SList<A11yProfileManager.Profile> profile_list = A11yProfileManager.list_profiles();

	foreach (var profile in profile_list)
	{
		if (detailed)
		{
			stdout.printf(_("Profile directory name: %s\n"), profile.dir_name);
			stdout.printf(_("Name: %s\n"), profile.name);
			stdout.printf(_("Description: %s\n"), profile.description);
			stdout.printf("\n");
		}
		else
		{
			stdout.printf("%s\n", profile.dir_name);
		}
	}
}


public static int
main(string[] args)
{
	/* Initialize i18n */
	Intl.setlocale (LocaleCategory.ALL, "");
	Intl.bindtextdomain (Config.GETTEXT_PACKAGE, Config.LOCALEDIR);
	Intl.bind_textdomain_codeset (Config.GETTEXT_PACKAGE, "UTF-8");
	Intl.textdomain (Config.GETTEXT_PACKAGE);

	/* Command-line arguments */
	bool show_version = false;
	bool list_profiles = false;
	bool list_profiles_detailed = false;
	bool get_active_profile = false;
	bool deactivate_profile = false;
	bool toggle_indicator = false;
	string active_profile = null;
	string profile_name = null;
	string dconf_file = null;

	OptionEntry version_option = { "version", 'v', 0, OptionArg.NONE, ref show_version, N_("Show release version"), null };
	OptionEntry list_option = { "list", 'l', 0, OptionArg.NONE, ref list_profiles, N_("List available accessibility profiles"), null };
	OptionEntry detail_option = { "list-detailed", 'L', 0, OptionArg.NONE, ref list_profiles_detailed, N_("List available accessibility profiles with more detail"), null };
	OptionEntry get_option = { "get", 'g', 0, OptionArg.NONE, ref get_active_profile, N_("Get the currently active profile"), null };
	OptionEntry set_option = { "set", 's', 0, OptionArg.STRING, ref profile_name, N_("Set the active profile, using its directory name, as shown with the -l or -L option output"), "DIR" };
	OptionEntry dconf_option = { "dconf", 'D', 0, OptionArg.STRING, ref dconf_file, N_("Write profile settings to a file compatible with the dconf utility"), "FILE" };
	OptionEntry deact_option = { "deactivate", 'd', 0, OptionArg.NONE, ref deactivate_profile, N_("Deactivate the currently active profile"), null };
	OptionEntry tindc_option = { "indicator-toggle", 'i', 0, OptionArg.NONE, ref toggle_indicator, N_("Toggle the accessibility profile indicator, will be enabled/disabled depending on whether a profile is being activated or deactivated"), null };
	OptionEntry nulloption = { null };
	OptionEntry[] options = { version_option, list_option, detail_option, get_option, set_option, dconf_option, deact_option, tindc_option, nulloption  };

	var context = new OptionContext(_("- Accessibility Profile Manager"));
	context.add_main_entries(options, Config.GETTEXT_PACKAGE);

	try
	{
		context.parse(ref args);
	}
	catch (Error e)
	{
		stderr.printf("%s\n", e.message);
		stderr.printf(_("Run '%s --help' to see a full list of available command line options."), args[0]);
		stderr.printf ("\n");
		return Posix.EXIT_FAILURE;
	}

	if (show_version)
	{
		stderr.printf ("accessibility-profile-manager %s\n", Config.VERSION);
	}

	if (list_profiles)
	{
		print_profile_list(false);
	}

	if (list_profiles_detailed)
	{
		print_profile_list(true);
	}

	if (get_active_profile)
	{
		var profile_settings = new Settings("com.canonical.a11y-profile-manager");

		active_profile = profile_settings.get_string("active-profile");

		if (active_profile != null && active_profile != "")
		{
			stdout.printf("%s\n", active_profile);
		}
		else
		{
			stdout.printf(_("There is currently no active profile set.\n"));
		}
	}

	if (deactivate_profile)
	{
		var profile_settings = new Settings("com.canonical.a11y-profile-manager");
		active_profile = profile_settings.get_string("active-profile");
		A11yProfileManager.Profile profile = null;

		if (active_profile != null && active_profile != "")
		{
			profile = A11yProfileManager.get_profile_by_dir_name(active_profile);
			if (profile != null && profile.valid)
			{
				try
				{
					profile.indicator_toggle = toggle_indicator;
					profile.reset_settings();
				}
				catch (Error e)
				{
					stdout.printf(_("Unable to reset the settings in the profile %s: %s"), profile.dir_name, e.message);
					return Posix.EXIT_FAILURE;
				}
			}

			profile_settings.reset("active-profile");
			Settings.sync();
		}
	}

	if (profile_name != null)
	{
		bool profile_set = false;
		A11yProfileManager.Profile profile = A11yProfileManager.get_profile_by_dir_name(profile_name);

		if (profile != null && profile.valid && profile.dir_name == profile_name)
		{
			try
			{
				profile.indicator_toggle = toggle_indicator;

				if (dconf_file != null && dconf_file != "")
				{
					profile.write_dconf_file(dconf_file);
				}
				else
				{
					A11yProfileManager.set_active_profile(profile);
				}
			}
			catch (Error e)
			{
				stderr.printf(_("The profile %s cannot be set as the active profile: %s"), profile.dir_name, e.message);
				return Posix.EXIT_FAILURE;
			}
			profile_set = true;
		}

		if (!profile_set)
		{
			stderr.printf(_("The profile %s cannot be found in the list of available accessibility profiles.\n"), active_profile);
			return Posix.EXIT_FAILURE;
		}
	}

	return Posix.EXIT_SUCCESS;
}
