// ---------------------------------------------------------------------------
// - Solid.cpp                                                               -
// - afnix:geo service - solid geoemtry class implementation                 -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2019 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Solid.hpp"
#include "Solid.hxx"
#include "Vector.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - private section                                                       -
  // -------------------------------------------------------------------------

  // the default axis rotation angle
  static const t_real GEO_RX_DEF   = 0.0;
  static const t_real GEO_RY_DEF   = 0.0;
  static const t_real GEO_RZ_DEF   = 0.0;

  // the solid xml name
  static const String XML_TAG_NAME = GEO_SLD_NAME;
  static const String XML_TAG_INFO = GEO_SLD_INFO;
  // the axis rotation angles
  static const String XML_RX_ATTR  = "rx";
  static const String XML_RY_ATTR  = "ry";
  static const String XML_RZ_ATTR  = "rz";

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a default solid

  Solid::Solid (void) {
  }

  // copy construct this solid

  Solid::Solid (const Solid& that) {
    that.rdlock ();
    try {
      // assign base object
      Geometry::operator = (that);
      // copy locally
      d_matl = that.d_matl;
      d_ortn = that.d_ortn;
      that.unlock ();
    } catch (...) {
      that.unlock ();
      throw;
    }
  }
  
  // copy move this solid

  Solid::Solid (Solid&& that) noexcept :
    // copy move base object
    Geometry (static_cast<Geometry&&>(that)) {
    that.wrlock ();
    try {
      // copy move locally
      d_matl = that.d_matl; that.d_matl.reset ();
      d_ortn = that.d_ortn; that.d_ortn.reset ();
      that.unlock ();
    } catch (...) {
      d_matl.reset ();
      d_ortn.reset ();
      that.unlock ();
    } 
  }

  // assign a solid into this one

  Solid& Solid::operator = (const Solid& that) {
    // check for self-assignation
    if (this == &that) return *this;
    // lock and assign
    wrlock ();
    that.rdlock ();
    try {
      // assign base object
      Geometry::operator = (that);
      // assign locally
      d_matl = that.d_matl;
      d_ortn = that.d_ortn;
      unlock ();
      that.unlock ();
      return *this;
    } catch (...) {
      unlock ();
      that.unlock ();
      throw;
    }
  }
  
  // move this solid into this one

  Solid& Solid::operator = (Solid&& that) noexcept {
    // check for self-move
    if (this == &that) return *this;
    // lock and assign
    wrlock ();
    that.wrlock ();
    try {
      // move base object
      Geometry::operator = (static_cast<Geometry&&>(that));
      // move locally
      d_matl = that.d_matl; that.d_matl.reset ();
      d_ortn = that.d_ortn; that.d_ortn.reset ();
      unlock ();
      that.unlock ();
    } catch (...) {
      d_matl.reset ();
      d_ortn.reset ();
      unlock ();
      that.unlock ();
    }
    return *this;
  }

  // set the solid by plist

  void Solid::setplst (const Plist& plst) {
    wrlock ();
    try {
      // set the material
      d_matl.setplst (plst);
      // extract quaternion info
      t_real rx = plst.exists (XML_RX_ATTR) ?
	plst.toreal (XML_RX_ATTR) : GEO_RX_DEF;
      t_real ry = plst.exists (XML_RY_ATTR) ?
	plst.toreal (XML_RY_ATTR) : GEO_RY_DEF;
      t_real rz = plst.exists (XML_RZ_ATTR) ?
	plst.toreal (XML_RZ_ATTR) : GEO_RZ_DEF;
      d_ortn.setra (rx, ry, rz);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the solid property list

  Plist Solid::getplst (void) const {
    rdlock ();
    try {
      // the material plist
      Plist result = d_matl.getplst ();
      result.setname (XML_TAG_NAME); result.setinfo (XML_TAG_INFO);
      // set the quaternion rotation if any
      if (d_ortn.isrx () == true) result.add (XML_RX_ATTR, d_ortn.getxa ());
      if (d_ortn.isry () == true) result.add (XML_RY_ATTR, d_ortn.getxa ());
      if (d_ortn.isrz () == true) result.add (XML_RZ_ATTR, d_ortn.getxa ());
      // here it is
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the solid material

  Material Solid::getmatl (void) const {
    rdlock ();
    try {
      Material result = d_matl;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set the quaternion object

  void Solid::setqo (const Quaternion& q) {
    wrlock ();
    try {
      d_ortn = q;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the solid quaternion

  Quaternion Solid::getqo (void) const {
    rdlock ();
    try {
      Quaternion result = d_ortn;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set the x-axis angle

  void Solid::setrx (const t_real a) {
    wrlock ();
    try {
      d_ortn.setrx (a);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set the y-axis angle

  void Solid::setry (const t_real a) {
    wrlock ();
    try {
      d_ortn.setry (a);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set the z-axis angle

  void Solid::setrz (const t_real a) {
    wrlock ();
    try {
      d_ortn.setrz (a);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 6;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_SETRX     = zone.intern ("set-rotation-x");
  static const long QUARK_SETRY     = zone.intern ("set-rotation-y");
  static const long QUARK_SETRZ     = zone.intern ("set-rotation-z");
  static const long QUARK_SETQOBJ   = zone.intern ("set-orientation");
  static const long QUARK_GETQOBJ   = zone.intern ("get-orientation");
  static const long QUARK_GETORIGIN = zone.intern ("get-origin");

  // return true if the given quark is defined
  
  bool Solid::isquark (const long quark, const bool hflg) const {
    rdlock ();
    try {
      if (zone.exists (quark) == true) {
	unlock ();
	return true;
      }
      bool result = hflg ? Geometry::isquark (quark, hflg) : false;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // apply this object with a set of arguments and a quark

  Object* Solid::apply (Runnable* robj, Nameset* nset, const long quark,
			Vector* argv) {
    // get the number of arguments
    long argc = (argv == nullptr) ? 0 : argv->length ();
    
    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETQOBJ)   return new Quaternion (getqo ());
      if (quark == QUARK_GETORIGIN) return new Point3 (getorigin ());
    }
    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_SETQOBJ) {
	Object* obj = argv->get (1);
	Quaternion* qobj = dynamic_cast <Quaternion*> (obj);
	if (qobj == nullptr) {
	  throw Exception ("type-error", "invalid object with set-orientation",
			   Object::repr (qobj));
	}
	setqo (*qobj);
	return nullptr;
      }
      if (quark == QUARK_SETRX) {
	t_real a = argv->getreal (0);
	setrx (a);
	return nullptr;
      }
      if (quark == QUARK_SETRY) {
	t_real a = argv->getreal (0);
	setry (a);
	return nullptr;
      }
      if (quark == QUARK_SETRZ) {
	t_real a = argv->getreal (0);
	setrz (a);
	return nullptr;
      }
    }
    // call the geometry method
    return Geometry::apply (robj, nset, quark, argv);
  }
}
