#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "canlxx.h"

namespace AuthN {

const int IO::CommunicationTimeout(1);
const int IO::NotImplemented(2);

IO::IO(const AuthN::Context& ctx):
      valid_(false),
      timeout_(1000),
      validator_(new Validator(ctx)),
      node_(new Credentials(ctx)),
      peer_(new Credentials(Context())),
      context_(&ctx.Copy()) {
}

IO::~IO(void) {
  if(peer_) delete peer_;
  if(node_) delete node_;
  if(validator_) delete validator_;
  if(context_) delete context_;
}

IO::operator bool(void) const {
  return valid_;
}

bool IO::operator!(void) const {
  return !valid_;
}

void IO::SetOwnCredentials(const AuthN::Credentials& cred) {
  if(node_) delete node_;
  node_ = &cred.Copy();
}

const AuthN::Credentials& IO::GetOwnCredentials(void) const {
  return *node_;
}

const AuthN::Credentials& IO::GetPeerCredentials(void) const {
  return *peer_;
}

void IO::SetValidator(const AuthN::Validator& validator) {
  if(validator_) delete validator_;
  validator_ = &validator.Copy();
}

const AuthN::Validator& IO::GetValidator(void) const {
  return *validator_;
}

AuthN::Status IO::Read(void *buffer, size_t& size) {
  return last_error_ = WireRead(buffer, size);
}

AuthN::Status IO::Read(std::string& buffer, size_t size) {
  if(size > 0) {
    buffer.resize(size);
    last_error_ = WireRead((void*)buffer.c_str(),size);
    if(last_error_) {
      buffer.resize(0);
    } else {
      buffer.resize(size);
    };
  } else {
    buffer.clear();
    std::string::size_type p = 0;
    int timeout = timeout_;
    timeout_ = 0; // to read as much as available right now
    size_t default_size = 4096;
    for(;;) {
      size_t l = default_size;
      buffer.resize(p + l);
      last_error_ = WireRead((void*)(buffer.c_str()+p),l);
      if(!last_error_) break;
      p += l;
      if(l < default_size) break;
    };
    timeout_ = timeout;
    // Read with timeout if nothing could be read immediately
    if((p == 0) && (timeout_ > 0)) {
      size_t l = 4096;
      buffer.resize(p + l);
      last_error_ = WireRead((void*)(buffer.c_str()+p),l);
      if(last_error_) p += l;
    };
    buffer.resize(p);
    if(p > 0) last_error_ = Status();
  };
  return last_error_;
}

AuthN::Status IO::Write(const void *buffer, size_t size) {
  return last_error_ = WireWrite(buffer, size);
}

AuthN::Status IO::Write(const std::string& buffer) {
  return last_error_ = WireWrite(buffer.c_str(), buffer.length());
}

AuthN::Status IO::Close(void) {
  return last_error_ = WireClose();
}

void IO::SetTimeout(int ms) {
  timeout_ = ms;
}

int IO::GetTimeout(void) {
  return timeout_;
}

AuthN::Status IO::GetStatus(void) const {
  return last_error_;
}

AuthN::Status IO::WireWrite(const void *buffer, size_t size) {
  return Status(NotImplemented,"Not implemented");
}

AuthN::Status IO::WireRead(void *buffer, size_t& size) {
  return Status(NotImplemented,"Not implemented");
}

AuthN::Status IO::WireClose(void) {
  return Status(NotImplemented,"Not implemented");
}

} // namespace AuthN

