#ifndef __AUTHNLIB_OPENSSLEXTUTI_H_
#define __AUTHNLIB_OPENSSLEXTUTI_H_

#include <openssl/asn1.h>
#include <openssl/pem.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>

#include "canlxx.h"

namespace AuthN {
namespace OpenSSL {

  enum KeyUsageType {
    // KeyUsage
    DigitalSignature,    ///Certificate can be used to create digital signatures
    NonRepudiation,      ///Certificate can be used for non-repudiation
    KeyEncipherment,     ///Certificate can be used for encrypting / decrypting keys
    DataEncipherment,    ///Certificate can be used for encrypting / decrypting data
    KeyAgreement,        ///Certificate can be used for key agreement
    KeyCertificateSign,  ///Certificate can be used for key certificate signing
    CRLSign,             ///Certificate can be used to sign Certificate Revocation Lists
    EncipherOnly,        ///Certificate can only be used for encryption
    DecipherOnly,        ///Certificate can only be used for decryption

    // ExtKeyUsage
    ServerAuth,       ///Certificate can be used for server authentication (e.g. web server), id = "1.3.6.1.5.5.7.3.1".
    ClientAuth,       ///Certificate can be used for client authentication (e.g. web browser), id = "1.3.6.1.5.5.7.3.2".
    CodeSigning,      ///Certificate can be used to sign code, id = "1.3.6.1.5.5.7.3.3".
    EmailProtection,  ///Certificate can be used to sign / encrypt email, id = "1.3.6.1.5.5.7.3.4".
    IPSecEndSystem,   ///Certificate can be used to authenticate a endpoint in IPSEC, id = "1.3.6.1.5.5.7.3.5".
    IPSecTunnel,      ///Certificate can be used to authenticate a tunnel in IPSEC, id = "1.3.6.1.5.5.7.3.6".
    IPSecUser,        ///Certificate can be used to authenticate a user in IPSEC, id = "1.3.6.1.5.5.7.3.7".
    TimeStamping,     ///Certificate can be used to create a "time stamp" signature, id = "1.3.6.1.5.5.7.3.8".
    OCSPSigning       ///Certificate can be used to sign an Online %Certificate Status Protocol (OCSP) assertion, id = "1.3.6.1.5.5.7.3.9".
  };


  class KeyUsage {
    public:
      KeyUsage() : type_(-1), extended_(false){ };

      KeyUsage(KeyUsageType type);

      /*@param id the type as an identifier string (OID or internal)
        @param section the section this type belongs in
      */
      KeyUsage(const std::string &id, bool extended);

      KeyUsage(const KeyUsage &from) : id_(from.id_), type_(from.type_), extended_(from.extended_) { };

      ~KeyUsage(){ };

      KeyUsage & operator=(const KeyUsage &from) {
        id_ = from.id_;
        type_ = from.type_;
        extended_ = from.extended_;
        return *this;
      };

      bool extended() const { return extended_; };

      KeyUsageType type() const { return (KeyUsageType)type_; };

      std::string id() const { return id_; };

      bool operator<(const KeyUsage &other) const;

      bool operator==(const KeyUsage &other) const;

      inline bool operator!=(const KeyUsage &other) const {
        return !(*this == other);
      }

    private:
        std::string id_;
        int type_;
        bool extended_;
  };

  class KeyUsages {
    private:
      std::list<KeyUsage> keyusages;
      KeyUsage dummy;
    public:
      KeyUsages& operator+= ( const KeyUsages & other ); //the overlap KeyUsage will only be count once
      KeyUsages& operator+= ( const KeyUsage& value );
      KeyUsages& operator= ( const KeyUsages& other ) {
        keyusages.clear();
        for(std::list<KeyUsage>::const_iterator i=other.keyusages.begin(); i!=other.keyusages.end(); i++) keyusages.push_back(*i); return *this;
      }
      const KeyUsage& operator[] ( int num ) const {
        int j = 0;
        for(std::list<KeyUsage>::const_iterator i=keyusages.begin(); i != keyusages.end(); i++, j++) {
          if(j==num) return *i;
        }
        return dummy;
      }
      KeyUsages& Intersection( const KeyUsages & other );
      void Clear() { keyusages.clear(); }
      int Size() { return keyusages.size(); }
      const int Size() const { return keyusages.size(); }
      operator std::string(void) const;
  };

  typedef KeyUsages Constraints;

  struct IssuerId{
    std::string keyid;
    std::string issuer;
    int serial;
  };

  enum ConstraintBit {
    Bit_DigitalSignature   = 0,
    Bit_NonRepudiation     = 1,
    Bit_KeyEncipherment    = 2,
    Bit_DataEncipherment   = 3,
    Bit_KeyAgreement       = 4,
    Bit_KeyCertificateSign = 5,
    Bit_CRLSign            = 6,
    Bit_EncipherOnly       = 7,
    Bit_DecipherOnly       = 8
  };


//----------
// X509ExtUtil --- an ulility class for manuplicate the extension of X509 object,
//                 including the inserting and parsing some common used extensions
//                 the X509 object will not be deleted inside this class
//----------
  class X509ExtUtil {
    enum ConstraintBit {
      Bit_DigitalSignature   = 0,
      Bit_NonRepudiation     = 1,
      Bit_KeyEncipherment    = 2,
      Bit_DataEncipherment   = 3,
      Bit_KeyAgreement       = 4,
      Bit_KeyCertificateSign = 5,
      Bit_CRLSign            = 6,
      Bit_EncipherOnly       = 7,
      Bit_DecipherOnly       = 8
    };

    private:
      X509* cert;
      X509_REQ* req;
      Context* logctx;
    public:
      X509ExtUtil() : cert(NULL), req(NULL), logctx(NULL) { };
      X509ExtUtil(X509* x509, Context* context) : cert(x509), req(NULL),
        logctx(context) {
      };
      X509ExtUtil(X509_REQ* x509req, Context* context) : cert(NULL),
          req(x509req), logctx(context) {
      };
      ~X509ExtUtil() { };

      void add_ext2req(X509_EXTENSION* ext);

      // parameter is the configuration for setting
      // subject key ID, e.g. "hash", which is the same
      // as the value definition in openssl.cnf
      bool set_subject_key_id(const std::string& sub_kid_conf);

      std::string get_subject_key_id();

      // parameter is the configuration for setting
      // authority key ID, e.g. "keyid:always,issuer:always",
      // which is the same as the value definition in openssl.cnf
      bool set_authority_key_id(const std::string& auth_kid_conf);

      IssuerId get_authority_key_id();

      bool set_subject_alt_name(const std::string& alt_name);

      void get_subject_alt_name(std::string& alt_name);

      bool set_basic_constraints(bool ca, int pathlen);

      void get_basic_constraints(bool* is_ca, int* pathlen);

      bool set_key_usage(KeyUsages& keyusages, bool issuer_is_ca = true);

      KeyUsages get_key_usage();

      bool set_ext_key_usage(KeyUsages& keyusages);

      KeyUsages get_ext_key_usage();

      bool set_cert_policies(const std::vector<std::string>& policies);

      void get_cert_policies(std::vector<std::string>& policies);

#ifdef HAVE_OPENSSL_PROXY
      Status set_proxy_policy(PROXY_CERT_INFO_EXTENSION* proxy_info);
#endif

      Status set_proxy_policy(const std::string& pci_policy = "",
                const std::string& pci_language = "", const std::string& pci_pathlen = "");

      Status get_proxy_policy(Credentials::Extension& policy) const;

      Status get_extension(int pos, Credentials::Extension& ext) const;

      Status get_extension(const std::string& name, Credentials::Extension& ext) const;

      Status set_extension(Credentials::Extension& ext);

      Status set_attributes(const std::string& name, const std::string& attrs_value);

      Status set_attributes(const std::string& name, std::list<std::string>& attrs);

      Status get_attributes(const std::string& name, std::list<std::string>& attrs) const;

  };

  bool retrieve_extension(Context& ctx, X509_EXTENSION* x509_ext, AuthN::Credentials::Extension& ext);

  bool parse_extension(X509_EXTENSION* x509_ext, bool& crit, std::string& oid, std::string& ext_val);

  STACK_OF(X509_EXTENSION)* x509_req_get_extensions(X509_REQ *req);

}
}


#endif /* __AUTHNLIB_OPENSSLEXTUTI_H_ */
