;; hyperclim.el: Looks up symbols in the CLIM spec from inside xemacs.
;; Written and tested using XEmacs.
;; This code is in the public domain.

;; Orginally written by Andy Hefner (andy.hefner@verizon.net)

(require 'cl)
(require 'browse-url)
(require 'thingatpt)


(defvar clim-sybmol-table nil
  "The symbol table for looking up CLIM symbols")

;;; Override base URL of CLIM specification, set this if you want
;;; to use one of the above from your local disk.
(setq clim-url-base-override nil)

;;; Selection of CLIM spec according to preference -- uncomment one of the lines below

;; MikeMac conversion of the CLIM specification
;(setq clim-dataset 'clim-mikemac-spec)

;; Gilbert Baumann conversion of the CLIM specification
(setq clim-dataset 'clim-gilberth-spec)

;; Alternate URL for Gilbert's spec..
; (setq clim-url-base-override "http://bauhh.dyndns.org:8000/clim-spec/"


;;; Internal bits

(defvar clim-history nil)

(defun clim-get-dataset ()
  (symbol-value clim-dataset))

(defun clim-get-symbol-table ()
  (rest (clim-get-dataset)))

(defun clim-get-url-base ()
  (first (clim-get-dataset)))

(defun clim-find-symbol (string)
  (assoc (downcase symbol-name)
         (clim-get-symbol-table)))

(defun clim-find-symbol-url (string)
  (let ((syminfo (clim-find-symbol string)))
    (when syminfo
      (concat (or clim-url-base-override 
                  (clim-get-url-base))
              (cdr syminfo)))))

;;; CLIM-LOOKUP command
;;    Look up a symbol in MikeMac's CLIM documentation.
;;    By default it looks up the symbol under the point, but if it isn't over
;;    something resembling a symbol, it will prompt you. 
;;    Also, you can use a prefix arg to force prompting.
(defun clim-lookup (p)
  (interactive "p")  
  (let ((symbol-name (thing-at-point 'symbol)))
    (unless (and (= 1 p) (stringp symbol-name))
      (setq symbol-name (read-from-minibuffer "Symbol name: " "" nil nil 'clim-history)))
    (let ((url (clim-find-symbol-url (downcase symbol-name))))
      (if url 
          (browse-url url)
          (message "Symbol %s not found." symbol-name)))))


;;; CLIM spec datasets

(setq clim-gilberth-spec 
 '("http://www.stud.uni-karlsruhe.de/~unk6/clim-spec/"
   ("+yellow+" . "13-3.html#_700") ("+white+" . "13-3.html#_702")
   ("+transparent-ink+" . "13-4.html#_710")
   ("+textual-view+" . "23-6.html#_1218")
   ("+textual-menu-view+" . "23-6.html#_1219")
   ("+textual-dialog-view+" . "23-6.html#_1220")
   ("+super-key+" . "8-2.html#_381") ("+shift-key+" . "8-2.html#_378")
   ("+red+" . "13-3.html#_695") ("+pointer-right-button+" . "8-2.html#_377")
   ("+pointer-middle-button+" . "8-2.html#_376")
   ("+pointer-left-button+" . "8-2.html#_375")
   ("+pointer-documentation-view+" . "23-6.html#_1224")
   ("+nowhere+" . "3-1.html#_49") ("+meta-key+" . "8-2.html#_380")
   ("+magenta+" . "13-3.html#_699")
   ("+identity-transformation+" . "5-1.html#_174")
   ("+hyper-key+" . "8-2.html#_382") ("+green+" . "13-3.html#_696")
   ("+gadget-view+" . "23-6.html#_1221")
   ("+gadget-menu-view+" . "23-6.html#_1222")
   ("+gadget-dialog-view+" . "23-6.html#_1223")
   ("+foreground-ink+" . "13-6.html#_715") ("+flipping-ink+" . "13-7.html#_719")
   ("+fill+" . "29-3.html#_1611") ("+everywhere+" . "3-1.html#_48")
   ("+cyan+" . "13-3.html#_698") ("+control-key+" . "8-2.html#_379")
   ("+blue+" . "13-3.html#_697") ("+black+" . "13-3.html#_701")
   ("+background-ink+" . "13-6.html#_716")
   ("*unsupplied-argument-marker*" . "27-6.html#_1448")
   ("*undefined-text-style*" . "11-1.html#_571")
   ("*standard-activation-gestures*" . "24-2.html#_1325")
   ("*possibilities-gestures*" . "24-5.html#_1345")
   ("*pointer-documentation-output*" . "28-3.html#_1500")
   ("*pointer-button-press-handler*" . "22-2.html#_1080")
   ("*partial-command-parser*" . "27-6.html#_1447")
   ("*original-stream*" . "C-1.html#_1916")
   ("*numeric-argument-marker*" . "27-6.html#_1449")
   ("*null-presentation*" . "23-8.html#_1260")
   ("*multiprocessing-p*" . "B-2.html#_1879")
   ("*input-wait-test*" . "22-2.html#_1078")
   ("*input-wait-handler*" . "22-2.html#_1079")
   ("*input-context*" . "23-5.html#_1199")
   ("*help-gestures*" . "24-5.html#_1344")
   ("*delimiter-gestures*" . "24-2.html#_1328")
   ("*default-text-style*" . "11-1.html#_570")
   ("*default-server-path*" . "9-2.html#_457")
   ("*default-frame-manager*" . "28-5.html#_1550")
   ("*completion-gestures*" . "24-5.html#_1343")
   ("*command-unparser*" . "27-6.html#_1446")
   ("*command-parser*" . "27-6.html#_1445")
   ("*command-name-delimiters*" . "27-6.html#_1450")
   ("*command-dispatchers*" . "27-6.html#_1456")
   ("*command-argument-delimiters*" . "27-6.html#_1451")
   ("*application-frame*" . "28-2.html#_1482")
   ("*activation-gestures*" . "24-2.html#_1324")
   ("*accelerator-gestures*" . "22-2.html#_1090")
   (":y-spacing" . "29-3.html#_1615") (":y-spacing" . "17-3.html#_933")
   (":y-spacing" . "17-3.html#_965") (":y-position" . "16-2.html#_841")
   (":x-spacing" . "29-3.html#_1614") (":x-spacing" . "17-3.html#_932")
   (":x-spacing" . "17-3.html#_964") (":x-position" . "16-2.html#_840")
   ("write-token" . "24-4.html#_1340") ("without-scheduling" . "B-2.html#_1895")
   (":within-generation-separation" . "18-2.html#_988")
   ("with-translation" . "10-2.html#_533") ("with-text-style" . "11-2.html#_589")
   ("with-text-size" . "11-2.html#_593") ("with-text-family" . "11-2.html#_591")
   ("with-text-face" . "11-2.html#_592")
   ("with-sheet-medium-bound" . "8-3.html#_415")
   ("with-sheet-medium" . "8-3.html#_414") ("with-scaling" . "10-2.html#_534")
   ("with-rotation" . "10-2.html#_535")
   ("with-room-for-graphics" . "15-4.html#_815")
   ("with-recursive-lock-held" . "B-3.html#_1902")
   ("with-radio-box" . "30-4.html#_1835")
   ("with-presentation-type-parameters" . "23-3.html#_1179")
   ("with-presentation-type-options" . "23-3.html#_1180")
   ("with-presentation-type-decoded" . "23-3.html#_1177")
   ("with-port-locked" . "9-2.html#_463")
   ("with-output-to-postscript-stream" . "E-1.html#_1964")
   ("with-output-to-pixmap" . "12-6.html#_654")
   ("with-output-to-output-record" . "16-4.html#_914")
   ("with-output-recording-options" . "16-4.html#_910")
   ("with-output-buffered" . "15-6.html#_828")
   ("with-output-as-presentation" . "23-4.html#_1192")
   ("with-output-as-gadget" . "30-5.html#_1866")
   ("with-new-output-record" . "16-4.html#_912") ("with-menu" . "25.html#_1362")
   ("with-look-and-feel-realization" . "29-2.html#_1590")
   ("with-lock-held" . "B-3.html#_1900")
   ("with-local-coordinates" . "10-2.html#_538")
   ("with-input-focus" . "22-2.html#_1077")
   ("with-input-editor-typeout" . "24-1.html#_1303")
   ("with-input-editing" . "24-1.html#_1302")
   ("with-input-context" . "23-5.html#_1201")
   ("with-identity-transformation" . "10-2.html#_536")
   ("with-graft-locked" . "9-3.html#_478")
   ("with-frame-manager" . "28-5.html#_1551")
   ("with-first-quadrant-coordinates" . "10-2.html#_539")
   ("with-end-of-page-action" . "15-4.html#_822")
   ("with-end-of-line-action" . "15-4.html#_819")
   ("with-drawing-options" . "10-2.html#_525")
   ("with-delimiter-gestures" . "24-2.html#_1329")
   ("with-command-table-keystrokes" . "27-6.html#_1438")
   ("with-bounding-rectangle*" . "4-1.html#_160")
   ("with-application-frame" . "28-2.html#_1483")
   ("with-activation-gestures" . "24-2.html#_1326")
   ("with-accept-help" . "24-5.html#_1352")
   ("window-viewport-position" . "29-4.html#_1695")
   ("window-viewport" . "29-4.html#_1693")
   ("window-repaint-event" . "8-2.html#_370")
   ("window-refresh" . "29-4.html#_1692")
   ("window-manager-event" . "8-2.html#_371")
   ("window-manager-delete-event" . "8-2.html#_373")
   ("window-event-region" . "8-2.html#_366")
   ("window-event-native-region" . "8-2.html#_367")
   ("window-event-mirrored-sheet" . "8-2.html#_368")
   ("window-event" . "8-2.html#_364")
   ("window-erase-viewport" . "29-4.html#_1694")
   ("window-configuration-event" . "8-2.html#_369")
   ("window-clear" . "29-4.html#_1691") ("window" . "29-4.html#_1690")
   ("vrack-pane" . "29-3.html#_1623") ("volatile" . "2-4.html#_16")
   ("viewport" . "29-3.html#_1639") ("viewp" . "23-6.html#_1210")
   ("view" . "23-6.html#_1209") (":view" . "23-2.html#_1134")
   ("vertically" . "29-3.html#_1621") (":vertical-spacing" . "29-4.html#_1672")
   (":vertical-spacing" . "15-2.html#_776") ("vbox-pane" . "29-3.html#_1620")
   (":value-key" . "30-4.html#_1841") (":value-key" . "30-4.html#_1849")
   ("value-gadget" . "30-3.html#_1732")
   (":value-changed-callback" . "30-3.html#_1734")
   ("value-changed-callback" . "30-3.html#_1738")
   ("using-resource" . "B-1.html#_1873")
   ("user-command-table" . "27-2.html#_1412") ("user" . "2-1.html#_5")
   ("updating-output-record-p" . "21-3.html#_1025")
   ("updating-output-record" . "21-3.html#_1024")
   ("updating-output" . "21-2.html#_1019")
   ("untransform-region" . "5-3.html#_215")
   ("untransform-rectangle*" . "5-3.html#_221")
   ("untransform-position" . "5-3.html#_217")
   ("untransform-distance" . "5-3.html#_219")
   ("unread-gesture" . "22-2.html#_1084") (":unique-id" . "21-3.html#_1026")
   ("uniform" . "13-2.html#_678")
   ("unhighlight-highlighted-presentation" . "23-7.html#_1249")
   ("unbounded" . "13-2.html#_677") ("type-or-string" . "23-8.html#_1292")
   (":type" . "23-2.html#_1133") ("tree-recompute-extent" . "16-2.html#_869")
   ("translucent" . "13-2.html#_681")
   ("translation-transformation-p" . "5-3.html#_196")
   ("translating" . "23-7.html#_1228") ("transformationp" . "5-1.html#_173")
   ("transformation-underspecified" . "5-1.html#_177")
   ("transformation-error" . "5-1.html#_176")
   ("transformation-equal" . "5-3.html#_193")
   ("transformation" . "5-1.html#_172") (":transformation" . "10-2.html#_528")
   ("transform-region" . "5-3.html#_214")
   ("transform-rectangle*" . "5-3.html#_220")
   ("transform-position" . "5-3.html#_216")
   ("transform-distance" . "5-3.html#_218")
   ("tracking-pointer" . "22-5.html#_1121") ("token-or-type" . "23-8.html#_1290")
   ("toggle-button-pane" . "30-4.html#_1776")
   ("toggle-button-indicator-type" . "30-4.html#_1774")
   ("toggle-button" . "30-4.html#_1772") ("title-pane" . "29-4.html#_1683")
   (":timestamp" . "8-2.html#_327") ("timer-event" . "8-2.html#_374")
   ("tiling" . "14-3.html#_743")
   ("throw-highlighted-presentation" . "23-7.html#_1246")
   ("textual-view" . "23-6.html#_1211") ("textual-menu-view" . "23-6.html#_1212")
   ("textual-dialog-view" . "23-6.html#_1213")
   ("text-style-width" . "11-1.html#_585") ("text-style-size" . "11-1.html#_579")
   ("text-style-p" . "11-1.html#_567") ("text-style-mapping" . "11-3.html#_595")
   ("text-style-height" . "11-1.html#_584")
   ("text-style-fixed-width-p" . "11-1.html#_586")
   ("text-style-family" . "11-1.html#_575")
   ("text-style-face" . "11-1.html#_577")
   ("text-style-descent" . "11-1.html#_583")
   ("text-style-components" . "11-1.html#_573")
   ("text-style-ascent" . "11-1.html#_582") ("text-style" . "11-1.html#_566")
   (":text-style" . "10-2.html#_531") (":text-style" . "29-2.html#_1594")
   (":text-style" . "15-2.html#_775") (":text-size" . "11-1.html#_578")
   ("text-size" . "11-1.html#_587") (":text-margin" . "29-4.html#_1671")
   (":text-margin" . "15-2.html#_777") ("text-field-pane" . "30-4.html#_1857")
   ("text-field" . "30-4.html#_1854") (":text-family" . "11-1.html#_574")
   (":text-face" . "11-1.html#_576") ("text-editor-pane" . "30-4.html#_1863")
   ("text-editor" . "30-4.html#_1859")
   ("text-displayed-output-record-string" . "16-3.html#_882")
   ("text-displayed-output-record-p" . "16-3.html#_879")
   ("text-displayed-output-record" . "16-3.html#_878")
   (":text-cursor" . "22-2.html#_1069")
   ("test-presentation-translator" . "23-7.html#_1238")
   (":test" . "30-4.html#_1842") (":test" . "30-4.html#_1850")
   ("temporary-medium-sheet-output-mixin" . "8-3.html#_412")
   ("tabling" . "29-3.html#_1625") ("table-pane" . "29-3.html#_1624")
   ("table-output-record-p" . "17-3.html#_931")
   ("table-output-record" . "17-3.html#_930") ("symbol" . "23-8.html#_1257")
   ("surrounding-output-with-border" . "19.html#_1006")
   ("suggest" . "24-5.html#_1349")
   ("substitute-numeric-argument-marker" . "27-4.html#_1430")
   ("subset-sequence" . "23-8.html#_1281")
   ("subset-completion" . "23-8.html#_1279") ("subset-alist" . "23-8.html#_1282")
   ("subset" . "23-8.html#_1280") ("structured" . "2-3.html#_10")
   ("string" . "23-8.html#_1271") ("streams" . "D.html#_1918")
   ("streamp" . "D-1.html#_1923") ("stream-write-string" . "15-1.html#_761")
   ("stream-write-string" . "D-4.html#_1952")
   ("stream-write-char" . "15-1.html#_760")
   ("stream-write-char" . "D-4.html#_1949")
   ("stream-write-byte" . "D-5.html#_1961")
   ("stream-vertical-spacing" . "15-4.html#_812")
   ("stream-unread-gesture" . "24-1.html#_1320")
   ("stream-unread-gesture" . "22-2.html#_1085")
   ("stream-unread-char" . "22-1.html#_1059")
   ("stream-unread-char" . "D-3.html#_1942")
   ("stream-truename" . "D-2.html#_1939")
   ("stream-text-output-record" . "16-4.html#_905")
   ("stream-text-margin" . "15-4.html#_809")
   ("stream-text-cursor" . "15-3.html#_801") ("stream-terpri" . "15-1.html#_762")
   ("stream-terpri" . "D-4.html#_1953")
   ("stream-string-width" . "15-4.html#_808")
   ("stream-start-line-p" . "15-1.html#_768")
   ("stream-start-line-p" . "D-4.html#_1951")
   ("stream-set-input-focus" . "22-2.html#_1076")
   ("stream-scan-pointer" . "24-1.html#_1309")
   ("stream-rescanning-p" . "24-1.html#_1311")
   ("stream-replay" . "16-4.html#_900")
   ("stream-redisplaying-p" . "21-4.html#_1051")
   ("stream-recording-p" . "16-4.html#_892")
   ("stream-read-line" . "22-1.html#_1062")
   ("stream-read-line" . "D-3.html#_1946")
   ("stream-read-gesture" . "24-1.html#_1319")
   ("stream-read-gesture" . "22-2.html#_1082")
   ("stream-read-char-no-hang" . "22-1.html#_1058")
   ("stream-read-char-no-hang" . "D-3.html#_1943")
   ("stream-read-char" . "22-1.html#_1057")
   ("stream-read-char" . "D-3.html#_1941")
   ("stream-read-byte" . "D-5.html#_1960")
   ("stream-process-gesture" . "24-1.html#_1318")
   ("stream-present" . "23-4.html#_1194")
   ("stream-pointer-position" . "22-2.html#_1074")
   ("stream-peek-char" . "22-1.html#_1060")
   ("stream-peek-char" . "D-3.html#_1944") ("stream-pathname" . "D-2.html#_1938")
   ("stream-output-history-mixin" . "16-3.html#_884")
   ("stream-output-history" . "16-4.html#_896")
   ("stream-listen" . "22-1.html#_1061") ("stream-listen" . "D-3.html#_1945")
   ("stream-line-height" . "15-4.html#_811")
   ("stream-line-column" . "15-1.html#_767")
   ("stream-line-column" . "D-4.html#_1950")
   ("stream-insertion-pointer" . "24-1.html#_1307")
   ("stream-input-wait" . "22-2.html#_1083")
   ("stream-input-buffer" . "24-1.html#_1306")
   ("stream-input-buffer" . "22-2.html#_1072")
   ("stream-increment-cursor-position" . "15-3.html#_805")
   ("stream-fresh-line" . "15-1.html#_763")
   ("stream-fresh-line" . "D-4.html#_1954")
   ("stream-force-output" . "15-1.html#_765")
   ("stream-force-output" . "D-4.html#_1956")
   ("stream-finish-output" . "15-1.html#_764")
   ("stream-finish-output" . "D-4.html#_1955")
   ("stream-end-of-page-action" . "15-4.html#_820")
   ("stream-end-of-line-action" . "15-4.html#_817")
   ("stream-element-type" . "D-2.html#_1935")
   ("stream-drawing-p" . "16-4.html#_894")
   ("stream-default-view" . "23-6.html#_1225")
   ("stream-cursor-position" . "15-3.html#_803")
   ("stream-current-output-record" . "16-4.html#_897")
   ("stream-close-text-output-record" . "16-4.html#_906")
   ("stream-clear-output" . "15-1.html#_766")
   ("stream-clear-output" . "D-4.html#_1957")
   ("stream-clear-input" . "22-1.html#_1063")
   ("stream-clear-input" . "D-3.html#_1947")
   ("stream-character-width" . "15-4.html#_807")
   ("stream-baseline" . "15-4.html#_813")
   ("stream-advance-to-column" . "15-1.html#_769")
   ("stream-advance-to-column" . "D-4.html#_1958")
   ("stream-add-string-output" . "16-4.html#_908")
   ("stream-add-output-record" . "16-4.html#_899")
   ("stream-add-character-output" . "16-4.html#_907")
   ("stream-accept" . "23-5.html#_1203") (":stream" . "C-1.html#_1911")
   ("stencil" . "14-2.html#_737") (":state" . "28-2.html#_1477")
   ("state" . "15-3.html#_785")
   ("standard-updating-output-record" . "21-3.html#_1031")
   ("standard-tree-output-record" . "16-3.html#_873")
   ("standard-tree-output-history" . "16-3.html#_885")
   ("standard-text-style" . "11-1.html#_568")
   ("standard-text-cursor" . "15-3.html#_789")
   ("standard-table-output-record" . "17-3.html#_936")
   ("standard-sheet-output-mixin" . "8-3.html#_408")
   ("standard-sheet-input-mixin" . "8-1.html#_316")
   ("standard-sequence-output-record" . "16-3.html#_872")
   ("standard-sequence-output-history" . "16-3.html#_886")
   ("standard-row-output-record" . "17-3.html#_943")
   ("standard-repainting-mixin" . "8-4.html#_432")
   ("standard-region-union" . "3-1.html#_59")
   ("standard-region-intersection" . "3-1.html#_60")
   ("standard-region-difference" . "3-1.html#_61")
   ("standard-rectangle" . "3-2.html#_114")
   ("standard-presentation" . "23-2.html#_1131")
   ("standard-polyline" . "3-2.html#_87") ("standard-polygon" . "3-2.html#_92")
   ("standard-pointer" . "22-4.html#_1111") ("standard-point" . "3-2.html#_75")
   ("standard-output-stream" . "15-1.html#_759")
   ("standard-output-recording-stream" . "16-4.html#_890")
   ("standard-line-style" . "10-3.html#_544") ("standard-line" . "3-2.html#_103")
   ("standard-item-list-output-record" . "17-3.html#_972")
   ("standard-input-stream" . "22-1.html#_1056")
   ("standard-input-editing-stream" . "24-1.html#_1301")
   ("standard-graph-output-record" . "18-2.html#_982")
   ("standard-graph-node-output-record" . "18-2.html#_998")
   ("standard-extended-output-stream" . "15-2.html#_781")
   ("standard-extended-input-stream" . "22-2.html#_1070")
   ("standard-encapsulating-stream" . "C-1.html#_1912")
   ("standard-elliptical-arc" . "3-2.html#_139")
   ("standard-ellipse" . "3-2.html#_134")
   ("standard-command-table" . "27-2.html#_1387")
   ("standard-column-output-record" . "17-3.html#_947")
   ("standard-cell-output-record" . "17-3.html#_956")
   ("standard-bounding-rectangle" . "4-1.html#_154")
   ("standard-application-frame" . "28-2.html#_1479") ("spread" . "2-3.html#_11")
   ("spacing-pane" . "29-3.html#_1627") (":spacing" . "29-3.html#_1616")
   ("spacing" . "29-3.html#_1628")
   ("space-requirement-width" . "29-3.html#_1650")
   ("space-requirement-min-width" . "29-3.html#_1651")
   ("space-requirement-min-height" . "29-3.html#_1654")
   ("space-requirement-max-width" . "29-3.html#_1652")
   ("space-requirement-max-height" . "29-3.html#_1655")
   ("space-requirement-height" . "29-3.html#_1653")
   ("space-requirement-components" . "29-3.html#_1656")
   ("space-requirement-combine" . "29-3.html#_1657")
   ("space-requirement" . "29-3.html#_1648")
   ("space-requirement+" . "29-3.html#_1658")
   ("space-requirement+*" . "29-3.html#_1659") ("solid" . "13-2.html#_680")
   ("slider-pane" . "30-4.html#_1819")
   ("slider-drag-callback" . "30-4.html#_1816") ("slider" . "30-4.html#_1806")
   (":size" . "16-2.html#_843") ("singular-transformation" . "5-1.html#_179")
   (":single-box" . "23-2.html#_1135") ("simple-parse-error" . "24-3.html#_1333")
   ("simple-parse-error" . "24-3.html#_1332")
   ("simple-completion-error" . "24-5.html#_1347") ("shrunk" . "28-5.html#_1544")
   ("shrink-frame" . "28-5.html#_1563") (":show-value-p" . "30-4.html#_1808")
   (":show-as-default" . "30-4.html#_1768") ("sheetp" . "7-1.html#_239")
   ("sheet-y-inverting-transformation-mixin" . "7-3.html#_292")
   ("sheet-with-medium-mixin" . "8-3.html#_410")
   ("sheet-viewable-p" . "7-2.html#_261")
   ("sheet-translation-mixin" . "7-3.html#_291")
   ("sheet-transformation-mixin" . "7-3.html#_293")
   ("sheet-transformation" . "7-3.html#_271")
   ("sheet-single-child-mixin" . "7-2.html#_267")
   ("sheet-siblings" . "7-2.html#_253") ("sheet-region" . "7-3.html#_273")
   ("sheet-parent-mixin" . "7-2.html#_265") ("sheet-parent" . "7-2.html#_249")
   ("sheet-occluding-sheets" . "7-2.html#_262")
   ("sheet-native-transformation" . "9-4.html#_498")
   ("sheet-native-region" . "9-4.html#_499")
   ("sheet-mute-repainting-mixin" . "8-4.html#_434")
   ("sheet-mute-output-mixin" . "8-3.html#_409")
   ("sheet-mute-input-mixin" . "8-1.html#_318")
   ("sheet-multiple-child-mixin" . "7-2.html#_268")
   ("sheet-mirrored-ancestor" . "9-4.html#_490")
   ("sheet-mirror" . "9-4.html#_491") ("sheet-medium" . "8-3.html#_416")
   ("sheet-leaf-mixin" . "7-2.html#_266")
   ("sheet-identity-transformation-mixin" . "7-3.html#_290")
   ("sheet-grafted-p" . "9-3.html#_474") ("sheet-event-queue" . "8-1.html#_302")
   ("sheet-enabled-p" . "7-2.html#_259")
   ("sheet-enabled-children" . "7-2.html#_254")
   ("sheet-disown-child" . "7-2.html#_252")
   ("sheet-direct-mirror" . "9-4.html#_489")
   ("sheet-device-transformation" . "9-4.html#_500")
   ("sheet-device-region" . "9-4.html#_501")
   ("sheet-delta-transformation" . "7-3.html#_287")
   ("sheet-children" . "7-2.html#_250") ("sheet-ancestor-p" . "7-2.html#_255")
   ("sheet-allocated-region" . "7-3.html#_288")
   ("sheet-adopt-child" . "7-2.html#_251") ("sheet" . "7-1.html#_238")
   (":sheet" . "8-2.html#_331") (":sheet" . "8-2.html#_372")
   (":sheet" . "15-3.html#_788") ("sheet" . "6-1.html#_226")
   ("set-highlighted-presentation" . "23-7.html#_1248")
   ("sequence-enumerated" . "23-8.html#_1285") ("sequence" . "23-8.html#_1284")
   ("scrolling" . "29-3.html#_1638") ("scroller-pane" . "29-3.html#_1637")
   (":scroll-up-page-callback" . "30-4.html#_1788")
   ("scroll-up-page-callback" . "30-4.html#_1800")
   (":scroll-up-line-callback" . "30-4.html#_1786")
   ("scroll-up-line-callback" . "30-4.html#_1799")
   (":scroll-to-top-callback" . "30-4.html#_1784")
   ("scroll-to-top-callback" . "30-4.html#_1797")
   (":scroll-to-bottom-callback" . "30-4.html#_1783")
   ("scroll-to-bottom-callback" . "30-4.html#_1798")
   ("scroll-extent" . "29-3.html#_1643")
   (":scroll-down-page-callback" . "30-4.html#_1787")
   ("scroll-down-page-callback" . "30-4.html#_1802")
   (":scroll-down-line-callback" . "30-4.html#_1785")
   ("scroll-down-line-callback" . "30-4.html#_1801")
   ("scroll-bar-scroll-up-page-callback" . "30-4.html#_1795")
   ("scroll-bar-scroll-up-line-callback" . "30-4.html#_1793")
   ("scroll-bar-scroll-to-top-callback" . "30-4.html#_1791")
   ("scroll-bar-scroll-to-bottom-callback" . "30-4.html#_1790")
   ("scroll-bar-scroll-down-page-callback" . "30-4.html#_1794")
   ("scroll-bar-scroll-down-line-callback" . "30-4.html#_1792")
   ("scroll-bar-pane" . "30-4.html#_1804")
   ("scroll-bar-drag-callback" . "30-4.html#_1789")
   ("scroll-bar" . "30-4.html#_1781") ("run-frame-top-level" . "28-4.html#_1532")
   ("run-frame-top-level" . "28-4.html#_1533") (":row-wise" . "17-3.html#_967")
   ("row-output-record-p" . "17-3.html#_942")
   ("row-output-record" . "17-3.html#_941")
   ("rigid-transformation-p" . "5-3.html#_198") ("right-handed" . "3.html#_32")
   ("restraining-pane" . "29-3.html#_1631") ("restraining" . "29-3.html#_1632")
   ("restart-process" . "B-2.html#_1894") ("restart-port" . "9-2.html#_470")
   ("resources" . "B-1.html#_1871") ("resize-sheet" . "7-3.html#_276")
   ("reset-scan-pointer" . "24-1.html#_1312") ("reset-frame" . "28-5.html#_1575")
   ("rescan-if-necessary" . "24-1.html#_1315")
   ("replay-output-record" . "16-2.html#_853") ("replay" . "16-2.html#_852")
   ("replace-input" . "24-4.html#_1337") ("repaint-sheet" . "8-4.html#_430")
   ("reorder-sheets" . "7-2.html#_258")
   ("remove-presentation-translator-from-command-table" . "27-5.html#_1433")
   ("remove-menu-item-from-command-table" . "27-3.html#_1415")
   ("remove-keystroke-from-command-table" . "27-4.html#_1425")
   ("remove-command-from-command-table" . "27-2.html#_1401")
   ("regionp" . "3-1.html#_40") ("region-union" . "3-1.html#_65")
   ("region-set-regions" . "3-1.html#_62") ("region-set-p" . "3-1.html#_58")
   ("region-set" . "3-1.html#_57")
   ("region-intersects-region-p" . "3-1.html#_54")
   ("region-intersection" . "3-1.html#_66") ("region-equal" . "3-1.html#_51")
   ("region-difference" . "3-1.html#_67")
   ("region-contains-region-p" . "3-1.html#_52")
   ("region-contains-position-p" . "3-1.html#_53") ("region" . "3-1.html#_39")
   (":region" . "8-2.html#_365") ("region" . "3-1.html#_35")
   ("reflection-underspecified" . "5-1.html#_178")
   ("reflection-transformation-p" . "5-3.html#_197")
   ("redraw-input-buffer" . "24-1.html#_1317")
   ("redisplayable-stream-p" . "21-4.html#_1050")
   ("redisplay-output-record" . "21-2.html#_1022")
   ("redisplay-frame-panes" . "28-3.html#_1519")
   ("redisplay-frame-pane" . "28-3.html#_1518") ("redisplay" . "21-2.html#_1021")
   ("rectilinear-transformation-p" . "5-3.html#_201")
   ("rectanglep" . "3-2.html#_113") ("rectangle-width" . "3-2.html#_125")
   ("rectangle-size" . "3-2.html#_127") ("rectangle-min-y" . "3-2.html#_122")
   ("rectangle-min-x" . "3-2.html#_121")
   ("rectangle-min-point" . "3-2.html#_119")
   ("rectangle-max-y" . "3-2.html#_124") ("rectangle-max-x" . "3-2.html#_123")
   ("rectangle-max-point" . "3-2.html#_120")
   ("rectangle-height" . "3-2.html#_126") ("rectangle-edges*" . "3-2.html#_118")
   ("rectangle" . "3-2.html#_112") (":record" . "29-4.html#_1677")
   ("recompute-extent-for-new-child" . "16-2.html#_867")
   ("recompute-extent-for-changed-child" . "16-2.html#_868")
   ("recompute-contents-ok" . "21-3.html#_1045")
   ("realize-mirror" . "9-4.html#_492") ("real" . "23-8.html#_1264")
   ("read-token" . "24-4.html#_1339") ("read-gesture" . "22-2.html#_1081")
   ("read-frame-command" . "28-4.html#_1535")
   ("read-command-using-keystrokes" . "27-6.html#_1439")
   ("read-command" . "27-6.html#_1437") ("read-bitmap-file" . "E-2.html#_1967")
   ("rational" . "23-8.html#_1265") ("ratio" . "23-8.html#_1267")
   (":range-label-text-style" . "30-4.html#_1812")
   ("range-gadget-mixin" . "30-3.html#_1756") ("raise-sheet" . "7-2.html#_256")
   ("raise-mirror" . "9-4.html#_494") ("raise-frame" . "28-2.html#_1486")
   ("radio-box-selections" . "30-4.html#_1825")
   ("radio-box-pane" . "30-4.html#_1827")
   ("radio-box-current-selection" . "30-4.html#_1823")
   ("queue-rescan" . "24-1.html#_1314") ("queue-repaint" . "8-4.html#_428")
   ("queue-event" . "8-1.html#_308")
   ("push-button-show-as-default" . "30-4.html#_1769")
   ("push-button-pane" . "30-4.html#_1770") ("push-button" . "30-4.html#_1767")
   (":properties" . "28-2.html#_1478")
   ("propagate-output-record-changes-p" . "21-3.html#_1040")
   ("propagate-output-record-changes" . "21-3.html#_1041")
   ("prompt-for-accept-1" . "23-5.html#_1207")
   ("prompt-for-accept" . "23-5.html#_1206") ("programmer" . "2-1.html#_6")
   ("processp" . "B-2.html#_1884") ("process-yield" . "B-2.html#_1890")
   ("process-whostate" . "B-2.html#_1887")
   ("process-wait-with-timeout" . "B-2.html#_1889")
   ("process-wait" . "B-2.html#_1888") ("process-state" . "B-2.html#_1886")
   ("process-next-event" . "8-1.html#_303") ("process-name" . "B-2.html#_1885")
   ("process-interrupt" . "B-2.html#_1891") ("print-menu-item" . "25.html#_1358")
   (":pretty-name" . "28-2.html#_1471") ("presentationp" . "23-2.html#_1130")
   ("presentation-typep" . "23-3.html#_1165")
   ("presentation-typep" . "23-3.html#_1182")
   ("presentation-type-specifier-p" . "23-3.html#_1164")
   ("presentation-type-specifier-p" . "23-3.html#_1181")
   ("presentation-type-parameters" . "23-3.html#_1175")
   ("presentation-type-options" . "23-3.html#_1176")
   ("presentation-type-of" . "23-3.html#_1183")
   ("presentation-type-name" . "23-3.html#_1178")
   ("presentation-type-history" . "23-3.html#_1169")
   ("presentation-type-direct-supertypes" . "23-3.html#_1186")
   ("presentation-type" . "23-2.html#_1140")
   ("presentation-subtypep" . "23-3.html#_1166")
   ("presentation-subtypep" . "23-3.html#_1184")
   ("presentation-single-box" . "23-2.html#_1142")
   ("presentation-replace-input" . "24-4.html#_1338")
   ("presentation-refined-position-test" . "23-3.html#_1171")
   ("presentation-object" . "23-2.html#_1138")
   ("presentation-modifier" . "23-2.html#_1144")
   ("presentation-matches-context-type" . "23-7.html#_1240")
   ("presentation-default-preprocessor" . "23-3.html#_1170")
   ("presentation" . "23-2.html#_1129") ("presentation" . "23-2.html#_1128")
   ("present-to-string" . "23-4.html#_1195") ("present" . "23-3.html#_1161")
   ("present" . "23-4.html#_1193") ("present" . "27-2.html#_1383")
   ("ports" . "9-1.html#_448") ("ports" . "6-1.html#_232")
   ("portp" . "9-2.html#_454") ("port-type" . "9-2.html#_467")
   ("port-server-path" . "9-2.html#_465") ("port-properties" . "9-2.html#_468")
   ("port-name" . "9-2.html#_466")
   ("port-keyboard-input-focus" . "8-1.html#_304") ("port" . "9-2.html#_453")
   ("port" . "8-3.html#_419") ("port" . "9-4.html#_496")
   ("port" . "22-4.html#_1119") ("port" . "28-5.html#_1558")
   ("port" . "28-5.html#_1559") (":port" . "22-4.html#_1110")
   ("port" . "9-2.html#_462") ("port" . "9-2.html#_451")
   ("polylinep" . "3-2.html#_86") ("polyline-closed" . "3-2.html#_99")
   ("polyline" . "3-2.html#_85") ("polyline" . "3-2.html#_82")
   ("polygonp" . "3-2.html#_91") ("polygon-points" . "3-2.html#_96")
   ("polygon" . "3-2.html#_90") ("polygon" . "3-2.html#_84")
   ("pointp" . "3-2.html#_74") ("pointerp" . "22-4.html#_1109")
   ("pointer-sheet" . "22-4.html#_1112") ("pointer-position" . "22-4.html#_1115")
   ("pointer-motion-event" . "8-2.html#_359")
   ("pointer-exit-event" . "8-2.html#_363") ("pointer-event-y" . "8-2.html#_347")
   ("pointer-event-x" . "8-2.html#_346")
   ("pointer-event-pointer" . "8-2.html#_350")
   ("pointer-event-native-y" . "8-2.html#_349")
   ("pointer-event-native-x" . "8-2.html#_348")
   ("pointer-event-button" . "8-2.html#_352") ("pointer-event" . "8-2.html#_341")
   ("pointer-enter-event" . "8-2.html#_362")
   ("pointer-double-click-event" . "8-2.html#_357")
   ("pointer-documentation-view" . "23-6.html#_1217")
   ("pointer-documentation-pane" . "29-4.html#_1684")
   ("pointer-cursor" . "22-4.html#_1117")
   ("pointer-click-event" . "8-2.html#_356")
   ("pointer-click-and-hold-event" . "8-2.html#_358")
   ("pointer-button-state" . "22-4.html#_1114")
   ("pointer-button-release-event" . "8-2.html#_354")
   ("pointer-button-press-event" . "8-2.html#_353")
   ("pointer-button-hold-event" . "8-2.html#_355")
   ("pointer-button-event" . "8-2.html#_351")
   ("pointer-boundary-event-kind" . "8-2.html#_361")
   ("pointer-boundary-event" . "8-2.html#_360") ("pointer" . "22-4.html#_1108")
   (":pointer" . "8-2.html#_342") (":pointer" . "22-2.html#_1068")
   ("point-y" . "3-2.html#_80") ("point-x" . "3-2.html#_79")
   ("point-position" . "3-2.html#_78") ("point" . "3-2.html#_73")
   ("point" . "3-2.html#_72") ("pixmap-width" . "12-6.html#_647")
   ("pixmap-height" . "12-6.html#_648") ("pixmap-depth" . "12-6.html#_649")
   ("pixmap" . "12-6.html#_644")
   ("permanent-medium-sheet-output-mixin" . "8-3.html#_411")
   ("patterning" . "14-2.html#_736") ("pattern-width" . "14-2.html#_739")
   ("pattern-height" . "14-2.html#_740") ("pathp" . "3-1.html#_43")
   ("pathname" . "23-8.html#_1273") ("path" . "3-1.html#_41")
   ("partial-command-p" . "27-1.html#_1379")
   ("parse-text-style" . "11-1.html#_580") (":parent" . "16-2.html#_842")
   ("parent" . "6-1.html#_228") (":panes" . "28-2.html#_1474")
   ("panes" . "28-1.html#_1461") ("panep" . "29-2.html#_1586")
   ("pane-viewport-region" . "29-3.html#_1641")
   ("pane-viewport" . "29-3.html#_1640") ("pane-text-style" . "29-2.html#_1601")
   ("pane-scroller" . "29-3.html#_1642")
   ("pane-needs-redisplay" . "28-3.html#_1516") ("pane-name" . "29-2.html#_1598")
   ("pane-frame" . "29-2.html#_1597") ("pane-foreground" . "29-2.html#_1599")
   ("pane-background" . "29-2.html#_1600") ("pane" . "29-2.html#_1585")
   ("output-stream-p" . "D-1.html#_1927")
   ("output-recording-stream-p" . "16-4.html#_889")
   ("output-recording-stream" . "16-4.html#_888")
   ("output-record-unique-id" . "21-3.html#_1032")
   ("output-record-start-cursor-position" . "16-2.html#_847")
   ("output-record-refined-position-test" . "16-2.html#_855")
   ("output-record-position" . "16-2.html#_845")
   ("output-record-parent" . "16-2.html#_851")
   ("output-record-p" . "16-2.html#_837")
   ("output-record-hit-detection-rectangle*" . "16-2.html#_854")
   ("output-record-fixed-position" . "21-3.html#_1034")
   ("output-record-end-cursor-position" . "16-2.html#_849")
   ("output-record-displayer" . "21-3.html#_1035")
   ("output-record-count" . "16-2.html#_863")
   ("output-record-contents-ok" . "21-3.html#_1044")
   ("output-record-children" . "16-2.html#_859")
   ("output-record-cache-value" . "21-3.html#_1033")
   ("output-record" . "16-2.html#_836") (":output-record" . "29-4.html#_1675")
   ("outlining" . "29-3.html#_1630") ("outlined-pane" . "29-3.html#_1629")
   ("oriented-gadget-mixin" . "30-3.html#_1743")
   (":orientation" . "18-2.html#_983") (":orientation" . "30-3.html#_1744")
   ("or" . "23-8.html#_1287") ("option-pane" . "30-4.html#_1845")
   (":openlook" . "9-2.html#_460") ("open-window-stream" . "29-4.html#_1698")
   ("open-stream-p" . "D-2.html#_1936") ("opacityp" . "13-4.html#_708")
   ("opacity-value" . "13-4.html#_711") ("opacity" . "13-4.html#_707")
   (":number-of-tick-marks" . "30-4.html#_1813")
   (":number-of-quanta" . "30-4.html#_1814") ("number" . "23-8.html#_1262")
   ("null-or-type" . "23-8.html#_1291") ("null" . "23-8.html#_1255")
   ("notify-user" . "28-3.html#_1521")
   ("note-space-requirements-changed" . "29-3.html#_1663")
   ("note-sheet-transformation-changed" . "8-5.html#_445")
   ("note-sheet-region-changed" . "8-5.html#_444")
   ("note-sheet-grafted" . "8-5.html#_437")
   ("note-sheet-enabled" . "8-5.html#_441")
   ("note-sheet-disowned" . "8-5.html#_440")
   ("note-sheet-disabled" . "8-5.html#_442")
   ("note-sheet-degrafted" . "8-5.html#_438")
   ("note-sheet-adopted" . "8-5.html#_439")
   ("note-output-record-child-changed" . "21-3.html#_1039")
   ("note-gadget-deactivated" . "30-3.html#_1731")
   ("note-gadget-activated" . "30-3.html#_1730")
   ("note-frame-iconified" . "28-5.html#_1566")
   ("note-frame-enabled" . "28-5.html#_1564")
   ("note-frame-disabled" . "28-5.html#_1565")
   ("note-frame-deiconified" . "28-5.html#_1567")
   ("note-command-enabled" . "28-5.html#_1568")
   ("note-command-disabled" . "28-5.html#_1569")
   ("non-uniform" . "13-2.html#_679") (":nlines" . "30-4.html#_1861")
   ("nil" . "23-8.html#_1254") ("new-page" . "E-1.html#_1965")
   (":ncolumns" . "30-4.html#_1860") (":name-key" . "30-4.html#_1840")
   (":name-key" . "30-4.html#_1848") (":name" . "29-2.html#_1595")
   (":name" . "28-2.html#_1470") (":n-rows" . "17-3.html#_968")
   ("mutable" . "2-4.html#_14")
   (":multiple-columns-x-spacing" . "17-3.html#_934")
   ("move-sheet" . "7-3.html#_275") ("move-and-resize-sheet" . "7-3.html#_277")
   (":motif" . "9-2.html#_459")
   ("modifier-state-matches-gesture-name-p" . "22-3.html#_1104")
   (":modifier-state" . "8-2.html#_332") (":modifier" . "23-2.html#_1136")
   (":mode" . "30-4.html#_1838") (":mode" . "30-4.html#_1846")
   ("mirrored-sheet-mixin" . "9-4.html#_487") (":min-width" . "29-3.html#_1607")
   (":min-width" . "17-3.html#_954") (":min-value" . "30-3.html#_1757")
   (":min-label" . "30-4.html#_1810") (":min-height" . "29-3.html#_1610")
   (":min-height" . "17-3.html#_955") ("merge-text-styles" . "11-1.html#_581")
   (":merge-duplicates" . "18-2.html#_986")
   ("menu-read-remaining-arguments-for-partial-command" . "27-6.html#_1444")
   ("menu-item-value" . "25.html#_1359") ("menu-item-options" . "25.html#_1361")
   ("menu-item-display" . "25.html#_1360")
   ("menu-command-parser" . "27-6.html#_1443")
   ("menu-choose-from-drawer" . "25.html#_1356")
   ("menu-choose-command-from-command-table" . "27-3.html#_1422")
   ("menu-choose" . "25.html#_1354") ("menu-button-pane" . "30-4.html#_1779")
   ("menu-button" . "30-4.html#_1778") (":menu-bar" . "28-2.html#_1475")
   ("member-sequence" . "23-8.html#_1277") ("member-alist" . "23-8.html#_1278")
   ("member" . "23-8.html#_1276") ("mediump" . "8-3.html#_387")
   ("medium-transformation" . "8-3.html#_395")
   ("medium-transformation" . "10-1.html#_513")
   ("medium-text-style" . "8-3.html#_401")
   ("medium-text-style" . "10-1.html#_521") ("medium-sheet" . "8-3.html#_417")
   ("medium-merged-text-style" . "8-3.html#_405")
   ("medium-line-style" . "8-3.html#_399")
   ("medium-line-style" . "10-1.html#_517") ("medium-ink" . "8-3.html#_393")
   ("medium-ink" . "10-1.html#_511") ("medium-foreground" . "8-3.html#_389")
   ("medium-foreground" . "10-1.html#_507")
   ("medium-force-output" . "12-7.html#_669")
   ("medium-finish-output" . "12-7.html#_668")
   ("medium-drawable" . "8-3.html#_418") ("medium-draw-text*" . "12-7.html#_666")
   ("medium-draw-rectangles*" . "12-7.html#_664")
   ("medium-draw-rectangle*" . "12-7.html#_663")
   ("medium-draw-polygon*" . "12-7.html#_662")
   ("medium-draw-points*" . "12-7.html#_659")
   ("medium-draw-point*" . "12-7.html#_658")
   ("medium-draw-lines*" . "12-7.html#_661")
   ("medium-draw-line*" . "12-7.html#_660")
   ("medium-draw-ellipse*" . "12-7.html#_665")
   ("medium-default-text-style" . "8-3.html#_403")
   ("medium-default-text-style" . "10-1.html#_519")
   ("medium-current-text-style" . "10-1.html#_523")
   ("medium-copy-area" . "12-6.html#_653")
   ("medium-clipping-region" . "8-3.html#_397")
   ("medium-clipping-region" . "10-1.html#_515")
   ("medium-clear-area" . "12-7.html#_670")
   ("medium-buffering-output-p" . "15-6.html#_826")
   ("medium-beep" . "12-7.html#_671") ("medium-background" . "8-3.html#_391")
   ("medium-background" . "10-1.html#_508") ("medium" . "8-3.html#_386")
   ("medium" . "8-3.html#_385") (":max-width" . "29-3.html#_1606")
   (":max-width" . "17-3.html#_970") (":max-value" . "30-3.html#_1758")
   (":max-label" . "30-4.html#_1811") (":max-height" . "29-3.html#_1609")
   (":max-height" . "17-3.html#_971") ("matching" . "23-7.html#_1232")
   ("match-output-records" . "21-3.html#_1042")
   ("map-sheet-rectangle*-to-parent" . "7-3.html#_280")
   ("map-sheet-rectangle*-to-child" . "7-3.html#_281")
   ("map-sheet-position-to-parent" . "7-3.html#_278")
   ("map-sheet-position-to-child" . "7-3.html#_279")
   ("map-resource" . "B-1.html#_1877")
   ("map-over-table-elements" . "17-3.html#_937")
   ("map-over-sheets-overlapping-region" . "7-3.html#_283")
   ("map-over-sheets-containing-position" . "7-3.html#_282")
   ("map-over-sheets" . "7-2.html#_263")
   ("map-over-row-cells" . "17-3.html#_944")
   ("map-over-row-cells" . "17-3.html#_948")
   ("map-over-region-set-regions" . "3-1.html#_63")
   ("map-over-presentation-type-supertypes" . "23-3.html#_1167")
   ("map-over-presentation-type-supertypes" . "23-3.html#_1185")
   ("map-over-ports" . "9-2.html#_464")
   ("map-over-polygon-segments" . "3-2.html#_98")
   ("map-over-polygon-coordinates" . "3-2.html#_97")
   ("map-over-output-records-overlapping-region" . "16-2.html#_865")
   ("map-over-output-records-containing-position" . "16-2.html#_864")
   ("map-over-item-list-cells" . "17-3.html#_973")
   ("map-over-grafts" . "9-3.html#_477") ("map-over-frames" . "28-2.html#_1484")
   ("map-over-command-table-translators" . "27-5.html#_1434")
   ("map-over-command-table-names" . "27-2.html#_1405")
   ("map-over-command-table-menu-items" . "27-3.html#_1416")
   ("map-over-command-table-keystrokes" . "27-4.html#_1426")
   ("map-over-command-table-commands" . "27-2.html#_1404")
   ("make-translation-transformation" . "5-2.html#_181")
   ("make-transformation" . "5-2.html#_188")
   ("make-text-style" . "11-1.html#_569") ("make-stencil" . "14-2.html#_741")
   ("make-space-requirement" . "29-3.html#_1649")
   ("make-scaling-transformation" . "5-2.html#_184")
   ("make-scaling-transformation*" . "5-2.html#_185")
   ("make-rotation-transformation" . "5-2.html#_182")
   ("make-rotation-transformation*" . "5-2.html#_183")
   ("make-rgb-color" . "13-3.html#_688")
   ("make-reflection-transformation" . "5-2.html#_186")
   ("make-reflection-transformation*" . "5-2.html#_187")
   ("make-recursive-lock" . "B-3.html#_1901")
   ("make-rectangular-tile" . "14-3.html#_744")
   ("make-rectangle" . "3-2.html#_115") ("make-rectangle*" . "3-2.html#_116")
   ("make-process" . "B-2.html#_1880")
   ("make-presentation-type-specifier" . "23-3.html#_1190")
   ("make-polyline" . "3-2.html#_88") ("make-polyline*" . "3-2.html#_89")
   ("make-polygon" . "3-2.html#_93") ("make-polygon*" . "3-2.html#_94")
   ("make-point" . "3-2.html#_76")
   ("make-pattern-from-bitmap-file" . "E-2.html#_1968")
   ("make-pattern" . "14-2.html#_738") ("make-pane-1" . "29-2.html#_1589")
   ("make-pane" . "29-2.html#_1588") ("make-opacity" . "13-4.html#_709")
   ("make-modifier-state" . "22-3.html#_1105") ("make-medium" . "8-3.html#_423")
   ("make-lock" . "B-3.html#_1899") ("make-line-style" . "10-3.html#_545")
   ("make-line" . "3-2.html#_104") ("make-line*" . "3-2.html#_105")
   ("make-ihs-color" . "13-3.html#_689") ("make-gray-color" . "13-3.html#_690")
   ("make-flipping-ink" . "13-7.html#_718")
   ("make-elliptical-arc" . "3-2.html#_140")
   ("make-elliptical-arc*" . "3-2.html#_141") ("make-ellipse" . "3-2.html#_135")
   ("make-ellipse*" . "3-2.html#_136")
   ("make-device-font-text-style" . "11-3.html#_597")
   ("make-design-from-output-record" . "16-4.html#_916")
   ("make-contrasting-inks" . "13-3.html#_704")
   ("make-contrasting-dash-patterns" . "10-3.html#_560")
   ("make-command-table" . "27-2.html#_1392")
   ("make-clim-stream-pane" . "29-4.html#_1686")
   ("make-clim-interactor-pane" . "29-4.html#_1687")
   ("make-clim-application-pane" . "29-4.html#_1688")
   ("make-bounding-rectangle" . "4-1.html#_155")
   ("make-application-frame" . "28-2.html#_1481")
   ("make-3-point-transformation" . "5-2.html#_189")
   ("lookup-keystroke-item" . "27-4.html#_1428")
   ("lookup-keystroke-command-item" . "27-4.html#_1429")
   ("list-pane" . "30-4.html#_1837") ("linep" . "3-2.html#_102")
   (":line-unit" . "10-3.html#_547") (":line-thickness" . "10-3.html#_549")
   ("line-style-unit" . "10-3.html#_548")
   ("line-style-thickness" . "10-3.html#_550")
   ("line-style-p" . "10-3.html#_543")
   ("line-style-joint-shape" . "10-3.html#_552")
   ("line-style-dashes" . "10-3.html#_558")
   ("line-style-cap-shape" . "10-3.html#_555") ("line-style" . "10-3.html#_542")
   (":line-style" . "10-2.html#_530") ("line-start-point" . "3-2.html#_109")
   ("line-start-point*" . "3-2.html#_107")
   (":line-joint-shape" . "10-3.html#_551") ("line-end-point" . "3-2.html#_110")
   ("line-end-point*" . "3-2.html#_108") (":line-dashes" . "10-3.html#_557")
   (":line-cap-shape" . "10-3.html#_554") ("line" . "3-2.html#_101")
   ("left-handed" . "3.html#_33") ("layout-graph-nodes" . "18-2.html#_994")
   ("layout-graph-edges" . "18-2.html#_995") ("layout-frame" . "28-3.html#_1512")
   ("labelling" . "29-3.html#_1635")
   ("labelled-gadget-mixin" . "30-3.html#_1746")
   ("label-pane" . "29-3.html#_1634") ("keyword" . "23-8.html#_1258")
   ("keyboard-event-key-name" . "8-2.html#_337")
   ("keyboard-event-character" . "8-2.html#_338")
   ("keyboard-event" . "8-2.html#_335") ("key-release-event" . "8-2.html#_340")
   ("key-press-event" . "8-2.html#_339") (":items" . "30-4.html#_1839")
   (":items" . "30-4.html#_1847")
   ("item-list-output-record-p" . "17-3.html#_963")
   ("item-list-output-record" . "17-3.html#_962")
   ("invoke-with-text-style" . "11-2.html#_590")
   ("invoke-with-output-to-output-record" . "16-4.html#_915")
   ("invoke-with-output-recording-options" . "16-4.html#_911")
   ("invoke-with-new-output-record" . "16-4.html#_913")
   ("invoke-with-drawing-options" . "10-2.html#_526")
   ("invoke-updating-output" . "21-2.html#_1020")
   ("invoke-accept-values-command-button" . "26.html#_1370")
   ("invertible-transformation-p" . "5-3.html#_195")
   ("invert-transformation" . "5-3.html#_206")
   ("invalidate-cached-transformations" . "9-4.html#_502")
   ("invalidate-cached-regions" . "9-4.html#_503") ("interning" . "2-4.html#_15")
   ("interactor-pane" . "29-4.html#_1680")
   ("interactive-stream-p" . "24-1.html#_1298") ("integer" . "23-8.html#_1266")
   ("inside" . "12-4.html#_607") ("input-stream-p" . "D-1.html#_1925")
   ("input-not-of-required-type" . "24-3.html#_1335")
   ("input-not-of-required-type" . "24-3.html#_1334")
   ("input-editor-format" . "24-1.html#_1304")
   ("input-editing-stream-p" . "24-1.html#_1300")
   ("input-editing-stream" . "24-1.html#_1299")
   ("input-context-type" . "23-5.html#_1200")
   (":input-buffer" . "22-2.html#_1067") (":ink" . "10-2.html#_527")
   (":initial-spacing" . "17-3.html#_966")
   (":indicator-type" . "30-4.html#_1773")
   ("indenting-output" . "20-2.html#_1012")
   (":incremental-redisplay" . "29-4.html#_1670")
   ("incremental-redisplay" . "21-4.html#_1052") ("implementor" . "2-1.html#_7")
   ("immutable" . "2-4.html#_13")
   ("immediate-sheet-input-mixin" . "8-1.html#_317")
   ("immediate-rescan" . "24-1.html#_1313")
   ("immediate-repainting-mixin" . "8-4.html#_433")
   ("identity-transformation-p" . "5-3.html#_194")
   (":id-test" . "21-3.html#_1027") ("hrack-pane" . "29-3.html#_1622")
   ("horizontally" . "29-3.html#_1619")
   ("highlight-presentation" . "23-3.html#_1172")
   ("highlight-output-record" . "16-2.html#_856")
   ("highlight-applicable-presentation" . "23-7.html#_1247")
   (":height" . "29-3.html#_1608") ("hbox-pane" . "29-3.html#_1618")
   (":hash-table" . "18-2.html#_989") ("handle-repaint" . "8-4.html#_429")
   ("grid-pane" . "29-3.html#_1626")
   ("graphics-displayed-output-record-p" . "16-3.html#_876")
   ("graphics-displayed-output-record" . "16-3.html#_875")
   ("graph-root-nodes" . "18-2.html#_991")
   ("graph-output-record-p" . "18-2.html#_981")
   ("graph-output-record" . "18-2.html#_980")
   ("graph-node-parents" . "18-2.html#_999")
   ("graph-node-output-record-p" . "18-2.html#_997")
   ("graph-node-output-record" . "18-2.html#_996")
   ("graph-node-object" . "18-2.html#_1003")
   ("graph-node-children" . "18-2.html#_1001") ("grafts" . "6-1.html#_233")
   ("grafts" . "9-1.html#_449") ("grafted" . "7-2.html#_244")
   ("graft-width" . "9-3.html#_481") ("graft-units" . "9-3.html#_480")
   ("graft-pixels-per-millimeter" . "9-3.html#_483")
   ("graft-pixels-per-inch" . "9-3.html#_484")
   ("graft-orientation" . "9-3.html#_479") ("graft-height" . "9-3.html#_482")
   ("graft" . "9-3.html#_476") ("graft" . "9-3.html#_473")
   ("global-command-table" . "27-2.html#_1411")
   ("get-frame-pane" . "28-3.html#_1507") ("gesture" . "22-3.html#_1095")
   ("generic-option-pane" . "30-4.html#_1852")
   ("generic-list-pane" . "30-4.html#_1844")
   (":generation-separation" . "18-2.html#_987")
   ("generate-panes" . "28-5.html#_1571")
   ("generate-graph-nodes" . "18-2.html#_993") (":genera" . "9-2.html#_461")
   ("gadgets" . "30-1.html#_1704") ("gadgetp" . "30-3.html#_1713")
   ("gadget-view" . "23-6.html#_1214")
   ("gadget-value-changed-callback" . "30-3.html#_1737")
   ("gadget-value" . "30-4.html#_1775") ("gadget-value" . "30-4.html#_1803")
   ("gadget-value" . "30-4.html#_1818") ("gadget-value" . "30-4.html#_1826")
   ("gadget-value" . "30-4.html#_1833") ("gadget-value" . "30-4.html#_1843")
   ("gadget-value" . "30-4.html#_1851") ("gadget-value" . "30-4.html#_1856")
   ("gadget-value" . "30-4.html#_1862") ("gadget-value" . "30-3.html#_1735")
   ("gadget-show-value-p" . "30-4.html#_1815")
   ("gadget-range" . "30-3.html#_1763") ("gadget-range*" . "30-3.html#_1764")
   ("gadget-output-record" . "30-5.html#_1865")
   ("gadget-orientation" . "30-3.html#_1745")
   ("gadget-min-value" . "30-3.html#_1759")
   ("gadget-menu-view" . "23-6.html#_1215")
   ("gadget-max-value" . "30-3.html#_1761")
   ("gadget-label-align-y" . "30-3.html#_1754")
   ("gadget-label-align-x" . "30-3.html#_1752")
   ("gadget-label" . "30-3.html#_1750") ("gadget-id" . "30-3.html#_1719")
   ("gadget-disarmed-callback" . "30-3.html#_1724")
   ("gadget-dialog-view" . "23-6.html#_1216")
   ("gadget-client" . "30-3.html#_1721")
   ("gadget-armed-callback" . "30-3.html#_1723")
   ("gadget-active-p" . "30-3.html#_1729")
   ("gadget-activate-callback" . "30-3.html#_1741")
   ("gadget" . "30-3.html#_1712") ("fundamental-stream" . "D-1.html#_1922")
   ("fundamental-output-stream" . "D-1.html#_1926")
   ("fundamental-input-stream" . "D-1.html#_1924")
   ("fundamental-character-stream" . "D-1.html#_1928")
   ("fundamental-character-output-stream" . "D-1.html#_1931")
   ("fundamental-character-input-stream" . "D-1.html#_1930")
   ("fundamental-binary-stream" . "D-1.html#_1929")
   ("fundamental-binary-output-stream" . "D-1.html#_1933")
   ("fundamental-binary-input-stream" . "D-1.html#_1932")
   ("funcall-presentation-generic-function" . "23-3.html#_1159")
   ("frames" . "28-1.html#_1460") ("frame-top-level-sheet" . "28-3.html#_1505")
   ("frame-state" . "28-5.html#_1560")
   ("frame-standard-output" . "28-3.html#_1496")
   ("frame-standard-input" . "28-3.html#_1497")
   ("frame-replay" . "28-3.html#_1520") ("frame-query-io" . "28-3.html#_1498")
   ("frame-properties" . "28-3.html#_1522")
   ("frame-pretty-name" . "28-3.html#_1492")
   ("frame-pointer-documentation-output" . "28-3.html#_1501")
   ("frame-parent" . "28-3.html#_1503") ("frame-panes" . "28-3.html#_1504")
   ("frame-name" . "28-3.html#_1491") ("frame-mananger-p" . "28-5.html#_1547")
   ("frame-manager-notify-user" . "28-5.html#_1570")
   ("frame-manager-menu-choose" . "25.html#_1355")
   ("frame-manager-frames" . "28-5.html#_1555")
   ("frame-manager" . "28-5.html#_1546") ("frame-manager" . "28-5.html#_1553")
   ("frame-maintain-presentation-histories" . "28-3.html#_1525")
   ("frame-input-context-button-press-handler" . "28-3.html#_1527")
   ("frame-find-innermost-applicable-presentation" . "28-3.html#_1526")
   ("frame-exit-frame" . "28-3.html#_1514") ("frame-exit" . "28-3.html#_1515")
   ("frame-exit" . "28-3.html#_1513") ("frame-error-output" . "28-3.html#_1499")
   ("frame-drag-and-drop-highlighting" . "28-3.html#_1530")
   ("frame-drag-and-drop-feedback" . "28-3.html#_1529")
   ("frame-document-highlighted-presentation" . "28-3.html#_1528")
   ("frame-current-panes" . "28-3.html#_1506")
   ("frame-current-layout" . "28-3.html#_1509")
   ("frame-command-table" . "28-3.html#_1494")
   ("frame-calling-frame" . "28-3.html#_1502")
   ("frame-all-layouts" . "28-3.html#_1511")
   ("formatting-table" . "17-2.html#_922") ("formatting-row" . "17-2.html#_923")
   ("formatting-item-list" . "17-2.html#_926")
   ("formatting-column" . "17-2.html#_924")
   ("formatting-cell" . "17-2.html#_925")
   ("format-textual-list" . "20-1.html#_1010")
   ("format-items" . "17-2.html#_927")
   ("format-graph-from-roots" . "18-1.html#_978") ("form" . "23-8.html#_1295")
   (":foreground" . "29-2.html#_1592") (":foreground" . "15-2.html#_773")
   ("float" . "23-8.html#_1268") (":fixed-position" . "21-3.html#_1030")
   ("find-presentation-type-class" . "23-3.html#_1187")
   ("find-presentation-translators" . "23-7.html#_1237")
   ("find-presentation-translator" . "27-5.html#_1435")
   ("find-port" . "9-2.html#_456") ("find-pane-named" . "28-3.html#_1508")
   ("find-pane-for-frame" . "28-5.html#_1572")
   ("find-menu-item" . "27-3.html#_1417")
   ("find-keystroke-item" . "27-4.html#_1427")
   ("find-innermost-applicable-presentation" . "23-7.html#_1245")
   ("find-graft" . "9-3.html#_475") ("find-frame-manager" . "28-5.html#_1549")
   ("find-command-table" . "27-2.html#_1393")
   ("find-command-from-command-line-name" . "27-2.html#_1408")
   ("find-child-output-record" . "21-3.html#_1043")
   ("find-cached-output-record" . "21-3.html#_1048")
   ("find-applicable-translators" . "23-7.html#_1239")
   ("extended-output-stream-p" . "15-2.html#_772")
   ("extended-output-stream" . "15-2.html#_771")
   ("extended-input-stream-p" . "22-2.html#_1066")
   ("extended-input-stream" . "22-2.html#_1065")
   ("expression" . "23-8.html#_1294")
   ("expand-presentation-type-abbreviation-1" . "23-3.html#_1153")
   ("expand-presentation-type-abbreviation" . "23-3.html#_1154")
   ("execute-frame-command" . "28-4.html#_1536") ("eventp" . "8-2.html#_326")
   ("event-unread" . "8-1.html#_313") ("event-type" . "8-2.html#_329")
   ("event-timestamp" . "8-2.html#_328") ("event-sheet" . "8-2.html#_333")
   ("event-read-no-hang" . "8-1.html#_311") ("event-read" . "8-1.html#_310")
   ("event-peek" . "8-1.html#_312") ("event-modifier-state" . "8-2.html#_334")
   ("event-matches-gesture-name-p" . "22-3.html#_1103")
   ("event-listen" . "8-1.html#_314") ("event" . "8-2.html#_325")
   ("event" . "8-2.html#_323")
   ("even-scaling-transformation-p" . "5-3.html#_199")
   ("erase-output-record" . "16-4.html#_901")
   ("erase-input-buffer" . "24-1.html#_1316")
   (":equalize-column-widths" . "17-3.html#_935")
   ("engraft-medium" . "8-3.html#_424")
   (":end-of-page-action" . "29-4.html#_1674")
   (":end-of-page-action" . "15-2.html#_779")
   (":end-of-line-action" . "29-4.html#_1673")
   (":end-of-line-action" . "15-2.html#_778")
   ("encapsulating-stream-stream" . "C-1.html#_1914")
   ("encapsulating-stream-p" . "C-1.html#_1910")
   ("encapsulating-stream" . "C-1.html#_1909") ("enabled" . "7-2.html#_246")
   ("enabled" . "28-5.html#_1542") ("enable-process" . "B-2.html#_1893")
   ("enable-frame" . "28-5.html#_1561") ("elliptical-arc-p" . "3-2.html#_138")
   ("elliptical-arc" . "3-2.html#_137") ("ellipsep" . "3-2.html#_133")
   ("ellipse-start-angle" . "3-2.html#_146") ("ellipse-radii" . "3-2.html#_145")
   ("ellipse-end-angle" . "3-2.html#_147")
   ("ellipse-center-point" . "3-2.html#_144")
   ("ellipse-center-point*" . "3-2.html#_143") ("ellipse" . "3-2.html#_132")
   ("ellipse" . "3-2.html#_129") ("draw-text" . "12-5.html#_636")
   ("draw-text*" . "12-5.html#_637") ("draw-standard-menu" . "25.html#_1357")
   ("draw-rectangles" . "12-5.html#_630") ("draw-rectangles*" . "12-5.html#_631")
   ("draw-rectangle" . "12-5.html#_628") ("draw-rectangle*" . "12-5.html#_629")
   ("draw-polygon" . "12-5.html#_626") ("draw-polygon*" . "12-5.html#_627")
   ("draw-points" . "12-5.html#_620") ("draw-points*" . "12-5.html#_621")
   ("draw-point" . "12-5.html#_618") ("draw-point*" . "12-5.html#_619")
   ("draw-pattern*" . "14-5.html#_749") ("draw-oval" . "12-5.html#_641")
   ("draw-oval*" . "12-5.html#_642") ("draw-lines" . "12-5.html#_624")
   ("draw-lines*" . "12-5.html#_625") ("draw-line" . "12-5.html#_622")
   ("draw-line*" . "12-5.html#_623") ("draw-ellipse" . "12-5.html#_632")
   ("draw-ellipse*" . "12-5.html#_633") ("draw-design" . "14-5.html#_748")
   ("draw-circle" . "12-5.html#_634") ("draw-circle*" . "12-5.html#_635")
   ("draw-arrow" . "12-5.html#_639") ("draw-arrow*" . "12-5.html#_640")
   (":draw" . "29-4.html#_1676") ("dragging-output" . "22-5.html#_1123")
   ("drag-output-record" . "22-5.html#_1122")
   (":drag-callback" . "30-4.html#_1782") (":drag-callback" . "30-4.html#_1807")
   ("drag-callback" . "30-4.html#_1796") ("drag-callback" . "30-4.html#_1817")
   ("document-presentation-translator" . "23-7.html#_1242")
   ("do-command-table-inheritance" . "27-2.html#_1403")
   ("distributing" . "8-1.html#_297") ("distribute-event" . "8-1.html#_306")
   ("displayed-output-record-p" . "16-2.html#_839")
   ("displayed-output-record-ink" . "16-2.html#_857")
   ("displayed-output-record" . "16-2.html#_838")
   (":display-time" . "29-4.html#_1669")
   (":display-function" . "29-4.html#_1668")
   ("display-exit-boxes" . "26.html#_1367")
   ("display-command-table-menu" . "27-3.html#_1421")
   ("display-command-menu" . "28-4.html#_1539") ("dispatches" . "8-1.html#_298")
   ("dispatch-event" . "8-1.html#_307") ("disowned" . "7-2.html#_243")
   ("disowned" . "28-5.html#_1545") ("disown-frame" . "28-5.html#_1557")
   (":disarmed-callback" . "30-3.html#_1718")
   ("disarmed-callback" . "30-3.html#_1726")
   (":disabled-commands" . "28-2.html#_1473") ("disabled" . "7-2.html#_247")
   ("disabled" . "28-5.html#_1543") ("disable-process" . "B-2.html#_1892")
   ("disable-frame" . "28-5.html#_1562") ("dimensionality" . "3-1.html#_42")
   ("device-event" . "8-2.html#_330") ("destroy-process" . "B-2.html#_1881")
   ("destroy-port" . "9-2.html#_471") ("destroy-mirror" . "9-4.html#_493")
   ("destroy-frame" . "28-2.html#_1485") ("designp" . "13-2.html#_684")
   ("design" . "13-2.html#_675") ("design" . "3-1.html#_38")
   ("describe-presentation-type" . "23-3.html#_1163")
   ("describe-presentation-type" . "23-3.html#_1174")
   ("delimiter-gesture-p" . "24-2.html#_1330")
   ("delete-output-record" . "16-2.html#_861")
   ("delete-gesture-name" . "22-3.html#_1102")
   ("delegate-sheet-input-mixin" . "8-1.html#_319")
   ("delegate-sheet-delegate" . "8-1.html#_320") ("degrafted" . "7-2.html#_245")
   ("degraft-medium" . "8-3.html#_425") ("defresource" . "B-1.html#_1872")
   ("defmethod*" . "B-4.html#_1905")
   ("define-presentation-type-abbreviation" . "23-3.html#_1151")
   ("define-presentation-type" . "23-3.html#_1149")
   ("define-presentation-translator" . "23-7.html#_1230")
   ("define-presentation-to-command-translator" . "23-7.html#_1233")
   ("define-presentation-method" . "23-3.html#_1157")
   ("define-presentation-generic-function" . "23-3.html#_1156")
   ("define-presentation-action" . "23-7.html#_1234")
   ("define-graph-type" . "18-2.html#_990")
   ("define-gesture-name" . "22-3.html#_1100")
   ("define-drag-and-drop-translator" . "23-7.html#_1235")
   ("define-default-presentation-method" . "23-3.html#_1158")
   ("define-command-table" . "27-2.html#_1390")
   ("define-command" . "27-1.html#_1380")
   ("define-border-type" . "19.html#_1007")
   ("define-application-frame" . "28-2.html#_1480")
   ("defgeneric*" . "B-4.html#_1904") (":default-view" . "15-2.html#_780")
   ("default-frame-top-level" . "28-4.html#_1534")
   ("default-describe-presentation-type" . "23-3.html#_1189")
   (":decimal-places" . "30-4.html#_1809")
   ("decache-child-output-record" . "21-3.html#_1047")
   ("deallocate-resource" . "B-1.html#_1875")
   ("deallocate-pixmap" . "12-6.html#_646")
   ("deallocate-medium" . "8-3.html#_422") (":cutoff-depth" . "18-2.html#_985")
   ("cursorp" . "15-3.html#_787") ("cursor-visibility" . "15-3.html#_798")
   ("cursor-state" . "15-3.html#_795") ("cursor-sheet" . "15-3.html#_790")
   ("cursor-position" . "15-3.html#_791") ("cursor-focus" . "15-3.html#_797")
   ("cursor-active" . "15-3.html#_793") ("cursor" . "15-3.html#_786")
   (":current-selection" . "30-4.html#_1822")
   (":current-selection" . "30-4.html#_1829")
   ("current-process" . "B-2.html#_1882") ("copy-to-pixmap" . "12-6.html#_650")
   ("copy-textual-output-history" . "16-4.html#_902")
   ("copy-from-pixmap" . "12-6.html#_651") ("copy-area" . "12-6.html#_652")
   ("coordinate" . "3-1.html#_46") ("coordinate" . "3-1.html#_47")
   ("contrasting-inks-limit" . "13-3.html#_705")
   ("contrasting-dash-pattern-limit" . "10-3.html#_561")
   (":contents" . "29-3.html#_1604")
   ("compute-new-output-records" . "21-3.html#_1036")
   ("compute-difference-set" . "21-3.html#_1037")
   ("compositing" . "14-1.html#_727") ("composing" . "5-3.html#_204")
   ("compose-translation-with-transformation" . "5-3.html#_207")
   ("compose-transformations" . "5-3.html#_205")
   ("compose-transformation-with-translation" . "5-3.html#_210")
   ("compose-transformation-with-scaling" . "5-3.html#_211")
   ("compose-transformation-with-rotation" . "5-3.html#_212")
   ("compose-space" . "29-3.html#_1660")
   ("compose-scaling-with-transformation" . "5-3.html#_208")
   ("compose-rotation-with-transformation" . "5-3.html#_209")
   ("compose-over" . "14-1.html#_731") ("compose-out" . "14-1.html#_734")
   ("compose-in" . "14-1.html#_732") ("complex" . "23-8.html#_1263")
   ("completion" . "23-8.html#_1275") ("completion" . "24-5.html#_1342")
   ("completing-from-suggestions" . "24-5.html#_1348")
   ("complete-input" . "24-5.html#_1346")
   ("complete-from-possibilities" . "24-5.html#_1351")
   ("complete-from-generator" . "24-5.html#_1350")
   ("command-table-p" . "27-2.html#_1386")
   ("command-table-not-found" . "27-2.html#_1395")
   ("command-table-name" . "27-2.html#_1388")
   ("command-table-inherit-from" . "27-2.html#_1389")
   ("command-table-error" . "27-2.html#_1394")
   ("command-table-complete-input" . "27-2.html#_1410")
   ("command-table-already-exists" . "27-2.html#_1396")
   ("command-table" . "27-2.html#_1385") (":command-table" . "28-2.html#_1472")
   ("command-present-in-command-table-p" . "27-2.html#_1406")
   ("command-or-form" . "27-6.html#_1455")
   ("command-not-present" . "27-2.html#_1397")
   ("command-not-accessible" . "27-2.html#_1398")
   ("command-name-from-symbol" . "27-2.html#_1402")
   ("command-name" . "27-6.html#_1454") ("command-name" . "27-1.html#_1377")
   ("command-menu-pane" . "29-4.html#_1682")
   ("command-menu-item-value" . "27-3.html#_1419")
   ("command-menu-item-type" . "27-3.html#_1418")
   ("command-menu-item-options" . "27-3.html#_1420")
   ("command-line-read-remaining-arguments-for-partial-command"
    . "27-6.html#_1442")
   ("command-line-name-for-command" . "27-2.html#_1409")
   ("command-line-command-unparser" . "27-6.html#_1441")
   ("command-line-command-parser" . "27-6.html#_1440")
   ("command-enabled" . "28-4.html#_1537")
   ("command-arguments" . "27-1.html#_1378")
   ("command-already-present" . "27-2.html#_1399")
   ("command-accessible-in-command-table-p" . "27-2.html#_1407")
   ("command" . "27-6.html#_1453") ("command" . "27-1.html#_1374")
   ("column-output-record-p" . "17-3.html#_946")
   ("column-output-record" . "17-3.html#_945") ("colorp" . "13-3.html#_687")
   ("colorless" . "13-2.html#_682") ("colored" . "13-2.html#_683")
   ("color-rgb" . "13-3.html#_691") ("color-ihs" . "13-3.html#_692")
   ("color" . "13-3.html#_686") (":clx" . "9-2.html#_458")
   ("closed" . "3-2.html#_83") ("close" . "D-2.html#_1937")
   (":clipping-region" . "10-2.html#_529")
   ("clim-stream-pane" . "29-4.html#_1679") (":client" . "30-3.html#_1716")
   ("client" . "8-1.html#_296") ("client" . "30-2.html#_1708")
   ("clear-resource" . "B-1.html#_1876")
   ("clear-output-record" . "16-2.html#_862")
   ("class-presentation-type-name" . "23-3.html#_1188")
   ("children-overlapping-region" . "7-3.html#_285")
   ("children-overlapping-rectangle*" . "7-3.html#_286")
   ("children" . "6-1.html#_229") ("children" . "16-1.html#_833")
   ("child-containing-position" . "7-3.html#_284")
   ("check-box-selections" . "30-4.html#_1832")
   ("check-box-pane" . "30-4.html#_1834")
   ("check-box-current-selection" . "30-4.html#_1830")
   ("check-box" . "30-4.html#_1828") ("character" . "23-8.html#_1270")
   ("changing-space-requirements" . "29-3.html#_1664")
   ("change-space-requirements" . "29-3.html#_1662")
   (":center-nodes" . "18-2.html#_984")
   ("cell-output-record-p" . "17-3.html#_951")
   ("cell-output-record" . "17-3.html#_950")
   ("cell-min-width" . "17-3.html#_959") ("cell-min-height" . "17-3.html#_960")
   ("cell-align-y" . "17-3.html#_958") ("cell-align-x" . "17-3.html#_957")
   (":calling-frame" . "28-2.html#_1476")
   ("call-presentation-translator" . "23-7.html#_1241")
   ("call-presentation-menu" . "23-7.html#_1243")
   (":cache-value" . "21-3.html#_1028") (":cache-test" . "21-3.html#_1029")
   (":button" . "8-2.html#_343") ("bury-sheet" . "7-2.html#_257")
   ("bury-mirror" . "9-4.html#_495") ("bury-frame" . "28-2.html#_1487")
   ("bounding-rectangle-width" . "4-1.html#_166")
   ("bounding-rectangle-size" . "4-1.html#_168")
   ("bounding-rectangle-position" . "4-1.html#_161")
   ("bounding-rectangle-p" . "4-1.html#_153")
   ("bounding-rectangle-min-y" . "4-1.html#_163")
   ("bounding-rectangle-min-x" . "4-1.html#_162")
   ("bounding-rectangle-max-y" . "4-1.html#_165")
   ("bounding-rectangle-max-x" . "4-1.html#_164")
   ("bounding-rectangle-height" . "4-1.html#_167")
   ("bounding-rectangle" . "4-1.html#_152")
   ("bounding-rectangle" . "4-1.html#_158")
   ("bounding-rectangle*" . "4-1.html#_157") ("bounded" . "13-2.html#_676")
   ("boolean" . "23-8.html#_1256") ("blank-area" . "23-8.html#_1259")
   ("beep" . "15-5.html#_824") ("bboard-pane" . "29-3.html#_1633")
   ("basic-sheet" . "7-1.html#_240") ("basic-port" . "9-2.html#_455")
   ("basic-pane" . "29-2.html#_1587") ("basic-medium" . "8-3.html#_388")
   ("basic-gadget" . "30-3.html#_1714") (":background" . "29-2.html#_1593")
   ("augment-draw-set" . "21-3.html#_1038") ("atomic-incf" . "B-2.html#_1896")
   ("atomic-decf" . "B-2.html#_1897") (":armed-callback" . "30-3.html#_1717")
   ("armed-callback" . "30-3.html#_1725") ("areap" . "3-1.html#_45")
   ("area" . "3-1.html#_44")
   ("apply-presentation-generic-function" . "23-3.html#_1160")
   ("application-pane" . "29-4.html#_1681")
   ("application-frame-p" . "28-2.html#_1469")
   ("application-frame" . "28-2.html#_1468")
   ("applicability" . "23-7.html#_1231") ("anti-aliasing" . "12-4.html#_606")
   ("and" . "23-8.html#_1288") ("allocate-space" . "29-3.html#_1661")
   ("allocate-resource" . "B-1.html#_1874")
   ("allocate-pixmap" . "12-6.html#_645") ("allocate-medium" . "8-3.html#_421")
   ("all-processes" . "B-2.html#_1883") (":align-y" . "29-3.html#_1613")
   (":align-y" . "17-3.html#_953") (":align-y" . "30-3.html#_1749")
   (":align-x" . "29-3.html#_1612") (":align-x" . "17-3.html#_952")
   (":align-x" . "30-3.html#_1748") ("adopted" . "7-2.html#_242")
   ("adopted" . "28-5.html#_1541") ("adopt-frame" . "28-5.html#_1556")
   ("adjust-table-cells" . "17-3.html#_938")
   ("adjust-multiple-columns" . "17-3.html#_939")
   ("adjust-item-list-cells" . "17-3.html#_974")
   ("add-string-output-to-text-record" . "16-3.html#_881")
   ("add-presentation-translator-to-command-table" . "27-5.html#_1432")
   ("add-output-record" . "16-2.html#_860")
   ("add-menu-item-to-command-table" . "27-3.html#_1414")
   ("add-keystroke-to-command-table" . "27-4.html#_1424")
   ("add-input-editor-command" . "24-1.html#_1322")
   ("add-gesture-name" . "22-3.html#_1101")
   ("add-command-to-command-table" . "27-2.html#_1400")
   ("add-character-output-to-text-record" . "16-3.html#_880")
   ("active" . "15-3.html#_784") ("activation-gesture-p" . "24-2.html#_1327")
   ("activate-gadget" . "30-3.html#_1727")
   (":activate-callback" . "30-3.html#_1740")
   ("activate-callback" . "30-3.html#_1742")
   ("action-gadget" . "30-3.html#_1739") ("accessible" . "27-2.html#_1384")
   ("accepting-values" . "26.html#_1364")
   ("accept-values-resynchronize" . "26.html#_1368")
   ("accept-values-command-button" . "26.html#_1369")
   ("accept-values" . "26.html#_1366")
   ("accept-present-default" . "23-3.html#_1168")
   ("accept-from-string" . "23-5.html#_1205") ("accept-1" . "23-5.html#_1204")
   ("accept" . "23-3.html#_1162") ("accept" . "23-5.html#_1202")
   ("accelerator-gesture-numeric-argument" . "22-2.html#_1093")
   ("accelerator-gesture-event" . "22-2.html#_1092")
   ("accelerator-gesture" . "22-2.html#_1091")
   ("abort-gesture-event" . "22-2.html#_1089")
   ("abort-gesture" . "22-2.html#_1088")))
