#
# Copyright (C) 2005  Robert Collins  <robertc@squid-cache.org>
# 
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import tests
import unittest

# if this fails to import, then the config_manager module is bust
# and this test script will never be reached to be imported
import config_manager
import config_manager.implementations as implementations
from config_manager.tests import FunctionalTestCase

import os

class TestConfigEntry(unittest.TestCase):

    def test_imports(self):
        from config_manager import ConfigEntry

    def test_cannot_default_init(self):
        self.assertRaises(TypeError, config_manager.ConfigEntry)

    def test_init_abs(self):
        """We should be able to create an entry with a path and url."""
        entry = config_manager.ConfigEntry("./foo", "svn:///bar")
        self.assertEqual(entry.path, "./foo")
        self.assertEqual(entry.url, "svn:///bar")

    def test_eq(self):
        entry1 = config_manager.ConfigEntry("./foo", "svn:///bar")
        entry2 = config_manager.ConfigEntry("./foo", "svn:///bar")
        self.assertEqual(entry1, entry2)

    def test_get_implementations_fake(self):
        # fake:// is always available
        entry1 = config_manager.ConfigEntry("./foo", "fake:///bar")
        expected_implementations = implementations.schemes["fake"]
        self.assertEqual(expected_implementations, entry1.get_implementations())

    def test_unknown_scheme(self):
        self.failIf("unknownscheme" in implementations.schemes)
        entry1 = config_manager.ConfigEntry("./foo", "unknownscheme:///bar")
        self.assertRaises(config_manager.UnsupportedScheme,
                          entry1.get_implementations)

    def test_mapped_creation(self):
        mapper = config_manager.URLMapper()
        mapper.add_map("svn:///bar", "fake://for-real")
        entry = config_manager.ConfigEntry("./foo", "svn:///bar/gam", mapper)
        self.assertEqual(entry.path, "./foo")
        self.assertEqual(entry.url, "fake://for-real/gam")


class FunctionalConfigEntryTest(FunctionalTestCase):

    def test_build_arch_from_name(self):
        """Test that doing an arch name version based build works."""
        if not self.have_arch():
            return
        entry = config_manager.ConfigEntry("nested", "anarchive@example.com/foo--0")
        entry.build(os.path.join(self.tempdir, "build1"))
        self.assertEqual(open(
            os.path.join(self.tempdir, "build1", "nested", "proof"), "r").read(), 
                         "yo\n")

    def test_build_arch_version(self):
        """Test that doing an arch url version based build works."""
        if not self.have_arch():
            return
        entry = config_manager.ConfigEntry("nested", "arch://anarchive@example.com/foo--0")
        entry.build(os.path.join(self.tempdir, "build1"))
        self.assertEqual(open(
            os.path.join(self.tempdir, "build1", "nested", "proof"), "r").read(), 
                         "yo\n")

    def test_build_arch_revision(self):
        if not self.have_arch():
            return
        entry = config_manager.ConfigEntry("nested", 
            "arch://anarchive@example.com/foo--0--base-0")
        entry.build(os.path.join(self.tempdir, "build1"))
        self.failIf(os.path.exists(
            os.path.join(self.tempdir, "build1", "nested", "proof")))
        
    def test_update_arch_from_name(self):
        if not self.have_arch():
            return
        entry = config_manager.ConfigEntry("nested", 
            "arch://anarchive@example.com/foo--0--base-0")
        entry.build(os.path.join(self.tempdir, "build1"))
        entry = config_manager.ConfigEntry("nested", 
            "anarchive@example.com/foo--0")
        entry.update(os.path.join(self.tempdir, "build1"))
        
    def test_update_arch_version(self):
        if not self.have_arch():
            return
        entry = config_manager.ConfigEntry("nested", 
            "arch://anarchive@example.com/foo--0--base-0")
        entry.build(os.path.join(self.tempdir, "build1"))
        entry = config_manager.ConfigEntry("nested", 
            "arch://anarchive@example.com/foo--0")
        entry.update(os.path.join(self.tempdir, "build1"))
        
    def test_build_fake_version(self):
        """Test doing a fake:// based build works."""
        from config_manager.implementations.fake_vcs import fake_builds
        entry = config_manager.ConfigEntry("nested",
                                           "fake://anarchive@example.com")
        entry.build(os.path.join(self.tempdir, "build1"))
        self.assertEqual(fake_builds, [("fake://anarchive@example.com",
                                        os.path.join(self.tempdir, 
                                                     "build1", "nested"))])
        
    def test_update_fake_version(self):
        """Test doing a fake:// based update works."""
        entry = config_manager.ConfigEntry("nested",
                                           "fake://anarchive@example.com")
        entry.build(os.path.join(self.tempdir, "build1"))
        entry.update(os.path.join(self.tempdir, "build1"))
        from config_manager import fake_updates
        self.assertEqual(fake_updates, [("fake://anarchive@example.com",
                                         os.path.join(self.tempdir, 
                                                     "build1", "nested"))])
        
    def test_update_fake_missing_dir(self):
        """Test doing a fake:// based update falls back to a build."""
        from config_manager import fake_updates
        from config_manager.implementations.fake_vcs import fake_builds
        entry = config_manager.ConfigEntry("nested",
                                           "fake://anarchive@example.com")
        entry.update(os.path.join(self.tempdir, "build1"))
        self.assertEqual(fake_builds, [("fake://anarchive@example.com",
                                        os.path.join(self.tempdir, 
                                                     "build1", "nested"))])
        self.assertEqual(fake_updates, [])

    def test_file_url_with_arch(self):
        """Test building a file url that is actually an arch version."""
        if not self.have_arch():
            return
        basepath = self.tempdir.replace('\\', '/')
        entry = config_manager.ConfigEntry("nested", 
            "file://" + basepath + "/archive/foo--0")
        entry.build(os.path.join(self.tempdir, "build1"))
        self.assertEqual(open(
            os.path.join(self.tempdir, "build1", "nested", "proof"), "r").read(),
                         "yo\n")
        
    def test_file_url_with_baz2(self):
        """Test building a file url that is actually a baz2 branch."""
        basepath = self.tempdir.replace('\\', '/')
        entry = config_manager.ConfigEntry("nested",
            "file://" + basepath + "/bzr")
        entry.build(os.path.join(self.tempdir, "build1"))
        self.assertEqual(open(
            os.path.join(self.tempdir, "build1", "nested", "broom"), "r").read(),
                         "stick\n")

    def test_file_url_update_with_baz2(self):
        basepath = self.tempdir.replace('\\', '/')
        entry = config_manager.ConfigEntry("nested", 
            "file://" + basepath + "/bzr")
        entry.build(os.path.join(self.tempdir, "build1"))
        # FIXME: should update against something else for completeness
        entry.update(os.path.join(self.tempdir, "build1"))
        self.assertEqual(open(
            os.path.join(self.tempdir, "build1", "nested", "broom"), "r").read(), 
                         "stick\n")

    def test_build_absent_path(self):
        """Test that building fails when a path component is missing."""
        if not self.have_arch():
            return
        entry = config_manager.ConfigEntry("missing/nested",
            "arch://anarchive@example.com/foo--0")
        self.assertRaises(KeyError,
                          entry.build,
                          os.path.join(self.tempdir, "build1"))


def test_suite():
    loader = tests.TestLoader()
    result = loader.loadTestsFromName(__name__)
    return result
