/*
 *  docbook-to-html using dancer-XML parser
 *  Copyright (C) 2003 Junichi Uekawa
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


#define _GNU_SOURCE
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <getopt.h>
#include "dancer-xml.h"
#include "config.h"


/** chapter counter array */
int counter_chapter[5]={0,0,0,0,0};
const int c_chapter = 0;
const int c_sect1 = 1;
const int c_sect2 = 2;
const int c_sect3 = 3;


/**
 *   clear some counter
 */
static void 
clear_counter(int min /** the minimum counter number */)
{
  int i;
  for (i=min; i<5; i++)
    counter_chapter[i]=0;
}


static int 
replace_external_entity(const char * s)
{
  if (!strcmp(s,"gt"))
    printf("&gt;");
  else if (!strcmp(s,"lt"))
    printf("&lt;");
  else if (!strcmp(s,"amp"))
    printf("&amp;");
  else
    {
      fprintf(stderr, "unknown entity %s\n", s);
      exit (1);
    }
  return 0;
}


/**
 *   remove exceeding space, and remove special-chars
 */
static int
convert_print (const char * s /** The string to print */,  
	       int verbatim /** preserve carriage returns? */)
{
  const char * endptr = NULL;
  
  if (!s)
    {
      return 1;
    }
  
  while (*s && isspace(*s))
    {
      s++;
    }

  while (*s)
    {
      switch (*s) 
	{
	case '\n':
	  if (verbatim) printf ("<br>");
	  printf ("\n");
	  break;
	case '&':
	  if (NULL!=(endptr = strchr (s, ';')))
	    {
	      char * tmps;
	      tmps = strndup(s+1,endptr-s-1);
	      replace_external_entity(tmps);
	      free(tmps);
	      s=endptr;
	    }
	  else
	    {
	      fprintf(stderr, "Unterminated '&' \n");
	      exit (1);
	    }
	  break;
	default:
	  putc(*s,stdout);
	}
      
      s++;
      
    }
  
  return 0;
}

static const char * getatrstring(dxml_element * e, const char * attr)
{
  dxml_attribute * a = e->element_attribute;
  while (a)
    {
      if (!strcmp(a->attribute_name,attr))
	return a->attribute_data;
      a=a->next;
    }
  return NULL;
}

static int get_paratype(dxml_element *b, int verbatim);
static int get_bookchapter(dxml_element *b, int paraenter);

/**
   Add a label and self-link.

   Needs to be accompanied with maybe_label_end, and the return value of this 
   function needs to be sent there.
   
   @return 0 on no label
   @return 1 if need to terminate the label exists.
 */
static int
maybe_label_start(dxml_element *b)
{
  const char * s;
  if ((s = getatrstring(b, "id")))
    {
      printf ("<a href=\"#%s\" name=\"%s\"><!-- label -->\n", s, s);
      return 1;
    }
  return 0;
}

/**
   ends a label conditionally.
 */
static void
maybe_label_end(int labelt /** whether to require end of label */)
{
  if (labelt)
    printf ("</a>\n");
}


/**
   Process table element
 */
static int
get_table(dxml_element *b_parent)
{
  dxml_element *row, *entry, *b = b_parent -> child;
  int labelt;
  
  
  labelt=maybe_label_start(b_parent);
  printf ("<p class=\"tablecaption\">%s</p>\n<table>\n<tbody>\n",
	  dxml_get_PCDATA_bysimplepath(b,"title"));
  maybe_label_end (labelt);

  for (row = dxml_get_element_bysimplepath(b, "tgroup/tbody/row"); row; row = row->next )
    {
      printf ("<tr>\n");
      for (entry = row->child; entry; entry = entry -> next)
	{
	  printf ("<td>\n");
	  get_paratype(entry->child,0);
	  printf ("</td>\n");
	}
      printf ("</tr>\n");
    }
  printf (
	  "</tbody>\n</table>\n"
	  );
  return 0;
}

/**
   Process itemized list.
 */
static int get_itemizedlist(dxml_element *b)
{
  printf("<ul>\n");
  while (b)
    {
      if (!strcasecmp("listitem", b->element_name))
	{
	  printf("<li>");
	  get_bookchapter(b->child,0);
	  printf("</li>\n");
	}
      else
	dxml_dump_element(b);
      
      b=b->next;
    }
  printf("</ul>\n");
  return 0;
}

/**
   Something that appears inside the para elements and outside the
   para elements.

   @return 1 on match.
   @return 0 on no-match
 */
static int get_bothtypes(dxml_element *b, int inlinetype /** 1 if it is inside para*/)
{
  if (!strcasecmp("screen", b->element_name))
    {
      if (!inlinetype)
	puts("<pre>");	/* enter newline when not inline */
      puts("<span class=\"screen\">");
      get_paratype(b->child, 1);
      puts("</span>");
      if (!inlinetype)
	puts("</pre>");	/* enter newline when not inline */
    }
  else
    return 0;
  return 1;
}


/** 
 * handle para-type ones which can contain PCDATA
 *
 * @return 0 on success
 */
static int
get_paratype(dxml_element *b, 
	     int verbatim/** whether it is in screen env. or not. */)
{
  while (b)
    {
      if (b->element_type == element_type_pcdata)
	{
	  convert_print (b->element_name, verbatim);
	}
      else if (get_bothtypes(b, 1))
	{
	}
      else if ((!strcasecmp("command", b->element_name)) ||
	       (!strcasecmp("filename", b->element_name)) ||
	       (!strcasecmp("keysym", b->element_name)) ||
	       (!strcasecmp("keycombo", b->element_name)) ||
	       (!strcasecmp("otheraddr", b->element_name)) ||
	       (!strcasecmp("prompt", b->element_name)) ||
	       (!strcasecmp("address", b->element_name)) ||
	       (!strcasecmp("option", b->element_name)) ||
	       (!strcasecmp("parameter", b->element_name)))
	{
	  printf ("<span class=\"%s\">", b->element_name);
	  get_paratype(b->child, verbatim);
	  printf ("</span>\n");
	}
      else if (!strcasecmp("footnote", b->element_name))
	{
	  printf ("<span class=\"footnote\">");
	  get_bookchapter(b->child,0);
	  printf ("</span>\n");
	}
      else if (!strcasecmp("itemizedlist", b->element_name))
	{
	  get_itemizedlist(b->child);
	}
      else if ((!strcasecmp("xref", b->element_name))||
	       (!strcasecmp("link", b->element_name)))
	{
	  printf ("<a href=\"#%s\">", 
		  getatrstring(b, "linkend"));
	  get_paratype(b->child, verbatim);
	  printf ("</a>\n");
	}
      else if (!strcasecmp("ulink", b->element_name))
	{
	  printf ("<a href=\"%s\">", 
		  getatrstring(b, "url"));
	  get_paratype(b->child, verbatim);
	  printf ("</a>\n");
	}
      else
	{
	  fprintf(stderr, "Unknown tag: %s\n", b->element_name);
	  dxml_dump_element(b);
	}
      
      
      b=b->next;
    }
  return 0;
}



/**
   Handle chapter and sect* 

   @return 0 on success.
*/
static int
get_bookchapter(dxml_element *b, int paraenter)
{
  int labelt /** whether label was defined. (locally used for maybe_label_start) */;
  
  while (b)
    {
      if (b->element_type != element_type_element)
	{
	  exit (1);
	}
      else if (!strcasecmp("sect1", b->element_name))
	{
	  labelt=maybe_label_start(b);
	  printf ("\n<h3 class=\"sect1\">%i.%i. ", 
		  counter_chapter[c_chapter], 
		  ++counter_chapter[c_sect1]);
	  clear_counter(c_sect1+1);
	  convert_print(dxml_get_PCDATA_bysimplepath(b->child, "title"),0);
	  printf("</h3>\n\n");
	  maybe_label_end(labelt);
	  get_bookchapter(b->child,1);
	}
      else if (!strcasecmp("sect2", b->element_name))
	{
	  labelt=maybe_label_start(b);
	  printf ("\n<h3 class=\"sect2\">%i.%i.%i. ", 
		  counter_chapter[c_chapter], 
		  counter_chapter[c_sect1], 
		  ++counter_chapter[c_sect2]);
	  clear_counter(c_sect2+1);
	  convert_print(dxml_get_PCDATA_bysimplepath(b->child, "title"),0);
	  printf("</h3>\n\n");
	  maybe_label_end(labelt);
	  get_bookchapter(b->child,1);
	}
      else if (!strcasecmp("sect3", b->element_name))
	{
	  labelt=maybe_label_start(b);
	  printf ("\n<h3 class=\"sect3\">%i.%i.%i.%i. ", 
		  counter_chapter[c_chapter], 
		  counter_chapter[c_sect1], 
		  counter_chapter[c_sect2], 
		  ++counter_chapter[c_sect3]);
	  clear_counter(c_sect3+1);
	  convert_print(dxml_get_PCDATA_bysimplepath(b->child, "title"),0);
	  printf("</h3>\n\n");
	  maybe_label_end(labelt);
	  get_bookchapter(b->child,1);
	}
      else if (!strcasecmp("title", b->element_name))
	{
	}
      else if (get_bothtypes(b,0))
	{
	}
      else if (!strcasecmp("para", b->element_name))
	{
	  if (paraenter)puts("<p>\n");
	  get_paratype(b->child, 0);
	  if (paraenter)puts("</p>\n");
	}
      else if (!strcasecmp("itemizedlist", b->element_name))
	{
	  get_itemizedlist(b->child);
	}
      else if (!strcasecmp("table", b->element_name))
	{
	  get_table(b);
	}
      else
	{
	  fprintf(stderr, "Unknown tag: %s\n", b->element_name);
	  dxml_dump_element(b->child);
	}
      
      b=b->next;
    }
  return 0;
}

/**
   Handle book element, and check for each chapter in the book.
 */
static int
recursebook(dxml_element * b)
{
  dxml_element * tmp;
  int labelt = 0;

  for (tmp = dxml_get_element_byname(b, "chapter"); tmp; tmp=tmp->next)
    {
      labelt=maybe_label_start(tmp);
      printf ("\n<h2>%i. ", ++counter_chapter[c_chapter]);
      clear_counter(c_chapter+1);
      convert_print(dxml_get_PCDATA_bysimplepath(tmp->child, "title"),0);
      printf("</h2>\n\n");
      maybe_label_end(labelt);
      get_bookchapter (tmp->child,1);
    }
  return 0;
}


int
main(int argc, char ** argv)
{
  dxml_element * tmp ;
  int c;
  const char * stylesheet = "db2html.css";

  while ((c = getopt(argc, argv, "c:")) != -1)
    {
      switch (c)
	{
	case 'c':
	  stylesheet=optarg;
	  break;
	case '?':
	  return 1;
	default:
	  fprintf(stderr, "Unknown option -%c used\n", c);
	  return 1;
	}
    }

  printf (
	  "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">\n"
	  "<html>\n<head>\n<link rel=\"stylesheet\" type=\"text/css\" href=\"%s\">\n"
	  "<!-- Generated by dxml-db2html %s -->\n",
	  stylesheet,
	  VERSION);

  tmp = dxml_read_xml(stdin);
  if (tmp)
    {
      /* process the document */
      char * title = NULL;
      dxml_element * authorlink;
      

      printf ("<title>");
      title=dxml_get_PCDATA_bysimplepath(tmp,"book/title");
      if (!title)
	title=dxml_get_PCDATA_bysimplepath(tmp,"book/bookinfo/title");

      if (title)
	convert_print(title,0);
      printf ("</title>\n</head>\n<body>\n<h1>\n");
      if (title)
	convert_print(title,0);
      printf ("</h1>\n<p class=\"author\">");

      
      {
	int notfirst = 0;
	for (authorlink = dxml_get_element_bysimplepath(tmp,"book/bookinfo/authorgroup/author");
	     authorlink; 
	     authorlink = dxml_get_element_byname(authorlink->next, "author"))
	  {
	    if (notfirst) 
	      printf(", ");
	    else
	      notfirst = 1;
	    
	    convert_print(dxml_get_PCDATA_bysimplepath(authorlink, "author/firstname"),0);
	    printf(" ");
	    
	    convert_print(dxml_get_PCDATA_bysimplepath(authorlink, "author/surname"),0);
	  }
      }
      
      printf ("</p>\n");

   
      recursebook(dxml_get_element_byname(tmp, "book")->child);
      printf (
	      "</body></html>");
    }
  else
    {
      printf ("Something wicked happened!\n");
      return 1;
    }

  return 0;  
}
