/* $Id: fancy.c,v 1.7 2001/04/22 15:20:38 japh Exp $
 * fancy.c  -- fancy-login module. Uses ncurses to display a colorful login-
 *             screen with input-masks.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Written by Richard Bergmair.
 * Modified by Andreas Krennmair.
 */


#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#include "environment.h"
#include "macros.h"
#include "definitions.h"
#include "limits.h"

#include <stdlib.h>
#include <unistd.h>
#include <netdb.h>
#include <sys/utsname.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <termios.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <dirent.h>

#include "log_message.h"
#include "emergency.h"
#include "conf.h"

#define KEY_BACKSPACE 127
#define KEY_RETURN 10
#define KEY_TAB 9 


/*
 * couldn't find the header
 * again.
 */

int kill(pid_t pid, int sig);


/* 
 * simple imitiations of the ncurses 
 * functions move() and clear() 
 */

#define move(y,x) printf("%c[%d;%df",27,y,x)
#define clear()   printf("%c[2J",27);fflush(stdout)


static struct termios oldtermios;

static int pos_username_x;
static int pos_username_y;
static int pos_password_x;
static int pos_password_y;
static int len_username;
static int len_password;
static int color_username_fg;
static int color_username_bg;
static int color_password_fg;
static int color_password_bg;
static char normal_ansi_file[1024];
static char failed_ansi_file[1024];

static void set_pwfield_attrs(void);
static void set_userfield_attrs(void);
static void reset_field_attrs(void);
static void init_screen(void);
static int process_config(void);
static int draw_ansifile(char *ansifile);
static int draw_logon(void);
static int inputmask(int ypos, int xpos, const int maxchars, char *getback,
                     char mask, char emptyc);
static char readkey(void);
int initialize_prompt(void);
void fancy_prompt(char *user, char *password);
int close_prompt(void);
int draw_faillogon(void);





/***********************************************************************/
/* set_pwfield_attrs: sets the attributes for the password-mask from   */ 
/*                    the themefile.                                   */
/***********************************************************************/

static void 
set_pwfield_attrs(void)
{
  printf("%c[%d;%dm",27,color_password_fg,color_password_bg);
}



/***********************************************************************/
/* set_userfield_attrs: sets the attributes for the username-mask from */
/*                      the themefile.                                 */
/***********************************************************************/

static void 
set_userfield_attrs(void)
{
  printf("%c[%d;%dm",27,color_username_fg,color_username_bg);
}



/************************************************************************/
/* reset_field_colors: resets the colors to the terminal's default      */
/************************************************************************/

static void 
reset_field_attrs(void)
{
  printf("%c[m",27);
}



/*************************************************************************/
/* init_screen: initializes the screen to show special characters, e.g.  */
/*              card symbols, arrows, the famous framebars, etc.         */
/*************************************************************************/

static void 
init_screen(void)
{
  printf("\033[m\033]R\033]R\033[H\033[J\033[0;10;1;11m");
}



/*****************************************************************************/
/* process_config: processes a theme file                                    */
/*****************************************************************************/

static int
process_config(void)
{
  int have_theme_fl = 0;
  char line[__MAX_STR_LEN__];
  char normal_file[256];
  char failed_file[256];
  struct dirent * dir;
  char * theme_dir = get_theme_dir();
  char * theme_config_file = NULL;
  DIR * xdir = opendir(theme_dir);
  FILE * f;
  if (errno)
    return 0;
  while ((dir=readdir(xdir))!=NULL)
    {
       /* 
        * if we seem to have a theme directory,
        * we stop searching and confirm that we found a theme.fl
        */
       if (strcmp(dir->d_name,"theme.fl")==0)
         {
           have_theme_fl = 1;
           break;
         }
    }
  /**
   not having found a valid theme means aborting. 
   */
  if (have_theme_fl==0)
    return 0;
  
  theme_config_file = malloc(strlen(theme_dir)+strlen("/theme.fl")+1);

  if (theme_config_file==NULL)
    return 0;

  /* concatenating all parts to get the whole file name of the theme.fl */
  strcpy(theme_config_file,theme_dir);
  strcat(theme_config_file,"/theme.fl");

  if ((f=fopen(theme_config_file,"rt"))==NULL)
    return 0; /* just to be sure */

  while (fgets(line,sizeof line, f) !=NULL)
    {
      if (line[0]!='#' && line[0]!=0 && line[0]!=10)
        {
          switch (line[0])
            {
             /* username */
             case 'u':
               sscanf(line+8," %d %d %d %d %d",&pos_username_x,
                                               &pos_username_y,
                                               &len_username,
                                               &color_username_fg,
                                               &color_username_bg);
               break;
             /* password */
             case 'p':
               sscanf(line+8," %d %d %d %d %d",&pos_password_x,
                                               &pos_password_y,
                                               &len_password,
                                               &color_password_fg,
                                               &color_password_bg);
               break;
             /* files */
             case 'f':
               sscanf(line+5," %s %s", normal_file, failed_file);

               strcpy(normal_ansi_file,theme_dir);
               strcat(normal_ansi_file,"/");
               strcat(normal_ansi_file,normal_file);
               
               strcpy(failed_ansi_file,theme_dir);
               strcat(failed_ansi_file,"/");
               strcat(failed_ansi_file,failed_file);
               
               break;
            }
        }
    }
  closedir(xdir);
  fclose(f);
  return 1;
}



/*****************************************************************************/
/* close_prompt: stuff to do when terminating                                */
/*****************************************************************************/

int
close_prompt (void)
{
  /* set the new attributes */
  if (tcsetattr(0,TCSADRAIN,&oldtermios))
    return -1;

  setlinebuf(stdout); 
  fflush(stdout);
  printf("%cc", 27);
  fflush(stdout);
  return 0;
}



/*****************************************************************************/
/* draw_ansiscreen: draws the logon-screen, defined in an ansi file.         */
/*****************************************************************************/

static int
draw_ansifile (char * ansifile)
{
  char c;             /* color-character-buffer */
  FILE * f;
  struct utsname info;
  char * termname;
  
  /* 
   * we try to get the hostname and the terminal-name; 
   * if this fails, the hostname becomes an empty
   * string and the terminal-name is simply `tty'.            
   */

  termname = malloc(strlen(ttyname(0))-5+1);
  
  if (termname==NULL)
    termname = "tty";
  else 
    sscanf (ttyname(0), "/dev/%s", termname);
  uname(&info);

  init_screen();

  clear();
  move (0, 0);

  f = fopen(ansifile,"rt");

  if (f==NULL)
    return 0;
  
  while ((c=fgetc(f))!=EOF)  /* for every character you find in */
    {                            /*  the themefile...*/  
      if (c=='%')
        {
          int i;
          int k;
          int isopen=TRUE;
          c=fgetc(f);
          switch (c)
            {
              case 'h': put_out(info.nodename)
              case 't': put_out(termname)
              case 's': put_out(info.sysname)
              case 'm': put_out(info.machine)
            }
        }
      else
        putchar(c);   
    } /* while */


  fflush(stdout);
  fclose(f); 
  return 1;
}



/*****************************************************************************/
/* draw_logon: paint the login-screen                                        */
/*****************************************************************************/

static int
draw_logon(void)
{
  if(0==draw_ansifile(normal_ansi_file))
    {
      move(0,0);
      printf("Error: couldn't write file %s to screen. Please check your"
             " current theme for errors.\n",normal_ansi_file);
    }
  return 1;
}



/*****************************************************************************/
/* draw_faillogin: draw the failed-login-screen, and wait for the user to    */
/*                 press a key.                                              */
/*****************************************************************************/

int
draw_faillogon(void)
{
  int x;
  if (0==draw_ansifile(failed_ansi_file))
    {
      move(0,0);
      printf("Error: couldn't write file %s to screen. Please check your"
             " current theme for errors.\n",failed_ansi_file);
    }
  x=getchar();
  return 1;
}



/*****************************************************************************/
/* readkey: read a key from stdin, that is >= 0x20                           */
/*****************************************************************************/

static char
readkey(void)
{
  char c;

  do
    c=getchar();
  while (!((c>=0x20) ||
           (c==KEY_BACKSPACE) ||
           (c==KEY_RETURN) ||
           (c==KEY_TAB)));

  return c;
}



/*****************************************************************************/
/* inputmask: draws an input field at y, x which can take up to maxchars     */
/*            characters, returns the result in char *getback, uses char mask*/
/*            to display an input-mask for example for password-input. If you*/
/*            don't want a mask, set mask 0. char emptyc defines what        */
/*            character to use, if nothing is displayed, usually ' '.        */
/*            colp defines which color-pair to use.                          */
/*            returns 0 if the user terminates the input with <RETURN> and   */
/*            1 if the user terminates with <TAB>.                           */
/*****************************************************************************/

static int
inputmask(int ypos, int xpos, const int maxchars, char *getback,
          char mask, char emptyc)
{
  int x=0;     /* Indexcounter for the current editing position */
  int c;       /* Character-buffer */
  int i;       /* counter for several purposes */

  /* clear the part of the screen we'll need */
  move(ypos, xpos);
  for (i=0;i<maxchars;i++)
    putchar (emptyc);
  move(ypos, xpos);
  fflush (stdout);

  /* initialize the string to return */
  for (i=0;i<=maxchars;i++)
    getback[i]=0;

  for (;;)                                     /* forever do */
    {
      int i;
      x++;                                     /* increment indexcounter */
      c=readkey();                             /* get character from input */
      if ((c == KEY_RETURN) || (c == KEY_TAB))
        break;                                 /* break if <RETURN> or <TAB> */

      switch (c)
        {
          case KEY_BACKSPACE:
            /*
             * the user hit <BACKSPACE>
             */
            if (x>1)                           /* are there characters left */
              {                                /* to delete? */
                x--;                           /* yes-> decrement counter */
                move(ypos, xpos+x-1);          /* move one position back */
                getback[x-1]=0;                /* delete char from string */
                for (i=x;i<=maxchars;i++)
                  putchar(emptyc);             /* overwrite the ch on screen */
                move(ypos, xpos+x-1);          /* and go back again */
                fflush (stdout);               /* show everything */
              }
            x--;                               /* decrement counter */
            break;

          case 3:
            kill(getpid(), SIGINT); 

          default:
            if (x <= maxchars)                /* is there space left? */
              {                               /* yes-> */
                move(ypos, xpos+x-1);         /* go back one character */
                if (mask==0)                  /* do we need masks? */
                  putchar(c);                 /* no:write the char from inp.*/
                else
                  putchar(mask);              /* yes: write the mask-char */
                getback[x-1]=(char)c;         /* add the char to the str. */
                fflush(stdout);               /* show everything */
              }
            else
              x--;                            /* no: decrement counter */
            break;
        } /* switch */
    } /* for */

  if (c==10) return 0;                 /* return 0 when exited with <RETURN> */
  return 1;                            /* return 1 if not so (<TAB>) */
}



/*****************************************************************************/
/* initialize_prompt: Initializes everything                                 */
/*****************************************************************************/

int
initialize_prompt (void)
{
  struct termios tio;

  /* 
   load the configuration, and, if anything fails
   abort by calling fatal(). 
   */
  if (process_config()==0)
    fatal();

  /* get the current terminal attributes */
  if (tcgetattr(0,&tio))
    return -1;

  oldtermios=tio;

  tio.c_lflag &= ICANON;
  tio.c_lflag &= ECHO;

  /* set the new attributes */
  if (tcsetattr(0,TCSADRAIN,&tio))
    return -1;

  return 0;
}



/******************************************************************************/
/* main-function                                                              */
/******************************************************************************/

void
fancy_prompt (char *user, char *password)
{
  int rc;
  do 
    {
      process_config ();
      draw_logon ();
      set_userfield_attrs();
      inputmask (pos_username_y, pos_username_x, len_username, user, 0, ' ');
      call_after_username_hook();
      set_pwfield_attrs();
      rc = inputmask(pos_password_y,pos_password_x,len_password,
                     password,'*',' ');
      call_after_password_hook();              
    }
  while (rc);

  reset_field_attrs();
}



/*****************************************************************************/
/* (c) Copyright 1999-2000 Richard Bergmair, 2000-2001 Andreas Krennmair     */
/*****************************************************************************/
