/*
 * Copyright (C) 2019-2023 Intel Corporation
 *
 * SPDX-License-Identifier: MIT
 *
 */

#include "shared/source/compiler_interface/compiler_cache.h"

#include "shared/source/debug_settings/debug_settings_manager.h"
#include "shared/source/helpers/aligned_memory.h"
#include "shared/source/helpers/casts.h"
#include "shared/source/helpers/file_io.h"
#include "shared/source/helpers/hash.h"
#include "shared/source/helpers/hw_info.h"
#include "shared/source/utilities/debug_settings_reader.h"
#include "shared/source/utilities/io_functions.h"

#include "config.h"
#include "os_inc.h"

#include <cstring>
#include <iomanip>
#include <mutex>
#include <sstream>
#include <string>

namespace NEO {
std::mutex CompilerCache::cacheAccessMtx;

const std::string CompilerCache::getCachedFileName(const HardwareInfo &hwInfo, const ArrayRef<const char> input,
                                                   const ArrayRef<const char> options, const ArrayRef<const char> internalOptions,
                                                   const ArrayRef<const char> specIds, const ArrayRef<const char> specValues,
                                                   const ArrayRef<const char> igcRevision, size_t igcLibSize, time_t igcLibMTime) {
    Hash hash;

    hash.update("----", 4);
    hash.update(&*igcRevision.begin(), igcRevision.size());
    hash.update(safePodCast<const char *>(&igcLibSize), sizeof(igcLibSize));
    hash.update(safePodCast<const char *>(&igcLibMTime), sizeof(igcLibMTime));

    hash.update("----", 4);
    hash.update(&*input.begin(), input.size());
    hash.update("----", 4);
    hash.update(&*options.begin(), options.size());
    hash.update("----", 4);
    hash.update(&*internalOptions.begin(), internalOptions.size());
    hash.update("----", 4);
    hash.update(&*specIds.begin(), specIds.size());
    hash.update(&*specValues.begin(), specValues.size());
    hash.update("----", 4);
    hash.update(safePodCast<const char *>(&hwInfo.platform), sizeof(hwInfo.platform));
    hash.update("----", 4);

    const auto featureTableHashStr = std::to_string(hwInfo.featureTable.asHash());
    hash.update(featureTableHashStr.c_str(), featureTableHashStr.length());
    hash.update("----", 4);

    const auto workaroundTableHashStr = std::to_string(hwInfo.workaroundTable.asHash());
    hash.update(workaroundTableHashStr.c_str(), workaroundTableHashStr.length());

    auto res = hash.finish();
    std::stringstream stream;
    stream << std::setfill('0')
           << std::setw(sizeof(res) * 2)
           << std::hex
           << res;

    if (DebugManager.flags.BinaryCacheTrace.get()) {
        std::string traceFilePath = config.cacheDir + PATH_SEPARATOR + stream.str() + ".trace";
        std::string inputFilePath = config.cacheDir + PATH_SEPARATOR + stream.str() + ".input";
        std::lock_guard<std::mutex> lock(cacheAccessMtx);
        auto fp = NEO::IoFunctions::fopenPtr(traceFilePath.c_str(), "w");
        if (fp) {
            NEO::IoFunctions::fprintf(fp, "---- igcRevision ----\n");
            NEO::IoFunctions::fprintf(fp, "%s\n", &*igcRevision.begin());
            NEO::IoFunctions::fprintf(fp, "  libSize=%llu\n", igcLibSize);
            NEO::IoFunctions::fprintf(fp, "  libMTime=%llu\n", igcLibMTime);
            NEO::IoFunctions::fprintf(fp, "---- input ----\n");
            NEO::IoFunctions::fprintf(fp, "<%s>\n", inputFilePath.c_str());
            NEO::IoFunctions::fprintf(fp, "---- options ----\n");
            NEO::IoFunctions::fprintf(fp, "%s\n", &*options.begin());
            NEO::IoFunctions::fprintf(fp, "---- internal options ----\n");
            NEO::IoFunctions::fprintf(fp, "%s\n", &*internalOptions.begin());
            NEO::IoFunctions::fprintf(fp, "---- specialization constants ----\n");
            NEO::IoFunctions::fprintf(fp, "specIds=");
            for (size_t idx = 0; idx < specIds.size(); idx++) {
                NEO::IoFunctions::fprintf(fp, "%02x", specIds[idx]);
            }
            NEO::IoFunctions::fprintf(fp, "\n");
            NEO::IoFunctions::fprintf(fp, "specValues=");
            for (size_t idx = 0; idx < specValues.size(); idx++) {
                NEO::IoFunctions::fprintf(fp, "%02x", specValues[idx]);
            }
            NEO::IoFunctions::fprintf(fp, "\n");

            NEO::IoFunctions::fprintf(fp, "---- platform ----\n");
            NEO::IoFunctions::fprintf(fp, "  eProductFamily=%d\n", hwInfo.platform.eProductFamily);
            NEO::IoFunctions::fprintf(fp, "  ePCHProductFamily=%d\n", hwInfo.platform.ePCHProductFamily);
            NEO::IoFunctions::fprintf(fp, "  eDisplayCoreFamily=%d\n", hwInfo.platform.eDisplayCoreFamily);
            NEO::IoFunctions::fprintf(fp, "  eRenderCoreFamily=%d\n", hwInfo.platform.eRenderCoreFamily);
            NEO::IoFunctions::fprintf(fp, "  ePlatformType=%d\n", hwInfo.platform.ePlatformType);
            NEO::IoFunctions::fprintf(fp, "  usDeviceID=%d\n", hwInfo.platform.usDeviceID);
            NEO::IoFunctions::fprintf(fp, "  usRevId=%d\n", hwInfo.platform.usRevId);
            NEO::IoFunctions::fprintf(fp, "  usDeviceID_PCH=%d\n", hwInfo.platform.usDeviceID_PCH);
            NEO::IoFunctions::fprintf(fp, "  usRevId_PCH=%d\n", hwInfo.platform.usRevId_PCH);
            NEO::IoFunctions::fprintf(fp, "  eGTType=%d\n", hwInfo.platform.eGTType);

            NEO::IoFunctions::fprintf(fp, "---- feature table ----\n");
            auto featureTable = safePodCast<const char *>(&hwInfo.featureTable.packed);
            for (size_t idx = 0; idx < sizeof(hwInfo.featureTable.packed); idx++) {
                NEO::IoFunctions::fprintf(fp, "%02x.", (uint8_t)(featureTable[idx]));
            }
            NEO::IoFunctions::fprintf(fp, "\n");

            NEO::IoFunctions::fprintf(fp, "---- workaround table ----\n");
            auto workaroundTable = reinterpret_cast<const char *>(&hwInfo.workaroundTable);
            for (size_t idx = 0; idx < sizeof(hwInfo.workaroundTable); idx++) {
                NEO::IoFunctions::fprintf(fp, "%02x.", (uint8_t)(workaroundTable[idx]));
            }
            NEO::IoFunctions::fprintf(fp, "\n");

            NEO::IoFunctions::fclosePtr(fp);
        }
        fp = NEO::IoFunctions::fopenPtr(inputFilePath.c_str(), "w");
        if (fp) {
            NEO::IoFunctions::fwritePtr(&*input.begin(), input.size(), 1, fp);
            NEO::IoFunctions::fclosePtr(fp);
        }
    }

    return stream.str();
}

CompilerCache::CompilerCache(const CompilerCacheConfig &cacheConfig)
    : config(cacheConfig){};

} // namespace NEO
