package latexDraw.figures;

import java.awt.Color;
import java.awt.Point;
import java.io.IOException;
import java.io.ObjectInputStream;

import latexDraw.util.LaTeXDrawPoint2D;


/**
 * A DrawBorders allows the user to define the position of the origin
 * of his drawing (the point (0,0)) and to define too the dimensions
 * (the borders) of his drawing.<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE.  See the GNU General Public License for more details.<br>
 *<br>
 * 01/20/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>>
 */
public class DrawBorders extends LaTeXDrawRectangle
{
	private static final long serialVersionUID = 1L;

	/** Correspond to the origin point (0,0) */
	private LaTeXDrawPoint2D originPoint;
	
	/** The colour of the rectangle */
	public static final Color color = Color.DARK_GRAY;
	
	/** 
	 * The constructor by default 
	 */
	public DrawBorders(boolean increaseMeter)
	{
		this(new LaTeXDrawPoint2D(), new LaTeXDrawPoint2D(), false, false, increaseMeter);
	}



	/**
	 * The constructor using 2 points
	 * @param NWpt The North-West point of the rectangle
	 * @param SEpt The South-East point of the rectangle
	 */
	public DrawBorders(LaTeXDrawPoint2D NWpt, LaTeXDrawPoint2D SEpt, boolean autoAdjust,
			boolean inSelection, boolean increaseMeter)
	{

		super(NWpt,new LaTeXDrawPoint2D(SEpt.x, NWpt.y), new LaTeXDrawPoint2D(NWpt.x, SEpt.y), SEpt, increaseMeter);
	
		isBordersMovable = false;
		isDashableOrDotable = false;
		canHaveShadow = false;
		isDoubleBoundaryable = false;
		originPoint = new LaTeXDrawPoint2D(NWpt.x, (NWpt.y+SEpt.y)/2);
		isSelected = autoAdjust && inSelection;
		setLinesColor(color);
	}
	
	

	
	/**
	 * Allows to get the bottom left point of the drawing (South-West point)
	 * @return The bottom left point of the drawing
	 */
	public LaTeXDrawPoint2D getTheSWPoint()
	{
		LaTeXDrawPoint2D p1 = getPoint(0), p2 = getPoint(1);
		LaTeXDrawPoint2D p3 = getPoint(2), p4 = getPoint(3);

		if(p1.x<p2.x)
		{
			if(p1.y>p3.y)
				 return p1;
			return p3;
		}
		
		if(p2.y>p4.y)
			 return p2;
		return p4;
	}
	
	
	
	
	/**
	 * Allows to get the top right point of the drawing (North-East point)
	 * @return The top right point of the drawing
	 */	
	public LaTeXDrawPoint2D getTheNEPoint()
	{
		LaTeXDrawPoint2D p1 = getPoint(0), p2 = getPoint(1);
		LaTeXDrawPoint2D p3 = getPoint(2), p4 = getPoint(3);

		if(p1.x>p2.x)
		{
			if(p1.y<p3.y)
				 return p1;
			return p3;
		}
		
		if(p2.y<p4.y)
			 return p2;
		return p4;
	}
	
	
	
	
	/**
	 * Allows to get the origin point (the reference point of axes)
	 * @return The origin point
	 */
	public LaTeXDrawPoint2D getOriginPoint()
	{
		return originPoint;
	}
	
	

	@Override
	public void onDragged(Point formerPt, Point newPt) 
	{
		super.onDragged(formerPt, newPt);
		updateOriginPoint();
	}
	
	
	
	
	@Override
	public synchronized String getCodePSTricks(DrawBorders drawBorders, float ppc)
	{
		return null;
	}



	@Override
	public synchronized void onRelease() 
	{		
		System.out.println("DrawBorders : empty method !");//$NON-NLS-1$
	}



	@Override
	public void shift(double shiftX, double shiftY) 
	{
		if(shiftX==0 && shiftY==0) return ;
		super.shift(shiftX, shiftY);
		updateOriginPoint();
	}



	@Override
	public synchronized void setLastPoint(LaTeXDrawPoint2D pt)
	{
		super.setLastPoint(pt);
		updateOriginPoint();
	}
	
	

	
	@Override
	public synchronized void setFirstPoint(LaTeXDrawPoint2D pt)
	{
		super.setFirstPoint(pt);
		updateOriginPoint();
	}
	
	
	
	
	public void updateOriginPoint()
	{
		LaTeXDrawPoint2D NW = getTheNWPoint(), SE = getTheSEPoint();
		
		originPoint.x = NW.x;
		originPoint.y = (NW.y+SE.y)/2;
	}



	@Override
	public Object clone() throws CloneNotSupportedException
	{
		DrawBorders d = (DrawBorders) super.clone();
		if(borders!=null)
			d.borders = (LaTeXDrawRectangle) borders.clone();
		d.originPoint = (LaTeXDrawPoint2D) originPoint.clone();
		
		return d;
	}



	@Override
	public synchronized void updateGravityCenter() 
	{
		/* This function is disable. */
	}



	@Override
	public synchronized void updateStyleOfDelimitors() 
	{
		/* This function is disable : no style for delimiters. */
	}
	
	
	
	
	private void readObject(ObjectInputStream ois) throws IOException, ClassNotFoundException
	{
		isSelected = ois.readBoolean();
		borders = (LaTeXDrawRectangle) ois.readObject();
		originPoint = (LaTeXDrawPoint2D) ois.readObject();

		updateGravityCenter();
		updateStyleOfDelimitors();
	}
}
