package latexDraw.generators.svg;

import java.awt.geom.Point2D;
import java.util.Vector;

import latexDraw.figures.DrawBorders;
import latexDraw.figures.Figure;
import latexDraw.figures.FramedBox;
import latexDraw.figures.Text;
import latexDraw.parsers.latexdraw.LPSTParser;
import latexDraw.parsers.svg.CSSColors;
import latexDraw.parsers.svg.SVGAttributes;
import latexDraw.parsers.svg.SVGDocument;
import latexDraw.parsers.svg.SVGElements;
import latexDraw.parsers.svg.elements.SVGElement;
import latexDraw.parsers.svg.elements.SVGGElement;
import latexDraw.parsers.svg.elements.SVGTextElement;
import latexDraw.psTricks.DviPsColors;
import latexDraw.util.LaTeXDrawNamespace;
import latexDraw.util.LaTeXDrawPoint2D;

import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * Defines a SVG generator for a text.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 11/11/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.1<br>
 */
public class LTextSVGGenerator extends LShapeSVGGenerator
{

	public LTextSVGGenerator(Text f)
	{
		super(f);
	}

	
	
	/**
	 * Creates a text from an SVG text element.
	 * @param elt The source element.
	 * @since 2.0.0
	 */
	public LTextSVGGenerator(SVGTextElement elt)
	{
		this(new Text(null, false));
		
		if(elt==null)
			throw new IllegalArgumentException();
		
		String txt = elt.getText();
		
		if(txt==null || txt.length()==0)
			throw new IllegalArgumentException("This text is empty.");
		
		Text t = (Text)getShape();
		
		t.setText(txt);
		t.setPosition(new LaTeXDrawPoint2D(elt.getX(), elt.getY()));
		setTextAttributes(elt);
		applyTransformations(elt);
	}
	
	
	
	public LTextSVGGenerator(SVGGElement elt)
	{
		this(elt, true);
	}
	
	
	
	/**
	 * Creates a text from a latexdraw-SVG element.
	 * @param elt The source element.
	 * @since 2.0.0
	 */
	public LTextSVGGenerator(SVGGElement elt, boolean withTransformation)
	{
		this(new Text(null, false));

		if(elt==null)
			throw new IllegalArgumentException();
		
		double x, y = 0;
		String v;
		NodeList nl;
		String fontFam = elt.getAttribute(SVGAttributes.SVG_FONT_FAMILY);
		int fontSize; 
		Text t;
		
		try { fontSize = Double.valueOf(elt.getAttribute(SVGAttributes.SVG_FONT_SIZE)).intValue(); }
		catch(NumberFormatException e) { fontSize = Text.DEFAULT_SIZE.getSize(); }
		
		setNumber(elt);
		setTextAttributes(elt);
		v = elt.getAttribute(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE_URI+':'+SVGAttributes.SVG_X);
		x = v==null ? Double.MAX_VALUE : Double.valueOf(v).doubleValue();
		
		nl = elt.getElementsByTagNameNS(SVGDocument.SVG_NAMESPACE, SVGElements.SVG_TEXT);
		
		if(nl.getLength()>0)
		{
			SVGTextElement text = (SVGTextElement)nl.item(0);
			LPSTParser parser   = new LPSTParser("\\put(0,0){" + text.getText() + "}");
			Vector<Figure> shapes;
			
			try{ parser.parseFramedBox(); }
			catch(Exception e) {e.printStackTrace(); }
			shapes = parser.getShapes();
			
			if(shapes==null || shapes.isEmpty() || !(shapes.firstElement() instanceof Text))
				throw new IllegalArgumentException();
			
			t = (Text)shapes.firstElement();
			y = text.getY();
			x = text.getX()<x ? text.getX() : x;
			t.setPosition(new LaTeXDrawPoint2D(x, y));
			
			shape = t;
		}
		else
			throw new IllegalArgumentException();

		if(t.hasSimpleFramedBox())
		{
			FramedBox fb  = t.getSimpleBox();
			Figure figure = fb.getBox();
			
			if(((float)fb.getFrameSep())==0f && figure.getLinesColor().equals(figure.getInteriorColor()) &&
					figure.isFilled() && fb.getBoxType()==FramedBox.BOX_RECTANGLE)
			{
				t.setOpacityColor(figure.getLinesColor());
				t.removeAllBoxes();
				t.setOpaque(true);
				
			}
		}
		
		t = (Text)shape;
		t.setSize(fontSize);
		t.setTextFont(fontFam);
		t.setIsBold(SVGAttributes.SVG_FONT_WEIGHT_BOLD.equals(elt.getAttribute(SVGAttributes.SVG_FONT_WEIGHT)));
		t.setIsItalic(SVGAttributes.SVG_FONT_STYLE_ITALIC.equals(elt.getAttribute(SVGAttributes.SVG_FONT_STYLE)));
		
		if(withTransformation)
			applyTransformations(elt);
	}
	
	
	
	
	/**
	 * Sets the text attribute (is italic, font family,...) from the given SVG element.
	 * @param e The source SVG element.
	 * @since 2.0.0
	 */
	public void setTextAttributes(SVGElement e)
	{
		if(e==null)
			return ;
		
		Text t 		   = (Text)getShape();
		float fontSize = e.getFontSize();
		String fam	   = e.getFontFamily();
		String style   = e.getFontStyle();
		String weight  = e.getFontWeight();
		
		if(fontSize>0)
			t.setSize((int)fontSize);

		if(fam.length()>0)
			t.setTextFontByFamily(fam);
		
		t.setLinesColor(CSSColors.getRGBColour(e.getFill()));
		t.setIsItalic(style.equals(SVGAttributes.SVG_FONT_STYLE_ITALIC) || 
					  style.equals(SVGAttributes.SVG_FONT_STYLE_OBLIQUE));
		t.setIsBold(weight.equals(SVGAttributes.SVG_FONT_WEIGHT_BOLD));
	}
	
	
	
	
	/**
	 * Creates an SVGTextElement from the given part of text.
	 * @param txt The text to convert in SVG.
	 * @param doc The document to create the text element.
	 * @param position The position of the text.
	 * @return The SVG text element or null if a parameter is null.
	 * @since 2.0.0
	 */
	public SVGTextElement getSVGTextElement(String txt, SVGDocument doc, Point2D position)
	{
		if(txt==null || position==null || doc==null)
			return null;
		
		SVGTextElement textElt = new SVGTextElement(doc);
		
		textElt.setAttribute(SVGAttributes.SVG_X, String.valueOf(position.getX()));
		textElt.setAttribute(SVGAttributes.SVG_Y, String.valueOf(position.getY()));
		textElt.appendChild(doc.createCDATASection(txt));
		
		return textElt;
	}
	
	
	
	
	@Override
	public SVGElement toSVG(SVGDocument doc)
	{
		if(doc == null)
			return null;

		SVGElement root = new SVGGElement(doc);
		Text t = (Text)getShape();
		String ltdPref = LaTeXDrawNamespace.LATEXDRAW_NAMESPACE + ':';
		
		root.setAttribute(ltdPref + LaTeXDrawNamespace.XML_TYPE, LaTeXDrawNamespace.XML_TYPE_TEXT);
		root.setAttribute(SVGAttributes.SVG_ID, getSVGID());
		root.setAttribute(SVGAttributes.SVG_FONT_FAMILY, String.valueOf(t.getCurrentFont().getFamily()));
		root.setAttribute(SVGAttributes.SVG_FILL, CSSColors.getColorName(t.getLinesColor(), true));
		root.setAttribute(SVGAttributes.SVG_FONT_SIZE, String.valueOf(t.getCurrentFont().getSize()));
		root.setAttribute(ltdPref + SVGAttributes.SVG_X, String.valueOf(t.getPosition().x));
		
		if(t.isItalic())
			root.setAttribute(SVGAttributes.SVG_FONT_STYLE, SVGAttributes.SVG_FONT_STYLE_ITALIC);
		
		if(t.isBold())
			root.setAttribute(SVGAttributes.SVG_FONT_WEIGHT, SVGAttributes.SVG_FONT_WEIGHT_BOLD);
		
		Element txt = new SVGTextElement(doc);
		String str  = t.getCodePSTricksBody(new DrawBorders(false), Figure.PPC);
		String cols = DviPsColors.getUserColoursCode(str);
		
		if(cols!=null)
			cols = cols.replace(System.getProperty("line.separator"), "");
		
		txt.setAttribute(SVGAttributes.SVG_X, String.valueOf(t.getPosition().x));
		txt.setAttribute(SVGAttributes.SVG_Y, String.valueOf(t.getPosition().y));
		txt.appendChild(doc.createCDATASection(cols+str)); 
		root.appendChild(txt);
		
		setSVGRotationAttribute(root);
		
		return root;
	}
}
