/**
 * Copyright (c) 2022, Timothy Stack
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * * Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * * Neither the name of Timothy Stack nor the names of its contributors
 * may be used to endorse or promote products derived from this software
 * without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ''AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <iostream>

#include "attr_line.hh"

#include "config.h"
#include "doctest/doctest.h"

using namespace lnav::roles::literals;

TEST_CASE("line_range")
{
    line_range lr1{0, 95};
    line_range lr2{0, -1};

    CHECK(lr2 < lr1);
    CHECK(!(lr1 < lr2));
}

TEST_CASE("attr_line_t::basic-wrapping")
{
    text_wrap_settings tws = {3, 21};
    attr_line_t to_be_wrapped{"This line, right here, needs to be wrapped."};
    attr_line_t dst;

    to_be_wrapped.al_attrs.emplace_back(
        line_range{0, (int) to_be_wrapped.al_string.length()},
        VC_ROLE.value(role_t::VCR_ERROR));
    dst.append(to_be_wrapped, &tws);

    CHECK(dst.get_string() ==
          "This line, right\n"
          "   here, needs to be\n"
          "   wrapped.");

    for (const auto& attr : dst.al_attrs) {
        printf("attr %d:%d %s\n",
               attr.sa_range.lr_start,
               attr.sa_range.lr_end,
               attr.sa_type->sat_name);
    }
}

TEST_CASE("attr_line_t::unicode-wrap")
{
    text_wrap_settings tws = {3, 21};
    attr_line_t prefix;

    prefix.append(" ")
        .append("\u2022"_list_glyph)
        .append(" ")
        .with_attr_for_all(SA_PREFORMATTED.value());

    attr_line_t body;
    body.append("This is a long line that needs to be wrapped and indented");

    attr_line_t li;

    li.append(prefix)
        .append(body, &tws)
        .with_attr_for_all(SA_PREFORMATTED.value());

    attr_line_t dst;

    dst.append(li);

    CHECK(dst.get_string()
          == " \u2022 This is a long\n"
          "   line that needs to\n"
          "   be wrapped and\n"
          "   indented");
}

TEST_CASE("attr_line_t::pre-wrap")
{
    auto pre_al = attr_line_t(" Hello, World! ")
                      .with_attr_for_all(SA_PREFORMATTED.value());
    auto al = attr_line_t("This is a pre-formatted inline -- ")
                  .append(pre_al)
                  .append(" -- that should be wrapped");

    text_wrap_settings tws = {0, 36};

    auto body = attr_line_t().append(al, &tws);

    printf("body\n%s\n", body.get_string().c_str());
}

TEST_CASE("attr_line_t::from_table_cell_content")
{
    const unsigned char bits[] = {
        0x1f, 0x8b, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xed, 0x9c,
        0x5b, 0x6f, 0xda, 0x30, 0x14, 0x80, 0xdf, 0xfb, 0x2b, 0xaa, 0x88, 0xc7,
        0x29, 0x69, 0x81, 0x02, 0xdd, 0x1b, 0x50, 0x28, 0xb4, 0x5c, 0x52, 0x6e,
        0xdd, 0x58, 0xa7, 0xc8, 0x4d, 0x0c, 0x04, 0x82, 0x13, 0x6c, 0x87, 0x4b,
        0xab, 0xfe, 0xf7, 0x25, 0x30, 0xb6, 0xb5, 0x50, 0xc0, 0xac, 0x12, 0x9b,
        0x39, 0x91, 0x10, 0x22, 0xf9, 0x72, 0x8e, 0xe3, 0x93, 0x0f, 0x47, 0x89,
        0xe1, 0xf9, 0xe4, 0x34, 0x58, 0x94, 0x7a, 0xa3, 0xd6, 0xcc, 0x36, 0x9a,
        0xb5, 0x9c, 0xf2, 0xf9, 0xf4, 0x79, 0xbe, 0x6a, 0xbe, 0xda, 0x74, 0x87,
        0xea, 0x78, 0x38, 0x41, 0x14, 0xab, 0x98, 0x4d, 0x55, 0x86, 0x39, 0xb7,
        0x49, 0x97, 0x19, 0x16, 0xc2, 0x43, 0x97, 0xa8, 0xa6, 0x4b, 0x3a, 0x76,
        0xd7, 0xa7, 0x88, 0xdb, 0xe1, 0xa7, 0x1e, 0x36, 0x07, 0x46, 0xb0, 0x8b,
        0xe7, 0xd8, 0x88, 0x98, 0xd8, 0xa0, 0x98, 0xf9, 0x0e, 0x7f, 0x15, 0x71,
        0x19, 0x75, 0x89, 0xd8, 0xa4, 0xe3, 0xae, 0x00, 0x73, 0xa8, 0xaa, 0x37,
        0x8a, 0xd5, 0x4a, 0xba, 0xb4, 0x76, 0xeb, 0x86, 0x36, 0xaf, 0x60, 0x62,
        0xc7, 0xb0, 0x43, 0xd3, 0x5e, 0x45, 0x47, 0x96, 0xc5, 0xb6, 0x52, 0x6f,
        0x0f, 0x88, 0xf8, 0x8e, 0xb3, 0x71, 0x87, 0x97, 0x4f, 0x9b, 0xb3, 0x5a,
        0xd8, 0xc1, 0x1c, 0x8b, 0x27, 0xfe, 0xb6, 0x95, 0x0e, 0x97, 0xed, 0x31,
        0x7f, 0xc5, 0xde, 0xad, 0x06, 0x6b, 0x77, 0x15, 0xaa, 0xcb, 0xcf, 0xad,
        0x7f, 0x9c, 0x5d, 0xc2, 0x09, 0x17, 0x49, 0x7d, 0x4a, 0x31, 0x59, 0x3d,
        0x27, 0x77, 0x0e, 0x20, 0x52, 0xc5, 0x75, 0xcb, 0x96, 0xca, 0xae, 0xcd,
        0xe9, 0x21, 0xde, 0x0b, 0xf2, 0x29, 0x9a, 0x47, 0xdd, 0x8e, 0xed, 0x60,
        0x2d, 0xe8, 0x2f, 0x6d, 0x3c, 0x1c, 0x60, 0x4a, 0xb0, 0xa3, 0x0d, 0x5d,
        0xcb, 0x77, 0xb0, 0xb1, 0xec, 0x3f, 0xed, 0x4c, 0xd9, 0x23, 0x05, 0x47,
        0xb4, 0x8b, 0x0f, 0xd9, 0x2d, 0x42, 0x7b, 0xec, 0x4e, 0xef, 0x46, 0xee,
        0x58, 0x14, 0xf0, 0xe2, 0xfd, 0x00, 0xff, 0xbe, 0x17, 0xe7, 0xe0, 0x85,
        0x28, 0x09, 0x5e, 0x1c, 0x81, 0x17, 0x51, 0xf0, 0x42, 0x94, 0x04, 0x2f,
        0x8e, 0xc0, 0x8b, 0x18, 0x78, 0x21, 0x4a, 0x82, 0x17, 0x47, 0xe0, 0x45,
        0x1c, 0xbc, 0x10, 0x25, 0xc1, 0x8b, 0x23, 0xf0, 0xe2, 0x02, 0xbc, 0x10,
        0x25, 0xc1, 0x0b, 0x79, 0xbc, 0x20, 0x98, 0x4f, 0x5c, 0x3a, 0x08, 0xfd,
        0x20, 0xb6, 0x09, 0xb7, 0xa1, 0xf6, 0x20, 0x41, 0x07, 0x79, 0x75, 0x80,
        0xbb, 0x4f, 0xc2, 0x24, 0xe8, 0x20, 0x9f, 0x0e, 0xc1, 0xbb, 0xc1, 0x38,
        0x32, 0x07, 0x30, 0x40, 0xec, 0x41, 0x82, 0x11, 0xd2, 0x19, 0x61, 0x8c,
        0x19, 0xd3, 0xd8, 0xc4, 0xe6, 0x66, 0x0f, 0x83, 0x13, 0x7b, 0x90, 0xe0,
        0x84, 0xe4, 0x4e, 0xc0, 0x95, 0x93, 0x30, 0x09, 0x4e, 0xc8, 0xe3, 0x04,
        0x9b, 0x31, 0xc7, 0xed, 0x6a, 0xc1, 0xab, 0x8b, 0x29, 0xcc, 0xf2, 0xf8,
        0x0b, 0x12, 0xac, 0x90, 0xde, 0x0a, 0x18, 0x2b, 0x84, 0x49, 0xb0, 0x42,
        0x7a, 0x2b, 0x60, 0x86, 0x87, 0x30, 0x09, 0x56, 0x48, 0x6f, 0x05, 0xcc,
        0xef, 0x10, 0x26, 0xc1, 0x0a, 0xe9, 0xad, 0x80, 0xd9, 0x1d, 0xc2, 0x24,
        0x58, 0x21, 0xbd, 0x15, 0x30, 0xb7, 0x43, 0x98, 0x04, 0x2b, 0xa4, 0xb7,
        0x22, 0x01, 0x56, 0x88, 0x92, 0x60, 0x85, 0xf4, 0x56, 0x24, 0xc1, 0x0a,
        0x51, 0x12, 0xac, 0x90, 0xde, 0x8a, 0x14, 0x58, 0x21, 0x4a, 0x82, 0x15,
        0xd2, 0x5b, 0x71, 0x09, 0x56, 0x88, 0x92, 0x60, 0x85, 0xf4, 0x56, 0x9c,
        0xc3, 0xc3, 0x6d, 0x61, 0x12, 0xb4, 0x90, 0x5f, 0x0b, 0x78, 0xba, 0x2d,
        0x4c, 0x82, 0x16, 0xf2, 0x6b, 0x01, 0x8f, 0xb7, 0x85, 0xc9, 0xed, 0xd4,
        0xf7, 0x8d, 0xc4, 0xb6, 0xff, 0xa7, 0x0a, 0x8a, 0x03, 0x7f, 0x4e, 0x25,
        0x92, 0x70, 0x91, 0xf4, 0x03, 0x55, 0x55, 0x38, 0xf5, 0xb1, 0x72, 0x38,
        0x59, 0x97, 0x3f, 0xf8, 0xe8, 0xd8, 0x14, 0x4f, 0x90, 0xe3, 0x68, 0x98,
        0xa0, 0x47, 0x07, 0x5b, 0xe0, 0xaa, 0x28, 0x09, 0x43, 0xd8, 0x87, 0x7a,
        0xf1, 0x10, 0x1e, 0xc3, 0xf9, 0x83, 0x72, 0x40, 0x37, 0x90, 0x35, 0x0e,
        0x3b, 0xc3, 0x32, 0x5c, 0x2f, 0xec, 0x27, 0xa6, 0x0d, 0x6d, 0x66, 0x86,
        0xe3, 0x9a, 0xe1, 0xb9, 0x94, 0x83, 0x21, 0xa2, 0x24, 0x18, 0xf2, 0xc1,
        0x86, 0x44, 0x12, 0x91, 0xda, 0xa8, 0xd3, 0xcd, 0x4f, 0xed, 0xa7, 0x88,
        0x7a, 0x59, 0xf7, 0x12, 0xfe, 0x6d, 0xaf, 0xdf, 0x9a, 0xe5, 0x47, 0xbd,
        0x4c, 0x31, 0x31, 0xba, 0xbd, 0xd6, 0x13, 0xd9, 0x2a, 0x69, 0xb6, 0x73,
        0x85, 0x68, 0xdf, 0xd1, 0x4d, 0xaf, 0xd5, 0xef, 0x66, 0xd5, 0xc9, 0xe3,
        0xac, 0xdd, 0x2f, 0x50, 0x52, 0x47, 0x63, 0x32, 0xbc, 0x99, 0x9e, 0x95,
        0xe2, 0x9e, 0xd7, 0xb4, 0x1a, 0x76, 0xda, 0x4b, 0xa7, 0xf2, 0xb1, 0x62,
        0xa2, 0x7d, 0x7d, 0x31, 0x6a, 0xc7, 0x6b, 0xd1, 0x6c, 0x13, 0xf1, 0x8c,
        0x7a, 0x58, 0xff, 0x7c, 0xde, 0x0b, 0x7a, 0xcb, 0x36, 0xe7, 0x45, 0xd0,
        0x7c, 0x16, 0x5c, 0x4e, 0x22, 0xd3, 0x74, 0x7d, 0xc2, 0x99, 0x76, 0xa6,
        0x79, 0x88, 0xb1, 0x60, 0xec, 0xb2, 0x8c, 0x1e, 0x62, 0xbd, 0x03, 0xdb,
        0xb8, 0x28, 0x46, 0xd0, 0xce, 0x5c, 0xf1, 0xaa, 0x50, 0x56, 0x53, 0xb1,
        0x04, 0xfa, 0xda, 0xf2, 0x22, 0xc3, 0x69, 0xdd, 0xee, 0x5f, 0x96, 0x6f,
        0x72, 0x56, 0x3e, 0xdb, 0xb8, 0x4f, 0x60, 0xd2, 0xaa, 0x97, 0xd1, 0xcd,
        0x04, 0x75, 0x4c, 0xbd, 0x91, 0x2c, 0xdf, 0x8d, 0x6a, 0xa5, 0xd6, 0x24,
        0x3d, 0xbb, 0x1b, 0xf9, 0xf1, 0x64, 0xa1, 0x92, 0x4a, 0xeb, 0xed, 0xfb,
        0x8b, 0x6c, 0xe1, 0x29, 0xa3, 0x4e, 0xfa, 0xd9, 0xab, 0x72, 0xa6, 0x79,
        0xd5, 0x1f, 0x37, 0xf5, 0x5c, 0x7b, 0xe0, 0x3e, 0xc5, 0x8a, 0xd1, 0xe8,
        0xb8, 0xe0, 0xc7, 0x5b, 0x5f, 0xf4, 0xe4, 0x7e, 0x45, 0xf9, 0xbf, 0xaf,
        0x63, 0xdf, 0xdd, 0xba, 0x7e, 0xcb, 0xea, 0xda, 0xd7, 0x6b, 0xde, 0x9c,
        0xa2, 0x0a, 0xa6, 0xd4, 0xa5, 0xeb, 0xaf, 0x84, 0x37, 0x7e, 0xe7, 0xbe,
        0x8d, 0xc3, 0x38, 0xe2, 0x7e, 0x18, 0x47, 0xa9, 0x54, 0x2b, 0x46, 0xb6,
        0x5a, 0xd6, 0x4b, 0xc5, 0x74, 0xa5, 0xf1, 0xbb, 0x5c, 0x8b, 0x66, 0xbc,
        0x9c, 0xbc, 0xfc, 0x00, 0xa7, 0xa7, 0x1d, 0x26, 0xd0, 0x57, 0x00, 0x00};
    auto sf = string_fragment::from_bytes(bits, sizeof(bits));

    auto al = attr_line_t::from_table_cell_content(sf, 100);
}
