/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     disc_lagrange_2_3d.c                                           */
/*                                                                          */
/* description:  piecewise quadratic discontinuous Lagrange elements in 3d  */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#define N_BAS_3D N_VERTICES_3D+N_EDGES_3D

static const REAL d_bary2_3d[N_BAS_3D][N_LAMBDA] = {{1.0, 0.0, 0.0, 0.0},
						    {0.0, 1.0, 0.0, 0.0},
						    {0.0, 0.0, 1.0, 0.0},
						    {0.0, 0.0, 0.0, 1.0},
						    {0.5, 0.5, 0.0, 0.0},
						    {0.5, 0.0, 0.5, 0.0},
						    {0.5, 0.0, 0.0, 0.5},
						    {0.0, 0.5, 0.5, 0.0},
						    {0.0, 0.5, 0.0, 0.5},
						    {0.0, 0.0, 0.5, 0.5}};

/****************************************************************************/
/*  basisfunction at vertex 0                                               */
/****************************************************************************/

static REAL d_phi2v0_3d(const REAL lambda[N_LAMBDA])
{
  return(lambda[0]*(2.0*lambda[0] - 1.0));
}

static const REAL *d_grd_phi2v0_3d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[0] = 4.0*lambda[0] - 1.0;
  return((const REAL *) grd);
}

static const REAL (*d_D2_phi2v0_3d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{4, 0, 0, 0},
					      {0, 0, 0, 0},
					      {0, 0, 0, 0},
					      {0, 0, 0, 0}};

  return(D2);
}

/****************************************************************************/
/*  basisfunction at vertex 1                                               */
/****************************************************************************/

static REAL d_phi2v1_3d(const REAL lambda[N_LAMBDA])
{
  return(lambda[1]*(2.0*lambda[1] - 1.0));
}

static const REAL *d_grd_phi2v1_3d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[1] = 4.0*lambda[1] - 1.0;
  return((const REAL *) grd);
}

static const REAL (*d_D2_phi2v1_3d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 0, 0},
					      {0, 4, 0, 0},
					      {0, 0, 0, 0},
					      {0, 0, 0, 0}};

  return(D2);
}

/****************************************************************************/
/*  basisfunction at vertex 2                                               */
/****************************************************************************/

static REAL d_phi2v2_3d(const REAL lambda[N_LAMBDA])
{
  return(lambda[2]*(2.0*lambda[2] - 1.0));
}

static const REAL *d_grd_phi2v2_3d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[2] = 4.0*lambda[2] - 1.0;
  return((const REAL *) grd);
}

static const REAL (*d_D2_phi2v2_3d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 0, 0},
					      {0, 0, 0, 0},
					      {0, 0, 4, 0},
					      {0, 0, 0, 0}};

  return(D2);
}

/****************************************************************************/
/*  basisfunction at vertex 3                                               */
/****************************************************************************/

static REAL d_phi2v3_3d(const REAL lambda[N_LAMBDA])
{
  return(lambda[3]*(2.0*lambda[3] - 1.0));
}

static const REAL *d_grd_phi2v3_3d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[3] = 4.0*lambda[3] - 1.0;
  return((const REAL *) grd);
}

static const REAL (*d_D2_phi2v3_3d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 0, 0},
					      {0, 0, 0, 0},
					      {0, 0, 0, 0},
					      {0, 0, 0, 4}};

  return(D2);
}

/****************************************************************************/
/*  basisfunction at edge 0                                                 */
/****************************************************************************/

static REAL d_phi2e0_3d(const REAL lambda[N_LAMBDA])
{
  return(4.0*lambda[0]*lambda[1]);
}

static const REAL *d_grd_phi2e0_3d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[0] = 4.0*lambda[1];
  grd[1] = 4.0*lambda[0];
  return((const REAL *) grd);
}

static const REAL (*d_D2_phi2e0_3d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 4, 0, 0},
					      {4, 0, 0, 0},
					      {0, 0, 0, 0},
					      {0, 0, 0, 0}};
  return(D2);
}

/****************************************************************************/
/*  basisfunction at edge 1                                                 */
/****************************************************************************/

static REAL d_phi2e1_3d(const REAL lambda[N_LAMBDA])
{
  return(4.0*lambda[0]*lambda[2]);
}

static const REAL *d_grd_phi2e1_3d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[0] = 4.0*lambda[2];
  grd[2] = 4.0*lambda[0];
  return((const REAL *) grd);
}

static const REAL (*d_D2_phi2e1_3d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 4, 0},
					      {0, 0, 0, 0},
					      {4, 0, 0, 0},
					      {0, 0, 0, 0}};
  return(D2);
}

/****************************************************************************/
/*  basisfunction at edge 2                                                 */
/****************************************************************************/

static REAL d_phi2e2_3d(const REAL lambda[N_LAMBDA])
{
  return(4.0*lambda[0]*lambda[3]);
}

static const REAL *d_grd_phi2e2_3d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[0] = 4.0*lambda[3];
  grd[3] = 4.0*lambda[0];
  return((const REAL *) grd);
}

static const REAL (*d_D2_phi2e2_3d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 0, 4},
					      {0, 0, 0, 0},
					      {4, 0, 0, 0},
					      {0, 0, 0, 0}};
  return(D2);
}

/****************************************************************************/
/*  basisfunction at edge 3                                                 */
/****************************************************************************/

static REAL d_phi2e3_3d(const REAL lambda[N_LAMBDA])
{
  return(4.0*lambda[1]*lambda[2]);
}

static const REAL *d_grd_phi2e3_3d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[1] = 4.0*lambda[2];
  grd[2] = 4.0*lambda[1];
  return((const REAL *) grd);
}

static const REAL (*d_D2_phi2e3_3d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 0, 0},
					      {0, 0, 4, 0},
					      {0, 4, 0, 0},
					      {0, 0, 0, 0}};
  return(D2);
}

/****************************************************************************/
/*  basisfunction at edge 4                                                 */
/****************************************************************************/

static REAL d_phi2e4_3d(const REAL lambda[N_LAMBDA])
{
  return(4.0*lambda[1]*lambda[3]);
}

static const REAL *d_grd_phi2e4_3d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[1] = 4.0*lambda[3];
  grd[3] = 4.0*lambda[1];
  return((const REAL *) grd);
}

static const REAL (*d_D2_phi2e4_3d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 0, 0},
					      {0, 0, 0, 4},
					      {0, 0, 0, 0},
					      {0, 4, 0, 0}};
  return(D2);
}

/****************************************************************************/
/*  basisfunction at edge 5                                                 */
/****************************************************************************/

static REAL d_phi2e5_3d(const REAL lambda[N_LAMBDA])
{
  return(4.0*lambda[2]*lambda[3]);
}

static const REAL *d_grd_phi2e5_3d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[2] = 4.0*lambda[3];
  grd[3] = 4.0*lambda[2];
  return((const REAL *) grd);
}

static const REAL (*d_D2_phi2e5_3d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 0, 0},
					      {0, 0, 0, 0},
					      {0, 0, 0, 4},
					      {0, 0, 4, 0}};
  return(D2);
}

/****************************************************************************/
/*  functions for combining basisfunctions with coefficients                */
/****************************************************************************/

static const DOF *d_get_dof_indices2_3d(const EL *el, const DOF_ADMIN *admin,
					DOF *idof)
{
  static DOF  index_vec[N_BAS_3D];
  int         i, n0, node;
  DOF         *rvec = idof ? idof : index_vec;
  DOF         **dof = el->dof;

  n0 = admin->n0_dof[CENTER];
  node = admin->mesh->node[CENTER];

  for (i = 0; i < N_BAS_3D; i++)
    rvec[i] = dof[node][n0+i];

  return((const DOF *) rvec);
}

static const S_CHAR *d_get_bound2_3d(const EL_INFO *el_info, S_CHAR *bound)
{
  FUNCNAME("d_get_bound2_3d");
  static S_CHAR  bound_vec[N_BAS_3D];
  S_CHAR         *rvec = bound ? bound : bound_vec;
  int            i;

  TEST_FLAG(FILL_BOUND, el_info);

  for (i = 0; i < N_VERTICES_3D; i++)
    rvec[i] = el_info->vertex_bound[i];
  for (i = 0; i < N_EDGES_3D; i++)
    rvec[N_VERTICES_3D+i] = el_info->edge_bound[i];

  return((const S_CHAR *) rvec);
}

static const int *d_get_int_vec2_3d(const EL *el, const DOF_INT_VEC *vec,
				    int *ivec)
{
  FUNCNAME("d_get_int_vec2_3d");
  int           i, n0, node;
  static int    local_vec[N_BAS_3D];
  int           *v = nil, *rvec = ivec ? ivec : local_vec;
  DOF           **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[CENTER];
  node = vec->fe_space->admin->mesh->node[CENTER];

  for (i = 0; i < N_BAS_3D; i++)
    rvec[i] = v[dof[node][n0+i]];

  return((const int *) rvec);
}

static const REAL *d_get_real_vec2_3d(const EL *el, const DOF_REAL_VEC *vec,
				      REAL *Rvec)
{
  FUNCNAME("d_get_real_vec2_3d");
  int            i, n0, node;
  static REAL    local_vec[N_BAS_3D];
  REAL           *v = nil, *rvec = Rvec ? Rvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[CENTER];
  node = vec->fe_space->admin->mesh->node[CENTER];

  for (i = 0; i < N_BAS_3D; i++)
    rvec[i] = v[dof[node][n0+i]];

  return((const REAL *) rvec);
}

static const REAL_D *d_get_real_d_vec2_3d(const EL *el, 
					  const DOF_REAL_D_VEC *vec,
					  REAL_D *Rvec)
{
  FUNCNAME("d_get_real_d_vec2_3d");
  int            i, k, n0, node;
  static REAL_D  local_vec[N_BAS_3D];
  REAL_D         *v = nil, *rvec = Rvec ? Rvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[CENTER];
  node = vec->fe_space->admin->mesh->node[CENTER];

  for (i = 0; i < N_BAS_3D; i++)
    for (k = 0; k < DIM_OF_WORLD; k++)
      rvec[i][k] = v[dof[node][n0+i]][k];

  return((const REAL_D *) rvec);
}

static const U_CHAR *d_get_uchar_vec2_3d(const EL *el, const DOF_UCHAR_VEC *vec,
					 U_CHAR *uvec)
{
  FUNCNAME("d_get_uchar_vec2_3d");
  int            i, n0, node;
  static U_CHAR  local_vec[N_BAS_3D];
  U_CHAR         *v = nil, *rvec = uvec ? uvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[CENTER];
  node = vec->fe_space->admin->mesh->node[CENTER];

  for (i = 0; i < N_BAS_3D; i++)
    rvec[i] = v[dof[node][n0+i]];

  return((const U_CHAR *) rvec);
}

static const S_CHAR *d_get_schar_vec2_3d(const EL *el, const DOF_SCHAR_VEC *vec,
					 S_CHAR *svec)
{
  FUNCNAME("d_get_schar_vec2_3d");
  int            i, n0, node;
  static S_CHAR  local_vec[N_BAS_3D];
  S_CHAR         *v = nil, *rvec = svec ? svec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[CENTER];
  node = vec->fe_space->admin->mesh->node[CENTER];

  for (i = 0; i < N_BAS_3D; i++)
    rvec[i] = v[dof[node][n0+i]];

  return((const S_CHAR *) rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of scalar functions          ---*/
/*--------------------------------------------------------------------*/

GENERATE_INTERPOL(d_,2,3,10)

/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of vector functions          ---*/
/*--------------------------------------------------------------------*/

GENERATE_INTERPOL_D(d_,2,3,10)

/****************************************************************************/
/*  functions for interpolation/ restriction during refinement/coarsening   */
/****************************************************************************/

static void d_real_refine_inter2_3d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("d_real_refine_inter2_3d");
  EL        *el;
  REAL      *vec = nil;
  int       node, n0, i;
  U_CHAR    type;
  const DOF *cdof;
  DOF       pdof[N_BAS_3D];
  const DOF *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;

  if (n < 1) return;

  GET_DOF_VEC(vec, drv);
  if (!drv->fe_space)
  {
    ERROR("no fe_space in dof_real_vec %s\n", NAME(drv));
    return;
  }
  else if (!drv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drv->fe_space));
    return;
  }
  get_dof_indices = drv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drv->fe_space);

  for (i = 0; i < n; i++)
  {
    el = list[i].el_info.el;
    
    get_dof_indices(el, admin, pdof);
    
    node = drv->fe_space->mesh->node[CENTER];
    n0 = admin->n0_dof[CENTER];

/****************************************************************************/
/*  values on child[0]                                                      */
/****************************************************************************/

    cdof = get_dof_indices(el->child[0], admin, nil);
    
    vec[cdof[0]] = vec[pdof[0]];
    vec[cdof[1]] = vec[pdof[2]];
    vec[cdof[2]] = vec[pdof[3]];
    vec[cdof[3]] = vec[pdof[4]];
    vec[cdof[4]] = vec[pdof[5]];
    vec[cdof[5]] = vec[pdof[6]];
    vec[cdof[6]] = (0.375*vec[pdof[0]] - 0.125*vec[pdof[1]] 
		    + 0.75*vec[pdof[4]]);
    vec[cdof[7]] = vec[pdof[9]];
    vec[cdof[8]] = (0.125*(-vec[pdof[0]] - vec[pdof[1]]) + 0.25*vec[pdof[4]]
		    + 0.5*(vec[pdof[5]] + vec[pdof[7]]));
    vec[cdof[9]] = (0.125*(-vec[pdof[0]] - vec[pdof[1]]) + 0.25*vec[pdof[4]]
		    + 0.5*(vec[pdof[6]] + vec[pdof[8]]));
    
/****************************************************************************/
/*  values on child[1]                                                      */
/****************************************************************************/

#if NEIGH_IN_EL
    type = list[i].el_info.el->el_type;
#else
    type = list[i].el_info.el_type;
#endif
    
    cdof = get_dof_indices(el->child[1], admin, nil);
    vec[cdof[0]] = vec[pdof[1]];
    if (type == 0)
    {
      vec[cdof[1]] = vec[pdof[3]];
      vec[cdof[2]] = vec[pdof[2]];
    }
    else
    {
      vec[cdof[1]] = vec[pdof[2]];
      vec[cdof[2]] = vec[pdof[3]];
    }
    vec[cdof[3]] = vec[pdof[4]];
    if (type == 0)
    {
      vec[cdof[4]] = vec[pdof[8]];
      vec[cdof[5]] = vec[pdof[7]];
    }
    else
    {
      vec[cdof[4]] = vec[pdof[7]];
      vec[cdof[5]] = vec[pdof[8]];
    }
    vec[cdof[6]] = (0.375*vec[pdof[1]] - 0.125*vec[pdof[0]] 
		    + 0.75*vec[pdof[4]]);
    vec[cdof[7]] = vec[pdof[9]];
    if (type == 0)
    {
      vec[cdof[8]] = (0.125*(-vec[pdof[0]] - vec[pdof[1]]) + 0.25*vec[pdof[4]]
		      + 0.5*(vec[pdof[6]] + vec[pdof[8]]));
      vec[cdof[9]] = (0.125*(-vec[pdof[0]] - vec[pdof[1]]) + 0.25*vec[pdof[4]]
		      + 0.5*(vec[pdof[5]] + vec[pdof[7]]));
    }
    else
    {
      vec[cdof[8]] = (0.125*(-vec[pdof[0]] - vec[pdof[1]]) + 0.25*vec[pdof[4]]
		      + 0.5*(vec[pdof[5]] + vec[pdof[7]]));
      vec[cdof[9]] = (0.125*(-vec[pdof[0]] - vec[pdof[1]]) + 0.25*vec[pdof[4]]
		      + 0.5*(vec[pdof[6]] + vec[pdof[8]]));
    }
  }
  return;
}

static void d_real_coarse_inter2_3d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("d_real_coarse_inter2_3d");
  EL        *el;
  REAL      *v = nil;
  U_CHAR    type;
  DOF       pdof[N_BAS_3D], cdof0[N_BAS_3D], cdof1[N_BAS_3D];
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;
  MESH      *mesh = nil;
  int       i;

  if (n < 1) return;

  GET_DOF_VEC(v, drv);
  if (!drv->fe_space)
  {
    ERROR("no fe_space in dof_real_vec %s\n", NAME(drv));
    return;
  }
  else if (!drv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drv->fe_space));
    return;
  }

  get_dof_indices = drv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drv->fe_space);
  GET_STRUCT(mesh,drv->fe_space);

  for (i = 0; i < n; i++)
  {
    el = list[i].el_info.el;

#if NEIGH_IN_EL
    type = el->el_type;
#else
    type = list[i].el_info.el_type;
#endif

    get_dof_indices(el, admin, pdof);
    get_dof_indices(el->child[0], admin, cdof0);
    get_dof_indices(el->child[1], admin, cdof1);

    v[pdof[0]] = v[cdof0[0]];
    v[pdof[1]] = v[cdof1[0]];
    v[pdof[2]] = 0.5 * (v[cdof0[1]] + v[cdof1[ type ? 1 : 2 ]]);
    v[pdof[3]] = 0.5 * (v[cdof0[2]] + v[cdof1[ type ? 2 : 1 ]]);
    v[pdof[4]] = 0.5 * (v[cdof0[3]] + v[cdof1[3]]);
    v[pdof[5]] = 0.5 * (v[cdof0[9]] + v[cdof1[ type ? 9 : 8 ]]);
    v[pdof[6]] = v[cdof0[5]];
    v[pdof[7]] = v[cdof1[ type ? 4 : 5]];
    v[pdof[8]] = v[cdof1[ type ? 5 : 4]];
    v[pdof[9]] = 0.5 * (v[cdof0[7]] + v[cdof1[7]]);
  }

  return;
}


static BAS_FCT     *d_phi2_3d[N_BAS_3D]     = {d_phi2v0_3d, d_phi2v1_3d, 
					       d_phi2v2_3d, d_phi2v3_3d,
					       d_phi2e0_3d, d_phi2e1_3d,
					       d_phi2e2_3d, d_phi2e3_3d,
					       d_phi2e4_3d, d_phi2e5_3d};
static GRD_BAS_FCT *d_grd_phi2_3d[N_BAS_3D] = {d_grd_phi2v0_3d, d_grd_phi2v1_3d,
					       d_grd_phi2v2_3d, d_grd_phi2v3_3d,
					       d_grd_phi2e0_3d, d_grd_phi2e1_3d,
					       d_grd_phi2e2_3d, d_grd_phi2e3_3d,
					       d_grd_phi2e4_3d, d_grd_phi2e5_3d};
static D2_BAS_FCT  *d_D2_phi2_3d[N_BAS_3D]  = {d_D2_phi2v0_3d, d_D2_phi2v1_3d,
					       d_D2_phi2v2_3d, d_D2_phi2v3_3d,
					       d_D2_phi2e0_3d, d_D2_phi2e1_3d,
					       d_D2_phi2e2_3d, d_D2_phi2e3_3d,
					       d_D2_phi2e4_3d, d_D2_phi2e5_3d};

static BAS_FCTS  disc_lagrange2_3d = {"disc_lagrange2_3d", 3, N_BAS_3D, 2,
				      {0, N_BAS_3D, 0, 0}, 
				      nil,
				      d_phi2_3d, d_grd_phi2_3d, d_D2_phi2_3d, 
				      d_get_dof_indices2_3d, 
				      d_get_bound2_3d,
				      d_interpol2_3d,
				      d_interpol_d2_3d,
				      d_get_int_vec2_3d,
				      d_get_real_vec2_3d,
				      d_get_real_d_vec2_3d,
				      d_get_uchar_vec2_3d,
				      d_get_schar_vec2_3d,
				      d_real_refine_inter2_3d,
				      d_real_coarse_inter2_3d,
				      nil,
				      nil,
				      nil,
				      nil,
				      d_bary2_3d, };

#undef N_BAS_3D
