/* NVTV Philips TV-I2C access -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: tv_ph1_saa7102.c,v 1.6 2004/01/27 17:05:26 dthierbach Exp $
 *
 * Contents:
 *
 * Routines to access the Philips Model1 (SAA7102 etc.) encoder chip 
 * registers via the I2C bus.
 *
 */

#include "local.h" /* before everything else */

#include "bitmask.h"
#include "tv_chip.h"
#include "tv_i2c.h"
#include "tv_ph1_saa7102.h"

/* -------- Philips Model1 -------- */

typedef struct {
  int blckl_diff; /* not sure if it is really a difference, but we have */
  int blnnl_diff; /* to modify these levels somehow */
  int reg01; /* mask c0   */
  int reg03; /* mask c0   */
  int reg04; /* mask c0   */
  int reg05; /* mask 00   */
  int reg06; /* mask 00   */
  int reg07; /* mask 00   */
  int reg08; /* mask fc   */
  int reg10; /* mask c0   */
  int reg11; /* mask 00   */
  int reg12; /* mask 00   */
  int reg14; /* mask 00   */
  int reg15; /* mask 80   */
  int reg20; /* mask 40 ! */
  int reg21; /* mask 00   */
  int reg24; /* mask 00   */
  int reg25; /* mask 80   */
  int reg30; /* mask c0   */
  int reg31; /* mask f8   */
  int reg33; /* mask 00   */
  int reg36; /* mask 00   */
  int reg60; /* mask 80   */
  int reg7d; /* mask 16 ! */ 
} TVPhMacro;

TVPhMacro tvPhNtscMacro0 = {
  blckl_diff:0, blnnl_diff:0, reg7d:0x00 
};

TVPhMacro tvPhNtscMacro1 = {         
  blckl_diff: 31, /* 0x1b <- 0x3a */ 
  blnnl_diff: 32, /* 0x0e <- 0x2e */ 
  reg60:0x40, reg7d:0xc8, reg01:0x1e,  
  reg03:0x11, reg04:0x26, reg05:0x0a, reg06:0xc5, reg07:0xff, reg08:0x03,  
  reg10:0x11, reg11:0xad, reg12:0x60, reg14:0x0f, reg15:0x78, reg20:0x9b,  
  reg21:0x04, reg24:0x00, reg25:0x00, reg30:0x1b, reg31:0x04, reg33:0x0f,  
  reg36:0x0f 
};                                                          

TVPhMacro tvPhNtscMacro2 = {                                             
  blckl_diff: 31, /* 0x1b <- 0x3a */ 
  blnnl_diff: 32, /* 0x0e <- 0x2e */ 
  reg60:0x40, reg7d:0xe8, reg01:0x1e,  
  reg03:0x11, reg04:0x26, reg05:0x0a, reg06:0xc5, reg07:0xff, reg08:0x03,  
  reg10:0x11, reg11:0xad, reg12:0x60, reg14:0x0f, reg15:0x78, reg20:0x9b,  
  reg21:0x04, reg24:0x00, reg25:0x00, reg30:0x1b, reg31:0x04, reg33:0x0f,  
  reg36:0x0f 
};                                                          

TVPhMacro tvPhNtscMacro3 = {                                             
  blckl_diff: 31, /* 0x1b <- 0x3a */ 
  blnnl_diff: 32, /* 0x0e <- 0x2e */ 
  reg60:0x40, reg7d:0xe8, reg01:0x18,  
  reg03:0x15, reg04:0x22, reg05:0x0a, reg06:0xc5, reg07:0xff, reg08:0x03,  
  reg10:0x15, reg11:0xad, reg12:0x60, reg14:0x0f, reg15:0x78, reg20:0x9b,  
  reg21:0x04, reg24:0x00, reg25:0x00, reg30:0x1b, reg31:0x04, reg33:0x0f,  
  reg36:0x0f 
};                                                          

/* PAL diff: blckl: -0x1C  blnnl: -0x1C */

TVPhMacro tvPhPalMacro0 = {                                              
  blckl_diff:0, blnnl_diff:0, reg7d:0x00 
};                                  

TVPhMacro tvPhPalMacro1= {                                               
  blckl_diff: 28, /* 0x17 <- 0x33 */ 
  blnnl_diff: 28, /* 0x19 <- 0x35 */ 
  reg60:0x40, reg7d:0xc8, reg01:0x1b,  
  reg03:0x22, reg04:0x2b, reg05:0x08, reg06:0x74, reg07:0x55, reg08:0x01,  
  reg10:0x22, reg11:0x15, reg12:0x60, reg14:0x07, reg15:0x7e, reg20:0xbb,  
  reg21:0x72, reg24:0x02, reg25:0x54, reg30:0x2c, reg31:0x04, reg33:0xfe,  
  reg36:0x7e 
}; 

TVPhMacro *TVMacroPhRegs (TVEncoderRegs *r)
{
  RAISE (MSG_DEBUG, "tv macro ph %i", r->ph1.macro); 
  switch (r->ph1.macro) {
    case 0: /* NTSC APS 0 */
      return &tvPhNtscMacro0;
      break;
    case 1: /* NTSC APS 1 */
      return &tvPhNtscMacro1;
      break;
    case 2: /* NTSC APS 2 */
      return &tvPhNtscMacro2;
      break;
    case 3: /* NTSC APS 3 */
      return &tvPhNtscMacro3;
      break;
    case 4: /* PAL APS 0 */
      return &tvPhPalMacro0;
      break;
    case 5: /* PAL APS 1 */
    case 6:
    case 7:
      return &tvPhPalMacro1;
      break;
  }
  return NULL;
}

void TVPhCreate (TVEncoderObj *this, TVChip chip_type, void *ctrl)
{
  RAISE (MSG_DEBUG, "ph create");
  this->type = chip_type;
  this->ctrl = ctrl;
  this->hwconfig = 0; /* No hardware config for PH */
  this->hwstate = 0;  /* pcbn=0, slave=0 */
}

/*
  Registers:
  ----
  0x16:  Common DAC adjust fine - set gain to 0
  0x17:  Red DAC adjust fine - default
  0x18:  Green DAC adjust fine - default
  0x19:  Blue DAC adjust fine - default (max)
  0x1a:  Monitor sense mode - recommended setting
  0x1b:  Monitor sense mode off
  0x21:  Undocumented register in SAA7108E
  0x26:  Widescreen signal
  0x27:  Widescreen signal
  0x28:  Realtime control, burst start
  0x29:  Sync reset enable, burst end
  0x2a:  copy gen 0
  0x2b:  copy gen 1
  0x2c:  copy gen 2, Copy gen enable (off)
  0x2d:  Output port control - default connector
  0x38:  Gain luminance for RGB
  0x39:  Gain color difference for RGB
  0x3a:  Input control 1 - 0
  0x54:  Input cont. 2 (VPS enable) - edge1: falling, edge2: rising
  0x6c:  Horiz. trigger
  0x6d:  Horiz. trigger
  0x6e:  Multi control
  0x6f:  Closed caption, teletext - ttxen=0 ccen=01 sccln=2
  0x75:  Composite sync advance - 22h
  96-7   efs   = 0  FSVGC (FIELD) ignored
  96-6   pcbn  = 0  NCBO (BLANK) polarity
  96-5   slave = 0
  97-7   hfs   = 0  no hsync from FSVGC (FIELD)
  97-6   vfs   = 0  no vsync from FSVGC (FIELD)
  97-5   ofs   = 1
  97-4   pfs   = 0  fsync polarity (0:high, 1:low)
  97-3   ovs   = 1  vsync direction (0:in, 1:out)
  97-2   pvs   = 1  vsync polarity (0:high, 1:low)  (why 1 ???)
  97-1   ohs   = 1  hsync direction (0:in, 1:out)
  97-0   phs   = 0  hsync polarity (0:high, 1:low)
         Note: sync polarity varies wildly in different modes.
  0xa2:  Border color Y
  0xa3:  Border color U
  0xa4:  Border color V
  0xfc:  Vertical hot spot
  fd-7   lutoff = 1  (bypass LUT)
  fd-6   cmode  = 0  (cursor mode: invert)
  fd-5   lutl   = 0  (LUT loading via input inactive)
  fd-4:2 if     = 110 = 6 (R8G8B8 4:4:4 non-interlaced, special ordering)
  fd-1   matoff = 0  (RGB -> CrYCb active)
  fd-0   dfoff  = 0  (4:4:4 -> 4:4:2 active)
*/

void TVPhSetPort (TVEncoderObj *this, int port)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;

  RAISE (MSG_DEBUG, "ph port %08X", port);
  if ((port & PORT_FORMAT_MASK) != PORT_FORMAT_RGB) {
    RAISE (MSG_ABORT, "TVPh1SetPort: Only RGB multiplex format supported.");
  }
  if ((port & PORT_PCLK_MODE) != PORT_PCLK_MASTER) {
    RAISE (MSG_ABORT, "TVPh1SetPort: Only clock master mode supported.");
  }
  if ((port & PORT_BLANK_DIR) != PORT_BLANK_OUT) {
    RAISE (MSG_ABORT, "TVPh1SetPort: Only output blank supported.");
  }
  this->hwstate = ((port & PORT_BLANK_POLARITY) ? SetBit(6) : 0)
                | ((port & PORT_SYNC_DIR) ? SetBit(5) : 0);
  TVWriteBus (dev, 0x96, SetBitField(this->hwstate,6:5,6:5));
  /* FIXME Test sync polarity - this varies wildly in the observed modes. */
  TVWriteBus (dev, 0x97, 0x20 /* ofs=1, hfs=vfs=0 */
	      | ((port & PORT_SYNC_DIR) ? 0 : SetBit(1) | SetBit(3))
	      | ((port & PORT_VSYNC_POLARITY) ? 0 : SetBit(2))
	      | ((port & PORT_HSYNC_POLARITY) ? 0 : SetBit(0)));
}

void TVPh1SetPort (TVEncoderObj *this, int port)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;

  TVPhSetPort (this, port);
  if ((port & PORT_PCLK_POLARITY) == PORT_PCLK_HIGH) {
    TVWriteBus (dev, 0x54, 0x01); /* edge1: falling, edge2: rising */
  } else {
    TVWriteBus (dev, 0x54, 0x02); /* edge1: raising, edge2: falling */
  }
}

void TVPhGetPort (TVEncoderObj *this, int *port)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;
  I2CByte res;

  RAISE (MSG_DEBUG, "ph get port");
  *port = PORT_FORMAT_RGB | PORT_PCLK_MASTER | PORT_PCLK_HIGH | 
          PORT_BLANK_OUT;
  TVReadBus (dev, 0x96, &res);
  *port |= (GetBit(res,6) ? PORT_BLANK_HIGH : PORT_BLANK_LOW) 
        |  (GetBit(res,5) ? PORT_SYNC_SLAVE : PORT_SYNC_MASTER);
  TVReadBus (dev, 0x97, &res); 
  *port |= ((GetBit(res,1)|GetBit(res,3)) ? PORT_SYNC_OUT : PORT_SYNC_IN)
        |  (GetBit(res,2) ? PORT_VSYNC_LOW : PORT_VSYNC_HIGH) 
        |  (GetBit(res,0) ? PORT_HSYNC_LOW : PORT_HSYNC_HIGH);
}

void TVPh1GetPort (TVEncoderObj *this, int *port)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;
  I2CByte res;

  RAISE (MSG_DEBUG, "ph1 get port");
  TVPhGetPort (this, port);
  TVReadBus (dev, 0x54, &res); 
  *port |= ((res & 0x3) == 0x02 ? PORT_PCLK_LOW : PORT_PCLK_HIGH);
}

void TVPhInitRegs (TVEncoderObj *this)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;
  int i;

  RAISE (MSG_DEBUG, "ph init %04X", this->type);
  for (i = 0x01; i <= 0x16; i++) {
    TVWriteBus (dev, i, 0x00);  /* NV: 0x01 - 0x16 */ 
  }
  TVWriteBus (dev, 0x17, 0x1b); /* NV */ 
  TVWriteBus (dev, 0x18, 0x1b); /* NV */ 
  TVWriteBus (dev, 0x19, 0x1f); /* NV */ 
  TVWriteBus (dev, 0x1a, 0x46); /* NV */ 
  TVWriteBus (dev, 0x1b, 0x00); /* NV - monitor sense mode off */
  TVWriteBus (dev, 0x21, 0x1b); /* NV does it */
  TVWriteBus (dev, 0x26, 0xff); /* NV */ 
  TVWriteBus (dev, 0x27, 0x3f); /* NV */ 
  TVWriteBus (dev, 0x28, 0x19); /* NV */
  TVWriteBus (dev, 0x29, 0x1d); /* NV */
  TVWriteBus (dev, 0x2a, 0xff); /* NV - copy gen */
  TVWriteBus (dev, 0x2b, 0xff); /* NV - copy gen */
  TVWriteBus (dev, 0x2c, 0x0f); /* NV - copy gen off */
  TVWriteBus (dev, 0x2d, 0xb4); /* NV - default connector */
  for (i = 0x2e; i <= 0x36; i++) {
    TVWriteBus (dev, i, 0x00);   /* NV: 0x2e - 0x36*/ 
  }
  TVWriteBus (dev, 0x38, 0x1a); /* NV */
  TVWriteBus (dev, 0x39, 0x1a); /* NV */
  TVWriteBus (dev, 0x3a, 0x00); /* NV */ 
  TVWriteBus (dev, 0x55, 0x0f); /* NV */
  TVWriteBus (dev, 0x56, 0xc3); /* NV */
  TVWriteBus (dev, 0x57, 0x00); /* NV */
  TVWriteBus (dev, 0x58, 0x02); /* NV */
  TVWriteBus (dev, 0x59, 0x30); /* NV */
  TVWriteBus (dev, 0x6c, 0x01); /* NV */
  TVWriteBus (dev, 0x6d, 0x20); /* NV */
  TVWriteBus (dev, 0x6f, 0x42); /* NV ttxen=0 ccen=01 sccln=2 */
  TVWriteBus (dev, 0x75, 0xb0); /* NV - csynca */ 
  TVWriteBus (dev, 0xa2, 0x10); /* NV */
  TVWriteBus (dev, 0xa3, 0x80); /* NV */
  TVWriteBus (dev, 0xa4, 0x80); /* NV */
  TVWriteBus (dev, 0xfc, 0x03); /* NV */ 
  TVWriteBus (dev, 0xfd, 0x98); /* NV */ 
}

void TVPh1InitRegs (TVEncoderObj *this, int port)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;

  RAISE (MSG_DEBUG, "ph1 init %04X", this->type);
  TVPh1SetPort (this, port);
  TVPhInitRegs (this);
  TVWriteBus (dev, 0x37, 0x00); /* NV */ 
  TVWriteBus (dev, 0x54, 0x01); /* NV - edge1: falling, edge2: rising */
}

/*
  Registers:
  ----
  0x28: Starting point of burst in clock cycles
  0x29: Sync reset enable, burst end
  0x2d: Output port control
  0x38: Gain luminance for RGB
  0x39: Gain color difference for RGB
  0x5a: Chrominance phase
  0x5b: Gain U
  0x5c: Gain V
  0x5d: Gain U MSB, black level
  0x5e: Gain V MSB, blanking level
  0x5f: Blanking level, CCR
  0x61: Standard control
  0x62: Burst amplitude
  0x63: Subcarrier 0
  0x64: Subcarrier 1
  0x65: Subcarrier 2
  0x66: Subcarrier 3
  0x6E: Multi Control (Field length FLC, Carrier reset PHRES, ...)
  0x70: Active display window horizontal start
  0x71: Active display window horizontal end
  0x72: Active display window horizontal values MSB
  0x7a: First active line
  0x7b: Last active line
  0x7c: MSBs for fal and lal, teletext mode
  0x81: Pixel clock 0
  0x82: Pixel clock 1
  0x83: Pixel clock 2
  0x90: Horizontal offset
  0x91: Horizontal pixel count
  0x92: Vertical offset (odd)
  0x93: Vertical offset (even)
  0x94: MSBs for pcl, pixel number and vertical offsets
  0x95: Line number (vertical pixel count)
  0x96: MSB for line number, Scaler control
  0x98: Line length
  0x99: Line lenth MSB, Input delay
  0x9a: Horizontal increment
  0x9b: Vertical increment
  0x9c: MSBs for horizontal and vertical increments
  0x9d: Weighting factor (odd)
  0x9e: Weighting factor (even)
  0x9f: Weighting factor MSBs
  0xa0: Vertical line skip
  0xa1: Vertical line MSB, blank enable
  0xa2: Border color luminance
  0xa3: Border color chrominance red
  0xa4: Border color chrominance blue
*/

void TVPh1SetRegs (TVEncoderObj *this, TVEncoderRegs *r, TVState state)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;
  TVPhMacro *m;
  int blckl, blnnl;

  RAISE (MSG_DEBUG, "nv set ph %04X", this->type);
  m = TVMacroPhRegs (r);
  blckl = r->ph1.blckl;
  blnnl = r->ph1.blnnl;
  if (m) {
    blckl -= m->blckl_diff;
    blnnl -= m->blnnl_diff;
    if (blckl < 0) blckl = 0;
    if (blnnl < 0) blnnl = 0;
  }
  TVWriteBus (dev, 0x28, SetBitField(r->ph1.bs,5:0,5:0));
  TVWriteBus (dev, 0x29, SetBitField(r->ph1.be,5:0,5:0)
 	               | 0x00 /* sres=0 ttx pin */ );
  TVWriteBus (dev, 0x2d, (r->ph1.flags2 * PH_FLAG2_MASK)
 	               | 0x04 /* clk2en=1, encoff=0 */ );
  TVWriteBus (dev, 0x38, SetBitField(r->ph1.gy,4:0,4:0));
  TVWriteBus (dev, 0x39, SetBitField(r->ph1.gcd,4:0,4:0));

  TVWriteBus (dev, 0x5a, Set8Bits(r->ph1.chps));
  TVWriteBus (dev, 0x5b, Set8Bits(r->ph1.gainu));
  TVWriteBus (dev, 0x5c, Set8Bits(r->ph1.gainv));
  TVWriteBus (dev, 0x5d, SetBitField(blckl,5:0,5:0)
 	               | SetBitField(r->ph1.gainu,8:8,7:7));
  TVWriteBus (dev, 0x5e, SetBitField(blnnl,5:0,5:0)
 	               | SetBitField(r->ph1.gainv,8:8,7:7));
  TVWriteBus (dev, 0x5f, SetBitField(r->ph1.blnnl,5:0,5:0) /* blnvb */
                       | SetBitField(r->ph1.ccrs,1:0,7:6));
  TVWriteBus (dev, 0x61, Set8Bits(r->ph1.flags1 & PH_FLAG1_MASK));

  /* FIXME: State for DOWN flags !! Always up at the moment */
  /* #define PH_FLAG1_DOWNA (1 << 6)
     #define PH_FLAG1_DOWND (1 << 7) */
  TVWriteBus (dev, 0x62, SetBitField(r->ph1.bsta,6:0,6:0));
  TVWriteBus (dev, 0x63, SetBitField(r->ph1.fsc,7:0,7:0));
  TVWriteBus (dev, 0x64, SetBitField(r->ph1.fsc,15:8,7:0));
  TVWriteBus (dev, 0x65, SetBitField(r->ph1.fsc,23:16,7:0));
  TVWriteBus (dev, 0x66, SetBitField(r->ph1.fsc,31:24,7:0));
  TVWriteBus (dev, 0x6e, SetBitField(r->ph1.phres,1:0,5:4)
                       | SetBitField(r->ph1.flc,1:0,1:0)
 	               | 0x00 /* blckon, ldel are zero */ );
  TVWriteBus (dev, 0x70, Set8Bits(r->ph1.adwhs));
  TVWriteBus (dev, 0x71, Set8Bits(r->ph1.adwhe));
  TVWriteBus (dev, 0x72, SetBitField(r->ph1.adwhs,10:8,2:0)
                       | SetBitField(r->ph1.adwhe,10:8,6:4));
  TVWriteBus (dev, 0x7a, Set8Bits(r->ph1.fal));
  TVWriteBus (dev, 0x7b, Set8Bits(r->ph1.lal));
  TVWriteBus (dev, 0x7c, SetBitField(r->ph1.fal,8:8,4:4)
                       | SetBitField(r->ph1.lal,8:8,6:6)
 	               | 0x00 /* TTX */ );
  TVWriteBus (dev, 0x81, SetBitField(r->ph1.pcl,7:0,7:0));
  TVWriteBus (dev, 0x82, SetBitField(r->ph1.pcl,15:8,7:0));
  TVWriteBus (dev, 0x83, SetBitField(r->ph1.pcl,23:16,7:0));
  TVWriteBus (dev, 0x90, Set8Bits(r->ph1.xofs));
  TVWriteBus (dev, 0x91, Set8Bits(r->ph1.xpix));
  TVWriteBus (dev, 0x92, Set8Bits(r->ph1.yofso));
  TVWriteBus (dev, 0x93, Set8Bits(r->ph1.yofse));
  TVWriteBus (dev, 0x94, SetBitField(r->ph1.yofse,9:8,7:6)
                       | SetBitField(r->ph1.yofso,9:8,5:4)
                       | SetBitField(r->ph1.xpix,9:8,3:2)
	               | SetBitField(r->ph1.xofs,9:8,1:0));
  TVWriteBus (dev, 0x95, Set8Bits(r->ph1.ypix));
  TVWriteBus (dev, 0x96, SetBitField(r->ph1.ypix,9:8,1:0)
 	               | SetBitFlag(r->ph1.flags1,PH_FLAG1_YFIL,3)
 	               | SetBitField(this->hwstate,6:5,6:5)
	               | 0x00 /* rest all zero */ );
  TVWriteBus (dev, 0x98, Set8Bits(r->ph1.hlen));
  /* hlen extra bit for SAA7104, 0 in SAA7102 */
  TVWriteBus (dev, 0x99, SetBitField(r->ph1.hlen,11:8,3:0)
	               | SetBitField(r->ph1.idel,3:0,7:4));
  TVWriteBus (dev, 0x9a, Set8Bits(r->ph1.xinc));
  TVWriteBus (dev, 0x9b, Set8Bits(r->ph1.yinc));
  TVWriteBus (dev, 0x9c, SetBitField(r->ph1.xinc,11:8,3:0)
	               | SetBitField(r->ph1.yinc,11:8,7:4));
  TVWriteBus (dev, 0x9d, Set8Bits(r->ph1.yiwgto));
  TVWriteBus (dev, 0x9e, Set8Bits(r->ph1.yiwgte));
  TVWriteBus (dev, 0x9f, SetBitField(r->ph1.yiwgto,11:8,3:0)
	               | SetBitField(r->ph1.yiwgte,11:8,7:4));
  TVWriteBus (dev, 0xa0, Set8Bits(r->ph1.yskip));
  TVWriteBus (dev, 0xa1, SetBitField(r->ph1.yskip,11:8,3:0)
 	               | 0x00 /* blen=0 */ );
  TVWriteBus (dev, 0xa2, Set8Bits(r->ph1.bcy));
  TVWriteBus (dev, 0xa3, Set8Bits(r->ph1.bcu));
  TVWriteBus (dev, 0xa4, Set8Bits(r->ph1.bcv));

  if (m) {
    TVWriteBus (dev, 0x60, SetBitField(m->reg60,7:0,7:0));
    TVWriteBus (dev, 0x7d, Set8Bits(m->reg7d & ~0x16));
    TVWriteBus (dev, 0x01, SetBitField(m->reg01,5:0,5:0));
    TVWriteBus (dev, 0x03, SetBitField(m->reg03,5:0,5:0));
    TVWriteBus (dev, 0x04, SetBitField(m->reg04,5:0,5:0));
    TVWriteBus (dev, 0x05, SetBitField(m->reg05,7:0,7:0));
    TVWriteBus (dev, 0x06, SetBitField(m->reg06,7:0,7:0));
    TVWriteBus (dev, 0x07, SetBitField(m->reg07,7:0,7:0));
    TVWriteBus (dev, 0x08, SetBitField(m->reg08,1:0,1:0));
    TVWriteBus (dev, 0x10, SetBitField(m->reg10,5:0,5:0));
    TVWriteBus (dev, 0x11, SetBitField(m->reg11,7:0,7:0));
    TVWriteBus (dev, 0x12, SetBitField(m->reg12,7:0,7:0));
    TVWriteBus (dev, 0x14, SetBitField(m->reg14,7:0,7:0));
    TVWriteBus (dev, 0x15, SetBitField(m->reg15,6:0,6:0));
    TVWriteBus (dev, 0x20, Set8Bits(m->reg20 & ~0x40));
    TVWriteBus (dev, 0x21, SetBitField(m->reg21,7:0,7:0));
    TVWriteBus (dev, 0x24, SetBitField(m->reg24,7:0,7:0));
    TVWriteBus (dev, 0x25, SetBitField(m->reg25,6:0,6:0));
    TVWriteBus (dev, 0x30, SetBitField(m->reg30,5:0,5:0));
    TVWriteBus (dev, 0x31, SetBitField(m->reg31,3:0,3:0));
    TVWriteBus (dev, 0x33, SetBitField(m->reg33,7:0,7:0));
    TVWriteBus (dev, 0x36, SetBitField(m->reg36,7:0,7:0));
  }
}

void TVPh1GetRegs (TVEncoderObj *this, TVEncoderRegs *r)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;
  I2CByte regs[0x100];
  int i;

  for (i = 0x00; i < 0xa4; i++) TVReadBus (dev, i, &regs[i]);
  r->ph1.adwhs  = Set8Bits(regs[0x70])
               | SetBitField(regs[0x72],2:0,10:8);
  r->ph1.adwhe  = Set8Bits(regs[0x71])           
               | SetBitField(regs[0x72],6:4,10:8);
  r->ph1.xofs   = Set8Bits(regs[0x90])
               | SetBitField(regs[0x94],1:0,9:8);
  r->ph1.xpix   = Set8Bits(regs[0x91])          
               | SetBitField(regs[0x94],3:2,9:8);
  r->ph1.xinc   = Set8Bits(regs[0x9a])
               | SetBitField(regs[0x9c],3:0,11:8);
  r->ph1.hlen   = Set8Bits(regs[0x98])
               | SetBitField(regs[0x99],2:0,10:8);
  r->ph1.fal    = Set8Bits(regs[0x7a])
               | SetBitField(regs[0x7c],4:4,8:8);
  r->ph1.lal    = Set8Bits(regs[0x7b])          
               | SetBitField(regs[0x7c],6:6,8:8);
  r->ph1.yinc   = Set8Bits(regs[0x9b])
               | SetBitField(regs[0x9c],7:4,11:8);
  r->ph1.yskip  = Set8Bits(regs[0xa0])
               | SetBitField(regs[0xa1],3:0,11:8);
  r->ph1.yofso  = Set8Bits(regs[0x92])
               | SetBitField(regs[0x94],5:4,9:8);
  r->ph1.yofse  = Set8Bits(regs[0x93])          
               | SetBitField(regs[0x94],7:6,9:8);
  r->ph1.ypix   = Set8Bits(regs[0x95]) 
               | SetBitField(regs[0x96],1:0,9:8);
  r->ph1.yiwgto = Set8Bits(regs[0x9d])
               | SetBitField(regs[0x9f],3:0,11:8);
  r->ph1.yiwgte = Set8Bits(regs[0x9e])
               | SetBitField(regs[0x9f],7:4,11:8);
  r->ph1.pcl    = (unsigned long) Set8Bits(regs[0x81]) 
               | (unsigned long) Set8Bits(regs[0x82]) << 8
               | (unsigned long) Set8Bits(regs[0x83]) << 16;
  r->ph1.fsc    = (unsigned long) Set8Bits(regs[0x63]) 
               | (unsigned long) Set8Bits(regs[0x64]) << 8
               | (unsigned long) Set8Bits(regs[0x65]) << 16
               | (unsigned long) Set8Bits(regs[0x66]) << 24; 
  r->ph1.idel   = SetBitField(regs[0x99],7:4,3:0);
  r->ph1.bs     = SetBitField(regs[0x28],5:0,5:0);
  r->ph1.be     = SetBitField(regs[0x29],5:0,5:0);
  r->ph1.bsta   = SetBitField(regs[0x62],6:0,6:0);
  r->ph1.blckl  = SetBitField(regs[0x5d],5:0,5:0);
  r->ph1.blnnl  = SetBitField(regs[0x5e],5:0,5:0);
  r->ph1.chps   = Set8Bits(regs[0x5a]); 
  r->ph1.gy     = SetBitField(regs[0x38],4:0,4:0);
  r->ph1.gcd    = SetBitField(regs[0x39],4:0,4:0);
  r->ph1.bcy    = Set8Bits(regs[0xa2]);
  r->ph1.bcu    = Set8Bits(regs[0xa3]);
  r->ph1.bcv    = Set8Bits(regs[0xa4]);
  r->ph1.ccrs   = SetBitField(regs[0x5f],7:6,1:0);
  r->ph1.gainu  = Set8Bits(regs[0x5b])
               | SetBitField(regs[0x5d],7:7,8:8);
  r->ph1.gainv  = Set8Bits(regs[0x5c])
               | SetBitField(regs[0x5e],7:7,8:8);
  r->ph1.flc    = SetBitField(regs[0x6e],1:0,1:0);
  r->ph1.phres  = SetBitField(regs[0x6e],5:4,1:0);
  r->ph1.flags1 = (Set8Bits(regs [0x61]) & PH_FLAG1_MASK)
               | GetBitFlag(regs[0x96],3,PH_FLAG1_YFIL);
  r->ph1.flags2 = regs[0x2d] & PH_FLAG2_MASK;
  r->ph1.macro = 0;   
}

void TVPh1SetState (TVEncoderObj *this, TVEncoderRegs *r, TVState state)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;

  /* FIXME */
  /* 0x6e blckon, ... */
  /* FIXME 0x61 dacoff/chip off ... but need connect state ... always on*/

  tvState = state;
  switch (state)
  {
    case TV_OFF: 
      TVWriteBus (dev, 0x3a, 0x80); /* NV */ 
      TVWriteBus (dev, 0x61, Set8Bits(r->ph1.flags1 & PH_FLAG1_MASK));
      break;
    case TV_BARS: 
      TVWriteBus (dev, 0x3a, 0x80); /* NV + cbenb=1 */ 
      TVWriteBus (dev, 0x61, Set8Bits(r->ph1.flags1 & PH_FLAG1_MASK));
      break;
    case TV_ON: 
      TVWriteBus (dev, 0x3a, 0x00); /* NV */ 
      TVWriteBus (dev, 0x61, Set8Bits(r->ph1.flags1 & PH_FLAG1_MASK));
      break;
    default:
      break;
  }
}

long TVPh1GetStatus (TVEncoderObj *this, int index)
{ 
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;
  I2CByte fifo, status;

  TVReadBus (dev, 0x00, &status); /* FIFO status */
  TVReadBus (dev, 0x80, &fifo); /* FIFO status */
  
  return SetBitField (status,1:0,1:0) | SetBitField (fifo,1:0,3:2) |
    ((tvState == TV_ON) ? 0x100 : 0);
}

TVConnect TVPh1GetConnect (TVEncoderObj *this)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;
  TVConnect status = CONNECT_BOTH;
  I2CByte msense;

  /* Enable monitor sensing temporarily */
  TVWriteBus (dev, 0x1a, 0x2c); /* NV */
  TVWriteBus (dev, 0x1b, 0x80);

  /* Read monitor sense data */
  TVReadBus (dev, 0x1b, &msense); /* Monitor sense */

  RAISE (MSG_DEBUG, "TVPh1GetConnect %02X", msense);

  switch (msense & 0x7) { /* 0=active, 1=inactive */
    case 0x06: 
      status = CONNECT_COMPOSITE;
      break;
    case 0x01:
      status = CONNECT_SVIDEO;
      break;
    case 0x05:
      status = CONNECT_CONVERT;
      break;
    case 0x07:
      RAISE (MSG_WARNING, "No TV seems to be connected.");
      /* no break */
    default:
      status = CONNECT_BOTH;
      break;
  }

  /* Disable monitor sensing */
  TVWriteBus (dev, 0x1b, 0x00 );

  return status;
}

/*
 * Check for Philips chip on device dev. Return version string if found, 
 * NULL otherwise.
 */

char *TVDetectPhilips (I2CDevPtr dev, TVChip *encoder)
{
  I2CByte cid, ver;
  char *name;
  static char version [50];

#ifdef FAKE_PROBE_PHILIPS
#ifdef FAKE_PROBE_ID
  *encoder = FAKE_PROBE_ID;
#else
  *encoder = TV_PHILIPS;
#endif
  snprintf (version, 50, "Philips Fake (%1s:%02X)", I2C_ID(dev));
  return version;
#else
#ifdef FAKE_I2C
  return NULL;
#endif
#endif

  tvBusOk = TRUE;  
  TVReadBus (dev, 0x00, &ver); 
  TVReadBus (dev, 0x1c, &cid); 
  ver = (ver & 0xe0 ) >> 5; /* 3 highest bits are version info */
  RAISE (MSG_DEBUG, "ph check ver=%01x cid=%02x (%s)", ver, cid, 
	 tvBusOk?"ok":"err");
  *encoder = TV_PHILIPS;
  if (ver != 0x2 && ver != 0x5) {
    snprintf (version, 50, "Unknown Philips #%i (%1s:%02X)", ver, I2C_ID(dev));
    return version;
  }
  if (!tvBusOk) return NULL;
  /* Probe addr 0x40 and 0x42 to identify SAA7108, SAA7109 */
  if (TVProbeBus (dev->pI2CBus, 0x40) || 
      TVProbeBus (dev->pI2CBus, 0x42)) {
    switch (cid) {
      case 0x02: name = "7108";  *encoder = TV_PHILIPS_7108; break;
      case 0x03: name = "7109";  *encoder = TV_PHILIPS_7109; break;
      case 0x04: name = "7108A"; *encoder = TV_PHILIPS_7108A; break;
      case 0x05: name = "7109A"; *encoder = TV_PHILIPS_7109A; break;
      default:   name = NULL; break;
    }
  } else {
    switch (cid) {
      case 0x02: name = "7102"; *encoder = TV_PHILIPS_7102; break;
      case 0x03: name = "7103"; *encoder = TV_PHILIPS_7103; break;
      case 0x04: name = "7104"; *encoder = TV_PHILIPS_7104; break;
      case 0x05: name = "7105"; *encoder = TV_PHILIPS_7105; break;
      default:   name = NULL; break;
    }
  }
  if (name) {
    TVReadBus (dev, 0x00, &ver); 
    if (!tvBusOk) return NULL;
    snprintf (version, 50, "Philips SAA%s (%1s:%02X)", name, I2C_ID(dev));
    return version;
  } else {
    snprintf (version, 50, "Philips (id = %02X) (%1s:%02X)", 
	      cid, I2C_ID(dev));
    return version;
  }
}

TVEncoderObj tvPh1Template = {
  type: TV_PHILIPS_MODEL1, ctrl: NULL, minClock: 10000, maxClock: 44000, 
  Create:     TVPhCreate,
  InitRegs:   TVPh1InitRegs, 
  SetRegs:    TVPh1SetRegs, 
  GetRegs:    TVPh1GetRegs, 
  SetPort:    TVPh1SetPort,
  GetPort:    TVPh1GetPort,
  SetState:   TVPh1SetState,
  GetConnect: TVPh1GetConnect, 
  GetStatus:  TVPh1GetStatus
};
