/*	Dialog_Box

PIRL CVS ID: Dialog_Box.java,v 1.12 2012/04/16 06:22:58 castalia Exp

Copyright (C) 2003-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Viewers;

import	javax.swing.JOptionPane;
import	java.awt.Component;
import	java.awt.Frame;
import	java.awt.Toolkit;

/**	The <I>Dialog_Box</I> class provides a collection of static methods
	that are convenient for producing commonly used Dialog_Boxes.
<P>
	@author	Bradford Castalia, UA/PIRL
	@version 1.12
*/
public class Dialog_Box
	extends JOptionPane
{
private static final String
	ID = "PIRL.Viewers.Dialog_Box (1.12 2012/04/16 06:22:58)";

/**	The maximum number of characters per line in a dialog box.
	A width <= 0 is an unlimited width.
*/
public static int
	MAX_LINE_LENGTH		= 70;

//	For beep.
private static final Toolkit
	TOOLKIT				= Toolkit.getDefaultToolkit ();

//	For Fit_to_Width.
private static final String
	NL					= System.getProperty ("line.separator");

/*==============================================================================
	Constructors
*/
//	This class only contains static functions.
private Dialog_Box ()
{}

/*==============================================================================
	Functions
*/
/**	Displays an INFORMATION_MESSAGE MessageDialog box with a "Notice"
	title and an OK button.
<P>
	@param	message	A String message that will be displayed in the
		dialog box.
	@param	parent	The parent Component relative to which the dialog
		box will be positioned. This may be null in which case the
		Dialog_Box will be centered on the display.
*/
public static void Notice
	(
	String		message,
	Component	parent
	)
{
showMessageDialog
	(
	parent,
	Fit_to_Width (message),
	"Notice",
	INFORMATION_MESSAGE
	);
}

/**	Displays an INFORMATION_MESSAGE MessageDialog box centered on the
	screen with a "Notice" title and an OK button.
<P>
	@param	message	A String message that will be displayed in the
		dialog box.
*/
public static void Notice (String message)
{Notice (message, null);}

/**	Displays a WARNING_MESSAGE MessageDialog box, preceeded by an
	audible beep, with a "Warning" title and an OK button.
<P>
	@param	message	A String message that will be displayed in the
		dialog box.
	@param	parent	The parent Component relative to which the dialog
		box will be positioned. This may be null in which case the
		Dialog_Box will be centered on the display.
*/
public static void Warning
	(
	String		message,
	Component	parent
	)
{
TOOLKIT.beep ();
showMessageDialog
	(
	parent,
	Fit_to_Width (message),
	"Warning",
	WARNING_MESSAGE
	);
}

/**	Displays a WARNING_MESSAGE MessageDialog box centered on the
	screen, preceeded by an audible beep, with a "Warning" title and an
	OK button.
<P>
	@param	message	A String message that will be displayed in the
		dialog box.
*/
public static void Warning (String message)
{Warning (message, null);}

/**	Displays an ERROR_MESSAGE MessageDialog box, preceeded by an
	audible beep, with an "Error" title and an OK button.
<P>
	@param	message	A String message that will be displayed in the
		dialog box.
	@param	parent	The parent Component relative to which the dialog
		box will be positioned. This may be null in which case the
		Dialog_Box will be centered on the display.
*/
public static void Error
	(
	String		message,
	Component	parent
	)
{
TOOLKIT.beep ();
showMessageDialog
	(
	parent,
	Fit_to_Width (message),
	"Error",
	ERROR_MESSAGE
	);
}

/**	Displays an ERROR_MESSAGE MessageDialog box centered on the screen,
	preceeded by an audible beep, with an "Error" title and an OK
	button.
<P>
	@param	message	A String message that will be displayed in the
		dialog box.
*/
public static void Error (String message)
{Error (message, null);}

/**	Displays a ConfirmDialog box with a "Confirm" title and an OK and
	Cancel button.
<P>
	@param	message	A String message that will be displayed in the
		dialog box.
	@param	parent	The parent Component relative to which the dialog
		box will be positioned. This may be null in which case the
		Dialog_Box will be centered on the display.
	@return	true if the OK button was pressed; false otherwise.
*/
public static boolean Confirm
	(
	String		message,
	Component	parent
	)
{
return showConfirmDialog
	(
	parent,
	Fit_to_Width (message),
	"Confirm",
	OK_CANCEL_OPTION
	)
	== OK_OPTION;
}

/**	Displays a ConfirmDialog box centered on the screen with a
	"Confirm" title and an OK and Cancel button.
<P>
	@param	message	A String message that will be displayed in the
		dialog box.
	@return	true if the OK button was pressed; false otherwise.
*/
public static boolean Confirm (String message)
{return Confirm (message, null);}

/**	Displays a ConfirmDialog box with a "Check" title and a Yes, No
	and Cancel button.
<P>
	@param	message	A String message that will be displayed in the
		dialog box.
	@param	parent	The parent Component relative to which the dialog
		box will be positioned. This may be null in which case the
		Dialog_Box will be centered on the display.
	@return	1 if the Yes button was pressed, -1 if the No button was
		pressed, or 0 if the Cancel button was pressed.
*/
public static int Check
	(
	String		message,
	Component	parent
	)
{
int check = showConfirmDialog
	(
	parent,
	Fit_to_Width (message),
	"Check",
	YES_NO_CANCEL_OPTION
	);
switch (check)
	{
	case YES_OPTION:	return 1;
	case NO_OPTION:		return -1;
	default:			return 0;
	}
}

/**	Displays a ConfirmDialog box centered on the screen with a "Check"
	title and a Yes, No and Cancel button.
<P>
	@param	message	A String message that will be displayed in the
		dialog box.
	@return	1 if the Yes button was pressed, -1 if the No button was
		pressed, or 0 if the Cancel button was pressed.
*/
public static int Check (String message)
{return Check (message, null);}

/**	Displays a Password_Dialog with a password input field preceded by
	a prompt.
<p>
	@param	prompt	The text preceding the password input field. If
		null, "Password" will be used by default.
	@param	parent	The parent Frame relative to which the dialog
		box will be positioned. This may be null.
*/
public static String Password
	(
	String		prompt,
	Frame		parent
	)
{
Password_Dialog
	password_dialog = new Password_Dialog ("Password", prompt, parent);
password_dialog.setVisible (true);

String
	password_string = null;
char[]
	password = password_dialog.Password ();
if (password != null)
	{
	password_string = new String (password);
	password_dialog.Erase_Password ();
	}
return password_string;
}

/**	Displays a Password_Dialog with a password input field preceded by a
	prompt.
<p>
	@param	prompt	The text preceding the password input field. If
		null, "Password" will be used by default.
*/
public static String Password
	(
	String	prompt
	)
{return Password (prompt, null);}

/*==============================================================================
	Helpers
*/
/**	Fits (wraps using NL sequences) a String to a given width.
<P>
	A NL replaces the last whitespace sequence before the width
	of the current line is reached. If there is no whitespace, then
	the NL is inserted at the width. A width <= 0 is an
	unlimited width.
*/
public static String Fit_to_Width
	(
	String	in_string,
	int		width
	)
{
if (in_string == null)
	return "null";
int
	total = in_string.length ();
if (width >= total ||
	width <= 0)
	return in_string;

StringBuffer
	out_string = new StringBuffer (in_string);
int
	NL_length = NL.length (),
	count = 0,
	column = 0,
	break_column = 0;
while (column < total)
	{
	if ((column + NL_length) < total &&
		out_string.substring (column, column + NL_length).equals (NL))
		{
		//	Line break.
		column += NL_length;
		count = break_column = 0;
		}
	else if (count < width)
		{
		if (out_string.charAt (column) == ' ')
			break_column = column;
		column++;
		count++;
		}
	else
		{
		//	Wrap the long line.
		while (break_column > 0 &&
				out_string.charAt (break_column) == ' ')
			{
			//	Delete the break space(s).
			out_string.delete (break_column, break_column + 1);
			break_column--;
			column--;
			total--;
			}
		if (break_column == 0)
			break_column = column;
		else
			column = ++break_column;
		if (column != 0)
			{
			//	Insert a line break.
			out_string.insert (column, NL);
			column += NL_length;
			total += NL_length;
			}
		count = break_column = 0;
		}
	}
return out_string.toString ();
}

/**	Fits a String to the MAX_LINE_LENGTH.
<P>
	@param	string	The String to be fitted.
	@return	The fitted String.
	@see	#Fit_to_Width(String, int)
*/
public static String Fit_to_Width (String string)
{return Fit_to_Width (string, MAX_LINE_LENGTH);}

}	//	End of Dialog_Box class.
