# -*- coding:utf-8 -*-
"""
/***************************************************************************
Module to generate prepared APIs for calltips and auto-completion.
                             -------------------
begin                : 2013-08-29
copyright            : (C) 2013 Larry Shaffer
email                : larrys (at) dakotacarto (dot) com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
Portions of this file contain code from Eric4 APIsManager module.
"""

import sys
import os

from PyQt5.Qsci import QsciLexerPython, QsciAPIs
from PyQt5.QtWidgets import QApplication
from PyQt5.QtCore import QObject


class PrepareAPIs(QObject):

    def __init__(self, api_lexer, api_files, pap_file):
        QObject.__init__(self)
        self._api = None
        self._api_files = api_files
        self._api_lexer = api_lexer
        self._pap_file = pap_file

    def _clearLexer(self):
        self.qlexer = None

    def _stopPreparation(self):
        if self._api is not None:
            self._api.cancelPreparation()
        self._api = None
        sys.exit(1)

    def _preparationFinished(self):
        self._clearLexer()
        try:
            if os.path.exists(self._pap_file):
                os.remove(self._pap_file)
            prepd = self._api.savePrepared(str(self._pap_file))
            self._api = None
            sys.exit(0 if prepd else 1)
        except Exception as err:
            self._api = None
            sys.exit(1)

    def prepareAPI(self):
        try:
            self._api = QsciAPIs(self._api_lexer)
            self._api.apiPreparationFinished.connect(self._preparationFinished)
            for api_file in self._api_files:
                self._api.load(str(api_file))
            self._api.prepare()
        except Exception as err:
            self._api = None
            sys.exit(1)


if __name__ == '__main__':
    app = QApplication(sys.argv)
    args = app.arguments()

    if len(args) != 4:
        print(f'Usage: python {sys.argv[0]} [qt options] <pap_file> <apis_src_dir> <api_bin_dir>')
        sys.exit(1)
    pap_file = args[1]
    api_src_dir = args[2]
    api_bin_dir = args[3]

    api_files = [
        os.path.join(api_bin_dir, 'PyQGIS.api'),
        os.path.join(api_src_dir, 'Python-3.6.api'),
        os.path.join(api_src_dir, 'PyQt5.api'),
        os.path.join(api_src_dir, 'OSGeo_GEOS-3.6.2.api'),
        os.path.join(api_src_dir, 'OSGeo_GDAL-OGR-2.2.3.api')
    ]

    api_lexer = QsciLexerPython()
    prepap = PrepareAPIs(api_lexer, api_files, pap_file)
    prepap.prepareAPI()

    sys.exit(app.exec_())
