// Copyright (C) 2016 The Qt Company Ltd.
// SPDX-License-Identifier: LicenseRef-Qt-Commercial OR GPL-3.0-only WITH Qt-GPL-exception-1.0

#pragma once

#include "core_global.h"

#include <utils/id.h>

#include <QFile>
#include <QList>
#include <QSharedDataPointer>

namespace Utils { class FilePath; }

namespace Core {
class GeneratedFilePrivate;

class CORE_EXPORT GeneratedFile
{
public:
    enum Attribute {
        // Open this file in editor
        OpenEditorAttribute = 0x01,
        // Open project
        OpenProjectAttribute = 0x02,
        // File is generated by external scripts, do not write out, see BaseFileWizard::writeFiles()
        CustomGeneratorAttribute = 0x4,
        // File exists and the user indicated that he wants to keep it
        KeepExistingFileAttribute = 0x8,
        // Force overwriting of a file without asking the user to keep it
        ForceOverwrite = 0x10,
        // Mark the document temporary after opening the file
        TemporaryFile = 0x20
    };
    Q_DECLARE_FLAGS(Attributes, Attribute)

    GeneratedFile();
    explicit GeneratedFile(const Utils::FilePath &path);
    GeneratedFile(const GeneratedFile &);
    GeneratedFile &operator=(const GeneratedFile &);
    ~GeneratedFile();

    // Full path of the file should be created, or the suggested file name
    void setFilePath(const Utils::FilePath &p);
    Utils::FilePath filePath() const;

    void setPermissions(QFile::Permissions permissions);
    std::optional<QFile::Permissions> permissions() const;

    // Contents of the file (UTF8)
    QString contents() const;
    void setContents(const QString &c);

    QByteArray binaryContents() const;
    void setBinaryContents(const QByteArray &c);

    // Defaults to false (Text file).
    bool isBinary() const;
    void setBinary(bool b);

    // Id of editor to open the file with
    Utils::Id editorId() const;
    void setEditorId(Utils::Id id);

    bool write(QString *errorMessage) const;

    Attributes attributes() const;
    void setAttributes(Attributes a);

private:
    QSharedDataPointer<GeneratedFilePrivate> m_d;

    friend CORE_EXPORT QDebug &operator<<(QDebug &debug, const Core::GeneratedFile &file);
};

CORE_EXPORT QDebug &operator<<(QDebug &debug, const Core::GeneratedFile &file);

using GeneratedFiles = QList<GeneratedFile>;

} // namespace Core
