///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is sequential in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/geo.h"
#include "rheolef/geo_domain.h"
#include "rheolef/rheostream.h"
#include "rheolef/iorheo.h"

#include "geo_header.h"

namespace rheolef {

// =========================================================================
// cstors
// =========================================================================
template <class T, class M>
geo_base_rep<T,M>::geo_base_rep ()
 : geo_abstract_rep<T,M>(),
   _name("*nogeo*"),
   _version(0),
   _serial_number(0),
   _geo_element(),
   _map_dimension(0),
   _gs(),
   _domains(),
   _have_connectivity(false),
   _have_neighbour(false),
   _node(),
   _dimension(0),
   _sys_coord(space_constant::cartesian),
   _xmin(),
   _xmax(),
   _hmin(),
   _hmax(),
   _numbering("P1"),
   _locator(),
   _tracer_ray_boundary(),
   _nearestor()
{
}
template <class T, class M>
geo_base_rep<T,M>::geo_base_rep (const geo_base_rep<T,M>& o)
 : geo_abstract_rep<T,M>(o),
   _name                (o._name),
   _version             (o._version),
   _serial_number       (o._serial_number),
   _geo_element         (o._geo_element),
   _map_dimension       (o._map_dimension),
   _gs                  (o._gs),
   _domains             (o._domains),
   _have_connectivity   (o._have_connectivity),
   _have_neighbour      (o._have_neighbour),
   _node                (o._node),
   _dimension           (o._dimension),
   _sys_coord           (o._sys_coord),
   _xmin                (o._xmin),
   _xmax                (o._xmax),
   _hmin		(o._hmin),
   _hmax                (o._hmax),
   _numbering           (o._numbering),
   _locator             (o._locator),
   _tracer_ray_boundary (o._tracer_ray_boundary),
   _nearestor           (o._nearestor)
{
  trace_macro ("*** PHYSICAL COPY OF GEO_BASE_REP ***");
}
template <class T>
geo_rep<T,sequential>::geo_rep ()
 : geo_base_rep<T,sequential>()
{
}
template <class T>
geo_rep<T,sequential>::geo_rep (const geo_rep<T,sequential>& omega)
 : geo_base_rep<T,sequential>(omega)
{
  trace_macro ("*** PHYSICAL COPY OF GEO_REP<seq> ***");
}
template <class T>
geo_abstract_rep<T,sequential>*
geo_rep<T,sequential>::clone() const 
{
  trace_macro ("*** CLONE OF GEO_REP<seq> ***");
  typedef geo_rep<T,sequential> rep;
  return new_macro (rep(*this));
}
#ifdef _RHEOLEF_HAVE_MPI
template <class T>
geo_rep<T,distributed>::geo_rep ()
 : geo_base_rep<T,distributed>(),
   _inod2ios_dis_inod(),
   _ios_inod2dis_inod(),
   _ios_ige2dis_ige(),
   _ios_gs(),
   _igev2ios_dis_igev(),
   _ios_igev2dis_igev()
{
}
template <class T>
geo_rep<T,distributed>::geo_rep (const geo_rep<T,distributed>& o)
 : geo_base_rep<T,distributed>(o),
   _inod2ios_dis_inod(o._inod2ios_dis_inod),
   _ios_inod2dis_inod(o._ios_inod2dis_inod),
   _ios_ige2dis_ige  (o._ios_ige2dis_ige),
   _ios_gs           (o._ios_gs),
   _igev2ios_dis_igev(o._igev2ios_dis_igev),
   _ios_igev2dis_igev(o._ios_igev2dis_igev)
{
  trace_macro ("*** PHYSICAL COPY OF GEO_REP<dis> ***");
}
template <class T>
geo_abstract_rep<T,distributed>*
geo_rep<T,distributed>::clone() const 
{
  trace_macro ("*** CLONE OF GEO_REP<dis> ***");
  typedef geo_rep<T,distributed> rep;
  return new_macro (rep(*this));
}
#endif // _RHEOLEF_HAVE_MPI
// =========================================================================
// geo_load(name): handle name as "square[boundary]" or "square.boundary"
// =========================================================================
// geo(name) { base::operaor=(geo_load(name)); }
static
bool
is_domain (const std::string& name, std::string& bgd_name, std::string& dom_name)
{
  size_t n = name.length();
  trace_macro ("is_domain: n=" << n);
  if (n <= 2 || name[n-1] != ']') return false;
  size_t i = n-2;
  for (; i > 0 && name[i] != '[' && name[i] != '/'; i--) ;
  // ../../nfem/ptst/triangle_p10-v2 => name[i] == '/' : stop before "../../"
  if (i == 0 || i == n-2 || name[i] == '/') return false;
  // name[i] == '[' as "square[right]"
  trace_macro ("is_domain: i=" << i);
  bgd_name = name.substr(0,i); // range [0:i[
  dom_name = name.substr(i+1); // range [i+1:end[
  dom_name = dom_name.substr(0,dom_name.length()-1); // range [0:end-1[ : skip ']'
  return true;
}
template<class T, class M>
geo_basic<T,M>
geo_load (const std::string& name)
{
  trace_macro ("geo_load: " << name);
  std::string root_name = delete_suffix (delete_suffix(name, "gz"), "geo");
  std::string bgd_name, dom_name;
  if (is_domain(root_name,bgd_name,dom_name)) {
    trace_macro ("bgd_name="<<bgd_name<<", dom_name="<<dom_name);
    geo_basic<T,M> omega (bgd_name);
    return omega[dom_name];
  }
  // allocate a new geo_rep<mpi> object ; TODO geo_rep(name,comm)
  geo_rep<T,M>* ptr = new_macro((geo_rep<T,M>));
  ptr->load (name, communicator());
  geo_basic<T,M> omega;
  omega.geo_basic<T,M>::base::operator= (ptr);
  return omega;
}
// =========================================================================
// utility: vertex ownership follows node ownership, but dis_numbering differ
// for high order > 1 meshes. This function converts numbering.
// =========================================================================
template <class T, class M>
typename geo_base_rep<T,M>::size_type
geo_base_rep<T,M>::dis_inod2dis_iv (size_type dis_inod) const
{
  if (is_sequential<M>::value || order() == 1) return dis_inod;
  distributor vertex_ownership = geo_element_ownership(0);
  distributor   node_ownership = _node.ownership();
  size_type iproc          =   node_ownership.find_owner(dis_inod);
  size_type first_dis_inod =   node_ownership.first_index(iproc);
  size_type first_dis_iv   = vertex_ownership.first_index(iproc);
  size_type   inod = dis_inod - first_dis_inod;
  size_type     iv = inod;
  size_type dis_iv = first_dis_iv + iv;
  return dis_iv;
}
template <class T, class M>
typename geo_base_rep<T,M>::size_type
geo_base_rep<T,M>::dis_iv2dis_inod (size_type dis_iv) const
{
  if (is_sequential<M>::value || order() == 1) return dis_iv;
  distributor vertex_ownership = _gs.ownership_by_variant [reference_element::p];
  distributor   node_ownership = _gs.node_ownership;
  size_type iproc          = vertex_ownership.find_owner(dis_iv);
  size_type first_dis_iv   = vertex_ownership.first_index(iproc);
  size_type first_dis_inod =   node_ownership.first_index(iproc);
  size_type       iv = dis_iv - first_dis_iv;
  size_type     inod = iv;
  size_type dis_inod = first_dis_inod + inod;
  return dis_inod;
}
// --------------------------------------------------------------------------
// accessors to distributed data
// --------------------------------------------------------------------------
template <class T, class M>
typename geo_base_rep<T,M>::const_reference  
geo_base_rep<T,M>::dis_get_geo_element (size_type dim, size_type dis_ige) const
{
  if (is_sequential<M>::value) return get_geo_element (dim, dis_ige);
  if (_gs.ownership_by_dimension[dim].is_owned (dis_ige)) {
    size_type first_dis_ige = _gs.ownership_by_dimension[dim].first_index();
    size_type ige = dis_ige - first_dis_ige;
    return get_geo_element (dim, ige);
  }
  // element is owned by another proc ; get its variant and then its index-variant igev
  size_type variant  = _gs.dis_ige2variant  (dim, dis_ige);
  size_type dis_igev = _gs.dis_ige2dis_igev (dim, variant, dis_ige);
  return _geo_element [variant].dis_at (dis_igev);
}
// =========================================================================
// geo_base_rep members
// =========================================================================
template <class T, class M>
geo_base_rep<T,M>::~geo_base_rep()
{
#ifdef TO_CLEAN
  trace_macro ("dstor geo_base_rep: name="<<name()<<", dim="<<dimension()<<", map_dim="<<map_dimension()<<" size="<<size());
#endif // TO_CLEAN
}
template <class T, class M>
std::string
geo_base_rep<T,M>::name() const
{
  if (_serial_number == 0) return _name;
  return _name + "-" + itos(_serial_number);
}
template <class T, class M>
bool
geo_base_rep<T,M>::have_domain_indirect (const std::string& name) const
{
  for (typename std::vector<domain_indirect_basic<M> >::const_iterator
        iter = _domains.begin(), last = _domains.end(); iter != last; iter++) {
    const domain_indirect_basic<M>& dom = *iter;
    if (name == dom.name()) return true;
  }
#ifdef TO_CLEAN
  // predefined domains:
  if (name == "boundary" || name == "sides"  || name == "internal_sides") 
    return true;
#endif // TO_CLEAN
  return false;
}
template <class T, class M>
const domain_indirect_basic<M>&
geo_base_rep<T,M>::get_domain_indirect (const std::string& name) const
{
  for (typename std::vector<domain_indirect_basic<M> >::const_iterator
        iter = _domains.begin(), last = _domains.end(); iter != last; iter++) {
    const domain_indirect_basic<M>& dom = *iter;
    if (name == dom.name()) return dom;
  }
  error_macro ("undefined domain \""<<name<<"\" in mesh \"" << _name << "\"");
  return *(_domains.begin()); // not reached
}
template <class T, class M>
void
geo_base_rep<T,M>::insert_domain_indirect (const domain_indirect_basic<M>& dom) const
{
  for (typename std::vector<domain_indirect_basic<M> >::iterator
        iter = _domains.begin(), last = _domains.end(); iter != last; iter++) {
    domain_indirect_basic<M>& curr_dom = *iter;
    if (dom.name() == curr_dom.name()) {
      // a domain with the same name already exists: replace it
      curr_dom = dom;
      return;
    }
  }
  // insert a new domain:
  _domains.push_back (dom);
}
template <class T, class M>
typename geo_base_rep<T,M>::size_type
geo_base_rep<T,M>::size (size_type dim) const
{
  size_type sz = 0;
  for (size_type variant = reference_element::first_variant_by_dimension(dim);
                 variant < reference_element:: last_variant_by_dimension(dim); variant++) {
    sz += _geo_element [variant].size();
  }
  return sz;
}
template <class T, class M>
typename geo_base_rep<T,M>::size_type
geo_base_rep<T,M>::dis_size (size_type dim) const
{
  size_type sz = 0;
  for (size_type variant = reference_element::first_variant_by_dimension(dim);
                 variant < reference_element:: last_variant_by_dimension(dim); variant++) {
    sz += _geo_element [variant].dis_size();
  }
  return sz;
}
template <class T, class M>
void
geo_base_rep<T,M>::compute_bbox()
{
  // first, compute bbox sequentialy:
  for (size_type j = 0; j < _dimension; j++) {
    _xmin[j] =  std::numeric_limits<T>::max();
    _xmax[j] = -std::numeric_limits<T>::max();
  }
  for (size_type j = _dimension+1; j < 3; j++) {
    _xmin [j] = _xmax [j] = 0;
  }
  for (size_type inod = 0, nnod = _node.size(); inod < nnod; inod++) {
    const point_basic<T>& x = _node [inod];
    for (size_type j = 0 ; j < _dimension; j++) {
      _xmin[j] = min(x[j], _xmin[j]);
      _xmax[j] = max(x[j], _xmax[j]);
    }
  }
  // distributed case: min & max are grouped accross procs 
#ifdef _RHEOLEF_HAVE_MPI
  if (is_distributed<M>::value) {
    for (size_type j = 0 ; j < _dimension; j++) {
      _xmin[j] = mpi::all_reduce (comm(), _xmin[j], mpi::minimum<T>());
      _xmax[j] = mpi::all_reduce (comm(), _xmax[j], mpi::maximum<T>());
    }
  }
#endif // _RHEOLEF_HAVE_MPI

  // next, compute hmin & hmax
  _hmin = std::numeric_limits<T>::max();
  _hmax = 0;
  for (size_t iedg = 0, nedg = geo_element_ownership(1).size(); iedg < nedg; ++iedg) {
    const geo_element& E = get_geo_element (1, iedg);
    const point_basic<T>& x0 = dis_node(E[0]);
    const point_basic<T>& x1 = dis_node(E[1]);
    T hloc = dist(x0,x1);
    _hmin = min(_hmin, hloc);
    _hmax = max(_hmax, hloc);
  }
#ifdef _RHEOLEF_HAVE_MPI
  if (is_distributed<M>::value) {
    _hmin = mpi::all_reduce (comm(), _hmin, mpi::minimum<T>());
    _hmax = mpi::all_reduce (comm(), _hmax, mpi::maximum<T>());
  }
#endif // _RHEOLEF_HAVE_MPI
}
template <class T, class M>
typename geo_base_rep<T,M>::const_reference
geo_base_rep<T,M>::get_geo_element (size_type dim, size_type ige) const
{
  size_type first_ige = 0, last_ige = 0;
  for (size_type variant = reference_element::first_variant_by_dimension(dim);
                 variant < reference_element:: last_variant_by_dimension(dim); variant++) {
    last_ige += _geo_element [variant].size();
    if (ige < last_ige) return _geo_element [variant] [ige - first_ige];
    first_ige = last_ige;
  }
  error_macro ("geo_element index " << ige << " out of range [0:"<< last_ige << "[");
  return _geo_element [0][0]; // not reached
}
template <class T, class M>
typename geo_base_rep<T,M>::reference
geo_base_rep<T,M>::get_geo_element (size_type dim, size_type ige)
{
  size_type first_ige = 0, last_ige = 0;
  for (size_type variant = reference_element::first_variant_by_dimension(dim);
                 variant < reference_element:: last_variant_by_dimension(dim); variant++) {
    last_ige += _geo_element [variant].size();
    if (ige < last_ige) return _geo_element [variant] [ige - first_ige];
    first_ige = last_ige;
  }
  error_macro ("geo_element index " << ige << " out of range [0:"<< last_ige << "[");
  return _geo_element [0][0]; // not reached
}
template <class T, class M>
typename geo_base_rep<T,M>::node_type
geo_base_rep<T,M>::piola (const geo_element& K, const node_type& hat_x) const
{
  // TODO: use basis_on_lattice [K.variant] instead of values[]
  std::vector<T> values;
  reference_element hat_K (K.variant());
  _numbering.get_basis().eval (hat_K, hat_x, values);

  std::vector<size_type> dis_inod;
  _numbering.dis_idof (_gs, K, dis_inod);
  node_type x;
  for (size_type loc_inod = 0, loc_nnod = values.size(); loc_inod < loc_nnod; loc_inod++) {
    const node_type& xnod = dis_node(dis_inod[loc_inod]);
    const T& coeff = values[loc_inod];
    for (size_type i = 0, d = dimension(); i < d; i++) {
      x[i] += coeff*xnod[i];
    }
  }
  return x;
}
// ----------------------------------------------------------------------------
// set order & resize node array ; internal node are not computed
// since they depend on the curved boundary information (cad data)
// that is not aivailable here
// ----------------------------------------------------------------------------
#define _RHEOLEF_reset_order(M)						\
template <class T>							\
void									\
geo_rep<T,M>::reset_order (size_type new_order)				\
{									\
  if (new_order == base::_numbering.degree()) return;			\
  base::_numbering.set_degree (new_order);				\
  size_type dis_nnod = base::_numbering.dis_ndof (base::_gs, base::_map_dimension); \
  size_type     nnod = base::_numbering.ndof     (base::_gs, base::_map_dimension); \
  base::_gs.node_ownership = distributor (nnod, base::comm(), nnod);	\
  array<point_basic<T>, M> new_node (base::_gs.node_ownership);		\
  for (size_type iv = 0, nv = base::_gs.ownership_by_dimension[0].size(); iv < nv; iv++) { \
    new_node [iv] = base::_node [iv];					\
  }									\
  base::_node = new_node;						\
  build_external_entities ();						\
}
_RHEOLEF_reset_order(sequential)
#ifdef _RHEOLEF_HAVE_MPI
_RHEOLEF_reset_order(distributed)
#endif // _RHEOLEF_HAVE_MPI
#undef _RHEOLEF_reset_order
// -------------------------------------------------------------------
// guards for omega.boundary() and omega.internal_sides()
// -------------------------------------------------------------------
template <class T, class M>
void boundary_guard (const geo_basic<T,M>& omega)
{
  typedef typename geo_basic<T,M>::size_type size_type;
  if (omega.have_domain_indirect ("boundary")) return;
  omega.neighbour_guard();
  size_type map_dim = omega.map_dimension();
  check_macro (map_dim > 0, "undefined boundary for 0D geometry");
  size_type sid_dim = map_dim-1;
  std::vector<size_type> isid_list;
  for (size_type isid = 0, nsid = omega.size(sid_dim); isid < nsid; isid++) {
    const geo_element& S = omega.get_geo_element (sid_dim, isid);
    if (S.master(1) != std::numeric_limits<size_type>::max()) continue; // S is an internal side
    isid_list.push_back (isid);
  }
  communicator comm = omega.sizes().ownership_by_dimension[sid_dim].comm();
  domain_indirect_basic<M> isid_dom (omega, "boundary", sid_dim, comm, isid_list);
  omega.insert_domain_indirect (isid_dom);
}
template <class T, class M>
void internal_sides_guard (const geo_basic<T,M>& omega)
{
  typedef typename geo_basic<T,M>::size_type size_type;
  if (omega.have_domain_indirect ("internal_sides")) return;
  omega.neighbour_guard();
  size_type map_dim = omega.map_dimension();
  check_macro (map_dim > 0, "undefined internal_sides for 0D geometry");
  size_type sid_dim = map_dim-1;
  std::vector<size_type> isid_list;
  for (size_type isid = 0, nsid = omega.size(sid_dim); isid < nsid; isid++) {
    const geo_element& S = omega.get_geo_element (sid_dim, isid);
    if (S.master(1) == std::numeric_limits<size_type>::max()) continue; // S is on boundary
    isid_list.push_back (isid);
  }
  communicator comm = omega.sizes().ownership_by_dimension[sid_dim].comm();
  domain_indirect_basic<M> isid_dom (omega, "internal_sides", sid_dim, comm, isid_list);
  omega.insert_domain_indirect (isid_dom);
}
template <class T, class M>
void sides_guard (const geo_basic<T,M>& omega)
{
  typedef typename geo_basic<T,M>::size_type size_type;
  if (omega.have_domain_indirect ("sides")) return;
  omega.neighbour_guard();
  size_type map_dim = omega.map_dimension();
  check_macro (map_dim > 0, "undefined sides for 0D geometry");
  size_type sid_dim = map_dim-1;
  std::vector<size_type> isid_list;
  for (size_type isid = 0, nsid = omega.size(sid_dim); isid < nsid; isid++) {
    const geo_element& S = omega.get_geo_element (sid_dim, isid);
    isid_list.push_back (isid);
  }
  communicator comm = omega.sizes().ownership_by_dimension[sid_dim].comm();
  domain_indirect_basic<M> isid_dom (omega, "sides", sid_dim, comm, isid_list);
  omega.insert_domain_indirect (isid_dom);
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
#define _RHEOLEF_instanciation(T,M) 				\
template class geo_base_rep<T,M>;				\
template class geo_rep<T,M>;					\
template geo_basic<T,M> geo_load (const std::string& name); 	\
template geo_basic<T,M> compact    (const geo_basic<T,M>&);    	\
template void boundary_guard       (const geo_basic<T,M>&);	\
template void internal_sides_guard (const geo_basic<T,M>&);	\
template void sides_guard          (const geo_basic<T,M>&);

_RHEOLEF_instanciation(Float,sequential)
#ifdef _RHEOLEF_HAVE_MPI
_RHEOLEF_instanciation(Float,distributed)
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
