use std::fmt::Write;

use yaxpeax_arch::{AddressBase, Decoder, U8Reader, LengthedInstruction};
use yaxpeax_x86::protected_mode::InstDecoder;

#[allow(dead_code)]
fn test_invalid(data: &[u8]) {
    test_invalid_under(&InstDecoder::default(), data);
}

fn test_invalid_under(decoder: &InstDecoder, data: &[u8]) {
    let mut reader = U8Reader::new(data);
    if let Ok(inst) = decoder.decode(&mut reader) {
        // realistically, the chances an error only shows up under non-fmt builds seems unlikely,
        // but try to report *something* in such cases.
        cfg_if::cfg_if! {
            if #[cfg(feature="fmt")] {
                panic!("decoded {:?} from {:02x?} under decoder {}", inst.opcode(), data, decoder);
            } else {
                // don't warn about the unused inst here
                let _ = inst;
                panic!("decoded instruction <non-fmt build> from {:02x?} under decoder <non-fmt build>", data);
            }
        }
    } else {
        // this is fine
    }
}

#[allow(dead_code)]
fn test_display(data: &[u8], expected: &'static str) {
    test_display_under(&InstDecoder::default(), data, expected);
}

fn test_display_under(decoder: &InstDecoder, data: &[u8], expected: &'static str) {
    let mut hex = String::new();
    for b in data {
        write!(hex, "{:02x}", b).unwrap();
    }
    let mut reader = yaxpeax_arch::U8Reader::new(data);
    match decoder.decode(&mut reader) {
        Ok(instr) => {
            cfg_if::cfg_if! {
                if #[cfg(feature="fmt")] {
                    let text = format!("{}", instr);
                    assert!(
                        text == expected,
                        "display error for {}:\n  decoded: {:?} under decoder {}\n displayed: {}\n expected: {}\n",
                        hex,
                        instr,
                        decoder,
                        text,
                        expected
                    );
                } else {
                    eprintln!("non-fmt build cannot compare text equality")
                }
            }
            // while we're at it, test that the instruction is as long, and no longer, than its
            // input
            assert_eq!((0u32.wrapping_offset(instr.len()).to_linear()) as usize, data.len(), "instruction length is incorrect, wanted instruction {}", expected);
        },
        Err(e) => {
            cfg_if::cfg_if! {
                if #[cfg(feature="fmt")] {
                    assert!(false, "decode error ({}) for {} under decoder {}:\n  expected: {}\n", e, hex, decoder, expected);
                } else {
                    // avoid the unused `e` warning
                    let _ = e;
                    assert!(false, "decode error (<non-fmt build>) for {} under decoder <non-fmt build>:\n  expected: {}\n", hex, expected);
                }
            }
        }
    }
}

fn test_avx_full(bytes: &[u8], text: &'static str) {
    // test with a hypothetical CPU that supports all of AVX512. at time of writing, no such CPU
    // exists.
    test_display_under(&InstDecoder::minimal().with_avx512(), bytes, text);
    test_display_under(&InstDecoder::default(), bytes, text);
    test_invalid_under(&InstDecoder::minimal(), bytes);
}

#[allow(non_snake_case)]
#[test]
fn tests_None_0f() {
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x10, 0xca], "vmovups ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x10, 0x0a], "vmovups ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x10, 0x4a, 0x01], "vmovups ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x10, 0xca], "vmovups ymm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x10, 0xca], "vmovups ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x10, 0x0a], "vmovups ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x10, 0x0a], "vmovups ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x10, 0x4a, 0x01], "vmovups ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x10, 0x4a, 0x01], "vmovups ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x10, 0xca], "vmovups zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x10, 0x0a], "vmovups zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x10, 0x4a, 0x01], "vmovups zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x10, 0xca], "vmovups zmm1, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x10, 0xca], "vmovups zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x10, 0x0a], "vmovups zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x10, 0x0a], "vmovups zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x10, 0x4a, 0x01], "vmovups zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x10, 0x4a, 0x01], "vmovups zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x10, 0xca], "vmovups xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x10, 0x0a], "vmovups xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x10, 0x4a, 0x01], "vmovups xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x10, 0xca], "vmovups xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x10, 0xca], "vmovups xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x10, 0x0a], "vmovups xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x10, 0x0a], "vmovups xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x10, 0x4a, 0x01], "vmovups xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x10, 0x4a, 0x01], "vmovups xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x11, 0xca], "vmovups ymm2{k5}{z}, ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x11, 0xca], "vmovups ymm2, ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x11, 0xca], "vmovups ymm2{k5}, ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x11, 0x0a], "vmovups ymmword [edx], ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x11, 0x0a], "vmovups ymmword [edx]{k5}, ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x11, 0x4a, 0x01], "vmovups ymmword [edx + 0x20], ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x11, 0x4a, 0x01], "vmovups ymmword [edx + 0x20]{k5}, ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x11, 0xca], "vmovups zmm2{k5}{z}, zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x11, 0xca], "vmovups zmm2, zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x11, 0xca], "vmovups zmm2{k5}, zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x11, 0x0a], "vmovups zmmword [edx], zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x11, 0x0a], "vmovups zmmword [edx]{k5}, zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x11, 0x4a, 0x01], "vmovups zmmword [edx + 0x40], zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x11, 0x4a, 0x01], "vmovups zmmword [edx + 0x40]{k5}, zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x11, 0xca], "vmovups xmm2{k5}{z}, xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x11, 0xca], "vmovups xmm2, xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x11, 0xca], "vmovups xmm2{k5}, xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x11, 0x0a], "vmovups xmmword [edx], xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x11, 0x0a], "vmovups xmmword [edx]{k5}, xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x11, 0x4a, 0x01], "vmovups xmmword [edx + 0x10], xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x11, 0x4a, 0x01], "vmovups xmmword [edx + 0x10]{k5}, xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x12, 0xca], "vmovhlps xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x12, 0x0a], "vmovlps xmm1, xmm0, qword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x12, 0x4a, 0x01], "vmovlps xmm1, xmm0, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x13, 0x0a], "vmovlps qword [edx], xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x13, 0x4a, 0x01], "vmovlps qword [edx + 0x8], xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x14, 0x0a], "vunpcklps ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x14, 0x4a, 0x01], "vunpcklps ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x14, 0x0a], "vunpcklps ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x14, 0x0a], "vunpcklps ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x14, 0x4a, 0x01], "vunpcklps ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x14, 0x4a, 0x01], "vunpcklps ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x14, 0xca], "vunpcklps ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x14, 0x0a], "vunpcklps ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x14, 0x4a, 0x01], "vunpcklps ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x14, 0xca], "vunpcklps ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x14, 0xca], "vunpcklps ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x14, 0x0a], "vunpcklps ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x14, 0x0a], "vunpcklps ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x14, 0x4a, 0x01], "vunpcklps ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x14, 0x4a, 0x01], "vunpcklps ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x14, 0x0a], "vunpcklps zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x14, 0x4a, 0x01], "vunpcklps zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x14, 0x0a], "vunpcklps zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x14, 0x0a], "vunpcklps zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x14, 0x4a, 0x01], "vunpcklps zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x14, 0x4a, 0x01], "vunpcklps zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x14, 0x0a], "vunpcklps xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x14, 0x4a, 0x01], "vunpcklps xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x14, 0x0a], "vunpcklps xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x14, 0x0a], "vunpcklps xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x14, 0x4a, 0x01], "vunpcklps xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x14, 0x4a, 0x01], "vunpcklps xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x14, 0xca], "vunpcklps zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x14, 0x0a], "vunpcklps zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x14, 0x4a, 0x01], "vunpcklps zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x14, 0xca], "vunpcklps zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x14, 0xca], "vunpcklps zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x14, 0x0a], "vunpcklps zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x14, 0x0a], "vunpcklps zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x14, 0x4a, 0x01], "vunpcklps zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x14, 0x4a, 0x01], "vunpcklps zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x14, 0xca], "vunpcklps xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x14, 0x0a], "vunpcklps xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x14, 0x4a, 0x01], "vunpcklps xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x14, 0xca], "vunpcklps xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x14, 0xca], "vunpcklps xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x14, 0x0a], "vunpcklps xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x14, 0x0a], "vunpcklps xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x14, 0x4a, 0x01], "vunpcklps xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x14, 0x4a, 0x01], "vunpcklps xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x15, 0x0a], "vunpckhps ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x15, 0x4a, 0x01], "vunpckhps ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x15, 0x0a], "vunpckhps ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x15, 0x0a], "vunpckhps ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x15, 0x4a, 0x01], "vunpckhps ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x15, 0x4a, 0x01], "vunpckhps ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x15, 0xca], "vunpckhps ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x15, 0x0a], "vunpckhps ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x15, 0x4a, 0x01], "vunpckhps ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x15, 0xca], "vunpckhps ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x15, 0xca], "vunpckhps ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x15, 0x0a], "vunpckhps ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x15, 0x0a], "vunpckhps ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x15, 0x4a, 0x01], "vunpckhps ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x15, 0x4a, 0x01], "vunpckhps ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x15, 0x0a], "vunpckhps zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x15, 0x4a, 0x01], "vunpckhps zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x15, 0x0a], "vunpckhps zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x15, 0x0a], "vunpckhps zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x15, 0x4a, 0x01], "vunpckhps zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x15, 0x4a, 0x01], "vunpckhps zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x15, 0x0a], "vunpckhps xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x15, 0x4a, 0x01], "vunpckhps xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x15, 0x0a], "vunpckhps xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x15, 0x0a], "vunpckhps xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x15, 0x4a, 0x01], "vunpckhps xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x15, 0x4a, 0x01], "vunpckhps xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x15, 0xca], "vunpckhps zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x15, 0x0a], "vunpckhps zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x15, 0x4a, 0x01], "vunpckhps zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x15, 0xca], "vunpckhps zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x15, 0xca], "vunpckhps zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x15, 0x0a], "vunpckhps zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x15, 0x0a], "vunpckhps zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x15, 0x4a, 0x01], "vunpckhps zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x15, 0x4a, 0x01], "vunpckhps zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x15, 0xca], "vunpckhps xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x15, 0x0a], "vunpckhps xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x15, 0x4a, 0x01], "vunpckhps xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x15, 0xca], "vunpckhps xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x15, 0xca], "vunpckhps xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x15, 0x0a], "vunpckhps xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x15, 0x0a], "vunpckhps xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x15, 0x4a, 0x01], "vunpckhps xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x15, 0x4a, 0x01], "vunpckhps xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x16, 0xca], "vmovlhps xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x16, 0x0a], "vmovhps xmm1, xmm0, qword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x16, 0x4a, 0x01], "vmovhps xmm1, xmm0, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x17, 0x0a], "vmovhps qword [edx], xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x17, 0x4a, 0x01], "vmovhps qword [edx + 0x8], xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x28, 0xca], "vmovaps ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x28, 0x0a], "vmovaps ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x28, 0x4a, 0x01], "vmovaps ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x28, 0xca], "vmovaps ymm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x28, 0xca], "vmovaps ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x28, 0x0a], "vmovaps ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x28, 0x0a], "vmovaps ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x28, 0x4a, 0x01], "vmovaps ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x28, 0x4a, 0x01], "vmovaps ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x28, 0xca], "vmovaps zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x28, 0x0a], "vmovaps zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x28, 0x4a, 0x01], "vmovaps zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x28, 0xca], "vmovaps zmm1, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x28, 0xca], "vmovaps zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x28, 0x0a], "vmovaps zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x28, 0x0a], "vmovaps zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x28, 0x4a, 0x01], "vmovaps zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x28, 0x4a, 0x01], "vmovaps zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x28, 0xca], "vmovaps xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x28, 0x0a], "vmovaps xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x28, 0x4a, 0x01], "vmovaps xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x28, 0xca], "vmovaps xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x28, 0xca], "vmovaps xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x28, 0x0a], "vmovaps xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x28, 0x0a], "vmovaps xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x28, 0x4a, 0x01], "vmovaps xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x28, 0x4a, 0x01], "vmovaps xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x29, 0xca], "vmovaps ymm2{k5}{z}, ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x29, 0xca], "vmovaps ymm2, ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x29, 0xca], "vmovaps ymm2{k5}, ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x29, 0x0a], "vmovaps ymmword [edx], ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x29, 0x0a], "vmovaps ymmword [edx]{k5}, ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x29, 0x4a, 0x01], "vmovaps ymmword [edx + 0x20], ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x29, 0x4a, 0x01], "vmovaps ymmword [edx + 0x20]{k5}, ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x29, 0xca], "vmovaps zmm2{k5}{z}, zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x29, 0xca], "vmovaps zmm2, zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x29, 0xca], "vmovaps zmm2{k5}, zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x29, 0x0a], "vmovaps zmmword [edx], zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x29, 0x0a], "vmovaps zmmword [edx]{k5}, zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x29, 0x4a, 0x01], "vmovaps zmmword [edx + 0x40], zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x29, 0x4a, 0x01], "vmovaps zmmword [edx + 0x40]{k5}, zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x29, 0xca], "vmovaps xmm2{k5}{z}, xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x29, 0xca], "vmovaps xmm2, xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x29, 0xca], "vmovaps xmm2{k5}, xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x29, 0x0a], "vmovaps xmmword [edx], xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x29, 0x0a], "vmovaps xmmword [edx]{k5}, xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x29, 0x4a, 0x01], "vmovaps xmmword [edx + 0x10], xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x29, 0x4a, 0x01], "vmovaps xmmword [edx + 0x10]{k5}, xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x2b, 0x0a], "vmovntps ymmword [edx], ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x2b, 0x4a, 0x01], "vmovntps ymmword [edx + 0x20], ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x2b, 0x0a], "vmovntps zmmword [edx], zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x2b, 0x4a, 0x01], "vmovntps zmmword [edx + 0x40], zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x2b, 0x0a], "vmovntps xmmword [edx], xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x2b, 0x4a, 0x01], "vmovntps xmmword [edx + 0x10], xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x2e, 0xca], "vucomiss xmm1{sae}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x2e, 0xca], "vucomiss xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x2e, 0x0a], "vucomiss xmm1, dword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x2e, 0x4a, 0x01], "vucomiss xmm1, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x2f, 0xca], "vcomiss xmm1{sae}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x2f, 0xca], "vcomiss xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x2f, 0x0a], "vcomiss xmm1, dword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x2f, 0x4a, 0x01], "vcomiss xmm1, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xfd, 0x51, 0xca], "vsqrtps zmm1{k5}{z}{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x51, 0xca], "vsqrtps zmm1{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x7d, 0x51, 0xca], "vsqrtps zmm1{k5}{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x51, 0xca], "vsqrtps zmm1{k5}{z}{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x51, 0x0a], "vsqrtps ymm1{k5}{z}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x51, 0x4a, 0x01], "vsqrtps ymm1{k5}{z}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x51, 0xca], "vsqrtps zmm1{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x51, 0xca], "vsqrtps zmm1{k5}{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x51, 0x0a], "vsqrtps ymm1, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x51, 0x0a], "vsqrtps ymm1{k5}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x51, 0x4a, 0x01], "vsqrtps ymm1, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x51, 0x4a, 0x01], "vsqrtps ymm1{k5}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x51, 0xca], "vsqrtps ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x51, 0x0a], "vsqrtps ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x51, 0x4a, 0x01], "vsqrtps ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x51, 0xca], "vsqrtps ymm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x51, 0xca], "vsqrtps ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x51, 0x0a], "vsqrtps ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x51, 0x0a], "vsqrtps ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x51, 0x4a, 0x01], "vsqrtps ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x51, 0x4a, 0x01], "vsqrtps ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x51, 0xca], "vsqrtps zmm1{k5}{z}{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x51, 0x0a], "vsqrtps zmm1{k5}{z}, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x51, 0x4a, 0x01], "vsqrtps zmm1{k5}{z}, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x51, 0xca], "vsqrtps zmm1{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x51, 0xca], "vsqrtps zmm1{k5}{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x51, 0x0a], "vsqrtps zmm1, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x51, 0x0a], "vsqrtps zmm1{k5}, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x51, 0x4a, 0x01], "vsqrtps zmm1, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x51, 0x4a, 0x01], "vsqrtps zmm1{k5}, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x51, 0xca], "vsqrtps zmm1{k5}{z}{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x51, 0x0a], "vsqrtps xmm1{k5}{z}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x51, 0x4a, 0x01], "vsqrtps xmm1{k5}{z}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x51, 0xca], "vsqrtps zmm1{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x51, 0xca], "vsqrtps zmm1{k5}{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x51, 0x0a], "vsqrtps xmm1, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x51, 0x0a], "vsqrtps xmm1{k5}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x51, 0x4a, 0x01], "vsqrtps xmm1, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x51, 0x4a, 0x01], "vsqrtps xmm1{k5}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x51, 0xca], "vsqrtps zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x51, 0x0a], "vsqrtps zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x51, 0x4a, 0x01], "vsqrtps zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x51, 0xca], "vsqrtps zmm1, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x51, 0xca], "vsqrtps zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x51, 0x0a], "vsqrtps zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x51, 0x0a], "vsqrtps zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x51, 0x4a, 0x01], "vsqrtps zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x51, 0x4a, 0x01], "vsqrtps zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x51, 0xca], "vsqrtps xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x51, 0x0a], "vsqrtps xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x51, 0x4a, 0x01], "vsqrtps xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x51, 0xca], "vsqrtps xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x51, 0xca], "vsqrtps xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x51, 0x0a], "vsqrtps xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x51, 0x0a], "vsqrtps xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x51, 0x4a, 0x01], "vsqrtps xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x51, 0x4a, 0x01], "vsqrtps xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x54, 0x0a], "vandps ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x54, 0x4a, 0x01], "vandps ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x54, 0x0a], "vandps ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x54, 0x0a], "vandps ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x54, 0x4a, 0x01], "vandps ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x54, 0x4a, 0x01], "vandps ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x54, 0xca], "vandps ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x54, 0x0a], "vandps ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x54, 0x4a, 0x01], "vandps ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x54, 0xca], "vandps ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x54, 0xca], "vandps ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x54, 0x0a], "vandps ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x54, 0x0a], "vandps ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x54, 0x4a, 0x01], "vandps ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x54, 0x4a, 0x01], "vandps ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x54, 0x0a], "vandps zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x54, 0x4a, 0x01], "vandps zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x54, 0x0a], "vandps zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x54, 0x0a], "vandps zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x54, 0x4a, 0x01], "vandps zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x54, 0x4a, 0x01], "vandps zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x54, 0x0a], "vandps xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x54, 0x4a, 0x01], "vandps xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x54, 0x0a], "vandps xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x54, 0x0a], "vandps xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x54, 0x4a, 0x01], "vandps xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x54, 0x4a, 0x01], "vandps xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x54, 0xca], "vandps zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x54, 0x0a], "vandps zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x54, 0x4a, 0x01], "vandps zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x54, 0xca], "vandps zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x54, 0xca], "vandps zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x54, 0x0a], "vandps zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x54, 0x0a], "vandps zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x54, 0x4a, 0x01], "vandps zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x54, 0x4a, 0x01], "vandps zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x54, 0xca], "vandps xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x54, 0x0a], "vandps xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x54, 0x4a, 0x01], "vandps xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x54, 0xca], "vandps xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x54, 0xca], "vandps xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x54, 0x0a], "vandps xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x54, 0x0a], "vandps xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x54, 0x4a, 0x01], "vandps xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x54, 0x4a, 0x01], "vandps xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x55, 0x0a], "vandnps ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x55, 0x4a, 0x01], "vandnps ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x55, 0x0a], "vandnps ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x55, 0x0a], "vandnps ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x55, 0x4a, 0x01], "vandnps ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x55, 0x4a, 0x01], "vandnps ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x55, 0xca], "vandnps ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x55, 0x0a], "vandnps ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x55, 0x4a, 0x01], "vandnps ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x55, 0xca], "vandnps ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x55, 0xca], "vandnps ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x55, 0x0a], "vandnps ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x55, 0x0a], "vandnps ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x55, 0x4a, 0x01], "vandnps ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x55, 0x4a, 0x01], "vandnps ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x55, 0x0a], "vandnps zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x55, 0x4a, 0x01], "vandnps zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x55, 0x0a], "vandnps zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x55, 0x0a], "vandnps zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x55, 0x4a, 0x01], "vandnps zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x55, 0x4a, 0x01], "vandnps zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x55, 0x0a], "vandnps xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x55, 0x4a, 0x01], "vandnps xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x55, 0x0a], "vandnps xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x55, 0x0a], "vandnps xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x55, 0x4a, 0x01], "vandnps xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x55, 0x4a, 0x01], "vandnps xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x55, 0xca], "vandnps zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x55, 0x0a], "vandnps zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x55, 0x4a, 0x01], "vandnps zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x55, 0xca], "vandnps zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x55, 0xca], "vandnps zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x55, 0x0a], "vandnps zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x55, 0x0a], "vandnps zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x55, 0x4a, 0x01], "vandnps zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x55, 0x4a, 0x01], "vandnps zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x55, 0xca], "vandnps xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x55, 0x0a], "vandnps xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x55, 0x4a, 0x01], "vandnps xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x55, 0xca], "vandnps xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x55, 0xca], "vandnps xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x55, 0x0a], "vandnps xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x55, 0x0a], "vandnps xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x55, 0x4a, 0x01], "vandnps xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x55, 0x4a, 0x01], "vandnps xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x56, 0x0a], "vorps ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x56, 0x4a, 0x01], "vorps ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x56, 0x0a], "vorps ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x56, 0x0a], "vorps ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x56, 0x4a, 0x01], "vorps ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x56, 0x4a, 0x01], "vorps ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x56, 0xca], "vorps ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x56, 0x0a], "vorps ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x56, 0x4a, 0x01], "vorps ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x56, 0xca], "vorps ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x56, 0xca], "vorps ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x56, 0x0a], "vorps ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x56, 0x0a], "vorps ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x56, 0x4a, 0x01], "vorps ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x56, 0x4a, 0x01], "vorps ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x56, 0x0a], "vorps zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x56, 0x4a, 0x01], "vorps zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x56, 0x0a], "vorps zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x56, 0x0a], "vorps zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x56, 0x4a, 0x01], "vorps zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x56, 0x4a, 0x01], "vorps zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x56, 0x0a], "vorps xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x56, 0x4a, 0x01], "vorps xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x56, 0x0a], "vorps xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x56, 0x0a], "vorps xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x56, 0x4a, 0x01], "vorps xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x56, 0x4a, 0x01], "vorps xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x56, 0xca], "vorps zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x56, 0x0a], "vorps zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x56, 0x4a, 0x01], "vorps zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x56, 0xca], "vorps zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x56, 0xca], "vorps zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x56, 0x0a], "vorps zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x56, 0x0a], "vorps zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x56, 0x4a, 0x01], "vorps zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x56, 0x4a, 0x01], "vorps zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x56, 0xca], "vorps xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x56, 0x0a], "vorps xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x56, 0x4a, 0x01], "vorps xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x56, 0xca], "vorps xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x56, 0xca], "vorps xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x56, 0x0a], "vorps xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x56, 0x0a], "vorps xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x56, 0x4a, 0x01], "vorps xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x56, 0x4a, 0x01], "vorps xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x57, 0x0a], "vxorps ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x57, 0x4a, 0x01], "vxorps ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x57, 0x0a], "vxorps ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x57, 0x0a], "vxorps ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x57, 0x4a, 0x01], "vxorps ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x57, 0x4a, 0x01], "vxorps ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x57, 0xca], "vxorps ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x57, 0x0a], "vxorps ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x57, 0x4a, 0x01], "vxorps ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x57, 0xca], "vxorps ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x57, 0xca], "vxorps ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x57, 0x0a], "vxorps ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x57, 0x0a], "vxorps ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x57, 0x4a, 0x01], "vxorps ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x57, 0x4a, 0x01], "vxorps ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x57, 0x0a], "vxorps zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x57, 0x4a, 0x01], "vxorps zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x57, 0x0a], "vxorps zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x57, 0x0a], "vxorps zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x57, 0x4a, 0x01], "vxorps zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x57, 0x4a, 0x01], "vxorps zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x57, 0x0a], "vxorps xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x57, 0x4a, 0x01], "vxorps xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x57, 0x0a], "vxorps xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x57, 0x0a], "vxorps xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x57, 0x4a, 0x01], "vxorps xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x57, 0x4a, 0x01], "vxorps xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x57, 0xca], "vxorps zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x57, 0x0a], "vxorps zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x57, 0x4a, 0x01], "vxorps zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x57, 0xca], "vxorps zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x57, 0xca], "vxorps zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x57, 0x0a], "vxorps zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x57, 0x0a], "vxorps zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x57, 0x4a, 0x01], "vxorps zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x57, 0x4a, 0x01], "vxorps zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x57, 0xca], "vxorps xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x57, 0x0a], "vxorps xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x57, 0x4a, 0x01], "vxorps xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x57, 0xca], "vxorps xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x57, 0xca], "vxorps xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x57, 0x0a], "vxorps xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x57, 0x0a], "vxorps xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x57, 0x4a, 0x01], "vxorps xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x57, 0x4a, 0x01], "vxorps xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xfd, 0x58, 0xca], "vaddps zmm1{k5}{z}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x58, 0xca], "vaddps zmm1{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x7d, 0x58, 0xca], "vaddps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x58, 0xca], "vaddps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x58, 0x0a], "vaddps ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x58, 0x4a, 0x01], "vaddps ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x58, 0xca], "vaddps zmm1{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x58, 0xca], "vaddps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x58, 0x0a], "vaddps ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x58, 0x0a], "vaddps ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x58, 0x4a, 0x01], "vaddps ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x58, 0x4a, 0x01], "vaddps ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x58, 0xca], "vaddps ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x58, 0x0a], "vaddps ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x58, 0x4a, 0x01], "vaddps ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x58, 0xca], "vaddps ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x58, 0xca], "vaddps ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x58, 0x0a], "vaddps ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x58, 0x0a], "vaddps ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x58, 0x4a, 0x01], "vaddps ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x58, 0x4a, 0x01], "vaddps ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x58, 0xca], "vaddps zmm1{k5}{z}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x58, 0x0a], "vaddps zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x58, 0x4a, 0x01], "vaddps zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x58, 0xca], "vaddps zmm1{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x58, 0xca], "vaddps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x58, 0x0a], "vaddps zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x58, 0x0a], "vaddps zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x58, 0x4a, 0x01], "vaddps zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x58, 0x4a, 0x01], "vaddps zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x58, 0xca], "vaddps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x58, 0x0a], "vaddps xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x58, 0x4a, 0x01], "vaddps xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x58, 0xca], "vaddps zmm1{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x58, 0xca], "vaddps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x58, 0x0a], "vaddps xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x58, 0x0a], "vaddps xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x58, 0x4a, 0x01], "vaddps xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x58, 0x4a, 0x01], "vaddps xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x58, 0xca], "vaddps zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x58, 0x0a], "vaddps zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x58, 0x4a, 0x01], "vaddps zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x58, 0xca], "vaddps zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x58, 0xca], "vaddps zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x58, 0x0a], "vaddps zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x58, 0x0a], "vaddps zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x58, 0x4a, 0x01], "vaddps zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x58, 0x4a, 0x01], "vaddps zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x58, 0xca], "vaddps xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x58, 0x0a], "vaddps xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x58, 0x4a, 0x01], "vaddps xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x58, 0xca], "vaddps xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x58, 0xca], "vaddps xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x58, 0x0a], "vaddps xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x58, 0x0a], "vaddps xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x58, 0x4a, 0x01], "vaddps xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x58, 0x4a, 0x01], "vaddps xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xfd, 0x59, 0xca], "vmulps zmm1{k5}{z}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x59, 0xca], "vmulps zmm1{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x7d, 0x59, 0xca], "vmulps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x59, 0xca], "vmulps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x59, 0x0a], "vmulps ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x59, 0x4a, 0x01], "vmulps ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x59, 0xca], "vmulps zmm1{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x59, 0xca], "vmulps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x59, 0x0a], "vmulps ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x59, 0x0a], "vmulps ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x59, 0x4a, 0x01], "vmulps ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x59, 0x4a, 0x01], "vmulps ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x59, 0xca], "vmulps ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x59, 0x0a], "vmulps ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x59, 0x4a, 0x01], "vmulps ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x59, 0xca], "vmulps ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x59, 0xca], "vmulps ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x59, 0x0a], "vmulps ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x59, 0x0a], "vmulps ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x59, 0x4a, 0x01], "vmulps ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x59, 0x4a, 0x01], "vmulps ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x59, 0xca], "vmulps zmm1{k5}{z}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x59, 0x0a], "vmulps zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x59, 0x4a, 0x01], "vmulps zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x59, 0xca], "vmulps zmm1{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x59, 0xca], "vmulps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x59, 0x0a], "vmulps zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x59, 0x0a], "vmulps zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x59, 0x4a, 0x01], "vmulps zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x59, 0x4a, 0x01], "vmulps zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x59, 0xca], "vmulps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x59, 0x0a], "vmulps xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x59, 0x4a, 0x01], "vmulps xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x59, 0xca], "vmulps zmm1{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x59, 0xca], "vmulps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x59, 0x0a], "vmulps xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x59, 0x0a], "vmulps xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x59, 0x4a, 0x01], "vmulps xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x59, 0x4a, 0x01], "vmulps xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x59, 0xca], "vmulps zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x59, 0x0a], "vmulps zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x59, 0x4a, 0x01], "vmulps zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x59, 0xca], "vmulps zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x59, 0xca], "vmulps zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x59, 0x0a], "vmulps zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x59, 0x0a], "vmulps zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x59, 0x4a, 0x01], "vmulps zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x59, 0x4a, 0x01], "vmulps zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x59, 0xca], "vmulps xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x59, 0x0a], "vmulps xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x59, 0x4a, 0x01], "vmulps xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x59, 0xca], "vmulps xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x59, 0xca], "vmulps xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x59, 0x0a], "vmulps xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x59, 0x0a], "vmulps xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x59, 0x4a, 0x01], "vmulps xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x59, 0x4a, 0x01], "vmulps xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xfd, 0x5a, 0xca], "vcvtps2pd zmm1{k5}{z}{sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x5a, 0xca], "vcvtps2pd zmm1{sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x7d, 0x5a, 0xca], "vcvtps2pd zmm1{k5}{sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x5a, 0x0a], "vcvtps2pd ymm1{k5}{z}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x5a, 0x4a, 0x01], "vcvtps2pd ymm1{k5}{z}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x5a, 0x0a], "vcvtps2pd ymm1, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x5a, 0x0a], "vcvtps2pd ymm1{k5}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x5a, 0x4a, 0x01], "vcvtps2pd ymm1, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x5a, 0x4a, 0x01], "vcvtps2pd ymm1{k5}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5a, 0xca], "vcvtps2pd ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5a, 0x0a], "vcvtps2pd ymm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5a, 0x4a, 0x01], "vcvtps2pd ymm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5a, 0xca], "vcvtps2pd ymm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5a, 0xca], "vcvtps2pd ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5a, 0x0a], "vcvtps2pd ymm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5a, 0x0a], "vcvtps2pd ymm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5a, 0x4a, 0x01], "vcvtps2pd ymm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5a, 0x4a, 0x01], "vcvtps2pd ymm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x5a, 0x0a], "vcvtps2pd zmm1{k5}{z}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x5a, 0x4a, 0x01], "vcvtps2pd zmm1{k5}{z}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x5a, 0x0a], "vcvtps2pd zmm1, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x5a, 0x0a], "vcvtps2pd zmm1{k5}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x5a, 0x4a, 0x01], "vcvtps2pd zmm1, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x5a, 0x4a, 0x01], "vcvtps2pd zmm1{k5}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x5a, 0x0a], "vcvtps2pd xmm1{k5}{z}, dword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x5a, 0x4a, 0x01], "vcvtps2pd xmm1{k5}{z}, dword [edx + 0x4]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x5a, 0x0a], "vcvtps2pd xmm1, dword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x5a, 0x0a], "vcvtps2pd xmm1{k5}, dword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x5a, 0x4a, 0x01], "vcvtps2pd xmm1, dword [edx + 0x4]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x5a, 0x4a, 0x01], "vcvtps2pd xmm1{k5}, dword [edx + 0x4]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5a, 0xca], "vcvtps2pd zmm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5a, 0x0a], "vcvtps2pd zmm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5a, 0x4a, 0x01], "vcvtps2pd zmm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5a, 0xca], "vcvtps2pd zmm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5a, 0xca], "vcvtps2pd zmm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5a, 0x0a], "vcvtps2pd zmm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5a, 0x0a], "vcvtps2pd zmm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5a, 0x4a, 0x01], "vcvtps2pd zmm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5a, 0x4a, 0x01], "vcvtps2pd zmm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5a, 0xca], "vcvtps2pd xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5a, 0x0a], "vcvtps2pd xmm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5a, 0x4a, 0x01], "vcvtps2pd xmm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5a, 0xca], "vcvtps2pd xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5a, 0xca], "vcvtps2pd xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5a, 0x0a], "vcvtps2pd xmm1, qword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5a, 0x0a], "vcvtps2pd xmm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5a, 0x4a, 0x01], "vcvtps2pd xmm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5a, 0x4a, 0x01], "vcvtps2pd xmm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xfd, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{z}{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x78, 0x5b, 0xca], "vcvtqq2ps ymm1{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x7d, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xbd, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{z}{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xbd, 0x5b, 0x0a], "vcvtqq2ps xmm1{k5}{z}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xbd, 0x5b, 0x4a, 0x01], "vcvtqq2ps xmm1{k5}{z}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x38, 0x5b, 0xca], "vcvtqq2ps ymm1{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x3d, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x38, 0x5b, 0x0a], "vcvtqq2ps xmm1, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x3d, 0x5b, 0x0a], "vcvtqq2ps xmm1{k5}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x38, 0x5b, 0x4a, 0x01], "vcvtqq2ps xmm1, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x3d, 0x5b, 0x4a, 0x01], "vcvtqq2ps xmm1{k5}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xad, 0x5b, 0xca], "vcvtqq2ps xmm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xad, 0x5b, 0x0a], "vcvtqq2ps xmm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xad, 0x5b, 0x4a, 0x01], "vcvtqq2ps xmm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x28, 0x5b, 0xca], "vcvtqq2ps xmm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x2d, 0x5b, 0xca], "vcvtqq2ps xmm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x28, 0x5b, 0x0a], "vcvtqq2ps xmm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x2d, 0x5b, 0x0a], "vcvtqq2ps xmm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x28, 0x5b, 0x4a, 0x01], "vcvtqq2ps xmm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x2d, 0x5b, 0x4a, 0x01], "vcvtqq2ps xmm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xfd, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{z}{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x5b, 0xca], "vcvtdq2ps zmm1{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x7d, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{z}{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x5b, 0x0a], "vcvtdq2ps ymm1{k5}{z}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x5b, 0x4a, 0x01], "vcvtdq2ps ymm1{k5}{z}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x5b, 0xca], "vcvtdq2ps zmm1{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x5b, 0x0a], "vcvtdq2ps ymm1, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x5b, 0x0a], "vcvtdq2ps ymm1{k5}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x5b, 0x4a, 0x01], "vcvtdq2ps ymm1, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x5b, 0x4a, 0x01], "vcvtdq2ps ymm1{k5}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5b, 0xca], "vcvtdq2ps ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5b, 0x0a], "vcvtdq2ps ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5b, 0x4a, 0x01], "vcvtdq2ps ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5b, 0xca], "vcvtdq2ps ymm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5b, 0xca], "vcvtdq2ps ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5b, 0x0a], "vcvtdq2ps ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5b, 0x0a], "vcvtdq2ps ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5b, 0x4a, 0x01], "vcvtdq2ps ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5b, 0x4a, 0x01], "vcvtdq2ps ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xdd, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{z}{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xdd, 0x5b, 0x0a], "vcvtqq2ps ymm1{k5}{z}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xdd, 0x5b, 0x4a, 0x01], "vcvtqq2ps ymm1{k5}{z}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x58, 0x5b, 0xca], "vcvtqq2ps ymm1{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x5d, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x58, 0x5b, 0x0a], "vcvtqq2ps ymm1, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x5d, 0x5b, 0x0a], "vcvtqq2ps ymm1{k5}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x58, 0x5b, 0x4a, 0x01], "vcvtqq2ps ymm1, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x5d, 0x5b, 0x4a, 0x01], "vcvtqq2ps ymm1{k5}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x9d, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{z}{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x9d, 0x5b, 0x0a], "vcvtqq2ps xmm1{k5}{z}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x9d, 0x5b, 0x4a, 0x01], "vcvtqq2ps xmm1{k5}{z}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x18, 0x5b, 0xca], "vcvtqq2ps ymm1{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x1d, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x18, 0x5b, 0x0a], "vcvtqq2ps xmm1, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x1d, 0x5b, 0x0a], "vcvtqq2ps xmm1{k5}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x18, 0x5b, 0x4a, 0x01], "vcvtqq2ps xmm1, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x1d, 0x5b, 0x4a, 0x01], "vcvtqq2ps xmm1{k5}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xcd, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xcd, 0x5b, 0x0a], "vcvtqq2ps ymm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xcd, 0x5b, 0x4a, 0x01], "vcvtqq2ps ymm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x48, 0x5b, 0xca], "vcvtqq2ps ymm1, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x4d, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x48, 0x5b, 0x0a], "vcvtqq2ps ymm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x4d, 0x5b, 0x0a], "vcvtqq2ps ymm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x48, 0x5b, 0x4a, 0x01], "vcvtqq2ps ymm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x4d, 0x5b, 0x4a, 0x01], "vcvtqq2ps ymm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x8d, 0x5b, 0xca], "vcvtqq2ps xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x8d, 0x5b, 0x0a], "vcvtqq2ps xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x8d, 0x5b, 0x4a, 0x01], "vcvtqq2ps xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x08, 0x5b, 0xca], "vcvtqq2ps xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x0d, 0x5b, 0xca], "vcvtqq2ps xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x08, 0x5b, 0x0a], "vcvtqq2ps xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x0d, 0x5b, 0x0a], "vcvtqq2ps xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x08, 0x5b, 0x4a, 0x01], "vcvtqq2ps xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x0d, 0x5b, 0x4a, 0x01], "vcvtqq2ps xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{z}{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x5b, 0x0a], "vcvtdq2ps zmm1{k5}{z}, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x5b, 0x4a, 0x01], "vcvtdq2ps zmm1{k5}{z}, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x5b, 0xca], "vcvtdq2ps zmm1{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x5b, 0x0a], "vcvtdq2ps zmm1, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x5b, 0x0a], "vcvtdq2ps zmm1{k5}, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x5b, 0x4a, 0x01], "vcvtdq2ps zmm1, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x5b, 0x4a, 0x01], "vcvtdq2ps zmm1{k5}, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{z}{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x5b, 0x0a], "vcvtdq2ps xmm1{k5}{z}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x5b, 0x4a, 0x01], "vcvtdq2ps xmm1{k5}{z}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x5b, 0xca], "vcvtdq2ps zmm1{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x5b, 0x0a], "vcvtdq2ps xmm1, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x5b, 0x0a], "vcvtdq2ps xmm1{k5}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x5b, 0x4a, 0x01], "vcvtdq2ps xmm1, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x5b, 0x4a, 0x01], "vcvtdq2ps xmm1{k5}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5b, 0x0a], "vcvtdq2ps zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5b, 0x4a, 0x01], "vcvtdq2ps zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5b, 0xca], "vcvtdq2ps zmm1, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5b, 0x0a], "vcvtdq2ps zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5b, 0x0a], "vcvtdq2ps zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5b, 0x4a, 0x01], "vcvtdq2ps zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5b, 0x4a, 0x01], "vcvtdq2ps zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5b, 0xca], "vcvtdq2ps xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5b, 0x0a], "vcvtdq2ps xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5b, 0x4a, 0x01], "vcvtdq2ps xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5b, 0xca], "vcvtdq2ps xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5b, 0xca], "vcvtdq2ps xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5b, 0x0a], "vcvtdq2ps xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5b, 0x0a], "vcvtdq2ps xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5b, 0x4a, 0x01], "vcvtdq2ps xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5b, 0x4a, 0x01], "vcvtdq2ps xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xfd, 0x5c, 0xca], "vsubps zmm1{k5}{z}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x5c, 0xca], "vsubps zmm1{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x7d, 0x5c, 0xca], "vsubps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x5c, 0xca], "vsubps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x5c, 0x0a], "vsubps ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x5c, 0x4a, 0x01], "vsubps ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x5c, 0xca], "vsubps zmm1{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x5c, 0xca], "vsubps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x5c, 0x0a], "vsubps ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x5c, 0x0a], "vsubps ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x5c, 0x4a, 0x01], "vsubps ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x5c, 0x4a, 0x01], "vsubps ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5c, 0xca], "vsubps ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5c, 0x0a], "vsubps ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5c, 0x4a, 0x01], "vsubps ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5c, 0xca], "vsubps ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5c, 0xca], "vsubps ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5c, 0x0a], "vsubps ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5c, 0x0a], "vsubps ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5c, 0x4a, 0x01], "vsubps ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5c, 0x4a, 0x01], "vsubps ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x5c, 0xca], "vsubps zmm1{k5}{z}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x5c, 0x0a], "vsubps zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x5c, 0x4a, 0x01], "vsubps zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x5c, 0xca], "vsubps zmm1{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x5c, 0xca], "vsubps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x5c, 0x0a], "vsubps zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x5c, 0x0a], "vsubps zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x5c, 0x4a, 0x01], "vsubps zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x5c, 0x4a, 0x01], "vsubps zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x5c, 0xca], "vsubps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x5c, 0x0a], "vsubps xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x5c, 0x4a, 0x01], "vsubps xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x5c, 0xca], "vsubps zmm1{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x5c, 0xca], "vsubps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x5c, 0x0a], "vsubps xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x5c, 0x0a], "vsubps xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x5c, 0x4a, 0x01], "vsubps xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x5c, 0x4a, 0x01], "vsubps xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5c, 0xca], "vsubps zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5c, 0x0a], "vsubps zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5c, 0x4a, 0x01], "vsubps zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5c, 0xca], "vsubps zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5c, 0xca], "vsubps zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5c, 0x0a], "vsubps zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5c, 0x0a], "vsubps zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5c, 0x4a, 0x01], "vsubps zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5c, 0x4a, 0x01], "vsubps zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5c, 0xca], "vsubps xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5c, 0x0a], "vsubps xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5c, 0x4a, 0x01], "vsubps xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5c, 0xca], "vsubps xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5c, 0xca], "vsubps xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5c, 0x0a], "vsubps xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5c, 0x0a], "vsubps xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5c, 0x4a, 0x01], "vsubps xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5c, 0x4a, 0x01], "vsubps xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xfd, 0x5d, 0xca], "vminps zmm1{k5}{z}{sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x5d, 0xca], "vminps zmm1{sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x7d, 0x5d, 0xca], "vminps zmm1{k5}{sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x5d, 0x0a], "vminps ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x5d, 0x4a, 0x01], "vminps ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x5d, 0x0a], "vminps ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x5d, 0x0a], "vminps ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x5d, 0x4a, 0x01], "vminps ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x5d, 0x4a, 0x01], "vminps ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5d, 0xca], "vminps ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5d, 0x0a], "vminps ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5d, 0x4a, 0x01], "vminps ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5d, 0xca], "vminps ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5d, 0xca], "vminps ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5d, 0x0a], "vminps ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5d, 0x0a], "vminps ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5d, 0x4a, 0x01], "vminps ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5d, 0x4a, 0x01], "vminps ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x5d, 0x0a], "vminps zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x5d, 0x4a, 0x01], "vminps zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x5d, 0x0a], "vminps zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x5d, 0x0a], "vminps zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x5d, 0x4a, 0x01], "vminps zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x5d, 0x4a, 0x01], "vminps zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x5d, 0x0a], "vminps xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x5d, 0x4a, 0x01], "vminps xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x5d, 0x0a], "vminps xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x5d, 0x0a], "vminps xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x5d, 0x4a, 0x01], "vminps xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x5d, 0x4a, 0x01], "vminps xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5d, 0xca], "vminps zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5d, 0x0a], "vminps zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5d, 0x4a, 0x01], "vminps zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5d, 0xca], "vminps zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5d, 0xca], "vminps zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5d, 0x0a], "vminps zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5d, 0x0a], "vminps zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5d, 0x4a, 0x01], "vminps zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5d, 0x4a, 0x01], "vminps zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5d, 0xca], "vminps xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5d, 0x0a], "vminps xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5d, 0x4a, 0x01], "vminps xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5d, 0xca], "vminps xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5d, 0xca], "vminps xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5d, 0x0a], "vminps xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5d, 0x0a], "vminps xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5d, 0x4a, 0x01], "vminps xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5d, 0x4a, 0x01], "vminps xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xfd, 0x5e, 0xca], "vdivps zmm1{k5}{z}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x5e, 0xca], "vdivps zmm1{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x7d, 0x5e, 0xca], "vdivps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x5e, 0xca], "vdivps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x5e, 0x0a], "vdivps ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x5e, 0x4a, 0x01], "vdivps ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x5e, 0xca], "vdivps zmm1{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x5e, 0xca], "vdivps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x5e, 0x0a], "vdivps ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x5e, 0x0a], "vdivps ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x5e, 0x4a, 0x01], "vdivps ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x5e, 0x4a, 0x01], "vdivps ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5e, 0xca], "vdivps ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5e, 0x0a], "vdivps ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5e, 0x4a, 0x01], "vdivps ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5e, 0xca], "vdivps ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5e, 0xca], "vdivps ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5e, 0x0a], "vdivps ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5e, 0x0a], "vdivps ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5e, 0x4a, 0x01], "vdivps ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5e, 0x4a, 0x01], "vdivps ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x5e, 0xca], "vdivps zmm1{k5}{z}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x5e, 0x0a], "vdivps zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x5e, 0x4a, 0x01], "vdivps zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x5e, 0xca], "vdivps zmm1{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x5e, 0xca], "vdivps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x5e, 0x0a], "vdivps zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x5e, 0x0a], "vdivps zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x5e, 0x4a, 0x01], "vdivps zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x5e, 0x4a, 0x01], "vdivps zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x5e, 0xca], "vdivps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x5e, 0x0a], "vdivps xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x5e, 0x4a, 0x01], "vdivps xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x5e, 0xca], "vdivps zmm1{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x5e, 0xca], "vdivps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x5e, 0x0a], "vdivps xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x5e, 0x0a], "vdivps xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x5e, 0x4a, 0x01], "vdivps xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x5e, 0x4a, 0x01], "vdivps xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5e, 0xca], "vdivps zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5e, 0x0a], "vdivps zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5e, 0x4a, 0x01], "vdivps zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5e, 0xca], "vdivps zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5e, 0xca], "vdivps zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5e, 0x0a], "vdivps zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5e, 0x0a], "vdivps zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5e, 0x4a, 0x01], "vdivps zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5e, 0x4a, 0x01], "vdivps zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5e, 0xca], "vdivps xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5e, 0x0a], "vdivps xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5e, 0x4a, 0x01], "vdivps xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5e, 0xca], "vdivps xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5e, 0xca], "vdivps xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5e, 0x0a], "vdivps xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5e, 0x0a], "vdivps xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5e, 0x4a, 0x01], "vdivps xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5e, 0x4a, 0x01], "vdivps xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xfd, 0x5f, 0xca], "vmaxps zmm1{k5}{z}{sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x5f, 0xca], "vmaxps zmm1{sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x7d, 0x5f, 0xca], "vmaxps zmm1{k5}{sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x5f, 0x0a], "vmaxps ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x5f, 0x4a, 0x01], "vmaxps ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x5f, 0x0a], "vmaxps ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x5f, 0x0a], "vmaxps ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x5f, 0x4a, 0x01], "vmaxps ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x5f, 0x4a, 0x01], "vmaxps ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5f, 0xca], "vmaxps ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5f, 0x0a], "vmaxps ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5f, 0x4a, 0x01], "vmaxps ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5f, 0xca], "vmaxps ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5f, 0xca], "vmaxps ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5f, 0x0a], "vmaxps ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5f, 0x0a], "vmaxps ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5f, 0x4a, 0x01], "vmaxps ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5f, 0x4a, 0x01], "vmaxps ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x5f, 0x0a], "vmaxps zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x5f, 0x4a, 0x01], "vmaxps zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x5f, 0x0a], "vmaxps zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x5f, 0x0a], "vmaxps zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x5f, 0x4a, 0x01], "vmaxps zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x5f, 0x4a, 0x01], "vmaxps zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x5f, 0x0a], "vmaxps xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x5f, 0x4a, 0x01], "vmaxps xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x5f, 0x0a], "vmaxps xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x5f, 0x0a], "vmaxps xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x5f, 0x4a, 0x01], "vmaxps xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x5f, 0x4a, 0x01], "vmaxps xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5f, 0xca], "vmaxps zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5f, 0x0a], "vmaxps zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5f, 0x4a, 0x01], "vmaxps zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5f, 0xca], "vmaxps zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5f, 0xca], "vmaxps zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5f, 0x0a], "vmaxps zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5f, 0x0a], "vmaxps zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5f, 0x4a, 0x01], "vmaxps zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5f, 0x4a, 0x01], "vmaxps zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5f, 0xca], "vmaxps xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5f, 0x0a], "vmaxps xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5f, 0x4a, 0x01], "vmaxps xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5f, 0xca], "vmaxps xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5f, 0xca], "vmaxps xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5f, 0x0a], "vmaxps xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5f, 0x0a], "vmaxps xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5f, 0x4a, 0x01], "vmaxps xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5f, 0x4a, 0x01], "vmaxps xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xfd, 0x78, 0xca], "vcvttpd2udq ymm1{k5}{z}{sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x78, 0x78, 0xca], "vcvttpd2udq ymm1{sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x7d, 0x78, 0xca], "vcvttpd2udq ymm1{k5}{sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xbd, 0x78, 0x0a], "vcvttpd2udq xmm1{k5}{z}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xbd, 0x78, 0x4a, 0x01], "vcvttpd2udq xmm1{k5}{z}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x38, 0x78, 0x0a], "vcvttpd2udq xmm1, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x3d, 0x78, 0x0a], "vcvttpd2udq xmm1{k5}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x38, 0x78, 0x4a, 0x01], "vcvttpd2udq xmm1, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x3d, 0x78, 0x4a, 0x01], "vcvttpd2udq xmm1{k5}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xad, 0x78, 0xca], "vcvttpd2udq xmm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xad, 0x78, 0x0a], "vcvttpd2udq xmm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xad, 0x78, 0x4a, 0x01], "vcvttpd2udq xmm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x28, 0x78, 0xca], "vcvttpd2udq xmm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x2d, 0x78, 0xca], "vcvttpd2udq xmm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x28, 0x78, 0x0a], "vcvttpd2udq xmm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x2d, 0x78, 0x0a], "vcvttpd2udq xmm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x28, 0x78, 0x4a, 0x01], "vcvttpd2udq xmm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x2d, 0x78, 0x4a, 0x01], "vcvttpd2udq xmm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xfd, 0x78, 0xca], "vcvttps2udq zmm1{k5}{z}{sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x78, 0xca], "vcvttps2udq zmm1{sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x7d, 0x78, 0xca], "vcvttps2udq zmm1{k5}{sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x78, 0x0a], "vcvttps2udq ymm1{k5}{z}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x78, 0x4a, 0x01], "vcvttps2udq ymm1{k5}{z}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x78, 0x0a], "vcvttps2udq ymm1, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x78, 0x0a], "vcvttps2udq ymm1{k5}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x78, 0x4a, 0x01], "vcvttps2udq ymm1, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x78, 0x4a, 0x01], "vcvttps2udq ymm1{k5}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x78, 0xca], "vcvttps2udq ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x78, 0x0a], "vcvttps2udq ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x78, 0x4a, 0x01], "vcvttps2udq ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x78, 0xca], "vcvttps2udq ymm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x78, 0xca], "vcvttps2udq ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x78, 0x0a], "vcvttps2udq ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x78, 0x0a], "vcvttps2udq ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x78, 0x4a, 0x01], "vcvttps2udq ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x78, 0x4a, 0x01], "vcvttps2udq ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xdd, 0x78, 0x0a], "vcvttpd2udq ymm1{k5}{z}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xdd, 0x78, 0x4a, 0x01], "vcvttpd2udq ymm1{k5}{z}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x58, 0x78, 0x0a], "vcvttpd2udq ymm1, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x5d, 0x78, 0x0a], "vcvttpd2udq ymm1{k5}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x58, 0x78, 0x4a, 0x01], "vcvttpd2udq ymm1, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x5d, 0x78, 0x4a, 0x01], "vcvttpd2udq ymm1{k5}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x9d, 0x78, 0x0a], "vcvttpd2udq xmm1{k5}{z}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x9d, 0x78, 0x4a, 0x01], "vcvttpd2udq xmm1{k5}{z}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x18, 0x78, 0x0a], "vcvttpd2udq xmm1, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x1d, 0x78, 0x0a], "vcvttpd2udq xmm1{k5}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x18, 0x78, 0x4a, 0x01], "vcvttpd2udq xmm1, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x1d, 0x78, 0x4a, 0x01], "vcvttpd2udq xmm1{k5}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xcd, 0x78, 0xca], "vcvttpd2udq ymm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xcd, 0x78, 0x0a], "vcvttpd2udq ymm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xcd, 0x78, 0x4a, 0x01], "vcvttpd2udq ymm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x48, 0x78, 0xca], "vcvttpd2udq ymm1, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x4d, 0x78, 0xca], "vcvttpd2udq ymm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x48, 0x78, 0x0a], "vcvttpd2udq ymm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x4d, 0x78, 0x0a], "vcvttpd2udq ymm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x48, 0x78, 0x4a, 0x01], "vcvttpd2udq ymm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x4d, 0x78, 0x4a, 0x01], "vcvttpd2udq ymm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x8d, 0x78, 0xca], "vcvttpd2udq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x8d, 0x78, 0x0a], "vcvttpd2udq xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x8d, 0x78, 0x4a, 0x01], "vcvttpd2udq xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x08, 0x78, 0xca], "vcvttpd2udq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x0d, 0x78, 0xca], "vcvttpd2udq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x08, 0x78, 0x0a], "vcvttpd2udq xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x0d, 0x78, 0x0a], "vcvttpd2udq xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x08, 0x78, 0x4a, 0x01], "vcvttpd2udq xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x0d, 0x78, 0x4a, 0x01], "vcvttpd2udq xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x78, 0x0a], "vcvttps2udq zmm1{k5}{z}, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x78, 0x4a, 0x01], "vcvttps2udq zmm1{k5}{z}, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x78, 0x0a], "vcvttps2udq zmm1, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x78, 0x0a], "vcvttps2udq zmm1{k5}, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x78, 0x4a, 0x01], "vcvttps2udq zmm1, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x78, 0x4a, 0x01], "vcvttps2udq zmm1{k5}, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x78, 0x0a], "vcvttps2udq xmm1{k5}{z}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x78, 0x4a, 0x01], "vcvttps2udq xmm1{k5}{z}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x78, 0x0a], "vcvttps2udq xmm1, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x78, 0x0a], "vcvttps2udq xmm1{k5}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x78, 0x4a, 0x01], "vcvttps2udq xmm1, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x78, 0x4a, 0x01], "vcvttps2udq xmm1{k5}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x78, 0xca], "vcvttps2udq zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x78, 0x0a], "vcvttps2udq zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x78, 0x4a, 0x01], "vcvttps2udq zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x78, 0xca], "vcvttps2udq zmm1, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x78, 0xca], "vcvttps2udq zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x78, 0x0a], "vcvttps2udq zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x78, 0x0a], "vcvttps2udq zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x78, 0x4a, 0x01], "vcvttps2udq zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x78, 0x4a, 0x01], "vcvttps2udq zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x78, 0xca], "vcvttps2udq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x78, 0x0a], "vcvttps2udq xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x78, 0x4a, 0x01], "vcvttps2udq xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x78, 0xca], "vcvttps2udq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x78, 0xca], "vcvttps2udq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x78, 0x0a], "vcvttps2udq xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x78, 0x0a], "vcvttps2udq xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x78, 0x4a, 0x01], "vcvttps2udq xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x78, 0x4a, 0x01], "vcvttps2udq xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xfd, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{z}{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x78, 0x79, 0xca], "vcvtpd2udq ymm1{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x7d, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xbd, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{z}{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xbd, 0x79, 0x0a], "vcvtpd2udq xmm1{k5}{z}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xbd, 0x79, 0x4a, 0x01], "vcvtpd2udq xmm1{k5}{z}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x38, 0x79, 0xca], "vcvtpd2udq ymm1{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x3d, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x38, 0x79, 0x0a], "vcvtpd2udq xmm1, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x3d, 0x79, 0x0a], "vcvtpd2udq xmm1{k5}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x38, 0x79, 0x4a, 0x01], "vcvtpd2udq xmm1, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x3d, 0x79, 0x4a, 0x01], "vcvtpd2udq xmm1{k5}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xad, 0x79, 0xca], "vcvtpd2udq xmm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xad, 0x79, 0x0a], "vcvtpd2udq xmm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xad, 0x79, 0x4a, 0x01], "vcvtpd2udq xmm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x28, 0x79, 0xca], "vcvtpd2udq xmm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x2d, 0x79, 0xca], "vcvtpd2udq xmm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x28, 0x79, 0x0a], "vcvtpd2udq xmm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x2d, 0x79, 0x0a], "vcvtpd2udq xmm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x28, 0x79, 0x4a, 0x01], "vcvtpd2udq xmm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x2d, 0x79, 0x4a, 0x01], "vcvtpd2udq xmm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xfd, 0x79, 0xca], "vcvtps2udq zmm1{k5}{z}{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x79, 0xca], "vcvtps2udq zmm1{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x7d, 0x79, 0xca], "vcvtps2udq zmm1{k5}{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x79, 0xca], "vcvtps2udq zmm1{k5}{z}{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x79, 0x0a], "vcvtps2udq ymm1{k5}{z}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x79, 0x4a, 0x01], "vcvtps2udq ymm1{k5}{z}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x79, 0xca], "vcvtps2udq zmm1{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x79, 0xca], "vcvtps2udq zmm1{k5}{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x79, 0x0a], "vcvtps2udq ymm1, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x79, 0x0a], "vcvtps2udq ymm1{k5}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x79, 0x4a, 0x01], "vcvtps2udq ymm1, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x79, 0x4a, 0x01], "vcvtps2udq ymm1{k5}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x79, 0xca], "vcvtps2udq ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x79, 0x0a], "vcvtps2udq ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x79, 0x4a, 0x01], "vcvtps2udq ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x79, 0xca], "vcvtps2udq ymm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x79, 0xca], "vcvtps2udq ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x79, 0x0a], "vcvtps2udq ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x79, 0x0a], "vcvtps2udq ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x79, 0x4a, 0x01], "vcvtps2udq ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x79, 0x4a, 0x01], "vcvtps2udq ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xdd, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{z}{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xdd, 0x79, 0x0a], "vcvtpd2udq ymm1{k5}{z}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xdd, 0x79, 0x4a, 0x01], "vcvtpd2udq ymm1{k5}{z}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x58, 0x79, 0xca], "vcvtpd2udq ymm1{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x5d, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x58, 0x79, 0x0a], "vcvtpd2udq ymm1, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x5d, 0x79, 0x0a], "vcvtpd2udq ymm1{k5}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x58, 0x79, 0x4a, 0x01], "vcvtpd2udq ymm1, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x5d, 0x79, 0x4a, 0x01], "vcvtpd2udq ymm1{k5}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x9d, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{z}{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x9d, 0x79, 0x0a], "vcvtpd2udq xmm1{k5}{z}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x9d, 0x79, 0x4a, 0x01], "vcvtpd2udq xmm1{k5}{z}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x18, 0x79, 0xca], "vcvtpd2udq ymm1{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x1d, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x18, 0x79, 0x0a], "vcvtpd2udq xmm1, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x1d, 0x79, 0x0a], "vcvtpd2udq xmm1{k5}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x18, 0x79, 0x4a, 0x01], "vcvtpd2udq xmm1, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x1d, 0x79, 0x4a, 0x01], "vcvtpd2udq xmm1{k5}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xcd, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xcd, 0x79, 0x0a], "vcvtpd2udq ymm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xcd, 0x79, 0x4a, 0x01], "vcvtpd2udq ymm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x48, 0x79, 0xca], "vcvtpd2udq ymm1, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x4d, 0x79, 0xca], "vcvtpd2udq ymm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x48, 0x79, 0x0a], "vcvtpd2udq ymm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x4d, 0x79, 0x0a], "vcvtpd2udq ymm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x48, 0x79, 0x4a, 0x01], "vcvtpd2udq ymm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x4d, 0x79, 0x4a, 0x01], "vcvtpd2udq ymm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x8d, 0x79, 0xca], "vcvtpd2udq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x8d, 0x79, 0x0a], "vcvtpd2udq xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x8d, 0x79, 0x4a, 0x01], "vcvtpd2udq xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x08, 0x79, 0xca], "vcvtpd2udq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x0d, 0x79, 0xca], "vcvtpd2udq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x08, 0x79, 0x0a], "vcvtpd2udq xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x0d, 0x79, 0x0a], "vcvtpd2udq xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x08, 0x79, 0x4a, 0x01], "vcvtpd2udq xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x0d, 0x79, 0x4a, 0x01], "vcvtpd2udq xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x79, 0xca], "vcvtps2udq zmm1{k5}{z}{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x79, 0x0a], "vcvtps2udq zmm1{k5}{z}, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x79, 0x4a, 0x01], "vcvtps2udq zmm1{k5}{z}, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x79, 0xca], "vcvtps2udq zmm1{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x79, 0xca], "vcvtps2udq zmm1{k5}{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x79, 0x0a], "vcvtps2udq zmm1, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x79, 0x0a], "vcvtps2udq zmm1{k5}, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x79, 0x4a, 0x01], "vcvtps2udq zmm1, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x79, 0x4a, 0x01], "vcvtps2udq zmm1{k5}, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x79, 0xca], "vcvtps2udq zmm1{k5}{z}{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x79, 0x0a], "vcvtps2udq xmm1{k5}{z}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x79, 0x4a, 0x01], "vcvtps2udq xmm1{k5}{z}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x79, 0xca], "vcvtps2udq zmm1{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x79, 0xca], "vcvtps2udq zmm1{k5}{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x79, 0x0a], "vcvtps2udq xmm1, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x79, 0x0a], "vcvtps2udq xmm1{k5}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x79, 0x4a, 0x01], "vcvtps2udq xmm1, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x79, 0x4a, 0x01], "vcvtps2udq xmm1{k5}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x79, 0xca], "vcvtps2udq zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x79, 0x0a], "vcvtps2udq zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x79, 0x4a, 0x01], "vcvtps2udq zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x79, 0xca], "vcvtps2udq zmm1, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x79, 0xca], "vcvtps2udq zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x79, 0x0a], "vcvtps2udq zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x79, 0x0a], "vcvtps2udq zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x79, 0x4a, 0x01], "vcvtps2udq zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x79, 0x4a, 0x01], "vcvtps2udq zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x79, 0xca], "vcvtps2udq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x79, 0x0a], "vcvtps2udq xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x79, 0x4a, 0x01], "vcvtps2udq xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x79, 0xca], "vcvtps2udq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x79, 0xca], "vcvtps2udq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x79, 0x0a], "vcvtps2udq xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x79, 0x0a], "vcvtps2udq xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x79, 0x4a, 0x01], "vcvtps2udq xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x79, 0x4a, 0x01], "vcvtps2udq xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0xc2, 0xca, 0xcc], "vcmpps k1{sae}, zmm0, zmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x7d, 0xc2, 0xca, 0xcc], "vcmpps k1{k5}{sae}, zmm0, zmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0xc2, 0x0a, 0xcc], "vcmpps k1, ymm0, dword [edx]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0xc2, 0x0a, 0xcc], "vcmpps k1{k5}, ymm0, dword [edx]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0xc2, 0x4a, 0x01, 0xcc], "vcmpps k1, ymm0, dword [edx + 0x4]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0xc2, 0x4a, 0x01, 0xcc], "vcmpps k1{k5}, ymm0, dword [edx + 0x4]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0xc2, 0xca, 0xcc], "vcmpps k1, ymm0, ymm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0xc2, 0xca, 0xcc], "vcmpps k1{k5}, ymm0, ymm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0xc2, 0x0a, 0xcc], "vcmpps k1, ymm0, ymmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0xc2, 0x0a, 0xcc], "vcmpps k1{k5}, ymm0, ymmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0xc2, 0x4a, 0x01, 0xcc], "vcmpps k1, ymm0, ymmword [edx + 0x20], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0xc2, 0x4a, 0x01, 0xcc], "vcmpps k1{k5}, ymm0, ymmword [edx + 0x20], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0xc2, 0x0a, 0xcc], "vcmpps k1, zmm0, dword [edx]{1to16}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0xc2, 0x0a, 0xcc], "vcmpps k1{k5}, zmm0, dword [edx]{1to16}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0xc2, 0x4a, 0x01, 0xcc], "vcmpps k1, zmm0, dword [edx + 0x4]{1to16}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0xc2, 0x4a, 0x01, 0xcc], "vcmpps k1{k5}, zmm0, dword [edx + 0x4]{1to16}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0xc2, 0x0a, 0xcc], "vcmpps k1, xmm0, dword [edx]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0xc2, 0x0a, 0xcc], "vcmpps k1{k5}, xmm0, dword [edx]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0xc2, 0x4a, 0x01, 0xcc], "vcmpps k1, xmm0, dword [edx + 0x4]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0xc2, 0x4a, 0x01, 0xcc], "vcmpps k1{k5}, xmm0, dword [edx + 0x4]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0xc2, 0xca, 0xcc], "vcmpps k1, zmm0, zmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0xc2, 0xca, 0xcc], "vcmpps k1{k5}, zmm0, zmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0xc2, 0x0a, 0xcc], "vcmpps k1, zmm0, zmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0xc2, 0x0a, 0xcc], "vcmpps k1{k5}, zmm0, zmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0xc2, 0x4a, 0x01, 0xcc], "vcmpps k1, zmm0, zmmword [edx + 0x40], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0xc2, 0x4a, 0x01, 0xcc], "vcmpps k1{k5}, zmm0, zmmword [edx + 0x40], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0xc2, 0xca, 0xcc], "vcmpps k1, xmm0, xmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0xc2, 0xca, 0xcc], "vcmpps k1{k5}, xmm0, xmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0xc2, 0x0a, 0xcc], "vcmpps k1, xmm0, xmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0xc2, 0x0a, 0xcc], "vcmpps k1{k5}, xmm0, xmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0xc2, 0x4a, 0x01, 0xcc], "vcmpps k1, xmm0, xmmword [edx + 0x10], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0xc2, 0x4a, 0x01, 0xcc], "vcmpps k1{k5}, xmm0, xmmword [edx + 0x10], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0xc6, 0x0a, 0xcc], "vshufps ymm1{k5}{z}, ymm0, dword [edx]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0xc6, 0x4a, 0x01, 0xcc], "vshufps ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0xc6, 0x0a, 0xcc], "vshufps ymm1, ymm0, dword [edx]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0xc6, 0x0a, 0xcc], "vshufps ymm1{k5}, ymm0, dword [edx]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0xc6, 0x4a, 0x01, 0xcc], "vshufps ymm1, ymm0, dword [edx + 0x4]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0xc6, 0x4a, 0x01, 0xcc], "vshufps ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0xc6, 0xca, 0xcc], "vshufps ymm1{k5}{z}, ymm0, ymm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0xc6, 0x0a, 0xcc], "vshufps ymm1{k5}{z}, ymm0, ymmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0xc6, 0x4a, 0x01, 0xcc], "vshufps ymm1{k5}{z}, ymm0, ymmword [edx + 0x20], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0xc6, 0xca, 0xcc], "vshufps ymm1, ymm0, ymm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0xc6, 0xca, 0xcc], "vshufps ymm1{k5}, ymm0, ymm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0xc6, 0x0a, 0xcc], "vshufps ymm1, ymm0, ymmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0xc6, 0x0a, 0xcc], "vshufps ymm1{k5}, ymm0, ymmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0xc6, 0x4a, 0x01, 0xcc], "vshufps ymm1, ymm0, ymmword [edx + 0x20], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0xc6, 0x4a, 0x01, 0xcc], "vshufps ymm1{k5}, ymm0, ymmword [edx + 0x20], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0xc6, 0x0a, 0xcc], "vshufps zmm1{k5}{z}, zmm0, dword [edx]{1to16}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0xc6, 0x4a, 0x01, 0xcc], "vshufps zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0xc6, 0x0a, 0xcc], "vshufps zmm1, zmm0, dword [edx]{1to16}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0xc6, 0x0a, 0xcc], "vshufps zmm1{k5}, zmm0, dword [edx]{1to16}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0xc6, 0x4a, 0x01, 0xcc], "vshufps zmm1, zmm0, dword [edx + 0x4]{1to16}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0xc6, 0x4a, 0x01, 0xcc], "vshufps zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0xc6, 0x0a, 0xcc], "vshufps xmm1{k5}{z}, xmm0, dword [edx]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0xc6, 0x4a, 0x01, 0xcc], "vshufps xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0xc6, 0x0a, 0xcc], "vshufps xmm1, xmm0, dword [edx]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0xc6, 0x0a, 0xcc], "vshufps xmm1{k5}, xmm0, dword [edx]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0xc6, 0x4a, 0x01, 0xcc], "vshufps xmm1, xmm0, dword [edx + 0x4]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0xc6, 0x4a, 0x01, 0xcc], "vshufps xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0xc6, 0xca, 0xcc], "vshufps zmm1{k5}{z}, zmm0, zmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0xc6, 0x0a, 0xcc], "vshufps zmm1{k5}{z}, zmm0, zmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0xc6, 0x4a, 0x01, 0xcc], "vshufps zmm1{k5}{z}, zmm0, zmmword [edx + 0x40], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0xc6, 0xca, 0xcc], "vshufps zmm1, zmm0, zmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0xc6, 0xca, 0xcc], "vshufps zmm1{k5}, zmm0, zmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0xc6, 0x0a, 0xcc], "vshufps zmm1, zmm0, zmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0xc6, 0x0a, 0xcc], "vshufps zmm1{k5}, zmm0, zmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0xc6, 0x4a, 0x01, 0xcc], "vshufps zmm1, zmm0, zmmword [edx + 0x40], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0xc6, 0x4a, 0x01, 0xcc], "vshufps zmm1{k5}, zmm0, zmmword [edx + 0x40], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0xc6, 0xca, 0xcc], "vshufps xmm1{k5}{z}, xmm0, xmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0xc6, 0x0a, 0xcc], "vshufps xmm1{k5}{z}, xmm0, xmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0xc6, 0x4a, 0x01, 0xcc], "vshufps xmm1{k5}{z}, xmm0, xmmword [edx + 0x10], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0xc6, 0xca, 0xcc], "vshufps xmm1, xmm0, xmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0xc6, 0xca, 0xcc], "vshufps xmm1{k5}, xmm0, xmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0xc6, 0x0a, 0xcc], "vshufps xmm1, xmm0, xmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0xc6, 0x0a, 0xcc], "vshufps xmm1{k5}, xmm0, xmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0xc6, 0x4a, 0x01, 0xcc], "vshufps xmm1, xmm0, xmmword [edx + 0x10], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0xc6, 0x4a, 0x01, 0xcc], "vshufps xmm1{k5}, xmm0, xmmword [edx + 0x10], 0xcc");
}

#[test]
fn tests_66_0f() {
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x10, 0xca], "vmovupd ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x10, 0x0a], "vmovupd ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x10, 0x4a, 0x01], "vmovupd ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x10, 0xca], "vmovupd ymm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x10, 0xca], "vmovupd ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x10, 0x0a], "vmovupd ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x10, 0x0a], "vmovupd ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x10, 0x4a, 0x01], "vmovupd ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x10, 0x4a, 0x01], "vmovupd ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x10, 0xca], "vmovupd zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x10, 0x0a], "vmovupd zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x10, 0x4a, 0x01], "vmovupd zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x10, 0xca], "vmovupd zmm1, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x10, 0xca], "vmovupd zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x10, 0x0a], "vmovupd zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x10, 0x0a], "vmovupd zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x10, 0x4a, 0x01], "vmovupd zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x10, 0x4a, 0x01], "vmovupd zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x10, 0xca], "vmovupd xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x10, 0x0a], "vmovupd xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x10, 0x4a, 0x01], "vmovupd xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x10, 0xca], "vmovupd xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x10, 0xca], "vmovupd xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x10, 0x0a], "vmovupd xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x10, 0x0a], "vmovupd xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x10, 0x4a, 0x01], "vmovupd xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x10, 0x4a, 0x01], "vmovupd xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x11, 0xca], "vmovupd ymm2{k5}{z}, ymm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x11, 0xca], "vmovupd ymm2, ymm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x11, 0xca], "vmovupd ymm2{k5}, ymm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x11, 0x0a], "vmovupd ymmword [edx], ymm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x11, 0x0a], "vmovupd ymmword [edx]{k5}, ymm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x11, 0x4a, 0x01], "vmovupd ymmword [edx + 0x20], ymm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x11, 0x4a, 0x01], "vmovupd ymmword [edx + 0x20]{k5}, ymm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x11, 0xca], "vmovupd zmm2{k5}{z}, zmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x11, 0xca], "vmovupd zmm2, zmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x11, 0xca], "vmovupd zmm2{k5}, zmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x11, 0x0a], "vmovupd zmmword [edx], zmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x11, 0x0a], "vmovupd zmmword [edx]{k5}, zmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x11, 0x4a, 0x01], "vmovupd zmmword [edx + 0x40], zmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x11, 0x4a, 0x01], "vmovupd zmmword [edx + 0x40]{k5}, zmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x11, 0xca], "vmovupd xmm2{k5}{z}, xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x11, 0xca], "vmovupd xmm2, xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x11, 0xca], "vmovupd xmm2{k5}, xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x11, 0x0a], "vmovupd xmmword [edx], xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x11, 0x0a], "vmovupd xmmword [edx]{k5}, xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x11, 0x4a, 0x01], "vmovupd xmmword [edx + 0x10], xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x11, 0x4a, 0x01], "vmovupd xmmword [edx + 0x10]{k5}, xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x12, 0x0a], "vmovlpd xmm1, xmm0, qword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x12, 0x4a, 0x01], "vmovlpd xmm1, xmm0, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x13, 0x0a], "vmovlpd qword [edx], xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x13, 0x4a, 0x01], "vmovlpd qword [edx + 0x8], xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x14, 0x0a], "vunpcklpd ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x14, 0x4a, 0x01], "vunpcklpd ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x14, 0x0a], "vunpcklpd ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x14, 0x0a], "vunpcklpd ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x14, 0x4a, 0x01], "vunpcklpd ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x14, 0x4a, 0x01], "vunpcklpd ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x14, 0xca], "vunpcklpd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x14, 0x0a], "vunpcklpd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x14, 0x4a, 0x01], "vunpcklpd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x14, 0xca], "vunpcklpd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x14, 0xca], "vunpcklpd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x14, 0x0a], "vunpcklpd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x14, 0x0a], "vunpcklpd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x14, 0x4a, 0x01], "vunpcklpd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x14, 0x4a, 0x01], "vunpcklpd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x14, 0x0a], "vunpcklpd zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x14, 0x4a, 0x01], "vunpcklpd zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x14, 0x0a], "vunpcklpd zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x14, 0x0a], "vunpcklpd zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x14, 0x4a, 0x01], "vunpcklpd zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x14, 0x4a, 0x01], "vunpcklpd zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x14, 0x0a], "vunpcklpd xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x14, 0x4a, 0x01], "vunpcklpd xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x14, 0x0a], "vunpcklpd xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x14, 0x0a], "vunpcklpd xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x14, 0x4a, 0x01], "vunpcklpd xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x14, 0x4a, 0x01], "vunpcklpd xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x14, 0xca], "vunpcklpd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x14, 0x0a], "vunpcklpd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x14, 0x4a, 0x01], "vunpcklpd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x14, 0xca], "vunpcklpd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x14, 0xca], "vunpcklpd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x14, 0x0a], "vunpcklpd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x14, 0x0a], "vunpcklpd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x14, 0x4a, 0x01], "vunpcklpd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x14, 0x4a, 0x01], "vunpcklpd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x14, 0xca], "vunpcklpd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x14, 0x0a], "vunpcklpd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x14, 0x4a, 0x01], "vunpcklpd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x14, 0xca], "vunpcklpd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x14, 0xca], "vunpcklpd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x14, 0x0a], "vunpcklpd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x14, 0x0a], "vunpcklpd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x14, 0x4a, 0x01], "vunpcklpd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x14, 0x4a, 0x01], "vunpcklpd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x15, 0x0a], "vunpckhpd ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x15, 0x4a, 0x01], "vunpckhpd ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x15, 0x0a], "vunpckhpd ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x15, 0x0a], "vunpckhpd ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x15, 0x4a, 0x01], "vunpckhpd ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x15, 0x4a, 0x01], "vunpckhpd ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x15, 0xca], "vunpckhpd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x15, 0x0a], "vunpckhpd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x15, 0x4a, 0x01], "vunpckhpd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x15, 0xca], "vunpckhpd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x15, 0xca], "vunpckhpd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x15, 0x0a], "vunpckhpd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x15, 0x0a], "vunpckhpd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x15, 0x4a, 0x01], "vunpckhpd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x15, 0x4a, 0x01], "vunpckhpd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x15, 0x0a], "vunpckhpd zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x15, 0x4a, 0x01], "vunpckhpd zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x15, 0x0a], "vunpckhpd zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x15, 0x0a], "vunpckhpd zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x15, 0x4a, 0x01], "vunpckhpd zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x15, 0x4a, 0x01], "vunpckhpd zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x15, 0x0a], "vunpckhpd xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x15, 0x4a, 0x01], "vunpckhpd xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x15, 0x0a], "vunpckhpd xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x15, 0x0a], "vunpckhpd xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x15, 0x4a, 0x01], "vunpckhpd xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x15, 0x4a, 0x01], "vunpckhpd xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x15, 0xca], "vunpckhpd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x15, 0x0a], "vunpckhpd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x15, 0x4a, 0x01], "vunpckhpd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x15, 0xca], "vunpckhpd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x15, 0xca], "vunpckhpd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x15, 0x0a], "vunpckhpd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x15, 0x0a], "vunpckhpd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x15, 0x4a, 0x01], "vunpckhpd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x15, 0x4a, 0x01], "vunpckhpd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x15, 0xca], "vunpckhpd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x15, 0x0a], "vunpckhpd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x15, 0x4a, 0x01], "vunpckhpd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x15, 0xca], "vunpckhpd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x15, 0xca], "vunpckhpd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x15, 0x0a], "vunpckhpd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x15, 0x0a], "vunpckhpd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x15, 0x4a, 0x01], "vunpckhpd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x15, 0x4a, 0x01], "vunpckhpd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x16, 0x0a], "vmovhpd xmm1, xmm0, qword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x16, 0x4a, 0x01], "vmovhpd xmm1, xmm0, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x17, 0x0a], "vmovhpd qword [edx], xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x17, 0x4a, 0x01], "vmovhpd qword [edx + 0x8], xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x28, 0xca], "vmovapd ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x28, 0x0a], "vmovapd ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x28, 0x4a, 0x01], "vmovapd ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x28, 0xca], "vmovapd ymm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x28, 0xca], "vmovapd ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x28, 0x0a], "vmovapd ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x28, 0x0a], "vmovapd ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x28, 0x4a, 0x01], "vmovapd ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x28, 0x4a, 0x01], "vmovapd ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x28, 0xca], "vmovapd zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x28, 0x0a], "vmovapd zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x28, 0x4a, 0x01], "vmovapd zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x28, 0xca], "vmovapd zmm1, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x28, 0xca], "vmovapd zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x28, 0x0a], "vmovapd zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x28, 0x0a], "vmovapd zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x28, 0x4a, 0x01], "vmovapd zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x28, 0x4a, 0x01], "vmovapd zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x28, 0xca], "vmovapd xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x28, 0x0a], "vmovapd xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x28, 0x4a, 0x01], "vmovapd xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x28, 0xca], "vmovapd xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x28, 0xca], "vmovapd xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x28, 0x0a], "vmovapd xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x28, 0x0a], "vmovapd xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x28, 0x4a, 0x01], "vmovapd xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x28, 0x4a, 0x01], "vmovapd xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x29, 0xca], "vmovapd ymm2{k5}{z}, ymm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x29, 0xca], "vmovapd ymm2, ymm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x29, 0xca], "vmovapd ymm2{k5}, ymm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x29, 0x0a], "vmovapd ymmword [edx], ymm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x29, 0x0a], "vmovapd ymmword [edx]{k5}, ymm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x29, 0x4a, 0x01], "vmovapd ymmword [edx + 0x20], ymm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x29, 0x4a, 0x01], "vmovapd ymmword [edx + 0x20]{k5}, ymm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x29, 0xca], "vmovapd zmm2{k5}{z}, zmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x29, 0xca], "vmovapd zmm2, zmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x29, 0xca], "vmovapd zmm2{k5}, zmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x29, 0x0a], "vmovapd zmmword [edx], zmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x29, 0x0a], "vmovapd zmmword [edx]{k5}, zmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x29, 0x4a, 0x01], "vmovapd zmmword [edx + 0x40], zmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x29, 0x4a, 0x01], "vmovapd zmmword [edx + 0x40]{k5}, zmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x29, 0xca], "vmovapd xmm2{k5}{z}, xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x29, 0xca], "vmovapd xmm2, xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x29, 0xca], "vmovapd xmm2{k5}, xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x29, 0x0a], "vmovapd xmmword [edx], xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x29, 0x0a], "vmovapd xmmword [edx]{k5}, xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x29, 0x4a, 0x01], "vmovapd xmmword [edx + 0x10], xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x29, 0x4a, 0x01], "vmovapd xmmword [edx + 0x10]{k5}, xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x2b, 0x0a], "vmovntpd ymmword [edx], ymm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x2b, 0x4a, 0x01], "vmovntpd ymmword [edx + 0x20], ymm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x2b, 0x0a], "vmovntpd zmmword [edx], zmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x2b, 0x4a, 0x01], "vmovntpd zmmword [edx + 0x40], zmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x2b, 0x0a], "vmovntpd xmmword [edx], xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x2b, 0x4a, 0x01], "vmovntpd xmmword [edx + 0x10], xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x2e, 0xca], "vucomisd xmm1{sae}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x2e, 0xca], "vucomisd xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x2e, 0x0a], "vucomisd xmm1, qword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x2e, 0x4a, 0x01], "vucomisd xmm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x2f, 0xca], "vcomisd xmm1{sae}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x2f, 0xca], "vcomisd xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x2f, 0x0a], "vcomisd xmm1, qword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x2f, 0x4a, 0x01], "vcomisd xmm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0x51, 0xca], "vsqrtpd zmm1{k5}{z}{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x51, 0xca], "vsqrtpd zmm1{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0x51, 0xca], "vsqrtpd zmm1{k5}{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x51, 0xca], "vsqrtpd zmm1{k5}{z}{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x51, 0x0a], "vsqrtpd ymm1{k5}{z}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x51, 0x4a, 0x01], "vsqrtpd ymm1{k5}{z}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x51, 0xca], "vsqrtpd zmm1{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x51, 0xca], "vsqrtpd zmm1{k5}{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x51, 0x0a], "vsqrtpd ymm1, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x51, 0x0a], "vsqrtpd ymm1{k5}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x51, 0x4a, 0x01], "vsqrtpd ymm1, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x51, 0x4a, 0x01], "vsqrtpd ymm1{k5}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x51, 0xca], "vsqrtpd ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x51, 0x0a], "vsqrtpd ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x51, 0x4a, 0x01], "vsqrtpd ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x51, 0xca], "vsqrtpd ymm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x51, 0xca], "vsqrtpd ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x51, 0x0a], "vsqrtpd ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x51, 0x0a], "vsqrtpd ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x51, 0x4a, 0x01], "vsqrtpd ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x51, 0x4a, 0x01], "vsqrtpd ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x51, 0xca], "vsqrtpd zmm1{k5}{z}{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x51, 0x0a], "vsqrtpd zmm1{k5}{z}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x51, 0x4a, 0x01], "vsqrtpd zmm1{k5}{z}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x51, 0xca], "vsqrtpd zmm1{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x51, 0xca], "vsqrtpd zmm1{k5}{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x51, 0x0a], "vsqrtpd zmm1, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x51, 0x0a], "vsqrtpd zmm1{k5}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x51, 0x4a, 0x01], "vsqrtpd zmm1, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x51, 0x4a, 0x01], "vsqrtpd zmm1{k5}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x51, 0xca], "vsqrtpd zmm1{k5}{z}{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x51, 0x0a], "vsqrtpd xmm1{k5}{z}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x51, 0x4a, 0x01], "vsqrtpd xmm1{k5}{z}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x51, 0xca], "vsqrtpd zmm1{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x51, 0xca], "vsqrtpd zmm1{k5}{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x51, 0x0a], "vsqrtpd xmm1, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x51, 0x0a], "vsqrtpd xmm1{k5}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x51, 0x4a, 0x01], "vsqrtpd xmm1, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x51, 0x4a, 0x01], "vsqrtpd xmm1{k5}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x51, 0xca], "vsqrtpd zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x51, 0x0a], "vsqrtpd zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x51, 0x4a, 0x01], "vsqrtpd zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x51, 0xca], "vsqrtpd zmm1, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x51, 0xca], "vsqrtpd zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x51, 0x0a], "vsqrtpd zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x51, 0x0a], "vsqrtpd zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x51, 0x4a, 0x01], "vsqrtpd zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x51, 0x4a, 0x01], "vsqrtpd zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x51, 0xca], "vsqrtpd xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x51, 0x0a], "vsqrtpd xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x51, 0x4a, 0x01], "vsqrtpd xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x51, 0xca], "vsqrtpd xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x51, 0xca], "vsqrtpd xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x51, 0x0a], "vsqrtpd xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x51, 0x0a], "vsqrtpd xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x51, 0x4a, 0x01], "vsqrtpd xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x51, 0x4a, 0x01], "vsqrtpd xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x54, 0x0a], "vandpd ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x54, 0x4a, 0x01], "vandpd ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x54, 0x0a], "vandpd ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x54, 0x0a], "vandpd ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x54, 0x4a, 0x01], "vandpd ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x54, 0x4a, 0x01], "vandpd ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x54, 0xca], "vandpd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x54, 0x0a], "vandpd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x54, 0x4a, 0x01], "vandpd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x54, 0xca], "vandpd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x54, 0xca], "vandpd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x54, 0x0a], "vandpd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x54, 0x0a], "vandpd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x54, 0x4a, 0x01], "vandpd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x54, 0x4a, 0x01], "vandpd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x54, 0x0a], "vandpd zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x54, 0x4a, 0x01], "vandpd zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x54, 0x0a], "vandpd zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x54, 0x0a], "vandpd zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x54, 0x4a, 0x01], "vandpd zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x54, 0x4a, 0x01], "vandpd zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x54, 0x0a], "vandpd xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x54, 0x4a, 0x01], "vandpd xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x54, 0x0a], "vandpd xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x54, 0x0a], "vandpd xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x54, 0x4a, 0x01], "vandpd xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x54, 0x4a, 0x01], "vandpd xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x54, 0xca], "vandpd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x54, 0x0a], "vandpd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x54, 0x4a, 0x01], "vandpd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x54, 0xca], "vandpd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x54, 0xca], "vandpd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x54, 0x0a], "vandpd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x54, 0x0a], "vandpd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x54, 0x4a, 0x01], "vandpd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x54, 0x4a, 0x01], "vandpd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x54, 0xca], "vandpd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x54, 0x0a], "vandpd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x54, 0x4a, 0x01], "vandpd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x54, 0xca], "vandpd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x54, 0xca], "vandpd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x54, 0x0a], "vandpd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x54, 0x0a], "vandpd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x54, 0x4a, 0x01], "vandpd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x54, 0x4a, 0x01], "vandpd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x55, 0x0a], "vandnpd ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x55, 0x4a, 0x01], "vandnpd ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x55, 0x0a], "vandnpd ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x55, 0x0a], "vandnpd ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x55, 0x4a, 0x01], "vandnpd ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x55, 0x4a, 0x01], "vandnpd ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x55, 0xca], "vandnpd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x55, 0x0a], "vandnpd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x55, 0x4a, 0x01], "vandnpd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x55, 0xca], "vandnpd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x55, 0xca], "vandnpd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x55, 0x0a], "vandnpd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x55, 0x0a], "vandnpd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x55, 0x4a, 0x01], "vandnpd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x55, 0x4a, 0x01], "vandnpd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x55, 0x0a], "vandnpd zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x55, 0x4a, 0x01], "vandnpd zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x55, 0x0a], "vandnpd zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x55, 0x0a], "vandnpd zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x55, 0x4a, 0x01], "vandnpd zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x55, 0x4a, 0x01], "vandnpd zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x55, 0x0a], "vandnpd xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x55, 0x4a, 0x01], "vandnpd xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x55, 0x0a], "vandnpd xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x55, 0x0a], "vandnpd xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x55, 0x4a, 0x01], "vandnpd xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x55, 0x4a, 0x01], "vandnpd xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x55, 0xca], "vandnpd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x55, 0x0a], "vandnpd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x55, 0x4a, 0x01], "vandnpd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x55, 0xca], "vandnpd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x55, 0xca], "vandnpd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x55, 0x0a], "vandnpd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x55, 0x0a], "vandnpd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x55, 0x4a, 0x01], "vandnpd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x55, 0x4a, 0x01], "vandnpd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x55, 0xca], "vandnpd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x55, 0x0a], "vandnpd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x55, 0x4a, 0x01], "vandnpd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x55, 0xca], "vandnpd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x55, 0xca], "vandnpd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x55, 0x0a], "vandnpd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x55, 0x0a], "vandnpd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x55, 0x4a, 0x01], "vandnpd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x55, 0x4a, 0x01], "vandnpd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x56, 0x0a], "vorpd ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x56, 0x4a, 0x01], "vorpd ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x56, 0x0a], "vorpd ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x56, 0x0a], "vorpd ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x56, 0x4a, 0x01], "vorpd ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x56, 0x4a, 0x01], "vorpd ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x56, 0xca], "vorpd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x56, 0x0a], "vorpd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x56, 0x4a, 0x01], "vorpd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x56, 0xca], "vorpd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x56, 0xca], "vorpd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x56, 0x0a], "vorpd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x56, 0x0a], "vorpd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x56, 0x4a, 0x01], "vorpd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x56, 0x4a, 0x01], "vorpd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x56, 0x0a], "vorpd zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x56, 0x4a, 0x01], "vorpd zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x56, 0x0a], "vorpd zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x56, 0x0a], "vorpd zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x56, 0x4a, 0x01], "vorpd zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x56, 0x4a, 0x01], "vorpd zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x56, 0x0a], "vorpd xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x56, 0x4a, 0x01], "vorpd xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x56, 0x0a], "vorpd xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x56, 0x0a], "vorpd xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x56, 0x4a, 0x01], "vorpd xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x56, 0x4a, 0x01], "vorpd xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x56, 0xca], "vorpd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x56, 0x0a], "vorpd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x56, 0x4a, 0x01], "vorpd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x56, 0xca], "vorpd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x56, 0xca], "vorpd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x56, 0x0a], "vorpd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x56, 0x0a], "vorpd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x56, 0x4a, 0x01], "vorpd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x56, 0x4a, 0x01], "vorpd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x56, 0xca], "vorpd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x56, 0x0a], "vorpd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x56, 0x4a, 0x01], "vorpd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x56, 0xca], "vorpd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x56, 0xca], "vorpd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x56, 0x0a], "vorpd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x56, 0x0a], "vorpd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x56, 0x4a, 0x01], "vorpd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x56, 0x4a, 0x01], "vorpd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x57, 0x0a], "vxorpd ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x57, 0x4a, 0x01], "vxorpd ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x57, 0x0a], "vxorpd ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x57, 0x0a], "vxorpd ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x57, 0x4a, 0x01], "vxorpd ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x57, 0x4a, 0x01], "vxorpd ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x57, 0xca], "vxorpd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x57, 0x0a], "vxorpd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x57, 0x4a, 0x01], "vxorpd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x57, 0xca], "vxorpd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x57, 0xca], "vxorpd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x57, 0x0a], "vxorpd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x57, 0x0a], "vxorpd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x57, 0x4a, 0x01], "vxorpd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x57, 0x4a, 0x01], "vxorpd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x57, 0x0a], "vxorpd zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x57, 0x4a, 0x01], "vxorpd zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x57, 0x0a], "vxorpd zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x57, 0x0a], "vxorpd zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x57, 0x4a, 0x01], "vxorpd zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x57, 0x4a, 0x01], "vxorpd zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x57, 0x0a], "vxorpd xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x57, 0x4a, 0x01], "vxorpd xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x57, 0x0a], "vxorpd xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x57, 0x0a], "vxorpd xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x57, 0x4a, 0x01], "vxorpd xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x57, 0x4a, 0x01], "vxorpd xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x57, 0xca], "vxorpd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x57, 0x0a], "vxorpd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x57, 0x4a, 0x01], "vxorpd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x57, 0xca], "vxorpd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x57, 0xca], "vxorpd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x57, 0x0a], "vxorpd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x57, 0x0a], "vxorpd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x57, 0x4a, 0x01], "vxorpd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x57, 0x4a, 0x01], "vxorpd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x57, 0xca], "vxorpd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x57, 0x0a], "vxorpd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x57, 0x4a, 0x01], "vxorpd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x57, 0xca], "vxorpd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x57, 0xca], "vxorpd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x57, 0x0a], "vxorpd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x57, 0x0a], "vxorpd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x57, 0x4a, 0x01], "vxorpd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x57, 0x4a, 0x01], "vxorpd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0x58, 0xca], "vaddpd zmm1{k5}{z}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x58, 0xca], "vaddpd zmm1{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0x58, 0xca], "vaddpd zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x58, 0xca], "vaddpd zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x58, 0x0a], "vaddpd ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x58, 0x4a, 0x01], "vaddpd ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x58, 0xca], "vaddpd zmm1{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x58, 0xca], "vaddpd zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x58, 0x0a], "vaddpd ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x58, 0x0a], "vaddpd ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x58, 0x4a, 0x01], "vaddpd ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x58, 0x4a, 0x01], "vaddpd ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x58, 0xca], "vaddpd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x58, 0x0a], "vaddpd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x58, 0x4a, 0x01], "vaddpd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x58, 0xca], "vaddpd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x58, 0xca], "vaddpd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x58, 0x0a], "vaddpd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x58, 0x0a], "vaddpd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x58, 0x4a, 0x01], "vaddpd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x58, 0x4a, 0x01], "vaddpd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x58, 0xca], "vaddpd zmm1{k5}{z}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x58, 0x0a], "vaddpd zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x58, 0x4a, 0x01], "vaddpd zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x58, 0xca], "vaddpd zmm1{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x58, 0xca], "vaddpd zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x58, 0x0a], "vaddpd zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x58, 0x0a], "vaddpd zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x58, 0x4a, 0x01], "vaddpd zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x58, 0x4a, 0x01], "vaddpd zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x58, 0xca], "vaddpd zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x58, 0x0a], "vaddpd xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x58, 0x4a, 0x01], "vaddpd xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x58, 0xca], "vaddpd zmm1{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x58, 0xca], "vaddpd zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x58, 0x0a], "vaddpd xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x58, 0x0a], "vaddpd xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x58, 0x4a, 0x01], "vaddpd xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x58, 0x4a, 0x01], "vaddpd xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x58, 0xca], "vaddpd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x58, 0x0a], "vaddpd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x58, 0x4a, 0x01], "vaddpd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x58, 0xca], "vaddpd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x58, 0xca], "vaddpd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x58, 0x0a], "vaddpd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x58, 0x0a], "vaddpd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x58, 0x4a, 0x01], "vaddpd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x58, 0x4a, 0x01], "vaddpd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x58, 0xca], "vaddpd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x58, 0x0a], "vaddpd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x58, 0x4a, 0x01], "vaddpd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x58, 0xca], "vaddpd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x58, 0xca], "vaddpd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x58, 0x0a], "vaddpd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x58, 0x0a], "vaddpd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x58, 0x4a, 0x01], "vaddpd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x58, 0x4a, 0x01], "vaddpd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0x59, 0xca], "vmulpd zmm1{k5}{z}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x59, 0xca], "vmulpd zmm1{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0x59, 0xca], "vmulpd zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x59, 0xca], "vmulpd zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x59, 0x0a], "vmulpd ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x59, 0x4a, 0x01], "vmulpd ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x59, 0xca], "vmulpd zmm1{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x59, 0xca], "vmulpd zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x59, 0x0a], "vmulpd ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x59, 0x0a], "vmulpd ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x59, 0x4a, 0x01], "vmulpd ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x59, 0x4a, 0x01], "vmulpd ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x59, 0xca], "vmulpd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x59, 0x0a], "vmulpd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x59, 0x4a, 0x01], "vmulpd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x59, 0xca], "vmulpd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x59, 0xca], "vmulpd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x59, 0x0a], "vmulpd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x59, 0x0a], "vmulpd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x59, 0x4a, 0x01], "vmulpd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x59, 0x4a, 0x01], "vmulpd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x59, 0xca], "vmulpd zmm1{k5}{z}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x59, 0x0a], "vmulpd zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x59, 0x4a, 0x01], "vmulpd zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x59, 0xca], "vmulpd zmm1{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x59, 0xca], "vmulpd zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x59, 0x0a], "vmulpd zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x59, 0x0a], "vmulpd zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x59, 0x4a, 0x01], "vmulpd zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x59, 0x4a, 0x01], "vmulpd zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x59, 0xca], "vmulpd zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x59, 0x0a], "vmulpd xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x59, 0x4a, 0x01], "vmulpd xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x59, 0xca], "vmulpd zmm1{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x59, 0xca], "vmulpd zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x59, 0x0a], "vmulpd xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x59, 0x0a], "vmulpd xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x59, 0x4a, 0x01], "vmulpd xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x59, 0x4a, 0x01], "vmulpd xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x59, 0xca], "vmulpd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x59, 0x0a], "vmulpd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x59, 0x4a, 0x01], "vmulpd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x59, 0xca], "vmulpd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x59, 0xca], "vmulpd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x59, 0x0a], "vmulpd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x59, 0x0a], "vmulpd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x59, 0x4a, 0x01], "vmulpd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x59, 0x4a, 0x01], "vmulpd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x59, 0xca], "vmulpd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x59, 0x0a], "vmulpd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x59, 0x4a, 0x01], "vmulpd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x59, 0xca], "vmulpd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x59, 0xca], "vmulpd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x59, 0x0a], "vmulpd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x59, 0x0a], "vmulpd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x59, 0x4a, 0x01], "vmulpd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x59, 0x4a, 0x01], "vmulpd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{z}{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x5a, 0xca], "vcvtpd2ps ymm1{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{z}{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x5a, 0x0a], "vcvtpd2ps xmm1{k5}{z}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x5a, 0x4a, 0x01], "vcvtpd2ps xmm1{k5}{z}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x5a, 0xca], "vcvtpd2ps ymm1{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x5a, 0x0a], "vcvtpd2ps xmm1, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x5a, 0x0a], "vcvtpd2ps xmm1{k5}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x5a, 0x4a, 0x01], "vcvtpd2ps xmm1, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x5a, 0x4a, 0x01], "vcvtpd2ps xmm1{k5}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5a, 0xca], "vcvtpd2ps xmm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5a, 0x0a], "vcvtpd2ps xmm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5a, 0x4a, 0x01], "vcvtpd2ps xmm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5a, 0xca], "vcvtpd2ps xmm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5a, 0xca], "vcvtpd2ps xmm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5a, 0x0a], "vcvtpd2ps xmm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5a, 0x0a], "vcvtpd2ps xmm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5a, 0x4a, 0x01], "vcvtpd2ps xmm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5a, 0x4a, 0x01], "vcvtpd2ps xmm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{z}{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x5a, 0x0a], "vcvtpd2ps ymm1{k5}{z}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x5a, 0x4a, 0x01], "vcvtpd2ps ymm1{k5}{z}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x5a, 0xca], "vcvtpd2ps ymm1{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x5a, 0x0a], "vcvtpd2ps ymm1, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x5a, 0x0a], "vcvtpd2ps ymm1{k5}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x5a, 0x4a, 0x01], "vcvtpd2ps ymm1, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x5a, 0x4a, 0x01], "vcvtpd2ps ymm1{k5}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{z}{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x5a, 0x0a], "vcvtpd2ps xmm1{k5}{z}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x5a, 0x4a, 0x01], "vcvtpd2ps xmm1{k5}{z}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x5a, 0xca], "vcvtpd2ps ymm1{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x5a, 0x0a], "vcvtpd2ps xmm1, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x5a, 0x0a], "vcvtpd2ps xmm1{k5}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x5a, 0x4a, 0x01], "vcvtpd2ps xmm1, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x5a, 0x4a, 0x01], "vcvtpd2ps xmm1{k5}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5a, 0x0a], "vcvtpd2ps ymm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5a, 0x4a, 0x01], "vcvtpd2ps ymm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5a, 0xca], "vcvtpd2ps ymm1, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5a, 0x0a], "vcvtpd2ps ymm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5a, 0x0a], "vcvtpd2ps ymm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5a, 0x4a, 0x01], "vcvtpd2ps ymm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5a, 0x4a, 0x01], "vcvtpd2ps ymm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5a, 0xca], "vcvtpd2ps xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5a, 0x0a], "vcvtpd2ps xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5a, 0x4a, 0x01], "vcvtpd2ps xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5a, 0xca], "vcvtpd2ps xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5a, 0xca], "vcvtpd2ps xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5a, 0x0a], "vcvtpd2ps xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5a, 0x0a], "vcvtpd2ps xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5a, 0x4a, 0x01], "vcvtpd2ps xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5a, 0x4a, 0x01], "vcvtpd2ps xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xfd, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{z}{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x78, 0x5b, 0xca], "vcvtps2dq zmm1{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x7d, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{z}{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x5b, 0x0a], "vcvtps2dq ymm1{k5}{z}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x5b, 0x4a, 0x01], "vcvtps2dq ymm1{k5}{z}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x5b, 0xca], "vcvtps2dq zmm1{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x5b, 0x0a], "vcvtps2dq ymm1, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x5b, 0x0a], "vcvtps2dq ymm1{k5}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x5b, 0x4a, 0x01], "vcvtps2dq ymm1, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x5b, 0x4a, 0x01], "vcvtps2dq ymm1{k5}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x5b, 0xca], "vcvtps2dq ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x5b, 0x0a], "vcvtps2dq ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x5b, 0x4a, 0x01], "vcvtps2dq ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x5b, 0xca], "vcvtps2dq ymm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x5b, 0xca], "vcvtps2dq ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x5b, 0x0a], "vcvtps2dq ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x5b, 0x0a], "vcvtps2dq ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x5b, 0x4a, 0x01], "vcvtps2dq ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x5b, 0x4a, 0x01], "vcvtps2dq ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{z}{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x5b, 0x0a], "vcvtps2dq zmm1{k5}{z}, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x5b, 0x4a, 0x01], "vcvtps2dq zmm1{k5}{z}, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x5b, 0xca], "vcvtps2dq zmm1{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x5b, 0x0a], "vcvtps2dq zmm1, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x5b, 0x0a], "vcvtps2dq zmm1{k5}, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x5b, 0x4a, 0x01], "vcvtps2dq zmm1, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x5b, 0x4a, 0x01], "vcvtps2dq zmm1{k5}, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{z}{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x5b, 0x0a], "vcvtps2dq xmm1{k5}{z}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x5b, 0x4a, 0x01], "vcvtps2dq xmm1{k5}{z}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x5b, 0xca], "vcvtps2dq zmm1{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x5b, 0x0a], "vcvtps2dq xmm1, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x5b, 0x0a], "vcvtps2dq xmm1{k5}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x5b, 0x4a, 0x01], "vcvtps2dq xmm1, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x5b, 0x4a, 0x01], "vcvtps2dq xmm1{k5}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x5b, 0x0a], "vcvtps2dq zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x5b, 0x4a, 0x01], "vcvtps2dq zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x5b, 0xca], "vcvtps2dq zmm1, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x5b, 0xca], "vcvtps2dq zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x5b, 0x0a], "vcvtps2dq zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x5b, 0x0a], "vcvtps2dq zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x5b, 0x4a, 0x01], "vcvtps2dq zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x5b, 0x4a, 0x01], "vcvtps2dq zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x5b, 0xca], "vcvtps2dq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x5b, 0x0a], "vcvtps2dq xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x5b, 0x4a, 0x01], "vcvtps2dq xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x5b, 0xca], "vcvtps2dq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x5b, 0xca], "vcvtps2dq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x5b, 0x0a], "vcvtps2dq xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x5b, 0x0a], "vcvtps2dq xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x5b, 0x4a, 0x01], "vcvtps2dq xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x5b, 0x4a, 0x01], "vcvtps2dq xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0x5c, 0xca], "vsubpd zmm1{k5}{z}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x5c, 0xca], "vsubpd zmm1{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0x5c, 0xca], "vsubpd zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x5c, 0xca], "vsubpd zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x5c, 0x0a], "vsubpd ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x5c, 0x4a, 0x01], "vsubpd ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x5c, 0xca], "vsubpd zmm1{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x5c, 0xca], "vsubpd zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x5c, 0x0a], "vsubpd ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x5c, 0x0a], "vsubpd ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x5c, 0x4a, 0x01], "vsubpd ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x5c, 0x4a, 0x01], "vsubpd ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5c, 0xca], "vsubpd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5c, 0x0a], "vsubpd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5c, 0x4a, 0x01], "vsubpd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5c, 0xca], "vsubpd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5c, 0xca], "vsubpd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5c, 0x0a], "vsubpd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5c, 0x0a], "vsubpd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5c, 0x4a, 0x01], "vsubpd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5c, 0x4a, 0x01], "vsubpd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x5c, 0xca], "vsubpd zmm1{k5}{z}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x5c, 0x0a], "vsubpd zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x5c, 0x4a, 0x01], "vsubpd zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x5c, 0xca], "vsubpd zmm1{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x5c, 0xca], "vsubpd zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x5c, 0x0a], "vsubpd zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x5c, 0x0a], "vsubpd zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x5c, 0x4a, 0x01], "vsubpd zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x5c, 0x4a, 0x01], "vsubpd zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x5c, 0xca], "vsubpd zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x5c, 0x0a], "vsubpd xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x5c, 0x4a, 0x01], "vsubpd xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x5c, 0xca], "vsubpd zmm1{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x5c, 0xca], "vsubpd zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x5c, 0x0a], "vsubpd xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x5c, 0x0a], "vsubpd xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x5c, 0x4a, 0x01], "vsubpd xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x5c, 0x4a, 0x01], "vsubpd xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5c, 0xca], "vsubpd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5c, 0x0a], "vsubpd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5c, 0x4a, 0x01], "vsubpd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5c, 0xca], "vsubpd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5c, 0xca], "vsubpd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5c, 0x0a], "vsubpd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5c, 0x0a], "vsubpd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5c, 0x4a, 0x01], "vsubpd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5c, 0x4a, 0x01], "vsubpd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5c, 0xca], "vsubpd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5c, 0x0a], "vsubpd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5c, 0x4a, 0x01], "vsubpd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5c, 0xca], "vsubpd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5c, 0xca], "vsubpd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5c, 0x0a], "vsubpd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5c, 0x0a], "vsubpd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5c, 0x4a, 0x01], "vsubpd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5c, 0x4a, 0x01], "vsubpd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0x5d, 0xca], "vminpd zmm1{k5}{z}{sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x5d, 0xca], "vminpd zmm1{sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0x5d, 0xca], "vminpd zmm1{k5}{sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x5d, 0x0a], "vminpd ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x5d, 0x4a, 0x01], "vminpd ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x5d, 0x0a], "vminpd ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x5d, 0x0a], "vminpd ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x5d, 0x4a, 0x01], "vminpd ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x5d, 0x4a, 0x01], "vminpd ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5d, 0xca], "vminpd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5d, 0x0a], "vminpd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5d, 0x4a, 0x01], "vminpd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5d, 0xca], "vminpd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5d, 0xca], "vminpd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5d, 0x0a], "vminpd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5d, 0x0a], "vminpd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5d, 0x4a, 0x01], "vminpd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5d, 0x4a, 0x01], "vminpd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x5d, 0x0a], "vminpd zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x5d, 0x4a, 0x01], "vminpd zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x5d, 0x0a], "vminpd zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x5d, 0x0a], "vminpd zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x5d, 0x4a, 0x01], "vminpd zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x5d, 0x4a, 0x01], "vminpd zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x5d, 0x0a], "vminpd xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x5d, 0x4a, 0x01], "vminpd xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x5d, 0x0a], "vminpd xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x5d, 0x0a], "vminpd xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x5d, 0x4a, 0x01], "vminpd xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x5d, 0x4a, 0x01], "vminpd xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5d, 0xca], "vminpd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5d, 0x0a], "vminpd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5d, 0x4a, 0x01], "vminpd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5d, 0xca], "vminpd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5d, 0xca], "vminpd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5d, 0x0a], "vminpd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5d, 0x0a], "vminpd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5d, 0x4a, 0x01], "vminpd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5d, 0x4a, 0x01], "vminpd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5d, 0xca], "vminpd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5d, 0x0a], "vminpd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5d, 0x4a, 0x01], "vminpd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5d, 0xca], "vminpd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5d, 0xca], "vminpd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5d, 0x0a], "vminpd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5d, 0x0a], "vminpd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5d, 0x4a, 0x01], "vminpd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5d, 0x4a, 0x01], "vminpd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0x5e, 0xca], "vdivpd zmm1{k5}{z}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x5e, 0xca], "vdivpd zmm1{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0x5e, 0xca], "vdivpd zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x5e, 0xca], "vdivpd zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x5e, 0x0a], "vdivpd ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x5e, 0x4a, 0x01], "vdivpd ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x5e, 0xca], "vdivpd zmm1{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x5e, 0xca], "vdivpd zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x5e, 0x0a], "vdivpd ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x5e, 0x0a], "vdivpd ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x5e, 0x4a, 0x01], "vdivpd ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x5e, 0x4a, 0x01], "vdivpd ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5e, 0xca], "vdivpd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5e, 0x0a], "vdivpd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5e, 0x4a, 0x01], "vdivpd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5e, 0xca], "vdivpd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5e, 0xca], "vdivpd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5e, 0x0a], "vdivpd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5e, 0x0a], "vdivpd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5e, 0x4a, 0x01], "vdivpd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5e, 0x4a, 0x01], "vdivpd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x5e, 0xca], "vdivpd zmm1{k5}{z}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x5e, 0x0a], "vdivpd zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x5e, 0x4a, 0x01], "vdivpd zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x5e, 0xca], "vdivpd zmm1{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x5e, 0xca], "vdivpd zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x5e, 0x0a], "vdivpd zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x5e, 0x0a], "vdivpd zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x5e, 0x4a, 0x01], "vdivpd zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x5e, 0x4a, 0x01], "vdivpd zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x5e, 0xca], "vdivpd zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x5e, 0x0a], "vdivpd xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x5e, 0x4a, 0x01], "vdivpd xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x5e, 0xca], "vdivpd zmm1{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x5e, 0xca], "vdivpd zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x5e, 0x0a], "vdivpd xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x5e, 0x0a], "vdivpd xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x5e, 0x4a, 0x01], "vdivpd xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x5e, 0x4a, 0x01], "vdivpd xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5e, 0xca], "vdivpd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5e, 0x0a], "vdivpd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5e, 0x4a, 0x01], "vdivpd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5e, 0xca], "vdivpd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5e, 0xca], "vdivpd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5e, 0x0a], "vdivpd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5e, 0x0a], "vdivpd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5e, 0x4a, 0x01], "vdivpd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5e, 0x4a, 0x01], "vdivpd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5e, 0xca], "vdivpd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5e, 0x0a], "vdivpd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5e, 0x4a, 0x01], "vdivpd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5e, 0xca], "vdivpd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5e, 0xca], "vdivpd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5e, 0x0a], "vdivpd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5e, 0x0a], "vdivpd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5e, 0x4a, 0x01], "vdivpd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5e, 0x4a, 0x01], "vdivpd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0x5f, 0xca], "vmaxpd zmm1{k5}{z}{sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x5f, 0xca], "vmaxpd zmm1{sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0x5f, 0xca], "vmaxpd zmm1{k5}{sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x5f, 0x0a], "vmaxpd ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x5f, 0x4a, 0x01], "vmaxpd ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x5f, 0x0a], "vmaxpd ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x5f, 0x0a], "vmaxpd ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x5f, 0x4a, 0x01], "vmaxpd ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x5f, 0x4a, 0x01], "vmaxpd ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5f, 0xca], "vmaxpd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5f, 0x0a], "vmaxpd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5f, 0x4a, 0x01], "vmaxpd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5f, 0xca], "vmaxpd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5f, 0xca], "vmaxpd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5f, 0x0a], "vmaxpd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5f, 0x0a], "vmaxpd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5f, 0x4a, 0x01], "vmaxpd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5f, 0x4a, 0x01], "vmaxpd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x5f, 0x0a], "vmaxpd zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x5f, 0x4a, 0x01], "vmaxpd zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x5f, 0x0a], "vmaxpd zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x5f, 0x0a], "vmaxpd zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x5f, 0x4a, 0x01], "vmaxpd zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x5f, 0x4a, 0x01], "vmaxpd zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x5f, 0x0a], "vmaxpd xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x5f, 0x4a, 0x01], "vmaxpd xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x5f, 0x0a], "vmaxpd xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x5f, 0x0a], "vmaxpd xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x5f, 0x4a, 0x01], "vmaxpd xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x5f, 0x4a, 0x01], "vmaxpd xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5f, 0xca], "vmaxpd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5f, 0x0a], "vmaxpd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5f, 0x4a, 0x01], "vmaxpd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5f, 0xca], "vmaxpd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5f, 0xca], "vmaxpd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5f, 0x0a], "vmaxpd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5f, 0x0a], "vmaxpd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5f, 0x4a, 0x01], "vmaxpd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5f, 0x4a, 0x01], "vmaxpd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5f, 0xca], "vmaxpd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5f, 0x0a], "vmaxpd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5f, 0x4a, 0x01], "vmaxpd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5f, 0xca], "vmaxpd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5f, 0xca], "vmaxpd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5f, 0x0a], "vmaxpd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5f, 0x0a], "vmaxpd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5f, 0x4a, 0x01], "vmaxpd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5f, 0x4a, 0x01], "vmaxpd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x60, 0xca], "vpunpcklbw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x60, 0x0a], "vpunpcklbw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x60, 0x4a, 0x01], "vpunpcklbw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x60, 0xca], "vpunpcklbw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x60, 0xca], "vpunpcklbw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x60, 0x0a], "vpunpcklbw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x60, 0x0a], "vpunpcklbw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x60, 0x4a, 0x01], "vpunpcklbw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x60, 0x4a, 0x01], "vpunpcklbw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x60, 0xca], "vpunpcklbw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x60, 0x0a], "vpunpcklbw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x60, 0x4a, 0x01], "vpunpcklbw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x60, 0xca], "vpunpcklbw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x60, 0xca], "vpunpcklbw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x60, 0x0a], "vpunpcklbw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x60, 0x0a], "vpunpcklbw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x60, 0x4a, 0x01], "vpunpcklbw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x60, 0x4a, 0x01], "vpunpcklbw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x60, 0xca], "vpunpcklbw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x60, 0x0a], "vpunpcklbw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x60, 0x4a, 0x01], "vpunpcklbw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x60, 0xca], "vpunpcklbw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x60, 0xca], "vpunpcklbw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x60, 0x0a], "vpunpcklbw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x60, 0x0a], "vpunpcklbw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x60, 0x4a, 0x01], "vpunpcklbw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x60, 0x4a, 0x01], "vpunpcklbw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x61, 0xca], "vpunpcklwd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x61, 0x0a], "vpunpcklwd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x61, 0x4a, 0x01], "vpunpcklwd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x61, 0xca], "vpunpcklwd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x61, 0xca], "vpunpcklwd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x61, 0x0a], "vpunpcklwd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x61, 0x0a], "vpunpcklwd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x61, 0x4a, 0x01], "vpunpcklwd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x61, 0x4a, 0x01], "vpunpcklwd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x61, 0xca], "vpunpcklwd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x61, 0x0a], "vpunpcklwd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x61, 0x4a, 0x01], "vpunpcklwd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x61, 0xca], "vpunpcklwd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x61, 0xca], "vpunpcklwd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x61, 0x0a], "vpunpcklwd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x61, 0x0a], "vpunpcklwd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x61, 0x4a, 0x01], "vpunpcklwd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x61, 0x4a, 0x01], "vpunpcklwd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x61, 0xca], "vpunpcklwd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x61, 0x0a], "vpunpcklwd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x61, 0x4a, 0x01], "vpunpcklwd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x61, 0xca], "vpunpcklwd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x61, 0xca], "vpunpcklwd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x61, 0x0a], "vpunpcklwd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x61, 0x0a], "vpunpcklwd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x61, 0x4a, 0x01], "vpunpcklwd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x61, 0x4a, 0x01], "vpunpcklwd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x62, 0x0a], "vpunpckldq ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x62, 0x4a, 0x01], "vpunpckldq ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x62, 0x0a], "vpunpckldq ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x62, 0x0a], "vpunpckldq ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x62, 0x4a, 0x01], "vpunpckldq ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x62, 0x4a, 0x01], "vpunpckldq ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x62, 0xca], "vpunpckldq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x62, 0x0a], "vpunpckldq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x62, 0x4a, 0x01], "vpunpckldq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x62, 0xca], "vpunpckldq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x62, 0xca], "vpunpckldq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x62, 0x0a], "vpunpckldq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x62, 0x0a], "vpunpckldq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x62, 0x4a, 0x01], "vpunpckldq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x62, 0x4a, 0x01], "vpunpckldq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x62, 0x0a], "vpunpckldq zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x62, 0x4a, 0x01], "vpunpckldq zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x62, 0x0a], "vpunpckldq zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x62, 0x0a], "vpunpckldq zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x62, 0x4a, 0x01], "vpunpckldq zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x62, 0x4a, 0x01], "vpunpckldq zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x62, 0x0a], "vpunpckldq xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x62, 0x4a, 0x01], "vpunpckldq xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x62, 0x0a], "vpunpckldq xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x62, 0x0a], "vpunpckldq xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x62, 0x4a, 0x01], "vpunpckldq xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x62, 0x4a, 0x01], "vpunpckldq xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x62, 0xca], "vpunpckldq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x62, 0x0a], "vpunpckldq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x62, 0x4a, 0x01], "vpunpckldq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x62, 0xca], "vpunpckldq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x62, 0xca], "vpunpckldq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x62, 0x0a], "vpunpckldq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x62, 0x0a], "vpunpckldq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x62, 0x4a, 0x01], "vpunpckldq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x62, 0x4a, 0x01], "vpunpckldq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x62, 0xca], "vpunpckldq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x62, 0x0a], "vpunpckldq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x62, 0x4a, 0x01], "vpunpckldq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x62, 0xca], "vpunpckldq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x62, 0xca], "vpunpckldq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x62, 0x0a], "vpunpckldq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x62, 0x0a], "vpunpckldq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x62, 0x4a, 0x01], "vpunpckldq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x62, 0x4a, 0x01], "vpunpckldq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x63, 0xca], "vpacksswb ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x63, 0x0a], "vpacksswb ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x63, 0x4a, 0x01], "vpacksswb ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x63, 0xca], "vpacksswb ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x63, 0xca], "vpacksswb ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x63, 0x0a], "vpacksswb ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x63, 0x0a], "vpacksswb ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x63, 0x4a, 0x01], "vpacksswb ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x63, 0x4a, 0x01], "vpacksswb ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x63, 0xca], "vpacksswb zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x63, 0x0a], "vpacksswb zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x63, 0x4a, 0x01], "vpacksswb zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x63, 0xca], "vpacksswb zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x63, 0xca], "vpacksswb zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x63, 0x0a], "vpacksswb zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x63, 0x0a], "vpacksswb zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x63, 0x4a, 0x01], "vpacksswb zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x63, 0x4a, 0x01], "vpacksswb zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x63, 0xca], "vpacksswb xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x63, 0x0a], "vpacksswb xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x63, 0x4a, 0x01], "vpacksswb xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x63, 0xca], "vpacksswb xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x63, 0xca], "vpacksswb xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x63, 0x0a], "vpacksswb xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x63, 0x0a], "vpacksswb xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x63, 0x4a, 0x01], "vpacksswb xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x63, 0x4a, 0x01], "vpacksswb xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x64, 0xca], "vpcmpgtb k1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x64, 0xca], "vpcmpgtb k1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x64, 0x0a], "vpcmpgtb k1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x64, 0x0a], "vpcmpgtb k1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x64, 0x4a, 0x01], "vpcmpgtb k1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x64, 0x4a, 0x01], "vpcmpgtb k1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x64, 0xca], "vpcmpgtb k1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x64, 0xca], "vpcmpgtb k1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x64, 0x0a], "vpcmpgtb k1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x64, 0x0a], "vpcmpgtb k1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x64, 0x4a, 0x01], "vpcmpgtb k1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x64, 0x4a, 0x01], "vpcmpgtb k1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x64, 0xca], "vpcmpgtb k1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x64, 0xca], "vpcmpgtb k1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x64, 0x0a], "vpcmpgtb k1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x64, 0x0a], "vpcmpgtb k1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x64, 0x4a, 0x01], "vpcmpgtb k1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x64, 0x4a, 0x01], "vpcmpgtb k1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x65, 0xca], "vpcmpgtw k1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x65, 0xca], "vpcmpgtw k1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x65, 0x0a], "vpcmpgtw k1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x65, 0x0a], "vpcmpgtw k1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x65, 0x4a, 0x01], "vpcmpgtw k1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x65, 0x4a, 0x01], "vpcmpgtw k1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x65, 0xca], "vpcmpgtw k1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x65, 0xca], "vpcmpgtw k1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x65, 0x0a], "vpcmpgtw k1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x65, 0x0a], "vpcmpgtw k1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x65, 0x4a, 0x01], "vpcmpgtw k1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x65, 0x4a, 0x01], "vpcmpgtw k1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x65, 0xca], "vpcmpgtw k1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x65, 0xca], "vpcmpgtw k1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x65, 0x0a], "vpcmpgtw k1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x65, 0x0a], "vpcmpgtw k1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x65, 0x4a, 0x01], "vpcmpgtw k1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x65, 0x4a, 0x01], "vpcmpgtw k1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x66, 0x0a], "vpcmpgtd k1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x66, 0x0a], "vpcmpgtd k1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x66, 0x4a, 0x01], "vpcmpgtd k1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x66, 0x4a, 0x01], "vpcmpgtd k1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x66, 0xca], "vpcmpgtd k1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x66, 0xca], "vpcmpgtd k1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x66, 0x0a], "vpcmpgtd k1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x66, 0x0a], "vpcmpgtd k1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x66, 0x4a, 0x01], "vpcmpgtd k1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x66, 0x4a, 0x01], "vpcmpgtd k1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x66, 0x0a], "vpcmpgtd k1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x66, 0x0a], "vpcmpgtd k1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x66, 0x4a, 0x01], "vpcmpgtd k1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x66, 0x4a, 0x01], "vpcmpgtd k1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x66, 0x0a], "vpcmpgtd k1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x66, 0x0a], "vpcmpgtd k1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x66, 0x4a, 0x01], "vpcmpgtd k1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x66, 0x4a, 0x01], "vpcmpgtd k1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x66, 0xca], "vpcmpgtd k1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x66, 0xca], "vpcmpgtd k1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x66, 0x0a], "vpcmpgtd k1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x66, 0x0a], "vpcmpgtd k1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x66, 0x4a, 0x01], "vpcmpgtd k1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x66, 0x4a, 0x01], "vpcmpgtd k1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x66, 0xca], "vpcmpgtd k1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x66, 0xca], "vpcmpgtd k1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x66, 0x0a], "vpcmpgtd k1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x66, 0x0a], "vpcmpgtd k1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x66, 0x4a, 0x01], "vpcmpgtd k1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x66, 0x4a, 0x01], "vpcmpgtd k1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x67, 0xca], "vpackuswb ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x67, 0x0a], "vpackuswb ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x67, 0x4a, 0x01], "vpackuswb ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x67, 0xca], "vpackuswb ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x67, 0xca], "vpackuswb ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x67, 0x0a], "vpackuswb ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x67, 0x0a], "vpackuswb ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x67, 0x4a, 0x01], "vpackuswb ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x67, 0x4a, 0x01], "vpackuswb ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x67, 0xca], "vpackuswb zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x67, 0x0a], "vpackuswb zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x67, 0x4a, 0x01], "vpackuswb zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x67, 0xca], "vpackuswb zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x67, 0xca], "vpackuswb zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x67, 0x0a], "vpackuswb zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x67, 0x0a], "vpackuswb zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x67, 0x4a, 0x01], "vpackuswb zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x67, 0x4a, 0x01], "vpackuswb zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x67, 0xca], "vpackuswb xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x67, 0x0a], "vpackuswb xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x67, 0x4a, 0x01], "vpackuswb xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x67, 0xca], "vpackuswb xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x67, 0xca], "vpackuswb xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x67, 0x0a], "vpackuswb xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x67, 0x0a], "vpackuswb xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x67, 0x4a, 0x01], "vpackuswb xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x67, 0x4a, 0x01], "vpackuswb xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x68, 0xca], "vpunpckhbw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x68, 0x0a], "vpunpckhbw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x68, 0x4a, 0x01], "vpunpckhbw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x68, 0xca], "vpunpckhbw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x68, 0xca], "vpunpckhbw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x68, 0x0a], "vpunpckhbw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x68, 0x0a], "vpunpckhbw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x68, 0x4a, 0x01], "vpunpckhbw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x68, 0x4a, 0x01], "vpunpckhbw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x68, 0xca], "vpunpckhbw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x68, 0x0a], "vpunpckhbw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x68, 0x4a, 0x01], "vpunpckhbw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x68, 0xca], "vpunpckhbw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x68, 0xca], "vpunpckhbw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x68, 0x0a], "vpunpckhbw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x68, 0x0a], "vpunpckhbw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x68, 0x4a, 0x01], "vpunpckhbw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x68, 0x4a, 0x01], "vpunpckhbw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x68, 0xca], "vpunpckhbw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x68, 0x0a], "vpunpckhbw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x68, 0x4a, 0x01], "vpunpckhbw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x68, 0xca], "vpunpckhbw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x68, 0xca], "vpunpckhbw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x68, 0x0a], "vpunpckhbw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x68, 0x0a], "vpunpckhbw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x68, 0x4a, 0x01], "vpunpckhbw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x68, 0x4a, 0x01], "vpunpckhbw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x69, 0xca], "vpunpckhwd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x69, 0x0a], "vpunpckhwd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x69, 0x4a, 0x01], "vpunpckhwd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x69, 0xca], "vpunpckhwd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x69, 0xca], "vpunpckhwd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x69, 0x0a], "vpunpckhwd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x69, 0x0a], "vpunpckhwd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x69, 0x4a, 0x01], "vpunpckhwd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x69, 0x4a, 0x01], "vpunpckhwd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x69, 0xca], "vpunpckhwd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x69, 0x0a], "vpunpckhwd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x69, 0x4a, 0x01], "vpunpckhwd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x69, 0xca], "vpunpckhwd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x69, 0xca], "vpunpckhwd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x69, 0x0a], "vpunpckhwd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x69, 0x0a], "vpunpckhwd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x69, 0x4a, 0x01], "vpunpckhwd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x69, 0x4a, 0x01], "vpunpckhwd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x69, 0xca], "vpunpckhwd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x69, 0x0a], "vpunpckhwd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x69, 0x4a, 0x01], "vpunpckhwd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x69, 0xca], "vpunpckhwd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x69, 0xca], "vpunpckhwd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x69, 0x0a], "vpunpckhwd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x69, 0x0a], "vpunpckhwd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x69, 0x4a, 0x01], "vpunpckhwd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x69, 0x4a, 0x01], "vpunpckhwd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x6a, 0x0a], "vpunpckhdq ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x6a, 0x4a, 0x01], "vpunpckhdq ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x6a, 0x0a], "vpunpckhdq ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x6a, 0x0a], "vpunpckhdq ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x6a, 0x4a, 0x01], "vpunpckhdq ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x6a, 0x4a, 0x01], "vpunpckhdq ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x6a, 0xca], "vpunpckhdq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x6a, 0x0a], "vpunpckhdq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x6a, 0x4a, 0x01], "vpunpckhdq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x6a, 0xca], "vpunpckhdq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x6a, 0xca], "vpunpckhdq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x6a, 0x0a], "vpunpckhdq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x6a, 0x0a], "vpunpckhdq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x6a, 0x4a, 0x01], "vpunpckhdq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x6a, 0x4a, 0x01], "vpunpckhdq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x6a, 0x0a], "vpunpckhdq zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x6a, 0x4a, 0x01], "vpunpckhdq zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x6a, 0x0a], "vpunpckhdq zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x6a, 0x0a], "vpunpckhdq zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x6a, 0x4a, 0x01], "vpunpckhdq zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x6a, 0x4a, 0x01], "vpunpckhdq zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x6a, 0x0a], "vpunpckhdq xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x6a, 0x4a, 0x01], "vpunpckhdq xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x6a, 0x0a], "vpunpckhdq xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x6a, 0x0a], "vpunpckhdq xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x6a, 0x4a, 0x01], "vpunpckhdq xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x6a, 0x4a, 0x01], "vpunpckhdq xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x6a, 0xca], "vpunpckhdq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x6a, 0x0a], "vpunpckhdq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x6a, 0x4a, 0x01], "vpunpckhdq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x6a, 0xca], "vpunpckhdq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x6a, 0xca], "vpunpckhdq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x6a, 0x0a], "vpunpckhdq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x6a, 0x0a], "vpunpckhdq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x6a, 0x4a, 0x01], "vpunpckhdq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x6a, 0x4a, 0x01], "vpunpckhdq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x6a, 0xca], "vpunpckhdq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x6a, 0x0a], "vpunpckhdq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x6a, 0x4a, 0x01], "vpunpckhdq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x6a, 0xca], "vpunpckhdq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x6a, 0xca], "vpunpckhdq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x6a, 0x0a], "vpunpckhdq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x6a, 0x0a], "vpunpckhdq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x6a, 0x4a, 0x01], "vpunpckhdq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x6a, 0x4a, 0x01], "vpunpckhdq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x6b, 0x0a], "vpackssdw ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x6b, 0x4a, 0x01], "vpackssdw ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x6b, 0x0a], "vpackssdw ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x6b, 0x0a], "vpackssdw ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x6b, 0x4a, 0x01], "vpackssdw ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x6b, 0x4a, 0x01], "vpackssdw ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x6b, 0xca], "vpackssdw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x6b, 0x0a], "vpackssdw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x6b, 0x4a, 0x01], "vpackssdw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x6b, 0xca], "vpackssdw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x6b, 0xca], "vpackssdw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x6b, 0x0a], "vpackssdw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x6b, 0x0a], "vpackssdw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x6b, 0x4a, 0x01], "vpackssdw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x6b, 0x4a, 0x01], "vpackssdw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x6b, 0x0a], "vpackssdw zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x6b, 0x4a, 0x01], "vpackssdw zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x6b, 0x0a], "vpackssdw zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x6b, 0x0a], "vpackssdw zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x6b, 0x4a, 0x01], "vpackssdw zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x6b, 0x4a, 0x01], "vpackssdw zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x6b, 0x0a], "vpackssdw xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x6b, 0x4a, 0x01], "vpackssdw xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x6b, 0x0a], "vpackssdw xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x6b, 0x0a], "vpackssdw xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x6b, 0x4a, 0x01], "vpackssdw xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x6b, 0x4a, 0x01], "vpackssdw xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x6b, 0xca], "vpackssdw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x6b, 0x0a], "vpackssdw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x6b, 0x4a, 0x01], "vpackssdw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x6b, 0xca], "vpackssdw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x6b, 0xca], "vpackssdw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x6b, 0x0a], "vpackssdw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x6b, 0x0a], "vpackssdw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x6b, 0x4a, 0x01], "vpackssdw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x6b, 0x4a, 0x01], "vpackssdw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x6b, 0xca], "vpackssdw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x6b, 0x0a], "vpackssdw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x6b, 0x4a, 0x01], "vpackssdw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x6b, 0xca], "vpackssdw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x6b, 0xca], "vpackssdw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x6b, 0x0a], "vpackssdw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x6b, 0x0a], "vpackssdw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x6b, 0x4a, 0x01], "vpackssdw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x6b, 0x4a, 0x01], "vpackssdw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x6c, 0x0a], "vpunpcklqdq ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x6c, 0x4a, 0x01], "vpunpcklqdq ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x6c, 0x0a], "vpunpcklqdq ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x6c, 0x0a], "vpunpcklqdq ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x6c, 0x4a, 0x01], "vpunpcklqdq ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x6c, 0x4a, 0x01], "vpunpcklqdq ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x6c, 0xca], "vpunpcklqdq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x6c, 0x0a], "vpunpcklqdq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x6c, 0x4a, 0x01], "vpunpcklqdq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x6c, 0xca], "vpunpcklqdq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x6c, 0xca], "vpunpcklqdq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x6c, 0x0a], "vpunpcklqdq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x6c, 0x0a], "vpunpcklqdq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x6c, 0x4a, 0x01], "vpunpcklqdq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x6c, 0x4a, 0x01], "vpunpcklqdq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x6c, 0x0a], "vpunpcklqdq zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x6c, 0x4a, 0x01], "vpunpcklqdq zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x6c, 0x0a], "vpunpcklqdq zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x6c, 0x0a], "vpunpcklqdq zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x6c, 0x4a, 0x01], "vpunpcklqdq zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x6c, 0x4a, 0x01], "vpunpcklqdq zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x6c, 0x0a], "vpunpcklqdq xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x6c, 0x4a, 0x01], "vpunpcklqdq xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x6c, 0x0a], "vpunpcklqdq xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x6c, 0x0a], "vpunpcklqdq xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x6c, 0x4a, 0x01], "vpunpcklqdq xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x6c, 0x4a, 0x01], "vpunpcklqdq xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x6c, 0xca], "vpunpcklqdq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x6c, 0x0a], "vpunpcklqdq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x6c, 0x4a, 0x01], "vpunpcklqdq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x6c, 0xca], "vpunpcklqdq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x6c, 0xca], "vpunpcklqdq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x6c, 0x0a], "vpunpcklqdq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x6c, 0x0a], "vpunpcklqdq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x6c, 0x4a, 0x01], "vpunpcklqdq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x6c, 0x4a, 0x01], "vpunpcklqdq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x6c, 0xca], "vpunpcklqdq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x6c, 0x0a], "vpunpcklqdq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x6c, 0x4a, 0x01], "vpunpcklqdq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x6c, 0xca], "vpunpcklqdq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x6c, 0xca], "vpunpcklqdq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x6c, 0x0a], "vpunpcklqdq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x6c, 0x0a], "vpunpcklqdq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x6c, 0x4a, 0x01], "vpunpcklqdq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x6c, 0x4a, 0x01], "vpunpcklqdq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x6d, 0x0a], "vpunpckhqdq ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x6d, 0x4a, 0x01], "vpunpckhqdq ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x6d, 0x0a], "vpunpckhqdq ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x6d, 0x0a], "vpunpckhqdq ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x6d, 0x4a, 0x01], "vpunpckhqdq ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x6d, 0x4a, 0x01], "vpunpckhqdq ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x6d, 0xca], "vpunpckhqdq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x6d, 0x0a], "vpunpckhqdq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x6d, 0x4a, 0x01], "vpunpckhqdq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x6d, 0xca], "vpunpckhqdq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x6d, 0xca], "vpunpckhqdq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x6d, 0x0a], "vpunpckhqdq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x6d, 0x0a], "vpunpckhqdq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x6d, 0x4a, 0x01], "vpunpckhqdq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x6d, 0x4a, 0x01], "vpunpckhqdq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x6d, 0x0a], "vpunpckhqdq zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x6d, 0x4a, 0x01], "vpunpckhqdq zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x6d, 0x0a], "vpunpckhqdq zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x6d, 0x0a], "vpunpckhqdq zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x6d, 0x4a, 0x01], "vpunpckhqdq zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x6d, 0x4a, 0x01], "vpunpckhqdq zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x6d, 0x0a], "vpunpckhqdq xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x6d, 0x4a, 0x01], "vpunpckhqdq xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x6d, 0x0a], "vpunpckhqdq xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x6d, 0x0a], "vpunpckhqdq xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x6d, 0x4a, 0x01], "vpunpckhqdq xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x6d, 0x4a, 0x01], "vpunpckhqdq xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x6d, 0xca], "vpunpckhqdq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x6d, 0x0a], "vpunpckhqdq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x6d, 0x4a, 0x01], "vpunpckhqdq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x6d, 0xca], "vpunpckhqdq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x6d, 0xca], "vpunpckhqdq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x6d, 0x0a], "vpunpckhqdq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x6d, 0x0a], "vpunpckhqdq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x6d, 0x4a, 0x01], "vpunpckhqdq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x6d, 0x4a, 0x01], "vpunpckhqdq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x6d, 0xca], "vpunpckhqdq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x6d, 0x0a], "vpunpckhqdq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x6d, 0x4a, 0x01], "vpunpckhqdq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x6d, 0xca], "vpunpckhqdq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x6d, 0xca], "vpunpckhqdq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x6d, 0x0a], "vpunpckhqdq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x6d, 0x0a], "vpunpckhqdq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x6d, 0x4a, 0x01], "vpunpckhqdq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x6d, 0x4a, 0x01], "vpunpckhqdq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x6e, 0xca], "vmovd xmm1, edx");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x6e, 0x0a], "vmovd xmm1, dword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x6e, 0x4a, 0x01], "vmovd xmm1, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x6f, 0xca], "vmovdqa64 ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x6f, 0x0a], "vmovdqa64 ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x6f, 0x4a, 0x01], "vmovdqa64 ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x6f, 0xca], "vmovdqa64 ymm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x6f, 0xca], "vmovdqa64 ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x6f, 0x0a], "vmovdqa64 ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x6f, 0x0a], "vmovdqa64 ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x6f, 0x4a, 0x01], "vmovdqa64 ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x6f, 0x4a, 0x01], "vmovdqa64 ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x6f, 0xca], "vmovdqa32 ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x6f, 0x0a], "vmovdqa32 ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x6f, 0x4a, 0x01], "vmovdqa32 ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x6f, 0xca], "vmovdqa32 ymm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x6f, 0xca], "vmovdqa32 ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x6f, 0x0a], "vmovdqa32 ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x6f, 0x0a], "vmovdqa32 ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x6f, 0x4a, 0x01], "vmovdqa32 ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x6f, 0x4a, 0x01], "vmovdqa32 ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x6f, 0xca], "vmovdqa64 zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x6f, 0x0a], "vmovdqa64 zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x6f, 0x4a, 0x01], "vmovdqa64 zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x6f, 0xca], "vmovdqa64 zmm1, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x6f, 0xca], "vmovdqa64 zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x6f, 0x0a], "vmovdqa64 zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x6f, 0x0a], "vmovdqa64 zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x6f, 0x4a, 0x01], "vmovdqa64 zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x6f, 0x4a, 0x01], "vmovdqa64 zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x6f, 0xca], "vmovdqa64 xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x6f, 0x0a], "vmovdqa64 xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x6f, 0x4a, 0x01], "vmovdqa64 xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x6f, 0xca], "vmovdqa64 xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x6f, 0xca], "vmovdqa64 xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x6f, 0x0a], "vmovdqa64 xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x6f, 0x0a], "vmovdqa64 xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x6f, 0x4a, 0x01], "vmovdqa64 xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x6f, 0x4a, 0x01], "vmovdqa64 xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x6f, 0xca], "vmovdqa32 zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x6f, 0x0a], "vmovdqa32 zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x6f, 0x4a, 0x01], "vmovdqa32 zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x6f, 0xca], "vmovdqa32 zmm1, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x6f, 0xca], "vmovdqa32 zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x6f, 0x0a], "vmovdqa32 zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x6f, 0x0a], "vmovdqa32 zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x6f, 0x4a, 0x01], "vmovdqa32 zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x6f, 0x4a, 0x01], "vmovdqa32 zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x6f, 0xca], "vmovdqa32 xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x6f, 0x0a], "vmovdqa32 xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x6f, 0x4a, 0x01], "vmovdqa32 xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x6f, 0xca], "vmovdqa32 xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x6f, 0xca], "vmovdqa32 xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x6f, 0x0a], "vmovdqa32 xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x6f, 0x0a], "vmovdqa32 xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x6f, 0x4a, 0x01], "vmovdqa32 xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x6f, 0x4a, 0x01], "vmovdqa32 xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x70, 0x0a, 0xcc], "vpshufd ymm1{k5}{z}, dword [edx]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x70, 0x4a, 0x01, 0xcc], "vpshufd ymm1{k5}{z}, dword [edx + 0x4]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x70, 0x0a, 0xcc], "vpshufd ymm1, dword [edx]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x70, 0x0a, 0xcc], "vpshufd ymm1{k5}, dword [edx]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x70, 0x4a, 0x01, 0xcc], "vpshufd ymm1, dword [edx + 0x4]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x70, 0x4a, 0x01, 0xcc], "vpshufd ymm1{k5}, dword [edx + 0x4]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x70, 0xca, 0xcc], "vpshufd ymm1{k5}{z}, ymm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x70, 0x0a, 0xcc], "vpshufd ymm1{k5}{z}, ymmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x70, 0x4a, 0x01, 0xcc], "vpshufd ymm1{k5}{z}, ymmword [edx + 0x20], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x70, 0xca, 0xcc], "vpshufd ymm1, ymm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x70, 0xca, 0xcc], "vpshufd ymm1{k5}, ymm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x70, 0x0a, 0xcc], "vpshufd ymm1, ymmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x70, 0x0a, 0xcc], "vpshufd ymm1{k5}, ymmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x70, 0x4a, 0x01, 0xcc], "vpshufd ymm1, ymmword [edx + 0x20], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x70, 0x4a, 0x01, 0xcc], "vpshufd ymm1{k5}, ymmword [edx + 0x20], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x70, 0x0a, 0xcc], "vpshufd zmm1{k5}{z}, dword [edx]{1to16}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x70, 0x4a, 0x01, 0xcc], "vpshufd zmm1{k5}{z}, dword [edx + 0x4]{1to16}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x70, 0x0a, 0xcc], "vpshufd zmm1, dword [edx]{1to16}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x70, 0x0a, 0xcc], "vpshufd zmm1{k5}, dword [edx]{1to16}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x70, 0x4a, 0x01, 0xcc], "vpshufd zmm1, dword [edx + 0x4]{1to16}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x70, 0x4a, 0x01, 0xcc], "vpshufd zmm1{k5}, dword [edx + 0x4]{1to16}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x70, 0x0a, 0xcc], "vpshufd xmm1{k5}{z}, dword [edx]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x70, 0x4a, 0x01, 0xcc], "vpshufd xmm1{k5}{z}, dword [edx + 0x4]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x70, 0x0a, 0xcc], "vpshufd xmm1, dword [edx]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x70, 0x0a, 0xcc], "vpshufd xmm1{k5}, dword [edx]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x70, 0x4a, 0x01, 0xcc], "vpshufd xmm1, dword [edx + 0x4]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x70, 0x4a, 0x01, 0xcc], "vpshufd xmm1{k5}, dword [edx + 0x4]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x70, 0xca, 0xcc], "vpshufd zmm1{k5}{z}, zmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x70, 0x0a, 0xcc], "vpshufd zmm1{k5}{z}, zmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x70, 0x4a, 0x01, 0xcc], "vpshufd zmm1{k5}{z}, zmmword [edx + 0x40], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x70, 0xca, 0xcc], "vpshufd zmm1, zmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x70, 0xca, 0xcc], "vpshufd zmm1{k5}, zmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x70, 0x0a, 0xcc], "vpshufd zmm1, zmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x70, 0x0a, 0xcc], "vpshufd zmm1{k5}, zmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x70, 0x4a, 0x01, 0xcc], "vpshufd zmm1, zmmword [edx + 0x40], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x70, 0x4a, 0x01, 0xcc], "vpshufd zmm1{k5}, zmmword [edx + 0x40], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x70, 0xca, 0xcc], "vpshufd xmm1{k5}{z}, xmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x70, 0x0a, 0xcc], "vpshufd xmm1{k5}{z}, xmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x70, 0x4a, 0x01, 0xcc], "vpshufd xmm1{k5}{z}, xmmword [edx + 0x10], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x70, 0xca, 0xcc], "vpshufd xmm1, xmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x70, 0xca, 0xcc], "vpshufd xmm1{k5}, xmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x70, 0x0a, 0xcc], "vpshufd xmm1, xmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x70, 0x0a, 0xcc], "vpshufd xmm1{k5}, xmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x70, 0x4a, 0x01, 0xcc], "vpshufd xmm1, xmmword [edx + 0x10], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x70, 0x4a, 0x01, 0xcc], "vpshufd xmm1{k5}, xmmword [edx + 0x10], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x72, 0x0a, 0xcc], "vprolq ymm0{k5}{z}, qword [edx]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x72, 0x4a, 0x01, 0xcc], "vprolq ymm0{k5}{z}, qword [edx + 0x8]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x72, 0x0a, 0xcc], "vprolq ymm0, qword [edx]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x72, 0x0a, 0xcc], "vprolq ymm0{k5}, qword [edx]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x72, 0x4a, 0x01, 0xcc], "vprolq ymm0, qword [edx + 0x8]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x72, 0x4a, 0x01, 0xcc], "vprolq ymm0{k5}, qword [edx + 0x8]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x72, 0xca, 0xcc], "vprolq ymm0{k5}{z}, ymm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x72, 0x0a, 0xcc], "vprolq ymm0{k5}{z}, ymmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x72, 0x4a, 0x01, 0xcc], "vprolq ymm0{k5}{z}, ymmword [edx + 0x20], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x72, 0xca, 0xcc], "vprolq ymm0, ymm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x72, 0xca, 0xcc], "vprolq ymm0{k5}, ymm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x72, 0x0a, 0xcc], "vprolq ymm0, ymmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x72, 0x0a, 0xcc], "vprolq ymm0{k5}, ymmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x72, 0x4a, 0x01, 0xcc], "vprolq ymm0, ymmword [edx + 0x20], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x72, 0x4a, 0x01, 0xcc], "vprolq ymm0{k5}, ymmword [edx + 0x20], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x72, 0x0a, 0xcc], "vprold ymm0{k5}{z}, dword [edx]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x72, 0x4a, 0x01, 0xcc], "vprold ymm0{k5}{z}, dword [edx + 0x4]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x72, 0x0a, 0xcc], "vprold ymm0, dword [edx]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x72, 0x0a, 0xcc], "vprold ymm0{k5}, dword [edx]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x72, 0x4a, 0x01, 0xcc], "vprold ymm0, dword [edx + 0x4]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x72, 0x4a, 0x01, 0xcc], "vprold ymm0{k5}, dword [edx + 0x4]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x72, 0xca, 0xcc], "vprold ymm0{k5}{z}, ymm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x72, 0x0a, 0xcc], "vprold ymm0{k5}{z}, ymmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x72, 0x4a, 0x01, 0xcc], "vprold ymm0{k5}{z}, ymmword [edx + 0x20], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x72, 0xca, 0xcc], "vprold ymm0, ymm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x72, 0xca, 0xcc], "vprold ymm0{k5}, ymm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x72, 0x0a, 0xcc], "vprold ymm0, ymmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x72, 0x0a, 0xcc], "vprold ymm0{k5}, ymmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x72, 0x4a, 0x01, 0xcc], "vprold ymm0, ymmword [edx + 0x20], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x72, 0x4a, 0x01, 0xcc], "vprold ymm0{k5}, ymmword [edx + 0x20], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x72, 0x0a, 0xcc], "vprolq zmm0{k5}{z}, qword [edx]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x72, 0x4a, 0x01, 0xcc], "vprolq zmm0{k5}{z}, qword [edx + 0x8]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x72, 0x0a, 0xcc], "vprolq zmm0, qword [edx]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x72, 0x0a, 0xcc], "vprolq zmm0{k5}, qword [edx]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x72, 0x4a, 0x01, 0xcc], "vprolq zmm0, qword [edx + 0x8]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x72, 0x4a, 0x01, 0xcc], "vprolq zmm0{k5}, qword [edx + 0x8]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x72, 0x0a, 0xcc], "vprolq xmm0{k5}{z}, qword [edx]{1to2}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x72, 0x4a, 0x01, 0xcc], "vprolq xmm0{k5}{z}, qword [edx + 0x8]{1to2}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x72, 0x0a, 0xcc], "vprolq xmm0, qword [edx]{1to2}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x72, 0x0a, 0xcc], "vprolq xmm0{k5}, qword [edx]{1to2}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x72, 0x4a, 0x01, 0xcc], "vprolq xmm0, qword [edx + 0x8]{1to2}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x72, 0x4a, 0x01, 0xcc], "vprolq xmm0{k5}, qword [edx + 0x8]{1to2}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x72, 0xca, 0xcc], "vprolq zmm0{k5}{z}, zmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x72, 0x0a, 0xcc], "vprolq zmm0{k5}{z}, zmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x72, 0x4a, 0x01, 0xcc], "vprolq zmm0{k5}{z}, zmmword [edx + 0x40], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x72, 0xca, 0xcc], "vprolq zmm0, zmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x72, 0xca, 0xcc], "vprolq zmm0{k5}, zmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x72, 0x0a, 0xcc], "vprolq zmm0, zmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x72, 0x0a, 0xcc], "vprolq zmm0{k5}, zmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x72, 0x4a, 0x01, 0xcc], "vprolq zmm0, zmmword [edx + 0x40], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x72, 0x4a, 0x01, 0xcc], "vprolq zmm0{k5}, zmmword [edx + 0x40], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x72, 0xca, 0xcc], "vprolq xmm0{k5}{z}, xmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x72, 0x0a, 0xcc], "vprolq xmm0{k5}{z}, xmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x72, 0x4a, 0x01, 0xcc], "vprolq xmm0{k5}{z}, xmmword [edx + 0x10], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x72, 0xca, 0xcc], "vprolq xmm0, xmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x72, 0xca, 0xcc], "vprolq xmm0{k5}, xmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x72, 0x0a, 0xcc], "vprolq xmm0, xmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x72, 0x0a, 0xcc], "vprolq xmm0{k5}, xmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x72, 0x4a, 0x01, 0xcc], "vprolq xmm0, xmmword [edx + 0x10], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x72, 0x4a, 0x01, 0xcc], "vprolq xmm0{k5}, xmmword [edx + 0x10], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x72, 0x0a, 0xcc], "vprold zmm0{k5}{z}, dword [edx]{1to16}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x72, 0x4a, 0x01, 0xcc], "vprold zmm0{k5}{z}, dword [edx + 0x4]{1to16}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x72, 0x0a, 0xcc], "vprold zmm0, dword [edx]{1to16}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x72, 0x0a, 0xcc], "vprold zmm0{k5}, dword [edx]{1to16}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x72, 0x4a, 0x01, 0xcc], "vprold zmm0, dword [edx + 0x4]{1to16}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x72, 0x4a, 0x01, 0xcc], "vprold zmm0{k5}, dword [edx + 0x4]{1to16}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x72, 0x0a, 0xcc], "vprold xmm0{k5}{z}, dword [edx]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x72, 0x4a, 0x01, 0xcc], "vprold xmm0{k5}{z}, dword [edx + 0x4]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x72, 0x0a, 0xcc], "vprold xmm0, dword [edx]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x72, 0x0a, 0xcc], "vprold xmm0{k5}, dword [edx]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x72, 0x4a, 0x01, 0xcc], "vprold xmm0, dword [edx + 0x4]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x72, 0x4a, 0x01, 0xcc], "vprold xmm0{k5}, dword [edx + 0x4]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x72, 0xca, 0xcc], "vprold zmm0{k5}{z}, zmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x72, 0x0a, 0xcc], "vprold zmm0{k5}{z}, zmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x72, 0x4a, 0x01, 0xcc], "vprold zmm0{k5}{z}, zmmword [edx + 0x40], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x72, 0xca, 0xcc], "vprold zmm0, zmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x72, 0xca, 0xcc], "vprold zmm0{k5}, zmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x72, 0x0a, 0xcc], "vprold zmm0, zmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x72, 0x0a, 0xcc], "vprold zmm0{k5}, zmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x72, 0x4a, 0x01, 0xcc], "vprold zmm0, zmmword [edx + 0x40], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x72, 0x4a, 0x01, 0xcc], "vprold zmm0{k5}, zmmword [edx + 0x40], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x72, 0xca, 0xcc], "vprold xmm0{k5}{z}, xmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x72, 0x0a, 0xcc], "vprold xmm0{k5}{z}, xmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x72, 0x4a, 0x01, 0xcc], "vprold xmm0{k5}{z}, xmmword [edx + 0x10], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x72, 0xca, 0xcc], "vprold xmm0, xmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x72, 0xca, 0xcc], "vprold xmm0{k5}, xmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x72, 0x0a, 0xcc], "vprold xmm0, xmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x72, 0x0a, 0xcc], "vprold xmm0{k5}, xmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x72, 0x4a, 0x01, 0xcc], "vprold xmm0, xmmword [edx + 0x10], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x72, 0x4a, 0x01, 0xcc], "vprold xmm0{k5}, xmmword [edx + 0x10], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x74, 0xca], "vpcmpeqb k1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x74, 0xca], "vpcmpeqb k1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x74, 0x0a], "vpcmpeqb k1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x74, 0x0a], "vpcmpeqb k1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x74, 0x4a, 0x01], "vpcmpeqb k1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x74, 0x4a, 0x01], "vpcmpeqb k1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x74, 0xca], "vpcmpeqb k1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x74, 0xca], "vpcmpeqb k1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x74, 0x0a], "vpcmpeqb k1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x74, 0x0a], "vpcmpeqb k1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x74, 0x4a, 0x01], "vpcmpeqb k1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x74, 0x4a, 0x01], "vpcmpeqb k1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x74, 0xca], "vpcmpeqb k1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x74, 0xca], "vpcmpeqb k1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x74, 0x0a], "vpcmpeqb k1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x74, 0x0a], "vpcmpeqb k1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x74, 0x4a, 0x01], "vpcmpeqb k1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x74, 0x4a, 0x01], "vpcmpeqb k1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x75, 0xca], "vpcmpeqw k1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x75, 0xca], "vpcmpeqw k1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x75, 0x0a], "vpcmpeqw k1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x75, 0x0a], "vpcmpeqw k1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x75, 0x4a, 0x01], "vpcmpeqw k1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x75, 0x4a, 0x01], "vpcmpeqw k1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x75, 0xca], "vpcmpeqw k1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x75, 0xca], "vpcmpeqw k1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x75, 0x0a], "vpcmpeqw k1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x75, 0x0a], "vpcmpeqw k1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x75, 0x4a, 0x01], "vpcmpeqw k1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x75, 0x4a, 0x01], "vpcmpeqw k1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x75, 0xca], "vpcmpeqw k1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x75, 0xca], "vpcmpeqw k1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x75, 0x0a], "vpcmpeqw k1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x75, 0x0a], "vpcmpeqw k1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x75, 0x4a, 0x01], "vpcmpeqw k1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x75, 0x4a, 0x01], "vpcmpeqw k1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x76, 0x0a], "vpcmpeqd k1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x76, 0x0a], "vpcmpeqd k1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x76, 0x4a, 0x01], "vpcmpeqd k1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x76, 0x4a, 0x01], "vpcmpeqd k1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x76, 0xca], "vpcmpeqd k1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x76, 0xca], "vpcmpeqd k1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x76, 0x0a], "vpcmpeqd k1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x76, 0x0a], "vpcmpeqd k1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x76, 0x4a, 0x01], "vpcmpeqd k1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x76, 0x4a, 0x01], "vpcmpeqd k1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x76, 0x0a], "vpcmpeqd k1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x76, 0x0a], "vpcmpeqd k1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x76, 0x4a, 0x01], "vpcmpeqd k1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x76, 0x4a, 0x01], "vpcmpeqd k1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x76, 0x0a], "vpcmpeqd k1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x76, 0x0a], "vpcmpeqd k1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x76, 0x4a, 0x01], "vpcmpeqd k1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x76, 0x4a, 0x01], "vpcmpeqd k1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x76, 0xca], "vpcmpeqd k1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x76, 0xca], "vpcmpeqd k1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x76, 0x0a], "vpcmpeqd k1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x76, 0x0a], "vpcmpeqd k1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x76, 0x4a, 0x01], "vpcmpeqd k1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x76, 0x4a, 0x01], "vpcmpeqd k1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x76, 0xca], "vpcmpeqd k1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x76, 0xca], "vpcmpeqd k1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x76, 0x0a], "vpcmpeqd k1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x76, 0x0a], "vpcmpeqd k1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x76, 0x4a, 0x01], "vpcmpeqd k1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x76, 0x4a, 0x01], "vpcmpeqd k1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0x78, 0xca], "vcvttpd2uqq zmm1{k5}{z}{sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x78, 0xca], "vcvttpd2uqq zmm1{sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0x78, 0xca], "vcvttpd2uqq zmm1{k5}{sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x78, 0x0a], "vcvttpd2uqq ymm1{k5}{z}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x78, 0x4a, 0x01], "vcvttpd2uqq ymm1{k5}{z}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x78, 0x0a], "vcvttpd2uqq ymm1, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x78, 0x0a], "vcvttpd2uqq ymm1{k5}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x78, 0x4a, 0x01], "vcvttpd2uqq ymm1, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x78, 0x4a, 0x01], "vcvttpd2uqq ymm1{k5}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x78, 0xca], "vcvttpd2uqq ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x78, 0x0a], "vcvttpd2uqq ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x78, 0x4a, 0x01], "vcvttpd2uqq ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x78, 0xca], "vcvttpd2uqq ymm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x78, 0xca], "vcvttpd2uqq ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x78, 0x0a], "vcvttpd2uqq ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x78, 0x0a], "vcvttpd2uqq ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x78, 0x4a, 0x01], "vcvttpd2uqq ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x78, 0x4a, 0x01], "vcvttpd2uqq ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xfd, 0x78, 0xca], "vcvttps2uqq zmm1{k5}{z}{sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x78, 0x78, 0xca], "vcvttps2uqq zmm1{sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x7d, 0x78, 0xca], "vcvttps2uqq zmm1{k5}{sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x78, 0x0a], "vcvttps2uqq ymm1{k5}{z}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x78, 0x4a, 0x01], "vcvttps2uqq ymm1{k5}{z}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x78, 0x0a], "vcvttps2uqq ymm1, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x78, 0x0a], "vcvttps2uqq ymm1{k5}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x78, 0x4a, 0x01], "vcvttps2uqq ymm1, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x78, 0x4a, 0x01], "vcvttps2uqq ymm1{k5}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x78, 0xca], "vcvttps2uqq ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x78, 0x0a], "vcvttps2uqq ymm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x78, 0x4a, 0x01], "vcvttps2uqq ymm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x78, 0xca], "vcvttps2uqq ymm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x78, 0xca], "vcvttps2uqq ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x78, 0x0a], "vcvttps2uqq ymm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x78, 0x0a], "vcvttps2uqq ymm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x78, 0x4a, 0x01], "vcvttps2uqq ymm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x78, 0x4a, 0x01], "vcvttps2uqq ymm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x78, 0x0a], "vcvttpd2uqq zmm1{k5}{z}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x78, 0x4a, 0x01], "vcvttpd2uqq zmm1{k5}{z}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x78, 0x0a], "vcvttpd2uqq zmm1, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x78, 0x0a], "vcvttpd2uqq zmm1{k5}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x78, 0x4a, 0x01], "vcvttpd2uqq zmm1, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x78, 0x4a, 0x01], "vcvttpd2uqq zmm1{k5}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x78, 0x0a], "vcvttpd2uqq xmm1{k5}{z}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x78, 0x4a, 0x01], "vcvttpd2uqq xmm1{k5}{z}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x78, 0x0a], "vcvttpd2uqq xmm1, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x78, 0x0a], "vcvttpd2uqq xmm1{k5}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x78, 0x4a, 0x01], "vcvttpd2uqq xmm1, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x78, 0x4a, 0x01], "vcvttpd2uqq xmm1{k5}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x78, 0xca], "vcvttpd2uqq zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x78, 0x0a], "vcvttpd2uqq zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x78, 0x4a, 0x01], "vcvttpd2uqq zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x78, 0xca], "vcvttpd2uqq zmm1, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x78, 0xca], "vcvttpd2uqq zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x78, 0x0a], "vcvttpd2uqq zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x78, 0x0a], "vcvttpd2uqq zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x78, 0x4a, 0x01], "vcvttpd2uqq zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x78, 0x4a, 0x01], "vcvttpd2uqq zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x78, 0xca], "vcvttpd2uqq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x78, 0x0a], "vcvttpd2uqq xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x78, 0x4a, 0x01], "vcvttpd2uqq xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x78, 0xca], "vcvttpd2uqq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x78, 0xca], "vcvttpd2uqq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x78, 0x0a], "vcvttpd2uqq xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x78, 0x0a], "vcvttpd2uqq xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x78, 0x4a, 0x01], "vcvttpd2uqq xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x78, 0x4a, 0x01], "vcvttpd2uqq xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x78, 0x0a], "vcvttps2uqq zmm1{k5}{z}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x78, 0x4a, 0x01], "vcvttps2uqq zmm1{k5}{z}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x78, 0x0a], "vcvttps2uqq zmm1, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x78, 0x0a], "vcvttps2uqq zmm1{k5}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x78, 0x4a, 0x01], "vcvttps2uqq zmm1, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x78, 0x4a, 0x01], "vcvttps2uqq zmm1{k5}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x78, 0x0a], "vcvttps2uqq xmm1{k5}{z}, dword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x78, 0x4a, 0x01], "vcvttps2uqq xmm1{k5}{z}, dword [edx + 0x4]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x78, 0x0a], "vcvttps2uqq xmm1, dword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x78, 0x0a], "vcvttps2uqq xmm1{k5}, dword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x78, 0x4a, 0x01], "vcvttps2uqq xmm1, dword [edx + 0x4]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x78, 0x4a, 0x01], "vcvttps2uqq xmm1{k5}, dword [edx + 0x4]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x78, 0xca], "vcvttps2uqq zmm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x78, 0x0a], "vcvttps2uqq zmm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x78, 0x4a, 0x01], "vcvttps2uqq zmm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x78, 0xca], "vcvttps2uqq zmm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x78, 0xca], "vcvttps2uqq zmm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x78, 0x0a], "vcvttps2uqq zmm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x78, 0x0a], "vcvttps2uqq zmm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x78, 0x4a, 0x01], "vcvttps2uqq zmm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x78, 0x4a, 0x01], "vcvttps2uqq zmm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x78, 0xca], "vcvttps2uqq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x78, 0x0a], "vcvttps2uqq xmm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x78, 0x4a, 0x01], "vcvttps2uqq xmm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x78, 0xca], "vcvttps2uqq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x78, 0xca], "vcvttps2uqq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x78, 0x0a], "vcvttps2uqq xmm1, qword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x78, 0x0a], "vcvttps2uqq xmm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x78, 0x4a, 0x01], "vcvttps2uqq xmm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x78, 0x4a, 0x01], "vcvttps2uqq xmm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{z}{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x79, 0xca], "vcvtpd2uqq zmm1{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{z}{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x79, 0x0a], "vcvtpd2uqq ymm1{k5}{z}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x79, 0x4a, 0x01], "vcvtpd2uqq ymm1{k5}{z}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x79, 0xca], "vcvtpd2uqq zmm1{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x79, 0x0a], "vcvtpd2uqq ymm1, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x79, 0x0a], "vcvtpd2uqq ymm1{k5}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x79, 0x4a, 0x01], "vcvtpd2uqq ymm1, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x79, 0x4a, 0x01], "vcvtpd2uqq ymm1{k5}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x79, 0xca], "vcvtpd2uqq ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x79, 0x0a], "vcvtpd2uqq ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x79, 0x4a, 0x01], "vcvtpd2uqq ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x79, 0xca], "vcvtpd2uqq ymm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x79, 0xca], "vcvtpd2uqq ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x79, 0x0a], "vcvtpd2uqq ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x79, 0x0a], "vcvtpd2uqq ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x79, 0x4a, 0x01], "vcvtpd2uqq ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x79, 0x4a, 0x01], "vcvtpd2uqq ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xfd, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{z}{rz-sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x78, 0x79, 0xca], "vcvtps2uqq zmm1{rz-sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x7d, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{rz-sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{z}{rd-sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x79, 0x0a], "vcvtps2uqq ymm1{k5}{z}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x79, 0x4a, 0x01], "vcvtps2uqq ymm1{k5}{z}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x79, 0xca], "vcvtps2uqq zmm1{rd-sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{rd-sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x79, 0x0a], "vcvtps2uqq ymm1, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x79, 0x0a], "vcvtps2uqq ymm1{k5}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x79, 0x4a, 0x01], "vcvtps2uqq ymm1, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x79, 0x4a, 0x01], "vcvtps2uqq ymm1{k5}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x79, 0xca], "vcvtps2uqq ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x79, 0x0a], "vcvtps2uqq ymm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x79, 0x4a, 0x01], "vcvtps2uqq ymm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x79, 0xca], "vcvtps2uqq ymm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x79, 0xca], "vcvtps2uqq ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x79, 0x0a], "vcvtps2uqq ymm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x79, 0x0a], "vcvtps2uqq ymm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x79, 0x4a, 0x01], "vcvtps2uqq ymm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x79, 0x4a, 0x01], "vcvtps2uqq ymm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{z}{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x79, 0x0a], "vcvtpd2uqq zmm1{k5}{z}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x79, 0x4a, 0x01], "vcvtpd2uqq zmm1{k5}{z}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x79, 0xca], "vcvtpd2uqq zmm1{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x79, 0x0a], "vcvtpd2uqq zmm1, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x79, 0x0a], "vcvtpd2uqq zmm1{k5}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x79, 0x4a, 0x01], "vcvtpd2uqq zmm1, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x79, 0x4a, 0x01], "vcvtpd2uqq zmm1{k5}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{z}{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x79, 0x0a], "vcvtpd2uqq xmm1{k5}{z}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x79, 0x4a, 0x01], "vcvtpd2uqq xmm1{k5}{z}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x79, 0xca], "vcvtpd2uqq zmm1{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x79, 0x0a], "vcvtpd2uqq xmm1, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x79, 0x0a], "vcvtpd2uqq xmm1{k5}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x79, 0x4a, 0x01], "vcvtpd2uqq xmm1, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x79, 0x4a, 0x01], "vcvtpd2uqq xmm1{k5}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x79, 0x0a], "vcvtpd2uqq zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x79, 0x4a, 0x01], "vcvtpd2uqq zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x79, 0xca], "vcvtpd2uqq zmm1, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x79, 0x0a], "vcvtpd2uqq zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x79, 0x0a], "vcvtpd2uqq zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x79, 0x4a, 0x01], "vcvtpd2uqq zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x79, 0x4a, 0x01], "vcvtpd2uqq zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x79, 0xca], "vcvtpd2uqq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x79, 0x0a], "vcvtpd2uqq xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x79, 0x4a, 0x01], "vcvtpd2uqq xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x79, 0xca], "vcvtpd2uqq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x79, 0xca], "vcvtpd2uqq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x79, 0x0a], "vcvtpd2uqq xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x79, 0x0a], "vcvtpd2uqq xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x79, 0x4a, 0x01], "vcvtpd2uqq xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x79, 0x4a, 0x01], "vcvtpd2uqq xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{z}{ru-sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x79, 0x0a], "vcvtps2uqq zmm1{k5}{z}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x79, 0x4a, 0x01], "vcvtps2uqq zmm1{k5}{z}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x79, 0xca], "vcvtps2uqq zmm1{ru-sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{ru-sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x79, 0x0a], "vcvtps2uqq zmm1, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x79, 0x0a], "vcvtps2uqq zmm1{k5}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x79, 0x4a, 0x01], "vcvtps2uqq zmm1, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x79, 0x4a, 0x01], "vcvtps2uqq zmm1{k5}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{z}{rne-sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x79, 0x0a], "vcvtps2uqq xmm1{k5}{z}, dword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x79, 0x4a, 0x01], "vcvtps2uqq xmm1{k5}{z}, dword [edx + 0x4]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x79, 0xca], "vcvtps2uqq zmm1{rne-sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{rne-sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x79, 0x0a], "vcvtps2uqq xmm1, dword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x79, 0x0a], "vcvtps2uqq xmm1{k5}, dword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x79, 0x4a, 0x01], "vcvtps2uqq xmm1, dword [edx + 0x4]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x79, 0x4a, 0x01], "vcvtps2uqq xmm1{k5}, dword [edx + 0x4]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x79, 0x0a], "vcvtps2uqq zmm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x79, 0x4a, 0x01], "vcvtps2uqq zmm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x79, 0xca], "vcvtps2uqq zmm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x79, 0xca], "vcvtps2uqq zmm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x79, 0x0a], "vcvtps2uqq zmm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x79, 0x0a], "vcvtps2uqq zmm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x79, 0x4a, 0x01], "vcvtps2uqq zmm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x79, 0x4a, 0x01], "vcvtps2uqq zmm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x79, 0xca], "vcvtps2uqq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x79, 0x0a], "vcvtps2uqq xmm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x79, 0x4a, 0x01], "vcvtps2uqq xmm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x79, 0xca], "vcvtps2uqq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x79, 0xca], "vcvtps2uqq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x79, 0x0a], "vcvtps2uqq xmm1, qword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x79, 0x0a], "vcvtps2uqq xmm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x79, 0x4a, 0x01], "vcvtps2uqq xmm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x79, 0x4a, 0x01], "vcvtps2uqq xmm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0x7a, 0xca], "vcvttpd2qq zmm1{k5}{z}{sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x7a, 0xca], "vcvttpd2qq zmm1{sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0x7a, 0xca], "vcvttpd2qq zmm1{k5}{sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x7a, 0x0a], "vcvttpd2qq ymm1{k5}{z}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x7a, 0x4a, 0x01], "vcvttpd2qq ymm1{k5}{z}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x7a, 0x0a], "vcvttpd2qq ymm1, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x7a, 0x0a], "vcvttpd2qq ymm1{k5}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x7a, 0x4a, 0x01], "vcvttpd2qq ymm1, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x7a, 0x4a, 0x01], "vcvttpd2qq ymm1{k5}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x7a, 0xca], "vcvttpd2qq ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x7a, 0x0a], "vcvttpd2qq ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x7a, 0x4a, 0x01], "vcvttpd2qq ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x7a, 0xca], "vcvttpd2qq ymm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x7a, 0xca], "vcvttpd2qq ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x7a, 0x0a], "vcvttpd2qq ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x7a, 0x0a], "vcvttpd2qq ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x7a, 0x4a, 0x01], "vcvttpd2qq ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x7a, 0x4a, 0x01], "vcvttpd2qq ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xfd, 0x7a, 0xca], "vcvttps2qq zmm1{k5}{z}{sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x78, 0x7a, 0xca], "vcvttps2qq zmm1{sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x7d, 0x7a, 0xca], "vcvttps2qq zmm1{k5}{sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x7a, 0x0a], "vcvttps2qq ymm1{k5}{z}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x7a, 0x4a, 0x01], "vcvttps2qq ymm1{k5}{z}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x7a, 0x0a], "vcvttps2qq ymm1, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x7a, 0x0a], "vcvttps2qq ymm1{k5}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x7a, 0x4a, 0x01], "vcvttps2qq ymm1, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x7a, 0x4a, 0x01], "vcvttps2qq ymm1{k5}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x7a, 0xca], "vcvttps2qq ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x7a, 0x0a], "vcvttps2qq ymm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x7a, 0x4a, 0x01], "vcvttps2qq ymm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x7a, 0xca], "vcvttps2qq ymm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x7a, 0xca], "vcvttps2qq ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x7a, 0x0a], "vcvttps2qq ymm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x7a, 0x0a], "vcvttps2qq ymm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x7a, 0x4a, 0x01], "vcvttps2qq ymm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x7a, 0x4a, 0x01], "vcvttps2qq ymm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x7a, 0x0a], "vcvttpd2qq zmm1{k5}{z}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x7a, 0x4a, 0x01], "vcvttpd2qq zmm1{k5}{z}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x7a, 0x0a], "vcvttpd2qq zmm1, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x7a, 0x0a], "vcvttpd2qq zmm1{k5}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x7a, 0x4a, 0x01], "vcvttpd2qq zmm1, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x7a, 0x4a, 0x01], "vcvttpd2qq zmm1{k5}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x7a, 0x0a], "vcvttpd2qq xmm1{k5}{z}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x7a, 0x4a, 0x01], "vcvttpd2qq xmm1{k5}{z}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x7a, 0x0a], "vcvttpd2qq xmm1, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x7a, 0x0a], "vcvttpd2qq xmm1{k5}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x7a, 0x4a, 0x01], "vcvttpd2qq xmm1, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x7a, 0x4a, 0x01], "vcvttpd2qq xmm1{k5}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x7a, 0xca], "vcvttpd2qq zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x7a, 0x0a], "vcvttpd2qq zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x7a, 0x4a, 0x01], "vcvttpd2qq zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x7a, 0xca], "vcvttpd2qq zmm1, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x7a, 0xca], "vcvttpd2qq zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x7a, 0x0a], "vcvttpd2qq zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x7a, 0x0a], "vcvttpd2qq zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x7a, 0x4a, 0x01], "vcvttpd2qq zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x7a, 0x4a, 0x01], "vcvttpd2qq zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x7a, 0xca], "vcvttpd2qq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x7a, 0x0a], "vcvttpd2qq xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x7a, 0x4a, 0x01], "vcvttpd2qq xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x7a, 0xca], "vcvttpd2qq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x7a, 0xca], "vcvttpd2qq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x7a, 0x0a], "vcvttpd2qq xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x7a, 0x0a], "vcvttpd2qq xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x7a, 0x4a, 0x01], "vcvttpd2qq xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x7a, 0x4a, 0x01], "vcvttpd2qq xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x7a, 0x0a], "vcvttps2qq zmm1{k5}{z}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x7a, 0x4a, 0x01], "vcvttps2qq zmm1{k5}{z}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x7a, 0x0a], "vcvttps2qq zmm1, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x7a, 0x0a], "vcvttps2qq zmm1{k5}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x7a, 0x4a, 0x01], "vcvttps2qq zmm1, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x7a, 0x4a, 0x01], "vcvttps2qq zmm1{k5}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x7a, 0x0a], "vcvttps2qq xmm1{k5}{z}, dword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x7a, 0x4a, 0x01], "vcvttps2qq xmm1{k5}{z}, dword [edx + 0x4]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x7a, 0x0a], "vcvttps2qq xmm1, dword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x7a, 0x0a], "vcvttps2qq xmm1{k5}, dword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x7a, 0x4a, 0x01], "vcvttps2qq xmm1, dword [edx + 0x4]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x7a, 0x4a, 0x01], "vcvttps2qq xmm1{k5}, dword [edx + 0x4]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x7a, 0xca], "vcvttps2qq zmm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x7a, 0x0a], "vcvttps2qq zmm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x7a, 0x4a, 0x01], "vcvttps2qq zmm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x7a, 0xca], "vcvttps2qq zmm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x7a, 0xca], "vcvttps2qq zmm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x7a, 0x0a], "vcvttps2qq zmm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x7a, 0x0a], "vcvttps2qq zmm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x7a, 0x4a, 0x01], "vcvttps2qq zmm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x7a, 0x4a, 0x01], "vcvttps2qq zmm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x7a, 0xca], "vcvttps2qq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x7a, 0x0a], "vcvttps2qq xmm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x7a, 0x4a, 0x01], "vcvttps2qq xmm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x7a, 0xca], "vcvttps2qq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x7a, 0xca], "vcvttps2qq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x7a, 0x0a], "vcvttps2qq xmm1, qword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x7a, 0x0a], "vcvttps2qq xmm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x7a, 0x4a, 0x01], "vcvttps2qq xmm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x7a, 0x4a, 0x01], "vcvttps2qq xmm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{z}{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x7b, 0xca], "vcvtpd2qq zmm1{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{rz-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{z}{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x7b, 0x0a], "vcvtpd2qq ymm1{k5}{z}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x7b, 0x4a, 0x01], "vcvtpd2qq ymm1{k5}{z}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x7b, 0xca], "vcvtpd2qq zmm1{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{rd-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x7b, 0x0a], "vcvtpd2qq ymm1, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x7b, 0x0a], "vcvtpd2qq ymm1{k5}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x7b, 0x4a, 0x01], "vcvtpd2qq ymm1, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x7b, 0x4a, 0x01], "vcvtpd2qq ymm1{k5}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x7b, 0xca], "vcvtpd2qq ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x7b, 0x0a], "vcvtpd2qq ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x7b, 0x4a, 0x01], "vcvtpd2qq ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x7b, 0xca], "vcvtpd2qq ymm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x7b, 0xca], "vcvtpd2qq ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x7b, 0x0a], "vcvtpd2qq ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x7b, 0x0a], "vcvtpd2qq ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x7b, 0x4a, 0x01], "vcvtpd2qq ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x7b, 0x4a, 0x01], "vcvtpd2qq ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xfd, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{z}{rz-sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x78, 0x7b, 0xca], "vcvtps2qq zmm1{rz-sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x7d, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{rz-sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{z}{rd-sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x7b, 0x0a], "vcvtps2qq ymm1{k5}{z}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x7b, 0x4a, 0x01], "vcvtps2qq ymm1{k5}{z}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x7b, 0xca], "vcvtps2qq zmm1{rd-sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{rd-sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x7b, 0x0a], "vcvtps2qq ymm1, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x7b, 0x0a], "vcvtps2qq ymm1{k5}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x7b, 0x4a, 0x01], "vcvtps2qq ymm1, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x7b, 0x4a, 0x01], "vcvtps2qq ymm1{k5}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x7b, 0xca], "vcvtps2qq ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x7b, 0x0a], "vcvtps2qq ymm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x7b, 0x4a, 0x01], "vcvtps2qq ymm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x7b, 0xca], "vcvtps2qq ymm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x7b, 0xca], "vcvtps2qq ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x7b, 0x0a], "vcvtps2qq ymm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x7b, 0x0a], "vcvtps2qq ymm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x7b, 0x4a, 0x01], "vcvtps2qq ymm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x7b, 0x4a, 0x01], "vcvtps2qq ymm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{z}{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x7b, 0x0a], "vcvtpd2qq zmm1{k5}{z}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x7b, 0x4a, 0x01], "vcvtpd2qq zmm1{k5}{z}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x7b, 0xca], "vcvtpd2qq zmm1{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{ru-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x7b, 0x0a], "vcvtpd2qq zmm1, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x7b, 0x0a], "vcvtpd2qq zmm1{k5}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x7b, 0x4a, 0x01], "vcvtpd2qq zmm1, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x7b, 0x4a, 0x01], "vcvtpd2qq zmm1{k5}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{z}{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x7b, 0x0a], "vcvtpd2qq xmm1{k5}{z}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x7b, 0x4a, 0x01], "vcvtpd2qq xmm1{k5}{z}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x7b, 0xca], "vcvtpd2qq zmm1{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{rne-sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x7b, 0x0a], "vcvtpd2qq xmm1, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x7b, 0x0a], "vcvtpd2qq xmm1{k5}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x7b, 0x4a, 0x01], "vcvtpd2qq xmm1, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x7b, 0x4a, 0x01], "vcvtpd2qq xmm1{k5}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x7b, 0x0a], "vcvtpd2qq zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x7b, 0x4a, 0x01], "vcvtpd2qq zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x7b, 0xca], "vcvtpd2qq zmm1, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x7b, 0x0a], "vcvtpd2qq zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x7b, 0x0a], "vcvtpd2qq zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x7b, 0x4a, 0x01], "vcvtpd2qq zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x7b, 0x4a, 0x01], "vcvtpd2qq zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x7b, 0xca], "vcvtpd2qq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x7b, 0x0a], "vcvtpd2qq xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x7b, 0x4a, 0x01], "vcvtpd2qq xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x7b, 0xca], "vcvtpd2qq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x7b, 0xca], "vcvtpd2qq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x7b, 0x0a], "vcvtpd2qq xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x7b, 0x0a], "vcvtpd2qq xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x7b, 0x4a, 0x01], "vcvtpd2qq xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x7b, 0x4a, 0x01], "vcvtpd2qq xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{z}{ru-sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x7b, 0x0a], "vcvtps2qq zmm1{k5}{z}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x7b, 0x4a, 0x01], "vcvtps2qq zmm1{k5}{z}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x7b, 0xca], "vcvtps2qq zmm1{ru-sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{ru-sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x7b, 0x0a], "vcvtps2qq zmm1, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x7b, 0x0a], "vcvtps2qq zmm1{k5}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x7b, 0x4a, 0x01], "vcvtps2qq zmm1, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x7b, 0x4a, 0x01], "vcvtps2qq zmm1{k5}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{z}{rne-sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x7b, 0x0a], "vcvtps2qq xmm1{k5}{z}, dword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x7b, 0x4a, 0x01], "vcvtps2qq xmm1{k5}{z}, dword [edx + 0x4]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x7b, 0xca], "vcvtps2qq zmm1{rne-sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{rne-sae}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x7b, 0x0a], "vcvtps2qq xmm1, dword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x7b, 0x0a], "vcvtps2qq xmm1{k5}, dword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x7b, 0x4a, 0x01], "vcvtps2qq xmm1, dword [edx + 0x4]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x7b, 0x4a, 0x01], "vcvtps2qq xmm1{k5}, dword [edx + 0x4]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x7b, 0x0a], "vcvtps2qq zmm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x7b, 0x4a, 0x01], "vcvtps2qq zmm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x7b, 0xca], "vcvtps2qq zmm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x7b, 0xca], "vcvtps2qq zmm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x7b, 0x0a], "vcvtps2qq zmm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x7b, 0x0a], "vcvtps2qq zmm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x7b, 0x4a, 0x01], "vcvtps2qq zmm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x7b, 0x4a, 0x01], "vcvtps2qq zmm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x7b, 0xca], "vcvtps2qq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x7b, 0x0a], "vcvtps2qq xmm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x7b, 0x4a, 0x01], "vcvtps2qq xmm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x7b, 0xca], "vcvtps2qq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x7b, 0xca], "vcvtps2qq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x7b, 0x0a], "vcvtps2qq xmm1, qword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x7b, 0x0a], "vcvtps2qq xmm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x7b, 0x4a, 0x01], "vcvtps2qq xmm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x7b, 0x4a, 0x01], "vcvtps2qq xmm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x7e, 0xca], "vmovd edx, xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x7e, 0x0a], "vmovd dword [edx], xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x7e, 0x4a, 0x01], "vmovd dword [edx + 0x4], xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x7f, 0xca], "vmovdqa64 ymm2{k5}{z}, ymm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x7f, 0xca], "vmovdqa64 ymm2, ymm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x7f, 0xca], "vmovdqa64 ymm2{k5}, ymm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x7f, 0x0a], "vmovdqa64 ymmword [edx], ymm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x7f, 0x0a], "vmovdqa64 ymmword [edx]{k5}, ymm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x7f, 0x4a, 0x01], "vmovdqa64 ymmword [edx + 0x20], ymm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x7f, 0x4a, 0x01], "vmovdqa64 ymmword [edx + 0x20]{k5}, ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x7f, 0xca], "vmovdqa32 ymm2{k5}{z}, ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x7f, 0xca], "vmovdqa32 ymm2, ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x7f, 0xca], "vmovdqa32 ymm2{k5}, ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x7f, 0x0a], "vmovdqa32 ymmword [edx], ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x7f, 0x0a], "vmovdqa32 ymmword [edx]{k5}, ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x7f, 0x4a, 0x01], "vmovdqa32 ymmword [edx + 0x20], ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x7f, 0x4a, 0x01], "vmovdqa32 ymmword [edx + 0x20]{k5}, ymm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x7f, 0xca], "vmovdqa64 zmm2{k5}{z}, zmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x7f, 0xca], "vmovdqa64 zmm2, zmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x7f, 0xca], "vmovdqa64 zmm2{k5}, zmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x7f, 0x0a], "vmovdqa64 zmmword [edx], zmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x7f, 0x0a], "vmovdqa64 zmmword [edx]{k5}, zmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x7f, 0x4a, 0x01], "vmovdqa64 zmmword [edx + 0x40], zmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x7f, 0x4a, 0x01], "vmovdqa64 zmmword [edx + 0x40]{k5}, zmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x7f, 0xca], "vmovdqa64 xmm2{k5}{z}, xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x7f, 0xca], "vmovdqa64 xmm2, xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x7f, 0xca], "vmovdqa64 xmm2{k5}, xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x7f, 0x0a], "vmovdqa64 xmmword [edx], xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x7f, 0x0a], "vmovdqa64 xmmword [edx]{k5}, xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x7f, 0x4a, 0x01], "vmovdqa64 xmmword [edx + 0x10], xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x7f, 0x4a, 0x01], "vmovdqa64 xmmword [edx + 0x10]{k5}, xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x7f, 0xca], "vmovdqa32 zmm2{k5}{z}, zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x7f, 0xca], "vmovdqa32 zmm2, zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x7f, 0xca], "vmovdqa32 zmm2{k5}, zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x7f, 0x0a], "vmovdqa32 zmmword [edx], zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x7f, 0x0a], "vmovdqa32 zmmword [edx]{k5}, zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x7f, 0x4a, 0x01], "vmovdqa32 zmmword [edx + 0x40], zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x7f, 0x4a, 0x01], "vmovdqa32 zmmword [edx + 0x40]{k5}, zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x7f, 0xca], "vmovdqa32 xmm2{k5}{z}, xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x7f, 0xca], "vmovdqa32 xmm2, xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x7f, 0xca], "vmovdqa32 xmm2{k5}, xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x7f, 0x0a], "vmovdqa32 xmmword [edx], xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x7f, 0x0a], "vmovdqa32 xmmword [edx]{k5}, xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x7f, 0x4a, 0x01], "vmovdqa32 xmmword [edx + 0x10], xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x7f, 0x4a, 0x01], "vmovdqa32 xmmword [edx + 0x10]{k5}, xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0xc2, 0xca, 0xcc], "vcmppd k1{sae}, zmm0, zmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0xc2, 0xca, 0xcc], "vcmppd k1{k5}{sae}, zmm0, zmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xc2, 0x0a, 0xcc], "vcmppd k1, ymm0, qword [edx]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xc2, 0x0a, 0xcc], "vcmppd k1{k5}, ymm0, qword [edx]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xc2, 0x4a, 0x01, 0xcc], "vcmppd k1, ymm0, qword [edx + 0x8]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xc2, 0x4a, 0x01, 0xcc], "vcmppd k1{k5}, ymm0, qword [edx + 0x8]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xc2, 0xca, 0xcc], "vcmppd k1, ymm0, ymm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xc2, 0xca, 0xcc], "vcmppd k1{k5}, ymm0, ymm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xc2, 0x0a, 0xcc], "vcmppd k1, ymm0, ymmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xc2, 0x0a, 0xcc], "vcmppd k1{k5}, ymm0, ymmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xc2, 0x4a, 0x01, 0xcc], "vcmppd k1, ymm0, ymmword [edx + 0x20], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xc2, 0x4a, 0x01, 0xcc], "vcmppd k1{k5}, ymm0, ymmword [edx + 0x20], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xc2, 0x0a, 0xcc], "vcmppd k1, zmm0, qword [edx]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xc2, 0x0a, 0xcc], "vcmppd k1{k5}, zmm0, qword [edx]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xc2, 0x4a, 0x01, 0xcc], "vcmppd k1, zmm0, qword [edx + 0x8]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xc2, 0x4a, 0x01, 0xcc], "vcmppd k1{k5}, zmm0, qword [edx + 0x8]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xc2, 0x0a, 0xcc], "vcmppd k1, xmm0, qword [edx]{1to2}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xc2, 0x0a, 0xcc], "vcmppd k1{k5}, xmm0, qword [edx]{1to2}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xc2, 0x4a, 0x01, 0xcc], "vcmppd k1, xmm0, qword [edx + 0x8]{1to2}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xc2, 0x4a, 0x01, 0xcc], "vcmppd k1{k5}, xmm0, qword [edx + 0x8]{1to2}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xc2, 0xca, 0xcc], "vcmppd k1, zmm0, zmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xc2, 0xca, 0xcc], "vcmppd k1{k5}, zmm0, zmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xc2, 0x0a, 0xcc], "vcmppd k1, zmm0, zmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xc2, 0x0a, 0xcc], "vcmppd k1{k5}, zmm0, zmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xc2, 0x4a, 0x01, 0xcc], "vcmppd k1, zmm0, zmmword [edx + 0x40], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xc2, 0x4a, 0x01, 0xcc], "vcmppd k1{k5}, zmm0, zmmword [edx + 0x40], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xc2, 0xca, 0xcc], "vcmppd k1, xmm0, xmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xc2, 0xca, 0xcc], "vcmppd k1{k5}, xmm0, xmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xc2, 0x0a, 0xcc], "vcmppd k1, xmm0, xmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xc2, 0x0a, 0xcc], "vcmppd k1{k5}, xmm0, xmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xc2, 0x4a, 0x01, 0xcc], "vcmppd k1, xmm0, xmmword [edx + 0x10], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xc2, 0x4a, 0x01, 0xcc], "vcmppd k1{k5}, xmm0, xmmword [edx + 0x10], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xc4, 0xca, 0xcc], "vpinsrw xmm1, xmm0, edx, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xc4, 0x0a, 0xcc], "vpinsrw xmm1, xmm0, word [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xc4, 0x4a, 0x01, 0xcc], "vpinsrw xmm1, xmm0, word [edx + 0x2], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xc5, 0xca, 0xcc], "vpextrw ecx, xmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xc6, 0x0a, 0xcc], "vshufpd ymm1{k5}{z}, ymm0, qword [edx]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xc6, 0x0a, 0xcc], "vshufpd ymm1, ymm0, qword [edx]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xc6, 0x0a, 0xcc], "vshufpd ymm1{k5}, ymm0, qword [edx]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd ymm1, ymm0, qword [edx + 0x8]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xc6, 0xca, 0xcc], "vshufpd ymm1{k5}{z}, ymm0, ymm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xc6, 0x0a, 0xcc], "vshufpd ymm1{k5}{z}, ymm0, ymmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xc6, 0xca, 0xcc], "vshufpd ymm1, ymm0, ymm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xc6, 0xca, 0xcc], "vshufpd ymm1{k5}, ymm0, ymm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xc6, 0x0a, 0xcc], "vshufpd ymm1, ymm0, ymmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xc6, 0x0a, 0xcc], "vshufpd ymm1{k5}, ymm0, ymmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd ymm1, ymm0, ymmword [edx + 0x20], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd ymm1{k5}, ymm0, ymmword [edx + 0x20], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xc6, 0x0a, 0xcc], "vshufpd zmm1{k5}{z}, zmm0, qword [edx]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xc6, 0x0a, 0xcc], "vshufpd zmm1, zmm0, qword [edx]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xc6, 0x0a, 0xcc], "vshufpd zmm1{k5}, zmm0, qword [edx]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd zmm1, zmm0, qword [edx + 0x8]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xc6, 0x0a, 0xcc], "vshufpd xmm1{k5}{z}, xmm0, qword [edx]{1to2}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xc6, 0x0a, 0xcc], "vshufpd xmm1, xmm0, qword [edx]{1to2}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xc6, 0x0a, 0xcc], "vshufpd xmm1{k5}, xmm0, qword [edx]{1to2}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd xmm1, xmm0, qword [edx + 0x8]{1to2}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xc6, 0xca, 0xcc], "vshufpd zmm1{k5}{z}, zmm0, zmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xc6, 0x0a, 0xcc], "vshufpd zmm1{k5}{z}, zmm0, zmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xc6, 0xca, 0xcc], "vshufpd zmm1, zmm0, zmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xc6, 0xca, 0xcc], "vshufpd zmm1{k5}, zmm0, zmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xc6, 0x0a, 0xcc], "vshufpd zmm1, zmm0, zmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xc6, 0x0a, 0xcc], "vshufpd zmm1{k5}, zmm0, zmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd zmm1, zmm0, zmmword [edx + 0x40], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd zmm1{k5}, zmm0, zmmword [edx + 0x40], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xc6, 0xca, 0xcc], "vshufpd xmm1{k5}{z}, xmm0, xmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xc6, 0x0a, 0xcc], "vshufpd xmm1{k5}{z}, xmm0, xmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xc6, 0xca, 0xcc], "vshufpd xmm1, xmm0, xmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xc6, 0xca, 0xcc], "vshufpd xmm1{k5}, xmm0, xmm2, 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xc6, 0x0a, 0xcc], "vshufpd xmm1, xmm0, xmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xc6, 0x0a, 0xcc], "vshufpd xmm1{k5}, xmm0, xmmword [edx], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd xmm1, xmm0, xmmword [edx + 0x10], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd xmm1{k5}, xmm0, xmmword [edx + 0x10], 0xcc");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd1, 0xca], "vpsrlw ymm1{k5}{z}, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd1, 0x0a], "vpsrlw ymm1{k5}{z}, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd1, 0x4a, 0x01], "vpsrlw ymm1{k5}{z}, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd1, 0xca], "vpsrlw ymm1, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd1, 0xca], "vpsrlw ymm1{k5}, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd1, 0x0a], "vpsrlw ymm1, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd1, 0x0a], "vpsrlw ymm1{k5}, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd1, 0x4a, 0x01], "vpsrlw ymm1, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd1, 0x4a, 0x01], "vpsrlw ymm1{k5}, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd1, 0xca], "vpsrlw zmm1{k5}{z}, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd1, 0x0a], "vpsrlw zmm1{k5}{z}, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd1, 0x4a, 0x01], "vpsrlw zmm1{k5}{z}, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd1, 0xca], "vpsrlw zmm1, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd1, 0xca], "vpsrlw zmm1{k5}, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd1, 0x0a], "vpsrlw zmm1, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd1, 0x0a], "vpsrlw zmm1{k5}, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd1, 0x4a, 0x01], "vpsrlw zmm1, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd1, 0x4a, 0x01], "vpsrlw zmm1{k5}, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd1, 0xca], "vpsrlw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd1, 0x0a], "vpsrlw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd1, 0x4a, 0x01], "vpsrlw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd1, 0xca], "vpsrlw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd1, 0xca], "vpsrlw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd1, 0x0a], "vpsrlw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd1, 0x0a], "vpsrlw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd1, 0x4a, 0x01], "vpsrlw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd1, 0x4a, 0x01], "vpsrlw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xd2, 0xca], "vpsrld ymm1{k5}{z}, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xd2, 0x0a], "vpsrld ymm1{k5}{z}, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xd2, 0x4a, 0x01], "vpsrld ymm1{k5}{z}, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xd2, 0xca], "vpsrld ymm1, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xd2, 0xca], "vpsrld ymm1{k5}, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xd2, 0x0a], "vpsrld ymm1, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xd2, 0x0a], "vpsrld ymm1{k5}, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xd2, 0x4a, 0x01], "vpsrld ymm1, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xd2, 0x4a, 0x01], "vpsrld ymm1{k5}, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xd2, 0xca], "vpsrld zmm1{k5}{z}, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xd2, 0x0a], "vpsrld zmm1{k5}{z}, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xd2, 0x4a, 0x01], "vpsrld zmm1{k5}{z}, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xd2, 0xca], "vpsrld zmm1, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xd2, 0xca], "vpsrld zmm1{k5}, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xd2, 0x0a], "vpsrld zmm1, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xd2, 0x0a], "vpsrld zmm1{k5}, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xd2, 0x4a, 0x01], "vpsrld zmm1, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xd2, 0x4a, 0x01], "vpsrld zmm1{k5}, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xd2, 0xca], "vpsrld xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xd2, 0x0a], "vpsrld xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xd2, 0x4a, 0x01], "vpsrld xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xd2, 0xca], "vpsrld xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xd2, 0xca], "vpsrld xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xd2, 0x0a], "vpsrld xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xd2, 0x0a], "vpsrld xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xd2, 0x4a, 0x01], "vpsrld xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xd2, 0x4a, 0x01], "vpsrld xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd3, 0xca], "vpsrlq ymm1{k5}{z}, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd3, 0x0a], "vpsrlq ymm1{k5}{z}, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd3, 0x4a, 0x01], "vpsrlq ymm1{k5}{z}, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd3, 0xca], "vpsrlq ymm1, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd3, 0xca], "vpsrlq ymm1{k5}, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd3, 0x0a], "vpsrlq ymm1, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd3, 0x0a], "vpsrlq ymm1{k5}, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd3, 0x4a, 0x01], "vpsrlq ymm1, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd3, 0x4a, 0x01], "vpsrlq ymm1{k5}, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd3, 0xca], "vpsrlq zmm1{k5}{z}, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd3, 0x0a], "vpsrlq zmm1{k5}{z}, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd3, 0x4a, 0x01], "vpsrlq zmm1{k5}{z}, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd3, 0xca], "vpsrlq zmm1, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd3, 0xca], "vpsrlq zmm1{k5}, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd3, 0x0a], "vpsrlq zmm1, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd3, 0x0a], "vpsrlq zmm1{k5}, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd3, 0x4a, 0x01], "vpsrlq zmm1, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd3, 0x4a, 0x01], "vpsrlq zmm1{k5}, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd3, 0xca], "vpsrlq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd3, 0x0a], "vpsrlq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd3, 0x4a, 0x01], "vpsrlq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd3, 0xca], "vpsrlq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd3, 0xca], "vpsrlq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd3, 0x0a], "vpsrlq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd3, 0x0a], "vpsrlq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd3, 0x4a, 0x01], "vpsrlq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd3, 0x4a, 0x01], "vpsrlq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xd4, 0x0a], "vpaddq ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xd4, 0x4a, 0x01], "vpaddq ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xd4, 0x0a], "vpaddq ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xd4, 0x0a], "vpaddq ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xd4, 0x4a, 0x01], "vpaddq ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xd4, 0x4a, 0x01], "vpaddq ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd4, 0xca], "vpaddq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd4, 0x0a], "vpaddq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd4, 0x4a, 0x01], "vpaddq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd4, 0xca], "vpaddq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd4, 0xca], "vpaddq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd4, 0x0a], "vpaddq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd4, 0x0a], "vpaddq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd4, 0x4a, 0x01], "vpaddq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd4, 0x4a, 0x01], "vpaddq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xd4, 0x0a], "vpaddq zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xd4, 0x4a, 0x01], "vpaddq zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xd4, 0x0a], "vpaddq zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xd4, 0x0a], "vpaddq zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xd4, 0x4a, 0x01], "vpaddq zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xd4, 0x4a, 0x01], "vpaddq zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xd4, 0x0a], "vpaddq xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xd4, 0x4a, 0x01], "vpaddq xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xd4, 0x0a], "vpaddq xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xd4, 0x0a], "vpaddq xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xd4, 0x4a, 0x01], "vpaddq xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xd4, 0x4a, 0x01], "vpaddq xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd4, 0xca], "vpaddq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd4, 0x0a], "vpaddq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd4, 0x4a, 0x01], "vpaddq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd4, 0xca], "vpaddq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd4, 0xca], "vpaddq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd4, 0x0a], "vpaddq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd4, 0x0a], "vpaddq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd4, 0x4a, 0x01], "vpaddq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd4, 0x4a, 0x01], "vpaddq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd4, 0xca], "vpaddq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd4, 0x0a], "vpaddq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd4, 0x4a, 0x01], "vpaddq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd4, 0xca], "vpaddq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd4, 0xca], "vpaddq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd4, 0x0a], "vpaddq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd4, 0x0a], "vpaddq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd4, 0x4a, 0x01], "vpaddq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd4, 0x4a, 0x01], "vpaddq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd5, 0xca], "vpmullw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd5, 0x0a], "vpmullw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd5, 0x4a, 0x01], "vpmullw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd5, 0xca], "vpmullw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd5, 0xca], "vpmullw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd5, 0x0a], "vpmullw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd5, 0x0a], "vpmullw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd5, 0x4a, 0x01], "vpmullw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd5, 0x4a, 0x01], "vpmullw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd5, 0xca], "vpmullw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd5, 0x0a], "vpmullw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd5, 0x4a, 0x01], "vpmullw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd5, 0xca], "vpmullw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd5, 0xca], "vpmullw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd5, 0x0a], "vpmullw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd5, 0x0a], "vpmullw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd5, 0x4a, 0x01], "vpmullw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd5, 0x4a, 0x01], "vpmullw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd5, 0xca], "vpmullw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd5, 0x0a], "vpmullw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd5, 0x4a, 0x01], "vpmullw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd5, 0xca], "vpmullw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd5, 0xca], "vpmullw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd5, 0x0a], "vpmullw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd5, 0x0a], "vpmullw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd5, 0x4a, 0x01], "vpmullw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd5, 0x4a, 0x01], "vpmullw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd6, 0xca], "vmovq xmm2, xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd6, 0x0a], "vmovq qword [edx], xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd6, 0x4a, 0x01], "vmovq qword [edx + 0x8], xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd8, 0xca], "vpsubusb ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd8, 0x0a], "vpsubusb ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd8, 0x4a, 0x01], "vpsubusb ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd8, 0xca], "vpsubusb ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd8, 0xca], "vpsubusb ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd8, 0x0a], "vpsubusb ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd8, 0x0a], "vpsubusb ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd8, 0x4a, 0x01], "vpsubusb ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd8, 0x4a, 0x01], "vpsubusb ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd8, 0xca], "vpsubusb zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd8, 0x0a], "vpsubusb zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd8, 0x4a, 0x01], "vpsubusb zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd8, 0xca], "vpsubusb zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd8, 0xca], "vpsubusb zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd8, 0x0a], "vpsubusb zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd8, 0x0a], "vpsubusb zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd8, 0x4a, 0x01], "vpsubusb zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd8, 0x4a, 0x01], "vpsubusb zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd8, 0xca], "vpsubusb xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd8, 0x0a], "vpsubusb xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd8, 0x4a, 0x01], "vpsubusb xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd8, 0xca], "vpsubusb xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd8, 0xca], "vpsubusb xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd8, 0x0a], "vpsubusb xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd8, 0x0a], "vpsubusb xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd8, 0x4a, 0x01], "vpsubusb xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd8, 0x4a, 0x01], "vpsubusb xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd9, 0xca], "vpsubusw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd9, 0x0a], "vpsubusw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd9, 0x4a, 0x01], "vpsubusw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd9, 0xca], "vpsubusw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd9, 0xca], "vpsubusw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd9, 0x0a], "vpsubusw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd9, 0x0a], "vpsubusw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd9, 0x4a, 0x01], "vpsubusw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd9, 0x4a, 0x01], "vpsubusw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd9, 0xca], "vpsubusw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd9, 0x0a], "vpsubusw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd9, 0x4a, 0x01], "vpsubusw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd9, 0xca], "vpsubusw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd9, 0xca], "vpsubusw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd9, 0x0a], "vpsubusw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd9, 0x0a], "vpsubusw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd9, 0x4a, 0x01], "vpsubusw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd9, 0x4a, 0x01], "vpsubusw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd9, 0xca], "vpsubusw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd9, 0x0a], "vpsubusw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd9, 0x4a, 0x01], "vpsubusw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd9, 0xca], "vpsubusw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd9, 0xca], "vpsubusw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd9, 0x0a], "vpsubusw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd9, 0x0a], "vpsubusw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd9, 0x4a, 0x01], "vpsubusw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd9, 0x4a, 0x01], "vpsubusw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xda, 0xca], "vpminub ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xda, 0x0a], "vpminub ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xda, 0x4a, 0x01], "vpminub ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xda, 0xca], "vpminub ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xda, 0xca], "vpminub ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xda, 0x0a], "vpminub ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xda, 0x0a], "vpminub ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xda, 0x4a, 0x01], "vpminub ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xda, 0x4a, 0x01], "vpminub ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xda, 0xca], "vpminub zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xda, 0x0a], "vpminub zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xda, 0x4a, 0x01], "vpminub zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xda, 0xca], "vpminub zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xda, 0xca], "vpminub zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xda, 0x0a], "vpminub zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xda, 0x0a], "vpminub zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xda, 0x4a, 0x01], "vpminub zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xda, 0x4a, 0x01], "vpminub zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xda, 0xca], "vpminub xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xda, 0x0a], "vpminub xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xda, 0x4a, 0x01], "vpminub xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xda, 0xca], "vpminub xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xda, 0xca], "vpminub xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xda, 0x0a], "vpminub xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xda, 0x0a], "vpminub xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xda, 0x4a, 0x01], "vpminub xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xda, 0x4a, 0x01], "vpminub xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xdb, 0x0a], "vpandq ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xdb, 0x4a, 0x01], "vpandq ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xdb, 0x0a], "vpandq ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xdb, 0x0a], "vpandq ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xdb, 0x4a, 0x01], "vpandq ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xdb, 0x4a, 0x01], "vpandq ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xdb, 0xca], "vpandq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xdb, 0x0a], "vpandq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xdb, 0x4a, 0x01], "vpandq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xdb, 0xca], "vpandq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xdb, 0xca], "vpandq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xdb, 0x0a], "vpandq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xdb, 0x0a], "vpandq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xdb, 0x4a, 0x01], "vpandq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xdb, 0x4a, 0x01], "vpandq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0xdb, 0x0a], "vpandd ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0xdb, 0x4a, 0x01], "vpandd ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0xdb, 0x0a], "vpandd ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0xdb, 0x0a], "vpandd ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0xdb, 0x4a, 0x01], "vpandd ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0xdb, 0x4a, 0x01], "vpandd ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xdb, 0xca], "vpandd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xdb, 0x0a], "vpandd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xdb, 0x4a, 0x01], "vpandd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xdb, 0xca], "vpandd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xdb, 0xca], "vpandd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xdb, 0x0a], "vpandd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xdb, 0x0a], "vpandd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xdb, 0x4a, 0x01], "vpandd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xdb, 0x4a, 0x01], "vpandd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xdb, 0x0a], "vpandq zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xdb, 0x4a, 0x01], "vpandq zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xdb, 0x0a], "vpandq zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xdb, 0x0a], "vpandq zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xdb, 0x4a, 0x01], "vpandq zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xdb, 0x4a, 0x01], "vpandq zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xdb, 0x0a], "vpandq xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xdb, 0x4a, 0x01], "vpandq xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xdb, 0x0a], "vpandq xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xdb, 0x0a], "vpandq xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xdb, 0x4a, 0x01], "vpandq xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xdb, 0x4a, 0x01], "vpandq xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xdb, 0xca], "vpandq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xdb, 0x0a], "vpandq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xdb, 0x4a, 0x01], "vpandq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xdb, 0xca], "vpandq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xdb, 0xca], "vpandq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xdb, 0x0a], "vpandq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xdb, 0x0a], "vpandq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xdb, 0x4a, 0x01], "vpandq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xdb, 0x4a, 0x01], "vpandq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xdb, 0xca], "vpandq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xdb, 0x0a], "vpandq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xdb, 0x4a, 0x01], "vpandq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xdb, 0xca], "vpandq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xdb, 0xca], "vpandq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xdb, 0x0a], "vpandq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xdb, 0x0a], "vpandq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xdb, 0x4a, 0x01], "vpandq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xdb, 0x4a, 0x01], "vpandq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0xdb, 0x0a], "vpandd zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0xdb, 0x4a, 0x01], "vpandd zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0xdb, 0x0a], "vpandd zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0xdb, 0x0a], "vpandd zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0xdb, 0x4a, 0x01], "vpandd zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0xdb, 0x4a, 0x01], "vpandd zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0xdb, 0x0a], "vpandd xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0xdb, 0x4a, 0x01], "vpandd xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0xdb, 0x0a], "vpandd xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0xdb, 0x0a], "vpandd xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0xdb, 0x4a, 0x01], "vpandd xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0xdb, 0x4a, 0x01], "vpandd xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xdb, 0xca], "vpandd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xdb, 0x0a], "vpandd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xdb, 0x4a, 0x01], "vpandd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xdb, 0xca], "vpandd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xdb, 0xca], "vpandd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xdb, 0x0a], "vpandd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xdb, 0x0a], "vpandd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xdb, 0x4a, 0x01], "vpandd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xdb, 0x4a, 0x01], "vpandd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xdb, 0xca], "vpandd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xdb, 0x0a], "vpandd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xdb, 0x4a, 0x01], "vpandd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xdb, 0xca], "vpandd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xdb, 0xca], "vpandd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xdb, 0x0a], "vpandd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xdb, 0x0a], "vpandd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xdb, 0x4a, 0x01], "vpandd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xdb, 0x4a, 0x01], "vpandd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xdc, 0xca], "vpaddusb ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xdc, 0x0a], "vpaddusb ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xdc, 0x4a, 0x01], "vpaddusb ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xdc, 0xca], "vpaddusb ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xdc, 0xca], "vpaddusb ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xdc, 0x0a], "vpaddusb ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xdc, 0x0a], "vpaddusb ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xdc, 0x4a, 0x01], "vpaddusb ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xdc, 0x4a, 0x01], "vpaddusb ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xdc, 0xca], "vpaddusb zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xdc, 0x0a], "vpaddusb zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xdc, 0x4a, 0x01], "vpaddusb zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xdc, 0xca], "vpaddusb zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xdc, 0xca], "vpaddusb zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xdc, 0x0a], "vpaddusb zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xdc, 0x0a], "vpaddusb zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xdc, 0x4a, 0x01], "vpaddusb zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xdc, 0x4a, 0x01], "vpaddusb zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xdc, 0xca], "vpaddusb xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xdc, 0x0a], "vpaddusb xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xdc, 0x4a, 0x01], "vpaddusb xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xdc, 0xca], "vpaddusb xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xdc, 0xca], "vpaddusb xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xdc, 0x0a], "vpaddusb xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xdc, 0x0a], "vpaddusb xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xdc, 0x4a, 0x01], "vpaddusb xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xdc, 0x4a, 0x01], "vpaddusb xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xdd, 0xca], "vpaddusw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xdd, 0x0a], "vpaddusw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xdd, 0x4a, 0x01], "vpaddusw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xdd, 0xca], "vpaddusw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xdd, 0xca], "vpaddusw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xdd, 0x0a], "vpaddusw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xdd, 0x0a], "vpaddusw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xdd, 0x4a, 0x01], "vpaddusw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xdd, 0x4a, 0x01], "vpaddusw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xdd, 0xca], "vpaddusw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xdd, 0x0a], "vpaddusw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xdd, 0x4a, 0x01], "vpaddusw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xdd, 0xca], "vpaddusw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xdd, 0xca], "vpaddusw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xdd, 0x0a], "vpaddusw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xdd, 0x0a], "vpaddusw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xdd, 0x4a, 0x01], "vpaddusw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xdd, 0x4a, 0x01], "vpaddusw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xdd, 0xca], "vpaddusw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xdd, 0x0a], "vpaddusw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xdd, 0x4a, 0x01], "vpaddusw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xdd, 0xca], "vpaddusw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xdd, 0xca], "vpaddusw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xdd, 0x0a], "vpaddusw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xdd, 0x0a], "vpaddusw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xdd, 0x4a, 0x01], "vpaddusw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xdd, 0x4a, 0x01], "vpaddusw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xde, 0xca], "vpmaxub ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xde, 0x0a], "vpmaxub ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xde, 0x4a, 0x01], "vpmaxub ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xde, 0xca], "vpmaxub ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xde, 0xca], "vpmaxub ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xde, 0x0a], "vpmaxub ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xde, 0x0a], "vpmaxub ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xde, 0x4a, 0x01], "vpmaxub ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xde, 0x4a, 0x01], "vpmaxub ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xde, 0xca], "vpmaxub zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xde, 0x0a], "vpmaxub zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xde, 0x4a, 0x01], "vpmaxub zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xde, 0xca], "vpmaxub zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xde, 0xca], "vpmaxub zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xde, 0x0a], "vpmaxub zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xde, 0x0a], "vpmaxub zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xde, 0x4a, 0x01], "vpmaxub zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xde, 0x4a, 0x01], "vpmaxub zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xde, 0xca], "vpmaxub xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xde, 0x0a], "vpmaxub xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xde, 0x4a, 0x01], "vpmaxub xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xde, 0xca], "vpmaxub xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xde, 0xca], "vpmaxub xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xde, 0x0a], "vpmaxub xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xde, 0x0a], "vpmaxub xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xde, 0x4a, 0x01], "vpmaxub xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xde, 0x4a, 0x01], "vpmaxub xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xdf, 0x0a], "vpandnq ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xdf, 0x4a, 0x01], "vpandnq ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xdf, 0x0a], "vpandnq ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xdf, 0x0a], "vpandnq ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xdf, 0x4a, 0x01], "vpandnq ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xdf, 0x4a, 0x01], "vpandnq ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xdf, 0xca], "vpandnq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xdf, 0x0a], "vpandnq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xdf, 0x4a, 0x01], "vpandnq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xdf, 0xca], "vpandnq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xdf, 0xca], "vpandnq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xdf, 0x0a], "vpandnq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xdf, 0x0a], "vpandnq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xdf, 0x4a, 0x01], "vpandnq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xdf, 0x4a, 0x01], "vpandnq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0xdf, 0x0a], "vpandnd ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0xdf, 0x4a, 0x01], "vpandnd ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0xdf, 0x0a], "vpandnd ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0xdf, 0x0a], "vpandnd ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0xdf, 0x4a, 0x01], "vpandnd ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0xdf, 0x4a, 0x01], "vpandnd ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xdf, 0xca], "vpandnd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xdf, 0x0a], "vpandnd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xdf, 0x4a, 0x01], "vpandnd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xdf, 0xca], "vpandnd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xdf, 0xca], "vpandnd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xdf, 0x0a], "vpandnd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xdf, 0x0a], "vpandnd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xdf, 0x4a, 0x01], "vpandnd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xdf, 0x4a, 0x01], "vpandnd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xdf, 0x0a], "vpandnq zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xdf, 0x4a, 0x01], "vpandnq zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xdf, 0x0a], "vpandnq zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xdf, 0x0a], "vpandnq zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xdf, 0x4a, 0x01], "vpandnq zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xdf, 0x4a, 0x01], "vpandnq zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xdf, 0x0a], "vpandnq xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xdf, 0x4a, 0x01], "vpandnq xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xdf, 0x0a], "vpandnq xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xdf, 0x0a], "vpandnq xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xdf, 0x4a, 0x01], "vpandnq xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xdf, 0x4a, 0x01], "vpandnq xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xdf, 0xca], "vpandnq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xdf, 0x0a], "vpandnq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xdf, 0x4a, 0x01], "vpandnq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xdf, 0xca], "vpandnq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xdf, 0xca], "vpandnq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xdf, 0x0a], "vpandnq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xdf, 0x0a], "vpandnq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xdf, 0x4a, 0x01], "vpandnq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xdf, 0x4a, 0x01], "vpandnq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xdf, 0xca], "vpandnq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xdf, 0x0a], "vpandnq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xdf, 0x4a, 0x01], "vpandnq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xdf, 0xca], "vpandnq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xdf, 0xca], "vpandnq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xdf, 0x0a], "vpandnq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xdf, 0x0a], "vpandnq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xdf, 0x4a, 0x01], "vpandnq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xdf, 0x4a, 0x01], "vpandnq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0xdf, 0x0a], "vpandnd zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0xdf, 0x4a, 0x01], "vpandnd zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0xdf, 0x0a], "vpandnd zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0xdf, 0x0a], "vpandnd zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0xdf, 0x4a, 0x01], "vpandnd zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0xdf, 0x4a, 0x01], "vpandnd zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0xdf, 0x0a], "vpandnd xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0xdf, 0x4a, 0x01], "vpandnd xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0xdf, 0x0a], "vpandnd xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0xdf, 0x0a], "vpandnd xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0xdf, 0x4a, 0x01], "vpandnd xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0xdf, 0x4a, 0x01], "vpandnd xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xdf, 0xca], "vpandnd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xdf, 0x0a], "vpandnd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xdf, 0x4a, 0x01], "vpandnd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xdf, 0xca], "vpandnd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xdf, 0xca], "vpandnd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xdf, 0x0a], "vpandnd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xdf, 0x0a], "vpandnd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xdf, 0x4a, 0x01], "vpandnd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xdf, 0x4a, 0x01], "vpandnd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xdf, 0xca], "vpandnd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xdf, 0x0a], "vpandnd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xdf, 0x4a, 0x01], "vpandnd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xdf, 0xca], "vpandnd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xdf, 0xca], "vpandnd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xdf, 0x0a], "vpandnd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xdf, 0x0a], "vpandnd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xdf, 0x4a, 0x01], "vpandnd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xdf, 0x4a, 0x01], "vpandnd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe0, 0xca], "vpavgb ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe0, 0x0a], "vpavgb ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe0, 0x4a, 0x01], "vpavgb ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe0, 0xca], "vpavgb ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe0, 0xca], "vpavgb ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe0, 0x0a], "vpavgb ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe0, 0x0a], "vpavgb ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe0, 0x4a, 0x01], "vpavgb ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe0, 0x4a, 0x01], "vpavgb ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe0, 0xca], "vpavgb zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe0, 0x0a], "vpavgb zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe0, 0x4a, 0x01], "vpavgb zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe0, 0xca], "vpavgb zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe0, 0xca], "vpavgb zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe0, 0x0a], "vpavgb zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe0, 0x0a], "vpavgb zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe0, 0x4a, 0x01], "vpavgb zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe0, 0x4a, 0x01], "vpavgb zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe0, 0xca], "vpavgb xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe0, 0x0a], "vpavgb xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe0, 0x4a, 0x01], "vpavgb xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe0, 0xca], "vpavgb xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe0, 0xca], "vpavgb xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe0, 0x0a], "vpavgb xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe0, 0x0a], "vpavgb xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe0, 0x4a, 0x01], "vpavgb xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe0, 0x4a, 0x01], "vpavgb xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe1, 0xca], "vpsraw ymm1{k5}{z}, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe1, 0x0a], "vpsraw ymm1{k5}{z}, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe1, 0x4a, 0x01], "vpsraw ymm1{k5}{z}, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe1, 0xca], "vpsraw ymm1, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe1, 0xca], "vpsraw ymm1{k5}, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe1, 0x0a], "vpsraw ymm1, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe1, 0x0a], "vpsraw ymm1{k5}, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe1, 0x4a, 0x01], "vpsraw ymm1, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe1, 0x4a, 0x01], "vpsraw ymm1{k5}, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe1, 0xca], "vpsraw zmm1{k5}{z}, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe1, 0x0a], "vpsraw zmm1{k5}{z}, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe1, 0x4a, 0x01], "vpsraw zmm1{k5}{z}, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe1, 0xca], "vpsraw zmm1, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe1, 0xca], "vpsraw zmm1{k5}, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe1, 0x0a], "vpsraw zmm1, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe1, 0x0a], "vpsraw zmm1{k5}, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe1, 0x4a, 0x01], "vpsraw zmm1, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe1, 0x4a, 0x01], "vpsraw zmm1{k5}, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe1, 0xca], "vpsraw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe1, 0x0a], "vpsraw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe1, 0x4a, 0x01], "vpsraw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe1, 0xca], "vpsraw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe1, 0xca], "vpsraw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe1, 0x0a], "vpsraw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe1, 0x0a], "vpsraw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe1, 0x4a, 0x01], "vpsraw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe1, 0x4a, 0x01], "vpsraw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe2, 0xca], "vpsraq ymm1{k5}{z}, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe2, 0x0a], "vpsraq ymm1{k5}{z}, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe2, 0x4a, 0x01], "vpsraq ymm1{k5}{z}, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe2, 0xca], "vpsraq ymm1, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe2, 0xca], "vpsraq ymm1{k5}, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe2, 0x0a], "vpsraq ymm1, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe2, 0x0a], "vpsraq ymm1{k5}, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe2, 0x4a, 0x01], "vpsraq ymm1, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe2, 0x4a, 0x01], "vpsraq ymm1{k5}, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xe2, 0xca], "vpsrad ymm1{k5}{z}, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xe2, 0x0a], "vpsrad ymm1{k5}{z}, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xe2, 0x4a, 0x01], "vpsrad ymm1{k5}{z}, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xe2, 0xca], "vpsrad ymm1, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xe2, 0xca], "vpsrad ymm1{k5}, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xe2, 0x0a], "vpsrad ymm1, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xe2, 0x0a], "vpsrad ymm1{k5}, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xe2, 0x4a, 0x01], "vpsrad ymm1, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xe2, 0x4a, 0x01], "vpsrad ymm1{k5}, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe2, 0xca], "vpsraq zmm1{k5}{z}, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe2, 0x0a], "vpsraq zmm1{k5}{z}, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe2, 0x4a, 0x01], "vpsraq zmm1{k5}{z}, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe2, 0xca], "vpsraq zmm1, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe2, 0xca], "vpsraq zmm1{k5}, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe2, 0x0a], "vpsraq zmm1, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe2, 0x0a], "vpsraq zmm1{k5}, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe2, 0x4a, 0x01], "vpsraq zmm1, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe2, 0x4a, 0x01], "vpsraq zmm1{k5}, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe2, 0xca], "vpsraq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe2, 0x0a], "vpsraq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe2, 0x4a, 0x01], "vpsraq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe2, 0xca], "vpsraq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe2, 0xca], "vpsraq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe2, 0x0a], "vpsraq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe2, 0x0a], "vpsraq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe2, 0x4a, 0x01], "vpsraq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe2, 0x4a, 0x01], "vpsraq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xe2, 0xca], "vpsrad zmm1{k5}{z}, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xe2, 0x0a], "vpsrad zmm1{k5}{z}, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xe2, 0x4a, 0x01], "vpsrad zmm1{k5}{z}, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xe2, 0xca], "vpsrad zmm1, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xe2, 0xca], "vpsrad zmm1{k5}, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xe2, 0x0a], "vpsrad zmm1, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xe2, 0x0a], "vpsrad zmm1{k5}, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xe2, 0x4a, 0x01], "vpsrad zmm1, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xe2, 0x4a, 0x01], "vpsrad zmm1{k5}, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xe2, 0xca], "vpsrad xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xe2, 0x0a], "vpsrad xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xe2, 0x4a, 0x01], "vpsrad xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xe2, 0xca], "vpsrad xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xe2, 0xca], "vpsrad xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xe2, 0x0a], "vpsrad xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xe2, 0x0a], "vpsrad xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xe2, 0x4a, 0x01], "vpsrad xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xe2, 0x4a, 0x01], "vpsrad xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe3, 0xca], "vpavgw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe3, 0x0a], "vpavgw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe3, 0x4a, 0x01], "vpavgw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe3, 0xca], "vpavgw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe3, 0xca], "vpavgw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe3, 0x0a], "vpavgw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe3, 0x0a], "vpavgw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe3, 0x4a, 0x01], "vpavgw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe3, 0x4a, 0x01], "vpavgw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe3, 0xca], "vpavgw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe3, 0x0a], "vpavgw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe3, 0x4a, 0x01], "vpavgw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe3, 0xca], "vpavgw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe3, 0xca], "vpavgw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe3, 0x0a], "vpavgw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe3, 0x0a], "vpavgw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe3, 0x4a, 0x01], "vpavgw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe3, 0x4a, 0x01], "vpavgw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe3, 0xca], "vpavgw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe3, 0x0a], "vpavgw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe3, 0x4a, 0x01], "vpavgw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe3, 0xca], "vpavgw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe3, 0xca], "vpavgw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe3, 0x0a], "vpavgw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe3, 0x0a], "vpavgw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe3, 0x4a, 0x01], "vpavgw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe3, 0x4a, 0x01], "vpavgw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe4, 0xca], "vpmulhuw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe4, 0x0a], "vpmulhuw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe4, 0x4a, 0x01], "vpmulhuw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe4, 0xca], "vpmulhuw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe4, 0xca], "vpmulhuw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe4, 0x0a], "vpmulhuw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe4, 0x0a], "vpmulhuw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe4, 0x4a, 0x01], "vpmulhuw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe4, 0x4a, 0x01], "vpmulhuw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe4, 0xca], "vpmulhuw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe4, 0x0a], "vpmulhuw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe4, 0x4a, 0x01], "vpmulhuw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe4, 0xca], "vpmulhuw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe4, 0xca], "vpmulhuw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe4, 0x0a], "vpmulhuw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe4, 0x0a], "vpmulhuw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe4, 0x4a, 0x01], "vpmulhuw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe4, 0x4a, 0x01], "vpmulhuw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe4, 0xca], "vpmulhuw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe4, 0x0a], "vpmulhuw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe4, 0x4a, 0x01], "vpmulhuw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe4, 0xca], "vpmulhuw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe4, 0xca], "vpmulhuw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe4, 0x0a], "vpmulhuw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe4, 0x0a], "vpmulhuw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe4, 0x4a, 0x01], "vpmulhuw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe4, 0x4a, 0x01], "vpmulhuw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe5, 0xca], "vpmulhw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe5, 0x0a], "vpmulhw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe5, 0x4a, 0x01], "vpmulhw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe5, 0xca], "vpmulhw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe5, 0xca], "vpmulhw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe5, 0x0a], "vpmulhw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe5, 0x0a], "vpmulhw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe5, 0x4a, 0x01], "vpmulhw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe5, 0x4a, 0x01], "vpmulhw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe5, 0xca], "vpmulhw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe5, 0x0a], "vpmulhw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe5, 0x4a, 0x01], "vpmulhw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe5, 0xca], "vpmulhw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe5, 0xca], "vpmulhw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe5, 0x0a], "vpmulhw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe5, 0x0a], "vpmulhw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe5, 0x4a, 0x01], "vpmulhw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe5, 0x4a, 0x01], "vpmulhw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe5, 0xca], "vpmulhw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe5, 0x0a], "vpmulhw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe5, 0x4a, 0x01], "vpmulhw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe5, 0xca], "vpmulhw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe5, 0xca], "vpmulhw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe5, 0x0a], "vpmulhw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe5, 0x0a], "vpmulhw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe5, 0x4a, 0x01], "vpmulhw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe5, 0x4a, 0x01], "vpmulhw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0xe6, 0xca], "vcvttpd2dq ymm1{k5}{z}{sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0xe6, 0xca], "vcvttpd2dq ymm1{sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0xe6, 0xca], "vcvttpd2dq ymm1{k5}{sae}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xe6, 0x0a], "vcvttpd2dq xmm1{k5}{z}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xe6, 0x4a, 0x01], "vcvttpd2dq xmm1{k5}{z}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xe6, 0x0a], "vcvttpd2dq xmm1, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xe6, 0x0a], "vcvttpd2dq xmm1{k5}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xe6, 0x4a, 0x01], "vcvttpd2dq xmm1, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xe6, 0x4a, 0x01], "vcvttpd2dq xmm1{k5}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe6, 0xca], "vcvttpd2dq xmm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe6, 0x0a], "vcvttpd2dq xmm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe6, 0x4a, 0x01], "vcvttpd2dq xmm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe6, 0xca], "vcvttpd2dq xmm1, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe6, 0xca], "vcvttpd2dq xmm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe6, 0x0a], "vcvttpd2dq xmm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe6, 0x0a], "vcvttpd2dq xmm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe6, 0x4a, 0x01], "vcvttpd2dq xmm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe6, 0x4a, 0x01], "vcvttpd2dq xmm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xe6, 0x0a], "vcvttpd2dq ymm1{k5}{z}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xe6, 0x4a, 0x01], "vcvttpd2dq ymm1{k5}{z}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xe6, 0x0a], "vcvttpd2dq ymm1, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xe6, 0x0a], "vcvttpd2dq ymm1{k5}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xe6, 0x4a, 0x01], "vcvttpd2dq ymm1, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xe6, 0x4a, 0x01], "vcvttpd2dq ymm1{k5}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xe6, 0x0a], "vcvttpd2dq xmm1{k5}{z}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xe6, 0x4a, 0x01], "vcvttpd2dq xmm1{k5}{z}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xe6, 0x0a], "vcvttpd2dq xmm1, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xe6, 0x0a], "vcvttpd2dq xmm1{k5}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xe6, 0x4a, 0x01], "vcvttpd2dq xmm1, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xe6, 0x4a, 0x01], "vcvttpd2dq xmm1{k5}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe6, 0xca], "vcvttpd2dq ymm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe6, 0x0a], "vcvttpd2dq ymm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe6, 0x4a, 0x01], "vcvttpd2dq ymm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe6, 0xca], "vcvttpd2dq ymm1, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe6, 0xca], "vcvttpd2dq ymm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe6, 0x0a], "vcvttpd2dq ymm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe6, 0x0a], "vcvttpd2dq ymm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe6, 0x4a, 0x01], "vcvttpd2dq ymm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe6, 0x4a, 0x01], "vcvttpd2dq ymm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe6, 0xca], "vcvttpd2dq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe6, 0x0a], "vcvttpd2dq xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe6, 0x4a, 0x01], "vcvttpd2dq xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe6, 0xca], "vcvttpd2dq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe6, 0xca], "vcvttpd2dq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe6, 0x0a], "vcvttpd2dq xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe6, 0x0a], "vcvttpd2dq xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe6, 0x4a, 0x01], "vcvttpd2dq xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe6, 0x4a, 0x01], "vcvttpd2dq xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xe7, 0x0a], "vmovntdq ymmword [edx], ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xe7, 0x4a, 0x01], "vmovntdq ymmword [edx + 0x20], ymm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xe7, 0x0a], "vmovntdq zmmword [edx], zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xe7, 0x4a, 0x01], "vmovntdq zmmword [edx + 0x40], zmm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xe7, 0x0a], "vmovntdq xmmword [edx], xmm1");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xe7, 0x4a, 0x01], "vmovntdq xmmword [edx + 0x10], xmm1");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe8, 0xca], "vpsubsb ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe8, 0x0a], "vpsubsb ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe8, 0x4a, 0x01], "vpsubsb ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe8, 0xca], "vpsubsb ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe8, 0xca], "vpsubsb ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe8, 0x0a], "vpsubsb ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe8, 0x0a], "vpsubsb ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe8, 0x4a, 0x01], "vpsubsb ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe8, 0x4a, 0x01], "vpsubsb ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe8, 0xca], "vpsubsb zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe8, 0x0a], "vpsubsb zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe8, 0x4a, 0x01], "vpsubsb zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe8, 0xca], "vpsubsb zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe8, 0xca], "vpsubsb zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe8, 0x0a], "vpsubsb zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe8, 0x0a], "vpsubsb zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe8, 0x4a, 0x01], "vpsubsb zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe8, 0x4a, 0x01], "vpsubsb zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe8, 0xca], "vpsubsb xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe8, 0x0a], "vpsubsb xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe8, 0x4a, 0x01], "vpsubsb xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe8, 0xca], "vpsubsb xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe8, 0xca], "vpsubsb xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe8, 0x0a], "vpsubsb xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe8, 0x0a], "vpsubsb xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe8, 0x4a, 0x01], "vpsubsb xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe8, 0x4a, 0x01], "vpsubsb xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe9, 0xca], "vpsubsw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe9, 0x0a], "vpsubsw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe9, 0x4a, 0x01], "vpsubsw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe9, 0xca], "vpsubsw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe9, 0xca], "vpsubsw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe9, 0x0a], "vpsubsw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe9, 0x0a], "vpsubsw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe9, 0x4a, 0x01], "vpsubsw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe9, 0x4a, 0x01], "vpsubsw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe9, 0xca], "vpsubsw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe9, 0x0a], "vpsubsw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe9, 0x4a, 0x01], "vpsubsw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe9, 0xca], "vpsubsw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe9, 0xca], "vpsubsw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe9, 0x0a], "vpsubsw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe9, 0x0a], "vpsubsw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe9, 0x4a, 0x01], "vpsubsw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe9, 0x4a, 0x01], "vpsubsw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe9, 0xca], "vpsubsw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe9, 0x0a], "vpsubsw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe9, 0x4a, 0x01], "vpsubsw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe9, 0xca], "vpsubsw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe9, 0xca], "vpsubsw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe9, 0x0a], "vpsubsw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe9, 0x0a], "vpsubsw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe9, 0x4a, 0x01], "vpsubsw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe9, 0x4a, 0x01], "vpsubsw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xea, 0xca], "vpminsw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xea, 0x0a], "vpminsw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xea, 0x4a, 0x01], "vpminsw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xea, 0xca], "vpminsw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xea, 0xca], "vpminsw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xea, 0x0a], "vpminsw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xea, 0x0a], "vpminsw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xea, 0x4a, 0x01], "vpminsw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xea, 0x4a, 0x01], "vpminsw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xea, 0xca], "vpminsw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xea, 0x0a], "vpminsw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xea, 0x4a, 0x01], "vpminsw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xea, 0xca], "vpminsw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xea, 0xca], "vpminsw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xea, 0x0a], "vpminsw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xea, 0x0a], "vpminsw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xea, 0x4a, 0x01], "vpminsw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xea, 0x4a, 0x01], "vpminsw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xea, 0xca], "vpminsw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xea, 0x0a], "vpminsw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xea, 0x4a, 0x01], "vpminsw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xea, 0xca], "vpminsw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xea, 0xca], "vpminsw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xea, 0x0a], "vpminsw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xea, 0x0a], "vpminsw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xea, 0x4a, 0x01], "vpminsw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xea, 0x4a, 0x01], "vpminsw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xeb, 0x0a], "vporq ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xeb, 0x4a, 0x01], "vporq ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xeb, 0x0a], "vporq ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xeb, 0x0a], "vporq ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xeb, 0x4a, 0x01], "vporq ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xeb, 0x4a, 0x01], "vporq ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xeb, 0xca], "vporq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xeb, 0x0a], "vporq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xeb, 0x4a, 0x01], "vporq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xeb, 0xca], "vporq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xeb, 0xca], "vporq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xeb, 0x0a], "vporq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xeb, 0x0a], "vporq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xeb, 0x4a, 0x01], "vporq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xeb, 0x4a, 0x01], "vporq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0xeb, 0x0a], "vpord ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0xeb, 0x4a, 0x01], "vpord ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0xeb, 0x0a], "vpord ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0xeb, 0x0a], "vpord ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0xeb, 0x4a, 0x01], "vpord ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0xeb, 0x4a, 0x01], "vpord ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xeb, 0xca], "vpord ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xeb, 0x0a], "vpord ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xeb, 0x4a, 0x01], "vpord ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xeb, 0xca], "vpord ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xeb, 0xca], "vpord ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xeb, 0x0a], "vpord ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xeb, 0x0a], "vpord ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xeb, 0x4a, 0x01], "vpord ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xeb, 0x4a, 0x01], "vpord ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xeb, 0x0a], "vporq zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xeb, 0x4a, 0x01], "vporq zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xeb, 0x0a], "vporq zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xeb, 0x0a], "vporq zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xeb, 0x4a, 0x01], "vporq zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xeb, 0x4a, 0x01], "vporq zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xeb, 0x0a], "vporq xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xeb, 0x4a, 0x01], "vporq xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xeb, 0x0a], "vporq xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xeb, 0x0a], "vporq xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xeb, 0x4a, 0x01], "vporq xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xeb, 0x4a, 0x01], "vporq xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xeb, 0xca], "vporq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xeb, 0x0a], "vporq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xeb, 0x4a, 0x01], "vporq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xeb, 0xca], "vporq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xeb, 0xca], "vporq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xeb, 0x0a], "vporq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xeb, 0x0a], "vporq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xeb, 0x4a, 0x01], "vporq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xeb, 0x4a, 0x01], "vporq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xeb, 0xca], "vporq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xeb, 0x0a], "vporq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xeb, 0x4a, 0x01], "vporq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xeb, 0xca], "vporq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xeb, 0xca], "vporq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xeb, 0x0a], "vporq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xeb, 0x0a], "vporq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xeb, 0x4a, 0x01], "vporq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xeb, 0x4a, 0x01], "vporq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0xeb, 0x0a], "vpord zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0xeb, 0x4a, 0x01], "vpord zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0xeb, 0x0a], "vpord zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0xeb, 0x0a], "vpord zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0xeb, 0x4a, 0x01], "vpord zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0xeb, 0x4a, 0x01], "vpord zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0xeb, 0x0a], "vpord xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0xeb, 0x4a, 0x01], "vpord xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0xeb, 0x0a], "vpord xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0xeb, 0x0a], "vpord xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0xeb, 0x4a, 0x01], "vpord xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0xeb, 0x4a, 0x01], "vpord xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xeb, 0xca], "vpord zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xeb, 0x0a], "vpord zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xeb, 0x4a, 0x01], "vpord zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xeb, 0xca], "vpord zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xeb, 0xca], "vpord zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xeb, 0x0a], "vpord zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xeb, 0x0a], "vpord zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xeb, 0x4a, 0x01], "vpord zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xeb, 0x4a, 0x01], "vpord zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xeb, 0xca], "vpord xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xeb, 0x0a], "vpord xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xeb, 0x4a, 0x01], "vpord xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xeb, 0xca], "vpord xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xeb, 0xca], "vpord xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xeb, 0x0a], "vpord xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xeb, 0x0a], "vpord xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xeb, 0x4a, 0x01], "vpord xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xeb, 0x4a, 0x01], "vpord xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xec, 0xca], "vpaddsb ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xec, 0x0a], "vpaddsb ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xec, 0x4a, 0x01], "vpaddsb ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xec, 0xca], "vpaddsb ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xec, 0xca], "vpaddsb ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xec, 0x0a], "vpaddsb ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xec, 0x0a], "vpaddsb ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xec, 0x4a, 0x01], "vpaddsb ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xec, 0x4a, 0x01], "vpaddsb ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xec, 0xca], "vpaddsb zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xec, 0x0a], "vpaddsb zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xec, 0x4a, 0x01], "vpaddsb zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xec, 0xca], "vpaddsb zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xec, 0xca], "vpaddsb zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xec, 0x0a], "vpaddsb zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xec, 0x0a], "vpaddsb zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xec, 0x4a, 0x01], "vpaddsb zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xec, 0x4a, 0x01], "vpaddsb zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xec, 0xca], "vpaddsb xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xec, 0x0a], "vpaddsb xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xec, 0x4a, 0x01], "vpaddsb xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xec, 0xca], "vpaddsb xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xec, 0xca], "vpaddsb xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xec, 0x0a], "vpaddsb xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xec, 0x0a], "vpaddsb xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xec, 0x4a, 0x01], "vpaddsb xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xec, 0x4a, 0x01], "vpaddsb xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xed, 0xca], "vpaddsw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xed, 0x0a], "vpaddsw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xed, 0x4a, 0x01], "vpaddsw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xed, 0xca], "vpaddsw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xed, 0xca], "vpaddsw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xed, 0x0a], "vpaddsw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xed, 0x0a], "vpaddsw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xed, 0x4a, 0x01], "vpaddsw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xed, 0x4a, 0x01], "vpaddsw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xed, 0xca], "vpaddsw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xed, 0x0a], "vpaddsw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xed, 0x4a, 0x01], "vpaddsw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xed, 0xca], "vpaddsw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xed, 0xca], "vpaddsw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xed, 0x0a], "vpaddsw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xed, 0x0a], "vpaddsw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xed, 0x4a, 0x01], "vpaddsw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xed, 0x4a, 0x01], "vpaddsw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xed, 0xca], "vpaddsw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xed, 0x0a], "vpaddsw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xed, 0x4a, 0x01], "vpaddsw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xed, 0xca], "vpaddsw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xed, 0xca], "vpaddsw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xed, 0x0a], "vpaddsw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xed, 0x0a], "vpaddsw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xed, 0x4a, 0x01], "vpaddsw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xed, 0x4a, 0x01], "vpaddsw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xee, 0xca], "vpmaxsw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xee, 0x0a], "vpmaxsw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xee, 0x4a, 0x01], "vpmaxsw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xee, 0xca], "vpmaxsw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xee, 0xca], "vpmaxsw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xee, 0x0a], "vpmaxsw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xee, 0x0a], "vpmaxsw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xee, 0x4a, 0x01], "vpmaxsw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xee, 0x4a, 0x01], "vpmaxsw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xee, 0xca], "vpmaxsw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xee, 0x0a], "vpmaxsw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xee, 0x4a, 0x01], "vpmaxsw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xee, 0xca], "vpmaxsw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xee, 0xca], "vpmaxsw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xee, 0x0a], "vpmaxsw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xee, 0x0a], "vpmaxsw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xee, 0x4a, 0x01], "vpmaxsw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xee, 0x4a, 0x01], "vpmaxsw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xee, 0xca], "vpmaxsw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xee, 0x0a], "vpmaxsw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xee, 0x4a, 0x01], "vpmaxsw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xee, 0xca], "vpmaxsw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xee, 0xca], "vpmaxsw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xee, 0x0a], "vpmaxsw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xee, 0x0a], "vpmaxsw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xee, 0x4a, 0x01], "vpmaxsw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xee, 0x4a, 0x01], "vpmaxsw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xef, 0x0a], "vpxorq ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xef, 0x4a, 0x01], "vpxorq ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xef, 0x0a], "vpxorq ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xef, 0x0a], "vpxorq ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xef, 0x4a, 0x01], "vpxorq ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xef, 0x4a, 0x01], "vpxorq ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xef, 0xca], "vpxorq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xef, 0x0a], "vpxorq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xef, 0x4a, 0x01], "vpxorq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xef, 0xca], "vpxorq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xef, 0xca], "vpxorq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xef, 0x0a], "vpxorq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xef, 0x0a], "vpxorq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xef, 0x4a, 0x01], "vpxorq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xef, 0x4a, 0x01], "vpxorq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0xef, 0x0a], "vpxord ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0xef, 0x4a, 0x01], "vpxord ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0xef, 0x0a], "vpxord ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0xef, 0x0a], "vpxord ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0xef, 0x4a, 0x01], "vpxord ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0xef, 0x4a, 0x01], "vpxord ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xef, 0xca], "vpxord ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xef, 0x0a], "vpxord ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xef, 0x4a, 0x01], "vpxord ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xef, 0xca], "vpxord ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xef, 0xca], "vpxord ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xef, 0x0a], "vpxord ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xef, 0x0a], "vpxord ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xef, 0x4a, 0x01], "vpxord ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xef, 0x4a, 0x01], "vpxord ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xef, 0x0a], "vpxorq zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xef, 0x4a, 0x01], "vpxorq zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xef, 0x0a], "vpxorq zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xef, 0x0a], "vpxorq zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xef, 0x4a, 0x01], "vpxorq zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xef, 0x4a, 0x01], "vpxorq zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xef, 0x0a], "vpxorq xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xef, 0x4a, 0x01], "vpxorq xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xef, 0x0a], "vpxorq xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xef, 0x0a], "vpxorq xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xef, 0x4a, 0x01], "vpxorq xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xef, 0x4a, 0x01], "vpxorq xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xef, 0xca], "vpxorq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xef, 0x0a], "vpxorq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xef, 0x4a, 0x01], "vpxorq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xef, 0xca], "vpxorq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xef, 0xca], "vpxorq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xef, 0x0a], "vpxorq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xef, 0x0a], "vpxorq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xef, 0x4a, 0x01], "vpxorq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xef, 0x4a, 0x01], "vpxorq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xef, 0xca], "vpxorq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xef, 0x0a], "vpxorq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xef, 0x4a, 0x01], "vpxorq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xef, 0xca], "vpxorq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xef, 0xca], "vpxorq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xef, 0x0a], "vpxorq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xef, 0x0a], "vpxorq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xef, 0x4a, 0x01], "vpxorq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xef, 0x4a, 0x01], "vpxorq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0xef, 0x0a], "vpxord zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0xef, 0x4a, 0x01], "vpxord zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0xef, 0x0a], "vpxord zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0xef, 0x0a], "vpxord zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0xef, 0x4a, 0x01], "vpxord zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0xef, 0x4a, 0x01], "vpxord zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0xef, 0x0a], "vpxord xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0xef, 0x4a, 0x01], "vpxord xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0xef, 0x0a], "vpxord xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0xef, 0x0a], "vpxord xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0xef, 0x4a, 0x01], "vpxord xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0xef, 0x4a, 0x01], "vpxord xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xef, 0xca], "vpxord zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xef, 0x0a], "vpxord zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xef, 0x4a, 0x01], "vpxord zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xef, 0xca], "vpxord zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xef, 0xca], "vpxord zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xef, 0x0a], "vpxord zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xef, 0x0a], "vpxord zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xef, 0x4a, 0x01], "vpxord zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xef, 0x4a, 0x01], "vpxord zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xef, 0xca], "vpxord xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xef, 0x0a], "vpxord xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xef, 0x4a, 0x01], "vpxord xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xef, 0xca], "vpxord xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xef, 0xca], "vpxord xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xef, 0x0a], "vpxord xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xef, 0x0a], "vpxord xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xef, 0x4a, 0x01], "vpxord xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xef, 0x4a, 0x01], "vpxord xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf1, 0xca], "vpsllw ymm1{k5}{z}, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf1, 0x0a], "vpsllw ymm1{k5}{z}, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf1, 0x4a, 0x01], "vpsllw ymm1{k5}{z}, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf1, 0xca], "vpsllw ymm1, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf1, 0xca], "vpsllw ymm1{k5}, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf1, 0x0a], "vpsllw ymm1, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf1, 0x0a], "vpsllw ymm1{k5}, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf1, 0x4a, 0x01], "vpsllw ymm1, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf1, 0x4a, 0x01], "vpsllw ymm1{k5}, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf1, 0xca], "vpsllw zmm1{k5}{z}, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf1, 0x0a], "vpsllw zmm1{k5}{z}, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf1, 0x4a, 0x01], "vpsllw zmm1{k5}{z}, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf1, 0xca], "vpsllw zmm1, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf1, 0xca], "vpsllw zmm1{k5}, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf1, 0x0a], "vpsllw zmm1, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf1, 0x0a], "vpsllw zmm1{k5}, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf1, 0x4a, 0x01], "vpsllw zmm1, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf1, 0x4a, 0x01], "vpsllw zmm1{k5}, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf1, 0xca], "vpsllw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf1, 0x0a], "vpsllw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf1, 0x4a, 0x01], "vpsllw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf1, 0xca], "vpsllw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf1, 0xca], "vpsllw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf1, 0x0a], "vpsllw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf1, 0x0a], "vpsllw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf1, 0x4a, 0x01], "vpsllw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf1, 0x4a, 0x01], "vpsllw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xf2, 0xca], "vpslld ymm1{k5}{z}, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xf2, 0x0a], "vpslld ymm1{k5}{z}, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xf2, 0x4a, 0x01], "vpslld ymm1{k5}{z}, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xf2, 0xca], "vpslld ymm1, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xf2, 0xca], "vpslld ymm1{k5}, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xf2, 0x0a], "vpslld ymm1, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xf2, 0x0a], "vpslld ymm1{k5}, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xf2, 0x4a, 0x01], "vpslld ymm1, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xf2, 0x4a, 0x01], "vpslld ymm1{k5}, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xf2, 0xca], "vpslld zmm1{k5}{z}, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xf2, 0x0a], "vpslld zmm1{k5}{z}, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xf2, 0x4a, 0x01], "vpslld zmm1{k5}{z}, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xf2, 0xca], "vpslld zmm1, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xf2, 0xca], "vpslld zmm1{k5}, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xf2, 0x0a], "vpslld zmm1, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xf2, 0x0a], "vpslld zmm1{k5}, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xf2, 0x4a, 0x01], "vpslld zmm1, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xf2, 0x4a, 0x01], "vpslld zmm1{k5}, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xf2, 0xca], "vpslld xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xf2, 0x0a], "vpslld xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xf2, 0x4a, 0x01], "vpslld xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xf2, 0xca], "vpslld xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xf2, 0xca], "vpslld xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xf2, 0x0a], "vpslld xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xf2, 0x0a], "vpslld xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xf2, 0x4a, 0x01], "vpslld xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xf2, 0x4a, 0x01], "vpslld xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf3, 0xca], "vpsllq ymm1{k5}{z}, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf3, 0x0a], "vpsllq ymm1{k5}{z}, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf3, 0x4a, 0x01], "vpsllq ymm1{k5}{z}, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf3, 0xca], "vpsllq ymm1, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf3, 0xca], "vpsllq ymm1{k5}, ymm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf3, 0x0a], "vpsllq ymm1, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf3, 0x0a], "vpsllq ymm1{k5}, ymm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf3, 0x4a, 0x01], "vpsllq ymm1, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf3, 0x4a, 0x01], "vpsllq ymm1{k5}, ymm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf3, 0xca], "vpsllq zmm1{k5}{z}, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf3, 0x0a], "vpsllq zmm1{k5}{z}, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf3, 0x4a, 0x01], "vpsllq zmm1{k5}{z}, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf3, 0xca], "vpsllq zmm1, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf3, 0xca], "vpsllq zmm1{k5}, zmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf3, 0x0a], "vpsllq zmm1, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf3, 0x0a], "vpsllq zmm1{k5}, zmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf3, 0x4a, 0x01], "vpsllq zmm1, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf3, 0x4a, 0x01], "vpsllq zmm1{k5}, zmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf3, 0xca], "vpsllq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf3, 0x0a], "vpsllq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf3, 0x4a, 0x01], "vpsllq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf3, 0xca], "vpsllq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf3, 0xca], "vpsllq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf3, 0x0a], "vpsllq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf3, 0x0a], "vpsllq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf3, 0x4a, 0x01], "vpsllq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf3, 0x4a, 0x01], "vpsllq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xf4, 0x0a], "vpmuludq ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xf4, 0x4a, 0x01], "vpmuludq ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xf4, 0x0a], "vpmuludq ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xf4, 0x0a], "vpmuludq ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xf4, 0x4a, 0x01], "vpmuludq ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xf4, 0x4a, 0x01], "vpmuludq ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf4, 0xca], "vpmuludq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf4, 0x0a], "vpmuludq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf4, 0x4a, 0x01], "vpmuludq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf4, 0xca], "vpmuludq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf4, 0xca], "vpmuludq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf4, 0x0a], "vpmuludq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf4, 0x0a], "vpmuludq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf4, 0x4a, 0x01], "vpmuludq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf4, 0x4a, 0x01], "vpmuludq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xf4, 0x0a], "vpmuludq zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xf4, 0x4a, 0x01], "vpmuludq zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xf4, 0x0a], "vpmuludq zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xf4, 0x0a], "vpmuludq zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xf4, 0x4a, 0x01], "vpmuludq zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xf4, 0x4a, 0x01], "vpmuludq zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xf4, 0x0a], "vpmuludq xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xf4, 0x4a, 0x01], "vpmuludq xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xf4, 0x0a], "vpmuludq xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xf4, 0x0a], "vpmuludq xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xf4, 0x4a, 0x01], "vpmuludq xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xf4, 0x4a, 0x01], "vpmuludq xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf4, 0xca], "vpmuludq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf4, 0x0a], "vpmuludq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf4, 0x4a, 0x01], "vpmuludq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf4, 0xca], "vpmuludq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf4, 0xca], "vpmuludq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf4, 0x0a], "vpmuludq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf4, 0x0a], "vpmuludq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf4, 0x4a, 0x01], "vpmuludq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf4, 0x4a, 0x01], "vpmuludq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf4, 0xca], "vpmuludq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf4, 0x0a], "vpmuludq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf4, 0x4a, 0x01], "vpmuludq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf4, 0xca], "vpmuludq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf4, 0xca], "vpmuludq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf4, 0x0a], "vpmuludq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf4, 0x0a], "vpmuludq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf4, 0x4a, 0x01], "vpmuludq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf4, 0x4a, 0x01], "vpmuludq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf5, 0xca], "vpmaddwd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf5, 0x0a], "vpmaddwd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf5, 0x4a, 0x01], "vpmaddwd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf5, 0xca], "vpmaddwd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf5, 0xca], "vpmaddwd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf5, 0x0a], "vpmaddwd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf5, 0x0a], "vpmaddwd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf5, 0x4a, 0x01], "vpmaddwd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf5, 0x4a, 0x01], "vpmaddwd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf5, 0xca], "vpmaddwd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf5, 0x0a], "vpmaddwd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf5, 0x4a, 0x01], "vpmaddwd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf5, 0xca], "vpmaddwd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf5, 0xca], "vpmaddwd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf5, 0x0a], "vpmaddwd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf5, 0x0a], "vpmaddwd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf5, 0x4a, 0x01], "vpmaddwd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf5, 0x4a, 0x01], "vpmaddwd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf5, 0xca], "vpmaddwd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf5, 0x0a], "vpmaddwd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf5, 0x4a, 0x01], "vpmaddwd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf5, 0xca], "vpmaddwd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf5, 0xca], "vpmaddwd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf5, 0x0a], "vpmaddwd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf5, 0x0a], "vpmaddwd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf5, 0x4a, 0x01], "vpmaddwd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf5, 0x4a, 0x01], "vpmaddwd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf6, 0xca], "vpsadbw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf6, 0x0a], "vpsadbw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf6, 0x4a, 0x01], "vpsadbw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf6, 0xca], "vpsadbw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf6, 0x0a], "vpsadbw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf6, 0x4a, 0x01], "vpsadbw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf6, 0xca], "vpsadbw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf6, 0x0a], "vpsadbw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf6, 0x4a, 0x01], "vpsadbw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf8, 0xca], "vpsubb ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf8, 0x0a], "vpsubb ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf8, 0x4a, 0x01], "vpsubb ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf8, 0xca], "vpsubb ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf8, 0xca], "vpsubb ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf8, 0x0a], "vpsubb ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf8, 0x0a], "vpsubb ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf8, 0x4a, 0x01], "vpsubb ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf8, 0x4a, 0x01], "vpsubb ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf8, 0xca], "vpsubb zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf8, 0x0a], "vpsubb zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf8, 0x4a, 0x01], "vpsubb zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf8, 0xca], "vpsubb zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf8, 0xca], "vpsubb zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf8, 0x0a], "vpsubb zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf8, 0x0a], "vpsubb zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf8, 0x4a, 0x01], "vpsubb zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf8, 0x4a, 0x01], "vpsubb zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf8, 0xca], "vpsubb xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf8, 0x0a], "vpsubb xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf8, 0x4a, 0x01], "vpsubb xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf8, 0xca], "vpsubb xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf8, 0xca], "vpsubb xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf8, 0x0a], "vpsubb xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf8, 0x0a], "vpsubb xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf8, 0x4a, 0x01], "vpsubb xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf8, 0x4a, 0x01], "vpsubb xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf9, 0xca], "vpsubw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf9, 0x0a], "vpsubw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf9, 0x4a, 0x01], "vpsubw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf9, 0xca], "vpsubw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf9, 0xca], "vpsubw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf9, 0x0a], "vpsubw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf9, 0x0a], "vpsubw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf9, 0x4a, 0x01], "vpsubw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf9, 0x4a, 0x01], "vpsubw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf9, 0xca], "vpsubw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf9, 0x0a], "vpsubw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf9, 0x4a, 0x01], "vpsubw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf9, 0xca], "vpsubw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf9, 0xca], "vpsubw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf9, 0x0a], "vpsubw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf9, 0x0a], "vpsubw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf9, 0x4a, 0x01], "vpsubw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf9, 0x4a, 0x01], "vpsubw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf9, 0xca], "vpsubw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf9, 0x0a], "vpsubw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf9, 0x4a, 0x01], "vpsubw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf9, 0xca], "vpsubw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf9, 0xca], "vpsubw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf9, 0x0a], "vpsubw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf9, 0x0a], "vpsubw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf9, 0x4a, 0x01], "vpsubw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf9, 0x4a, 0x01], "vpsubw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0xfa, 0x0a], "vpsubd ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0xfa, 0x4a, 0x01], "vpsubd ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0xfa, 0x0a], "vpsubd ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0xfa, 0x0a], "vpsubd ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0xfa, 0x4a, 0x01], "vpsubd ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0xfa, 0x4a, 0x01], "vpsubd ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xfa, 0xca], "vpsubd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xfa, 0x0a], "vpsubd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xfa, 0x4a, 0x01], "vpsubd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xfa, 0xca], "vpsubd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xfa, 0xca], "vpsubd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xfa, 0x0a], "vpsubd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xfa, 0x0a], "vpsubd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xfa, 0x4a, 0x01], "vpsubd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xfa, 0x4a, 0x01], "vpsubd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0xfa, 0x0a], "vpsubd zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0xfa, 0x4a, 0x01], "vpsubd zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0xfa, 0x0a], "vpsubd zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0xfa, 0x0a], "vpsubd zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0xfa, 0x4a, 0x01], "vpsubd zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0xfa, 0x4a, 0x01], "vpsubd zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0xfa, 0x0a], "vpsubd xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0xfa, 0x4a, 0x01], "vpsubd xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0xfa, 0x0a], "vpsubd xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0xfa, 0x0a], "vpsubd xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0xfa, 0x4a, 0x01], "vpsubd xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0xfa, 0x4a, 0x01], "vpsubd xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xfa, 0xca], "vpsubd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xfa, 0x0a], "vpsubd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xfa, 0x4a, 0x01], "vpsubd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xfa, 0xca], "vpsubd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xfa, 0xca], "vpsubd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xfa, 0x0a], "vpsubd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xfa, 0x0a], "vpsubd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xfa, 0x4a, 0x01], "vpsubd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xfa, 0x4a, 0x01], "vpsubd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xfa, 0xca], "vpsubd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xfa, 0x0a], "vpsubd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xfa, 0x4a, 0x01], "vpsubd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xfa, 0xca], "vpsubd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xfa, 0xca], "vpsubd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xfa, 0x0a], "vpsubd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xfa, 0x0a], "vpsubd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xfa, 0x4a, 0x01], "vpsubd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xfa, 0x4a, 0x01], "vpsubd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xfb, 0x0a], "vpsubq ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xfb, 0x4a, 0x01], "vpsubq ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xfb, 0x0a], "vpsubq ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xfb, 0x0a], "vpsubq ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xfb, 0x4a, 0x01], "vpsubq ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xfb, 0x4a, 0x01], "vpsubq ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xfb, 0xca], "vpsubq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xfb, 0x0a], "vpsubq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xfb, 0x4a, 0x01], "vpsubq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xfb, 0xca], "vpsubq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xfb, 0xca], "vpsubq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xfb, 0x0a], "vpsubq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xfb, 0x0a], "vpsubq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xfb, 0x4a, 0x01], "vpsubq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xfb, 0x4a, 0x01], "vpsubq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xfb, 0x0a], "vpsubq zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xfb, 0x4a, 0x01], "vpsubq zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xfb, 0x0a], "vpsubq zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xfb, 0x0a], "vpsubq zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xfb, 0x4a, 0x01], "vpsubq zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xfb, 0x4a, 0x01], "vpsubq zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xfb, 0x0a], "vpsubq xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xfb, 0x4a, 0x01], "vpsubq xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xfb, 0x0a], "vpsubq xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xfb, 0x0a], "vpsubq xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xfb, 0x4a, 0x01], "vpsubq xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xfb, 0x4a, 0x01], "vpsubq xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xfb, 0xca], "vpsubq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xfb, 0x0a], "vpsubq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xfb, 0x4a, 0x01], "vpsubq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xfb, 0xca], "vpsubq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xfb, 0xca], "vpsubq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xfb, 0x0a], "vpsubq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xfb, 0x0a], "vpsubq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xfb, 0x4a, 0x01], "vpsubq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xfb, 0x4a, 0x01], "vpsubq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xfb, 0xca], "vpsubq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xfb, 0x0a], "vpsubq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xfb, 0x4a, 0x01], "vpsubq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xfb, 0xca], "vpsubq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xfb, 0xca], "vpsubq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xfb, 0x0a], "vpsubq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xfb, 0x0a], "vpsubq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xfb, 0x4a, 0x01], "vpsubq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xfb, 0x4a, 0x01], "vpsubq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xfc, 0xca], "vpaddb ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xfc, 0x0a], "vpaddb ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xfc, 0x4a, 0x01], "vpaddb ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xfc, 0xca], "vpaddb ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xfc, 0xca], "vpaddb ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xfc, 0x0a], "vpaddb ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xfc, 0x0a], "vpaddb ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xfc, 0x4a, 0x01], "vpaddb ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xfc, 0x4a, 0x01], "vpaddb ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xfc, 0xca], "vpaddb zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xfc, 0x0a], "vpaddb zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xfc, 0x4a, 0x01], "vpaddb zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xfc, 0xca], "vpaddb zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xfc, 0xca], "vpaddb zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xfc, 0x0a], "vpaddb zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xfc, 0x0a], "vpaddb zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xfc, 0x4a, 0x01], "vpaddb zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xfc, 0x4a, 0x01], "vpaddb zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xfc, 0xca], "vpaddb xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xfc, 0x0a], "vpaddb xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xfc, 0x4a, 0x01], "vpaddb xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xfc, 0xca], "vpaddb xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xfc, 0xca], "vpaddb xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xfc, 0x0a], "vpaddb xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xfc, 0x0a], "vpaddb xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xfc, 0x4a, 0x01], "vpaddb xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xfc, 0x4a, 0x01], "vpaddb xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xfd, 0xca], "vpaddw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xfd, 0x0a], "vpaddw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xfd, 0x4a, 0x01], "vpaddw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xfd, 0xca], "vpaddw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xfd, 0xca], "vpaddw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xfd, 0x0a], "vpaddw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xfd, 0x0a], "vpaddw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xfd, 0x4a, 0x01], "vpaddw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xfd, 0x4a, 0x01], "vpaddw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xfd, 0xca], "vpaddw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xfd, 0x0a], "vpaddw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xfd, 0x4a, 0x01], "vpaddw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xfd, 0xca], "vpaddw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xfd, 0xca], "vpaddw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xfd, 0x0a], "vpaddw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xfd, 0x0a], "vpaddw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xfd, 0x4a, 0x01], "vpaddw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xfd, 0x4a, 0x01], "vpaddw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xfd, 0xca], "vpaddw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xfd, 0x0a], "vpaddw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xfd, 0x4a, 0x01], "vpaddw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xfd, 0xca], "vpaddw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xfd, 0xca], "vpaddw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xfd, 0x0a], "vpaddw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xfd, 0x0a], "vpaddw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xfd, 0x4a, 0x01], "vpaddw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xfd, 0x4a, 0x01], "vpaddw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0xfe, 0x0a], "vpaddd ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0xfe, 0x4a, 0x01], "vpaddd ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0xfe, 0x0a], "vpaddd ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0xfe, 0x0a], "vpaddd ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0xfe, 0x4a, 0x01], "vpaddd ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0xfe, 0x4a, 0x01], "vpaddd ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xfe, 0xca], "vpaddd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xfe, 0x0a], "vpaddd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xfe, 0x4a, 0x01], "vpaddd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xfe, 0xca], "vpaddd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xfe, 0xca], "vpaddd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xfe, 0x0a], "vpaddd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xfe, 0x0a], "vpaddd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xfe, 0x4a, 0x01], "vpaddd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xfe, 0x4a, 0x01], "vpaddd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0xfe, 0x0a], "vpaddd zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0xfe, 0x4a, 0x01], "vpaddd zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0xfe, 0x0a], "vpaddd zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0xfe, 0x0a], "vpaddd zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0xfe, 0x4a, 0x01], "vpaddd zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0xfe, 0x4a, 0x01], "vpaddd zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0xfe, 0x0a], "vpaddd xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0xfe, 0x4a, 0x01], "vpaddd xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0xfe, 0x0a], "vpaddd xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0xfe, 0x0a], "vpaddd xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0xfe, 0x4a, 0x01], "vpaddd xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0xfe, 0x4a, 0x01], "vpaddd xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xfe, 0xca], "vpaddd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xfe, 0x0a], "vpaddd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xfe, 0x4a, 0x01], "vpaddd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xfe, 0xca], "vpaddd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xfe, 0xca], "vpaddd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xfe, 0x0a], "vpaddd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xfe, 0x0a], "vpaddd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xfe, 0x4a, 0x01], "vpaddd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xfe, 0x4a, 0x01], "vpaddd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xfe, 0xca], "vpaddd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xfe, 0x0a], "vpaddd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xfe, 0x4a, 0x01], "vpaddd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xfe, 0xca], "vpaddd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xfe, 0xca], "vpaddd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xfe, 0x0a], "vpaddd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xfe, 0x0a], "vpaddd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xfe, 0x4a, 0x01], "vpaddd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xfe, 0x4a, 0x01], "vpaddd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
}
#[test]
fn tests_66_0f38() {
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x00, 0xca], "vpshufb ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x00, 0x0a], "vpshufb ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x00, 0x4a, 0x01], "vpshufb ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x00, 0xca], "vpshufb ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x00, 0xca], "vpshufb ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x00, 0x0a], "vpshufb ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x00, 0x0a], "vpshufb ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x00, 0x4a, 0x01], "vpshufb ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x00, 0x4a, 0x01], "vpshufb ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x00, 0xca], "vpshufb zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x00, 0x0a], "vpshufb zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x00, 0x4a, 0x01], "vpshufb zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x00, 0xca], "vpshufb zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x00, 0xca], "vpshufb zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x00, 0x0a], "vpshufb zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x00, 0x0a], "vpshufb zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x00, 0x4a, 0x01], "vpshufb zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x00, 0x4a, 0x01], "vpshufb zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x00, 0xca], "vpshufb xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x00, 0x0a], "vpshufb xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x00, 0x4a, 0x01], "vpshufb xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x00, 0xca], "vpshufb xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x00, 0xca], "vpshufb xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x00, 0x0a], "vpshufb xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x00, 0x0a], "vpshufb xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x00, 0x4a, 0x01], "vpshufb xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x00, 0x4a, 0x01], "vpshufb xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x04, 0xca], "vpmaddubsw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x04, 0x0a], "vpmaddubsw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x04, 0x4a, 0x01], "vpmaddubsw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x04, 0xca], "vpmaddubsw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x04, 0xca], "vpmaddubsw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x04, 0x0a], "vpmaddubsw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x04, 0x0a], "vpmaddubsw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x04, 0x4a, 0x01], "vpmaddubsw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x04, 0x4a, 0x01], "vpmaddubsw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x04, 0xca], "vpmaddubsw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x04, 0x0a], "vpmaddubsw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x04, 0x4a, 0x01], "vpmaddubsw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x04, 0xca], "vpmaddubsw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x04, 0xca], "vpmaddubsw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x04, 0x0a], "vpmaddubsw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x04, 0x0a], "vpmaddubsw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x04, 0x4a, 0x01], "vpmaddubsw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x04, 0x4a, 0x01], "vpmaddubsw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x04, 0xca], "vpmaddubsw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x04, 0x0a], "vpmaddubsw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x04, 0x4a, 0x01], "vpmaddubsw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x04, 0xca], "vpmaddubsw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x04, 0xca], "vpmaddubsw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x04, 0x0a], "vpmaddubsw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x04, 0x0a], "vpmaddubsw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x04, 0x4a, 0x01], "vpmaddubsw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x04, 0x4a, 0x01], "vpmaddubsw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x0b, 0xca], "vpmulhrsw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x0b, 0x0a], "vpmulhrsw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x0b, 0x4a, 0x01], "vpmulhrsw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x0b, 0xca], "vpmulhrsw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x0b, 0xca], "vpmulhrsw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x0b, 0x0a], "vpmulhrsw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x0b, 0x0a], "vpmulhrsw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x0b, 0x4a, 0x01], "vpmulhrsw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x0b, 0x4a, 0x01], "vpmulhrsw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x0b, 0xca], "vpmulhrsw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x0b, 0x0a], "vpmulhrsw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x0b, 0x4a, 0x01], "vpmulhrsw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x0b, 0xca], "vpmulhrsw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x0b, 0xca], "vpmulhrsw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x0b, 0x0a], "vpmulhrsw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x0b, 0x0a], "vpmulhrsw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x0b, 0x4a, 0x01], "vpmulhrsw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x0b, 0x4a, 0x01], "vpmulhrsw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x0b, 0xca], "vpmulhrsw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x0b, 0x0a], "vpmulhrsw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x0b, 0x4a, 0x01], "vpmulhrsw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x0b, 0xca], "vpmulhrsw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x0b, 0xca], "vpmulhrsw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x0b, 0x0a], "vpmulhrsw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x0b, 0x0a], "vpmulhrsw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x0b, 0x4a, 0x01], "vpmulhrsw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x0b, 0x4a, 0x01], "vpmulhrsw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x0c, 0x0a], "vpermilps ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x0c, 0x4a, 0x01], "vpermilps ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x0c, 0x0a], "vpermilps ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x0c, 0x0a], "vpermilps ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x0c, 0x4a, 0x01], "vpermilps ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x0c, 0x4a, 0x01], "vpermilps ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x0c, 0xca], "vpermilps ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x0c, 0x0a], "vpermilps ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x0c, 0x4a, 0x01], "vpermilps ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x0c, 0xca], "vpermilps ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x0c, 0xca], "vpermilps ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x0c, 0x0a], "vpermilps ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x0c, 0x0a], "vpermilps ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x0c, 0x4a, 0x01], "vpermilps ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x0c, 0x4a, 0x01], "vpermilps ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x0c, 0x0a], "vpermilps zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x0c, 0x4a, 0x01], "vpermilps zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x0c, 0x0a], "vpermilps zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x0c, 0x0a], "vpermilps zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x0c, 0x4a, 0x01], "vpermilps zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x0c, 0x4a, 0x01], "vpermilps zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x0c, 0x0a], "vpermilps xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x0c, 0x4a, 0x01], "vpermilps xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x0c, 0x0a], "vpermilps xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x0c, 0x0a], "vpermilps xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x0c, 0x4a, 0x01], "vpermilps xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x0c, 0x4a, 0x01], "vpermilps xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x0c, 0xca], "vpermilps zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x0c, 0x0a], "vpermilps zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x0c, 0x4a, 0x01], "vpermilps zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x0c, 0xca], "vpermilps zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x0c, 0xca], "vpermilps zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x0c, 0x0a], "vpermilps zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x0c, 0x0a], "vpermilps zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x0c, 0x4a, 0x01], "vpermilps zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x0c, 0x4a, 0x01], "vpermilps zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x0c, 0xca], "vpermilps xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x0c, 0x0a], "vpermilps xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x0c, 0x4a, 0x01], "vpermilps xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x0c, 0xca], "vpermilps xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x0c, 0xca], "vpermilps xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x0c, 0x0a], "vpermilps xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x0c, 0x0a], "vpermilps xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x0c, 0x4a, 0x01], "vpermilps xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x0c, 0x4a, 0x01], "vpermilps xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x0d, 0x0a], "vpermilpd ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x0d, 0x4a, 0x01], "vpermilpd ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x0d, 0x0a], "vpermilpd ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x0d, 0x0a], "vpermilpd ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x0d, 0x4a, 0x01], "vpermilpd ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x0d, 0x4a, 0x01], "vpermilpd ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x0d, 0xca], "vpermilpd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x0d, 0x0a], "vpermilpd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x0d, 0x4a, 0x01], "vpermilpd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x0d, 0xca], "vpermilpd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x0d, 0xca], "vpermilpd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x0d, 0x0a], "vpermilpd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x0d, 0x0a], "vpermilpd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x0d, 0x4a, 0x01], "vpermilpd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x0d, 0x4a, 0x01], "vpermilpd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x0d, 0x0a], "vpermilpd zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x0d, 0x4a, 0x01], "vpermilpd zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x0d, 0x0a], "vpermilpd zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x0d, 0x0a], "vpermilpd zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x0d, 0x4a, 0x01], "vpermilpd zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x0d, 0x4a, 0x01], "vpermilpd zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x0d, 0x0a], "vpermilpd xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x0d, 0x4a, 0x01], "vpermilpd xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x0d, 0x0a], "vpermilpd xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x0d, 0x0a], "vpermilpd xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x0d, 0x4a, 0x01], "vpermilpd xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x0d, 0x4a, 0x01], "vpermilpd xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x0d, 0xca], "vpermilpd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x0d, 0x0a], "vpermilpd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x0d, 0x4a, 0x01], "vpermilpd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x0d, 0xca], "vpermilpd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x0d, 0xca], "vpermilpd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x0d, 0x0a], "vpermilpd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x0d, 0x0a], "vpermilpd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x0d, 0x4a, 0x01], "vpermilpd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x0d, 0x4a, 0x01], "vpermilpd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x0d, 0xca], "vpermilpd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x0d, 0x0a], "vpermilpd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x0d, 0x4a, 0x01], "vpermilpd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x0d, 0xca], "vpermilpd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x0d, 0xca], "vpermilpd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x0d, 0x0a], "vpermilpd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x0d, 0x0a], "vpermilpd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x0d, 0x4a, 0x01], "vpermilpd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x0d, 0x4a, 0x01], "vpermilpd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x10, 0xca], "vpsrlvw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x10, 0x0a], "vpsrlvw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x10, 0x4a, 0x01], "vpsrlvw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x10, 0xca], "vpsrlvw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x10, 0xca], "vpsrlvw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x10, 0x0a], "vpsrlvw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x10, 0x0a], "vpsrlvw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x10, 0x4a, 0x01], "vpsrlvw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x10, 0x4a, 0x01], "vpsrlvw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x10, 0xca], "vpsrlvw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x10, 0x0a], "vpsrlvw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x10, 0x4a, 0x01], "vpsrlvw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x10, 0xca], "vpsrlvw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x10, 0xca], "vpsrlvw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x10, 0x0a], "vpsrlvw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x10, 0x0a], "vpsrlvw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x10, 0x4a, 0x01], "vpsrlvw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x10, 0x4a, 0x01], "vpsrlvw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x10, 0xca], "vpsrlvw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x10, 0x0a], "vpsrlvw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x10, 0x4a, 0x01], "vpsrlvw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x10, 0xca], "vpsrlvw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x10, 0xca], "vpsrlvw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x10, 0x0a], "vpsrlvw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x10, 0x0a], "vpsrlvw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x10, 0x4a, 0x01], "vpsrlvw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x10, 0x4a, 0x01], "vpsrlvw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x11, 0xca], "vpsravw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x11, 0x0a], "vpsravw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x11, 0x4a, 0x01], "vpsravw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x11, 0xca], "vpsravw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x11, 0xca], "vpsravw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x11, 0x0a], "vpsravw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x11, 0x0a], "vpsravw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x11, 0x4a, 0x01], "vpsravw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x11, 0x4a, 0x01], "vpsravw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x11, 0xca], "vpsravw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x11, 0x0a], "vpsravw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x11, 0x4a, 0x01], "vpsravw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x11, 0xca], "vpsravw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x11, 0xca], "vpsravw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x11, 0x0a], "vpsravw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x11, 0x0a], "vpsravw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x11, 0x4a, 0x01], "vpsravw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x11, 0x4a, 0x01], "vpsravw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x11, 0xca], "vpsravw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x11, 0x0a], "vpsravw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x11, 0x4a, 0x01], "vpsravw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x11, 0xca], "vpsravw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x11, 0xca], "vpsravw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x11, 0x0a], "vpsravw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x11, 0x0a], "vpsravw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x11, 0x4a, 0x01], "vpsravw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x11, 0x4a, 0x01], "vpsravw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x12, 0xca], "vpsllvw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x12, 0x0a], "vpsllvw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x12, 0x4a, 0x01], "vpsllvw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x12, 0xca], "vpsllvw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x12, 0xca], "vpsllvw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x12, 0x0a], "vpsllvw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x12, 0x0a], "vpsllvw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x12, 0x4a, 0x01], "vpsllvw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x12, 0x4a, 0x01], "vpsllvw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x12, 0xca], "vpsllvw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x12, 0x0a], "vpsllvw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x12, 0x4a, 0x01], "vpsllvw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x12, 0xca], "vpsllvw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x12, 0xca], "vpsllvw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x12, 0x0a], "vpsllvw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x12, 0x0a], "vpsllvw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x12, 0x4a, 0x01], "vpsllvw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x12, 0x4a, 0x01], "vpsllvw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x12, 0xca], "vpsllvw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x12, 0x0a], "vpsllvw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x12, 0x4a, 0x01], "vpsllvw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x12, 0xca], "vpsllvw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x12, 0xca], "vpsllvw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x12, 0x0a], "vpsllvw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x12, 0x0a], "vpsllvw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x12, 0x4a, 0x01], "vpsllvw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x12, 0x4a, 0x01], "vpsllvw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xfd, 0x13, 0xca], "vcvtph2ps zmm1{k5}{z}{sae}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x78, 0x13, 0xca], "vcvtph2ps zmm1{sae}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x7d, 0x13, 0xca], "vcvtph2ps zmm1{k5}{sae}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x13, 0xca], "vcvtph2ps ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x13, 0x0a], "vcvtph2ps ymm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x13, 0x4a, 0x01], "vcvtph2ps ymm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x13, 0xca], "vcvtph2ps ymm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x13, 0xca], "vcvtph2ps ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x13, 0x0a], "vcvtph2ps ymm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x13, 0x0a], "vcvtph2ps ymm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x13, 0x4a, 0x01], "vcvtph2ps ymm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x13, 0x4a, 0x01], "vcvtph2ps ymm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x13, 0xca], "vcvtph2ps zmm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x13, 0x0a], "vcvtph2ps zmm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x13, 0x4a, 0x01], "vcvtph2ps zmm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x13, 0xca], "vcvtph2ps zmm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x13, 0xca], "vcvtph2ps zmm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x13, 0x0a], "vcvtph2ps zmm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x13, 0x0a], "vcvtph2ps zmm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x13, 0x4a, 0x01], "vcvtph2ps zmm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x13, 0x4a, 0x01], "vcvtph2ps zmm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x13, 0xca], "vcvtph2ps xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x13, 0x0a], "vcvtph2ps xmm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x13, 0x4a, 0x01], "vcvtph2ps xmm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x13, 0xca], "vcvtph2ps xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x13, 0xca], "vcvtph2ps xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x13, 0x0a], "vcvtph2ps xmm1, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x13, 0x0a], "vcvtph2ps xmm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x13, 0x4a, 0x01], "vcvtph2ps xmm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x13, 0x4a, 0x01], "vcvtph2ps xmm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x14, 0x0a], "vprorvq ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x14, 0x4a, 0x01], "vprorvq ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x14, 0x0a], "vprorvq ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x14, 0x0a], "vprorvq ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x14, 0x4a, 0x01], "vprorvq ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x14, 0x4a, 0x01], "vprorvq ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x14, 0xca], "vprorvq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x14, 0x0a], "vprorvq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x14, 0x4a, 0x01], "vprorvq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x14, 0xca], "vprorvq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x14, 0xca], "vprorvq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x14, 0x0a], "vprorvq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x14, 0x0a], "vprorvq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x14, 0x4a, 0x01], "vprorvq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x14, 0x4a, 0x01], "vprorvq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x14, 0x0a], "vprorvd ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x14, 0x4a, 0x01], "vprorvd ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x14, 0x0a], "vprorvd ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x14, 0x0a], "vprorvd ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x14, 0x4a, 0x01], "vprorvd ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x14, 0x4a, 0x01], "vprorvd ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x14, 0xca], "vprorvd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x14, 0x0a], "vprorvd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x14, 0x4a, 0x01], "vprorvd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x14, 0xca], "vprorvd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x14, 0xca], "vprorvd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x14, 0x0a], "vprorvd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x14, 0x0a], "vprorvd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x14, 0x4a, 0x01], "vprorvd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x14, 0x4a, 0x01], "vprorvd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x14, 0x0a], "vprorvq zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x14, 0x4a, 0x01], "vprorvq zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x14, 0x0a], "vprorvq zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x14, 0x0a], "vprorvq zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x14, 0x4a, 0x01], "vprorvq zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x14, 0x4a, 0x01], "vprorvq zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x14, 0x0a], "vprorvq xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x14, 0x4a, 0x01], "vprorvq xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x14, 0x0a], "vprorvq xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x14, 0x0a], "vprorvq xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x14, 0x4a, 0x01], "vprorvq xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x14, 0x4a, 0x01], "vprorvq xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x14, 0xca], "vprorvq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x14, 0x0a], "vprorvq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x14, 0x4a, 0x01], "vprorvq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x14, 0xca], "vprorvq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x14, 0xca], "vprorvq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x14, 0x0a], "vprorvq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x14, 0x0a], "vprorvq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x14, 0x4a, 0x01], "vprorvq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x14, 0x4a, 0x01], "vprorvq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x14, 0xca], "vprorvq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x14, 0x0a], "vprorvq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x14, 0x4a, 0x01], "vprorvq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x14, 0xca], "vprorvq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x14, 0xca], "vprorvq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x14, 0x0a], "vprorvq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x14, 0x0a], "vprorvq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x14, 0x4a, 0x01], "vprorvq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x14, 0x4a, 0x01], "vprorvq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x14, 0x0a], "vprorvd zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x14, 0x4a, 0x01], "vprorvd zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x14, 0x0a], "vprorvd zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x14, 0x0a], "vprorvd zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x14, 0x4a, 0x01], "vprorvd zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x14, 0x4a, 0x01], "vprorvd zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x14, 0x0a], "vprorvd xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x14, 0x4a, 0x01], "vprorvd xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x14, 0x0a], "vprorvd xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x14, 0x0a], "vprorvd xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x14, 0x4a, 0x01], "vprorvd xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x14, 0x4a, 0x01], "vprorvd xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x14, 0xca], "vprorvd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x14, 0x0a], "vprorvd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x14, 0x4a, 0x01], "vprorvd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x14, 0xca], "vprorvd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x14, 0xca], "vprorvd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x14, 0x0a], "vprorvd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x14, 0x0a], "vprorvd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x14, 0x4a, 0x01], "vprorvd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x14, 0x4a, 0x01], "vprorvd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x14, 0xca], "vprorvd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x14, 0x0a], "vprorvd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x14, 0x4a, 0x01], "vprorvd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x14, 0xca], "vprorvd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x14, 0xca], "vprorvd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x14, 0x0a], "vprorvd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x14, 0x0a], "vprorvd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x14, 0x4a, 0x01], "vprorvd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x14, 0x4a, 0x01], "vprorvd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x15, 0x0a], "vprolvq ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x15, 0x4a, 0x01], "vprolvq ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x15, 0x0a], "vprolvq ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x15, 0x0a], "vprolvq ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x15, 0x4a, 0x01], "vprolvq ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x15, 0x4a, 0x01], "vprolvq ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x15, 0xca], "vprolvq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x15, 0x0a], "vprolvq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x15, 0x4a, 0x01], "vprolvq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x15, 0xca], "vprolvq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x15, 0xca], "vprolvq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x15, 0x0a], "vprolvq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x15, 0x0a], "vprolvq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x15, 0x4a, 0x01], "vprolvq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x15, 0x4a, 0x01], "vprolvq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x15, 0x0a], "vprolvd ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x15, 0x4a, 0x01], "vprolvd ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x15, 0x0a], "vprolvd ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x15, 0x0a], "vprolvd ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x15, 0x4a, 0x01], "vprolvd ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x15, 0x4a, 0x01], "vprolvd ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x15, 0xca], "vprolvd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x15, 0x0a], "vprolvd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x15, 0x4a, 0x01], "vprolvd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x15, 0xca], "vprolvd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x15, 0xca], "vprolvd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x15, 0x0a], "vprolvd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x15, 0x0a], "vprolvd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x15, 0x4a, 0x01], "vprolvd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x15, 0x4a, 0x01], "vprolvd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x15, 0x0a], "vprolvq zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x15, 0x4a, 0x01], "vprolvq zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x15, 0x0a], "vprolvq zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x15, 0x0a], "vprolvq zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x15, 0x4a, 0x01], "vprolvq zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x15, 0x4a, 0x01], "vprolvq zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x15, 0x0a], "vprolvq xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x15, 0x4a, 0x01], "vprolvq xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x15, 0x0a], "vprolvq xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x15, 0x0a], "vprolvq xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x15, 0x4a, 0x01], "vprolvq xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x15, 0x4a, 0x01], "vprolvq xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x15, 0xca], "vprolvq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x15, 0x0a], "vprolvq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x15, 0x4a, 0x01], "vprolvq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x15, 0xca], "vprolvq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x15, 0xca], "vprolvq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x15, 0x0a], "vprolvq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x15, 0x0a], "vprolvq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x15, 0x4a, 0x01], "vprolvq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x15, 0x4a, 0x01], "vprolvq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x15, 0xca], "vprolvq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x15, 0x0a], "vprolvq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x15, 0x4a, 0x01], "vprolvq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x15, 0xca], "vprolvq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x15, 0xca], "vprolvq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x15, 0x0a], "vprolvq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x15, 0x0a], "vprolvq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x15, 0x4a, 0x01], "vprolvq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x15, 0x4a, 0x01], "vprolvq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x15, 0x0a], "vprolvd zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x15, 0x4a, 0x01], "vprolvd zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x15, 0x0a], "vprolvd zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x15, 0x0a], "vprolvd zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x15, 0x4a, 0x01], "vprolvd zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x15, 0x4a, 0x01], "vprolvd zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x15, 0x0a], "vprolvd xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x15, 0x4a, 0x01], "vprolvd xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x15, 0x0a], "vprolvd xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x15, 0x0a], "vprolvd xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x15, 0x4a, 0x01], "vprolvd xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x15, 0x4a, 0x01], "vprolvd xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x15, 0xca], "vprolvd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x15, 0x0a], "vprolvd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x15, 0x4a, 0x01], "vprolvd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x15, 0xca], "vprolvd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x15, 0xca], "vprolvd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x15, 0x0a], "vprolvd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x15, 0x0a], "vprolvd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x15, 0x4a, 0x01], "vprolvd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x15, 0x4a, 0x01], "vprolvd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x15, 0xca], "vprolvd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x15, 0x0a], "vprolvd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x15, 0x4a, 0x01], "vprolvd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x15, 0xca], "vprolvd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x15, 0xca], "vprolvd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x15, 0x0a], "vprolvd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x15, 0x0a], "vprolvd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x15, 0x4a, 0x01], "vprolvd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x15, 0x4a, 0x01], "vprolvd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x16, 0x0a], "vpermpd ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x16, 0x4a, 0x01], "vpermpd ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x16, 0x0a], "vpermpd ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x16, 0x0a], "vpermpd ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x16, 0x4a, 0x01], "vpermpd ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x16, 0x4a, 0x01], "vpermpd ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x16, 0xca], "vpermpd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x16, 0x0a], "vpermpd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x16, 0x4a, 0x01], "vpermpd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x16, 0xca], "vpermpd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x16, 0xca], "vpermpd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x16, 0x0a], "vpermpd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x16, 0x0a], "vpermpd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x16, 0x4a, 0x01], "vpermpd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x16, 0x4a, 0x01], "vpermpd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x16, 0x0a], "vpermps ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x16, 0x4a, 0x01], "vpermps ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x16, 0x0a], "vpermps ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x16, 0x0a], "vpermps ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x16, 0x4a, 0x01], "vpermps ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x16, 0x4a, 0x01], "vpermps ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x16, 0xca], "vpermps ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x16, 0x0a], "vpermps ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x16, 0x4a, 0x01], "vpermps ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x16, 0xca], "vpermps ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x16, 0xca], "vpermps ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x16, 0x0a], "vpermps ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x16, 0x0a], "vpermps ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x16, 0x4a, 0x01], "vpermps ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x16, 0x4a, 0x01], "vpermps ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x16, 0x0a], "vpermpd zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x16, 0x4a, 0x01], "vpermpd zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x16, 0x0a], "vpermpd zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x16, 0x0a], "vpermpd zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x16, 0x4a, 0x01], "vpermpd zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x16, 0x4a, 0x01], "vpermpd zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x16, 0xca], "vpermpd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x16, 0x0a], "vpermpd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x16, 0x4a, 0x01], "vpermpd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x16, 0xca], "vpermpd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x16, 0xca], "vpermpd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x16, 0x0a], "vpermpd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x16, 0x0a], "vpermpd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x16, 0x4a, 0x01], "vpermpd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x16, 0x4a, 0x01], "vpermpd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x16, 0x0a], "vpermps zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x16, 0x4a, 0x01], "vpermps zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x16, 0x0a], "vpermps zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x16, 0x0a], "vpermps zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x16, 0x4a, 0x01], "vpermps zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x16, 0x4a, 0x01], "vpermps zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x16, 0xca], "vpermps zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x16, 0x0a], "vpermps zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x16, 0x4a, 0x01], "vpermps zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x16, 0xca], "vpermps zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x16, 0xca], "vpermps zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x16, 0x0a], "vpermps zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x16, 0x0a], "vpermps zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x16, 0x4a, 0x01], "vpermps zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x16, 0x4a, 0x01], "vpermps zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x18, 0xca], "vbroadcastss ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x18, 0x0a], "vbroadcastss ymm1{k5}{z}, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x18, 0x4a, 0x01], "vbroadcastss ymm1{k5}{z}, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x18, 0xca], "vbroadcastss ymm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x18, 0xca], "vbroadcastss ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x18, 0x0a], "vbroadcastss ymm1, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x18, 0x0a], "vbroadcastss ymm1{k5}, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x18, 0x4a, 0x01], "vbroadcastss ymm1, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x18, 0x4a, 0x01], "vbroadcastss ymm1{k5}, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x18, 0xca], "vbroadcastss zmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x18, 0x0a], "vbroadcastss zmm1{k5}{z}, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x18, 0x4a, 0x01], "vbroadcastss zmm1{k5}{z}, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x18, 0xca], "vbroadcastss zmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x18, 0xca], "vbroadcastss zmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x18, 0x0a], "vbroadcastss zmm1, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x18, 0x0a], "vbroadcastss zmm1{k5}, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x18, 0x4a, 0x01], "vbroadcastss zmm1, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x18, 0x4a, 0x01], "vbroadcastss zmm1{k5}, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x18, 0xca], "vbroadcastss xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x18, 0x0a], "vbroadcastss xmm1{k5}{z}, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x18, 0x4a, 0x01], "vbroadcastss xmm1{k5}{z}, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x18, 0xca], "vbroadcastss xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x18, 0xca], "vbroadcastss xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x18, 0x0a], "vbroadcastss xmm1, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x18, 0x0a], "vbroadcastss xmm1{k5}, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x18, 0x4a, 0x01], "vbroadcastss xmm1, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x18, 0x4a, 0x01], "vbroadcastss xmm1{k5}, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x19, 0xca], "vbroadcastsd ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x19, 0x0a], "vbroadcastsd ymm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x19, 0x4a, 0x01], "vbroadcastsd ymm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x19, 0xca], "vbroadcastsd ymm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x19, 0xca], "vbroadcastsd ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x19, 0x0a], "vbroadcastsd ymm1, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x19, 0x0a], "vbroadcastsd ymm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x19, 0x4a, 0x01], "vbroadcastsd ymm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x19, 0x4a, 0x01], "vbroadcastsd ymm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x19, 0xca], "vbroadcastf32x2 ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x19, 0x0a], "vbroadcastf32x2 ymm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x19, 0x4a, 0x01], "vbroadcastf32x2 ymm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x19, 0xca], "vbroadcastf32x2 ymm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x19, 0xca], "vbroadcastf32x2 ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x19, 0x0a], "vbroadcastf32x2 ymm1, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x19, 0x0a], "vbroadcastf32x2 ymm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x19, 0x4a, 0x01], "vbroadcastf32x2 ymm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x19, 0x4a, 0x01], "vbroadcastf32x2 ymm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x19, 0xca], "vbroadcastsd zmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x19, 0x0a], "vbroadcastsd zmm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x19, 0x4a, 0x01], "vbroadcastsd zmm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x19, 0xca], "vbroadcastsd zmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x19, 0xca], "vbroadcastsd zmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x19, 0x0a], "vbroadcastsd zmm1, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x19, 0x0a], "vbroadcastsd zmm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x19, 0x4a, 0x01], "vbroadcastsd zmm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x19, 0x4a, 0x01], "vbroadcastsd zmm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x19, 0xca], "vbroadcastf32x2 zmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x19, 0x0a], "vbroadcastf32x2 zmm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x19, 0x4a, 0x01], "vbroadcastf32x2 zmm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x19, 0xca], "vbroadcastf32x2 zmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x19, 0xca], "vbroadcastf32x2 zmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x19, 0x0a], "vbroadcastf32x2 zmm1, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x19, 0x0a], "vbroadcastf32x2 zmm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x19, 0x4a, 0x01], "vbroadcastf32x2 zmm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x19, 0x4a, 0x01], "vbroadcastf32x2 zmm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x1a, 0x0a], "vbroadcastf64x2 ymm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x1a, 0x4a, 0x01], "vbroadcastf64x2 ymm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x1a, 0x0a], "vbroadcastf64x2 ymm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x1a, 0x0a], "vbroadcastf64x2 ymm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x1a, 0x4a, 0x01], "vbroadcastf64x2 ymm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x1a, 0x4a, 0x01], "vbroadcastf64x2 ymm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x1a, 0x0a], "vbroadcastf32x4 ymm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x1a, 0x4a, 0x01], "vbroadcastf32x4 ymm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x1a, 0x0a], "vbroadcastf32x4 ymm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x1a, 0x0a], "vbroadcastf32x4 ymm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x1a, 0x4a, 0x01], "vbroadcastf32x4 ymm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x1a, 0x4a, 0x01], "vbroadcastf32x4 ymm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x1a, 0x0a], "vbroadcastf64x2 zmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x1a, 0x4a, 0x01], "vbroadcastf64x2 zmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x1a, 0x0a], "vbroadcastf64x2 zmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x1a, 0x0a], "vbroadcastf64x2 zmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x1a, 0x4a, 0x01], "vbroadcastf64x2 zmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x1a, 0x4a, 0x01], "vbroadcastf64x2 zmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x1a, 0x0a], "vbroadcastf32x4 zmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x1a, 0x4a, 0x01], "vbroadcastf32x4 zmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x1a, 0x0a], "vbroadcastf32x4 zmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x1a, 0x0a], "vbroadcastf32x4 zmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x1a, 0x4a, 0x01], "vbroadcastf32x4 zmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x1a, 0x4a, 0x01], "vbroadcastf32x4 zmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x1b, 0x0a], "vbroadcastf64x4 zmm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x1b, 0x4a, 0x01], "vbroadcastf64x4 zmm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x1b, 0x0a], "vbroadcastf64x4 zmm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x1b, 0x0a], "vbroadcastf64x4 zmm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x1b, 0x4a, 0x01], "vbroadcastf64x4 zmm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x1b, 0x4a, 0x01], "vbroadcastf64x4 zmm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x1b, 0x0a], "vbroadcastf32x8 zmm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x1b, 0x4a, 0x01], "vbroadcastf32x8 zmm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x1b, 0x0a], "vbroadcastf32x8 zmm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x1b, 0x0a], "vbroadcastf32x8 zmm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x1b, 0x4a, 0x01], "vbroadcastf32x8 zmm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x1b, 0x4a, 0x01], "vbroadcastf32x8 zmm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x1c, 0xca], "vpabsb ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x1c, 0x0a], "vpabsb ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x1c, 0x4a, 0x01], "vpabsb ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x1c, 0xca], "vpabsb ymm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x1c, 0xca], "vpabsb ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x1c, 0x0a], "vpabsb ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x1c, 0x0a], "vpabsb ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x1c, 0x4a, 0x01], "vpabsb ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x1c, 0x4a, 0x01], "vpabsb ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x1c, 0xca], "vpabsb zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x1c, 0x0a], "vpabsb zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x1c, 0x4a, 0x01], "vpabsb zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x1c, 0xca], "vpabsb zmm1, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x1c, 0xca], "vpabsb zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x1c, 0x0a], "vpabsb zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x1c, 0x0a], "vpabsb zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x1c, 0x4a, 0x01], "vpabsb zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x1c, 0x4a, 0x01], "vpabsb zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x1c, 0xca], "vpabsb xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x1c, 0x0a], "vpabsb xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x1c, 0x4a, 0x01], "vpabsb xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x1c, 0xca], "vpabsb xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x1c, 0xca], "vpabsb xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x1c, 0x0a], "vpabsb xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x1c, 0x0a], "vpabsb xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x1c, 0x4a, 0x01], "vpabsb xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x1c, 0x4a, 0x01], "vpabsb xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x1d, 0xca], "vpabsw ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x1d, 0x0a], "vpabsw ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x1d, 0x4a, 0x01], "vpabsw ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x1d, 0xca], "vpabsw ymm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x1d, 0xca], "vpabsw ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x1d, 0x0a], "vpabsw ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x1d, 0x0a], "vpabsw ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x1d, 0x4a, 0x01], "vpabsw ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x1d, 0x4a, 0x01], "vpabsw ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x1d, 0xca], "vpabsw zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x1d, 0x0a], "vpabsw zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x1d, 0x4a, 0x01], "vpabsw zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x1d, 0xca], "vpabsw zmm1, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x1d, 0xca], "vpabsw zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x1d, 0x0a], "vpabsw zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x1d, 0x0a], "vpabsw zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x1d, 0x4a, 0x01], "vpabsw zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x1d, 0x4a, 0x01], "vpabsw zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x1d, 0xca], "vpabsw xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x1d, 0x0a], "vpabsw xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x1d, 0x4a, 0x01], "vpabsw xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x1d, 0xca], "vpabsw xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x1d, 0xca], "vpabsw xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x1d, 0x0a], "vpabsw xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x1d, 0x0a], "vpabsw xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x1d, 0x4a, 0x01], "vpabsw xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x1d, 0x4a, 0x01], "vpabsw xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x1e, 0x0a], "vpabsd ymm1{k5}{z}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x1e, 0x4a, 0x01], "vpabsd ymm1{k5}{z}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x1e, 0x0a], "vpabsd ymm1, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x1e, 0x0a], "vpabsd ymm1{k5}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x1e, 0x4a, 0x01], "vpabsd ymm1, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x1e, 0x4a, 0x01], "vpabsd ymm1{k5}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x1e, 0xca], "vpabsd ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x1e, 0x0a], "vpabsd ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x1e, 0x4a, 0x01], "vpabsd ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x1e, 0xca], "vpabsd ymm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x1e, 0xca], "vpabsd ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x1e, 0x0a], "vpabsd ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x1e, 0x0a], "vpabsd ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x1e, 0x4a, 0x01], "vpabsd ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x1e, 0x4a, 0x01], "vpabsd ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x1e, 0x0a], "vpabsd zmm1{k5}{z}, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x1e, 0x4a, 0x01], "vpabsd zmm1{k5}{z}, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x1e, 0x0a], "vpabsd zmm1, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x1e, 0x0a], "vpabsd zmm1{k5}, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x1e, 0x4a, 0x01], "vpabsd zmm1, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x1e, 0x4a, 0x01], "vpabsd zmm1{k5}, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x1e, 0x0a], "vpabsd xmm1{k5}{z}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x1e, 0x4a, 0x01], "vpabsd xmm1{k5}{z}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x1e, 0x0a], "vpabsd xmm1, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x1e, 0x0a], "vpabsd xmm1{k5}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x1e, 0x4a, 0x01], "vpabsd xmm1, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x1e, 0x4a, 0x01], "vpabsd xmm1{k5}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x1e, 0xca], "vpabsd zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x1e, 0x0a], "vpabsd zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x1e, 0x4a, 0x01], "vpabsd zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x1e, 0xca], "vpabsd zmm1, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x1e, 0xca], "vpabsd zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x1e, 0x0a], "vpabsd zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x1e, 0x0a], "vpabsd zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x1e, 0x4a, 0x01], "vpabsd zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x1e, 0x4a, 0x01], "vpabsd zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x1e, 0xca], "vpabsd xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x1e, 0x0a], "vpabsd xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x1e, 0x4a, 0x01], "vpabsd xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x1e, 0xca], "vpabsd xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x1e, 0xca], "vpabsd xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x1e, 0x0a], "vpabsd xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x1e, 0x0a], "vpabsd xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x1e, 0x4a, 0x01], "vpabsd xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x1e, 0x4a, 0x01], "vpabsd xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x1f, 0x0a], "vpabsq ymm1{k5}{z}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x1f, 0x4a, 0x01], "vpabsq ymm1{k5}{z}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x1f, 0x0a], "vpabsq ymm1, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x1f, 0x0a], "vpabsq ymm1{k5}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x1f, 0x4a, 0x01], "vpabsq ymm1, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x1f, 0x4a, 0x01], "vpabsq ymm1{k5}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x1f, 0xca], "vpabsq ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x1f, 0x0a], "vpabsq ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x1f, 0x4a, 0x01], "vpabsq ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x1f, 0xca], "vpabsq ymm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x1f, 0xca], "vpabsq ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x1f, 0x0a], "vpabsq ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x1f, 0x0a], "vpabsq ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x1f, 0x4a, 0x01], "vpabsq ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x1f, 0x4a, 0x01], "vpabsq ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x1f, 0x0a], "vpabsq zmm1{k5}{z}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x1f, 0x4a, 0x01], "vpabsq zmm1{k5}{z}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x1f, 0x0a], "vpabsq zmm1, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x1f, 0x0a], "vpabsq zmm1{k5}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x1f, 0x4a, 0x01], "vpabsq zmm1, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x1f, 0x4a, 0x01], "vpabsq zmm1{k5}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x1f, 0x0a], "vpabsq xmm1{k5}{z}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x1f, 0x4a, 0x01], "vpabsq xmm1{k5}{z}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x1f, 0x0a], "vpabsq xmm1, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x1f, 0x0a], "vpabsq xmm1{k5}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x1f, 0x4a, 0x01], "vpabsq xmm1, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x1f, 0x4a, 0x01], "vpabsq xmm1{k5}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x1f, 0xca], "vpabsq zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x1f, 0x0a], "vpabsq zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x1f, 0x4a, 0x01], "vpabsq zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x1f, 0xca], "vpabsq zmm1, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x1f, 0xca], "vpabsq zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x1f, 0x0a], "vpabsq zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x1f, 0x0a], "vpabsq zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x1f, 0x4a, 0x01], "vpabsq zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x1f, 0x4a, 0x01], "vpabsq zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x1f, 0xca], "vpabsq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x1f, 0x0a], "vpabsq xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x1f, 0x4a, 0x01], "vpabsq xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x1f, 0xca], "vpabsq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x1f, 0xca], "vpabsq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x1f, 0x0a], "vpabsq xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x1f, 0x0a], "vpabsq xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x1f, 0x4a, 0x01], "vpabsq xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x1f, 0x4a, 0x01], "vpabsq xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x20, 0xca], "vpmovsxbw ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x20, 0x0a], "vpmovsxbw ymm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x20, 0x4a, 0x01], "vpmovsxbw ymm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x20, 0xca], "vpmovsxbw ymm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x20, 0xca], "vpmovsxbw ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x20, 0x0a], "vpmovsxbw ymm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x20, 0x0a], "vpmovsxbw ymm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x20, 0x4a, 0x01], "vpmovsxbw ymm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x20, 0x4a, 0x01], "vpmovsxbw ymm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x20, 0xca], "vpmovsxbw zmm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x20, 0x0a], "vpmovsxbw zmm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x20, 0x4a, 0x01], "vpmovsxbw zmm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x20, 0xca], "vpmovsxbw zmm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x20, 0xca], "vpmovsxbw zmm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x20, 0x0a], "vpmovsxbw zmm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x20, 0x0a], "vpmovsxbw zmm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x20, 0x4a, 0x01], "vpmovsxbw zmm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x20, 0x4a, 0x01], "vpmovsxbw zmm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x20, 0xca], "vpmovsxbw xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x20, 0x0a], "vpmovsxbw xmm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x20, 0x4a, 0x01], "vpmovsxbw xmm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x20, 0xca], "vpmovsxbw xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x20, 0xca], "vpmovsxbw xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x20, 0x0a], "vpmovsxbw xmm1, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x20, 0x0a], "vpmovsxbw xmm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x20, 0x4a, 0x01], "vpmovsxbw xmm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x20, 0x4a, 0x01], "vpmovsxbw xmm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x21, 0xca], "vpmovsxbd ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x21, 0x0a], "vpmovsxbd ymm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x21, 0x4a, 0x01], "vpmovsxbd ymm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x21, 0xca], "vpmovsxbd ymm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x21, 0xca], "vpmovsxbd ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x21, 0x0a], "vpmovsxbd ymm1, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x21, 0x0a], "vpmovsxbd ymm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x21, 0x4a, 0x01], "vpmovsxbd ymm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x21, 0x4a, 0x01], "vpmovsxbd ymm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x21, 0xca], "vpmovsxbd zmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x21, 0x0a], "vpmovsxbd zmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x21, 0x4a, 0x01], "vpmovsxbd zmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x21, 0xca], "vpmovsxbd zmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x21, 0xca], "vpmovsxbd zmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x21, 0x0a], "vpmovsxbd zmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x21, 0x0a], "vpmovsxbd zmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x21, 0x4a, 0x01], "vpmovsxbd zmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x21, 0x4a, 0x01], "vpmovsxbd zmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x21, 0xca], "vpmovsxbd xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x21, 0x0a], "vpmovsxbd xmm1{k5}{z}, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x21, 0x4a, 0x01], "vpmovsxbd xmm1{k5}{z}, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x21, 0xca], "vpmovsxbd xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x21, 0xca], "vpmovsxbd xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x21, 0x0a], "vpmovsxbd xmm1, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x21, 0x0a], "vpmovsxbd xmm1{k5}, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x21, 0x4a, 0x01], "vpmovsxbd xmm1, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x21, 0x4a, 0x01], "vpmovsxbd xmm1{k5}, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x22, 0xca], "vpmovsxbq ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x22, 0x0a], "vpmovsxbq ymm1{k5}{z}, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x22, 0x4a, 0x01], "vpmovsxbq ymm1{k5}{z}, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x22, 0xca], "vpmovsxbq ymm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x22, 0xca], "vpmovsxbq ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x22, 0x0a], "vpmovsxbq ymm1, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x22, 0x0a], "vpmovsxbq ymm1{k5}, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x22, 0x4a, 0x01], "vpmovsxbq ymm1, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x22, 0x4a, 0x01], "vpmovsxbq ymm1{k5}, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x22, 0xca], "vpmovsxbq zmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x22, 0x0a], "vpmovsxbq zmm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x22, 0x4a, 0x01], "vpmovsxbq zmm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x22, 0xca], "vpmovsxbq zmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x22, 0xca], "vpmovsxbq zmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x22, 0x0a], "vpmovsxbq zmm1, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x22, 0x0a], "vpmovsxbq zmm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x22, 0x4a, 0x01], "vpmovsxbq zmm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x22, 0x4a, 0x01], "vpmovsxbq zmm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x22, 0xca], "vpmovsxbq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x22, 0x0a], "vpmovsxbq xmm1{k5}{z}, word [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x22, 0x4a, 0x01], "vpmovsxbq xmm1{k5}{z}, word [edx + 0x2]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x22, 0xca], "vpmovsxbq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x22, 0xca], "vpmovsxbq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x22, 0x0a], "vpmovsxbq xmm1, word [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x22, 0x0a], "vpmovsxbq xmm1{k5}, word [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x22, 0x4a, 0x01], "vpmovsxbq xmm1, word [edx + 0x2]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x22, 0x4a, 0x01], "vpmovsxbq xmm1{k5}, word [edx + 0x2]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x23, 0xca], "vpmovsxwd ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x23, 0x0a], "vpmovsxwd ymm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x23, 0x4a, 0x01], "vpmovsxwd ymm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x23, 0xca], "vpmovsxwd ymm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x23, 0xca], "vpmovsxwd ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x23, 0x0a], "vpmovsxwd ymm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x23, 0x0a], "vpmovsxwd ymm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x23, 0x4a, 0x01], "vpmovsxwd ymm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x23, 0x4a, 0x01], "vpmovsxwd ymm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x23, 0xca], "vpmovsxwd zmm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x23, 0x0a], "vpmovsxwd zmm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x23, 0x4a, 0x01], "vpmovsxwd zmm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x23, 0xca], "vpmovsxwd zmm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x23, 0xca], "vpmovsxwd zmm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x23, 0x0a], "vpmovsxwd zmm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x23, 0x0a], "vpmovsxwd zmm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x23, 0x4a, 0x01], "vpmovsxwd zmm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x23, 0x4a, 0x01], "vpmovsxwd zmm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x23, 0xca], "vpmovsxwd xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x23, 0x0a], "vpmovsxwd xmm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x23, 0x4a, 0x01], "vpmovsxwd xmm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x23, 0xca], "vpmovsxwd xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x23, 0xca], "vpmovsxwd xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x23, 0x0a], "vpmovsxwd xmm1, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x23, 0x0a], "vpmovsxwd xmm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x23, 0x4a, 0x01], "vpmovsxwd xmm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x23, 0x4a, 0x01], "vpmovsxwd xmm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x24, 0xca], "vpmovsxwq ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x24, 0x0a], "vpmovsxwq ymm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x24, 0x4a, 0x01], "vpmovsxwq ymm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x24, 0xca], "vpmovsxwq ymm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x24, 0xca], "vpmovsxwq ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x24, 0x0a], "vpmovsxwq ymm1, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x24, 0x0a], "vpmovsxwq ymm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x24, 0x4a, 0x01], "vpmovsxwq ymm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x24, 0x4a, 0x01], "vpmovsxwq ymm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x24, 0xca], "vpmovsxwq zmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x24, 0x0a], "vpmovsxwq zmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x24, 0x4a, 0x01], "vpmovsxwq zmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x24, 0xca], "vpmovsxwq zmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x24, 0xca], "vpmovsxwq zmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x24, 0x0a], "vpmovsxwq zmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x24, 0x0a], "vpmovsxwq zmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x24, 0x4a, 0x01], "vpmovsxwq zmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x24, 0x4a, 0x01], "vpmovsxwq zmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x24, 0xca], "vpmovsxwq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x24, 0x0a], "vpmovsxwq xmm1{k5}{z}, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x24, 0x4a, 0x01], "vpmovsxwq xmm1{k5}{z}, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x24, 0xca], "vpmovsxwq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x24, 0xca], "vpmovsxwq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x24, 0x0a], "vpmovsxwq xmm1, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x24, 0x0a], "vpmovsxwq xmm1{k5}, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x24, 0x4a, 0x01], "vpmovsxwq xmm1, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x24, 0x4a, 0x01], "vpmovsxwq xmm1{k5}, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x25, 0xca], "vpmovsxdq ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x25, 0x0a], "vpmovsxdq ymm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x25, 0x4a, 0x01], "vpmovsxdq ymm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x25, 0xca], "vpmovsxdq ymm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x25, 0xca], "vpmovsxdq ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x25, 0x0a], "vpmovsxdq ymm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x25, 0x0a], "vpmovsxdq ymm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x25, 0x4a, 0x01], "vpmovsxdq ymm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x25, 0x4a, 0x01], "vpmovsxdq ymm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x25, 0xca], "vpmovsxdq zmm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x25, 0x0a], "vpmovsxdq zmm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x25, 0x4a, 0x01], "vpmovsxdq zmm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x25, 0xca], "vpmovsxdq zmm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x25, 0xca], "vpmovsxdq zmm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x25, 0x0a], "vpmovsxdq zmm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x25, 0x0a], "vpmovsxdq zmm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x25, 0x4a, 0x01], "vpmovsxdq zmm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x25, 0x4a, 0x01], "vpmovsxdq zmm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x25, 0xca], "vpmovsxdq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x25, 0x0a], "vpmovsxdq xmm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x25, 0x4a, 0x01], "vpmovsxdq xmm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x25, 0xca], "vpmovsxdq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x25, 0xca], "vpmovsxdq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x25, 0x0a], "vpmovsxdq xmm1, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x25, 0x0a], "vpmovsxdq xmm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x25, 0x4a, 0x01], "vpmovsxdq xmm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x25, 0x4a, 0x01], "vpmovsxdq xmm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x26, 0xca], "vptestmw k1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x26, 0xca], "vptestmw k1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x26, 0x0a], "vptestmw k1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x26, 0x0a], "vptestmw k1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x26, 0x4a, 0x01], "vptestmw k1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x26, 0x4a, 0x01], "vptestmw k1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x26, 0xca], "vptestmb k1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x26, 0xca], "vptestmb k1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x26, 0x0a], "vptestmb k1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x26, 0x0a], "vptestmb k1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x26, 0x4a, 0x01], "vptestmb k1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x26, 0x4a, 0x01], "vptestmb k1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x26, 0xca], "vptestmw k1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x26, 0xca], "vptestmw k1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x26, 0x0a], "vptestmw k1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x26, 0x0a], "vptestmw k1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x26, 0x4a, 0x01], "vptestmw k1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x26, 0x4a, 0x01], "vptestmw k1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x26, 0xca], "vptestmw k1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x26, 0xca], "vptestmw k1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x26, 0x0a], "vptestmw k1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x26, 0x0a], "vptestmw k1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x26, 0x4a, 0x01], "vptestmw k1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x26, 0x4a, 0x01], "vptestmw k1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x26, 0xca], "vptestmb k1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x26, 0xca], "vptestmb k1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x26, 0x0a], "vptestmb k1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x26, 0x0a], "vptestmb k1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x26, 0x4a, 0x01], "vptestmb k1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x26, 0x4a, 0x01], "vptestmb k1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x26, 0xca], "vptestmb k1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x26, 0xca], "vptestmb k1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x26, 0x0a], "vptestmb k1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x26, 0x0a], "vptestmb k1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x26, 0x4a, 0x01], "vptestmb k1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x26, 0x4a, 0x01], "vptestmb k1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x27, 0x0a], "vptestmq k1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x27, 0x0a], "vptestmq k1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x27, 0x4a, 0x01], "vptestmq k1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x27, 0x4a, 0x01], "vptestmq k1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x27, 0xca], "vptestmq k1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x27, 0xca], "vptestmq k1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x27, 0x0a], "vptestmq k1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x27, 0x0a], "vptestmq k1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x27, 0x4a, 0x01], "vptestmq k1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x27, 0x4a, 0x01], "vptestmq k1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x27, 0x0a], "vptestmd k1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x27, 0x0a], "vptestmd k1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x27, 0x4a, 0x01], "vptestmd k1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x27, 0x4a, 0x01], "vptestmd k1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x27, 0xca], "vptestmd k1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x27, 0xca], "vptestmd k1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x27, 0x0a], "vptestmd k1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x27, 0x0a], "vptestmd k1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x27, 0x4a, 0x01], "vptestmd k1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x27, 0x4a, 0x01], "vptestmd k1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x27, 0x0a], "vptestmq k1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x27, 0x0a], "vptestmq k1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x27, 0x4a, 0x01], "vptestmq k1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x27, 0x4a, 0x01], "vptestmq k1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x27, 0x0a], "vptestmq k1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x27, 0x0a], "vptestmq k1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x27, 0x4a, 0x01], "vptestmq k1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x27, 0x4a, 0x01], "vptestmq k1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x27, 0xca], "vptestmq k1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x27, 0xca], "vptestmq k1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x27, 0x0a], "vptestmq k1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x27, 0x0a], "vptestmq k1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x27, 0x4a, 0x01], "vptestmq k1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x27, 0x4a, 0x01], "vptestmq k1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x27, 0xca], "vptestmq k1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x27, 0xca], "vptestmq k1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x27, 0x0a], "vptestmq k1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x27, 0x0a], "vptestmq k1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x27, 0x4a, 0x01], "vptestmq k1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x27, 0x4a, 0x01], "vptestmq k1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x27, 0x0a], "vptestmd k1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x27, 0x0a], "vptestmd k1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x27, 0x4a, 0x01], "vptestmd k1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x27, 0x4a, 0x01], "vptestmd k1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x27, 0x0a], "vptestmd k1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x27, 0x0a], "vptestmd k1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x27, 0x4a, 0x01], "vptestmd k1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x27, 0x4a, 0x01], "vptestmd k1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x27, 0xca], "vptestmd k1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x27, 0xca], "vptestmd k1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x27, 0x0a], "vptestmd k1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x27, 0x0a], "vptestmd k1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x27, 0x4a, 0x01], "vptestmd k1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x27, 0x4a, 0x01], "vptestmd k1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x27, 0xca], "vptestmd k1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x27, 0xca], "vptestmd k1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x27, 0x0a], "vptestmd k1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x27, 0x0a], "vptestmd k1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x27, 0x4a, 0x01], "vptestmd k1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x27, 0x4a, 0x01], "vptestmd k1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x28, 0x0a], "vpmuldq ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x28, 0x4a, 0x01], "vpmuldq ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x28, 0x0a], "vpmuldq ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x28, 0x0a], "vpmuldq ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x28, 0x4a, 0x01], "vpmuldq ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x28, 0x4a, 0x01], "vpmuldq ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x28, 0xca], "vpmuldq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x28, 0x0a], "vpmuldq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x28, 0x4a, 0x01], "vpmuldq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x28, 0xca], "vpmuldq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x28, 0xca], "vpmuldq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x28, 0x0a], "vpmuldq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x28, 0x0a], "vpmuldq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x28, 0x4a, 0x01], "vpmuldq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x28, 0x4a, 0x01], "vpmuldq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x28, 0x0a], "vpmuldq zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x28, 0x4a, 0x01], "vpmuldq zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x28, 0x0a], "vpmuldq zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x28, 0x0a], "vpmuldq zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x28, 0x4a, 0x01], "vpmuldq zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x28, 0x4a, 0x01], "vpmuldq zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x28, 0x0a], "vpmuldq xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x28, 0x4a, 0x01], "vpmuldq xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x28, 0x0a], "vpmuldq xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x28, 0x0a], "vpmuldq xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x28, 0x4a, 0x01], "vpmuldq xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x28, 0x4a, 0x01], "vpmuldq xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x28, 0xca], "vpmuldq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x28, 0x0a], "vpmuldq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x28, 0x4a, 0x01], "vpmuldq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x28, 0xca], "vpmuldq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x28, 0xca], "vpmuldq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x28, 0x0a], "vpmuldq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x28, 0x0a], "vpmuldq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x28, 0x4a, 0x01], "vpmuldq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x28, 0x4a, 0x01], "vpmuldq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x28, 0xca], "vpmuldq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x28, 0x0a], "vpmuldq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x28, 0x4a, 0x01], "vpmuldq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x28, 0xca], "vpmuldq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x28, 0xca], "vpmuldq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x28, 0x0a], "vpmuldq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x28, 0x0a], "vpmuldq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x28, 0x4a, 0x01], "vpmuldq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x28, 0x4a, 0x01], "vpmuldq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x29, 0x0a], "vpcmpeqq k1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x29, 0x0a], "vpcmpeqq k1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x29, 0x4a, 0x01], "vpcmpeqq k1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x29, 0x4a, 0x01], "vpcmpeqq k1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x29, 0xca], "vpcmpeqq k1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x29, 0xca], "vpcmpeqq k1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x29, 0x0a], "vpcmpeqq k1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x29, 0x0a], "vpcmpeqq k1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x29, 0x4a, 0x01], "vpcmpeqq k1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x29, 0x4a, 0x01], "vpcmpeqq k1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x29, 0x0a], "vpcmpeqq k1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x29, 0x0a], "vpcmpeqq k1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x29, 0x4a, 0x01], "vpcmpeqq k1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x29, 0x4a, 0x01], "vpcmpeqq k1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x29, 0x0a], "vpcmpeqq k1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x29, 0x0a], "vpcmpeqq k1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x29, 0x4a, 0x01], "vpcmpeqq k1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x29, 0x4a, 0x01], "vpcmpeqq k1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x29, 0xca], "vpcmpeqq k1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x29, 0xca], "vpcmpeqq k1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x29, 0x0a], "vpcmpeqq k1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x29, 0x0a], "vpcmpeqq k1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x29, 0x4a, 0x01], "vpcmpeqq k1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x29, 0x4a, 0x01], "vpcmpeqq k1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x29, 0xca], "vpcmpeqq k1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x29, 0xca], "vpcmpeqq k1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x29, 0x0a], "vpcmpeqq k1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x29, 0x0a], "vpcmpeqq k1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x29, 0x4a, 0x01], "vpcmpeqq k1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x29, 0x4a, 0x01], "vpcmpeqq k1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x2a, 0x0a], "vmovntdqa ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x2a, 0x4a, 0x01], "vmovntdqa ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x2a, 0x0a], "vmovntdqa zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x2a, 0x4a, 0x01], "vmovntdqa zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x2a, 0x0a], "vmovntdqa xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x2a, 0x4a, 0x01], "vmovntdqa xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x2b, 0x0a], "vpackusdw ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x2b, 0x4a, 0x01], "vpackusdw ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x2b, 0x0a], "vpackusdw ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x2b, 0x0a], "vpackusdw ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x2b, 0x4a, 0x01], "vpackusdw ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x2b, 0x4a, 0x01], "vpackusdw ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x2b, 0xca], "vpackusdw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x2b, 0x0a], "vpackusdw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x2b, 0x4a, 0x01], "vpackusdw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x2b, 0xca], "vpackusdw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x2b, 0xca], "vpackusdw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x2b, 0x0a], "vpackusdw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x2b, 0x0a], "vpackusdw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x2b, 0x4a, 0x01], "vpackusdw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x2b, 0x4a, 0x01], "vpackusdw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x2b, 0x0a], "vpackusdw zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x2b, 0x4a, 0x01], "vpackusdw zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x2b, 0x0a], "vpackusdw zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x2b, 0x0a], "vpackusdw zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x2b, 0x4a, 0x01], "vpackusdw zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x2b, 0x4a, 0x01], "vpackusdw zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x2b, 0x0a], "vpackusdw xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x2b, 0x4a, 0x01], "vpackusdw xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x2b, 0x0a], "vpackusdw xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x2b, 0x0a], "vpackusdw xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x2b, 0x4a, 0x01], "vpackusdw xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x2b, 0x4a, 0x01], "vpackusdw xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x2b, 0xca], "vpackusdw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x2b, 0x0a], "vpackusdw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x2b, 0x4a, 0x01], "vpackusdw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x2b, 0xca], "vpackusdw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x2b, 0xca], "vpackusdw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x2b, 0x0a], "vpackusdw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x2b, 0x0a], "vpackusdw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x2b, 0x4a, 0x01], "vpackusdw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x2b, 0x4a, 0x01], "vpackusdw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x2b, 0xca], "vpackusdw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x2b, 0x0a], "vpackusdw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x2b, 0x4a, 0x01], "vpackusdw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x2b, 0xca], "vpackusdw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x2b, 0xca], "vpackusdw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x2b, 0x0a], "vpackusdw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x2b, 0x0a], "vpackusdw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x2b, 0x4a, 0x01], "vpackusdw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x2b, 0x4a, 0x01], "vpackusdw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xfd, 0x2c, 0xca], "vscalefpd zmm1{k5}{z}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x78, 0x2c, 0xca], "vscalefpd zmm1{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x7d, 0x2c, 0xca], "vscalefpd zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x2c, 0xca], "vscalefpd zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x2c, 0x0a], "vscalefpd ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x2c, 0x4a, 0x01], "vscalefpd ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x2c, 0xca], "vscalefpd zmm1{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x2c, 0xca], "vscalefpd zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x2c, 0x0a], "vscalefpd ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x2c, 0x0a], "vscalefpd ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x2c, 0x4a, 0x01], "vscalefpd ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x2c, 0x4a, 0x01], "vscalefpd ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x2c, 0xca], "vscalefpd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x2c, 0x0a], "vscalefpd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x2c, 0x4a, 0x01], "vscalefpd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x2c, 0xca], "vscalefpd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x2c, 0xca], "vscalefpd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x2c, 0x0a], "vscalefpd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x2c, 0x0a], "vscalefpd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x2c, 0x4a, 0x01], "vscalefpd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x2c, 0x4a, 0x01], "vscalefpd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xfd, 0x2c, 0xca], "vscalefps zmm1{k5}{z}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x78, 0x2c, 0xca], "vscalefps zmm1{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x7d, 0x2c, 0xca], "vscalefps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x2c, 0xca], "vscalefps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x2c, 0x0a], "vscalefps ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x2c, 0x4a, 0x01], "vscalefps ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x2c, 0xca], "vscalefps zmm1{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x2c, 0xca], "vscalefps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x2c, 0x0a], "vscalefps ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x2c, 0x0a], "vscalefps ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x2c, 0x4a, 0x01], "vscalefps ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x2c, 0x4a, 0x01], "vscalefps ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x2c, 0xca], "vscalefps ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x2c, 0x0a], "vscalefps ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x2c, 0x4a, 0x01], "vscalefps ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x2c, 0xca], "vscalefps ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x2c, 0xca], "vscalefps ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x2c, 0x0a], "vscalefps ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x2c, 0x0a], "vscalefps ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x2c, 0x4a, 0x01], "vscalefps ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x2c, 0x4a, 0x01], "vscalefps ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x2c, 0xca], "vscalefpd zmm1{k5}{z}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x2c, 0x0a], "vscalefpd zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x2c, 0x4a, 0x01], "vscalefpd zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x2c, 0xca], "vscalefpd zmm1{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x2c, 0xca], "vscalefpd zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x2c, 0x0a], "vscalefpd zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x2c, 0x0a], "vscalefpd zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x2c, 0x4a, 0x01], "vscalefpd zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x2c, 0x4a, 0x01], "vscalefpd zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x2c, 0xca], "vscalefpd zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x2c, 0x0a], "vscalefpd xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x2c, 0x4a, 0x01], "vscalefpd xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x2c, 0xca], "vscalefpd zmm1{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x2c, 0xca], "vscalefpd zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x2c, 0x0a], "vscalefpd xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x2c, 0x0a], "vscalefpd xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x2c, 0x4a, 0x01], "vscalefpd xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x2c, 0x4a, 0x01], "vscalefpd xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x2c, 0xca], "vscalefpd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x2c, 0x0a], "vscalefpd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x2c, 0x4a, 0x01], "vscalefpd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x2c, 0xca], "vscalefpd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x2c, 0xca], "vscalefpd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x2c, 0x0a], "vscalefpd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x2c, 0x0a], "vscalefpd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x2c, 0x4a, 0x01], "vscalefpd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x2c, 0x4a, 0x01], "vscalefpd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x2c, 0xca], "vscalefpd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x2c, 0x0a], "vscalefpd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x2c, 0x4a, 0x01], "vscalefpd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x2c, 0xca], "vscalefpd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x2c, 0xca], "vscalefpd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x2c, 0x0a], "vscalefpd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x2c, 0x0a], "vscalefpd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x2c, 0x4a, 0x01], "vscalefpd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x2c, 0x4a, 0x01], "vscalefpd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x2c, 0xca], "vscalefps zmm1{k5}{z}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x2c, 0x0a], "vscalefps zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x2c, 0x4a, 0x01], "vscalefps zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x2c, 0xca], "vscalefps zmm1{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x2c, 0xca], "vscalefps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x2c, 0x0a], "vscalefps zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x2c, 0x0a], "vscalefps zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x2c, 0x4a, 0x01], "vscalefps zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x2c, 0x4a, 0x01], "vscalefps zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x2c, 0xca], "vscalefps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x2c, 0x0a], "vscalefps xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x2c, 0x4a, 0x01], "vscalefps xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x2c, 0xca], "vscalefps zmm1{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x2c, 0xca], "vscalefps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x2c, 0x0a], "vscalefps xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x2c, 0x0a], "vscalefps xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x2c, 0x4a, 0x01], "vscalefps xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x2c, 0x4a, 0x01], "vscalefps xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x2c, 0xca], "vscalefps zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x2c, 0x0a], "vscalefps zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x2c, 0x4a, 0x01], "vscalefps zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x2c, 0xca], "vscalefps zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x2c, 0xca], "vscalefps zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x2c, 0x0a], "vscalefps zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x2c, 0x0a], "vscalefps zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x2c, 0x4a, 0x01], "vscalefps zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x2c, 0x4a, 0x01], "vscalefps zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x2c, 0xca], "vscalefps xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x2c, 0x0a], "vscalefps xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x2c, 0x4a, 0x01], "vscalefps xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x2c, 0xca], "vscalefps xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x2c, 0xca], "vscalefps xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x2c, 0x0a], "vscalefps xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x2c, 0x0a], "vscalefps xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x2c, 0x4a, 0x01], "vscalefps xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x2c, 0x4a, 0x01], "vscalefps xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xfd, 0x2d, 0xca], "vscalefsd xmm1{k5}{z}{rz-sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x78, 0x2d, 0xca], "vscalefsd xmm1{rz-sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x7d, 0x2d, 0xca], "vscalefsd xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x2d, 0xca], "vscalefsd xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x2d, 0xca], "vscalefsd xmm1{rd-sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x2d, 0xca], "vscalefsd xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x2d, 0xca], "vscalefsd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x2d, 0x0a], "vscalefsd xmm1{k5}{z}, xmm0, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x2d, 0x4a, 0x01], "vscalefsd xmm1{k5}{z}, xmm0, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x2d, 0xca], "vscalefsd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x2d, 0xca], "vscalefsd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x2d, 0x0a], "vscalefsd xmm1, xmm0, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x2d, 0x0a], "vscalefsd xmm1{k5}, xmm0, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x2d, 0x4a, 0x01], "vscalefsd xmm1, xmm0, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x2d, 0x4a, 0x01], "vscalefsd xmm1{k5}, xmm0, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xfd, 0x2d, 0xca], "vscalefss xmm1{k5}{z}{rz-sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x78, 0x2d, 0xca], "vscalefss xmm1{rz-sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x7d, 0x2d, 0xca], "vscalefss xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x2d, 0xca], "vscalefss xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x2d, 0xca], "vscalefss xmm1{rd-sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x2d, 0xca], "vscalefss xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x2d, 0xca], "vscalefss xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x2d, 0x0a], "vscalefss xmm1{k5}{z}, xmm0, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x2d, 0x4a, 0x01], "vscalefss xmm1{k5}{z}, xmm0, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x2d, 0xca], "vscalefss xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x2d, 0xca], "vscalefss xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x2d, 0x0a], "vscalefss xmm1, xmm0, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x2d, 0x0a], "vscalefss xmm1{k5}, xmm0, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x2d, 0x4a, 0x01], "vscalefss xmm1, xmm0, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x2d, 0x4a, 0x01], "vscalefss xmm1{k5}, xmm0, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x2d, 0xca], "vscalefsd xmm1{k5}{z}{ru-sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x2d, 0xca], "vscalefsd xmm1{ru-sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x2d, 0xca], "vscalefsd xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x2d, 0xca], "vscalefsd xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x2d, 0xca], "vscalefsd xmm1{rne-sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x2d, 0xca], "vscalefsd xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x2d, 0xca], "vscalefss xmm1{k5}{z}{ru-sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x2d, 0xca], "vscalefss xmm1{ru-sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x2d, 0xca], "vscalefss xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x2d, 0xca], "vscalefss xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x2d, 0xca], "vscalefss xmm1{rne-sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x2d, 0xca], "vscalefss xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x30, 0xca], "vpmovzxbw ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x30, 0x0a], "vpmovzxbw ymm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x30, 0x4a, 0x01], "vpmovzxbw ymm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x30, 0xca], "vpmovzxbw ymm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x30, 0xca], "vpmovzxbw ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x30, 0x0a], "vpmovzxbw ymm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x30, 0x0a], "vpmovzxbw ymm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x30, 0x4a, 0x01], "vpmovzxbw ymm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x30, 0x4a, 0x01], "vpmovzxbw ymm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x30, 0xca], "vpmovzxbw zmm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x30, 0x0a], "vpmovzxbw zmm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x30, 0x4a, 0x01], "vpmovzxbw zmm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x30, 0xca], "vpmovzxbw zmm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x30, 0xca], "vpmovzxbw zmm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x30, 0x0a], "vpmovzxbw zmm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x30, 0x0a], "vpmovzxbw zmm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x30, 0x4a, 0x01], "vpmovzxbw zmm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x30, 0x4a, 0x01], "vpmovzxbw zmm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x30, 0xca], "vpmovzxbw xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x30, 0x0a], "vpmovzxbw xmm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x30, 0x4a, 0x01], "vpmovzxbw xmm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x30, 0xca], "vpmovzxbw xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x30, 0xca], "vpmovzxbw xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x30, 0x0a], "vpmovzxbw xmm1, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x30, 0x0a], "vpmovzxbw xmm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x30, 0x4a, 0x01], "vpmovzxbw xmm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x30, 0x4a, 0x01], "vpmovzxbw xmm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x31, 0xca], "vpmovzxbd ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x31, 0x0a], "vpmovzxbd ymm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x31, 0x4a, 0x01], "vpmovzxbd ymm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x31, 0xca], "vpmovzxbd ymm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x31, 0xca], "vpmovzxbd ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x31, 0x0a], "vpmovzxbd ymm1, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x31, 0x0a], "vpmovzxbd ymm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x31, 0x4a, 0x01], "vpmovzxbd ymm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x31, 0x4a, 0x01], "vpmovzxbd ymm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x31, 0xca], "vpmovzxbd zmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x31, 0x0a], "vpmovzxbd zmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x31, 0x4a, 0x01], "vpmovzxbd zmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x31, 0xca], "vpmovzxbd zmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x31, 0xca], "vpmovzxbd zmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x31, 0x0a], "vpmovzxbd zmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x31, 0x0a], "vpmovzxbd zmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x31, 0x4a, 0x01], "vpmovzxbd zmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x31, 0x4a, 0x01], "vpmovzxbd zmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x31, 0xca], "vpmovzxbd xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x31, 0x0a], "vpmovzxbd xmm1{k5}{z}, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x31, 0x4a, 0x01], "vpmovzxbd xmm1{k5}{z}, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x31, 0xca], "vpmovzxbd xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x31, 0xca], "vpmovzxbd xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x31, 0x0a], "vpmovzxbd xmm1, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x31, 0x0a], "vpmovzxbd xmm1{k5}, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x31, 0x4a, 0x01], "vpmovzxbd xmm1, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x31, 0x4a, 0x01], "vpmovzxbd xmm1{k5}, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x32, 0xca], "vpmovzxbq ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x32, 0x0a], "vpmovzxbq ymm1{k5}{z}, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x32, 0x4a, 0x01], "vpmovzxbq ymm1{k5}{z}, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x32, 0xca], "vpmovzxbq ymm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x32, 0xca], "vpmovzxbq ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x32, 0x0a], "vpmovzxbq ymm1, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x32, 0x0a], "vpmovzxbq ymm1{k5}, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x32, 0x4a, 0x01], "vpmovzxbq ymm1, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x32, 0x4a, 0x01], "vpmovzxbq ymm1{k5}, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x32, 0xca], "vpmovzxbq zmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x32, 0x0a], "vpmovzxbq zmm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x32, 0x4a, 0x01], "vpmovzxbq zmm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x32, 0xca], "vpmovzxbq zmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x32, 0xca], "vpmovzxbq zmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x32, 0x0a], "vpmovzxbq zmm1, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x32, 0x0a], "vpmovzxbq zmm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x32, 0x4a, 0x01], "vpmovzxbq zmm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x32, 0x4a, 0x01], "vpmovzxbq zmm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x32, 0xca], "vpmovzxbq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x32, 0x0a], "vpmovzxbq xmm1{k5}{z}, word [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x32, 0x4a, 0x01], "vpmovzxbq xmm1{k5}{z}, word [edx + 0x2]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x32, 0xca], "vpmovzxbq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x32, 0xca], "vpmovzxbq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x32, 0x0a], "vpmovzxbq xmm1, word [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x32, 0x0a], "vpmovzxbq xmm1{k5}, word [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x32, 0x4a, 0x01], "vpmovzxbq xmm1, word [edx + 0x2]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x32, 0x4a, 0x01], "vpmovzxbq xmm1{k5}, word [edx + 0x2]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x33, 0xca], "vpmovzxwd ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x33, 0x0a], "vpmovzxwd ymm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x33, 0x4a, 0x01], "vpmovzxwd ymm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x33, 0xca], "vpmovzxwd ymm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x33, 0xca], "vpmovzxwd ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x33, 0x0a], "vpmovzxwd ymm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x33, 0x0a], "vpmovzxwd ymm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x33, 0x4a, 0x01], "vpmovzxwd ymm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x33, 0x4a, 0x01], "vpmovzxwd ymm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x33, 0xca], "vpmovzxwd zmm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x33, 0x0a], "vpmovzxwd zmm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x33, 0x4a, 0x01], "vpmovzxwd zmm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x33, 0xca], "vpmovzxwd zmm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x33, 0xca], "vpmovzxwd zmm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x33, 0x0a], "vpmovzxwd zmm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x33, 0x0a], "vpmovzxwd zmm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x33, 0x4a, 0x01], "vpmovzxwd zmm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x33, 0x4a, 0x01], "vpmovzxwd zmm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x33, 0xca], "vpmovzxwd xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x33, 0x0a], "vpmovzxwd xmm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x33, 0x4a, 0x01], "vpmovzxwd xmm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x33, 0xca], "vpmovzxwd xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x33, 0xca], "vpmovzxwd xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x33, 0x0a], "vpmovzxwd xmm1, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x33, 0x0a], "vpmovzxwd xmm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x33, 0x4a, 0x01], "vpmovzxwd xmm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x33, 0x4a, 0x01], "vpmovzxwd xmm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x34, 0xca], "vpmovzxwq ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x34, 0x0a], "vpmovzxwq ymm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x34, 0x4a, 0x01], "vpmovzxwq ymm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x34, 0xca], "vpmovzxwq ymm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x34, 0xca], "vpmovzxwq ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x34, 0x0a], "vpmovzxwq ymm1, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x34, 0x0a], "vpmovzxwq ymm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x34, 0x4a, 0x01], "vpmovzxwq ymm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x34, 0x4a, 0x01], "vpmovzxwq ymm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x34, 0xca], "vpmovzxwq zmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x34, 0x0a], "vpmovzxwq zmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x34, 0x4a, 0x01], "vpmovzxwq zmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x34, 0xca], "vpmovzxwq zmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x34, 0xca], "vpmovzxwq zmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x34, 0x0a], "vpmovzxwq zmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x34, 0x0a], "vpmovzxwq zmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x34, 0x4a, 0x01], "vpmovzxwq zmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x34, 0x4a, 0x01], "vpmovzxwq zmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x34, 0xca], "vpmovzxwq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x34, 0x0a], "vpmovzxwq xmm1{k5}{z}, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x34, 0x4a, 0x01], "vpmovzxwq xmm1{k5}{z}, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x34, 0xca], "vpmovzxwq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x34, 0xca], "vpmovzxwq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x34, 0x0a], "vpmovzxwq xmm1, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x34, 0x0a], "vpmovzxwq xmm1{k5}, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x34, 0x4a, 0x01], "vpmovzxwq xmm1, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x34, 0x4a, 0x01], "vpmovzxwq xmm1{k5}, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x35, 0xca], "vpmovzxdq ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x35, 0x0a], "vpmovzxdq ymm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x35, 0x4a, 0x01], "vpmovzxdq ymm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x35, 0xca], "vpmovzxdq ymm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x35, 0xca], "vpmovzxdq ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x35, 0x0a], "vpmovzxdq ymm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x35, 0x0a], "vpmovzxdq ymm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x35, 0x4a, 0x01], "vpmovzxdq ymm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x35, 0x4a, 0x01], "vpmovzxdq ymm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x35, 0xca], "vpmovzxdq zmm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x35, 0x0a], "vpmovzxdq zmm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x35, 0x4a, 0x01], "vpmovzxdq zmm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x35, 0xca], "vpmovzxdq zmm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x35, 0xca], "vpmovzxdq zmm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x35, 0x0a], "vpmovzxdq zmm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x35, 0x0a], "vpmovzxdq zmm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x35, 0x4a, 0x01], "vpmovzxdq zmm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x35, 0x4a, 0x01], "vpmovzxdq zmm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x35, 0xca], "vpmovzxdq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x35, 0x0a], "vpmovzxdq xmm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x35, 0x4a, 0x01], "vpmovzxdq xmm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x35, 0xca], "vpmovzxdq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x35, 0xca], "vpmovzxdq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x35, 0x0a], "vpmovzxdq xmm1, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x35, 0x0a], "vpmovzxdq xmm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x35, 0x4a, 0x01], "vpmovzxdq xmm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x35, 0x4a, 0x01], "vpmovzxdq xmm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x36, 0x0a], "vpermq ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x36, 0x4a, 0x01], "vpermq ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x36, 0x0a], "vpermq ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x36, 0x0a], "vpermq ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x36, 0x4a, 0x01], "vpermq ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x36, 0x4a, 0x01], "vpermq ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x36, 0xca], "vpermq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x36, 0x0a], "vpermq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x36, 0x4a, 0x01], "vpermq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x36, 0xca], "vpermq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x36, 0xca], "vpermq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x36, 0x0a], "vpermq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x36, 0x0a], "vpermq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x36, 0x4a, 0x01], "vpermq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x36, 0x4a, 0x01], "vpermq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x36, 0x0a], "vpermd ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x36, 0x4a, 0x01], "vpermd ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x36, 0x0a], "vpermd ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x36, 0x0a], "vpermd ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x36, 0x4a, 0x01], "vpermd ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x36, 0x4a, 0x01], "vpermd ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x36, 0xca], "vpermd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x36, 0x0a], "vpermd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x36, 0x4a, 0x01], "vpermd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x36, 0xca], "vpermd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x36, 0xca], "vpermd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x36, 0x0a], "vpermd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x36, 0x0a], "vpermd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x36, 0x4a, 0x01], "vpermd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x36, 0x4a, 0x01], "vpermd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x36, 0x0a], "vpermq zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x36, 0x4a, 0x01], "vpermq zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x36, 0x0a], "vpermq zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x36, 0x0a], "vpermq zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x36, 0x4a, 0x01], "vpermq zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x36, 0x4a, 0x01], "vpermq zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x36, 0xca], "vpermq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x36, 0x0a], "vpermq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x36, 0x4a, 0x01], "vpermq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x36, 0xca], "vpermq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x36, 0xca], "vpermq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x36, 0x0a], "vpermq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x36, 0x0a], "vpermq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x36, 0x4a, 0x01], "vpermq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x36, 0x4a, 0x01], "vpermq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x36, 0x0a], "vpermd zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x36, 0x4a, 0x01], "vpermd zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x36, 0x0a], "vpermd zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x36, 0x0a], "vpermd zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x36, 0x4a, 0x01], "vpermd zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x36, 0x4a, 0x01], "vpermd zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x36, 0xca], "vpermd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x36, 0x0a], "vpermd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x36, 0x4a, 0x01], "vpermd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x36, 0xca], "vpermd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x36, 0xca], "vpermd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x36, 0x0a], "vpermd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x36, 0x0a], "vpermd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x36, 0x4a, 0x01], "vpermd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x36, 0x4a, 0x01], "vpermd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x37, 0x0a], "vpcmpgtq k1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x37, 0x0a], "vpcmpgtq k1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x37, 0x4a, 0x01], "vpcmpgtq k1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x37, 0x4a, 0x01], "vpcmpgtq k1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x37, 0xca], "vpcmpgtq k1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x37, 0xca], "vpcmpgtq k1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x37, 0x0a], "vpcmpgtq k1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x37, 0x0a], "vpcmpgtq k1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x37, 0x4a, 0x01], "vpcmpgtq k1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x37, 0x4a, 0x01], "vpcmpgtq k1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x37, 0x0a], "vpcmpgtq k1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x37, 0x0a], "vpcmpgtq k1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x37, 0x4a, 0x01], "vpcmpgtq k1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x37, 0x4a, 0x01], "vpcmpgtq k1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x37, 0x0a], "vpcmpgtq k1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x37, 0x0a], "vpcmpgtq k1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x37, 0x4a, 0x01], "vpcmpgtq k1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x37, 0x4a, 0x01], "vpcmpgtq k1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x37, 0xca], "vpcmpgtq k1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x37, 0xca], "vpcmpgtq k1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x37, 0x0a], "vpcmpgtq k1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x37, 0x0a], "vpcmpgtq k1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x37, 0x4a, 0x01], "vpcmpgtq k1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x37, 0x4a, 0x01], "vpcmpgtq k1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x37, 0xca], "vpcmpgtq k1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x37, 0xca], "vpcmpgtq k1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x37, 0x0a], "vpcmpgtq k1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x37, 0x0a], "vpcmpgtq k1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x37, 0x4a, 0x01], "vpcmpgtq k1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x37, 0x4a, 0x01], "vpcmpgtq k1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x38, 0xca], "vpminsb ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x38, 0x0a], "vpminsb ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x38, 0x4a, 0x01], "vpminsb ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x38, 0xca], "vpminsb ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x38, 0xca], "vpminsb ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x38, 0x0a], "vpminsb ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x38, 0x0a], "vpminsb ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x38, 0x4a, 0x01], "vpminsb ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x38, 0x4a, 0x01], "vpminsb ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x38, 0xca], "vpminsb zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x38, 0x0a], "vpminsb zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x38, 0x4a, 0x01], "vpminsb zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x38, 0xca], "vpminsb zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x38, 0xca], "vpminsb zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x38, 0x0a], "vpminsb zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x38, 0x0a], "vpminsb zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x38, 0x4a, 0x01], "vpminsb zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x38, 0x4a, 0x01], "vpminsb zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x38, 0xca], "vpminsb xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x38, 0x0a], "vpminsb xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x38, 0x4a, 0x01], "vpminsb xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x38, 0xca], "vpminsb xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x38, 0xca], "vpminsb xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x38, 0x0a], "vpminsb xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x38, 0x0a], "vpminsb xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x38, 0x4a, 0x01], "vpminsb xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x38, 0x4a, 0x01], "vpminsb xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x39, 0x0a], "vpminsq ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x39, 0x4a, 0x01], "vpminsq ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x39, 0x0a], "vpminsq ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x39, 0x0a], "vpminsq ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x39, 0x4a, 0x01], "vpminsq ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x39, 0x4a, 0x01], "vpminsq ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x39, 0xca], "vpminsq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x39, 0x0a], "vpminsq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x39, 0x4a, 0x01], "vpminsq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x39, 0xca], "vpminsq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x39, 0xca], "vpminsq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x39, 0x0a], "vpminsq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x39, 0x0a], "vpminsq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x39, 0x4a, 0x01], "vpminsq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x39, 0x4a, 0x01], "vpminsq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x39, 0x0a], "vpminsd ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x39, 0x4a, 0x01], "vpminsd ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x39, 0x0a], "vpminsd ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x39, 0x0a], "vpminsd ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x39, 0x4a, 0x01], "vpminsd ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x39, 0x4a, 0x01], "vpminsd ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x39, 0xca], "vpminsd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x39, 0x0a], "vpminsd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x39, 0x4a, 0x01], "vpminsd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x39, 0xca], "vpminsd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x39, 0xca], "vpminsd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x39, 0x0a], "vpminsd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x39, 0x0a], "vpminsd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x39, 0x4a, 0x01], "vpminsd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x39, 0x4a, 0x01], "vpminsd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x39, 0x0a], "vpminsq zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x39, 0x4a, 0x01], "vpminsq zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x39, 0x0a], "vpminsq zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x39, 0x0a], "vpminsq zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x39, 0x4a, 0x01], "vpminsq zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x39, 0x4a, 0x01], "vpminsq zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x39, 0x0a], "vpminsq xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x39, 0x4a, 0x01], "vpminsq xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x39, 0x0a], "vpminsq xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x39, 0x0a], "vpminsq xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x39, 0x4a, 0x01], "vpminsq xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x39, 0x4a, 0x01], "vpminsq xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x39, 0xca], "vpminsq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x39, 0x0a], "vpminsq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x39, 0x4a, 0x01], "vpminsq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x39, 0xca], "vpminsq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x39, 0xca], "vpminsq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x39, 0x0a], "vpminsq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x39, 0x0a], "vpminsq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x39, 0x4a, 0x01], "vpminsq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x39, 0x4a, 0x01], "vpminsq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x39, 0xca], "vpminsq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x39, 0x0a], "vpminsq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x39, 0x4a, 0x01], "vpminsq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x39, 0xca], "vpminsq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x39, 0xca], "vpminsq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x39, 0x0a], "vpminsq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x39, 0x0a], "vpminsq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x39, 0x4a, 0x01], "vpminsq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x39, 0x4a, 0x01], "vpminsq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x39, 0x0a], "vpminsd zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x39, 0x4a, 0x01], "vpminsd zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x39, 0x0a], "vpminsd zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x39, 0x0a], "vpminsd zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x39, 0x4a, 0x01], "vpminsd zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x39, 0x4a, 0x01], "vpminsd zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x39, 0x0a], "vpminsd xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x39, 0x4a, 0x01], "vpminsd xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x39, 0x0a], "vpminsd xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x39, 0x0a], "vpminsd xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x39, 0x4a, 0x01], "vpminsd xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x39, 0x4a, 0x01], "vpminsd xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x39, 0xca], "vpminsd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x39, 0x0a], "vpminsd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x39, 0x4a, 0x01], "vpminsd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x39, 0xca], "vpminsd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x39, 0xca], "vpminsd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x39, 0x0a], "vpminsd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x39, 0x0a], "vpminsd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x39, 0x4a, 0x01], "vpminsd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x39, 0x4a, 0x01], "vpminsd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x39, 0xca], "vpminsd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x39, 0x0a], "vpminsd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x39, 0x4a, 0x01], "vpminsd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x39, 0xca], "vpminsd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x39, 0xca], "vpminsd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x39, 0x0a], "vpminsd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x39, 0x0a], "vpminsd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x39, 0x4a, 0x01], "vpminsd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x39, 0x4a, 0x01], "vpminsd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3a, 0xca], "vpminuw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3a, 0x0a], "vpminuw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3a, 0x4a, 0x01], "vpminuw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3a, 0xca], "vpminuw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3a, 0xca], "vpminuw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3a, 0x0a], "vpminuw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3a, 0x0a], "vpminuw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3a, 0x4a, 0x01], "vpminuw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3a, 0x4a, 0x01], "vpminuw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3a, 0xca], "vpminuw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3a, 0x0a], "vpminuw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3a, 0x4a, 0x01], "vpminuw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3a, 0xca], "vpminuw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3a, 0xca], "vpminuw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3a, 0x0a], "vpminuw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3a, 0x0a], "vpminuw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3a, 0x4a, 0x01], "vpminuw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3a, 0x4a, 0x01], "vpminuw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3a, 0xca], "vpminuw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3a, 0x0a], "vpminuw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3a, 0x4a, 0x01], "vpminuw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3a, 0xca], "vpminuw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3a, 0xca], "vpminuw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3a, 0x0a], "vpminuw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3a, 0x0a], "vpminuw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3a, 0x4a, 0x01], "vpminuw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3a, 0x4a, 0x01], "vpminuw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x3b, 0x0a], "vpminuq ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x3b, 0x4a, 0x01], "vpminuq ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x3b, 0x0a], "vpminuq ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x3b, 0x0a], "vpminuq ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x3b, 0x4a, 0x01], "vpminuq ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x3b, 0x4a, 0x01], "vpminuq ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3b, 0xca], "vpminuq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3b, 0x0a], "vpminuq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3b, 0x4a, 0x01], "vpminuq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3b, 0xca], "vpminuq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3b, 0xca], "vpminuq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3b, 0x0a], "vpminuq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3b, 0x0a], "vpminuq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3b, 0x4a, 0x01], "vpminuq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3b, 0x4a, 0x01], "vpminuq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x3b, 0x0a], "vpminud ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x3b, 0x4a, 0x01], "vpminud ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x3b, 0x0a], "vpminud ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x3b, 0x0a], "vpminud ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x3b, 0x4a, 0x01], "vpminud ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x3b, 0x4a, 0x01], "vpminud ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x3b, 0xca], "vpminud ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x3b, 0x0a], "vpminud ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x3b, 0x4a, 0x01], "vpminud ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x3b, 0xca], "vpminud ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x3b, 0xca], "vpminud ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x3b, 0x0a], "vpminud ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x3b, 0x0a], "vpminud ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x3b, 0x4a, 0x01], "vpminud ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x3b, 0x4a, 0x01], "vpminud ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x3b, 0x0a], "vpminuq zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x3b, 0x4a, 0x01], "vpminuq zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x3b, 0x0a], "vpminuq zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x3b, 0x0a], "vpminuq zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x3b, 0x4a, 0x01], "vpminuq zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x3b, 0x4a, 0x01], "vpminuq zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x3b, 0x0a], "vpminuq xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x3b, 0x4a, 0x01], "vpminuq xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x3b, 0x0a], "vpminuq xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x3b, 0x0a], "vpminuq xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x3b, 0x4a, 0x01], "vpminuq xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x3b, 0x4a, 0x01], "vpminuq xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3b, 0xca], "vpminuq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3b, 0x0a], "vpminuq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3b, 0x4a, 0x01], "vpminuq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3b, 0xca], "vpminuq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3b, 0xca], "vpminuq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3b, 0x0a], "vpminuq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3b, 0x0a], "vpminuq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3b, 0x4a, 0x01], "vpminuq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3b, 0x4a, 0x01], "vpminuq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3b, 0xca], "vpminuq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3b, 0x0a], "vpminuq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3b, 0x4a, 0x01], "vpminuq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3b, 0xca], "vpminuq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3b, 0xca], "vpminuq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3b, 0x0a], "vpminuq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3b, 0x0a], "vpminuq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3b, 0x4a, 0x01], "vpminuq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3b, 0x4a, 0x01], "vpminuq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x3b, 0x0a], "vpminud zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x3b, 0x4a, 0x01], "vpminud zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x3b, 0x0a], "vpminud zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x3b, 0x0a], "vpminud zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x3b, 0x4a, 0x01], "vpminud zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x3b, 0x4a, 0x01], "vpminud zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x3b, 0x0a], "vpminud xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x3b, 0x4a, 0x01], "vpminud xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x3b, 0x0a], "vpminud xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x3b, 0x0a], "vpminud xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x3b, 0x4a, 0x01], "vpminud xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x3b, 0x4a, 0x01], "vpminud xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x3b, 0xca], "vpminud zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x3b, 0x0a], "vpminud zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x3b, 0x4a, 0x01], "vpminud zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x3b, 0xca], "vpminud zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x3b, 0xca], "vpminud zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x3b, 0x0a], "vpminud zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x3b, 0x0a], "vpminud zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x3b, 0x4a, 0x01], "vpminud zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x3b, 0x4a, 0x01], "vpminud zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x3b, 0xca], "vpminud xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x3b, 0x0a], "vpminud xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x3b, 0x4a, 0x01], "vpminud xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x3b, 0xca], "vpminud xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x3b, 0xca], "vpminud xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x3b, 0x0a], "vpminud xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x3b, 0x0a], "vpminud xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x3b, 0x4a, 0x01], "vpminud xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x3b, 0x4a, 0x01], "vpminud xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3c, 0xca], "vpmaxsb ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3c, 0x0a], "vpmaxsb ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3c, 0x4a, 0x01], "vpmaxsb ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3c, 0xca], "vpmaxsb ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3c, 0xca], "vpmaxsb ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3c, 0x0a], "vpmaxsb ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3c, 0x0a], "vpmaxsb ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3c, 0x4a, 0x01], "vpmaxsb ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3c, 0x4a, 0x01], "vpmaxsb ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3c, 0xca], "vpmaxsb zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3c, 0x0a], "vpmaxsb zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3c, 0x4a, 0x01], "vpmaxsb zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3c, 0xca], "vpmaxsb zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3c, 0xca], "vpmaxsb zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3c, 0x0a], "vpmaxsb zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3c, 0x0a], "vpmaxsb zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3c, 0x4a, 0x01], "vpmaxsb zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3c, 0x4a, 0x01], "vpmaxsb zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3c, 0xca], "vpmaxsb xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3c, 0x0a], "vpmaxsb xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3c, 0x4a, 0x01], "vpmaxsb xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3c, 0xca], "vpmaxsb xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3c, 0xca], "vpmaxsb xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3c, 0x0a], "vpmaxsb xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3c, 0x0a], "vpmaxsb xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3c, 0x4a, 0x01], "vpmaxsb xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3c, 0x4a, 0x01], "vpmaxsb xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x3d, 0x0a], "vpmaxsq ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x3d, 0x4a, 0x01], "vpmaxsq ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x3d, 0x0a], "vpmaxsq ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x3d, 0x0a], "vpmaxsq ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x3d, 0x4a, 0x01], "vpmaxsq ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x3d, 0x4a, 0x01], "vpmaxsq ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3d, 0xca], "vpmaxsq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3d, 0x0a], "vpmaxsq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3d, 0x4a, 0x01], "vpmaxsq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3d, 0xca], "vpmaxsq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3d, 0xca], "vpmaxsq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3d, 0x0a], "vpmaxsq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3d, 0x0a], "vpmaxsq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3d, 0x4a, 0x01], "vpmaxsq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3d, 0x4a, 0x01], "vpmaxsq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x3d, 0x0a], "vpmaxsd ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x3d, 0x4a, 0x01], "vpmaxsd ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x3d, 0x0a], "vpmaxsd ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x3d, 0x0a], "vpmaxsd ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x3d, 0x4a, 0x01], "vpmaxsd ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x3d, 0x4a, 0x01], "vpmaxsd ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x3d, 0xca], "vpmaxsd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x3d, 0x0a], "vpmaxsd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x3d, 0x4a, 0x01], "vpmaxsd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x3d, 0xca], "vpmaxsd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x3d, 0xca], "vpmaxsd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x3d, 0x0a], "vpmaxsd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x3d, 0x0a], "vpmaxsd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x3d, 0x4a, 0x01], "vpmaxsd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x3d, 0x4a, 0x01], "vpmaxsd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x3d, 0x0a], "vpmaxsq zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x3d, 0x4a, 0x01], "vpmaxsq zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x3d, 0x0a], "vpmaxsq zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x3d, 0x0a], "vpmaxsq zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x3d, 0x4a, 0x01], "vpmaxsq zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x3d, 0x4a, 0x01], "vpmaxsq zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x3d, 0x0a], "vpmaxsq xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x3d, 0x4a, 0x01], "vpmaxsq xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x3d, 0x0a], "vpmaxsq xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x3d, 0x0a], "vpmaxsq xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x3d, 0x4a, 0x01], "vpmaxsq xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x3d, 0x4a, 0x01], "vpmaxsq xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3d, 0xca], "vpmaxsq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3d, 0x0a], "vpmaxsq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3d, 0x4a, 0x01], "vpmaxsq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3d, 0xca], "vpmaxsq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3d, 0xca], "vpmaxsq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3d, 0x0a], "vpmaxsq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3d, 0x0a], "vpmaxsq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3d, 0x4a, 0x01], "vpmaxsq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3d, 0x4a, 0x01], "vpmaxsq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3d, 0xca], "vpmaxsq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3d, 0x0a], "vpmaxsq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3d, 0x4a, 0x01], "vpmaxsq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3d, 0xca], "vpmaxsq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3d, 0xca], "vpmaxsq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3d, 0x0a], "vpmaxsq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3d, 0x0a], "vpmaxsq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3d, 0x4a, 0x01], "vpmaxsq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3d, 0x4a, 0x01], "vpmaxsq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x3d, 0x0a], "vpmaxsd zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x3d, 0x4a, 0x01], "vpmaxsd zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x3d, 0x0a], "vpmaxsd zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x3d, 0x0a], "vpmaxsd zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x3d, 0x4a, 0x01], "vpmaxsd zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x3d, 0x4a, 0x01], "vpmaxsd zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x3d, 0x0a], "vpmaxsd xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x3d, 0x4a, 0x01], "vpmaxsd xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x3d, 0x0a], "vpmaxsd xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x3d, 0x0a], "vpmaxsd xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x3d, 0x4a, 0x01], "vpmaxsd xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x3d, 0x4a, 0x01], "vpmaxsd xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x3d, 0xca], "vpmaxsd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x3d, 0x0a], "vpmaxsd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x3d, 0x4a, 0x01], "vpmaxsd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x3d, 0xca], "vpmaxsd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x3d, 0xca], "vpmaxsd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x3d, 0x0a], "vpmaxsd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x3d, 0x0a], "vpmaxsd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x3d, 0x4a, 0x01], "vpmaxsd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x3d, 0x4a, 0x01], "vpmaxsd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x3d, 0xca], "vpmaxsd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x3d, 0x0a], "vpmaxsd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x3d, 0x4a, 0x01], "vpmaxsd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x3d, 0xca], "vpmaxsd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x3d, 0xca], "vpmaxsd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x3d, 0x0a], "vpmaxsd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x3d, 0x0a], "vpmaxsd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x3d, 0x4a, 0x01], "vpmaxsd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x3d, 0x4a, 0x01], "vpmaxsd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3e, 0xca], "vpmaxuw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3e, 0x0a], "vpmaxuw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3e, 0x4a, 0x01], "vpmaxuw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3e, 0xca], "vpmaxuw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3e, 0xca], "vpmaxuw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3e, 0x0a], "vpmaxuw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3e, 0x0a], "vpmaxuw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3e, 0x4a, 0x01], "vpmaxuw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3e, 0x4a, 0x01], "vpmaxuw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3e, 0xca], "vpmaxuw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3e, 0x0a], "vpmaxuw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3e, 0x4a, 0x01], "vpmaxuw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3e, 0xca], "vpmaxuw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3e, 0xca], "vpmaxuw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3e, 0x0a], "vpmaxuw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3e, 0x0a], "vpmaxuw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3e, 0x4a, 0x01], "vpmaxuw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3e, 0x4a, 0x01], "vpmaxuw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3e, 0xca], "vpmaxuw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3e, 0x0a], "vpmaxuw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3e, 0x4a, 0x01], "vpmaxuw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3e, 0xca], "vpmaxuw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3e, 0xca], "vpmaxuw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3e, 0x0a], "vpmaxuw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3e, 0x0a], "vpmaxuw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3e, 0x4a, 0x01], "vpmaxuw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3e, 0x4a, 0x01], "vpmaxuw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x3f, 0x0a], "vpmaxuq ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x3f, 0x4a, 0x01], "vpmaxuq ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x3f, 0x0a], "vpmaxuq ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x3f, 0x0a], "vpmaxuq ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x3f, 0x4a, 0x01], "vpmaxuq ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x3f, 0x4a, 0x01], "vpmaxuq ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3f, 0xca], "vpmaxuq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3f, 0x0a], "vpmaxuq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3f, 0x4a, 0x01], "vpmaxuq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3f, 0xca], "vpmaxuq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3f, 0xca], "vpmaxuq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3f, 0x0a], "vpmaxuq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3f, 0x0a], "vpmaxuq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3f, 0x4a, 0x01], "vpmaxuq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3f, 0x4a, 0x01], "vpmaxuq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x3f, 0x0a], "vpmaxud ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x3f, 0x4a, 0x01], "vpmaxud ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x3f, 0x0a], "vpmaxud ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x3f, 0x0a], "vpmaxud ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x3f, 0x4a, 0x01], "vpmaxud ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x3f, 0x4a, 0x01], "vpmaxud ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x3f, 0xca], "vpmaxud ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x3f, 0x0a], "vpmaxud ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x3f, 0x4a, 0x01], "vpmaxud ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x3f, 0xca], "vpmaxud ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x3f, 0xca], "vpmaxud ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x3f, 0x0a], "vpmaxud ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x3f, 0x0a], "vpmaxud ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x3f, 0x4a, 0x01], "vpmaxud ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x3f, 0x4a, 0x01], "vpmaxud ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x3f, 0x0a], "vpmaxuq zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x3f, 0x4a, 0x01], "vpmaxuq zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x3f, 0x0a], "vpmaxuq zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x3f, 0x0a], "vpmaxuq zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x3f, 0x4a, 0x01], "vpmaxuq zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x3f, 0x4a, 0x01], "vpmaxuq zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x3f, 0x0a], "vpmaxuq xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x3f, 0x4a, 0x01], "vpmaxuq xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x3f, 0x0a], "vpmaxuq xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x3f, 0x0a], "vpmaxuq xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x3f, 0x4a, 0x01], "vpmaxuq xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x3f, 0x4a, 0x01], "vpmaxuq xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3f, 0xca], "vpmaxuq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3f, 0x0a], "vpmaxuq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3f, 0x4a, 0x01], "vpmaxuq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3f, 0xca], "vpmaxuq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3f, 0xca], "vpmaxuq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3f, 0x0a], "vpmaxuq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3f, 0x0a], "vpmaxuq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3f, 0x4a, 0x01], "vpmaxuq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3f, 0x4a, 0x01], "vpmaxuq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3f, 0xca], "vpmaxuq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3f, 0x0a], "vpmaxuq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3f, 0x4a, 0x01], "vpmaxuq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3f, 0xca], "vpmaxuq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3f, 0xca], "vpmaxuq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3f, 0x0a], "vpmaxuq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3f, 0x0a], "vpmaxuq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3f, 0x4a, 0x01], "vpmaxuq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3f, 0x4a, 0x01], "vpmaxuq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x3f, 0x0a], "vpmaxud zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x3f, 0x4a, 0x01], "vpmaxud zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x3f, 0x0a], "vpmaxud zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x3f, 0x0a], "vpmaxud zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x3f, 0x4a, 0x01], "vpmaxud zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x3f, 0x4a, 0x01], "vpmaxud zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x3f, 0x0a], "vpmaxud xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x3f, 0x4a, 0x01], "vpmaxud xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x3f, 0x0a], "vpmaxud xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x3f, 0x0a], "vpmaxud xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x3f, 0x4a, 0x01], "vpmaxud xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x3f, 0x4a, 0x01], "vpmaxud xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x3f, 0xca], "vpmaxud zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x3f, 0x0a], "vpmaxud zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x3f, 0x4a, 0x01], "vpmaxud zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x3f, 0xca], "vpmaxud zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x3f, 0xca], "vpmaxud zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x3f, 0x0a], "vpmaxud zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x3f, 0x0a], "vpmaxud zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x3f, 0x4a, 0x01], "vpmaxud zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x3f, 0x4a, 0x01], "vpmaxud zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x3f, 0xca], "vpmaxud xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x3f, 0x0a], "vpmaxud xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x3f, 0x4a, 0x01], "vpmaxud xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x3f, 0xca], "vpmaxud xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x3f, 0xca], "vpmaxud xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x3f, 0x0a], "vpmaxud xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x3f, 0x0a], "vpmaxud xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x3f, 0x4a, 0x01], "vpmaxud xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x3f, 0x4a, 0x01], "vpmaxud xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x40, 0x0a], "vpmullq ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x40, 0x4a, 0x01], "vpmullq ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x40, 0x0a], "vpmullq ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x40, 0x0a], "vpmullq ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x40, 0x4a, 0x01], "vpmullq ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x40, 0x4a, 0x01], "vpmullq ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x40, 0xca], "vpmullq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x40, 0x0a], "vpmullq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x40, 0x4a, 0x01], "vpmullq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x40, 0xca], "vpmullq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x40, 0xca], "vpmullq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x40, 0x0a], "vpmullq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x40, 0x0a], "vpmullq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x40, 0x4a, 0x01], "vpmullq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x40, 0x4a, 0x01], "vpmullq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x40, 0x0a], "vpmulld ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x40, 0x4a, 0x01], "vpmulld ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x40, 0x0a], "vpmulld ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x40, 0x0a], "vpmulld ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x40, 0x4a, 0x01], "vpmulld ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x40, 0x4a, 0x01], "vpmulld ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x40, 0xca], "vpmulld ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x40, 0x0a], "vpmulld ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x40, 0x4a, 0x01], "vpmulld ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x40, 0xca], "vpmulld ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x40, 0xca], "vpmulld ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x40, 0x0a], "vpmulld ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x40, 0x0a], "vpmulld ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x40, 0x4a, 0x01], "vpmulld ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x40, 0x4a, 0x01], "vpmulld ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x40, 0x0a], "vpmullq zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x40, 0x4a, 0x01], "vpmullq zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x40, 0x0a], "vpmullq zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x40, 0x0a], "vpmullq zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x40, 0x4a, 0x01], "vpmullq zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x40, 0x4a, 0x01], "vpmullq zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x40, 0x0a], "vpmullq xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x40, 0x4a, 0x01], "vpmullq xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x40, 0x0a], "vpmullq xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x40, 0x0a], "vpmullq xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x40, 0x4a, 0x01], "vpmullq xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x40, 0x4a, 0x01], "vpmullq xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x40, 0xca], "vpmullq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x40, 0x0a], "vpmullq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x40, 0x4a, 0x01], "vpmullq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x40, 0xca], "vpmullq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x40, 0xca], "vpmullq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x40, 0x0a], "vpmullq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x40, 0x0a], "vpmullq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x40, 0x4a, 0x01], "vpmullq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x40, 0x4a, 0x01], "vpmullq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x40, 0xca], "vpmullq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x40, 0x0a], "vpmullq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x40, 0x4a, 0x01], "vpmullq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x40, 0xca], "vpmullq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x40, 0xca], "vpmullq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x40, 0x0a], "vpmullq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x40, 0x0a], "vpmullq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x40, 0x4a, 0x01], "vpmullq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x40, 0x4a, 0x01], "vpmullq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x40, 0x0a], "vpmulld zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x40, 0x4a, 0x01], "vpmulld zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x40, 0x0a], "vpmulld zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x40, 0x0a], "vpmulld zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x40, 0x4a, 0x01], "vpmulld zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x40, 0x4a, 0x01], "vpmulld zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x40, 0x0a], "vpmulld xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x40, 0x4a, 0x01], "vpmulld xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x40, 0x0a], "vpmulld xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x40, 0x0a], "vpmulld xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x40, 0x4a, 0x01], "vpmulld xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x40, 0x4a, 0x01], "vpmulld xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x40, 0xca], "vpmulld zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x40, 0x0a], "vpmulld zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x40, 0x4a, 0x01], "vpmulld zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x40, 0xca], "vpmulld zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x40, 0xca], "vpmulld zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x40, 0x0a], "vpmulld zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x40, 0x0a], "vpmulld zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x40, 0x4a, 0x01], "vpmulld zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x40, 0x4a, 0x01], "vpmulld zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x40, 0xca], "vpmulld xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x40, 0x0a], "vpmulld xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x40, 0x4a, 0x01], "vpmulld xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x40, 0xca], "vpmulld xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x40, 0xca], "vpmulld xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x40, 0x0a], "vpmulld xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x40, 0x0a], "vpmulld xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x40, 0x4a, 0x01], "vpmulld xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x40, 0x4a, 0x01], "vpmulld xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xfd, 0x42, 0xca], "vgetexppd zmm1{k5}{z}{sae}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x78, 0x42, 0xca], "vgetexppd zmm1{sae}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x7d, 0x42, 0xca], "vgetexppd zmm1{k5}{sae}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x42, 0x0a], "vgetexppd ymm1{k5}{z}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x42, 0x4a, 0x01], "vgetexppd ymm1{k5}{z}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x42, 0x0a], "vgetexppd ymm1, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x42, 0x0a], "vgetexppd ymm1{k5}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x42, 0x4a, 0x01], "vgetexppd ymm1, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x42, 0x4a, 0x01], "vgetexppd ymm1{k5}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x42, 0xca], "vgetexppd ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x42, 0x0a], "vgetexppd ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x42, 0x4a, 0x01], "vgetexppd ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x42, 0xca], "vgetexppd ymm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x42, 0xca], "vgetexppd ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x42, 0x0a], "vgetexppd ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x42, 0x0a], "vgetexppd ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x42, 0x4a, 0x01], "vgetexppd ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x42, 0x4a, 0x01], "vgetexppd ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xfd, 0x42, 0xca], "vgetexpps zmm1{k5}{z}{sae}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x78, 0x42, 0xca], "vgetexpps zmm1{sae}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x7d, 0x42, 0xca], "vgetexpps zmm1{k5}{sae}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x42, 0x0a], "vgetexpps ymm1{k5}{z}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x42, 0x4a, 0x01], "vgetexpps ymm1{k5}{z}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x42, 0x0a], "vgetexpps ymm1, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x42, 0x0a], "vgetexpps ymm1{k5}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x42, 0x4a, 0x01], "vgetexpps ymm1, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x42, 0x4a, 0x01], "vgetexpps ymm1{k5}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x42, 0xca], "vgetexpps ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x42, 0x0a], "vgetexpps ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x42, 0x4a, 0x01], "vgetexpps ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x42, 0xca], "vgetexpps ymm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x42, 0xca], "vgetexpps ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x42, 0x0a], "vgetexpps ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x42, 0x0a], "vgetexpps ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x42, 0x4a, 0x01], "vgetexpps ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x42, 0x4a, 0x01], "vgetexpps ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x42, 0x0a], "vgetexppd zmm1{k5}{z}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x42, 0x4a, 0x01], "vgetexppd zmm1{k5}{z}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x42, 0x0a], "vgetexppd zmm1, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x42, 0x0a], "vgetexppd zmm1{k5}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x42, 0x4a, 0x01], "vgetexppd zmm1, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x42, 0x4a, 0x01], "vgetexppd zmm1{k5}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x42, 0x0a], "vgetexppd xmm1{k5}{z}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x42, 0x4a, 0x01], "vgetexppd xmm1{k5}{z}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x42, 0x0a], "vgetexppd xmm1, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x42, 0x0a], "vgetexppd xmm1{k5}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x42, 0x4a, 0x01], "vgetexppd xmm1, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x42, 0x4a, 0x01], "vgetexppd xmm1{k5}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x42, 0xca], "vgetexppd zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x42, 0x0a], "vgetexppd zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x42, 0x4a, 0x01], "vgetexppd zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x42, 0xca], "vgetexppd zmm1, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x42, 0xca], "vgetexppd zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x42, 0x0a], "vgetexppd zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x42, 0x0a], "vgetexppd zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x42, 0x4a, 0x01], "vgetexppd zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x42, 0x4a, 0x01], "vgetexppd zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x42, 0xca], "vgetexppd xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x42, 0x0a], "vgetexppd xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x42, 0x4a, 0x01], "vgetexppd xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x42, 0xca], "vgetexppd xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x42, 0xca], "vgetexppd xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x42, 0x0a], "vgetexppd xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x42, 0x0a], "vgetexppd xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x42, 0x4a, 0x01], "vgetexppd xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x42, 0x4a, 0x01], "vgetexppd xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x42, 0x0a], "vgetexpps zmm1{k5}{z}, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x42, 0x4a, 0x01], "vgetexpps zmm1{k5}{z}, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x42, 0x0a], "vgetexpps zmm1, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x42, 0x0a], "vgetexpps zmm1{k5}, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x42, 0x4a, 0x01], "vgetexpps zmm1, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x42, 0x4a, 0x01], "vgetexpps zmm1{k5}, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x42, 0x0a], "vgetexpps xmm1{k5}{z}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x42, 0x4a, 0x01], "vgetexpps xmm1{k5}{z}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x42, 0x0a], "vgetexpps xmm1, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x42, 0x0a], "vgetexpps xmm1{k5}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x42, 0x4a, 0x01], "vgetexpps xmm1, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x42, 0x4a, 0x01], "vgetexpps xmm1{k5}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x42, 0xca], "vgetexpps zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x42, 0x0a], "vgetexpps zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x42, 0x4a, 0x01], "vgetexpps zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x42, 0xca], "vgetexpps zmm1, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x42, 0xca], "vgetexpps zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x42, 0x0a], "vgetexpps zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x42, 0x0a], "vgetexpps zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x42, 0x4a, 0x01], "vgetexpps zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x42, 0x4a, 0x01], "vgetexpps zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x42, 0xca], "vgetexpps xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x42, 0x0a], "vgetexpps xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x42, 0x4a, 0x01], "vgetexpps xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x42, 0xca], "vgetexpps xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x42, 0xca], "vgetexpps xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x42, 0x0a], "vgetexpps xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x42, 0x0a], "vgetexpps xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x42, 0x4a, 0x01], "vgetexpps xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x42, 0x4a, 0x01], "vgetexpps xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xfd, 0x43, 0xca], "vgetexpsd xmm1{k5}{z}{sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x78, 0x43, 0xca], "vgetexpsd xmm1{sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x7d, 0x43, 0xca], "vgetexpsd xmm1{k5}{sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x43, 0xca], "vgetexpsd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x43, 0x0a], "vgetexpsd xmm1{k5}{z}, xmm0, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x43, 0x4a, 0x01], "vgetexpsd xmm1{k5}{z}, xmm0, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x43, 0xca], "vgetexpsd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x43, 0xca], "vgetexpsd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x43, 0x0a], "vgetexpsd xmm1, xmm0, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x43, 0x0a], "vgetexpsd xmm1{k5}, xmm0, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x43, 0x4a, 0x01], "vgetexpsd xmm1, xmm0, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x43, 0x4a, 0x01], "vgetexpsd xmm1{k5}, xmm0, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xfd, 0x43, 0xca], "vgetexpss xmm1{k5}{z}{sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x78, 0x43, 0xca], "vgetexpss xmm1{sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x7d, 0x43, 0xca], "vgetexpss xmm1{k5}{sae}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x43, 0xca], "vgetexpss xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x43, 0x0a], "vgetexpss xmm1{k5}{z}, xmm0, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x43, 0x4a, 0x01], "vgetexpss xmm1{k5}{z}, xmm0, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x43, 0xca], "vgetexpss xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x43, 0xca], "vgetexpss xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x43, 0x0a], "vgetexpss xmm1, xmm0, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x43, 0x0a], "vgetexpss xmm1{k5}, xmm0, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x43, 0x4a, 0x01], "vgetexpss xmm1, xmm0, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x43, 0x4a, 0x01], "vgetexpss xmm1{k5}, xmm0, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x44, 0x0a], "vplzcntq ymm1{k5}{z}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x44, 0x4a, 0x01], "vplzcntq ymm1{k5}{z}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x44, 0x0a], "vplzcntq ymm1, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x44, 0x0a], "vplzcntq ymm1{k5}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x44, 0x4a, 0x01], "vplzcntq ymm1, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x44, 0x4a, 0x01], "vplzcntq ymm1{k5}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x44, 0xca], "vplzcntq ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x44, 0x0a], "vplzcntq ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x44, 0x4a, 0x01], "vplzcntq ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x44, 0xca], "vplzcntq ymm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x44, 0xca], "vplzcntq ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x44, 0x0a], "vplzcntq ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x44, 0x0a], "vplzcntq ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x44, 0x4a, 0x01], "vplzcntq ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x44, 0x4a, 0x01], "vplzcntq ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x44, 0x0a], "vplzcntd ymm1{k5}{z}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x44, 0x4a, 0x01], "vplzcntd ymm1{k5}{z}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x44, 0x0a], "vplzcntd ymm1, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x44, 0x0a], "vplzcntd ymm1{k5}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x44, 0x4a, 0x01], "vplzcntd ymm1, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x44, 0x4a, 0x01], "vplzcntd ymm1{k5}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x44, 0xca], "vplzcntd ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x44, 0x0a], "vplzcntd ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x44, 0x4a, 0x01], "vplzcntd ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x44, 0xca], "vplzcntd ymm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x44, 0xca], "vplzcntd ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x44, 0x0a], "vplzcntd ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x44, 0x0a], "vplzcntd ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x44, 0x4a, 0x01], "vplzcntd ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x44, 0x4a, 0x01], "vplzcntd ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x44, 0x0a], "vplzcntq zmm1{k5}{z}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x44, 0x4a, 0x01], "vplzcntq zmm1{k5}{z}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x44, 0x0a], "vplzcntq zmm1, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x44, 0x0a], "vplzcntq zmm1{k5}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x44, 0x4a, 0x01], "vplzcntq zmm1, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x44, 0x4a, 0x01], "vplzcntq zmm1{k5}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x44, 0x0a], "vplzcntq xmm1{k5}{z}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x44, 0x4a, 0x01], "vplzcntq xmm1{k5}{z}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x44, 0x0a], "vplzcntq xmm1, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x44, 0x0a], "vplzcntq xmm1{k5}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x44, 0x4a, 0x01], "vplzcntq xmm1, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x44, 0x4a, 0x01], "vplzcntq xmm1{k5}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x44, 0xca], "vplzcntq zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x44, 0x0a], "vplzcntq zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x44, 0x4a, 0x01], "vplzcntq zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x44, 0xca], "vplzcntq zmm1, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x44, 0xca], "vplzcntq zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x44, 0x0a], "vplzcntq zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x44, 0x0a], "vplzcntq zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x44, 0x4a, 0x01], "vplzcntq zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x44, 0x4a, 0x01], "vplzcntq zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x44, 0xca], "vplzcntq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x44, 0x0a], "vplzcntq xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x44, 0x4a, 0x01], "vplzcntq xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x44, 0xca], "vplzcntq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x44, 0xca], "vplzcntq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x44, 0x0a], "vplzcntq xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x44, 0x0a], "vplzcntq xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x44, 0x4a, 0x01], "vplzcntq xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x44, 0x4a, 0x01], "vplzcntq xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x44, 0x0a], "vplzcntd zmm1{k5}{z}, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x44, 0x4a, 0x01], "vplzcntd zmm1{k5}{z}, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x44, 0x0a], "vplzcntd zmm1, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x44, 0x0a], "vplzcntd zmm1{k5}, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x44, 0x4a, 0x01], "vplzcntd zmm1, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x44, 0x4a, 0x01], "vplzcntd zmm1{k5}, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x44, 0x0a], "vplzcntd xmm1{k5}{z}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x44, 0x4a, 0x01], "vplzcntd xmm1{k5}{z}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x44, 0x0a], "vplzcntd xmm1, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x44, 0x0a], "vplzcntd xmm1{k5}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x44, 0x4a, 0x01], "vplzcntd xmm1, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x44, 0x4a, 0x01], "vplzcntd xmm1{k5}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x44, 0xca], "vplzcntd zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x44, 0x0a], "vplzcntd zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x44, 0x4a, 0x01], "vplzcntd zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x44, 0xca], "vplzcntd zmm1, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x44, 0xca], "vplzcntd zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x44, 0x0a], "vplzcntd zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x44, 0x0a], "vplzcntd zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x44, 0x4a, 0x01], "vplzcntd zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x44, 0x4a, 0x01], "vplzcntd zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x44, 0xca], "vplzcntd xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x44, 0x0a], "vplzcntd xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x44, 0x4a, 0x01], "vplzcntd xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x44, 0xca], "vplzcntd xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x44, 0xca], "vplzcntd xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x44, 0x0a], "vplzcntd xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x44, 0x0a], "vplzcntd xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x44, 0x4a, 0x01], "vplzcntd xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x44, 0x4a, 0x01], "vplzcntd xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x45, 0x0a], "vpsrlvq ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x45, 0x4a, 0x01], "vpsrlvq ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x45, 0x0a], "vpsrlvq ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x45, 0x0a], "vpsrlvq ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x45, 0x4a, 0x01], "vpsrlvq ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x45, 0x4a, 0x01], "vpsrlvq ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x45, 0xca], "vpsrlvq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x45, 0x0a], "vpsrlvq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x45, 0x4a, 0x01], "vpsrlvq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x45, 0xca], "vpsrlvq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x45, 0xca], "vpsrlvq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x45, 0x0a], "vpsrlvq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x45, 0x0a], "vpsrlvq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x45, 0x4a, 0x01], "vpsrlvq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x45, 0x4a, 0x01], "vpsrlvq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x45, 0x0a], "vpsrlvd ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x45, 0x4a, 0x01], "vpsrlvd ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x45, 0x0a], "vpsrlvd ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x45, 0x0a], "vpsrlvd ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x45, 0x4a, 0x01], "vpsrlvd ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x45, 0x4a, 0x01], "vpsrlvd ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x45, 0xca], "vpsrlvd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x45, 0x0a], "vpsrlvd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x45, 0x4a, 0x01], "vpsrlvd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x45, 0xca], "vpsrlvd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x45, 0xca], "vpsrlvd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x45, 0x0a], "vpsrlvd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x45, 0x0a], "vpsrlvd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x45, 0x4a, 0x01], "vpsrlvd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x45, 0x4a, 0x01], "vpsrlvd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x45, 0x0a], "vpsrlvq zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x45, 0x4a, 0x01], "vpsrlvq zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x45, 0x0a], "vpsrlvq zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x45, 0x0a], "vpsrlvq zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x45, 0x4a, 0x01], "vpsrlvq zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x45, 0x4a, 0x01], "vpsrlvq zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x45, 0x0a], "vpsrlvq xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x45, 0x4a, 0x01], "vpsrlvq xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x45, 0x0a], "vpsrlvq xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x45, 0x0a], "vpsrlvq xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x45, 0x4a, 0x01], "vpsrlvq xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x45, 0x4a, 0x01], "vpsrlvq xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x45, 0xca], "vpsrlvq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x45, 0x0a], "vpsrlvq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x45, 0x4a, 0x01], "vpsrlvq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x45, 0xca], "vpsrlvq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x45, 0xca], "vpsrlvq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x45, 0x0a], "vpsrlvq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x45, 0x0a], "vpsrlvq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x45, 0x4a, 0x01], "vpsrlvq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x45, 0x4a, 0x01], "vpsrlvq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x45, 0xca], "vpsrlvq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x45, 0x0a], "vpsrlvq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x45, 0x4a, 0x01], "vpsrlvq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x45, 0xca], "vpsrlvq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x45, 0xca], "vpsrlvq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x45, 0x0a], "vpsrlvq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x45, 0x0a], "vpsrlvq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x45, 0x4a, 0x01], "vpsrlvq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x45, 0x4a, 0x01], "vpsrlvq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x45, 0x0a], "vpsrlvd zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x45, 0x4a, 0x01], "vpsrlvd zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x45, 0x0a], "vpsrlvd zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x45, 0x0a], "vpsrlvd zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x45, 0x4a, 0x01], "vpsrlvd zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x45, 0x4a, 0x01], "vpsrlvd zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x45, 0x0a], "vpsrlvd xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x45, 0x4a, 0x01], "vpsrlvd xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x45, 0x0a], "vpsrlvd xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x45, 0x0a], "vpsrlvd xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x45, 0x4a, 0x01], "vpsrlvd xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x45, 0x4a, 0x01], "vpsrlvd xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x45, 0xca], "vpsrlvd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x45, 0x0a], "vpsrlvd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x45, 0x4a, 0x01], "vpsrlvd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x45, 0xca], "vpsrlvd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x45, 0xca], "vpsrlvd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x45, 0x0a], "vpsrlvd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x45, 0x0a], "vpsrlvd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x45, 0x4a, 0x01], "vpsrlvd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x45, 0x4a, 0x01], "vpsrlvd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x45, 0xca], "vpsrlvd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x45, 0x0a], "vpsrlvd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x45, 0x4a, 0x01], "vpsrlvd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x45, 0xca], "vpsrlvd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x45, 0xca], "vpsrlvd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x45, 0x0a], "vpsrlvd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x45, 0x0a], "vpsrlvd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x45, 0x4a, 0x01], "vpsrlvd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x45, 0x4a, 0x01], "vpsrlvd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x46, 0x0a], "vpsravq ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x46, 0x4a, 0x01], "vpsravq ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x46, 0x0a], "vpsravq ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x46, 0x0a], "vpsravq ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x46, 0x4a, 0x01], "vpsravq ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x46, 0x4a, 0x01], "vpsravq ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x46, 0xca], "vpsravq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x46, 0x0a], "vpsravq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x46, 0x4a, 0x01], "vpsravq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x46, 0xca], "vpsravq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x46, 0xca], "vpsravq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x46, 0x0a], "vpsravq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x46, 0x0a], "vpsravq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x46, 0x4a, 0x01], "vpsravq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x46, 0x4a, 0x01], "vpsravq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x46, 0x0a], "vpsravd ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x46, 0x4a, 0x01], "vpsravd ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x46, 0x0a], "vpsravd ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x46, 0x0a], "vpsravd ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x46, 0x4a, 0x01], "vpsravd ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x46, 0x4a, 0x01], "vpsravd ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x46, 0xca], "vpsravd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x46, 0x0a], "vpsravd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x46, 0x4a, 0x01], "vpsravd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x46, 0xca], "vpsravd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x46, 0xca], "vpsravd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x46, 0x0a], "vpsravd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x46, 0x0a], "vpsravd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x46, 0x4a, 0x01], "vpsravd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x46, 0x4a, 0x01], "vpsravd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x46, 0x0a], "vpsravq zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x46, 0x4a, 0x01], "vpsravq zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x46, 0x0a], "vpsravq zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x46, 0x0a], "vpsravq zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x46, 0x4a, 0x01], "vpsravq zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x46, 0x4a, 0x01], "vpsravq zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x46, 0x0a], "vpsravq xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x46, 0x4a, 0x01], "vpsravq xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x46, 0x0a], "vpsravq xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x46, 0x0a], "vpsravq xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x46, 0x4a, 0x01], "vpsravq xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x46, 0x4a, 0x01], "vpsravq xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x46, 0xca], "vpsravq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x46, 0x0a], "vpsravq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x46, 0x4a, 0x01], "vpsravq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x46, 0xca], "vpsravq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x46, 0xca], "vpsravq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x46, 0x0a], "vpsravq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x46, 0x0a], "vpsravq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x46, 0x4a, 0x01], "vpsravq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x46, 0x4a, 0x01], "vpsravq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x46, 0xca], "vpsravq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x46, 0x0a], "vpsravq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x46, 0x4a, 0x01], "vpsravq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x46, 0xca], "vpsravq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x46, 0xca], "vpsravq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x46, 0x0a], "vpsravq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x46, 0x0a], "vpsravq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x46, 0x4a, 0x01], "vpsravq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x46, 0x4a, 0x01], "vpsravq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x46, 0x0a], "vpsravd zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x46, 0x4a, 0x01], "vpsravd zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x46, 0x0a], "vpsravd zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x46, 0x0a], "vpsravd zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x46, 0x4a, 0x01], "vpsravd zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x46, 0x4a, 0x01], "vpsravd zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x46, 0x0a], "vpsravd xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x46, 0x4a, 0x01], "vpsravd xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x46, 0x0a], "vpsravd xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x46, 0x0a], "vpsravd xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x46, 0x4a, 0x01], "vpsravd xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x46, 0x4a, 0x01], "vpsravd xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x46, 0xca], "vpsravd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x46, 0x0a], "vpsravd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x46, 0x4a, 0x01], "vpsravd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x46, 0xca], "vpsravd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x46, 0xca], "vpsravd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x46, 0x0a], "vpsravd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x46, 0x0a], "vpsravd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x46, 0x4a, 0x01], "vpsravd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x46, 0x4a, 0x01], "vpsravd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x46, 0xca], "vpsravd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x46, 0x0a], "vpsravd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x46, 0x4a, 0x01], "vpsravd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x46, 0xca], "vpsravd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x46, 0xca], "vpsravd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x46, 0x0a], "vpsravd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x46, 0x0a], "vpsravd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x46, 0x4a, 0x01], "vpsravd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x46, 0x4a, 0x01], "vpsravd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x47, 0x0a], "vpsllvq ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x47, 0x4a, 0x01], "vpsllvq ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x47, 0x0a], "vpsllvq ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x47, 0x0a], "vpsllvq ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x47, 0x4a, 0x01], "vpsllvq ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x47, 0x4a, 0x01], "vpsllvq ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x47, 0xca], "vpsllvq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x47, 0x0a], "vpsllvq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x47, 0x4a, 0x01], "vpsllvq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x47, 0xca], "vpsllvq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x47, 0xca], "vpsllvq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x47, 0x0a], "vpsllvq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x47, 0x0a], "vpsllvq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x47, 0x4a, 0x01], "vpsllvq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x47, 0x4a, 0x01], "vpsllvq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x47, 0x0a], "vpsllvd ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x47, 0x4a, 0x01], "vpsllvd ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x47, 0x0a], "vpsllvd ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x47, 0x0a], "vpsllvd ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x47, 0x4a, 0x01], "vpsllvd ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x47, 0x4a, 0x01], "vpsllvd ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x47, 0xca], "vpsllvd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x47, 0x0a], "vpsllvd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x47, 0x4a, 0x01], "vpsllvd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x47, 0xca], "vpsllvd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x47, 0xca], "vpsllvd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x47, 0x0a], "vpsllvd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x47, 0x0a], "vpsllvd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x47, 0x4a, 0x01], "vpsllvd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x47, 0x4a, 0x01], "vpsllvd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x47, 0x0a], "vpsllvq zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x47, 0x4a, 0x01], "vpsllvq zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x47, 0x0a], "vpsllvq zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x47, 0x0a], "vpsllvq zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x47, 0x4a, 0x01], "vpsllvq zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x47, 0x4a, 0x01], "vpsllvq zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x47, 0x0a], "vpsllvq xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x47, 0x4a, 0x01], "vpsllvq xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x47, 0x0a], "vpsllvq xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x47, 0x0a], "vpsllvq xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x47, 0x4a, 0x01], "vpsllvq xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x47, 0x4a, 0x01], "vpsllvq xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x47, 0xca], "vpsllvq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x47, 0x0a], "vpsllvq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x47, 0x4a, 0x01], "vpsllvq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x47, 0xca], "vpsllvq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x47, 0xca], "vpsllvq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x47, 0x0a], "vpsllvq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x47, 0x0a], "vpsllvq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x47, 0x4a, 0x01], "vpsllvq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x47, 0x4a, 0x01], "vpsllvq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x47, 0xca], "vpsllvq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x47, 0x0a], "vpsllvq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x47, 0x4a, 0x01], "vpsllvq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x47, 0xca], "vpsllvq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x47, 0xca], "vpsllvq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x47, 0x0a], "vpsllvq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x47, 0x0a], "vpsllvq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x47, 0x4a, 0x01], "vpsllvq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x47, 0x4a, 0x01], "vpsllvq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x47, 0x0a], "vpsllvd zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x47, 0x4a, 0x01], "vpsllvd zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x47, 0x0a], "vpsllvd zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x47, 0x0a], "vpsllvd zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x47, 0x4a, 0x01], "vpsllvd zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x47, 0x4a, 0x01], "vpsllvd zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x47, 0x0a], "vpsllvd xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x47, 0x4a, 0x01], "vpsllvd xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x47, 0x0a], "vpsllvd xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x47, 0x0a], "vpsllvd xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x47, 0x4a, 0x01], "vpsllvd xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x47, 0x4a, 0x01], "vpsllvd xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x47, 0xca], "vpsllvd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x47, 0x0a], "vpsllvd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x47, 0x4a, 0x01], "vpsllvd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x47, 0xca], "vpsllvd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x47, 0xca], "vpsllvd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x47, 0x0a], "vpsllvd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x47, 0x0a], "vpsllvd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x47, 0x4a, 0x01], "vpsllvd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x47, 0x4a, 0x01], "vpsllvd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x47, 0xca], "vpsllvd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x47, 0x0a], "vpsllvd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x47, 0x4a, 0x01], "vpsllvd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x47, 0xca], "vpsllvd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x47, 0xca], "vpsllvd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x47, 0x0a], "vpsllvd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x47, 0x0a], "vpsllvd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x47, 0x4a, 0x01], "vpsllvd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x47, 0x4a, 0x01], "vpsllvd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x4c, 0x0a], "vrcp14pd ymm1{k5}{z}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x4c, 0x4a, 0x01], "vrcp14pd ymm1{k5}{z}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x4c, 0x0a], "vrcp14pd ymm1, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x4c, 0x0a], "vrcp14pd ymm1{k5}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x4c, 0x4a, 0x01], "vrcp14pd ymm1, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x4c, 0x4a, 0x01], "vrcp14pd ymm1{k5}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x4c, 0xca], "vrcp14pd ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x4c, 0x0a], "vrcp14pd ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x4c, 0x4a, 0x01], "vrcp14pd ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x4c, 0xca], "vrcp14pd ymm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x4c, 0xca], "vrcp14pd ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x4c, 0x0a], "vrcp14pd ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x4c, 0x0a], "vrcp14pd ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x4c, 0x4a, 0x01], "vrcp14pd ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x4c, 0x4a, 0x01], "vrcp14pd ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x4c, 0x0a], "vrcp14ps ymm1{k5}{z}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x4c, 0x4a, 0x01], "vrcp14ps ymm1{k5}{z}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x4c, 0x0a], "vrcp14ps ymm1, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x4c, 0x0a], "vrcp14ps ymm1{k5}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x4c, 0x4a, 0x01], "vrcp14ps ymm1, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x4c, 0x4a, 0x01], "vrcp14ps ymm1{k5}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x4c, 0xca], "vrcp14ps ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x4c, 0x0a], "vrcp14ps ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x4c, 0x4a, 0x01], "vrcp14ps ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x4c, 0xca], "vrcp14ps ymm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x4c, 0xca], "vrcp14ps ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x4c, 0x0a], "vrcp14ps ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x4c, 0x0a], "vrcp14ps ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x4c, 0x4a, 0x01], "vrcp14ps ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x4c, 0x4a, 0x01], "vrcp14ps ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x4c, 0x0a], "vrcp14pd zmm1{k5}{z}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x4c, 0x4a, 0x01], "vrcp14pd zmm1{k5}{z}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x4c, 0x0a], "vrcp14pd zmm1, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x4c, 0x0a], "vrcp14pd zmm1{k5}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x4c, 0x4a, 0x01], "vrcp14pd zmm1, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x4c, 0x4a, 0x01], "vrcp14pd zmm1{k5}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x4c, 0x0a], "vrcp14pd xmm1{k5}{z}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x4c, 0x4a, 0x01], "vrcp14pd xmm1{k5}{z}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x4c, 0x0a], "vrcp14pd xmm1, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x4c, 0x0a], "vrcp14pd xmm1{k5}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x4c, 0x4a, 0x01], "vrcp14pd xmm1, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x4c, 0x4a, 0x01], "vrcp14pd xmm1{k5}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x4c, 0xca], "vrcp14pd zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x4c, 0x0a], "vrcp14pd zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x4c, 0x4a, 0x01], "vrcp14pd zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x4c, 0xca], "vrcp14pd zmm1, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x4c, 0xca], "vrcp14pd zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x4c, 0x0a], "vrcp14pd zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x4c, 0x0a], "vrcp14pd zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x4c, 0x4a, 0x01], "vrcp14pd zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x4c, 0x4a, 0x01], "vrcp14pd zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x4c, 0xca], "vrcp14pd xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x4c, 0x0a], "vrcp14pd xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x4c, 0x4a, 0x01], "vrcp14pd xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x4c, 0xca], "vrcp14pd xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x4c, 0xca], "vrcp14pd xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x4c, 0x0a], "vrcp14pd xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x4c, 0x0a], "vrcp14pd xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x4c, 0x4a, 0x01], "vrcp14pd xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x4c, 0x4a, 0x01], "vrcp14pd xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x4c, 0x0a], "vrcp14ps zmm1{k5}{z}, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x4c, 0x4a, 0x01], "vrcp14ps zmm1{k5}{z}, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x4c, 0x0a], "vrcp14ps zmm1, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x4c, 0x0a], "vrcp14ps zmm1{k5}, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x4c, 0x4a, 0x01], "vrcp14ps zmm1, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x4c, 0x4a, 0x01], "vrcp14ps zmm1{k5}, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x4c, 0x0a], "vrcp14ps xmm1{k5}{z}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x4c, 0x4a, 0x01], "vrcp14ps xmm1{k5}{z}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x4c, 0x0a], "vrcp14ps xmm1, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x4c, 0x0a], "vrcp14ps xmm1{k5}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x4c, 0x4a, 0x01], "vrcp14ps xmm1, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x4c, 0x4a, 0x01], "vrcp14ps xmm1{k5}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x4c, 0xca], "vrcp14ps zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x4c, 0x0a], "vrcp14ps zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x4c, 0x4a, 0x01], "vrcp14ps zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x4c, 0xca], "vrcp14ps zmm1, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x4c, 0xca], "vrcp14ps zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x4c, 0x0a], "vrcp14ps zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x4c, 0x0a], "vrcp14ps zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x4c, 0x4a, 0x01], "vrcp14ps zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x4c, 0x4a, 0x01], "vrcp14ps zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x4c, 0xca], "vrcp14ps xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x4c, 0x0a], "vrcp14ps xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x4c, 0x4a, 0x01], "vrcp14ps xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x4c, 0xca], "vrcp14ps xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x4c, 0xca], "vrcp14ps xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x4c, 0x0a], "vrcp14ps xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x4c, 0x0a], "vrcp14ps xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x4c, 0x4a, 0x01], "vrcp14ps xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x4c, 0x4a, 0x01], "vrcp14ps xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x4d, 0xca], "vrcp14sd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x4d, 0x0a], "vrcp14sd xmm1{k5}{z}, xmm0, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x4d, 0x4a, 0x01], "vrcp14sd xmm1{k5}{z}, xmm0, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x4d, 0xca], "vrcp14sd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x4d, 0xca], "vrcp14sd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x4d, 0x0a], "vrcp14sd xmm1, xmm0, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x4d, 0x0a], "vrcp14sd xmm1{k5}, xmm0, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x4d, 0x4a, 0x01], "vrcp14sd xmm1, xmm0, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x4d, 0x4a, 0x01], "vrcp14sd xmm1{k5}, xmm0, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x4d, 0xca], "vrcp14ss xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x4d, 0x0a], "vrcp14ss xmm1{k5}{z}, xmm0, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x4d, 0x4a, 0x01], "vrcp14ss xmm1{k5}{z}, xmm0, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x4d, 0xca], "vrcp14ss xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x4d, 0xca], "vrcp14ss xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x4d, 0x0a], "vrcp14ss xmm1, xmm0, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x4d, 0x0a], "vrcp14ss xmm1{k5}, xmm0, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x4d, 0x4a, 0x01], "vrcp14ss xmm1, xmm0, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x4d, 0x4a, 0x01], "vrcp14ss xmm1{k5}, xmm0, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x4e, 0x0a], "vrsqrt14pd ymm1{k5}{z}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x4e, 0x4a, 0x01], "vrsqrt14pd ymm1{k5}{z}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x4e, 0x0a], "vrsqrt14pd ymm1, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x4e, 0x0a], "vrsqrt14pd ymm1{k5}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x4e, 0x4a, 0x01], "vrsqrt14pd ymm1, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x4e, 0x4a, 0x01], "vrsqrt14pd ymm1{k5}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x4e, 0xca], "vrsqrt14pd ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x4e, 0x0a], "vrsqrt14pd ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x4e, 0x4a, 0x01], "vrsqrt14pd ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x4e, 0xca], "vrsqrt14pd ymm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x4e, 0xca], "vrsqrt14pd ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x4e, 0x0a], "vrsqrt14pd ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x4e, 0x0a], "vrsqrt14pd ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x4e, 0x4a, 0x01], "vrsqrt14pd ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x4e, 0x4a, 0x01], "vrsqrt14pd ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x4e, 0x0a], "vrsqrt14ps ymm1{k5}{z}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x4e, 0x4a, 0x01], "vrsqrt14ps ymm1{k5}{z}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x4e, 0x0a], "vrsqrt14ps ymm1, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x4e, 0x0a], "vrsqrt14ps ymm1{k5}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x4e, 0x4a, 0x01], "vrsqrt14ps ymm1, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x4e, 0x4a, 0x01], "vrsqrt14ps ymm1{k5}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x4e, 0xca], "vrsqrt14ps ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x4e, 0x0a], "vrsqrt14ps ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x4e, 0x4a, 0x01], "vrsqrt14ps ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x4e, 0xca], "vrsqrt14ps ymm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x4e, 0xca], "vrsqrt14ps ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x4e, 0x0a], "vrsqrt14ps ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x4e, 0x0a], "vrsqrt14ps ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x4e, 0x4a, 0x01], "vrsqrt14ps ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x4e, 0x4a, 0x01], "vrsqrt14ps ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x4e, 0x0a], "vrsqrt14pd zmm1{k5}{z}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x4e, 0x4a, 0x01], "vrsqrt14pd zmm1{k5}{z}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x4e, 0x0a], "vrsqrt14pd zmm1, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x4e, 0x0a], "vrsqrt14pd zmm1{k5}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x4e, 0x4a, 0x01], "vrsqrt14pd zmm1, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x4e, 0x4a, 0x01], "vrsqrt14pd zmm1{k5}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x4e, 0x0a], "vrsqrt14pd xmm1{k5}{z}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x4e, 0x4a, 0x01], "vrsqrt14pd xmm1{k5}{z}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x4e, 0x0a], "vrsqrt14pd xmm1, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x4e, 0x0a], "vrsqrt14pd xmm1{k5}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x4e, 0x4a, 0x01], "vrsqrt14pd xmm1, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x4e, 0x4a, 0x01], "vrsqrt14pd xmm1{k5}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x4e, 0xca], "vrsqrt14pd zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x4e, 0x0a], "vrsqrt14pd zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x4e, 0x4a, 0x01], "vrsqrt14pd zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x4e, 0xca], "vrsqrt14pd zmm1, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x4e, 0xca], "vrsqrt14pd zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x4e, 0x0a], "vrsqrt14pd zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x4e, 0x0a], "vrsqrt14pd zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x4e, 0x4a, 0x01], "vrsqrt14pd zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x4e, 0x4a, 0x01], "vrsqrt14pd zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x4e, 0xca], "vrsqrt14pd xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x4e, 0x0a], "vrsqrt14pd xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x4e, 0x4a, 0x01], "vrsqrt14pd xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x4e, 0xca], "vrsqrt14pd xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x4e, 0xca], "vrsqrt14pd xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x4e, 0x0a], "vrsqrt14pd xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x4e, 0x0a], "vrsqrt14pd xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x4e, 0x4a, 0x01], "vrsqrt14pd xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x4e, 0x4a, 0x01], "vrsqrt14pd xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x4e, 0x0a], "vrsqrt14ps zmm1{k5}{z}, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x4e, 0x4a, 0x01], "vrsqrt14ps zmm1{k5}{z}, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x4e, 0x0a], "vrsqrt14ps zmm1, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x4e, 0x0a], "vrsqrt14ps zmm1{k5}, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x4e, 0x4a, 0x01], "vrsqrt14ps zmm1, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x4e, 0x4a, 0x01], "vrsqrt14ps zmm1{k5}, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x4e, 0x0a], "vrsqrt14ps xmm1{k5}{z}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x4e, 0x4a, 0x01], "vrsqrt14ps xmm1{k5}{z}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x4e, 0x0a], "vrsqrt14ps xmm1, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x4e, 0x0a], "vrsqrt14ps xmm1{k5}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x4e, 0x4a, 0x01], "vrsqrt14ps xmm1, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x4e, 0x4a, 0x01], "vrsqrt14ps xmm1{k5}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x4e, 0xca], "vrsqrt14ps zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x4e, 0x0a], "vrsqrt14ps zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x4e, 0x4a, 0x01], "vrsqrt14ps zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x4e, 0xca], "vrsqrt14ps zmm1, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x4e, 0xca], "vrsqrt14ps zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x4e, 0x0a], "vrsqrt14ps zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x4e, 0x0a], "vrsqrt14ps zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x4e, 0x4a, 0x01], "vrsqrt14ps zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x4e, 0x4a, 0x01], "vrsqrt14ps zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x4e, 0xca], "vrsqrt14ps xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x4e, 0x0a], "vrsqrt14ps xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x4e, 0x4a, 0x01], "vrsqrt14ps xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x4e, 0xca], "vrsqrt14ps xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x4e, 0xca], "vrsqrt14ps xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x4e, 0x0a], "vrsqrt14ps xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x4e, 0x0a], "vrsqrt14ps xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x4e, 0x4a, 0x01], "vrsqrt14ps xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x4e, 0x4a, 0x01], "vrsqrt14ps xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x4f, 0xca], "vrsqrt14sd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x4f, 0x0a], "vrsqrt14sd xmm1{k5}{z}, xmm0, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x4f, 0x4a, 0x01], "vrsqrt14sd xmm1{k5}{z}, xmm0, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x4f, 0xca], "vrsqrt14sd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x4f, 0xca], "vrsqrt14sd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x4f, 0x0a], "vrsqrt14sd xmm1, xmm0, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x4f, 0x0a], "vrsqrt14sd xmm1{k5}, xmm0, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x4f, 0x4a, 0x01], "vrsqrt14sd xmm1, xmm0, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x4f, 0x4a, 0x01], "vrsqrt14sd xmm1{k5}, xmm0, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x4f, 0xca], "vrsqrt14ss xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x4f, 0x0a], "vrsqrt14ss xmm1{k5}{z}, xmm0, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x4f, 0x4a, 0x01], "vrsqrt14ss xmm1{k5}{z}, xmm0, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x4f, 0xca], "vrsqrt14ss xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x4f, 0xca], "vrsqrt14ss xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x4f, 0x0a], "vrsqrt14ss xmm1, xmm0, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x4f, 0x0a], "vrsqrt14ss xmm1{k5}, xmm0, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x4f, 0x4a, 0x01], "vrsqrt14ss xmm1, xmm0, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x4f, 0x4a, 0x01], "vrsqrt14ss xmm1{k5}, xmm0, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x50, 0x0a], "vpdpbusd ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x50, 0x4a, 0x01], "vpdpbusd ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x50, 0x0a], "vpdpbusd ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x50, 0x0a], "vpdpbusd ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x50, 0x4a, 0x01], "vpdpbusd ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x50, 0x4a, 0x01], "vpdpbusd ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x50, 0xca], "vpdpbusd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x50, 0x0a], "vpdpbusd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x50, 0x4a, 0x01], "vpdpbusd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x50, 0xca], "vpdpbusd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x50, 0xca], "vpdpbusd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x50, 0x0a], "vpdpbusd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x50, 0x0a], "vpdpbusd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x50, 0x4a, 0x01], "vpdpbusd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x50, 0x4a, 0x01], "vpdpbusd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x50, 0x0a], "vpdpbusd zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x50, 0x4a, 0x01], "vpdpbusd zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x50, 0x0a], "vpdpbusd zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x50, 0x0a], "vpdpbusd zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x50, 0x4a, 0x01], "vpdpbusd zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x50, 0x4a, 0x01], "vpdpbusd zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x50, 0x0a], "vpdpbusd xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x50, 0x4a, 0x01], "vpdpbusd xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x50, 0x0a], "vpdpbusd xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x50, 0x0a], "vpdpbusd xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x50, 0x4a, 0x01], "vpdpbusd xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x50, 0x4a, 0x01], "vpdpbusd xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x50, 0xca], "vpdpbusd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x50, 0x0a], "vpdpbusd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x50, 0x4a, 0x01], "vpdpbusd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x50, 0xca], "vpdpbusd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x50, 0xca], "vpdpbusd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x50, 0x0a], "vpdpbusd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x50, 0x0a], "vpdpbusd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x50, 0x4a, 0x01], "vpdpbusd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x50, 0x4a, 0x01], "vpdpbusd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x50, 0xca], "vpdpbusd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x50, 0x0a], "vpdpbusd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x50, 0x4a, 0x01], "vpdpbusd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x50, 0xca], "vpdpbusd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x50, 0xca], "vpdpbusd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x50, 0x0a], "vpdpbusd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x50, 0x0a], "vpdpbusd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x50, 0x4a, 0x01], "vpdpbusd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x50, 0x4a, 0x01], "vpdpbusd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x51, 0x0a], "vpdpbusds ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x51, 0x4a, 0x01], "vpdpbusds ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x51, 0x0a], "vpdpbusds ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x51, 0x0a], "vpdpbusds ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x51, 0x4a, 0x01], "vpdpbusds ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x51, 0x4a, 0x01], "vpdpbusds ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x51, 0xca], "vpdpbusds ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x51, 0x0a], "vpdpbusds ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x51, 0x4a, 0x01], "vpdpbusds ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x51, 0xca], "vpdpbusds ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x51, 0xca], "vpdpbusds ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x51, 0x0a], "vpdpbusds ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x51, 0x0a], "vpdpbusds ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x51, 0x4a, 0x01], "vpdpbusds ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x51, 0x4a, 0x01], "vpdpbusds ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x51, 0x0a], "vpdpbusds zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x51, 0x4a, 0x01], "vpdpbusds zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x51, 0x0a], "vpdpbusds zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x51, 0x0a], "vpdpbusds zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x51, 0x4a, 0x01], "vpdpbusds zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x51, 0x4a, 0x01], "vpdpbusds zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x51, 0x0a], "vpdpbusds xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x51, 0x4a, 0x01], "vpdpbusds xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x51, 0x0a], "vpdpbusds xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x51, 0x0a], "vpdpbusds xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x51, 0x4a, 0x01], "vpdpbusds xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x51, 0x4a, 0x01], "vpdpbusds xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x51, 0xca], "vpdpbusds zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x51, 0x0a], "vpdpbusds zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x51, 0x4a, 0x01], "vpdpbusds zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x51, 0xca], "vpdpbusds zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x51, 0xca], "vpdpbusds zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x51, 0x0a], "vpdpbusds zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x51, 0x0a], "vpdpbusds zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x51, 0x4a, 0x01], "vpdpbusds zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x51, 0x4a, 0x01], "vpdpbusds zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x51, 0xca], "vpdpbusds xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x51, 0x0a], "vpdpbusds xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x51, 0x4a, 0x01], "vpdpbusds xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x51, 0xca], "vpdpbusds xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x51, 0xca], "vpdpbusds xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x51, 0x0a], "vpdpbusds xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x51, 0x0a], "vpdpbusds xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x51, 0x4a, 0x01], "vpdpbusds xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x51, 0x4a, 0x01], "vpdpbusds xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x52, 0x0a], "vpdpwssd ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x52, 0x4a, 0x01], "vpdpwssd ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x52, 0x0a], "vpdpwssd ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x52, 0x0a], "vpdpwssd ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x52, 0x4a, 0x01], "vpdpwssd ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x52, 0x4a, 0x01], "vpdpwssd ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x52, 0xca], "vpdpwssd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x52, 0x0a], "vpdpwssd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x52, 0x4a, 0x01], "vpdpwssd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x52, 0xca], "vpdpwssd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x52, 0xca], "vpdpwssd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x52, 0x0a], "vpdpwssd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x52, 0x0a], "vpdpwssd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x52, 0x4a, 0x01], "vpdpwssd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x52, 0x4a, 0x01], "vpdpwssd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x52, 0x0a], "vpdpwssd zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x52, 0x4a, 0x01], "vpdpwssd zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x52, 0x0a], "vpdpwssd zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x52, 0x0a], "vpdpwssd zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x52, 0x4a, 0x01], "vpdpwssd zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x52, 0x4a, 0x01], "vpdpwssd zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x52, 0x0a], "vpdpwssd xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x52, 0x4a, 0x01], "vpdpwssd xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x52, 0x0a], "vpdpwssd xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x52, 0x0a], "vpdpwssd xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x52, 0x4a, 0x01], "vpdpwssd xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x52, 0x4a, 0x01], "vpdpwssd xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x52, 0xca], "vpdpwssd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x52, 0x0a], "vpdpwssd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x52, 0x4a, 0x01], "vpdpwssd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x52, 0xca], "vpdpwssd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x52, 0xca], "vpdpwssd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x52, 0x0a], "vpdpwssd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x52, 0x0a], "vpdpwssd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x52, 0x4a, 0x01], "vpdpwssd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x52, 0x4a, 0x01], "vpdpwssd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x52, 0xca], "vpdpwssd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x52, 0x0a], "vpdpwssd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x52, 0x4a, 0x01], "vpdpwssd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x52, 0xca], "vpdpwssd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x52, 0xca], "vpdpwssd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x52, 0x0a], "vpdpwssd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x52, 0x0a], "vpdpwssd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x52, 0x4a, 0x01], "vpdpwssd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x52, 0x4a, 0x01], "vpdpwssd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x53, 0x0a], "vpdpwssds ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x53, 0x4a, 0x01], "vpdpwssds ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x53, 0x0a], "vpdpwssds ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x53, 0x0a], "vpdpwssds ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x53, 0x4a, 0x01], "vpdpwssds ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x53, 0x4a, 0x01], "vpdpwssds ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x53, 0xca], "vpdpwssds ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x53, 0x0a], "vpdpwssds ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x53, 0x4a, 0x01], "vpdpwssds ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x53, 0xca], "vpdpwssds ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x53, 0xca], "vpdpwssds ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x53, 0x0a], "vpdpwssds ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x53, 0x0a], "vpdpwssds ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x53, 0x4a, 0x01], "vpdpwssds ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x53, 0x4a, 0x01], "vpdpwssds ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x53, 0x0a], "vpdpwssds zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x53, 0x4a, 0x01], "vpdpwssds zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x53, 0x0a], "vpdpwssds zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x53, 0x0a], "vpdpwssds zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x53, 0x4a, 0x01], "vpdpwssds zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x53, 0x4a, 0x01], "vpdpwssds zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x53, 0x0a], "vpdpwssds xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x53, 0x4a, 0x01], "vpdpwssds xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x53, 0x0a], "vpdpwssds xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x53, 0x0a], "vpdpwssds xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x53, 0x4a, 0x01], "vpdpwssds xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x53, 0x4a, 0x01], "vpdpwssds xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x53, 0xca], "vpdpwssds zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x53, 0x0a], "vpdpwssds zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x53, 0x4a, 0x01], "vpdpwssds zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x53, 0xca], "vpdpwssds zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x53, 0xca], "vpdpwssds zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x53, 0x0a], "vpdpwssds zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x53, 0x0a], "vpdpwssds zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x53, 0x4a, 0x01], "vpdpwssds zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x53, 0x4a, 0x01], "vpdpwssds zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x53, 0xca], "vpdpwssds xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x53, 0x0a], "vpdpwssds xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x53, 0x4a, 0x01], "vpdpwssds xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x53, 0xca], "vpdpwssds xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x53, 0xca], "vpdpwssds xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x53, 0x0a], "vpdpwssds xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x53, 0x0a], "vpdpwssds xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x53, 0x4a, 0x01], "vpdpwssds xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x53, 0x4a, 0x01], "vpdpwssds xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x54, 0xca], "vpopcntw ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x54, 0x0a], "vpopcntw ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x54, 0x4a, 0x01], "vpopcntw ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x54, 0xca], "vpopcntw ymm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x54, 0xca], "vpopcntw ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x54, 0x0a], "vpopcntw ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x54, 0x0a], "vpopcntw ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x54, 0x4a, 0x01], "vpopcntw ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x54, 0x4a, 0x01], "vpopcntw ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x54, 0xca], "vpopcntb ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x54, 0x0a], "vpopcntb ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x54, 0x4a, 0x01], "vpopcntb ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x54, 0xca], "vpopcntb ymm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x54, 0xca], "vpopcntb ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x54, 0x0a], "vpopcntb ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x54, 0x0a], "vpopcntb ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x54, 0x4a, 0x01], "vpopcntb ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x54, 0x4a, 0x01], "vpopcntb ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x54, 0xca], "vpopcntw zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x54, 0x0a], "vpopcntw zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x54, 0x4a, 0x01], "vpopcntw zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x54, 0xca], "vpopcntw zmm1, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x54, 0xca], "vpopcntw zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x54, 0x0a], "vpopcntw zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x54, 0x0a], "vpopcntw zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x54, 0x4a, 0x01], "vpopcntw zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x54, 0x4a, 0x01], "vpopcntw zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x54, 0xca], "vpopcntw xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x54, 0x0a], "vpopcntw xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x54, 0x4a, 0x01], "vpopcntw xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x54, 0xca], "vpopcntw xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x54, 0xca], "vpopcntw xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x54, 0x0a], "vpopcntw xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x54, 0x0a], "vpopcntw xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x54, 0x4a, 0x01], "vpopcntw xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x54, 0x4a, 0x01], "vpopcntw xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x54, 0xca], "vpopcntb zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x54, 0x0a], "vpopcntb zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x54, 0x4a, 0x01], "vpopcntb zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x54, 0xca], "vpopcntb zmm1, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x54, 0xca], "vpopcntb zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x54, 0x0a], "vpopcntb zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x54, 0x0a], "vpopcntb zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x54, 0x4a, 0x01], "vpopcntb zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x54, 0x4a, 0x01], "vpopcntb zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x54, 0xca], "vpopcntb xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x54, 0x0a], "vpopcntb xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x54, 0x4a, 0x01], "vpopcntb xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x54, 0xca], "vpopcntb xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x54, 0xca], "vpopcntb xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x54, 0x0a], "vpopcntb xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x54, 0x0a], "vpopcntb xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x54, 0x4a, 0x01], "vpopcntb xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x54, 0x4a, 0x01], "vpopcntb xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x55, 0x0a], "vpopcntq ymm1{k5}{z}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x55, 0x4a, 0x01], "vpopcntq ymm1{k5}{z}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x55, 0x0a], "vpopcntq ymm1, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x55, 0x0a], "vpopcntq ymm1{k5}, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x55, 0x4a, 0x01], "vpopcntq ymm1, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x55, 0x4a, 0x01], "vpopcntq ymm1{k5}, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x55, 0xca], "vpopcntq ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x55, 0x0a], "vpopcntq ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x55, 0x4a, 0x01], "vpopcntq ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x55, 0xca], "vpopcntq ymm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x55, 0xca], "vpopcntq ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x55, 0x0a], "vpopcntq ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x55, 0x0a], "vpopcntq ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x55, 0x4a, 0x01], "vpopcntq ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x55, 0x4a, 0x01], "vpopcntq ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x55, 0x0a], "vpopcntd ymm1{k5}{z}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x55, 0x4a, 0x01], "vpopcntd ymm1{k5}{z}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x55, 0x0a], "vpopcntd ymm1, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x55, 0x0a], "vpopcntd ymm1{k5}, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x55, 0x4a, 0x01], "vpopcntd ymm1, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x55, 0x4a, 0x01], "vpopcntd ymm1{k5}, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x55, 0xca], "vpopcntd ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x55, 0x0a], "vpopcntd ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x55, 0x4a, 0x01], "vpopcntd ymm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x55, 0xca], "vpopcntd ymm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x55, 0xca], "vpopcntd ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x55, 0x0a], "vpopcntd ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x55, 0x0a], "vpopcntd ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x55, 0x4a, 0x01], "vpopcntd ymm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x55, 0x4a, 0x01], "vpopcntd ymm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x55, 0x0a], "vpopcntq zmm1{k5}{z}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x55, 0x4a, 0x01], "vpopcntq zmm1{k5}{z}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x55, 0x0a], "vpopcntq zmm1, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x55, 0x0a], "vpopcntq zmm1{k5}, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x55, 0x4a, 0x01], "vpopcntq zmm1, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x55, 0x4a, 0x01], "vpopcntq zmm1{k5}, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x55, 0x0a], "vpopcntq xmm1{k5}{z}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x55, 0x4a, 0x01], "vpopcntq xmm1{k5}{z}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x55, 0x0a], "vpopcntq xmm1, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x55, 0x0a], "vpopcntq xmm1{k5}, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x55, 0x4a, 0x01], "vpopcntq xmm1, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x55, 0x4a, 0x01], "vpopcntq xmm1{k5}, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x55, 0xca], "vpopcntq zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x55, 0x0a], "vpopcntq zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x55, 0x4a, 0x01], "vpopcntq zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x55, 0xca], "vpopcntq zmm1, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x55, 0xca], "vpopcntq zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x55, 0x0a], "vpopcntq zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x55, 0x0a], "vpopcntq zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x55, 0x4a, 0x01], "vpopcntq zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x55, 0x4a, 0x01], "vpopcntq zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x55, 0xca], "vpopcntq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x55, 0x0a], "vpopcntq xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x55, 0x4a, 0x01], "vpopcntq xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x55, 0xca], "vpopcntq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x55, 0xca], "vpopcntq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x55, 0x0a], "vpopcntq xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x55, 0x0a], "vpopcntq xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x55, 0x4a, 0x01], "vpopcntq xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x55, 0x4a, 0x01], "vpopcntq xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x55, 0x0a], "vpopcntd zmm1{k5}{z}, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x55, 0x4a, 0x01], "vpopcntd zmm1{k5}{z}, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x55, 0x0a], "vpopcntd zmm1, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x55, 0x0a], "vpopcntd zmm1{k5}, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x55, 0x4a, 0x01], "vpopcntd zmm1, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x55, 0x4a, 0x01], "vpopcntd zmm1{k5}, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x55, 0x0a], "vpopcntd xmm1{k5}{z}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x55, 0x4a, 0x01], "vpopcntd xmm1{k5}{z}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x55, 0x0a], "vpopcntd xmm1, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x55, 0x0a], "vpopcntd xmm1{k5}, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x55, 0x4a, 0x01], "vpopcntd xmm1, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x55, 0x4a, 0x01], "vpopcntd xmm1{k5}, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x55, 0xca], "vpopcntd zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x55, 0x0a], "vpopcntd zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x55, 0x4a, 0x01], "vpopcntd zmm1{k5}{z}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x55, 0xca], "vpopcntd zmm1, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x55, 0xca], "vpopcntd zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x55, 0x0a], "vpopcntd zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x55, 0x0a], "vpopcntd zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x55, 0x4a, 0x01], "vpopcntd zmm1, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x55, 0x4a, 0x01], "vpopcntd zmm1{k5}, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x55, 0xca], "vpopcntd xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x55, 0x0a], "vpopcntd xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x55, 0x4a, 0x01], "vpopcntd xmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x55, 0xca], "vpopcntd xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x55, 0xca], "vpopcntd xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x55, 0x0a], "vpopcntd xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x55, 0x0a], "vpopcntd xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x55, 0x4a, 0x01], "vpopcntd xmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x55, 0x4a, 0x01], "vpopcntd xmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x58, 0xca], "vpbroadcastd ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x58, 0x0a], "vpbroadcastd ymm1{k5}{z}, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x58, 0x4a, 0x01], "vpbroadcastd ymm1{k5}{z}, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x58, 0xca], "vpbroadcastd ymm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x58, 0xca], "vpbroadcastd ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x58, 0x0a], "vpbroadcastd ymm1, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x58, 0x0a], "vpbroadcastd ymm1{k5}, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x58, 0x4a, 0x01], "vpbroadcastd ymm1, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x58, 0x4a, 0x01], "vpbroadcastd ymm1{k5}, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x58, 0xca], "vpbroadcastd zmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x58, 0x0a], "vpbroadcastd zmm1{k5}{z}, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x58, 0x4a, 0x01], "vpbroadcastd zmm1{k5}{z}, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x58, 0xca], "vpbroadcastd zmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x58, 0xca], "vpbroadcastd zmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x58, 0x0a], "vpbroadcastd zmm1, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x58, 0x0a], "vpbroadcastd zmm1{k5}, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x58, 0x4a, 0x01], "vpbroadcastd zmm1, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x58, 0x4a, 0x01], "vpbroadcastd zmm1{k5}, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x58, 0xca], "vpbroadcastd xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x58, 0x0a], "vpbroadcastd xmm1{k5}{z}, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x58, 0x4a, 0x01], "vpbroadcastd xmm1{k5}{z}, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x58, 0xca], "vpbroadcastd xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x58, 0xca], "vpbroadcastd xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x58, 0x0a], "vpbroadcastd xmm1, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x58, 0x0a], "vpbroadcastd xmm1{k5}, dword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x58, 0x4a, 0x01], "vpbroadcastd xmm1, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x58, 0x4a, 0x01], "vpbroadcastd xmm1{k5}, dword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x59, 0xca], "vpbroadcastq ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x59, 0x0a], "vpbroadcastq ymm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x59, 0x4a, 0x01], "vpbroadcastq ymm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x59, 0xca], "vpbroadcastq ymm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x59, 0xca], "vpbroadcastq ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x59, 0x0a], "vpbroadcastq ymm1, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x59, 0x0a], "vpbroadcastq ymm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x59, 0x4a, 0x01], "vpbroadcastq ymm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x59, 0x4a, 0x01], "vpbroadcastq ymm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x59, 0xca], "vbroadcasti32x2 ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x59, 0x0a], "vbroadcasti32x2 ymm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x59, 0x4a, 0x01], "vbroadcasti32x2 ymm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x59, 0xca], "vbroadcasti32x2 ymm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x59, 0xca], "vbroadcasti32x2 ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x59, 0x0a], "vbroadcasti32x2 ymm1, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x59, 0x0a], "vbroadcasti32x2 ymm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x59, 0x4a, 0x01], "vbroadcasti32x2 ymm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x59, 0x4a, 0x01], "vbroadcasti32x2 ymm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x59, 0xca], "vpbroadcastq zmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x59, 0x0a], "vpbroadcastq zmm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x59, 0x4a, 0x01], "vpbroadcastq zmm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x59, 0xca], "vpbroadcastq zmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x59, 0xca], "vpbroadcastq zmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x59, 0x0a], "vpbroadcastq zmm1, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x59, 0x0a], "vpbroadcastq zmm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x59, 0x4a, 0x01], "vpbroadcastq zmm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x59, 0x4a, 0x01], "vpbroadcastq zmm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x59, 0xca], "vpbroadcastq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x59, 0x0a], "vpbroadcastq xmm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x59, 0x4a, 0x01], "vpbroadcastq xmm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x59, 0xca], "vpbroadcastq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x59, 0xca], "vpbroadcastq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x59, 0x0a], "vpbroadcastq xmm1, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x59, 0x0a], "vpbroadcastq xmm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x59, 0x4a, 0x01], "vpbroadcastq xmm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x59, 0x4a, 0x01], "vpbroadcastq xmm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x59, 0xca], "vbroadcasti32x2 zmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x59, 0x0a], "vbroadcasti32x2 zmm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x59, 0x4a, 0x01], "vbroadcasti32x2 zmm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x59, 0xca], "vbroadcasti32x2 zmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x59, 0xca], "vbroadcasti32x2 zmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x59, 0x0a], "vbroadcasti32x2 zmm1, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x59, 0x0a], "vbroadcasti32x2 zmm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x59, 0x4a, 0x01], "vbroadcasti32x2 zmm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x59, 0x4a, 0x01], "vbroadcasti32x2 zmm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x59, 0xca], "vbroadcasti32x2 xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x59, 0x0a], "vbroadcasti32x2 xmm1{k5}{z}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x59, 0x4a, 0x01], "vbroadcasti32x2 xmm1{k5}{z}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x59, 0xca], "vbroadcasti32x2 xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x59, 0xca], "vbroadcasti32x2 xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x59, 0x0a], "vbroadcasti32x2 xmm1, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x59, 0x0a], "vbroadcasti32x2 xmm1{k5}, qword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x59, 0x4a, 0x01], "vbroadcasti32x2 xmm1, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x59, 0x4a, 0x01], "vbroadcasti32x2 xmm1{k5}, qword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x5a, 0x0a], "vbroadcasti64x2 ymm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x5a, 0x4a, 0x01], "vbroadcasti64x2 ymm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x5a, 0x0a], "vbroadcasti64x2 ymm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x5a, 0x0a], "vbroadcasti64x2 ymm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x5a, 0x4a, 0x01], "vbroadcasti64x2 ymm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x5a, 0x4a, 0x01], "vbroadcasti64x2 ymm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x5a, 0x0a], "vbroadcasti32x4 ymm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x5a, 0x4a, 0x01], "vbroadcasti32x4 ymm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x5a, 0x0a], "vbroadcasti32x4 ymm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x5a, 0x0a], "vbroadcasti32x4 ymm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x5a, 0x4a, 0x01], "vbroadcasti32x4 ymm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x5a, 0x4a, 0x01], "vbroadcasti32x4 ymm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x5a, 0x0a], "vbroadcasti64x2 zmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x5a, 0x4a, 0x01], "vbroadcasti64x2 zmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x5a, 0x0a], "vbroadcasti64x2 zmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x5a, 0x0a], "vbroadcasti64x2 zmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x5a, 0x4a, 0x01], "vbroadcasti64x2 zmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x5a, 0x4a, 0x01], "vbroadcasti64x2 zmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x5a, 0x0a], "vbroadcasti32x4 zmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x5a, 0x4a, 0x01], "vbroadcasti32x4 zmm1{k5}{z}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x5a, 0x0a], "vbroadcasti32x4 zmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x5a, 0x0a], "vbroadcasti32x4 zmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x5a, 0x4a, 0x01], "vbroadcasti32x4 zmm1, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x5a, 0x4a, 0x01], "vbroadcasti32x4 zmm1{k5}, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x5b, 0x0a], "vbroadcasti64x4 zmm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x5b, 0x4a, 0x01], "vbroadcasti64x4 zmm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x5b, 0x0a], "vbroadcasti64x4 zmm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x5b, 0x0a], "vbroadcasti64x4 zmm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x5b, 0x4a, 0x01], "vbroadcasti64x4 zmm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x5b, 0x4a, 0x01], "vbroadcasti64x4 zmm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x5b, 0x0a], "vbroadcasti32x8 zmm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x5b, 0x4a, 0x01], "vbroadcasti32x8 zmm1{k5}{z}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x5b, 0x0a], "vbroadcasti32x8 zmm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x5b, 0x0a], "vbroadcasti32x8 zmm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x5b, 0x4a, 0x01], "vbroadcasti32x8 zmm1, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x5b, 0x4a, 0x01], "vbroadcasti32x8 zmm1{k5}, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x62, 0xca], "vpexpandw ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x62, 0x0a], "vpexpandw ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x62, 0x4a, 0x01], "vpexpandw ymm1{k5}{z}, ymmword [edx + 0x2]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x62, 0xca], "vpexpandw ymm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x62, 0xca], "vpexpandw ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x62, 0x0a], "vpexpandw ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x62, 0x0a], "vpexpandw ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x62, 0x4a, 0x01], "vpexpandw ymm1, ymmword [edx + 0x2]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x62, 0x4a, 0x01], "vpexpandw ymm1{k5}, ymmword [edx + 0x2]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x62, 0xca], "vpexpandb ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x62, 0x0a], "vpexpandb ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x62, 0x4a, 0x01], "vpexpandb ymm1{k5}{z}, ymmword [edx + 0x1]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x62, 0xca], "vpexpandb ymm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x62, 0xca], "vpexpandb ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x62, 0x0a], "vpexpandb ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x62, 0x0a], "vpexpandb ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x62, 0x4a, 0x01], "vpexpandb ymm1, ymmword [edx + 0x1]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x62, 0x4a, 0x01], "vpexpandb ymm1{k5}, ymmword [edx + 0x1]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x62, 0xca], "vpexpandw zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x62, 0x0a], "vpexpandw zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x62, 0x4a, 0x01], "vpexpandw zmm1{k5}{z}, zmmword [edx + 0x2]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x62, 0xca], "vpexpandw zmm1, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x62, 0xca], "vpexpandw zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x62, 0x0a], "vpexpandw zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x62, 0x0a], "vpexpandw zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x62, 0x4a, 0x01], "vpexpandw zmm1, zmmword [edx + 0x2]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x62, 0x4a, 0x01], "vpexpandw zmm1{k5}, zmmword [edx + 0x2]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x62, 0xca], "vpexpandw xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x62, 0x0a], "vpexpandw xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x62, 0x4a, 0x01], "vpexpandw xmm1{k5}{z}, xmmword [edx + 0x2]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x62, 0xca], "vpexpandw xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x62, 0xca], "vpexpandw xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x62, 0x0a], "vpexpandw xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x62, 0x0a], "vpexpandw xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x62, 0x4a, 0x01], "vpexpandw xmm1, xmmword [edx + 0x2]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x62, 0x4a, 0x01], "vpexpandw xmm1{k5}, xmmword [edx + 0x2]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x62, 0xca], "vpexpandb zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x62, 0x0a], "vpexpandb zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x62, 0x4a, 0x01], "vpexpandb zmm1{k5}{z}, zmmword [edx + 0x1]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x62, 0xca], "vpexpandb zmm1, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x62, 0xca], "vpexpandb zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x62, 0x0a], "vpexpandb zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x62, 0x0a], "vpexpandb zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x62, 0x4a, 0x01], "vpexpandb zmm1, zmmword [edx + 0x1]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x62, 0x4a, 0x01], "vpexpandb zmm1{k5}, zmmword [edx + 0x1]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x62, 0xca], "vpexpandb xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x62, 0x0a], "vpexpandb xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x62, 0x4a, 0x01], "vpexpandb xmm1{k5}{z}, xmmword [edx + 0x1]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x62, 0xca], "vpexpandb xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x62, 0xca], "vpexpandb xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x62, 0x0a], "vpexpandb xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x62, 0x0a], "vpexpandb xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x62, 0x4a, 0x01], "vpexpandb xmm1, xmmword [edx + 0x1]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x62, 0x4a, 0x01], "vpexpandb xmm1{k5}, xmmword [edx + 0x1]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x63, 0xca], "vpcompressw ymm2{k5}{z}, ymm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x63, 0xca], "vpcompressw ymm2, ymm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x63, 0xca], "vpcompressw ymm2{k5}, ymm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x63, 0x0a], "vpcompressw ymmword [edx], ymm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x63, 0x0a], "vpcompressw ymmword [edx]{k5}, ymm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x63, 0x4a, 0x01], "vpcompressw ymmword [edx + 0x2], ymm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x63, 0x4a, 0x01], "vpcompressw ymmword [edx + 0x2]{k5}, ymm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x63, 0xca], "vpcompressb ymm2{k5}{z}, ymm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x63, 0xca], "vpcompressb ymm2, ymm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x63, 0xca], "vpcompressb ymm2{k5}, ymm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x63, 0x0a], "vpcompressb ymmword [edx], ymm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x63, 0x0a], "vpcompressb ymmword [edx]{k5}, ymm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x63, 0x4a, 0x01], "vpcompressb ymmword [edx + 0x1], ymm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x63, 0x4a, 0x01], "vpcompressb ymmword [edx + 0x1]{k5}, ymm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x63, 0xca], "vpcompressw zmm2{k5}{z}, zmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x63, 0xca], "vpcompressw zmm2, zmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x63, 0xca], "vpcompressw zmm2{k5}, zmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x63, 0x0a], "vpcompressw zmmword [edx], zmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x63, 0x0a], "vpcompressw zmmword [edx]{k5}, zmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x63, 0x4a, 0x01], "vpcompressw zmmword [edx + 0x2], zmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x63, 0x4a, 0x01], "vpcompressw zmmword [edx + 0x2]{k5}, zmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x63, 0xca], "vpcompressw xmm2{k5}{z}, xmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x63, 0xca], "vpcompressw xmm2, xmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x63, 0xca], "vpcompressw xmm2{k5}, xmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x63, 0x0a], "vpcompressw xmmword [edx], xmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x63, 0x0a], "vpcompressw xmmword [edx]{k5}, xmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x63, 0x4a, 0x01], "vpcompressw xmmword [edx + 0x2], xmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x63, 0x4a, 0x01], "vpcompressw xmmword [edx + 0x2]{k5}, xmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x63, 0xca], "vpcompressb zmm2{k5}{z}, zmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x63, 0xca], "vpcompressb zmm2, zmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x63, 0xca], "vpcompressb zmm2{k5}, zmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x63, 0x0a], "vpcompressb zmmword [edx], zmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x63, 0x0a], "vpcompressb zmmword [edx]{k5}, zmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x63, 0x4a, 0x01], "vpcompressb zmmword [edx + 0x1], zmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x63, 0x4a, 0x01], "vpcompressb zmmword [edx + 0x1]{k5}, zmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x63, 0xca], "vpcompressb xmm2{k5}{z}, xmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x63, 0xca], "vpcompressb xmm2, xmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x63, 0xca], "vpcompressb xmm2{k5}, xmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x63, 0x0a], "vpcompressb xmmword [edx], xmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x63, 0x0a], "vpcompressb xmmword [edx]{k5}, xmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x63, 0x4a, 0x01], "vpcompressb xmmword [edx + 0x1], xmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x63, 0x4a, 0x01], "vpcompressb xmmword [edx + 0x1]{k5}, xmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x64, 0x0a], "vpblendmq ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x64, 0x4a, 0x01], "vpblendmq ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x64, 0x0a], "vpblendmq ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x64, 0x0a], "vpblendmq ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x64, 0x4a, 0x01], "vpblendmq ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x64, 0x4a, 0x01], "vpblendmq ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x64, 0xca], "vpblendmq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x64, 0x0a], "vpblendmq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x64, 0x4a, 0x01], "vpblendmq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x64, 0xca], "vpblendmq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x64, 0xca], "vpblendmq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x64, 0x0a], "vpblendmq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x64, 0x0a], "vpblendmq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x64, 0x4a, 0x01], "vpblendmq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x64, 0x4a, 0x01], "vpblendmq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x64, 0x0a], "vpblendmd ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x64, 0x4a, 0x01], "vpblendmd ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x64, 0x0a], "vpblendmd ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x64, 0x0a], "vpblendmd ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x64, 0x4a, 0x01], "vpblendmd ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x64, 0x4a, 0x01], "vpblendmd ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x64, 0xca], "vpblendmd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x64, 0x0a], "vpblendmd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x64, 0x4a, 0x01], "vpblendmd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x64, 0xca], "vpblendmd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x64, 0xca], "vpblendmd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x64, 0x0a], "vpblendmd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x64, 0x0a], "vpblendmd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x64, 0x4a, 0x01], "vpblendmd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x64, 0x4a, 0x01], "vpblendmd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x64, 0x0a], "vpblendmq zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x64, 0x4a, 0x01], "vpblendmq zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x64, 0x0a], "vpblendmq zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x64, 0x0a], "vpblendmq zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x64, 0x4a, 0x01], "vpblendmq zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x64, 0x4a, 0x01], "vpblendmq zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x64, 0x0a], "vpblendmq xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x64, 0x4a, 0x01], "vpblendmq xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x64, 0x0a], "vpblendmq xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x64, 0x0a], "vpblendmq xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x64, 0x4a, 0x01], "vpblendmq xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x64, 0x4a, 0x01], "vpblendmq xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x64, 0xca], "vpblendmq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x64, 0x0a], "vpblendmq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x64, 0x4a, 0x01], "vpblendmq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x64, 0xca], "vpblendmq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x64, 0xca], "vpblendmq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x64, 0x0a], "vpblendmq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x64, 0x0a], "vpblendmq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x64, 0x4a, 0x01], "vpblendmq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x64, 0x4a, 0x01], "vpblendmq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x64, 0xca], "vpblendmq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x64, 0x0a], "vpblendmq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x64, 0x4a, 0x01], "vpblendmq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x64, 0xca], "vpblendmq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x64, 0xca], "vpblendmq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x64, 0x0a], "vpblendmq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x64, 0x0a], "vpblendmq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x64, 0x4a, 0x01], "vpblendmq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x64, 0x4a, 0x01], "vpblendmq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x64, 0x0a], "vpblendmd zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x64, 0x4a, 0x01], "vpblendmd zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x64, 0x0a], "vpblendmd zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x64, 0x0a], "vpblendmd zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x64, 0x4a, 0x01], "vpblendmd zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x64, 0x4a, 0x01], "vpblendmd zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x64, 0x0a], "vpblendmd xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x64, 0x4a, 0x01], "vpblendmd xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x64, 0x0a], "vpblendmd xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x64, 0x0a], "vpblendmd xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x64, 0x4a, 0x01], "vpblendmd xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x64, 0x4a, 0x01], "vpblendmd xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x64, 0xca], "vpblendmd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x64, 0x0a], "vpblendmd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x64, 0x4a, 0x01], "vpblendmd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x64, 0xca], "vpblendmd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x64, 0xca], "vpblendmd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x64, 0x0a], "vpblendmd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x64, 0x0a], "vpblendmd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x64, 0x4a, 0x01], "vpblendmd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x64, 0x4a, 0x01], "vpblendmd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x64, 0xca], "vpblendmd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x64, 0x0a], "vpblendmd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x64, 0x4a, 0x01], "vpblendmd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x64, 0xca], "vpblendmd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x64, 0xca], "vpblendmd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x64, 0x0a], "vpblendmd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x64, 0x0a], "vpblendmd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x64, 0x4a, 0x01], "vpblendmd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x64, 0x4a, 0x01], "vpblendmd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x65, 0x0a], "vblendmpd ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x65, 0x4a, 0x01], "vblendmpd ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x65, 0x0a], "vblendmpd ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x65, 0x0a], "vblendmpd ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x65, 0x4a, 0x01], "vblendmpd ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x65, 0x4a, 0x01], "vblendmpd ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x65, 0xca], "vblendmpd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x65, 0x0a], "vblendmpd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x65, 0x4a, 0x01], "vblendmpd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x65, 0xca], "vblendmpd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x65, 0xca], "vblendmpd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x65, 0x0a], "vblendmpd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x65, 0x0a], "vblendmpd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x65, 0x4a, 0x01], "vblendmpd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x65, 0x4a, 0x01], "vblendmpd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x65, 0x0a], "vblendmps ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x65, 0x4a, 0x01], "vblendmps ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x65, 0x0a], "vblendmps ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x65, 0x0a], "vblendmps ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x65, 0x4a, 0x01], "vblendmps ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x65, 0x4a, 0x01], "vblendmps ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x65, 0xca], "vblendmps ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x65, 0x0a], "vblendmps ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x65, 0x4a, 0x01], "vblendmps ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x65, 0xca], "vblendmps ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x65, 0xca], "vblendmps ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x65, 0x0a], "vblendmps ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x65, 0x0a], "vblendmps ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x65, 0x4a, 0x01], "vblendmps ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x65, 0x4a, 0x01], "vblendmps ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x65, 0x0a], "vblendmpd zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x65, 0x4a, 0x01], "vblendmpd zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x65, 0x0a], "vblendmpd zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x65, 0x0a], "vblendmpd zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x65, 0x4a, 0x01], "vblendmpd zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x65, 0x4a, 0x01], "vblendmpd zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x65, 0x0a], "vblendmpd xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x65, 0x4a, 0x01], "vblendmpd xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x65, 0x0a], "vblendmpd xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x65, 0x0a], "vblendmpd xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x65, 0x4a, 0x01], "vblendmpd xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x65, 0x4a, 0x01], "vblendmpd xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x65, 0xca], "vblendmpd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x65, 0x0a], "vblendmpd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x65, 0x4a, 0x01], "vblendmpd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x65, 0xca], "vblendmpd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x65, 0xca], "vblendmpd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x65, 0x0a], "vblendmpd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x65, 0x0a], "vblendmpd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x65, 0x4a, 0x01], "vblendmpd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x65, 0x4a, 0x01], "vblendmpd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x65, 0xca], "vblendmpd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x65, 0x0a], "vblendmpd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x65, 0x4a, 0x01], "vblendmpd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x65, 0xca], "vblendmpd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x65, 0xca], "vblendmpd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x65, 0x0a], "vblendmpd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x65, 0x0a], "vblendmpd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x65, 0x4a, 0x01], "vblendmpd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x65, 0x4a, 0x01], "vblendmpd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x65, 0x0a], "vblendmps zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x65, 0x4a, 0x01], "vblendmps zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x65, 0x0a], "vblendmps zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x65, 0x0a], "vblendmps zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x65, 0x4a, 0x01], "vblendmps zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x65, 0x4a, 0x01], "vblendmps zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x65, 0x0a], "vblendmps xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x65, 0x4a, 0x01], "vblendmps xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x65, 0x0a], "vblendmps xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x65, 0x0a], "vblendmps xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x65, 0x4a, 0x01], "vblendmps xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x65, 0x4a, 0x01], "vblendmps xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x65, 0xca], "vblendmps zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x65, 0x0a], "vblendmps zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x65, 0x4a, 0x01], "vblendmps zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x65, 0xca], "vblendmps zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x65, 0xca], "vblendmps zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x65, 0x0a], "vblendmps zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x65, 0x0a], "vblendmps zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x65, 0x4a, 0x01], "vblendmps zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x65, 0x4a, 0x01], "vblendmps zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x65, 0xca], "vblendmps xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x65, 0x0a], "vblendmps xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x65, 0x4a, 0x01], "vblendmps xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x65, 0xca], "vblendmps xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x65, 0xca], "vblendmps xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x65, 0x0a], "vblendmps xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x65, 0x0a], "vblendmps xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x65, 0x4a, 0x01], "vblendmps xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x65, 0x4a, 0x01], "vblendmps xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x66, 0xca], "vpblendmw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x66, 0x0a], "vpblendmw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x66, 0x4a, 0x01], "vpblendmw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x66, 0xca], "vpblendmw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x66, 0xca], "vpblendmw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x66, 0x0a], "vpblendmw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x66, 0x0a], "vpblendmw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x66, 0x4a, 0x01], "vpblendmw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x66, 0x4a, 0x01], "vpblendmw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x66, 0xca], "vpblendmb ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x66, 0x0a], "vpblendmb ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x66, 0x4a, 0x01], "vpblendmb ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x66, 0xca], "vpblendmb ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x66, 0xca], "vpblendmb ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x66, 0x0a], "vpblendmb ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x66, 0x0a], "vpblendmb ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x66, 0x4a, 0x01], "vpblendmb ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x66, 0x4a, 0x01], "vpblendmb ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x66, 0xca], "vpblendmw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x66, 0x0a], "vpblendmw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x66, 0x4a, 0x01], "vpblendmw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x66, 0xca], "vpblendmw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x66, 0xca], "vpblendmw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x66, 0x0a], "vpblendmw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x66, 0x0a], "vpblendmw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x66, 0x4a, 0x01], "vpblendmw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x66, 0x4a, 0x01], "vpblendmw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x66, 0xca], "vpblendmw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x66, 0x0a], "vpblendmw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x66, 0x4a, 0x01], "vpblendmw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x66, 0xca], "vpblendmw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x66, 0xca], "vpblendmw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x66, 0x0a], "vpblendmw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x66, 0x0a], "vpblendmw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x66, 0x4a, 0x01], "vpblendmw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x66, 0x4a, 0x01], "vpblendmw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x66, 0xca], "vpblendmb zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x66, 0x0a], "vpblendmb zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x66, 0x4a, 0x01], "vpblendmb zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x66, 0xca], "vpblendmb zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x66, 0xca], "vpblendmb zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x66, 0x0a], "vpblendmb zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x66, 0x0a], "vpblendmb zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x66, 0x4a, 0x01], "vpblendmb zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x66, 0x4a, 0x01], "vpblendmb zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x66, 0xca], "vpblendmb xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x66, 0x0a], "vpblendmb xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x66, 0x4a, 0x01], "vpblendmb xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x66, 0xca], "vpblendmb xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x66, 0xca], "vpblendmb xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x66, 0x0a], "vpblendmb xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x66, 0x0a], "vpblendmb xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x66, 0x4a, 0x01], "vpblendmb xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x66, 0x4a, 0x01], "vpblendmb xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x70, 0xca], "vpshldvw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x70, 0x0a], "vpshldvw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x70, 0x4a, 0x01], "vpshldvw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x70, 0xca], "vpshldvw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x70, 0xca], "vpshldvw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x70, 0x0a], "vpshldvw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x70, 0x0a], "vpshldvw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x70, 0x4a, 0x01], "vpshldvw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x70, 0x4a, 0x01], "vpshldvw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x70, 0xca], "vpshldvw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x70, 0x0a], "vpshldvw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x70, 0x4a, 0x01], "vpshldvw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x70, 0xca], "vpshldvw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x70, 0xca], "vpshldvw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x70, 0x0a], "vpshldvw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x70, 0x0a], "vpshldvw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x70, 0x4a, 0x01], "vpshldvw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x70, 0x4a, 0x01], "vpshldvw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x70, 0xca], "vpshldvw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x70, 0x0a], "vpshldvw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x70, 0x4a, 0x01], "vpshldvw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x70, 0xca], "vpshldvw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x70, 0xca], "vpshldvw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x70, 0x0a], "vpshldvw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x70, 0x0a], "vpshldvw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x70, 0x4a, 0x01], "vpshldvw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x70, 0x4a, 0x01], "vpshldvw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x71, 0x0a], "vpshldvq ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x71, 0x4a, 0x01], "vpshldvq ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x71, 0x0a], "vpshldvq ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x71, 0x0a], "vpshldvq ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x71, 0x4a, 0x01], "vpshldvq ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x71, 0x4a, 0x01], "vpshldvq ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x71, 0xca], "vpshldvq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x71, 0x0a], "vpshldvq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x71, 0x4a, 0x01], "vpshldvq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x71, 0xca], "vpshldvq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x71, 0xca], "vpshldvq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x71, 0x0a], "vpshldvq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x71, 0x0a], "vpshldvq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x71, 0x4a, 0x01], "vpshldvq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x71, 0x4a, 0x01], "vpshldvq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x71, 0x0a], "vpshldvd ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x71, 0x4a, 0x01], "vpshldvd ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x71, 0x0a], "vpshldvd ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x71, 0x0a], "vpshldvd ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x71, 0x4a, 0x01], "vpshldvd ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x71, 0x4a, 0x01], "vpshldvd ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x71, 0xca], "vpshldvd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x71, 0x0a], "vpshldvd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x71, 0x4a, 0x01], "vpshldvd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x71, 0xca], "vpshldvd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x71, 0xca], "vpshldvd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x71, 0x0a], "vpshldvd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x71, 0x0a], "vpshldvd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x71, 0x4a, 0x01], "vpshldvd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x71, 0x4a, 0x01], "vpshldvd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x71, 0x0a], "vpshldvq zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x71, 0x4a, 0x01], "vpshldvq zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x71, 0x0a], "vpshldvq zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x71, 0x0a], "vpshldvq zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x71, 0x4a, 0x01], "vpshldvq zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x71, 0x4a, 0x01], "vpshldvq zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x71, 0x0a], "vpshldvq xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x71, 0x4a, 0x01], "vpshldvq xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x71, 0x0a], "vpshldvq xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x71, 0x0a], "vpshldvq xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x71, 0x4a, 0x01], "vpshldvq xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x71, 0x4a, 0x01], "vpshldvq xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x71, 0xca], "vpshldvq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x71, 0x0a], "vpshldvq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x71, 0x4a, 0x01], "vpshldvq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x71, 0xca], "vpshldvq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x71, 0xca], "vpshldvq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x71, 0x0a], "vpshldvq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x71, 0x0a], "vpshldvq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x71, 0x4a, 0x01], "vpshldvq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x71, 0x4a, 0x01], "vpshldvq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x71, 0xca], "vpshldvq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x71, 0x0a], "vpshldvq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x71, 0x4a, 0x01], "vpshldvq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x71, 0xca], "vpshldvq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x71, 0xca], "vpshldvq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x71, 0x0a], "vpshldvq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x71, 0x0a], "vpshldvq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x71, 0x4a, 0x01], "vpshldvq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x71, 0x4a, 0x01], "vpshldvq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x71, 0x0a], "vpshldvd zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x71, 0x4a, 0x01], "vpshldvd zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x71, 0x0a], "vpshldvd zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x71, 0x0a], "vpshldvd zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x71, 0x4a, 0x01], "vpshldvd zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x71, 0x4a, 0x01], "vpshldvd zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x71, 0x0a], "vpshldvd xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x71, 0x4a, 0x01], "vpshldvd xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x71, 0x0a], "vpshldvd xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x71, 0x0a], "vpshldvd xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x71, 0x4a, 0x01], "vpshldvd xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x71, 0x4a, 0x01], "vpshldvd xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x71, 0xca], "vpshldvd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x71, 0x0a], "vpshldvd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x71, 0x4a, 0x01], "vpshldvd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x71, 0xca], "vpshldvd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x71, 0xca], "vpshldvd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x71, 0x0a], "vpshldvd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x71, 0x0a], "vpshldvd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x71, 0x4a, 0x01], "vpshldvd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x71, 0x4a, 0x01], "vpshldvd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x71, 0xca], "vpshldvd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x71, 0x0a], "vpshldvd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x71, 0x4a, 0x01], "vpshldvd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x71, 0xca], "vpshldvd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x71, 0xca], "vpshldvd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x71, 0x0a], "vpshldvd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x71, 0x0a], "vpshldvd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x71, 0x4a, 0x01], "vpshldvd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x71, 0x4a, 0x01], "vpshldvd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x72, 0xca], "vpshrdvw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x72, 0x0a], "vpshrdvw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x72, 0x4a, 0x01], "vpshrdvw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x72, 0xca], "vpshrdvw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x72, 0xca], "vpshrdvw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x72, 0x0a], "vpshrdvw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x72, 0x0a], "vpshrdvw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x72, 0x4a, 0x01], "vpshrdvw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x72, 0x4a, 0x01], "vpshrdvw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x72, 0xca], "vpshrdvw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x72, 0x0a], "vpshrdvw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x72, 0x4a, 0x01], "vpshrdvw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x72, 0xca], "vpshrdvw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x72, 0xca], "vpshrdvw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x72, 0x0a], "vpshrdvw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x72, 0x0a], "vpshrdvw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x72, 0x4a, 0x01], "vpshrdvw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x72, 0x4a, 0x01], "vpshrdvw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x72, 0xca], "vpshrdvw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x72, 0x0a], "vpshrdvw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x72, 0x4a, 0x01], "vpshrdvw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x72, 0xca], "vpshrdvw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x72, 0xca], "vpshrdvw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x72, 0x0a], "vpshrdvw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x72, 0x0a], "vpshrdvw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x72, 0x4a, 0x01], "vpshrdvw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x72, 0x4a, 0x01], "vpshrdvw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x73, 0x0a], "vpshrdvq ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x73, 0x4a, 0x01], "vpshrdvq ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x73, 0x0a], "vpshrdvq ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x73, 0x0a], "vpshrdvq ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x73, 0x4a, 0x01], "vpshrdvq ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x73, 0x4a, 0x01], "vpshrdvq ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x73, 0xca], "vpshrdvq ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x73, 0x0a], "vpshrdvq ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x73, 0x4a, 0x01], "vpshrdvq ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x73, 0xca], "vpshrdvq ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x73, 0xca], "vpshrdvq ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x73, 0x0a], "vpshrdvq ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x73, 0x0a], "vpshrdvq ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x73, 0x4a, 0x01], "vpshrdvq ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x73, 0x4a, 0x01], "vpshrdvq ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x73, 0x0a], "vpshrdvd ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x73, 0x4a, 0x01], "vpshrdvd ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x73, 0x0a], "vpshrdvd ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x73, 0x0a], "vpshrdvd ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x73, 0x4a, 0x01], "vpshrdvd ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x73, 0x4a, 0x01], "vpshrdvd ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x73, 0xca], "vpshrdvd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x73, 0x0a], "vpshrdvd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x73, 0x4a, 0x01], "vpshrdvd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x73, 0xca], "vpshrdvd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x73, 0xca], "vpshrdvd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x73, 0x0a], "vpshrdvd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x73, 0x0a], "vpshrdvd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x73, 0x4a, 0x01], "vpshrdvd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x73, 0x4a, 0x01], "vpshrdvd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x73, 0x0a], "vpshrdvq zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x73, 0x4a, 0x01], "vpshrdvq zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x73, 0x0a], "vpshrdvq zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x73, 0x0a], "vpshrdvq zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x73, 0x4a, 0x01], "vpshrdvq zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x73, 0x4a, 0x01], "vpshrdvq zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x73, 0x0a], "vpshrdvq xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x73, 0x4a, 0x01], "vpshrdvq xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x73, 0x0a], "vpshrdvq xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x73, 0x0a], "vpshrdvq xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x73, 0x4a, 0x01], "vpshrdvq xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x73, 0x4a, 0x01], "vpshrdvq xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x73, 0xca], "vpshrdvq zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x73, 0x0a], "vpshrdvq zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x73, 0x4a, 0x01], "vpshrdvq zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x73, 0xca], "vpshrdvq zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x73, 0xca], "vpshrdvq zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x73, 0x0a], "vpshrdvq zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x73, 0x0a], "vpshrdvq zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x73, 0x4a, 0x01], "vpshrdvq zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x73, 0x4a, 0x01], "vpshrdvq zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x73, 0xca], "vpshrdvq xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x73, 0x0a], "vpshrdvq xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x73, 0x4a, 0x01], "vpshrdvq xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x73, 0xca], "vpshrdvq xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x73, 0xca], "vpshrdvq xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x73, 0x0a], "vpshrdvq xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x73, 0x0a], "vpshrdvq xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x73, 0x4a, 0x01], "vpshrdvq xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x73, 0x4a, 0x01], "vpshrdvq xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x73, 0x0a], "vpshrdvd zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x73, 0x4a, 0x01], "vpshrdvd zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x73, 0x0a], "vpshrdvd zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x73, 0x0a], "vpshrdvd zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x73, 0x4a, 0x01], "vpshrdvd zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x73, 0x4a, 0x01], "vpshrdvd zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x73, 0x0a], "vpshrdvd xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x73, 0x4a, 0x01], "vpshrdvd xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x73, 0x0a], "vpshrdvd xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x73, 0x0a], "vpshrdvd xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x73, 0x4a, 0x01], "vpshrdvd xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x73, 0x4a, 0x01], "vpshrdvd xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x73, 0xca], "vpshrdvd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x73, 0x0a], "vpshrdvd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x73, 0x4a, 0x01], "vpshrdvd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x73, 0xca], "vpshrdvd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x73, 0xca], "vpshrdvd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x73, 0x0a], "vpshrdvd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x73, 0x0a], "vpshrdvd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x73, 0x4a, 0x01], "vpshrdvd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x73, 0x4a, 0x01], "vpshrdvd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x73, 0xca], "vpshrdvd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x73, 0x0a], "vpshrdvd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x73, 0x4a, 0x01], "vpshrdvd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x73, 0xca], "vpshrdvd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x73, 0xca], "vpshrdvd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x73, 0x0a], "vpshrdvd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x73, 0x0a], "vpshrdvd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x73, 0x4a, 0x01], "vpshrdvd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x73, 0x4a, 0x01], "vpshrdvd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x75, 0xca], "vpermi2w ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x75, 0x0a], "vpermi2w ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x75, 0x4a, 0x01], "vpermi2w ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x75, 0xca], "vpermi2w ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x75, 0xca], "vpermi2w ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x75, 0x0a], "vpermi2w ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x75, 0x0a], "vpermi2w ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x75, 0x4a, 0x01], "vpermi2w ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x75, 0x4a, 0x01], "vpermi2w ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x75, 0xca], "vpermi2b ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x75, 0x0a], "vpermi2b ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x75, 0x4a, 0x01], "vpermi2b ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x75, 0xca], "vpermi2b ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x75, 0xca], "vpermi2b ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x75, 0x0a], "vpermi2b ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x75, 0x0a], "vpermi2b ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x75, 0x4a, 0x01], "vpermi2b ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x75, 0x4a, 0x01], "vpermi2b ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x75, 0xca], "vpermi2w zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x75, 0x0a], "vpermi2w zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x75, 0x4a, 0x01], "vpermi2w zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x75, 0xca], "vpermi2w zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x75, 0xca], "vpermi2w zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x75, 0x0a], "vpermi2w zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x75, 0x0a], "vpermi2w zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x75, 0x4a, 0x01], "vpermi2w zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x75, 0x4a, 0x01], "vpermi2w zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x75, 0xca], "vpermi2w xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x75, 0x0a], "vpermi2w xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x75, 0x4a, 0x01], "vpermi2w xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x75, 0xca], "vpermi2w xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x75, 0xca], "vpermi2w xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x75, 0x0a], "vpermi2w xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x75, 0x0a], "vpermi2w xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x75, 0x4a, 0x01], "vpermi2w xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x75, 0x4a, 0x01], "vpermi2w xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x75, 0xca], "vpermi2b zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x75, 0x0a], "vpermi2b zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x75, 0x4a, 0x01], "vpermi2b zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x75, 0xca], "vpermi2b zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x75, 0xca], "vpermi2b zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x75, 0x0a], "vpermi2b zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x75, 0x0a], "vpermi2b zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x75, 0x4a, 0x01], "vpermi2b zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x75, 0x4a, 0x01], "vpermi2b zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x75, 0xca], "vpermi2b xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x75, 0x0a], "vpermi2b xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x75, 0x4a, 0x01], "vpermi2b xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x75, 0xca], "vpermi2b xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x75, 0xca], "vpermi2b xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x75, 0x0a], "vpermi2b xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x75, 0x0a], "vpermi2b xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x75, 0x4a, 0x01], "vpermi2b xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x75, 0x4a, 0x01], "vpermi2b xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x76, 0x0a], "vpermi2q ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x76, 0x4a, 0x01], "vpermi2q ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x76, 0x0a], "vpermi2q ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x76, 0x0a], "vpermi2q ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x76, 0x4a, 0x01], "vpermi2q ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x76, 0x4a, 0x01], "vpermi2q ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x76, 0xca], "vpermi2q ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x76, 0x0a], "vpermi2q ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x76, 0x4a, 0x01], "vpermi2q ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x76, 0xca], "vpermi2q ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x76, 0xca], "vpermi2q ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x76, 0x0a], "vpermi2q ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x76, 0x0a], "vpermi2q ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x76, 0x4a, 0x01], "vpermi2q ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x76, 0x4a, 0x01], "vpermi2q ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x76, 0x0a], "vpermi2d ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x76, 0x4a, 0x01], "vpermi2d ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x76, 0x0a], "vpermi2d ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x76, 0x0a], "vpermi2d ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x76, 0x4a, 0x01], "vpermi2d ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x76, 0x4a, 0x01], "vpermi2d ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x76, 0xca], "vpermi2d ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x76, 0x0a], "vpermi2d ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x76, 0x4a, 0x01], "vpermi2d ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x76, 0xca], "vpermi2d ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x76, 0xca], "vpermi2d ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x76, 0x0a], "vpermi2d ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x76, 0x0a], "vpermi2d ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x76, 0x4a, 0x01], "vpermi2d ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x76, 0x4a, 0x01], "vpermi2d ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x76, 0x0a], "vpermi2q zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x76, 0x4a, 0x01], "vpermi2q zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x76, 0x0a], "vpermi2q zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x76, 0x0a], "vpermi2q zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x76, 0x4a, 0x01], "vpermi2q zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x76, 0x4a, 0x01], "vpermi2q zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x76, 0x0a], "vpermi2q xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x76, 0x4a, 0x01], "vpermi2q xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x76, 0x0a], "vpermi2q xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x76, 0x0a], "vpermi2q xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x76, 0x4a, 0x01], "vpermi2q xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x76, 0x4a, 0x01], "vpermi2q xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x76, 0xca], "vpermi2q zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x76, 0x0a], "vpermi2q zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x76, 0x4a, 0x01], "vpermi2q zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x76, 0xca], "vpermi2q zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x76, 0xca], "vpermi2q zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x76, 0x0a], "vpermi2q zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x76, 0x0a], "vpermi2q zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x76, 0x4a, 0x01], "vpermi2q zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x76, 0x4a, 0x01], "vpermi2q zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x76, 0xca], "vpermi2q xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x76, 0x0a], "vpermi2q xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x76, 0x4a, 0x01], "vpermi2q xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x76, 0xca], "vpermi2q xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x76, 0xca], "vpermi2q xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x76, 0x0a], "vpermi2q xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x76, 0x0a], "vpermi2q xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x76, 0x4a, 0x01], "vpermi2q xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x76, 0x4a, 0x01], "vpermi2q xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x76, 0x0a], "vpermi2d zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x76, 0x4a, 0x01], "vpermi2d zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x76, 0x0a], "vpermi2d zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x76, 0x0a], "vpermi2d zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x76, 0x4a, 0x01], "vpermi2d zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x76, 0x4a, 0x01], "vpermi2d zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x76, 0x0a], "vpermi2d xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x76, 0x4a, 0x01], "vpermi2d xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x76, 0x0a], "vpermi2d xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x76, 0x0a], "vpermi2d xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x76, 0x4a, 0x01], "vpermi2d xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x76, 0x4a, 0x01], "vpermi2d xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x76, 0xca], "vpermi2d zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x76, 0x0a], "vpermi2d zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x76, 0x4a, 0x01], "vpermi2d zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x76, 0xca], "vpermi2d zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x76, 0xca], "vpermi2d zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x76, 0x0a], "vpermi2d zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x76, 0x0a], "vpermi2d zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x76, 0x4a, 0x01], "vpermi2d zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x76, 0x4a, 0x01], "vpermi2d zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x76, 0xca], "vpermi2d xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x76, 0x0a], "vpermi2d xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x76, 0x4a, 0x01], "vpermi2d xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x76, 0xca], "vpermi2d xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x76, 0xca], "vpermi2d xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x76, 0x0a], "vpermi2d xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x76, 0x0a], "vpermi2d xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x76, 0x4a, 0x01], "vpermi2d xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x76, 0x4a, 0x01], "vpermi2d xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x77, 0x0a], "vpermi2pd ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x77, 0x4a, 0x01], "vpermi2pd ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x77, 0x0a], "vpermi2pd ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x77, 0x0a], "vpermi2pd ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x77, 0x4a, 0x01], "vpermi2pd ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x77, 0x4a, 0x01], "vpermi2pd ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x77, 0xca], "vpermi2pd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x77, 0x0a], "vpermi2pd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x77, 0x4a, 0x01], "vpermi2pd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x77, 0xca], "vpermi2pd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x77, 0xca], "vpermi2pd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x77, 0x0a], "vpermi2pd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x77, 0x0a], "vpermi2pd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x77, 0x4a, 0x01], "vpermi2pd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x77, 0x4a, 0x01], "vpermi2pd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x77, 0x0a], "vpermi2ps ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x77, 0x4a, 0x01], "vpermi2ps ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x77, 0x0a], "vpermi2ps ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x77, 0x0a], "vpermi2ps ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x77, 0x4a, 0x01], "vpermi2ps ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x77, 0x4a, 0x01], "vpermi2ps ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x77, 0xca], "vpermi2ps ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x77, 0x0a], "vpermi2ps ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x77, 0x4a, 0x01], "vpermi2ps ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x77, 0xca], "vpermi2ps ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x77, 0xca], "vpermi2ps ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x77, 0x0a], "vpermi2ps ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x77, 0x0a], "vpermi2ps ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x77, 0x4a, 0x01], "vpermi2ps ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x77, 0x4a, 0x01], "vpermi2ps ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x77, 0x0a], "vpermi2pd zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x77, 0x4a, 0x01], "vpermi2pd zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x77, 0x0a], "vpermi2pd zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x77, 0x0a], "vpermi2pd zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x77, 0x4a, 0x01], "vpermi2pd zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x77, 0x4a, 0x01], "vpermi2pd zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x77, 0x0a], "vpermi2pd xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x77, 0x4a, 0x01], "vpermi2pd xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x77, 0x0a], "vpermi2pd xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x77, 0x0a], "vpermi2pd xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x77, 0x4a, 0x01], "vpermi2pd xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x77, 0x4a, 0x01], "vpermi2pd xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x77, 0xca], "vpermi2pd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x77, 0x0a], "vpermi2pd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x77, 0x4a, 0x01], "vpermi2pd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x77, 0xca], "vpermi2pd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x77, 0xca], "vpermi2pd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x77, 0x0a], "vpermi2pd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x77, 0x0a], "vpermi2pd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x77, 0x4a, 0x01], "vpermi2pd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x77, 0x4a, 0x01], "vpermi2pd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x77, 0xca], "vpermi2pd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x77, 0x0a], "vpermi2pd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x77, 0x4a, 0x01], "vpermi2pd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x77, 0xca], "vpermi2pd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x77, 0xca], "vpermi2pd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x77, 0x0a], "vpermi2pd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x77, 0x0a], "vpermi2pd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x77, 0x4a, 0x01], "vpermi2pd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x77, 0x4a, 0x01], "vpermi2pd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x77, 0x0a], "vpermi2ps zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x77, 0x4a, 0x01], "vpermi2ps zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x77, 0x0a], "vpermi2ps zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x77, 0x0a], "vpermi2ps zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x77, 0x4a, 0x01], "vpermi2ps zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x77, 0x4a, 0x01], "vpermi2ps zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x77, 0x0a], "vpermi2ps xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x77, 0x4a, 0x01], "vpermi2ps xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x77, 0x0a], "vpermi2ps xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x77, 0x0a], "vpermi2ps xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x77, 0x4a, 0x01], "vpermi2ps xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x77, 0x4a, 0x01], "vpermi2ps xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x77, 0xca], "vpermi2ps zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x77, 0x0a], "vpermi2ps zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x77, 0x4a, 0x01], "vpermi2ps zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x77, 0xca], "vpermi2ps zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x77, 0xca], "vpermi2ps zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x77, 0x0a], "vpermi2ps zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x77, 0x0a], "vpermi2ps zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x77, 0x4a, 0x01], "vpermi2ps zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x77, 0x4a, 0x01], "vpermi2ps zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x77, 0xca], "vpermi2ps xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x77, 0x0a], "vpermi2ps xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x77, 0x4a, 0x01], "vpermi2ps xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x77, 0xca], "vpermi2ps xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x77, 0xca], "vpermi2ps xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x77, 0x0a], "vpermi2ps xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x77, 0x0a], "vpermi2ps xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x77, 0x4a, 0x01], "vpermi2ps xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x77, 0x4a, 0x01], "vpermi2ps xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x78, 0xca], "vpbroadcastb ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x78, 0x0a], "vpbroadcastb ymm1{k5}{z}, byte [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x78, 0x4a, 0x01], "vpbroadcastb ymm1{k5}{z}, byte [edx + 0x1]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x78, 0xca], "vpbroadcastb ymm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x78, 0xca], "vpbroadcastb ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x78, 0x0a], "vpbroadcastb ymm1, byte [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x78, 0x0a], "vpbroadcastb ymm1{k5}, byte [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x78, 0x4a, 0x01], "vpbroadcastb ymm1, byte [edx + 0x1]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x78, 0x4a, 0x01], "vpbroadcastb ymm1{k5}, byte [edx + 0x1]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x78, 0xca], "vpbroadcastb zmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x78, 0x0a], "vpbroadcastb zmm1{k5}{z}, byte [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x78, 0x4a, 0x01], "vpbroadcastb zmm1{k5}{z}, byte [edx + 0x1]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x78, 0xca], "vpbroadcastb zmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x78, 0xca], "vpbroadcastb zmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x78, 0x0a], "vpbroadcastb zmm1, byte [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x78, 0x0a], "vpbroadcastb zmm1{k5}, byte [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x78, 0x4a, 0x01], "vpbroadcastb zmm1, byte [edx + 0x1]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x78, 0x4a, 0x01], "vpbroadcastb zmm1{k5}, byte [edx + 0x1]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x78, 0xca], "vpbroadcastb xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x78, 0x0a], "vpbroadcastb xmm1{k5}{z}, byte [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x78, 0x4a, 0x01], "vpbroadcastb xmm1{k5}{z}, byte [edx + 0x1]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x78, 0xca], "vpbroadcastb xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x78, 0xca], "vpbroadcastb xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x78, 0x0a], "vpbroadcastb xmm1, byte [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x78, 0x0a], "vpbroadcastb xmm1{k5}, byte [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x78, 0x4a, 0x01], "vpbroadcastb xmm1, byte [edx + 0x1]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x78, 0x4a, 0x01], "vpbroadcastb xmm1{k5}, byte [edx + 0x1]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x79, 0xca], "vpbroadcastw ymm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x79, 0x0a], "vpbroadcastw ymm1{k5}{z}, word [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x79, 0x4a, 0x01], "vpbroadcastw ymm1{k5}{z}, word [edx + 0x2]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x79, 0xca], "vpbroadcastw ymm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x79, 0xca], "vpbroadcastw ymm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x79, 0x0a], "vpbroadcastw ymm1, word [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x79, 0x0a], "vpbroadcastw ymm1{k5}, word [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x79, 0x4a, 0x01], "vpbroadcastw ymm1, word [edx + 0x2]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x79, 0x4a, 0x01], "vpbroadcastw ymm1{k5}, word [edx + 0x2]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x79, 0xca], "vpbroadcastw zmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x79, 0x0a], "vpbroadcastw zmm1{k5}{z}, word [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x79, 0x4a, 0x01], "vpbroadcastw zmm1{k5}{z}, word [edx + 0x2]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x79, 0xca], "vpbroadcastw zmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x79, 0xca], "vpbroadcastw zmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x79, 0x0a], "vpbroadcastw zmm1, word [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x79, 0x0a], "vpbroadcastw zmm1{k5}, word [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x79, 0x4a, 0x01], "vpbroadcastw zmm1, word [edx + 0x2]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x79, 0x4a, 0x01], "vpbroadcastw zmm1{k5}, word [edx + 0x2]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x79, 0xca], "vpbroadcastw xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x79, 0x0a], "vpbroadcastw xmm1{k5}{z}, word [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x79, 0x4a, 0x01], "vpbroadcastw xmm1{k5}{z}, word [edx + 0x2]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x79, 0xca], "vpbroadcastw xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x79, 0xca], "vpbroadcastw xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x79, 0x0a], "vpbroadcastw xmm1, word [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x79, 0x0a], "vpbroadcastw xmm1{k5}, word [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x79, 0x4a, 0x01], "vpbroadcastw xmm1, word [edx + 0x2]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x79, 0x4a, 0x01], "vpbroadcastw xmm1{k5}, word [edx + 0x2]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x7a, 0xca], "vpbroadcastb ymm1{k5}{z}, edx");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x7a, 0xca], "vpbroadcastb ymm1, edx");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x7a, 0xca], "vpbroadcastb ymm1{k5}, edx");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x7a, 0xca], "vpbroadcastb zmm1{k5}{z}, edx");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x7a, 0xca], "vpbroadcastb zmm1, edx");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x7a, 0xca], "vpbroadcastb zmm1{k5}, edx");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x7a, 0xca], "vpbroadcastb xmm1{k5}{z}, edx");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x7a, 0xca], "vpbroadcastb xmm1, edx");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x7a, 0xca], "vpbroadcastb xmm1{k5}, edx");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x7b, 0xca], "vpbroadcastw ymm1{k5}{z}, edx");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x7b, 0xca], "vpbroadcastw ymm1, edx");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x7b, 0xca], "vpbroadcastw ymm1{k5}, edx");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x7b, 0xca], "vpbroadcastw zmm1{k5}{z}, edx");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x7b, 0xca], "vpbroadcastw zmm1, edx");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x7b, 0xca], "vpbroadcastw zmm1{k5}, edx");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x7b, 0xca], "vpbroadcastw xmm1{k5}{z}, edx");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x7b, 0xca], "vpbroadcastw xmm1, edx");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x7b, 0xca], "vpbroadcastw xmm1{k5}, edx");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x7c, 0xca], "vpbroadcastd ymm1{k5}{z}, edx");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x7c, 0xca], "vpbroadcastd ymm1, edx");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x7c, 0xca], "vpbroadcastd ymm1{k5}, edx");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x7c, 0xca], "vpbroadcastd zmm1{k5}{z}, edx");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x7c, 0xca], "vpbroadcastd zmm1, edx");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x7c, 0xca], "vpbroadcastd zmm1{k5}, edx");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x7c, 0xca], "vpbroadcastd xmm1{k5}{z}, edx");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x7c, 0xca], "vpbroadcastd xmm1, edx");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x7c, 0xca], "vpbroadcastd xmm1{k5}, edx");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x7d, 0xca], "vpermt2w ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x7d, 0x0a], "vpermt2w ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x7d, 0x4a, 0x01], "vpermt2w ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x7d, 0xca], "vpermt2w ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x7d, 0xca], "vpermt2w ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x7d, 0x0a], "vpermt2w ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x7d, 0x0a], "vpermt2w ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x7d, 0x4a, 0x01], "vpermt2w ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x7d, 0x4a, 0x01], "vpermt2w ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x7d, 0xca], "vpermt2b ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x7d, 0x0a], "vpermt2b ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x7d, 0x4a, 0x01], "vpermt2b ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x7d, 0xca], "vpermt2b ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x7d, 0xca], "vpermt2b ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x7d, 0x0a], "vpermt2b ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x7d, 0x0a], "vpermt2b ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x7d, 0x4a, 0x01], "vpermt2b ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x7d, 0x4a, 0x01], "vpermt2b ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x7d, 0xca], "vpermt2w zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x7d, 0x0a], "vpermt2w zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x7d, 0x4a, 0x01], "vpermt2w zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x7d, 0xca], "vpermt2w zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x7d, 0xca], "vpermt2w zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x7d, 0x0a], "vpermt2w zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x7d, 0x0a], "vpermt2w zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x7d, 0x4a, 0x01], "vpermt2w zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x7d, 0x4a, 0x01], "vpermt2w zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x7d, 0xca], "vpermt2w xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x7d, 0x0a], "vpermt2w xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x7d, 0x4a, 0x01], "vpermt2w xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x7d, 0xca], "vpermt2w xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x7d, 0xca], "vpermt2w xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x7d, 0x0a], "vpermt2w xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x7d, 0x0a], "vpermt2w xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x7d, 0x4a, 0x01], "vpermt2w xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x7d, 0x4a, 0x01], "vpermt2w xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x7d, 0xca], "vpermt2b zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x7d, 0x0a], "vpermt2b zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x7d, 0x4a, 0x01], "vpermt2b zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x7d, 0xca], "vpermt2b zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x7d, 0xca], "vpermt2b zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x7d, 0x0a], "vpermt2b zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x7d, 0x0a], "vpermt2b zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x7d, 0x4a, 0x01], "vpermt2b zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x7d, 0x4a, 0x01], "vpermt2b zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x7d, 0xca], "vpermt2b xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x7d, 0x0a], "vpermt2b xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x7d, 0x4a, 0x01], "vpermt2b xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x7d, 0xca], "vpermt2b xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x7d, 0xca], "vpermt2b xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x7d, 0x0a], "vpermt2b xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x7d, 0x0a], "vpermt2b xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x7d, 0x4a, 0x01], "vpermt2b xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x7d, 0x4a, 0x01], "vpermt2b xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x7e, 0x0a], "vpermt2q ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x7e, 0x4a, 0x01], "vpermt2q ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x7e, 0x0a], "vpermt2q ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x7e, 0x0a], "vpermt2q ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x7e, 0x4a, 0x01], "vpermt2q ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x7e, 0x4a, 0x01], "vpermt2q ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x7e, 0xca], "vpermt2q ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x7e, 0x0a], "vpermt2q ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x7e, 0x4a, 0x01], "vpermt2q ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x7e, 0xca], "vpermt2q ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x7e, 0xca], "vpermt2q ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x7e, 0x0a], "vpermt2q ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x7e, 0x0a], "vpermt2q ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x7e, 0x4a, 0x01], "vpermt2q ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x7e, 0x4a, 0x01], "vpermt2q ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x7e, 0x0a], "vpermt2d ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x7e, 0x4a, 0x01], "vpermt2d ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x7e, 0x0a], "vpermt2d ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x7e, 0x0a], "vpermt2d ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x7e, 0x4a, 0x01], "vpermt2d ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x7e, 0x4a, 0x01], "vpermt2d ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x7e, 0xca], "vpermt2d ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x7e, 0x0a], "vpermt2d ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x7e, 0x4a, 0x01], "vpermt2d ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x7e, 0xca], "vpermt2d ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x7e, 0xca], "vpermt2d ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x7e, 0x0a], "vpermt2d ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x7e, 0x0a], "vpermt2d ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x7e, 0x4a, 0x01], "vpermt2d ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x7e, 0x4a, 0x01], "vpermt2d ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x7e, 0x0a], "vpermt2q zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x7e, 0x4a, 0x01], "vpermt2q zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x7e, 0x0a], "vpermt2q zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x7e, 0x0a], "vpermt2q zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x7e, 0x4a, 0x01], "vpermt2q zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x7e, 0x4a, 0x01], "vpermt2q zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x7e, 0x0a], "vpermt2q xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x7e, 0x4a, 0x01], "vpermt2q xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x7e, 0x0a], "vpermt2q xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x7e, 0x0a], "vpermt2q xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x7e, 0x4a, 0x01], "vpermt2q xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x7e, 0x4a, 0x01], "vpermt2q xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x7e, 0xca], "vpermt2q zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x7e, 0x0a], "vpermt2q zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x7e, 0x4a, 0x01], "vpermt2q zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x7e, 0xca], "vpermt2q zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x7e, 0xca], "vpermt2q zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x7e, 0x0a], "vpermt2q zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x7e, 0x0a], "vpermt2q zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x7e, 0x4a, 0x01], "vpermt2q zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x7e, 0x4a, 0x01], "vpermt2q zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x7e, 0xca], "vpermt2q xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x7e, 0x0a], "vpermt2q xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x7e, 0x4a, 0x01], "vpermt2q xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x7e, 0xca], "vpermt2q xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x7e, 0xca], "vpermt2q xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x7e, 0x0a], "vpermt2q xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x7e, 0x0a], "vpermt2q xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x7e, 0x4a, 0x01], "vpermt2q xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x7e, 0x4a, 0x01], "vpermt2q xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x7e, 0x0a], "vpermt2d zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x7e, 0x4a, 0x01], "vpermt2d zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x7e, 0x0a], "vpermt2d zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x7e, 0x0a], "vpermt2d zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x7e, 0x4a, 0x01], "vpermt2d zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x7e, 0x4a, 0x01], "vpermt2d zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x7e, 0x0a], "vpermt2d xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x7e, 0x4a, 0x01], "vpermt2d xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x7e, 0x0a], "vpermt2d xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x7e, 0x0a], "vpermt2d xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x7e, 0x4a, 0x01], "vpermt2d xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x7e, 0x4a, 0x01], "vpermt2d xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x7e, 0xca], "vpermt2d zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x7e, 0x0a], "vpermt2d zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x7e, 0x4a, 0x01], "vpermt2d zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x7e, 0xca], "vpermt2d zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x7e, 0xca], "vpermt2d zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x7e, 0x0a], "vpermt2d zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x7e, 0x0a], "vpermt2d zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x7e, 0x4a, 0x01], "vpermt2d zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x7e, 0x4a, 0x01], "vpermt2d zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x7e, 0xca], "vpermt2d xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x7e, 0x0a], "vpermt2d xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x7e, 0x4a, 0x01], "vpermt2d xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x7e, 0xca], "vpermt2d xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x7e, 0xca], "vpermt2d xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x7e, 0x0a], "vpermt2d xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x7e, 0x0a], "vpermt2d xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x7e, 0x4a, 0x01], "vpermt2d xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x7e, 0x4a, 0x01], "vpermt2d xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x7f, 0x0a], "vpermt2pd ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x7f, 0x4a, 0x01], "vpermt2pd ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x7f, 0x0a], "vpermt2pd ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x7f, 0x0a], "vpermt2pd ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x7f, 0x4a, 0x01], "vpermt2pd ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x7f, 0x4a, 0x01], "vpermt2pd ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x7f, 0xca], "vpermt2pd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x7f, 0x0a], "vpermt2pd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x7f, 0x4a, 0x01], "vpermt2pd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x7f, 0xca], "vpermt2pd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x7f, 0xca], "vpermt2pd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x7f, 0x0a], "vpermt2pd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x7f, 0x0a], "vpermt2pd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x7f, 0x4a, 0x01], "vpermt2pd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x7f, 0x4a, 0x01], "vpermt2pd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x7f, 0x0a], "vpermt2ps ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x7f, 0x4a, 0x01], "vpermt2ps ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x7f, 0x0a], "vpermt2ps ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x7f, 0x0a], "vpermt2ps ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x7f, 0x4a, 0x01], "vpermt2ps ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x7f, 0x4a, 0x01], "vpermt2ps ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x7f, 0xca], "vpermt2ps ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x7f, 0x0a], "vpermt2ps ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x7f, 0x4a, 0x01], "vpermt2ps ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x7f, 0xca], "vpermt2ps ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x7f, 0xca], "vpermt2ps ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x7f, 0x0a], "vpermt2ps ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x7f, 0x0a], "vpermt2ps ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x7f, 0x4a, 0x01], "vpermt2ps ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x7f, 0x4a, 0x01], "vpermt2ps ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x7f, 0x0a], "vpermt2pd zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x7f, 0x4a, 0x01], "vpermt2pd zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x7f, 0x0a], "vpermt2pd zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x7f, 0x0a], "vpermt2pd zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x7f, 0x4a, 0x01], "vpermt2pd zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x7f, 0x4a, 0x01], "vpermt2pd zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x7f, 0x0a], "vpermt2pd xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x7f, 0x4a, 0x01], "vpermt2pd xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x7f, 0x0a], "vpermt2pd xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x7f, 0x0a], "vpermt2pd xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x7f, 0x4a, 0x01], "vpermt2pd xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x7f, 0x4a, 0x01], "vpermt2pd xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x7f, 0xca], "vpermt2pd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x7f, 0x0a], "vpermt2pd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x7f, 0x4a, 0x01], "vpermt2pd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x7f, 0xca], "vpermt2pd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x7f, 0xca], "vpermt2pd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x7f, 0x0a], "vpermt2pd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x7f, 0x0a], "vpermt2pd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x7f, 0x4a, 0x01], "vpermt2pd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x7f, 0x4a, 0x01], "vpermt2pd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x7f, 0xca], "vpermt2pd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x7f, 0x0a], "vpermt2pd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x7f, 0x4a, 0x01], "vpermt2pd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x7f, 0xca], "vpermt2pd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x7f, 0xca], "vpermt2pd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x7f, 0x0a], "vpermt2pd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x7f, 0x0a], "vpermt2pd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x7f, 0x4a, 0x01], "vpermt2pd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x7f, 0x4a, 0x01], "vpermt2pd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x7f, 0x0a], "vpermt2ps zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x7f, 0x4a, 0x01], "vpermt2ps zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x7f, 0x0a], "vpermt2ps zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x7f, 0x0a], "vpermt2ps zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x7f, 0x4a, 0x01], "vpermt2ps zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x7f, 0x4a, 0x01], "vpermt2ps zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x7f, 0x0a], "vpermt2ps xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x7f, 0x4a, 0x01], "vpermt2ps xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x7f, 0x0a], "vpermt2ps xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x7f, 0x0a], "vpermt2ps xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x7f, 0x4a, 0x01], "vpermt2ps xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x7f, 0x4a, 0x01], "vpermt2ps xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x7f, 0xca], "vpermt2ps zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x7f, 0x0a], "vpermt2ps zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x7f, 0x4a, 0x01], "vpermt2ps zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x7f, 0xca], "vpermt2ps zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x7f, 0xca], "vpermt2ps zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x7f, 0x0a], "vpermt2ps zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x7f, 0x0a], "vpermt2ps zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x7f, 0x4a, 0x01], "vpermt2ps zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x7f, 0x4a, 0x01], "vpermt2ps zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x7f, 0xca], "vpermt2ps xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x7f, 0x0a], "vpermt2ps xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x7f, 0x4a, 0x01], "vpermt2ps xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x7f, 0xca], "vpermt2ps xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x7f, 0xca], "vpermt2ps xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x7f, 0x0a], "vpermt2ps xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x7f, 0x0a], "vpermt2ps xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x7f, 0x4a, 0x01], "vpermt2ps xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x7f, 0x4a, 0x01], "vpermt2ps xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x83, 0x0a], "vpmultishiftqb ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x83, 0x4a, 0x01], "vpmultishiftqb ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x83, 0x0a], "vpmultishiftqb ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x83, 0x0a], "vpmultishiftqb ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x83, 0x4a, 0x01], "vpmultishiftqb ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x83, 0x4a, 0x01], "vpmultishiftqb ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x83, 0xca], "vpmultishiftqb ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x83, 0x0a], "vpmultishiftqb ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x83, 0x4a, 0x01], "vpmultishiftqb ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x83, 0xca], "vpmultishiftqb ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x83, 0xca], "vpmultishiftqb ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x83, 0x0a], "vpmultishiftqb ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x83, 0x0a], "vpmultishiftqb ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x83, 0x4a, 0x01], "vpmultishiftqb ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x83, 0x4a, 0x01], "vpmultishiftqb ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x83, 0x0a], "vpmultishiftqb zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x83, 0x4a, 0x01], "vpmultishiftqb zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x83, 0x0a], "vpmultishiftqb zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x83, 0x0a], "vpmultishiftqb zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x83, 0x4a, 0x01], "vpmultishiftqb zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x83, 0x4a, 0x01], "vpmultishiftqb zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x83, 0x0a], "vpmultishiftqb xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x83, 0x4a, 0x01], "vpmultishiftqb xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x83, 0x0a], "vpmultishiftqb xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x83, 0x0a], "vpmultishiftqb xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x83, 0x4a, 0x01], "vpmultishiftqb xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x83, 0x4a, 0x01], "vpmultishiftqb xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x83, 0xca], "vpmultishiftqb zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x83, 0x0a], "vpmultishiftqb zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x83, 0x4a, 0x01], "vpmultishiftqb zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x83, 0xca], "vpmultishiftqb zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x83, 0xca], "vpmultishiftqb zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x83, 0x0a], "vpmultishiftqb zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x83, 0x0a], "vpmultishiftqb zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x83, 0x4a, 0x01], "vpmultishiftqb zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x83, 0x4a, 0x01], "vpmultishiftqb zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x83, 0xca], "vpmultishiftqb xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x83, 0x0a], "vpmultishiftqb xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x83, 0x4a, 0x01], "vpmultishiftqb xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x83, 0xca], "vpmultishiftqb xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x83, 0xca], "vpmultishiftqb xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x83, 0x0a], "vpmultishiftqb xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x83, 0x0a], "vpmultishiftqb xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x83, 0x4a, 0x01], "vpmultishiftqb xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x83, 0x4a, 0x01], "vpmultishiftqb xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x88, 0xca], "vexpandpd ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x88, 0x0a], "vexpandpd ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x88, 0x4a, 0x01], "vexpandpd ymm1{k5}{z}, ymmword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x88, 0xca], "vexpandpd ymm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x88, 0xca], "vexpandpd ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x88, 0x0a], "vexpandpd ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x88, 0x0a], "vexpandpd ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x88, 0x4a, 0x01], "vexpandpd ymm1, ymmword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x88, 0x4a, 0x01], "vexpandpd ymm1{k5}, ymmword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x88, 0xca], "vexpandps ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x88, 0x0a], "vexpandps ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x88, 0x4a, 0x01], "vexpandps ymm1{k5}{z}, ymmword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x88, 0xca], "vexpandps ymm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x88, 0xca], "vexpandps ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x88, 0x0a], "vexpandps ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x88, 0x0a], "vexpandps ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x88, 0x4a, 0x01], "vexpandps ymm1, ymmword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x88, 0x4a, 0x01], "vexpandps ymm1{k5}, ymmword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x88, 0xca], "vexpandpd zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x88, 0x0a], "vexpandpd zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x88, 0x4a, 0x01], "vexpandpd zmm1{k5}{z}, zmmword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x88, 0xca], "vexpandpd zmm1, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x88, 0xca], "vexpandpd zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x88, 0x0a], "vexpandpd zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x88, 0x0a], "vexpandpd zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x88, 0x4a, 0x01], "vexpandpd zmm1, zmmword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x88, 0x4a, 0x01], "vexpandpd zmm1{k5}, zmmword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x88, 0xca], "vexpandpd xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x88, 0x0a], "vexpandpd xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x88, 0x4a, 0x01], "vexpandpd xmm1{k5}{z}, xmmword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x88, 0xca], "vexpandpd xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x88, 0xca], "vexpandpd xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x88, 0x0a], "vexpandpd xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x88, 0x0a], "vexpandpd xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x88, 0x4a, 0x01], "vexpandpd xmm1, xmmword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x88, 0x4a, 0x01], "vexpandpd xmm1{k5}, xmmword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x88, 0xca], "vexpandps zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x88, 0x0a], "vexpandps zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x88, 0x4a, 0x01], "vexpandps zmm1{k5}{z}, zmmword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x88, 0xca], "vexpandps zmm1, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x88, 0xca], "vexpandps zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x88, 0x0a], "vexpandps zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x88, 0x0a], "vexpandps zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x88, 0x4a, 0x01], "vexpandps zmm1, zmmword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x88, 0x4a, 0x01], "vexpandps zmm1{k5}, zmmword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x88, 0xca], "vexpandps xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x88, 0x0a], "vexpandps xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x88, 0x4a, 0x01], "vexpandps xmm1{k5}{z}, xmmword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x88, 0xca], "vexpandps xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x88, 0xca], "vexpandps xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x88, 0x0a], "vexpandps xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x88, 0x0a], "vexpandps xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x88, 0x4a, 0x01], "vexpandps xmm1, xmmword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x88, 0x4a, 0x01], "vexpandps xmm1{k5}, xmmword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x89, 0xca], "vpexpandq ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x89, 0x0a], "vpexpandq ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x89, 0x4a, 0x01], "vpexpandq ymm1{k5}{z}, ymmword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x89, 0xca], "vpexpandq ymm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x89, 0xca], "vpexpandq ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x89, 0x0a], "vpexpandq ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x89, 0x0a], "vpexpandq ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x89, 0x4a, 0x01], "vpexpandq ymm1, ymmword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x89, 0x4a, 0x01], "vpexpandq ymm1{k5}, ymmword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x89, 0xca], "vpexpandd ymm1{k5}{z}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x89, 0x0a], "vpexpandd ymm1{k5}{z}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x89, 0x4a, 0x01], "vpexpandd ymm1{k5}{z}, ymmword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x89, 0xca], "vpexpandd ymm1, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x89, 0xca], "vpexpandd ymm1{k5}, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x89, 0x0a], "vpexpandd ymm1, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x89, 0x0a], "vpexpandd ymm1{k5}, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x89, 0x4a, 0x01], "vpexpandd ymm1, ymmword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x89, 0x4a, 0x01], "vpexpandd ymm1{k5}, ymmword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x89, 0xca], "vpexpandq zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x89, 0x0a], "vpexpandq zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x89, 0x4a, 0x01], "vpexpandq zmm1{k5}{z}, zmmword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x89, 0xca], "vpexpandq zmm1, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x89, 0xca], "vpexpandq zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x89, 0x0a], "vpexpandq zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x89, 0x0a], "vpexpandq zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x89, 0x4a, 0x01], "vpexpandq zmm1, zmmword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x89, 0x4a, 0x01], "vpexpandq zmm1{k5}, zmmword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x89, 0xca], "vpexpandq xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x89, 0x0a], "vpexpandq xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x89, 0x4a, 0x01], "vpexpandq xmm1{k5}{z}, xmmword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x89, 0xca], "vpexpandq xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x89, 0xca], "vpexpandq xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x89, 0x0a], "vpexpandq xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x89, 0x0a], "vpexpandq xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x89, 0x4a, 0x01], "vpexpandq xmm1, xmmword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x89, 0x4a, 0x01], "vpexpandq xmm1{k5}, xmmword [edx + 0x8]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x89, 0xca], "vpexpandd zmm1{k5}{z}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x89, 0x0a], "vpexpandd zmm1{k5}{z}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x89, 0x4a, 0x01], "vpexpandd zmm1{k5}{z}, zmmword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x89, 0xca], "vpexpandd zmm1, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x89, 0xca], "vpexpandd zmm1{k5}, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x89, 0x0a], "vpexpandd zmm1, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x89, 0x0a], "vpexpandd zmm1{k5}, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x89, 0x4a, 0x01], "vpexpandd zmm1, zmmword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x89, 0x4a, 0x01], "vpexpandd zmm1{k5}, zmmword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x89, 0xca], "vpexpandd xmm1{k5}{z}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x89, 0x0a], "vpexpandd xmm1{k5}{z}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x89, 0x4a, 0x01], "vpexpandd xmm1{k5}{z}, xmmword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x89, 0xca], "vpexpandd xmm1, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x89, 0xca], "vpexpandd xmm1{k5}, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x89, 0x0a], "vpexpandd xmm1, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x89, 0x0a], "vpexpandd xmm1{k5}, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x89, 0x4a, 0x01], "vpexpandd xmm1, xmmword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x89, 0x4a, 0x01], "vpexpandd xmm1{k5}, xmmword [edx + 0x4]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x8a, 0xca], "vcompresspd ymm2{k5}{z}, ymm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x8a, 0xca], "vcompresspd ymm2, ymm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x8a, 0xca], "vcompresspd ymm2{k5}, ymm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x8a, 0x0a], "vcompresspd ymmword [edx], ymm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x8a, 0x0a], "vcompresspd ymmword [edx]{k5}, ymm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x8a, 0x4a, 0x01], "vcompresspd ymmword [edx + 0x8], ymm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x8a, 0x4a, 0x01], "vcompresspd ymmword [edx + 0x8]{k5}, ymm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x8a, 0xca], "vcompressps ymm2{k5}{z}, ymm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x8a, 0xca], "vcompressps ymm2, ymm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x8a, 0xca], "vcompressps ymm2{k5}, ymm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x8a, 0x0a], "vcompressps ymmword [edx], ymm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x8a, 0x0a], "vcompressps ymmword [edx]{k5}, ymm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x8a, 0x4a, 0x01], "vcompressps ymmword [edx + 0x4], ymm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x8a, 0x4a, 0x01], "vcompressps ymmword [edx + 0x4]{k5}, ymm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x8a, 0xca], "vcompresspd zmm2{k5}{z}, zmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x8a, 0xca], "vcompresspd zmm2, zmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x8a, 0xca], "vcompresspd zmm2{k5}, zmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x8a, 0x0a], "vcompresspd zmmword [edx], zmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x8a, 0x0a], "vcompresspd zmmword [edx]{k5}, zmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x8a, 0x4a, 0x01], "vcompresspd zmmword [edx + 0x8], zmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x8a, 0x4a, 0x01], "vcompresspd zmmword [edx + 0x8]{k5}, zmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x8a, 0xca], "vcompresspd xmm2{k5}{z}, xmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x8a, 0xca], "vcompresspd xmm2, xmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x8a, 0xca], "vcompresspd xmm2{k5}, xmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x8a, 0x0a], "vcompresspd xmmword [edx], xmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x8a, 0x0a], "vcompresspd xmmword [edx]{k5}, xmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x8a, 0x4a, 0x01], "vcompresspd xmmword [edx + 0x8], xmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x8a, 0x4a, 0x01], "vcompresspd xmmword [edx + 0x8]{k5}, xmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x8a, 0xca], "vcompressps zmm2{k5}{z}, zmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x8a, 0xca], "vcompressps zmm2, zmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x8a, 0xca], "vcompressps zmm2{k5}, zmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x8a, 0x0a], "vcompressps zmmword [edx], zmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x8a, 0x0a], "vcompressps zmmword [edx]{k5}, zmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x8a, 0x4a, 0x01], "vcompressps zmmword [edx + 0x4], zmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x8a, 0x4a, 0x01], "vcompressps zmmword [edx + 0x4]{k5}, zmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x8a, 0xca], "vcompressps xmm2{k5}{z}, xmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x8a, 0xca], "vcompressps xmm2, xmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x8a, 0xca], "vcompressps xmm2{k5}, xmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x8a, 0x0a], "vcompressps xmmword [edx], xmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x8a, 0x0a], "vcompressps xmmword [edx]{k5}, xmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x8a, 0x4a, 0x01], "vcompressps xmmword [edx + 0x4], xmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x8a, 0x4a, 0x01], "vcompressps xmmword [edx + 0x4]{k5}, xmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x8b, 0xca], "vpcompressq ymm2{k5}{z}, ymm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x8b, 0xca], "vpcompressq ymm2, ymm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x8b, 0xca], "vpcompressq ymm2{k5}, ymm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x8b, 0x0a], "vpcompressq ymmword [edx], ymm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x8b, 0x0a], "vpcompressq ymmword [edx]{k5}, ymm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x8b, 0x4a, 0x01], "vpcompressq ymmword [edx + 0x8], ymm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x8b, 0x4a, 0x01], "vpcompressq ymmword [edx + 0x8]{k5}, ymm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x8b, 0xca], "vpcompressd ymm2{k5}{z}, ymm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x8b, 0xca], "vpcompressd ymm2, ymm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x8b, 0xca], "vpcompressd ymm2{k5}, ymm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x8b, 0x0a], "vpcompressd ymmword [edx], ymm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x8b, 0x0a], "vpcompressd ymmword [edx]{k5}, ymm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x8b, 0x4a, 0x01], "vpcompressd ymmword [edx + 0x4], ymm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x8b, 0x4a, 0x01], "vpcompressd ymmword [edx + 0x4]{k5}, ymm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x8b, 0xca], "vpcompressq zmm2{k5}{z}, zmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x8b, 0xca], "vpcompressq zmm2, zmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x8b, 0xca], "vpcompressq zmm2{k5}, zmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x8b, 0x0a], "vpcompressq zmmword [edx], zmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x8b, 0x0a], "vpcompressq zmmword [edx]{k5}, zmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x8b, 0x4a, 0x01], "vpcompressq zmmword [edx + 0x8], zmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x8b, 0x4a, 0x01], "vpcompressq zmmword [edx + 0x8]{k5}, zmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x8b, 0xca], "vpcompressq xmm2{k5}{z}, xmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x8b, 0xca], "vpcompressq xmm2, xmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x8b, 0xca], "vpcompressq xmm2{k5}, xmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x8b, 0x0a], "vpcompressq xmmword [edx], xmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x8b, 0x0a], "vpcompressq xmmword [edx]{k5}, xmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x8b, 0x4a, 0x01], "vpcompressq xmmword [edx + 0x8], xmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x8b, 0x4a, 0x01], "vpcompressq xmmword [edx + 0x8]{k5}, xmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x8b, 0xca], "vpcompressd zmm2{k5}{z}, zmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x8b, 0xca], "vpcompressd zmm2, zmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x8b, 0xca], "vpcompressd zmm2{k5}, zmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x8b, 0x0a], "vpcompressd zmmword [edx], zmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x8b, 0x0a], "vpcompressd zmmword [edx]{k5}, zmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x8b, 0x4a, 0x01], "vpcompressd zmmword [edx + 0x4], zmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x8b, 0x4a, 0x01], "vpcompressd zmmword [edx + 0x4]{k5}, zmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x8b, 0xca], "vpcompressd xmm2{k5}{z}, xmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x8b, 0xca], "vpcompressd xmm2, xmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x8b, 0xca], "vpcompressd xmm2{k5}, xmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x8b, 0x0a], "vpcompressd xmmword [edx], xmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x8b, 0x0a], "vpcompressd xmmword [edx]{k5}, xmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x8b, 0x4a, 0x01], "vpcompressd xmmword [edx + 0x4], xmm1");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x8b, 0x4a, 0x01], "vpcompressd xmmword [edx + 0x4]{k5}, xmm1");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x8d, 0xca], "vpermw ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x8d, 0x0a], "vpermw ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x8d, 0x4a, 0x01], "vpermw ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x8d, 0xca], "vpermw ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x8d, 0xca], "vpermw ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x8d, 0x0a], "vpermw ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x8d, 0x0a], "vpermw ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x8d, 0x4a, 0x01], "vpermw ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x8d, 0x4a, 0x01], "vpermw ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x8d, 0xca], "vpermb ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x8d, 0x0a], "vpermb ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x8d, 0x4a, 0x01], "vpermb ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x8d, 0xca], "vpermb ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x8d, 0xca], "vpermb ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x8d, 0x0a], "vpermb ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x8d, 0x0a], "vpermb ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x8d, 0x4a, 0x01], "vpermb ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x8d, 0x4a, 0x01], "vpermb ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x8d, 0xca], "vpermw zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x8d, 0x0a], "vpermw zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x8d, 0x4a, 0x01], "vpermw zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x8d, 0xca], "vpermw zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x8d, 0xca], "vpermw zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x8d, 0x0a], "vpermw zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x8d, 0x0a], "vpermw zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x8d, 0x4a, 0x01], "vpermw zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x8d, 0x4a, 0x01], "vpermw zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x8d, 0xca], "vpermw xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x8d, 0x0a], "vpermw xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x8d, 0x4a, 0x01], "vpermw xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x8d, 0xca], "vpermw xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x8d, 0xca], "vpermw xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x8d, 0x0a], "vpermw xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x8d, 0x0a], "vpermw xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x8d, 0x4a, 0x01], "vpermw xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x8d, 0x4a, 0x01], "vpermw xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x8d, 0xca], "vpermb zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x8d, 0x0a], "vpermb zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x8d, 0x4a, 0x01], "vpermb zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x8d, 0xca], "vpermb zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x8d, 0xca], "vpermb zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x8d, 0x0a], "vpermb zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x8d, 0x0a], "vpermb zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x8d, 0x4a, 0x01], "vpermb zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x8d, 0x4a, 0x01], "vpermb zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x8d, 0xca], "vpermb xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x8d, 0x0a], "vpermb xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x8d, 0x4a, 0x01], "vpermb xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x8d, 0xca], "vpermb xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x8d, 0xca], "vpermb xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x8d, 0x0a], "vpermb xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x8d, 0x0a], "vpermb xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x8d, 0x4a, 0x01], "vpermb xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x8d, 0x4a, 0x01], "vpermb xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x8f, 0xca], "vpshufbitqmb k1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x8f, 0xca], "vpshufbitqmb k1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x8f, 0x0a], "vpshufbitqmb k1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x8f, 0x0a], "vpshufbitqmb k1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x8f, 0x4a, 0x01], "vpshufbitqmb k1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x8f, 0x4a, 0x01], "vpshufbitqmb k1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x8f, 0xca], "vpshufbitqmb k1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x8f, 0xca], "vpshufbitqmb k1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x8f, 0x0a], "vpshufbitqmb k1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x8f, 0x0a], "vpshufbitqmb k1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x8f, 0x4a, 0x01], "vpshufbitqmb k1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x8f, 0x4a, 0x01], "vpshufbitqmb k1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x8f, 0xca], "vpshufbitqmb k1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x8f, 0xca], "vpshufbitqmb k1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x8f, 0x0a], "vpshufbitqmb k1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x8f, 0x0a], "vpshufbitqmb k1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x8f, 0x4a, 0x01], "vpshufbitqmb k1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x8f, 0x4a, 0x01], "vpshufbitqmb k1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xfd, 0x96, 0xca], "vfmaddsub132pd zmm1{k5}{z}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x78, 0x96, 0xca], "vfmaddsub132pd zmm1{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x7d, 0x96, 0xca], "vfmaddsub132pd zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x96, 0xca], "vfmaddsub132pd zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x96, 0x0a], "vfmaddsub132pd ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x96, 0x4a, 0x01], "vfmaddsub132pd ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x96, 0xca], "vfmaddsub132pd zmm1{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x96, 0xca], "vfmaddsub132pd zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x96, 0x0a], "vfmaddsub132pd ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x96, 0x0a], "vfmaddsub132pd ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x96, 0x4a, 0x01], "vfmaddsub132pd ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x96, 0x4a, 0x01], "vfmaddsub132pd ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x96, 0xca], "vfmaddsub132pd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x96, 0x0a], "vfmaddsub132pd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x96, 0x4a, 0x01], "vfmaddsub132pd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x96, 0xca], "vfmaddsub132pd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x96, 0xca], "vfmaddsub132pd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x96, 0x0a], "vfmaddsub132pd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x96, 0x0a], "vfmaddsub132pd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x96, 0x4a, 0x01], "vfmaddsub132pd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x96, 0x4a, 0x01], "vfmaddsub132pd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xfd, 0x96, 0xca], "vfmaddsub132ps zmm1{k5}{z}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x78, 0x96, 0xca], "vfmaddsub132ps zmm1{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x7d, 0x96, 0xca], "vfmaddsub132ps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x96, 0xca], "vfmaddsub132ps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x96, 0x0a], "vfmaddsub132ps ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x96, 0x4a, 0x01], "vfmaddsub132ps ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x96, 0xca], "vfmaddsub132ps zmm1{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x96, 0xca], "vfmaddsub132ps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x96, 0x0a], "vfmaddsub132ps ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x96, 0x0a], "vfmaddsub132ps ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x96, 0x4a, 0x01], "vfmaddsub132ps ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x96, 0x4a, 0x01], "vfmaddsub132ps ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x96, 0xca], "vfmaddsub132ps ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x96, 0x0a], "vfmaddsub132ps ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x96, 0x4a, 0x01], "vfmaddsub132ps ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x96, 0xca], "vfmaddsub132ps ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x96, 0xca], "vfmaddsub132ps ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x96, 0x0a], "vfmaddsub132ps ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x96, 0x0a], "vfmaddsub132ps ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x96, 0x4a, 0x01], "vfmaddsub132ps ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x96, 0x4a, 0x01], "vfmaddsub132ps ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x96, 0xca], "vfmaddsub132pd zmm1{k5}{z}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x96, 0x0a], "vfmaddsub132pd zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x96, 0x4a, 0x01], "vfmaddsub132pd zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x96, 0xca], "vfmaddsub132pd zmm1{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x96, 0xca], "vfmaddsub132pd zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x96, 0x0a], "vfmaddsub132pd zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x96, 0x0a], "vfmaddsub132pd zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x96, 0x4a, 0x01], "vfmaddsub132pd zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x96, 0x4a, 0x01], "vfmaddsub132pd zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x96, 0xca], "vfmaddsub132pd zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x96, 0x0a], "vfmaddsub132pd xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x96, 0x4a, 0x01], "vfmaddsub132pd xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x96, 0xca], "vfmaddsub132pd zmm1{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x96, 0xca], "vfmaddsub132pd zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x96, 0x0a], "vfmaddsub132pd xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x96, 0x0a], "vfmaddsub132pd xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x96, 0x4a, 0x01], "vfmaddsub132pd xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x96, 0x4a, 0x01], "vfmaddsub132pd xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x96, 0xca], "vfmaddsub132pd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x96, 0x0a], "vfmaddsub132pd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x96, 0x4a, 0x01], "vfmaddsub132pd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x96, 0xca], "vfmaddsub132pd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x96, 0xca], "vfmaddsub132pd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x96, 0x0a], "vfmaddsub132pd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x96, 0x0a], "vfmaddsub132pd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x96, 0x4a, 0x01], "vfmaddsub132pd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x96, 0x4a, 0x01], "vfmaddsub132pd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x96, 0xca], "vfmaddsub132pd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x96, 0x0a], "vfmaddsub132pd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x96, 0x4a, 0x01], "vfmaddsub132pd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x96, 0xca], "vfmaddsub132pd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x96, 0xca], "vfmaddsub132pd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x96, 0x0a], "vfmaddsub132pd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x96, 0x0a], "vfmaddsub132pd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x96, 0x4a, 0x01], "vfmaddsub132pd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x96, 0x4a, 0x01], "vfmaddsub132pd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x96, 0xca], "vfmaddsub132ps zmm1{k5}{z}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x96, 0x0a], "vfmaddsub132ps zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x96, 0x4a, 0x01], "vfmaddsub132ps zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x96, 0xca], "vfmaddsub132ps zmm1{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x96, 0xca], "vfmaddsub132ps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x96, 0x0a], "vfmaddsub132ps zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x96, 0x0a], "vfmaddsub132ps zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x96, 0x4a, 0x01], "vfmaddsub132ps zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x96, 0x4a, 0x01], "vfmaddsub132ps zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x96, 0xca], "vfmaddsub132ps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x96, 0x0a], "vfmaddsub132ps xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x96, 0x4a, 0x01], "vfmaddsub132ps xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x96, 0xca], "vfmaddsub132ps zmm1{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x96, 0xca], "vfmaddsub132ps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x96, 0x0a], "vfmaddsub132ps xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x96, 0x0a], "vfmaddsub132ps xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x96, 0x4a, 0x01], "vfmaddsub132ps xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x96, 0x4a, 0x01], "vfmaddsub132ps xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x96, 0xca], "vfmaddsub132ps zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x96, 0x0a], "vfmaddsub132ps zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x96, 0x4a, 0x01], "vfmaddsub132ps zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x96, 0xca], "vfmaddsub132ps zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x96, 0xca], "vfmaddsub132ps zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x96, 0x0a], "vfmaddsub132ps zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x96, 0x0a], "vfmaddsub132ps zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x96, 0x4a, 0x01], "vfmaddsub132ps zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x96, 0x4a, 0x01], "vfmaddsub132ps zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x96, 0xca], "vfmaddsub132ps xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x96, 0x0a], "vfmaddsub132ps xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x96, 0x4a, 0x01], "vfmaddsub132ps xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x96, 0xca], "vfmaddsub132ps xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x96, 0xca], "vfmaddsub132ps xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x96, 0x0a], "vfmaddsub132ps xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x96, 0x0a], "vfmaddsub132ps xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x96, 0x4a, 0x01], "vfmaddsub132ps xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x96, 0x4a, 0x01], "vfmaddsub132ps xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xfd, 0x97, 0xca], "vfmsubadd132pd zmm1{k5}{z}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x78, 0x97, 0xca], "vfmsubadd132pd zmm1{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x7d, 0x97, 0xca], "vfmsubadd132pd zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x97, 0xca], "vfmsubadd132pd zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x97, 0x0a], "vfmsubadd132pd ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x97, 0x4a, 0x01], "vfmsubadd132pd ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x97, 0xca], "vfmsubadd132pd zmm1{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x97, 0xca], "vfmsubadd132pd zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x97, 0x0a], "vfmsubadd132pd ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x97, 0x0a], "vfmsubadd132pd ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x97, 0x4a, 0x01], "vfmsubadd132pd ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x97, 0x4a, 0x01], "vfmsubadd132pd ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x97, 0xca], "vfmsubadd132pd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x97, 0x0a], "vfmsubadd132pd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x97, 0x4a, 0x01], "vfmsubadd132pd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x97, 0xca], "vfmsubadd132pd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x97, 0xca], "vfmsubadd132pd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x97, 0x0a], "vfmsubadd132pd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x97, 0x0a], "vfmsubadd132pd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x97, 0x4a, 0x01], "vfmsubadd132pd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x97, 0x4a, 0x01], "vfmsubadd132pd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xfd, 0x97, 0xca], "vfmsubadd132ps zmm1{k5}{z}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x78, 0x97, 0xca], "vfmsubadd132ps zmm1{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x7d, 0x97, 0xca], "vfmsubadd132ps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x97, 0xca], "vfmsubadd132ps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x97, 0x0a], "vfmsubadd132ps ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x97, 0x4a, 0x01], "vfmsubadd132ps ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x97, 0xca], "vfmsubadd132ps zmm1{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x97, 0xca], "vfmsubadd132ps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x97, 0x0a], "vfmsubadd132ps ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x97, 0x0a], "vfmsubadd132ps ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x97, 0x4a, 0x01], "vfmsubadd132ps ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x97, 0x4a, 0x01], "vfmsubadd132ps ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x97, 0xca], "vfmsubadd132ps ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x97, 0x0a], "vfmsubadd132ps ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x97, 0x4a, 0x01], "vfmsubadd132ps ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x97, 0xca], "vfmsubadd132ps ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x97, 0xca], "vfmsubadd132ps ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x97, 0x0a], "vfmsubadd132ps ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x97, 0x0a], "vfmsubadd132ps ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x97, 0x4a, 0x01], "vfmsubadd132ps ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x97, 0x4a, 0x01], "vfmsubadd132ps ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x97, 0xca], "vfmsubadd132pd zmm1{k5}{z}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x97, 0x0a], "vfmsubadd132pd zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x97, 0x4a, 0x01], "vfmsubadd132pd zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x97, 0xca], "vfmsubadd132pd zmm1{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x97, 0xca], "vfmsubadd132pd zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x97, 0x0a], "vfmsubadd132pd zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x97, 0x0a], "vfmsubadd132pd zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x97, 0x4a, 0x01], "vfmsubadd132pd zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x97, 0x4a, 0x01], "vfmsubadd132pd zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x97, 0xca], "vfmsubadd132pd zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x97, 0x0a], "vfmsubadd132pd xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x97, 0x4a, 0x01], "vfmsubadd132pd xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x97, 0xca], "vfmsubadd132pd zmm1{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x97, 0xca], "vfmsubadd132pd zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x97, 0x0a], "vfmsubadd132pd xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x97, 0x0a], "vfmsubadd132pd xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x97, 0x4a, 0x01], "vfmsubadd132pd xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x97, 0x4a, 0x01], "vfmsubadd132pd xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x97, 0xca], "vfmsubadd132pd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x97, 0x0a], "vfmsubadd132pd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x97, 0x4a, 0x01], "vfmsubadd132pd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x97, 0xca], "vfmsubadd132pd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x97, 0xca], "vfmsubadd132pd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x97, 0x0a], "vfmsubadd132pd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x97, 0x0a], "vfmsubadd132pd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x97, 0x4a, 0x01], "vfmsubadd132pd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x97, 0x4a, 0x01], "vfmsubadd132pd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x97, 0xca], "vfmsubadd132pd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x97, 0x0a], "vfmsubadd132pd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x97, 0x4a, 0x01], "vfmsubadd132pd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x97, 0xca], "vfmsubadd132pd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x97, 0xca], "vfmsubadd132pd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x97, 0x0a], "vfmsubadd132pd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x97, 0x0a], "vfmsubadd132pd xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x97, 0x4a, 0x01], "vfmsubadd132pd xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x97, 0x4a, 0x01], "vfmsubadd132pd xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x97, 0xca], "vfmsubadd132ps zmm1{k5}{z}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x97, 0x0a], "vfmsubadd132ps zmm1{k5}{z}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x97, 0x4a, 0x01], "vfmsubadd132ps zmm1{k5}{z}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x97, 0xca], "vfmsubadd132ps zmm1{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x97, 0xca], "vfmsubadd132ps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x97, 0x0a], "vfmsubadd132ps zmm1, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x97, 0x0a], "vfmsubadd132ps zmm1{k5}, zmm0, dword [edx]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x97, 0x4a, 0x01], "vfmsubadd132ps zmm1, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x97, 0x4a, 0x01], "vfmsubadd132ps zmm1{k5}, zmm0, dword [edx + 0x4]{1to16}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x97, 0xca], "vfmsubadd132ps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x97, 0x0a], "vfmsubadd132ps xmm1{k5}{z}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x97, 0x4a, 0x01], "vfmsubadd132ps xmm1{k5}{z}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x97, 0xca], "vfmsubadd132ps zmm1{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x97, 0xca], "vfmsubadd132ps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x97, 0x0a], "vfmsubadd132ps xmm1, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x97, 0x0a], "vfmsubadd132ps xmm1{k5}, xmm0, dword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x97, 0x4a, 0x01], "vfmsubadd132ps xmm1, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x97, 0x4a, 0x01], "vfmsubadd132ps xmm1{k5}, xmm0, dword [edx + 0x4]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x97, 0xca], "vfmsubadd132ps zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x97, 0x0a], "vfmsubadd132ps zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x97, 0x4a, 0x01], "vfmsubadd132ps zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x97, 0xca], "vfmsubadd132ps zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x97, 0xca], "vfmsubadd132ps zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x97, 0x0a], "vfmsubadd132ps zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x97, 0x0a], "vfmsubadd132ps zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x97, 0x4a, 0x01], "vfmsubadd132ps zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x97, 0x4a, 0x01], "vfmsubadd132ps zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x97, 0xca], "vfmsubadd132ps xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x97, 0x0a], "vfmsubadd132ps xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x97, 0x4a, 0x01], "vfmsubadd132ps xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x97, 0xca], "vfmsubadd132ps xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x97, 0xca], "vfmsubadd132ps xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x97, 0x0a], "vfmsubadd132ps xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x97, 0x0a], "vfmsubadd132ps xmm1{k5}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x97, 0x4a, 0x01], "vfmsubadd132ps xmm1, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x97, 0x4a, 0x01], "vfmsubadd132ps xmm1{k5}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xfd, 0x98, 0xca], "vfmadd132pd zmm1{k5}{z}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x78, 0x98, 0xca], "vfmadd132pd zmm1{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x7d, 0x98, 0xca], "vfmadd132pd zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x98, 0xca], "vfmadd132pd zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x98, 0x0a], "vfmadd132pd ymm1{k5}{z}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x98, 0x4a, 0x01], "vfmadd132pd ymm1{k5}{z}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x98, 0xca], "vfmadd132pd zmm1{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x98, 0xca], "vfmadd132pd zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x98, 0x0a], "vfmadd132pd ymm1, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x98, 0x0a], "vfmadd132pd ymm1{k5}, ymm0, qword [edx]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x98, 0x4a, 0x01], "vfmadd132pd ymm1, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x98, 0x4a, 0x01], "vfmadd132pd ymm1{k5}, ymm0, qword [edx + 0x8]{1to4}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x98, 0xca], "vfmadd132pd ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x98, 0x0a], "vfmadd132pd ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x98, 0x4a, 0x01], "vfmadd132pd ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x98, 0xca], "vfmadd132pd ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x98, 0xca], "vfmadd132pd ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x98, 0x0a], "vfmadd132pd ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x98, 0x0a], "vfmadd132pd ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x98, 0x4a, 0x01], "vfmadd132pd ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x98, 0x4a, 0x01], "vfmadd132pd ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xfd, 0x98, 0xca], "vfmadd132ps zmm1{k5}{z}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x78, 0x98, 0xca], "vfmadd132ps zmm1{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x7d, 0x98, 0xca], "vfmadd132ps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x98, 0xca], "vfmadd132ps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x98, 0x0a], "vfmadd132ps ymm1{k5}{z}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x98, 0x4a, 0x01], "vfmadd132ps ymm1{k5}{z}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x98, 0xca], "vfmadd132ps zmm1{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x98, 0xca], "vfmadd132ps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x98, 0x0a], "vfmadd132ps ymm1, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x98, 0x0a], "vfmadd132ps ymm1{k5}, ymm0, dword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x98, 0x4a, 0x01], "vfmadd132ps ymm1, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x98, 0x4a, 0x01], "vfmadd132ps ymm1{k5}, ymm0, dword [edx + 0x4]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x98, 0xca], "vfmadd132ps ymm1{k5}{z}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x98, 0x0a], "vfmadd132ps ymm1{k5}{z}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x98, 0x4a, 0x01], "vfmadd132ps ymm1{k5}{z}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x98, 0xca], "vfmadd132ps ymm1, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x98, 0xca], "vfmadd132ps ymm1{k5}, ymm0, ymm2");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x98, 0x0a], "vfmadd132ps ymm1, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x98, 0x0a], "vfmadd132ps ymm1{k5}, ymm0, ymmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x98, 0x4a, 0x01], "vfmadd132ps ymm1, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x98, 0x4a, 0x01], "vfmadd132ps ymm1{k5}, ymm0, ymmword [edx + 0x20]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x98, 0xca], "vfmadd132pd zmm1{k5}{z}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x98, 0x0a], "vfmadd132pd zmm1{k5}{z}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x98, 0x4a, 0x01], "vfmadd132pd zmm1{k5}{z}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x98, 0xca], "vfmadd132pd zmm1{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x98, 0xca], "vfmadd132pd zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x98, 0x0a], "vfmadd132pd zmm1, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x98, 0x0a], "vfmadd132pd zmm1{k5}, zmm0, qword [edx]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x98, 0x4a, 0x01], "vfmadd132pd zmm1, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x98, 0x4a, 0x01], "vfmadd132pd zmm1{k5}, zmm0, qword [edx + 0x8]{1to8}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x98, 0xca], "vfmadd132pd zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x98, 0x0a], "vfmadd132pd xmm1{k5}{z}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x98, 0x4a, 0x01], "vfmadd132pd xmm1{k5}{z}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x98, 0xca], "vfmadd132pd zmm1{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x98, 0xca], "vfmadd132pd zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x98, 0x0a], "vfmadd132pd xmm1, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x98, 0x0a], "vfmadd132pd xmm1{k5}, xmm0, qword [edx]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x98, 0x4a, 0x01], "vfmadd132pd xmm1, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x98, 0x4a, 0x01], "vfmadd132pd xmm1{k5}, xmm0, qword [edx + 0x8]{1to2}");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x98, 0xca], "vfmadd132pd zmm1{k5}{z}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x98, 0x0a], "vfmadd132pd zmm1{k5}{z}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x98, 0x4a, 0x01], "vfmadd132pd zmm1{k5}{z}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x98, 0xca], "vfmadd132pd zmm1, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x98, 0xca], "vfmadd132pd zmm1{k5}, zmm0, zmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x98, 0x0a], "vfmadd132pd zmm1, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x98, 0x0a], "vfmadd132pd zmm1{k5}, zmm0, zmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x98, 0x4a, 0x01], "vfmadd132pd zmm1, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x98, 0x4a, 0x01], "vfmadd132pd zmm1{k5}, zmm0, zmmword [edx + 0x40]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x98, 0xca], "vfmadd132pd xmm1{k5}{z}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x98, 0x0a], "vfmadd132pd xmm1{k5}{z}, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x98, 0x4a, 0x01], "vfmadd132pd xmm1{k5}{z}, xmm0, xmmword [edx + 0x10]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x98, 0xca], "vfmadd132pd xmm1, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x98, 0xca], "vfmadd132pd xmm1{k5}, xmm0, xmm2");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x98, 0x0a], "vfmadd132pd xmm1, xmm0, xmmword [edx]");
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x9