// Copyright (C) 2007-2012  CEA/DEN, EDF R&D, OPEN CASCADE
//
// Copyright (C) 2003-2007  OPEN CASCADE, EADS/CCR, LIP6, CEA/DEN,
// CEDRAT, EDF R&D, LEG, PRINCIPIA R&D, BUREAU VERITAS
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
//
// See http://www.salome-platform.org/ or email : webmaster.salome@opencascade.com
//

//  File   : CorbaTypeManipulator.hxx
//  Author : Eric Fayolle (EDF)
//  Module : KERNEL
// Modified by : $LastChangedBy$
// Date        : $LastChangedDate: 2007-02-07 18:26:44 +0100 (mer, 07 fév 2007) $
// Id          : $Id: CorbaTypeManipulator.hxx,v 1.3.2.2.10.2.12.1 2012-04-12 14:05:06 vsr Exp $
//
#ifndef _CORBA_TYPE_MANIPULATION_HXX_
#define _CORBA_TYPE_MANIPULATION_HXX_

#include <iostream>
#include <cstring>
#include <CORBA.h>

//#define MYDEBUG

// Classes manipulation
// -------------------
//
// Ces diffrentes classes permettent d'unifier la manipulation des
// diffrents types de donnes dans un port datastream
// Les donnes sont manies par valeur ou par pointeur 
// pour viter les recopies de gros volume de donnes 

// Les classes prsentes quatre mthodes :
// - clone
// - get_data
// - delete_data
// - dump
// et
// trois types :
// - Type      : Le type CORBA de la donne manipule
// - InType    : Le mapping CORBA pour un paramtre IN du type manipul
// - InnerType : Type interne des valeurs d'un type contenant 

// Cette classe permet de manipuler des types CORBA 
// any, struct, union et sequence (utiliser plutt les seq_manipulator)
// Ces types sont manipuls par pointeur.
// Les donnes reues de CORBA sont systmatiquement
// dupliques pour tre conserves.
// Quelque soit le type de donne, les donnes sont considres 
// comme une donne unique (retour de size() == 1)
template <typename T >
class user_type_manipulation
{
public:
  typedef T *       Type;
  // correspond au mapping corba des type any, struct, 
  //                  union, squence en paramtre IN
  typedef const T & CorbaInType; 
  typedef T         InnerType;

  // Operation de recuperation des donnees venant de l'ORB et
  //  creation d'une copie (memoire spcialement allouee)
  static inline Type get_data(CorbaInType data) {
    return new T(data);
  }

  // Pb si ownerShip == True car appel par l'utilisateur de relPointer !
  static inline InnerType * const getPointer(Type data, bool ownerShip = false) {
    return data;
  }

  static inline void relPointer(InnerType * dataPtr) {
    delete dataPtr;
  }

  // Operation de clonage : par defaut, creation d'une copie en memoire allouee pour l'occasion
  static inline Type clone(Type data) { 
    return new T (* data);
  } 
  static inline Type clone(CorbaInType data) {
    return new T (data);
  }

  // Operation de cration
  static inline Type create (size_t size=1) { 
    return new T();
  } 

  // Operation de destruction d'une donnee
  static inline void delete_data(Type data) {
    delete data;
  }
  
  // Renvoie la taille de la donne
  static inline size_t size(Type data) { 
    return 1;
  } 

  // Dump de l'objet pour deboguage: neant car on ne connait pas sa structure
  static inline void dump (CorbaInType data) {}
};


// Gre les types CORBA atomiques ('Int', 'Char', 'Float', ...)
// Gre les types enums
// Gre les rfrences d'objets CORBA
// Ces types sont manipuls par valeur
// Les mthodes getPointer ... ne devrait pas tre utilise
// pour ce types de donnes
template <typename T>
class atom_manipulation
{
public:
  typedef T Type;
  // correspond au mapping corba des types simples en paramtre IN
  typedef T CorbaInType; 
  typedef T InnerType; 

    
  // Operation de recuperation des donnees venant de l'ORB : une copie par affectation simple
  static inline Type get_data(CorbaInType data) {
    return data;
  }

 static inline InnerType * const getPointer(Type & data, bool getOwnerShip = false) {
//    InnerType * ptr;
//     if (getOwnerShip) {
//       ptr =new InnerType[1];*ptr=data;
//       return ptr;
//     } else
//      return &data;
   return &data;
 }

//   static inline void relPointer(InnerType * dataPtr) {
//     return;
//         delete[] dataPtr;
//   }

// Je ne sais pas comment l'implmenter sans faire
// d'allocation heap
//static inline InnerType * allocPointer(size_t size=1) {
//    return  new InnerType[1];
  //}

  // Operation de clonage : une copie par affectation simple
  static inline Type clone(Type data) {
    return data;
  }

  // Inutile car Type == CorbaInType
  //   static inline Type clone(CorbaInType data) {
  //     return data;
  //   }

  // Operation de cration
//   static inline Type create(size_t size=1,InnerType * data=NULL,
//                          bool giveOwnerShip=false) {
//     Type dummy;
//     if (dataPtr)
//       return *data;
//     else
//       return dummy;
//   } 
    
  // Operation de destruction d'une donnee: rien a faire car pas de memoire a liberer
  static inline void delete_data(Type data) {}
    // Renvoie la taille de la donne

  static inline size_t size(Type data) { 
    return 1;
  } 

  // Dump de l'objet pour deboguage : Affiche la donnee
  static void inline dump (CorbaInType data) {
    std::cerr << "[atom_manipulation] Data : " << data << std::endl;
  }
};


// Gre un type sequence de taille illimitee (Unbounded)
// Ces types sont manipuls par pointeur
template <typename seq_T,typename elem_T>
class seq_u_manipulation {
  
public:
  typedef seq_T *       Type;        // Type de donne abstrait manipul par GenericPort::Put,Get,..
  typedef const seq_T & CorbaInType; // Mapping corba de la squence en paramtre IN
  typedef elem_T        InnerType;   // Il n'existe pas dans CORBA de seq_T::elem_T
                                     // C'est la raison d'tre du second paramtre template de seq_u_mani
 
  // Operation de recuperation des donnees venant de l'ORB
  // Remarque : On a un paramtre d'entre de type const seq_T &
  //            et en sortie un seq_T *
  static inline Type get_data(CorbaInType data) {
    CORBA::Long len = data.length();
    CORBA::Long max = data.maximum();
    // Rcupre et devient propritaire des donnes reues dans la squence. 
    // La squence reue (mais pas le buffer) sera dsalloue au retour 
    // de la mthode CORBA qui a reu le type CorbaInType en paramtre
    // (ex: GenericPort::put)
    // REM : Le mapping CORBA du type squence IN est : const seq &

    // OLD : On ne teste pas si le flag release de la squence est  true ou false 
    // OLD : ( pour des squences de chaines ou d'objrefs )
    // OLD :   -> Si on est collocalis le port uses doit crer une copie pour viter la modification
    // OLD : du contenu de la squence lorsque l'utilisateur modifie ses donnes dans son programme (0 copie)
    // OLD : ATTENTION TESTER p194 si le pointeur est null (release flag==false)
    // OLD :   -> La squence n'tait pas propritaire des donnes !

    // Le flag release() de la squence est  false si elle n'est pas propritaire du buffer
    // En  collocalit release() renvoie false car 
    // l'appel n'est pas propritaire de la squence. On effectue alors
    // une copie pour viter de perturber les structures de donnes de l'appelant.
    // En non collocalis on recre une squence avec le buffer de la premire dont on
    // a demand la proprit.

#ifdef MYDEBUG
    std::cout << "----seq_u_manipulation::get_data(..)-- MARK 1 ------------------" << std::endl;
#endif
    if ( data.release() ) {
      InnerType * p_data = const_cast<seq_T &>(data).get_buffer(true);

    // Cre une nouvelle sequence propritaire des donnes du buffer (pas de recopie)
    // Les donnes de la nouvelle squence seront automatiquement dsalloues 
    // par appel  la mthode freebuf dans le destructeur de la squence (cf  delete_data).
#ifdef MYDEBUG
      std::cout << "----seq_u_manipulation::get_data(..)-- MARK 1(0 copy) bis ------"<<  p_data <<"------------" << std::endl;
#endif
    
      return  new seq_T (max, len, p_data, true);
    }
#ifdef MYDEBUG
    std::cout << "----seq_u_manipulation::get_data(..)-- MARK 1(recopie) bis ------"<<  &data <<"------------" << std::endl;
#endif
    // Cre une nouvelle sequence propritaire des donnes du buffer (avec recopie)    
    return new seq_T(data);

  }

  static inline size_t size(Type data) { 
    return data->length();
  } 

  // Operation de destruction d'une donnee
  static inline void delete_data(Type data) {
    //La squence est dtruite par appel  son destructeur
    //Ce destructeur prend en compte la ncessit de dtruire ou non
    //les donnes contenues en fonction de son flag interne release()
    delete data;
  }

  // Operation de clonage : par defaut creation d'une copie en memoire allouee pour l'occasion
  // Utilisation du constructeur du type seq_T
  static inline Type clone(Type data) {
    return new seq_T (*data) ;
  }
  static inline Type clone(CorbaInType data) {
    return new seq_T (data);
  }

  // Permet d'obtenir un pointeur sur le buffer de la squence :
  // Si ownerShip=True, la squence n'est plus propritaire du buffer
  //         (son pointeur de buffer interne est aussi rinitialis) 
  //       On dtruit galement explicitement la squence (mais pas le buffer !)
  // Si ownerShip=False, la squence reste propritaire du buffer
  //    et l'utilisateur devra appeler delete_data sur la squence contenante pour
  //    dtruire  la fois la squence et le buffer contenu.
  static inline InnerType * const getPointer(Type data, bool ownerShip = false) {
    InnerType * p_data;
    if (ownerShip) {
      p_data = data->get_buffer(true);
      delete_data(data);
    } else
      p_data = data->get_buffer(false);
    return p_data;
  }

  // Permet de dsallouer le buffer dont on dtient le pointeur aprs appel
  //  la mthode getPointer avec ownerShip=True 
  static inline void relPointer(InnerType * dataPtr) {
    seq_T::freebuf(dataPtr);
  }

  // Permet d'allouer un buffer compatible avec le type squence
  static inline InnerType *  allocPointer(size_t size ) {
    return seq_T::allocbuf(size);
  }

  // Opration de cration de la squence CORBA soit
  // - Vide et de taille size
  // - Utilisant les donnes du pointeur *data de taille size 
  // (gnralement pas de recopie qlq soit l'ownership )
  // data doit avoir t allou par allocPointer si giveOwnerShip = true  
  static inline Type create(size_t size, InnerType * const data = NULL,
                            bool giveOwnerShip = false ) { 
    Type tmp;
    if (!data) {
      tmp = new seq_T();
      tmp->length(size);
    } else {
      tmp = new seq_T(size,size,data,giveOwnerShip); 
    }
    return tmp;
  } 

  // Copie le contenu de la squence dans le buffer idata de taille isize
  // pour les types non pointeur
  template <typename T >
  static inline void copy( Type data, T * const idata, size_t isize ) { 
    
    InnerType *dataPtr  = getPointer(data,false);

    for (int i = 0; i< isize; ++i) 
      idata[i]=dataPtr[i];

    // Le mode de recopie suivant ne permet pas  la conversion de type (ex int -> CORBA::Long)
    //OLD:     Type tmp = new seq_T(isize,isize,idata,false); 
    //OLD:     // giveOwnerShip == false -> seul le contenu du buffer data est dtruit et remplac
    //OLD:     // par celui de data dans l'affectation suivante :
    //OLD:     //       ---> ATTENTION SI LA TAILLE DU BUFFER EST TROP PETITE, QUE FAIT CORBA !
    //OLD:     //              corruption mmoire
    //OLD:     // Cependant ce cas devrait pas arriv (on s'assure dans les couches suprieures
    //OLD:     //  de la taille correcte du buffer de recopie)
    //OLD:     // Si giveOwnerShip tait == true -> le buffer et son contenu serait dtruit puis une 
    //OLD:     // allocation de la taille du buffer de data serait effectu avant la copie des donnes  
    //OLD:     // tmp = data;
  } 

  // Copie le contenu de la squence de char* dans le buffer idata de taille isize
  // La gnralisation de la recopie profonde est difficile du fait que CORBA ne renvoie pas
  // pas des objets de haut niveau de type std::vector<std::string> (avec des  interfaces d'accs identiques) 
  // mais un type simple C comme char *Tab[N]. On doit alors utiliser une mthode de recopie spcifique
  // comme l'appel C strcpy.
  static inline void copy( Type data, char* * const idata, size_t isize ) { 

    char* *dataPtr  = getPointer(data,false);

    // Si idata[i] n'a pas t allou suffisament grand,
    // il y a corruption de la mmoire
    for (int i = 0; i< isize; ++i) 
      strcpy(idata[i],dataPtr[i]);
  }
  
  // Dump de l'objet pour deboguage
  static void inline dump (CorbaInType data) {
    // Affiche la longueur des donnees
    std::cerr << "[seq_u_manipulation] Data length: " << data.length() << std::endl;
    // Affiche la longueur des donnees
    std::cerr << "[seq_u_manipulation] Data max: " << data.maximum() << std::endl;
  }
};


// TODO : Vrifier la conformit de l'implmentation par rapport
//        au type unbounded

// Gre un type sequence de taille limite (bounded)
// Ces types sont manipuls par pointeur
// Cette classe diffre de la seq_u_manipulation
// par la signature du constructeur de la squence
// utilis dans le methode get_data
template <typename seq_T,typename elem_T>
class seq_b_manipulation {
  
public:
  typedef seq_T *       Type;
  typedef const seq_T & CorbaInType;
  typedef elem_T        InnerType;


  // Operation de recuperation des donnees venant de l'ORB
  // Sans opration de notre part, ces donnes seraient perdues
  // au retour de la mthode put de GenericPort.
  // Remarque : On a un paramtre d'entre de type const seq_T &
  //            et en sortie un seq_T *
  static inline Type get_data(CorbaInType data) {
    CORBA::Long len = data.length();
    // Rcupre et devient propritaire des donnes reues dans la squence 
    // la squence sera dsallou (mais pas le buffer)
    // au retour de la mthode put (car mapping de type IN : const seq & )
     if ( data.release() ) {
       InnerType * p_data = const_cast<seq_T &>(data).get_buffer(true);

    // Cre une nouvelle sequence propritaire des donnes du buffer (gnralement pas de recopie)
    // Les donnes seront automatiquement dsalloues par appel interne  la mthode freebuf
    // lors de la destruction de l'objet par appel  delete_data.
#ifdef MYDEBUG
    std::cout << "----seq_u_manipulation::get_data(..)-- MARK 1bis Pas de Duplication  -----------" << std::endl;
#endif
       return new seq_T (len, p_data, true);
     }
#ifdef MYDEBUG
    std::cout << "----seq_u_manipulation::get_data(..)-- MARK 1bis Duplication pour en devenir propritaire -----------" << std::endl;
#endif
    // Cre une nouvelle sequence propritaire des donnes du buffer (avec recopie)    
    return new seq_T(data);

  }

  static inline size_t size(Type data) { 
    return data->length();
  } 

  // Operation de clonage : par defaut creation d'une copie en memoire allouee pour l'occasion
  // Utilisation du constructeur du type seq_T  
  static inline Type clone(Type data) {
    return new seq_T (* data);
  }
  static inline Type clone(CorbaInType data) {
    return new seq_T (data);
  }

  // Operation de destruction d'une donnee CORBA
  static inline void delete_data(Type data) {
    delete data;
  }

  // Permet d'obtenir un pointeur sur le buffer de la squence :
  // Si ownerShip=True, la squence n'est plus propritaire du buffer
  //         (son pointeur de buffer interne est aussi rinitialis) 
  //       On dtruit galement explicitement la squence (mais pas le buffer !)
  // Si ownerShip=False, la squence reste propritaire du buffer
  //    et l'utilisateur devra appeler delete_data sur la squence contenante pour
  //    dtruire  la fois la squence et le buffer contenu.
  static inline InnerType * const getPointer(Type data, bool getOwnerShip = false) {
    InnerType * p_data;
    if (getOwnerShip) {
      p_data = data->get_buffer(true);
      delete_data(data);
    } else
      p_data = data->get_buffer(false);
    return p_data;
  }

  // Permet de dsallouer le buffer dont on dtient le pointeur par appel
  //  la mthode getPointer avec ownerShip=True si la squence contenante
  //  t dtruite.
  static inline void relPointer(InnerType * dataPtr) {
    seq_T::freebuf(dataPtr);
  }

  // Permet d'allouer un buffer pour la squence
  static inline InnerType *  allocPointer(size_t size ) {
    return seq_T::allocbuf(size);
  }

  // Operation de cration du type corba soit
  // - Vide et de taille size
  // - Utilisant les donnes du pointeur *data de taille size 
  // (gnralement pas de recopie qlq soit l'ownership )
  // data doit avoir t allou par allocPointer si giveOwnerShip = true  
  static inline Type create(size_t size, InnerType * const data = NULL,
                            bool giveOwnerShip = false ) { 
    Type tmp;
    if (!data) {
      tmp = new seq_T();
      tmp->length(size);
    } else {
      tmp = new seq_T(size,data,giveOwnerShip); 
    }
    return tmp;
  } 

  
  // Dump de l'objet pour deboguage
  static inline void dump (CorbaInType data) {
    // Affiche la longueur des donnees
    std::cerr << "[seq_b_manipulation] Data length: " << data.length() << std::endl;
  }
};

#endif
