// Copyright (C) 2025 EDF
// All Rights Reserved
// This code is published under the GNU Lesser General Public License (GNU LGPL)
#ifndef  CONTINUATIONCUTSGRIDADAPTNONCONCAVEGENERS_H
#define  CONTINUATIONCUTSGRIDADAPTNONCONCAVEGENERS_H
#include <Eigen/Dense>
#include "geners/GenericIO.hh"
#include "StOpt/regression/ContinuationCutsGridAdaptNonConcave.h"
#include "StOpt/regression/BaseRegressionGeners.h"
#include "StOpt/regression/LocalLinearRegressionGeners.h"
#include "StOpt/core/grids/GridAdapt1DGeners.h"
#include "StOpt/core/grids/GridAdapt2DGeners.h"
#include "StOpt/core/utils/eigenGeners.h"

/** \file ContinuationCutsGridAdaptNonConcaveGeners.h
 * \brief Define non intrusive serialization with random access
*  \author Xavier Warin
 */

/// specialize the ClassIdSpecialization template
/// so that a ClassId object can be associated with the class we want to
/// serialize.  The second argument is the version number.
///@{
gs_specialize_class_id(StOpt::ContinuationCutsGridAdaptNonConcave, 1)
/// an external class
gs_declare_type_external(StOpt::ContinuationCutsGridAdaptNonConcave)
///@}

namespace gs
{
//
/// \brief  This is how the specialization of GenericWriter should look like
//
template <class Stream, class State >
struct GenericWriter < Stream, State, StOpt::ContinuationCutsGridAdaptNonConcave,
           Int2Type<IOTraits<int>::ISEXTERNAL> >
{
    inline static bool process(const StOpt::ContinuationCutsGridAdaptNonConcave  &p_regression, Stream &p_os,
                               State *, const bool p_processClassId)
    {
        // If necessary, serialize the class id
        static const ClassId current(ClassId::makeId<StOpt::ContinuationCutsGridAdaptNonConcave >());
        const bool status = p_processClassId ? ClassId::makeId<StOpt::ContinuationCutsGridAdaptNonConcave >().write(p_os) : true;
        // Serialize object data if the class id was successfully
        // written out
        if (status)
        {
            const Eigen::Array< Eigen::ArrayXXd, Eigen::Dynamic, 1  >   &values = p_regression.getValues();
            int isizeVal = values.size();
            write_pod(p_os, isizeVal);
            for (int ic = 0; ic < isizeVal; ++ic)
                write_item(p_os, values(ic));
            std::shared_ptr< StOpt::GridAdaptBase > ptrGrid = p_regression.getGrid();
            bool bSharedPtr = (ptrGrid ? true : false);
            write_pod(p_os, bSharedPtr);
            if (bSharedPtr)
                write_item(p_os, *p_regression.getGrid());
            write_item(p_os, *p_regression.getCondExp());
        }
        // Return "true" on success, "false" on failure
        return status && !p_os.fail();
    }
};

/// \brief  And this is the specialization of GenericReader
//
template <class Stream, class State  >
struct GenericReader < Stream, State, StOpt::ContinuationCutsGridAdaptNonConcave, Int2Type<IOTraits<int>::ISEXTERNAL> >
{
    inline static bool readIntoPtr(StOpt::ContinuationCutsGridAdaptNonConcave  *&ptr, Stream &p_is,
                                   State *p_st, const bool p_processClassId)
    {

        if (p_processClassId)
        {
            static const ClassId current(ClassId::makeId<StOpt::ContinuationCutsGridAdaptNonConcave>());
            ClassId id(p_is, 1);
            current.ensureSameName(id);
        }

        // Deserialize object data.
        int isizeVal = 0;
        read_pod(p_is, &isizeVal);
        Eigen::Array< Eigen::ArrayXXd, Eigen::Dynamic, 1> valReg(isizeVal);
        for (int ic = 0; ic < isizeVal ; ++ic)
        {
            std::unique_ptr<Eigen::ArrayXXd> valRegLoc  = read_item<Eigen::ArrayXXd>(p_is);
            valReg(ic) = *valRegLoc;
        }
        bool bSharedPtr ;
        read_pod(p_is, &bSharedPtr);
        std::unique_ptr<StOpt::GridAdaptBase> pgrid ;
        if (bSharedPtr)
        {
            pgrid  = read_item<StOpt::GridAdaptBase>(p_is);
        }
        std::shared_ptr<StOpt::GridAdaptBase > pgridShared(std::move(pgrid));
        std::unique_ptr<StOpt::BaseRegression> pcond  = read_item<StOpt::BaseRegression>(p_is);
        std::shared_ptr<StOpt::BaseRegression> pcondShared(std::move(pcond));
        if (p_is.fail())
            // Return "false" on failure
            return false;
        //Build the object from the stored data
        if (ptr)
        {
            *ptr = StOpt::ContinuationCutsGridAdaptNonConcave();
            ptr->loadForSimulation(pgridShared, pcondShared, valReg) ; // values);
            return true;
        }
        return false;
    }

    inline static bool process(StOpt::ContinuationCutsGridAdaptNonConcave &s, Stream &is,
                               State *st, const bool p_processClassId)
    {
        // Simply convert reading by reference into reading by pointer
        StOpt::ContinuationCutsGridAdaptNonConcave *ps = &s;
        return readIntoPtr(ps, is, st, p_processClassId);
    }
};
}



#endif/*  CONTINUATIONCUTSGRIDADAPTNONCONCAVEGENERS_H */
